;;; easymenu.el - Easy menu support for FSF and Lucid Emacs 19.
;; 
;; $Id: easymenu.el,v 5.10 1994/01/05 19:08:46 amanda Exp $
;;
;; LCD Archive Entry:
;; easymenu|Per Abrahamsen|abraham@iesd.auc.dk|
;; Easy menu support for FSF and Lucid Emacs 19|
;; $Date: 1994/01/05 19:08:46 $|$Revision: 5.10 $|~/misc/easymenu.el.gz|

;; Copyright (C) 1993 Per Abrahamsen <abraham@iesd.auc.dk>
;; Copyright (C) 1992, 1993 Free Software Foundation, Inc.
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;
;; The code to add and remove menus for Lucid Emacs was originally
;; from Alastair Burt <burt@dfki.uni-kl.de>.  The function
;; `easy-menu-create-keymaps' is derived from code from the file
;; `lemacs.el' in the FSF Emacs 19.15 distribution.

;; Commentary:
;;
;; Easymenu allows you to define menus for both FSF and Lucid Emacs
;; 19.  The advantages of using easymenu are:
;;
;; - Easier to use than either the FSF or Lucid menu syntax.
;;
;; - Common interface for Emacs 18, FSF Emacs 19, and Lucid Emacs.  
;;   (The code does nothing when run under Emacs 18).
;;
;; Otherwise easymenu is less powerful than either Lucid or FSF
;; menus, for example there is no way to make a menu item inactive.
;; An alternative to easymenu is to use Lucid menus, and enable the
;; Lucid menu emulator for FSF Emacs 19.  You can find that in
;; `lmenu.el' in the lisp directory.
;;
;; The public functions are:
;; 
;; - Function: easy-menu-define SYMBOL MAPS DOC MENU
;;     SYMBOL is the name of the variable that holds the menu. 
;;     MAPS is a list of keymaps where the menu should appear.
;;     DOC is the documentation string for the variable.
;;     MENU is a Lucid style menu description.  
;;
;;     A lucid style menu is a list where the first element is
;;     a string with the name of the menu, and the remaining elements
;;     are the menu items.  Each item can be either a
;;     - Menu: for nested menus.
;;     - String: for menu items that can not be selected.
;;     - Vector: for normal items.  It has three elements:
;;       1. A string with the name of the menu item.
;;       2. The function to be executed when the item is selected.
;;          This can be either a function name or a lisp expression.
;;       3. The symbol `t'
;;
;; - Function: easy-menu-add MENU [ MAP ]
;;     Add MENU to the current menubar in MAP.
;;
;; - Function: easy-menu-remove MENU
;;     Remove MENU from the current menubar.
;;
;; FSF Emacs 19 never uses `easy-menu-add' or `easy-menu-remove',
;; menus automatically appear and disappear when the keymaps
;; specified by the MAPS argument to `easy-menu-define' are
;; activated.
;;
;; Lucid Emacs never uses the MAPS argument to `easy-menu-define',
;; instead menus must explicitly be added and removed with
;; `easy-menu-add' and `easy-menu-remove'.
;;
;; If you want keyboard accelerators on the FSF EMacs 19 menus,
;; install powerkey.el.

;;; Code:

(cond 

;;; Emacs 18

((< (string-to-int emacs-version) 19)

(defmacro easy-menu-define (symbol maps doc menu))

(defmacro easy-menu-remove (menu))

(defmacro easy-menu-add (menu &optional map))

)					;Emacs 18

;;; Lucid Emacs

((string-match "Lucid" emacs-version)

(defmacro easy-menu-define (symbol maps doc menu)
  "Define SYMBOL to be a menu for keymaps MAPS.
DOC is the documentation string, and MENU is a Lucid style menu."
  (` (progn
       (defvar (, symbol) (, menu) (, doc))
       (defun (, symbol) (e)
	 (, doc)
	 (interactive "@e")
	 (setq zmacs-region-stays 't)
	 (popup-menu (, symbol))))))

(defun easy-menu-add (menu &optional map)
  "Add MENU to the current menu bar."
  (if current-menubar
      (if (assoc (car menu) current-menubar)
	  nil
	(set-buffer-menubar (copy-sequence current-menubar))
	(add-menu nil (car menu) (cdr menu)))))

(defun easy-menu-remove (menu)
  "Remove MENU from the current menu bar."
	  (if current-menubar
	      (if (assoc (car menu) current-menubar)
		  (delete-menu-item (list (car menu))))))

)					;Lucid Emacs

;;; FSF Emacs 19

(t

(defun easy-menu-create-keymaps (menu-name menu-items)
  (let ((menu (make-sparse-keymap menu-name))
	(loop menu-items))
    ;; Process items in reverse order,
    ;; since the define-key loop reverses them again.
    (setq menu-items (reverse menu-items))

    (while menu-items
      (let* ((item (car menu-items))
	     (callback (if (vectorp item) (aref item 1)))
	     command name spaces
	     (desc ""))
	(cond ((stringp item)
	       (setq command nil)
	       (setq name item))
	      ((consp item)
	       (setq command
		     (easy-menu-create-keymaps (car item) (cdr item)))
	       (setq name (concat (car item) "...")))
	      ((vectorp item)
	       (setq command 
		     (if (symbolp callback)
			 callback
		       (list 'lambda () '(interactive) callback)))
	       (setq name (aref item 0))
	       (and (stringp (aref item 2))
		    (boundp 'power-keys-use-tab-p)
		    (boundp 'power-keys-field-width)
		    (boundp 'power-keys-width)
		    (let ((spaces (max 1
				       (- power-keys-width
					  1
					  (length name)
					  (max (length (aref item 2))
					       power-keys-field-width)))))
		      (setq desc
			    (concat (if power-keys-use-tab-p
					"\t"
				      (make-string spaces ?\ ))
				    (aref item 2)))))))
	(if name 
	    (define-key menu (vector (intern name))
	      (cons (concat name desc) command))))
      (setq menu-items (cdr menu-items)))
    menu))

(defmacro easy-menu-define (symbol maps doc menu)
  "Define SYMBOL to be a menu for keymaps MAPS.
DOC is the documentation string, and MENU is a Lucid style menu."
  (` (let* ((maps (, maps))
	    (menu (, menu))
	    (keymap (easy-menu-create-keymaps (car menu) (cdr menu))))
       (mapcar (function (lambda (map) 
		 (define-key map (vector 'menu-bar (intern (car menu)))
		   (cons (car menu) keymap))
		 (if (fboundp 'update-power-keys)
		     (update-power-keys map))))
	       (if (keymapp maps) (list maps) maps)))))

(defmacro easy-menu-remove (menu))

(defmacro easy-menu-add (menu &optional map))

)					;FSF Emacs 19

)					;cond

(provide 'easymenu)

;;; easymenu.el ends here
