;;; ltx-env.el - environments for auc-tex.
;; 
;; $Id: ltx-env.el,v 1.30 1993/11/18 20:09:37 amanda Exp $

;; Copyright (C) 1987,1993 Per Abrahamsen
;; Copyright (C) 1991 Kresten Krab Thorup (krab@iesd.auc.dk).
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Code: 

;;; Customization

(defvar LaTeX-default-environment "itemize"
  "*The default environment when creating new ones with LaTeX-environment.")

 (make-variable-buffer-local 'LaTeX-default-environment)

(defvar LaTeX-float "htbp"
  "*Default float when creating figure and table environments.")

 (make-variable-buffer-local 'LaTeX-float)

(defvar LaTeX-figure-label "fig:"
  "*Default prefix to figure labels.")

 (make-variable-buffer-local 'LaTeX-figure-label)

(defvar LaTeX-table-label "tab:"
  "*Default prefix to table labels.")

 (make-variable-buffer-local 'LaTeX-table-label)

(defvar LaTeX-default-format ""
  "Specifies the default format string for array and tabular environments.")

 (make-variable-buffer-local 'LaTeX-default-format)

(defvar LaTeX-default-position ""
  "Specifies the default position string for array and tabular environments.")

 (make-variable-buffer-local 'LaTeX-default-position)

;;; Interactive Funtions

;;;###autoload
(defun LaTeX-environment (arg)
  "Make LaTeX environment (\\begin{...}-\\end{...} pair).
With optional ARG, modify current environment.

It may be customized with the following variables:

LaTeX-default-environment	Your favorite environment.
LaTeX-default-style		Your favorite document style.
LaTeX-default-options		Your favorite document style options.
LaTeX-float                     Where you want figures and tables to float.
LaTeX-table-label               Your prefix to labels in tables.
LaTeX-figure-label              Your prefix to labels in figures.
LaTeX-default-format		Format for array and tabular.
LaTeX-default-position		Position for array and tabular."
  
  (interactive "*P")
  (let ((environment (completing-read (concat "Environment type: (default "
					       (if (TeX-near-bobp)
						   "document"
						 LaTeX-default-environment)
					       ") ")
				       (LaTeX-environment-list))))
    ;; Get default
    (cond ((and (zerop (length environment))
		(TeX-near-bobp))
	   (setq environment "document"))
	  ((zerop (length environment))
	   (setq environment LaTeX-default-environment))
	  (t 
	   (setq LaTeX-default-environment environment)))

    (let ((entry (assoc environment (LaTeX-environment-list))))
      (if (null entry)
	  (LaTeX-add-environments (list environment)))
      
      (cond (arg
	     (LaTeX-modify-environment environment))
	    ((not (and entry (nth 1 entry)))
	     (LaTeX-insert-environment environment))
	    ((numberp (nth 1 entry))
	     (let ((count (nth 1 entry))
		   (args ""))
	       (while (> count 0)
		 (setq args (concat args TeX-grop TeX-grcl))
		 (setq count (- count 1)))
	       (LaTeX-insert-environment environment args)))
	    ((stringp (nth 1 entry))
	     (let ((prompts (cdr entry))
		   (args ""))
	       (while prompts
		 (setq args (concat args
				    TeX-grop
				    (read-from-minibuffer (concat (car prompts)
								  ": "))
				    TeX-grcl))
		 (setq prompts (cdr prompts)))
	       (LaTeX-insert-environment environment args)))
	    (t
	     (apply (nth 1 entry) environment (nthcdr 2 entry)))))))

;;;###autoload
(defun LaTeX-insert-item ()
  "Insert a new item in an environment.
You may use LaTeX-item-list to change the routines used to insert the item."
  (interactive "*")
  (let ((environment (LaTeX-current-environment)))
    (newline)
    (if (assoc environment LaTeX-item-list)
	(funcall (cdr (assoc environment LaTeX-item-list)))
      (insert TeX-esc "item "))
    (LaTeX-indent-line)))

;;;###autoload
(defun LaTeX-close-environment ()
  "Creates an \\end{...} to match the current environment."
  (interactive "*")
  (if (not (save-excursion
             (beginning-of-line)
             (looking-at "^[ \t]*$")))
      (insert "\n"))
  (insert "\\end{" (LaTeX-current-environment 1) "}")
  (LaTeX-indent-line)
  (insert "\n")
  (LaTeX-indent-line))

;;;###autoload
(defun LaTeX-hide-environment ()
  "Hide current LaTeX environment using selective display."
  (interactive)
  (outline-flag-region (save-excursion (LaTeX-find-matching-begin) (point))
		       (save-excursion (LaTeX-find-matching-end) (point))
		       ?\r))

;;;###autoload
(defun LaTeX-show-environment ()
  "Show current LaTeX environment."
  (interactive)
  (outline-flag-region (save-excursion (LaTeX-find-matching-begin) (point))
		       (save-excursion (LaTeX-find-matching-end) (point))
		       ?\n))

;;; Help Functions

(defun LaTeX-insert-environment (environment &optional extra)
  "Insert environment of type ENV, with optional argument EXTRA."
  (LaTeX-indent-line)
  (insert TeX-esc "begin" TeX-grop environment TeX-grcl)
  (if extra (insert extra))
  (newline-and-indent)
  (newline)
  (insert TeX-esc "end" TeX-grop environment TeX-grcl)
  (LaTeX-indent-line)
  (end-of-line 0))

(defun LaTeX-modify-environment (environment)
  ;; Modify current environment.
  (save-excursion
    (LaTeX-find-matching-end)
    (re-search-backward (concat (regexp-quote TeX-esc)
				"end"
				(regexp-quote TeX-grop)
				" *\\([a-zA-Z]*\\)"
				(regexp-quote TeX-grcl))
			(save-excursion (beginning-of-line 1) (point)))
    (replace-match (concat TeX-esc "end" TeX-grop environment TeX-grcl) t t)
    (beginning-of-line 1)
    (LaTeX-find-matching-begin)
    (re-search-forward (concat (regexp-quote TeX-esc)
			       "begin"
			       (regexp-quote TeX-grop)
			       " *\\([a-zA-Z]*\\)"
			       (regexp-quote TeX-grcl))
		       (save-excursion (end-of-line 1) (point)))
    (replace-match (concat TeX-esc "begin" TeX-grop environment TeX-grcl) t t)))

(defun LaTeX-current-environment (&optional arg)
  "Return the name (a string) of the enclosing LaTeX environment.
With optional ARG, find that outer level."
  (setq arg (if arg arg 1))
  (save-excursion
    (if (re-search-backward
	 (concat (regexp-quote TeX-esc) "begin" (regexp-quote TeX-grop)
		 "\\|"
		 (regexp-quote TeX-esc) "end" (regexp-quote TeX-grop)) nil t 1)
	(if (looking-at (concat (regexp-quote TeX-esc)
				"end" (regexp-quote TeX-grop)))
	    (LaTeX-current-environment (1+ arg))
	  (if (/= arg 1)
	      (LaTeX-current-environment (1- arg))
	    (search-forward TeX-grop)
	    (let ((beg (point)))
	      (search-forward TeX-grcl)
	      (backward-char 1)
	      (buffer-substring beg (point)))))
      "document")))

(defun TeX-near-bobp ()
  ;; Return t iff there's nothing but whitespace between (bob) and (point).
  (save-excursion
    (skip-chars-backward " \t\n")
    (bobp)))

;;; Environment Hooks

(defvar LaTeX-document-style-hook nil
  "List of hooks to run when inserting a document style environment.

To insert a hook here, you must insert it in the appropiate style file.")

(defun LaTeX-env-document (&optional ignore)
  "Create new LaTeX document."

  (TeX-insert-macro (if (string-equal LaTeX-version "2")
			"documentstyle"
		      "documentclass"))

  (newline 3)
  (end-of-line 0)
  (LaTeX-insert-environment "document")
  (run-hooks 'LaTeX-document-style-hook)
  (setq LaTeX-document-style-hook nil))

(defun LaTeX-env-item (environment)
  "Insert ENVIRONMENT and the first item."
  (LaTeX-insert-environment environment)
  (end-of-line 0)
  (delete-char 1)
  (delete-horizontal-space)
  (LaTeX-insert-item))

(defun LaTeX-env-figure (environment)
  "Create ENVIRONMENT with \\label and \\caption commands."
  (let ((float (read-string "Float to: " LaTeX-float))
	(caption (read-string "Caption: "))
	(label (read-string "Label: " (cond ((string= "figure" environment)
					     LaTeX-figure-label)
					    ((string= "table" environment)
					     LaTeX-table-label)
					    (t nil))))
        ; gf: ask if this should be centered
        (center (y-or-n-p "Center: ")))

    (setq LaTeX-float (if (zerop (length float))
			  LaTeX-float
			float))
	  
    (LaTeX-insert-environment environment
			      (concat LaTeX-optop LaTeX-float LaTeX-optcl))
    
    (if (or (zerop (length label))
	    (and (string= "figure" environment)
		 (equal LaTeX-figure-label label))
	    (and (string= "table" environment)
		 (equal LaTeX-table-label label)))
	()
      (newline-and-indent)
      (insert TeX-esc "label" TeX-grop label TeX-grcl)
      (LaTeX-add-labels label)
      (end-of-line 0)
      (LaTeX-indent-line))

    (if (zerop (length caption))
	()
      (newline-and-indent)
      (insert TeX-esc "caption" TeX-grop caption TeX-grcl)
      (end-of-line 0)
      (LaTeX-indent-line))
    
    ;; gf: add center if requested
    (if center
	(progn
	  (LaTeX-insert-environment "center")
	  (insert "\\leavevmode")
	  (newline-and-indent)))
    
    ;; gf: Add tabular if we're in a floating table
    (if (string= environment "table") (LaTeX-env-array "tabular"))))

(defun LaTeX-env-array (environment)
  "Insert ENVIRONMENT with position and column specifications 
like array and tabular."
  (let ((pos (read-string "Position: "))
	(fmt (read-string "Format: " LaTeX-default-format)))
    (setq LaTeX-default-position pos)
      (setq LaTeX-default-format fmt)
    (LaTeX-insert-environment environment
			      (concat
			        (if (not (zerop (length pos)))
				    (format "[%s]" pos))
				(format "{%s}" fmt)))
    (end-of-line 0)
    (next-line 1)
    (delete-horizontal-space)))

(defun LaTeX-env-label (environment)
  "Insert ENVIRONMENT and prompt for label."
  (let ((label (read-string "Label: ")))
    (LaTeX-insert-environment environment)
    (if (not (zerop (length label)))
	(progn
	  (newline-and-indent)
	  (insert TeX-esc "label" TeX-grop label TeX-grcl)
	  (LaTeX-add-labels label)
	  (end-of-line 0)))))

(defun LaTeX-env-list (environment)
  "Insert ENVIRONMENT and the first item."
  (let ((label (read-string "Default Label: ")))
    (LaTeX-insert-environment environment
			      (format "{%s}{}" label))
    (end-of-line 0)
    (delete-char 1)
    (delete-horizontal-space))
  (LaTeX-insert-item))

(defun LaTeX-env-minipage (environment)
  "Create new LaTeX minipage."
  (let ((pos (read-string "Position: " LaTeX-default-position))
	(width (read-string "Width: ")))
    (setq LaTeX-default-position pos)
    (if (zerop (length width))
	(setq width "4cm"))
    (LaTeX-insert-environment environment
			      (concat (if (not (zerop (length pos)))
					  (format "[%s]" pos))
				      (format "{%s}" width)))
    (end-of-line 0)
    (next-line 1)
    (delete-horizontal-space)))

(defun LaTeX-env-tabular* (environment)
  "Insert ENVIRONMENT with width, position and column specifications."
  (let ((width (read-string "Width: "))
	(pos (read-string "Position: " LaTeX-default-position))
	(fmt (read-string "Format: " LaTeX-default-format)))
    (setq LaTeX-default-position pos)
    (setq LaTeX-default-format fmt)
    (LaTeX-insert-environment environment
			      (concat
			        (if (not (zerop (length width)))
				    (format "{%s}" width))
			        (if (not (zerop (length pos)))
					  (format "[%s]" pos))
				(format "{%s}" fmt)))
    (end-of-line 0)
    (next-line 1)
    (delete-horizontal-space)))

(defun LaTeX-env-picture (environment)
  "Insert ENVIRONMENT with width, height specifications."
  (let ((width (read-string "Width: "))
	(height (read-string "Height: "))
	(x-offset (read-string "X Offset: "))
	(y-offset (read-string "Y Offset: ")))
    (if (zerop (length x-offset))
	(setq x-offset "0"))
    (if (zerop (length y-offset))
	(setq y-offset "0"))
    (LaTeX-insert-environment environment
			      (concat (format "(%s,%s)" width height)
				      (if (not (and (string= x-offset "0")
						    (string= y-offset "0")))
					  (format "(%s,%s)" x-offset y-offset))))
				      
    (end-of-line 0)
    (next-line 1)
    (delete-horizontal-space)))

(defun LaTeX-env-bib (environment)
  "Insert ENVIRONMENT with label for bibitem."
  (LaTeX-insert-environment environment
			    (concat TeX-grop
				    (read-string "Label for BibItem: " "99")
				    TeX-grcl))
  (end-of-line 0)
  (delete-char 1)
  (delete-horizontal-space)
  (LaTeX-insert-item))

;;; Item hooks

(defvar LaTeX-item-list '(("description" . LaTeX-item-argument)
			  ("thebibliography" . LaTeX-item-bib))
  "An list of environments where items have a special syntax. 
The cdr is the name of the function, used to insert this kind of items.")

(defun LaTeX-item-argument ()
  "Insert a new item with an optional argument."
  (insert TeX-esc "item[] ")
  (backward-char 2))

(defun LaTeX-item-bib ()
  "Insert a new bibitem."
  (TeX-insert-macro "bibitem"))

(provide 'ltx-env)

;;; ltx-env.el ends here
