;;; ltx-misc.el - Miscellaneous utilility functions for TeX mode.
;; 
;; $Id: ltx-misc.el,v 5.24 1993/11/18 20:09:47 amanda Exp $

;; Copyright 1991 Kresten Krab Thorup
;; Copyright 1993 Per Abrahamsen
;; 
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Code:

(require 'tex-init)
(require 'easymenu)

;;; Syntax

(defvar LaTeX-optop "["
  "The LaTeX optional argument opening character.")
(make-variable-buffer-local 'LaTeX-optop)

(defvar LaTeX-optcl "]"
  "The LaTeX optional argument closeing character.")
(make-variable-buffer-local 'LaTeX-optcl)

;;; Style

(defvar LaTeX-default-style "article"
  "*Default when creating new documents.")

  (make-variable-buffer-local 'LaTeX-default-style)

(defvar LaTeX-default-options nil
  "*Default options to documentstyle.
A list of strings.")

 (make-variable-buffer-local 'LaTeX-default-options)

;;; Syntax Table

(defvar LaTeX-mode-syntax-table (copy-syntax-table TeX-mode-syntax-table)
  "Syntax table used in LaTeX mode.")

(progn ; set [] to match for LaTeX.
  (modify-syntax-entry (string-to-char LaTeX-optop)
		       (concat "(" LaTeX-optcl) 
		       LaTeX-mode-syntax-table)  
  (modify-syntax-entry (string-to-char LaTeX-optcl)
		       (concat ")" LaTeX-optop)
		       LaTeX-mode-syntax-table))

;;; Sections

(defvar LaTeX-section-list '(("part" 0)
			     ("chapter" 1)
			     ("section" 2)
			     ("subsection" 3)
			     ("subsubsection" 4)
			     ("paragraph" 5)
			     ("subparagraph" 6))
  "List which elements is the names of the sections used by LaTeX.")

(defun LaTeX-section-name (level)
  "Return the name of the section corresponding to LEVEL."
  (let ((entry (TeX-member level LaTeX-section-list
			   (function (lambda (a b) (equal a (nth 1 b)))))))
    (if entry
	(nth 0 entry)
      nil)))

(defun LaTeX-section-level (name)
  "Return the level of the section NAME."
  (let ((entry (TeX-member name LaTeX-section-list
			   (function (lambda (a b) (equal a (nth 0 b)))))))

    (if entry
	(nth 1 entry)
      nil)))

(defvar TeX-outline-extra nil
  "*List of extra TeX outline levels.

Each element is a list with two entries.  The first entry is the
regular expression matching a header, and the second is the level of
the header.  See LaTeX-section-list for existing header levels.")

(defun LaTeX-outline-regexp (&optional anywhere)
  "Return regexp for LaTeX sections.  

If optional argument ANYWHERE is not nil, do not require that the
header is at the start of a line."
  (concat (if anywhere "" "^")
	  "[ \t]*"
	  (regexp-quote TeX-esc)
	  "\\(appendix\\|documentstyle\\|"
	  (mapconcat 'car LaTeX-section-list "\\|")
	  "\\)\\b"
	  (if TeX-outline-extra
	      "\\|"
	    "")
	  (mapconcat 'car TeX-outline-extra "\\|")
	  "\\|" TeX-header-end
	  "\\|" TeX-trailer-start))

(defvar LaTeX-largest-level nil
  "Largest sectioning level with current document style")

(make-variable-buffer-local 'LaTeX-largest-level)

(defun LaTeX-largest-level ()
  (TeX-update-style)
  LaTeX-largest-level)

(defun LaTeX-outline-offset ()
  "Offset to add to LaTeX-section-list levels to get outline level."
  (- 2 (LaTeX-largest-level)))

(defun TeX-look-at (list)
  "Check if we are looking at the first element of a member of LIST.
If so, return the second element, otherwise return nil."
  (while (and list
	      (not (looking-at (nth 0 (car list)))))
    (setq list (cdr list)))
  (if list
      (nth 1 (car list))
    nil))

(defun LaTeX-outline-level ()
  "Find the level of current outline heading in an LaTeX document."
  (cond ((looking-at TeX-header-end) 1)
	((looking-at TeX-trailer-start) 1)
	((TeX-look-at TeX-outline-extra)
	 (max 1 (+ (TeX-look-at TeX-outline-extra)
		   (LaTeX-outline-offset))))
	(t
	 (save-excursion
	  (skip-chars-forward " \t")
	  (forward-char 1)
	  (cond ((looking-at "appendix") 1)
		((looking-at "documentstyle") 1)
		((TeX-look-at LaTeX-section-list)
		 (max 1 (+ (TeX-look-at LaTeX-section-list)
			   (LaTeX-outline-offset))))
		(t
		 (error "Unrecognized header.")))))))

(add-hook 'TeX-remove-style-hook
	  (function (lambda () (setq LaTeX-largest-level nil))))

;;; Environments

(autoload 'LaTeX-environment "ltx-env" no-doc t)
(autoload 'LaTeX-insert-item "ltx-env" no-doc t)
(autoload 'LaTeX-close-environment "ltx-env" no-doc t)
(autoload 'LaTeX-current-environment "ltx-env" no-doc)

(defvar LaTeX-environment-list nil
  "List of known LaTeX Environments.")

(make-variable-buffer-local 'LaTeX-environment-list)

(defvar LaTeX-environment-list-changed nil
  "True if the environment list has changed since last time it was sorted.")

(make-variable-buffer-local 'LaTeX-environment-list-changed)

(defun LaTeX-add-environments (&rest environments)
  "Add arguments to the list of known environments."
  (if LaTeX-environment-list-changed
      (setq LaTeX-environment-list (cons environments LaTeX-environment-list))
    (setq LaTeX-environment-list-changed t)
    (setq LaTeX-environment-list (list LaTeX-environment-list environments))))

(defun LaTeX-environment-list ()
  "Return list of environments TeX knows about.
Each element is a list with the environment (a string) as the first entry."
  (TeX-update-style)
  (if LaTeX-environment-list-changed
      (progn
	(setq LaTeX-environment-list-changed nil)
	;; Sort it
	(setq LaTeX-environment-list
	      (sort (mapcar 'TeX-listify
			    (apply 'append LaTeX-environment-list))
		    'TeX-car-string-lessp))
	;; Make it unique
	(let ((entry LaTeX-environment-list))
	  (if entry
	      (while (cdr entry)
		(let ((this (car entry))
		      (next (car (cdr entry))))
		  (if (string-equal (car this) (car next))
		      (progn
			;; We have to equal environments.  Use the one with
			;; most arguments.
			(if (> (length next) (length this))
			    (setcdr this (cdr next)))
			(setcdr entry (cdr (cdr entry))))
		    (setq entry (cdr entry)))))))))
  LaTeX-environment-list)

(add-hook 'TeX-remove-style-hook
	  (function (lambda () (setq LaTeX-environment-list nil))))

;;; Bibliographies

(defvar LaTeX-bibliography-list nil
  "List of known LaTeX Bibliographies.")

(make-variable-buffer-local 'LaTeX-bibliography-list)

(defvar LaTeX-bibliography-list-changed nil
  "True if the bibliography list has changed since last time it was sorted.")

(make-variable-buffer-local 'LaTeX-bibliography-list-changed)

(defun LaTeX-add-bibliographies (&rest bibliographies)
  "Add arguments to the list of known bibliographies."
  (if LaTeX-bibliography-list-changed
      (setq LaTeX-bibliography-list
	    (cons bibliographies LaTeX-bibliography-list))
    (setq LaTeX-bibliography-list-changed t)
    (setq LaTeX-bibliography-list
	  (list LaTeX-bibliography-list bibliographies))))

(defun LaTeX-bibliography-list ()
  "Return list of bibliographies TeX knows about.
Each element is a list with the bibliography (a string) as the first entry."
  (TeX-update-style)
  (if LaTeX-bibliography-list-changed
      (progn
	(setq LaTeX-bibliography-list-changed nil)
	;; Sort it
	(setq LaTeX-bibliography-list
	      (sort (mapcar 'TeX-listify
			    (apply 'append LaTeX-bibliography-list))
		    'TeX-car-string-lessp))
	;; Make it unique
	(let ((entry LaTeX-bibliography-list))
	  (if entry
	      (while (cdr entry)
		(let ((this (car entry))
		      (next (car (cdr entry))))
		  (if (string-equal (car this) (car next))
		      (progn
			;; We have to equal bibliographies.  Use the one with
			;; most arguments.
			(if (> (length next) (length this))
			    (setcdr this (cdr next)))
			(setcdr entry (cdr (cdr entry))))))
		(setq entry (cdr entry)))))))
  LaTeX-bibliography-list)

(add-hook 'TeX-remove-style-hook
	  (function (lambda () (setq LaTeX-bibliographies-list nil))))

;;; BibTeX styles

;;; Bibitems

(defvar LaTeX-bibitem-list nil
  "List of known LaTeX Bibitems.")

(make-variable-buffer-local 'LaTeX-bibitem-list)

(defvar LaTeX-bibitem-list-changed nil
  "True if the bibitem list has changed since last time it was sorted.")

(make-variable-buffer-local 'LaTeX-bibitem-list-changed)

(defun LaTeX-add-bibitems (&rest bibitems)
  "Add arguments to the list of known bibitems."
  (if LaTeX-bibitem-list-changed
      (setq LaTeX-bibitem-list (cons bibitems LaTeX-bibitem-list))
    (setq LaTeX-bibitem-list-changed t)
    (setq LaTeX-bibitem-list (list LaTeX-bibitem-list bibitems))))

(defun LaTeX-bibitem-list ()
  "Return list of bibitems TeX knows about.
Each element is a list with the bibitem (a string) as the first entry."
  (TeX-update-style)
  (if LaTeX-bibitem-list-changed
      (progn
	(setq LaTeX-bibitem-list-changed nil)
	;; Sort it
	(setq LaTeX-bibitem-list
	      (sort (mapcar 'TeX-listify
			    (apply 'append LaTeX-bibitem-list))
		    'TeX-car-string-lessp))
	;; Make it unique
	(let ((entry LaTeX-bibitem-list))
	  (if entry
	      (while (cdr entry)
		(let ((this (car entry))
		      (next (car (cdr entry))))
		  (if (string-equal (car this) (car next))
		      (progn
			;; We have to equal bibitems.  Use the one with
			;; most arguments.
			(if (> (length next) (length this))
			    (setcdr this (cdr next)))
			(setcdr entry (cdr (cdr entry))))))
		(setq entry (cdr entry)))))))
  LaTeX-bibitem-list)

(add-hook 'TeX-remove-style-hook
	  (function (lambda () (setq LaTeX-bibitem-list nil))))

;;; Labels

(defvar LaTeX-label-list nil
  "List of known LaTeX Labels.")

(make-variable-buffer-local 'LaTeX-label-list)

(defvar LaTeX-label-list-changed nil
  "True if the label list has changed since last time it was sorted.")

(make-variable-buffer-local 'LaTeX-label-list-changed)

(defun LaTeX-add-labels (&rest labels)
  "Add arguments to the list of known labels."
  (if LaTeX-label-list-changed
      (setq LaTeX-label-list (cons labels LaTeX-label-list))
    (setq LaTeX-label-list-changed t)
    (setq LaTeX-label-list (list LaTeX-label-list labels))))

(defun LaTeX-label-list ()
  "Return list of labels TeX knows about.
Each element is a list with the label (a string) as the first entry."
  (TeX-update-style)
  (if LaTeX-label-list-changed
      (progn
	(setq LaTeX-label-list-changed nil)
	;; Sort it
	(setq LaTeX-label-list
	      (sort (mapcar 'TeX-listify
			    (apply 'append LaTeX-label-list))
		    'TeX-car-string-lessp))
	;; Make it unique
	(let ((entry LaTeX-label-list))
	  (if entry
	      (while (cdr entry)
		(let ((this (car entry))
		      (next (car (cdr entry))))
		  (if (string-equal (car this) (car next))
		      (progn
			;; We have to equal labels.  Use the one with
			;; most arguments.
			(if (> (length next) (length this))
			    (setcdr this (cdr next)))
			(setcdr entry (cdr (cdr entry))))))
		(setq entry (cdr entry)))))))
  LaTeX-label-list)

(add-hook 'TeX-remove-style-hook
	  (function (lambda () (setq LaTeX-label-list nil))))

;;; Indentation

(defvar LaTeX-indent-level 2
  "*Indentation of begin-end blocks in LaTeX.")

(defvar LaTeX-item-indent -2
  "*Extra indentation for lines beginning with an item.")

(defvar LaTeX-item-regexp "\\(bib\\)?item\\b"
  "*Regular expression matching macros considered items.")

(defun LaTeX-indent-line ()
  "Indent the line containing point, as LaTeX source.
Add LaTeX-indent-level indentation in each \\begin{ - \\end{ block.
Lines starting with an item is given an extra indentation of
LaTeX-item-indent."
  (interactive)
  (save-excursion
    (let ((indent (calculate-LaTeX-indentation)))
      (if (/= (current-indentation) indent)
	  (let ((beg (progn
		       (beginning-of-line)
		       (point))))
	    (back-to-indentation)
	    (delete-region beg (point))
	    (indent-to indent)))))
  (if (< (current-column) (calculate-LaTeX-indentation))
      (back-to-indentation)))

(defun LaTeX-fill-region-as-paragraph (from to &optional justify-flag)
  "Fill region as one paragraph: break lines to fit fill-column.
Prefix arg means justify too.
From program, pass args FROM, TO and JUSTIFY-FLAG."
  (interactive "*r\nP")
  (save-restriction
    (goto-char from)
    (skip-chars-forward " \n")
    (LaTeX-indent-line)
    (beginning-of-line)
    (narrow-to-region (point) to)
    (setq from (point))
    
    ;; from is now before the text to fill,
    ;; but after any fill prefix on the first line.
    
    ;; Make sure sentences ending at end of line get an extra space.
    (goto-char from)
    (while (re-search-forward "[.?!][])\"']*$" nil t)
      (insert ? ))
    ;; The change all newlines to spaces.
    (subst-char-in-region from (point-max) ?\n ?\ )
    ;; Flush excess spaces, except in the paragraph indentation.
    (goto-char from)
    (skip-chars-forward " \t")
    (while (re-search-forward "   *" nil t)
      (delete-region
       (+ (match-beginning 0)
	  (if (save-excursion
		(skip-chars-backward " ])\"'")
		(memq (preceding-char) '(?. ?? ?!)))
	      2 1))
       (match-end 0)))
    (goto-char (point-max))
    (delete-horizontal-space)
    (insert "  ")
    (goto-char (point-min))
    (let ((prefixcol 0))
      (while (not (eobp))
	(move-to-column (1+ fill-column))
	(if (eobp)
	    nil
	  (skip-chars-backward "^ \n")
	  (if (if (zerop prefixcol)
		  (bolp)
		(>= prefixcol (current-column)))
	      (skip-chars-forward "^ \n")
	    (forward-char -1)))
	(delete-horizontal-space)
	(if (equal (preceding-char) ?\\)
	    (insert ? ))
	(insert ?\n)
	(LaTeX-indent-line)
	(setq prefixcol (current-column))
	(and justify-flag (not (eobp))
	     (progn
	       (forward-line -1)
	       (justify-current-line)
	       (forward-line 1)))
	)
      (goto-char (point-max))
      (delete-horizontal-space))))


(defun LaTeX-format-paragraph (prefix)
  "Fill and indent paragraph at or after point.
Prefix arg means justify as well."
  (interactive "*P")
  (save-excursion
    (beginning-of-line)
    (forward-paragraph)
    (or (bolp) (newline 1))
    (and (eobp) (open-line 1))
    (let ((end (point-marker))
	  (start (progn
		   (backward-paragraph)
		   (point))))
      (LaTeX-fill-region-as-paragraph start end prefix))))

(defun LaTeX-format-region (from to &optional justify what)
  "Fill and indent each of the paragraphs in the region as LaTeX text.
Prefix arg (non-nil third arg, if called from program)
means justify as well. Fourth arg WHAT is a word to be displayed when
formatting."
  (interactive "*r\nP")
  (save-restriction
    (save-excursion
      (let ((length (- to from))) 
	(goto-char from)
	(beginning-of-line)
	(while (< (point) to)
	  (message "Formatting%s ... %d%%"
		   (if (not what)
		       ""
		     what)
		   (/ (* 100 (- (point) from)) length))
	  (save-excursion (LaTeX-format-paragraph justify))
	  (forward-paragraph 2)
	  (if (not (eobp))
	      (backward-paragraph)))))
    (message "Finished")))

(defun LaTeX-find-matching-end ()
  "Move point to the \\end of the current environment"
  (interactive)
  (let ((regexp (concat (regexp-quote TeX-esc) "\\(begin\\|end\\)\\b"))
	(level 1))
    (beginning-of-line 1)
    (if (looking-at (concat " *" (regexp-quote TeX-esc) "begin\\b"))
	(end-of-line 1))
    (while (and (> level 0) (re-search-forward regexp nil t))
      (if (= (char-after (1+ (match-beginning 0))) ?b);;begin
	  (setq level (1+ level))
	(setq level (1- level))))
    (if (= level 0)
	(search-forward "}")
      (error "Can't locate end of current environment"))))

(defun LaTeX-find-matching-begin ()
  "Move point to the \\begin of the current environment"
  (interactive)
  (let ((regexp (concat (regexp-quote TeX-esc) "\\(begin\\|end\\)\\b"))
	(level 1))
    (beginning-of-line 1)
    (if (looking-at (concat " *" (regexp-quote TeX-esc) "begin\\b"))
	(end-of-line 1))
    (while (and (> level 0) (re-search-backward regexp nil t))
      (if (= (char-after (1+ (match-beginning 0))) ?e);;end
	  (setq level (1+ level))
	(setq level (1- level))))
    (or (= level 0)
	(error "Can't locate beginning of current environment"))))

(defun LaTeX-mark-environment ()
  "Set mark to end of current environment and point to the matching begin
will not work properly if there are unbalanced begin-end pairs in
comments and verbatim environments"
  (interactive)
  (let ((cur (point)))
    (LaTeX-find-matching-end)
    (beginning-of-line 2)
    (set-mark (point))
    (goto-char cur)
    (LaTeX-find-matching-begin)))

(defun LaTeX-format-environment (justify)
  "Fill and indent current environment as LaTeX text."
  (interactive "*P")
  (save-excursion
    (LaTeX-mark-environment)
    (re-search-forward "{\\([^}]+\\)}")
    (LaTeX-format-region
     (region-beginning)
     (region-end)
     justify
     (concat " environment " (TeX-match-buffer 1)))))

(defun LaTeX-format-section (justify)
  "Fill and indent current logical section as LaTeX text."
  (interactive "*P")
  (save-excursion
    (LaTeX-mark-section)
    (re-search-forward "{\\([^}]+\\)}")
    (LaTeX-format-region
     (region-beginning)
     (region-end)
     justify
     (concat " section " (TeX-match-buffer 1)))))

(defun LaTeX-mark-section ()
  "Set mark at end of current logical section, and point at top."
  (interactive)
  (re-search-forward (concat  "\\(" (LaTeX-outline-regexp)
			      "\\|\\'\\)"))
  (re-search-backward "^")
  (set-mark (point))
  (re-search-backward (concat "\\(" (LaTeX-outline-regexp)
			      "\\|\\`\\)")))

(defun LaTeX-format-buffer (justify)
  "Fill and indent current buffer as LaTeX text."
  (interactive "*P")
  (save-excursion
    (LaTeX-format-region
     (point-min)
     (point-max)
     justify
     (concat " buffer " (buffer-name)))))

(defun calculate-LaTeX-indentation ()
  ;; Return the correct indentation of line of LaTeX source. (I hope...)
  (save-excursion
    (back-to-indentation)
    (cond ((looking-at (concat (regexp-quote TeX-esc) "end{verbatim}"))
	   (save-excursion
	     (search-backward "\\begin{verbatim}")
	     (current-indentation)))
	  ((looking-at (concat "\\("
			       (regexp-quote TeX-esc)
			       "end *{\\|"
			       (regexp-quote TeX-esc)
			       "right\\)"))
	   (- (calculate-normal-LaTeX-indentation) LaTeX-indent-level))
	  ((looking-at (concat (regexp-quote TeX-esc) LaTeX-item-regexp))
	   (+ (calculate-normal-LaTeX-indentation) LaTeX-item-indent))
	  (t (calculate-normal-LaTeX-indentation)))))

(defun calculate-normal-LaTeX-indentation ()
  "Return the correct indentation of a normal line of text.
The point is supposed to be at the beginning of the current line."
  (skip-chars-backward "\n\t ")
  (move-to-column (current-indentation))
  (cond ((looking-at (concat (regexp-quote TeX-esc) "begin{document}"))
	 ;; I dislike having all of the document indented...
	 (current-indentation))
	((looking-at (concat (regexp-quote TeX-esc) "begin"
			     (regexp-quote TeX-grop)
			     "verbatim"))
	 0)
	(t (+ (TeX-brace-count-line)
	      (cond ((looking-at (concat (regexp-quote TeX-esc) "begin *"
					 (regexp-quote TeX-grop)))
		     (+ (current-indentation) LaTeX-indent-level))
		    ((looking-at (concat (regexp-quote TeX-esc)
					 LaTeX-item-regexp))
		     (- (current-indentation) LaTeX-item-indent))
		    (t (current-indentation)))))))

;;; Keymap

(defvar LaTeX-mode-map
  (let ((map (copy-keymap TeX-mode-map)))
    
    ;; Standard
    (define-key map "\n"      'reindent-then-newline-and-indent)
    
    ;; From ltx-misc.el
    (define-key map "\t"      'LaTeX-indent-line)
    (define-key map "\eq"     'LaTeX-format-paragraph) ;*** Alias
    (define-key map "\eg"     'LaTeX-format-region) ;*** Alias
    
    (define-key map "\C-c\C-q\C-p" 'LaTeX-format-paragraph)
    (define-key map "\C-c\C-q\C-r" 'LaTeX-format-region)
    (define-key map "\C-c\C-q\C-s" 'LaTeX-format-section)
    (define-key map "\C-c\C-q\C-e" 'LaTeX-format-environment)
    
    (define-key map "\C-c."    'LaTeX-mark-environment) ;*** Dubious
    (define-key map "\C-c*"    'LaTeX-mark-section) ;*** Dubious

    ;; From ltx-env.el
    (define-key map "\C-c\C-e" 'LaTeX-environment)
    (define-key map "\C-c\n"   'LaTeX-insert-item)
    (define-key map "\e\r"    'LaTeX-insert-item) ;*** Alias
    (define-key map "\C-c]" 'LaTeX-close-environment)

    (let ((prefix (if (boundp 'outline-minor-mode-prefix)
		      outline-minor-mode-prefix
		    "\C-c")))
      (define-key map (concat prefix "\C-z") 'LaTeX-hide-environment)
      (define-key map (concat prefix "\C-x") 'LaTeX-show-environment))
    
    ;; From tex-math.el
    (define-key map "\C-c~"    'LaTeX-math-mode) ;*** Dubious
    
    ;; From ltx-sec.el
    (define-key map "\C-c\C-s" 'LaTeX-section)
    map)
  "Keymap used in LaTeX-mode.")

(easy-menu-define LaTeX-mode-menu
    LaTeX-mode-map
    "Menu used in LaTeX mode."
  (list "AUC TeX"
	(list "Templates"
	      ["Environment..." LaTeX-environment t]
	      ["Section..." LaTeX-section t]
	      ["Macro..." TeX-insert-macro t]
	      ["Complete" TeX-complete-symbol t]
	      ["Item" LaTeX-insert-item "C-c LFD"])
	(list "Insert Font"
	      ["Emphasize"  (TeX-font nil ?\C-e) "C-c C-f C-e"]
	      ["Bold"       (TeX-font nil ?\C-b) "C-c C-f C-b"]
	      ["Typewriter" (TeX-font nil ?\C-t) "C-c C-f C-t"]
	      ["Small Caps" (TeX-font nil ?\C-c) "C-c C-f C-c"]
	      ["Italic"     (TeX-font nil ?\C-i) "C-c C-f C-i"]
	      ["Slanted"    (TeX-font nil ?\C-s) "C-c C-f C-s"]
	      ["Roman"      (TeX-font nil ?\C-r) "C-c C-f C-r"])
	(list "Insert Font around Region"
	      ["Emphasize"
	       (progn (kill-region (point) (mark)) (TeX-font nil ?\C-e) (yank))
	       "C-w C-c C-f C-e C-y"]
	      ["Bold"
	       (progn (kill-region (point) (mark)) (TeX-font nil ?\C-b) (yank))
	       "C-w C-c C-f C-b C-y"]
	      ["Typewriter"
	       (progn (kill-region (point) (mark))
		      (TeX-font nil ?\C-t)
		      (yank))
	       "C-w C-c C-f C-t C-y"]
	      ["Small Caps"
	       (progn (kill-region (point) (mark)) 
		      (TeX-font nil ?\C-c)
		      (yank))
	       "C-w C-c C-f C-c C-y"]
	      ["Italic"
	       (progn (kill-region (point) (mark))
		      (TeX-font nil ?\C-i)
		      (yank))
	       "C-w C-c C-f C-i C-y"]
	      ["Slanted"
	       (progn (kill-region (point) (mark))
		      (TeX-font nil ?\C-s)
		      (yank))
	       "C-w C-c C-f C-s C-y"]
	      ["Roman"
	       (progn (kill-region (point) (mark))
		      (TeX-font nil ?\C-r)
		      (yank))
	       "C-w C-c C-f C-r C-y"])
	(list "Change Font"
	      ["Emphasize"  (TeX-font t ?\C-e) "C-u C-c C-f C-e"]
	      ["Bold"       (TeX-font t ?\C-b) "C-u C-c C-f C-b"]
	      ["Typewriter" (TeX-font t ?\C-t) "C-u C-c C-f C-t"]
	      ["Small Caps" (TeX-font t ?\C-c) "C-u C-c C-f C-c"]
	      ["Italic"     (TeX-font t ?\C-i) "C-u C-c C-f C-i"]
	      ["Slanted"    (TeX-font t ?\C-s) "C-u C-c C-f C-s"]
	      ["Roman"      (TeX-font t ?\C-r) "C-u C-c C-f C-r"])
	["Delete Font" (TeX-font t ?\C-d) "C-c C-f C-d"]
	"-"
	["Save Document" TeX-save-document t]
	(TeX-command-create-menu "Command on Master File" 'TeX-master-file)
	(TeX-command-create-menu "Command on Region" 'TeX-region-file)
	["Next Error" TeX-next-error t]
	(list "TeX Output"
	      ["Kill Job" TeX-kill-job t]
	      ["Toggle debug of boxes" TeX-toggle-debug-boxes t]
	      ["Switch to original file" TeX-home-buffer t]
	      ["Recenter Output Buffer" TeX-recenter-output-buffer t])
	"--"
	(list "Formatting and Marking"
	      ["Format Environment" LaTeX-format-environment t]
	      ["Format Paragraph" LaTeX-format-paragraph t]
	      ["Format Region" LaTeX-format-region t]
	      ["Format Section" LaTeX-format-section t]
	      ["Mark Environment" LaTeX-mark-environment t]
	      ["Mark Section" LaTeX-mark-section t]
	      ["Hide Environment" LaTeX-hide-environment t]
	      ["Show Environment" LaTeX-show-environment t])
	;; ["Uncomment" TeX-un-comment t]
	["Uncomment Region" TeX-un-comment-region t]
	;; ["Comment Out Paragraph" TeX-comment-out-paragraph t]
	;; ["Comment Paragraph" TeX-comment-paragraph t]
	["Comment Region" TeX-comment-region t]
	["Switch to Master file" TeX-home-buffer t]
	["Math Mode" LaTeX-math-mode t]
	["Reset Buffer" TeX-normal-mode t]
	["Reset AUC TeX" (TeX-normal-mode t) "C-u C-c C-n"]))

;; Seems to be a Lucid convention
(if (fboundp 'LaTeX-mode-menu)
    (define-key LaTeX-mode-map 'button3 'LaTeX-mode-menu))
    
;;; Mode

(defun LaTeX-mode-initialization ()
  "Common initialization for LaTeX and SliTeX modes."
  (set-syntax-table LaTeX-mode-syntax-table)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'LaTeX-indent-line)
  (use-local-map LaTeX-mode-map)
  (easy-menu-add LaTeX-mode-menu LaTeX-mode-map)

  (or LaTeX-largest-level 
      (setq LaTeX-largest-level (LaTeX-section-level "section")))
  
  (setq TeX-header-end (concat (regexp-quote TeX-esc) "begin *"
			       TeX-grop "document" TeX-grcl))
  (setq TeX-trailer-start (concat (regexp-quote TeX-esc) "end *"
				  TeX-grop "document" TeX-grcl))

  (if (featurep 'tex-19)
      (require 'outline)		;Must be loaded first.
    (require 'outln-18))
  (make-local-variable 'outline-level)
  (setq outline-level 'LaTeX-outline-level)
  (make-local-variable 'outline-regexp)
  (setq outline-regexp (LaTeX-outline-regexp t))
  
  (setq paragraph-start
	(concat
	 "\\("
	 "^.*[^" (regexp-quote TeX-esc) "]%.*$\\|"
	 "^%.*$\\|"
	 "^[ \t]*$"
	 "\\|"
	 "^[ \t]*"
	 (regexp-quote TeX-esc)
	 "\\("
	 "begin\\|end\\|item\\|part\\|chapter\\|label\\|caption\\|"
	 "section\\|subsection\\|subsubsection\\|par\\|noindent\\|"
	 "paragraph\\|include\\|includeonly\\|"
	 "tableofcontents\\|appendix\\|"
	 "\\[\\|\\]"  ; display math delimitors
	 "\\)"
	 "\\|"
	 "^[ \t]*\\$\\$" ; display math delimitor
	 "\\)" ))
  (setq paragraph-separate
	(concat
	 "\\("
	 "^.*[^" (regexp-quote TeX-esc) "]%.*$\\|"
	 "^%.*$\\|"
	 (regexp-quote TeX-esc)
	 "par\\|"
	 "^[ \t]*$\\|"
	 "^[ \t]*"
	 (regexp-quote TeX-esc)
	 "\\("
	 "begin\\|end\\|part\\|chapter\\|label\\|caption\\|"
	 "section\\|subsection\\|subsubsection\\|"
	 "paragraph\\|include\\|includeonly\\|"
	 "tableofcontents\\|appendix\\|" (regexp-quote TeX-esc)
	 "\\)"
	 "\\)"))
  (setq selective-display t)
  
  (run-hooks 'text-mode-hook 'TeX-mode-hook 'LaTeX-mode-hook))

(provide 'ltx-misc)

;;; ltx-misc.el ends here
