;;; tex-auto.el - Automatically generate style information.

;; $Id: tex-auto.el,v 5.22 1993/12/15 21:42:18 amanda Exp $

;; Copyright (C) 1993 Per Abrahamsen 
;; 
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Code:

(require 'tex-init)
(require 'ltx-misc)

;;; Buffer Local Information:

(defun TeX-auto-find-files ()
  "Find TeX information in the current buffer."
  (TeX-auto-parse)
  
  (if TeX-auto-symbol
      (apply 'TeX-add-symbols TeX-auto-symbol))
  (if TeX-auto-environment
      (apply 'LaTeX-add-environments TeX-auto-environment))
  (if TeX-auto-label
      (apply 'LaTeX-add-labels TeX-auto-label))
  (if TeX-auto-bibitem
      (apply 'LaTeX-add-bibitems TeX-auto-bibitem))
  (if TeX-auto-bibliography
      (apply 'LaTeX-add-bibliographies TeX-auto-bibliography))

  (append TeX-auto-file TeX-auto-bibliography))

;;; Local Auto Directory:

(defvar TeX-auto-save nil
  "*Automatically save style information when saving the buffer.")
  
(defvar TeX-auto-untabify t
  "*Automatically untabify when saving the buffer.")

(defvar TeX-auto-update nil
  "Automatically update the lisp information for this TeX file.")
 (make-variable-buffer-local 'TeX-auto-update)

(defun TeX-auto-write ()
  "Save all relevant TeX information from the current buffer."
  (if (not TeX-auto-update)
      ()
    (if TeX-auto-untabify
	(untabify (point-min) (point-max)))
    (if (and TeX-auto-save TeX-auto-local)
        (let* ((file (concat TeX-auto-local
			     (if (string-match "/$" TeX-auto-local) "" "/")
			     (TeX-strip-extension nil TeX-all-extensions t)
			     ".el"))
	       (dir (file-name-directory file)))
	  ;; Create auto directory if possible.
	  (if (not (file-exists-p dir))
	      (condition-case name
		  (make-directory (substring dir 0 -1))
		(error nil)))
	  (if (file-writable-p file)
              (save-excursion
		(TeX-update-style)
                (TeX-auto-store file))
	    (message "Can't write style information.")))))
  ;; Continue with the other write file hooks.
  nil)

;;; Private Auto Directory:

(defvar TeX-macro-default (car TeX-macro-private)
  "*Default directory to search for TeX macros.")

(defvar TeX-auto-default TeX-auto-private
  "*Default directory to place automatically generated TeX information.")

;;;###autoload
(defun TeX-auto-generate (tex auto)
  "Generate style file for TEX and store it in AUTO.  
If TEX is a directory, generate style files for all files in the directory."
  (interactive (list (setq TeX-macro-default
                           (expand-file-name (read-file-name
                                              "TeX file or directory: "
                                              TeX-macro-default
                                              TeX-macro-default 'confirm)))
                     (setq TeX-auto-default
                           (expand-file-name (read-file-name
                                              "AUTO lisp directory: "
                                              TeX-auto-default
                                              TeX-auto-default 'confirm)))))
  (cond ((not (file-readable-p tex)))
	((string-match TeX-ignore-file tex))
        ((file-directory-p tex)
         (let ((files (directory-files tex))
               (default-directory (concat (if (string-match "^/" tex)
                                              ""
                                            default-directory)
                                          (if (string-match "/$" tex)
                                              tex
                                            (concat tex "/")))))
           (mapcar (function (lambda (file)
		      (if (or TeX-file-recurse
			      (not (file-directory-p file)))
			  (TeX-auto-generate file auto))))
                   files)))
        ((TeX-match-extension tex (append TeX-file-extensions
					  BibTeX-file-extensions))
         (save-excursion
           (set-buffer (find-file-noselect tex))
           (message "Parsing %s..." tex)
           (TeX-auto-store (concat auto
                                   (if (string-match "/$" auto) "" "/")
                                   (TeX-strip-extension tex
							TeX-all-extensions
							t)
                                   ".el"))
           (kill-buffer (current-buffer))
           (message "Parsing %s... done" tex)))))

;;; Global Auto Directory:

;;;###autoload
(defun TeX-auto-generate-global ()
  "Create global auto directory for global TeX macro definitions."
  (interactive)
  (mapcar (function (lambda (macro) (TeX-auto-generate macro TeX-auto-global)))
          TeX-macro-global))

;;; Store File:

(defun TeX-auto-store (file)
  "Extract information for auc tex from current buffer and store it in FILE."
  (TeX-auto-parse)
  
  (if (or TeX-auto-symbol
          TeX-auto-environment
          TeX-auto-file
          TeX-auto-label
          TeX-auto-bibitem
          TeX-auto-bibliography)
      (let ((style (TeX-strip-extension nil TeX-all-extensions t)))
        (TeX-unload-style style)
	(save-excursion
	  (set-buffer (find-file-noselect file))
	  (erase-buffer)
	  (insert "(TeX-add-style-hook \"" style "\"\n"
		  " (function\n"
		  "  (lambda ()")
	  (TeX-auto-insert "TeX-add-symbols" TeX-auto-symbol)
	  (TeX-auto-insert "LaTeX-add-environments" TeX-auto-environment)
	  (TeX-auto-insert "LaTeX-add-labels" TeX-auto-label)
	  (TeX-auto-insert "LaTeX-add-bibitems" TeX-auto-bibitem)
	  (TeX-auto-insert "LaTeX-add-bibliographies" TeX-auto-bibliography)
	  (TeX-auto-insert "TeX-run-style-hooks"
			   (append TeX-auto-file TeX-auto-bibliography))
	  (insert ")))\n\n")
	  (save-buffer 0)
	  (kill-buffer (current-buffer))))
    (if (file-exists-p (concat "file" "c"))
	(delete-file (concat "file" "c")))
    (if (file-exists-p file)
	(delete-file file))))

(defun TeX-auto-insert (name list)
  "Insert hook NAME in current buffer called with LIST."
  (if (null list)
      ()
    (insert "\n    (" name)
    (while list
      (newline-and-indent)
      (if (stringp (car list))
          (insert (prin1-to-string (car list)))
        (insert "'" (prin1-to-string (car list))))
      (setq list (cdr list)))
    (insert ")")
    (if (> (current-column) fill-column)
        (do-auto-fill))))

;;;###autoload
(defun BibTeX-auto-store ()
  "This function should be called from bibtex-mode-hook.
It will setup BibTeX to store keys in an auto file."
  ;; We want this to be early in the list, so we do not
  ;; add it before we enter BibTeX mode the first time. 
  (add-hook 'write-file-hooks 'TeX-auto-write)
  (make-local-variable 'TeX-auto-update)
  (setq TeX-auto-update 'BibTeX)
  (make-local-variable 'TeX-auto-untabify)
  (setq TeX-auto-untabify nil)
  (make-local-variable 'TeX-auto-regexp-list)
  (setq TeX-auto-regexp-list BibTeX-auto-regexp-list))

;;; Data:

(defvar TeX-auto-ignore
  '("csname" "filedate" "fileversion" "docdate" "next" "labelitemi"
    "labelitemii" "labelitemiii" "labelitemiv" "labelitemv"
    "labelenumi" "labelenumii" "labelenumiii" "labelenumiv"
    "labelenumv" "theenumi" "theenumii" "theenumiii" "theenumiv"
    "theenumv" "document" "par" "do" "expandafter")
  "List of symbols to ignore when scanning a TeX style file.")

(defun TeX-auto-add-regexp (regexp)
  "Add REGEXP to TeX-auto-regexp-list if not already a member."
  (if (symbolp TeX-auto-regexp-list)
      (setq TeX-auto-regexp-list (symbol-value TeX-auto-regexp-list)))
  (or (memq regexp TeX-auto-regexp-list)
      (setq TeX-auto-regexp-list (cons regexp TeX-auto-regexp-list))))

(defvar TeX-auto-empty-regexp-list
  '(("<IMPOSSIBLE>\\(\\'\\`\\)" 1 ignore))
  "List of regular expressions guaranteed to match nothing.")

(defvar LaTeX-auto-minimal-regexp-list 
  '(("\\\\document\\(style\\|class\\)\
\\(\\[\\(\\([^#\\\\\\.%]\\|%[^\n\r]*[\n\r]\\)+\\)\\]\\)?\
{\\([^#\\\\\\.\n\r]+\\)}"
     (3 5) TeX-auto-style))
  "Minimal list of regular expressions matching LaTeX macro definitions.")

(defvar LaTeX-auto-label-regexp-list
  '(("\\\\label{\\([^\n\r%\\{}]+\\)}" 1 TeX-auto-label))
  "List of regular expression matching LaTeX labels only.")

(defvar LaTeX-auto-regexp-list 
  (append
   '(("\\\\newcommand{?\\\\\\([a-zA-Z]+\\)}?\\[\\([0-9]+\\)\\]\
\\[\\([^\]\\\\\n\r]+\\)\\]"
      (1 2 3) TeX-auto-optional)
     ("\\\\newcommand{?\\\\\\([a-zA-Z]+\\)}?\\[\\([0-9]+\\)\\]"
      (1 2) TeX-auto-arguments)
     ("\\\\newcommand{?\\\\\\([a-zA-Z]+\\)}?" 1 TeX-auto-symbol)
     ("\\\\newenvironment{?\\([a-zA-Z]+\\)}?\\[\\([0-9]+\\)\\]"
      (1 2) TeX-auto-env-args)
     ("\\\\newenvironment{?\\([a-zA-Z]+\\)}?" 1 TeX-auto-environment)
     ("\\\\newtheorem{\\([a-zA-Z]+\\)}" 1 TeX-auto-environment)
     ("\\\\input{\\([^#%\\\\\\.\n\r]+\\)\\(\\.[^#%\\\\\\.\n\r]+\\)?}"
      1 TeX-auto-file)
     ("\\\\include{\\([^#%\\\\\\.\n\r]+\\)\\(\\.^#%\\\\\\.\n\r]+\\)?}"
      1 TeX-auto-file)
     ("\\\\usepackage\\(\\[[^\]\\\\]*\\]\\)?\
{\\(\\([^#\\\\\\.%]\\|%[^\n\r]*[\n\r]\\)+\\)}"
      (2) TeX-auto-style)
     ("\\\\bibitem{\\([a-zA-Z][^, \n\r\t%\"#'()={}]*\\)}" 1 TeX-auto-bibitem)
     ("\\\\bibitem\\[[^][\n\r]+\\]{\\([a-zA-Z][^, \n\r\t%\"#'()={}]*\\)}"
      1 TeX-auto-bibitem)
     ("\\\\bibliography{\\([^#\\\\\\.\n\r]+\\)}" 1 TeX-auto-bibliography))
   LaTeX-auto-label-regexp-list
   LaTeX-auto-minimal-regexp-list)
  "List of regular expression matching common LaTeX macro definitions.")

(defvar plain-TeX-auto-regexp-list
  '(("\\\\def\\\\\\([a-zA-Z]+\\)[^a-zA-Z@]" 1 TeX-auto-symbol-check)
    ("\\\\let\\\\\\([a-zA-Z]+\\)[^a-zA-Z@]" 1 TeX-auto-symbol-check)
    ("\\\\font\\\\\\([a-zA-Z]+\\)[^a-zA-Z@]" 1 TeX-auto-symbol)
    ("\\\\chardef\\\\\\([a-zA-Z]+\\)[^a-zA-Z@]" 1 TeX-auto-symbol)
    ("\\\\new\\(count|dimen|muskip|skip\\)\\\\\\([a-z]+\\)[^a-zA-Z@]"
     2 TeX-auto-symbol)
    ("\\\\newfont{?\\\\\\([a-zA-Z]+\\)}?" 1 TeX-auto-symbol)
    ("\\\\typein\\[\\\\\\([a-zA-Z]+\\)\\]" 1 TeX-auto-symbol)
    ("\\\\input +\\([^#%\\\\\\.\n\r]+\\)\\(\\.[^#%\\\\\\.\n\r]+\\)?"
     1 TeX-auto-file)
    ("\\\\mathchardef\\\\\\([a-zA-Z]+\\)[^a-zA-Z@]" 1 TeX-auto-symbol))
  "List of regular expression matching common LaTeX macro definitions.")

(defvar TeX-auto-full-regexp-list 
  (append LaTeX-auto-regexp-list plain-TeX-auto-regexp-list)
  "Full list of regular expression matching TeX macro definitions.")

(defvar BibTeX-auto-regexp-list
  '(("@[Ss][Tt][Rr][Ii][Nn][Gg]" 1 ignore)
    ("@[a-zA-Z]+[{(][ \t]*\\([a-zA-Z][^, \n\r\t%\"#'()={}]*\\)"
     1 TeX-auto-bibitem))
  "List of regexp-list expressions matching BibTeX items.")

;;; Variables:

(defvar TeX-auto-symbol nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-arguments nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-optional nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-env-args nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-environment nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-file nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-label nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-bibitem nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-bibliography nil
  "Local variable used when parsing TeX code.")

(defvar TeX-auto-style nil
"Local variable used when parsing TeX code.")

(defvar TeX-auto-end-symbol nil
  "Local variable used when parsing TeX code.")

;;; Parse File:

(defvar TeX-auto-prepare-hook nil
  "List of hooks to be called before parsing a TeX file.")

(defvar TeX-auto-cleanup-hook nil
  "List of hooks to be called after partsing a TeX file.")

(defvar TeX-auto-parse-length 999999
  "*Maximal length of TeX file that will be parsed.")
  (make-variable-buffer-local 'TeX-auto-parse-length)

(defun TeX-auto-parse ()
  "Parse TeX information in current buffer.

Call the functions in TeX-auto-prepare-hook before parsing, and the
functions in TeX-auto-cleanup-hook after parsing."

  (let ((case-fold-search nil)
	(regexp-list (if (symbolp TeX-auto-regexp-list)
			 (symbol-value TeX-auto-regexp-list)
		       TeX-auto-regexp-list)))

    ;; Prepare 
    (setq TeX-auto-symbol nil
	  TeX-auto-arguments nil
	  TeX-auto-optional nil
	  TeX-auto-env-args nil
	  TeX-auto-environment nil
	  TeX-auto-file nil
	  TeX-auto-label nil
	  TeX-auto-bibitem nil
	  TeX-auto-bibliography nil
	  TeX-auto-style nil
	  TeX-auto-end-symbol nil)
    (run-hooks 'TeX-auto-prepare-hook)
    
    ;; Parse
    (save-excursion
      (goto-char (min (point-max) TeX-auto-parse-length))
      ;; Extract the information.
      (let ((regexp (concat "\\("
			    (mapconcat 'car regexp-list "\\)\\|\\(")
			    "\\)")))
	(while (re-search-backward regexp nil t)
	  (if (TeX-in-comment)
	      ()
	    (let* ((entry (TeX-member nil regexp-list
				      (function (lambda (a b)
						  (looking-at (nth 0 b))))))
		   (symbol (nth 2 entry))
		   (match (nth 1 entry)))
	      (if (fboundp symbol)
		  (funcall symbol match)
		(set symbol (cons (if (listp match)
				      (mapcar 'TeX-match-buffer match)
				    (TeX-match-buffer match))
				  (symbol-value symbol)))))))))
    
    ;; Cleanup ignored symbols.
    
    ;; NOTE: This is O(N M) where it could be O(N log N + M log M) if we 
    ;; sorted the lists first.
    (while (member (car TeX-auto-symbol) TeX-auto-ignore)
      (setq TeX-auto-symbol (cdr TeX-auto-symbol)))
    (let ((list TeX-auto-symbol))
      (while (and list (cdr list))
	(if (member (car (cdr list)) TeX-auto-ignore)
	    (setcdr list (cdr (cdr list)))
	  (setq list (cdr list)))))
    
    ;; Cleanup BibTeX files
    (setq TeX-auto-bibliography
	  (apply 'append (mapcar (function (lambda (arg)
					     (TeX-split-string "," arg)))
				 TeX-auto-bibliography)))
    
    ;; Cleanup document styles and packages
    (if (null TeX-auto-style)
	()
      (while TeX-auto-style
	(let* ((entry (car TeX-auto-style))
	       (options (nth 0 entry))
	       (style (nth 1 entry)))

	  ;; Next document style.
	  (setq TeX-auto-style (cdr TeX-auto-style))
	  
	  ;; Get the options.
	  (setq options (TeX-split-string 
			 "\\([ \t\r\n]\\|%[^\n\r]*[\n\r]\\|,\\)+"
			 options))
	  
	  ;; Strip empty options.
	  (if (string-equal (car options) "")
	      (setq options (cdr options)))
	  (let ((index options))
	    (while (cdr-safe index)
	      (if (string-equal (car (cdr index)) "")
		  (setcdr index (cdr (cdr index)))
		(setq index (cdr index)))))

	  ;; Add them, to the style list.
	  (setq TeX-auto-file (append options TeX-auto-file))

	  ;; The second argument if present is a normal style file.
	  (if (null style)
	      ()
	    (setq TeX-auto-file (cons style TeX-auto-file))
	  
	    ;; And a special "art10" style file combining style and size.
	    (setq TeX-auto-file
		  (cons (concat 
			 (cond ((string-equal "article" style)
				"art")
			       ((string-equal "book" style)
				"bk")
			       ((string-equal "report" style)
				"rep")
			       ((string-equal "jarticle" style)
				"jart")
			       ((string-equal "jbook" style)
				"jbk")
			       ((string-equal "jreport" style)
				"jrep")
			       ((string-equal "j-article" style)
				"j-art")
			       ((string-equal "j-book" style)
				"j-bk")
			       ((string-equal "j-report" style )
				"j-rep")
			       (t style))
			 (cond ((member "11pt" options)
				"11")
			       ((member "12pt" options)
				"12")
			       (t
				"10")))
			TeX-auto-file))))))
    
    ;; Cleanup optional arguments
    (mapcar (function (lambda (entry)
			(setq TeX-auto-symbol
			      (cons (list (nth 0 entry)
					  (string-to-int (nth 1 entry)))
				    TeX-auto-symbol))))
	    TeX-auto-arguments)

    ;; Cleanup default optional arguments
    (mapcar (function (lambda (entry)
			(setq TeX-auto-symbol
			      (cons (list (nth 0 entry)
					  (vector "argument")
					  (1- (string-to-int (nth 1 entry))))
				    TeX-auto-symbol))))
	    TeX-auto-optional)

    (mapcar (function (lambda (entry)
			(setq TeX-auto-environment
			      (cons (list (nth 0 entry)
					  (string-to-int (nth 1 entry)))
				    TeX-auto-environment))))
	    TeX-auto-env-args)
    
    ;; Cleanup use of def to add environments
    ;; NOTE: This use a O(N^2) algorithm, while a O(N log N) algorthm is
    ;; possible.
    (mapcar (function (lambda (symbol)
			(if (not (TeX-member symbol TeX-auto-symbol 'equal))
			    ;; No matching symbol, insert in list
			    (setq TeX-auto-symbol
				  (cons (concat "end" symbol) TeX-auto-symbol))
			  ;; Matching symbol found, remove from list
			  (if (equal (car TeX-auto-symbol) symbol)
			      ;; Is it the first symbol?
			      (setq TeX-auto-symbol (cdr TeX-auto-symbol))
			    ;; Nope!  Travel the list
			    (let ((list TeX-auto-symbol))
			      (while (consp (cdr list))
				;; Until we find it.
				(if (equal (car (cdr list)) symbol)
				    ;; Then remove it.
				    (setcdr list (cdr (cdr list))))
				(setq list (cdr list)))))
			  ;; and add the symbol as an environment.
			  (setq TeX-auto-environment
				(cons symbol TeX-auto-environment)))))
	    TeX-auto-end-symbol)
    
    (run-hooks 'TeX-auto-cleanup-hook)))

(defun TeX-auto-symbol-check (symbol)
  "Add MATCH to TeX-auto-symbol.
Check for potential LaTeX environments."
  (let ((symbol (if (listp match)
                    (mapcar 'TeX-match-buffer match)
                  (TeX-match-buffer match))))
    (if (and (stringp symbol)
             (string-match "^end\\(.+\\)$" symbol))
        (setq TeX-auto-end-symbol
              (cons (substring symbol (match-beginning 1) (match-end 1))
                    TeX-auto-end-symbol))
      (setq TeX-auto-symbol (cons symbol TeX-auto-symbol)))))

(provide 'tex-auto)

;;; tex-auto.el ends here
