;;; tex-cpl.el - completion for TeX/LaTeX modes

;; $Id: tex-cpl.el,v 1.20 1993/11/18 20:10:28 amanda Exp $

;;; Code:

(require 'tex-init)

;;; Interactive Functions

(defun TeX-complete-symbol ()
  "Perform completion on TeX/LaTeX symbol preceding point.
That symbol is compared against the symbols that exist
and any additional characters determined by what is there
are inserted."
  (interactive "*")
  (let* ((end (point))
	 (TeX-command-p nil)
         (beg (save-excursion
		(if (= (preceding-char) (string-to-char TeX-esc)) 
		    (progn					  
		      (setq TeX-command-p t)			  
		      (point))
		  (backward-sexp 1)
		  (while (= (following-char) (string-to-char TeX-esc))
		    (forward-char 1)
		    (setq TeX-command-p t))
		  (point))))
         (pattern (buffer-substring beg end))
         (completion (try-completion pattern (TeX-symbol-list) nil)))
    (if TeX-command-p
	(cond ((eq completion t))
	      ((null completion)
	       (error "Can't find completion for \"%s%s\"" TeX-esc pattern))
	      ((not (string= pattern completion))
	       (delete-region beg end)
	       (insert completion))
	      (t
	       (message "Making completion list...")
	       (let ((list (all-completions pattern (TeX-symbol-list) nil)))
		 (with-output-to-temp-buffer "*Completions*"
		   (display-completion-list list)))
	       (message "Making completion list...done")))
      (funcall TeX-complete-word))))

(defvar TeX-default-macro "ref"
  "*The default macro when creating new ones with TeX-insert-macro.")

 (make-variable-buffer-local 'TeX-default-macro)

(defun TeX-insert-macro (symbol)
  "Insert TeX macro with completion.

AUC TeX knows of some macros, and may query for extra arguments."
  (interactive (list (completing-read (concat "Macro (default "
					      TeX-default-macro
					      "): " 
					      TeX-esc)
				      (TeX-symbol-list))))
  (cond ((string-equal symbol "")
	 (setq symbol TeX-default-macro))
	((interactive-p)
	 (setq TeX-default-macro symbol)))
  
  (let ((entry (assoc symbol (TeX-symbol-list))))
    (if (consp entry)
	(TeX-parse-macro symbol (cdr entry))
      (insert TeX-esc symbol TeX-grop TeX-grcl))))

(defvar TeX-electric-macro-map nil)

(if TeX-electric-macro-map
    ()
  (setq TeX-electric-macro-map (copy-keymap minibuffer-local-completion-map))
  (define-key TeX-electric-macro-map " " 'minibuffer-complete-and-exit))

(defun TeX-electric-macro ()
  "Insert TeX macro with completion.

AUC TeX knows of some macros, and may query for extra arguments.
Space will complete and exit."
  (interactive)
  (let ((minibuffer-local-completion-map TeX-electric-macro-map))
    (call-interactively 'TeX-insert-macro)))

;;; Parsing Macro Arguments

(defun TeX-parse-macro (symbol args)
  "How to parse TeX macros which takes one or more arguments.

First argument is the name of the macro.  

If called with no additional arguments, insert macro with point inside
braces.  Otherwise, each argument of this function should match an
argument to the TeX macro.  What is done depend on the argument type.

string: Use the string as a prompt to prompt for the argument.

number: Insert that many braces, leave point inside the first.

nil: Insert empty braces.

t: Insert empty braces, leave point between the braces.

other symbols: Call the symbol as a function.  You can define your
own hook, or use one of the predefined argument hooks.

list: If the car is a string, insert it as a prompt and the next
element as initial input.  Otherwise, call the car of the list with
the remaining elements as arguments. 

vector: Optional argument.  If it has more than one element, parse it
as a list, otherwise parse the only element as above.  Use square
brackets instead of curly braces, and is not inserted on empty user
input.

Existing hooks:

Unless noted otherwise, these hooks take at one optional argument, a
string to be used as a prompt instead of the default.

TeX-arg-conditional: Implement if EXPR THEN ELSE.  If EXPR
evaluate to true, parse THEN as an argument list, else parse ELSE as
an argument list.

TeX-arg-literal: Insert its arguments into the buffer.  Used
for specifying extra syntax for a macro.

TeX-arg-free: Parse its arguments but use no braces when
they are inserted.

TeX-arg-eval:  Evaluate arguments and insert the result in
the buffer.

TeX-arg-file: Prompt for a filename.

TeX-arg-input-file: Prompt for a tex or sty filename, and use it
without the extension.  Run the style hooks for the file.

TeX-arg-label: Prompt for a label completing with known
labels.

TeX-arg-macro: Prompt for a TeX macro with completion.

TeX-arg-environment: Prompt for a LaTeX environment with
completion.

TeX-arg-cite: Prompt for a BibTeX citation.

TeX-arg-counter: Prompt for a LaTeX counter.

TeX-arg-savebox: Prompt for a LaTeX savebox.

TeX-arg-define-label: Prompt for a label completing with known
labels.  Add label to list of defined labels.

TeX-arg-define-macro: Prompt for a TeX macro with
completion.  Add macro to list of defined macros.

TeX-arg-define-environment: Prompt for a LaTeX environment
with completion.  Add environment to list of defined environments.

TeX-arg-define-cite: Prompt for a BibTeX citation.

TeX-arg-define-counter: Prompt for a LaTeX counter.

TeX-arg-define-savebox: Prompt for a LaTeX savebox.

TeX-arg-corner: Prompt for a LaTeX side or corner position
with completion.

TeX-arg-lr: Prompt for a LaTeX side with completion.

TeX-arg-tb: Prompt for a LaTeX side with completion.

TeX-arg-pagestyle: Prompt for a LaTeX pagestyle with
completion. 

TeX-arg-verb: Prompt for delimiter and text.  Arguments are
ignored.  

TeX-arg-pair: Insert a pair of number, use arguments for
prompt. The numbers are surounded by parenthesizes and separated with
a comma.

TeX-arg-size: Insert width and height as a pair.  No
arguments.

TeX-arg-coordinate: Insert x and y coordinate as a pair.  No
arguments.

If you add new hooks, you can assume that point is placed directly
after the previous argument, or after the macro name if this is the
first argument.  Please leave point located efter the argument you are
inserting.  If you want point to be located somewhere else after all
hooks have been processed, set the value of `exit-mark'.  It will
point nowhere, until the argument hook set it."

  (insert TeX-esc symbol)
  (let ((exit-mark (make-marker))
	(position (point)))
    (TeX-parse-arguments args)
    (cond ((marker-position exit-mark)
	   (goto-char (marker-position exit-mark))
	   (set-marker exit-mark nil))
	  ((equal position (point))
	   (insert TeX-grop TeX-grcl)))))

(defun TeX-parse-arguments (args)
  "Parse TeX macro arguments.

See TeX-parse-macro for details."
  (let ((last-optional-rejected nil))
    (while args
      (if (vectorp (car args))
	  (if last-optional-rejected
	      ()
	    (let ((< LaTeX-optop)
		  (> LaTeX-optcl))
	      (TeX-parse-argument t (if (equal (length (car args)) 1)
					(aref (car args) 0)
				      (append (car args) nil)))))
	(let ((< TeX-grop)
	      (> TeX-grcl))
	  (setq last-optional-rejected nil)
	  (TeX-parse-argument nil (car args))))
      (setq args (cdr args)))))

(defun TeX-parse-argument (optional arg)
  "Depending on OPTIONAL, insert TeX macro argument ARG in curly braces.
If OPTIONAL is set, only insert if there is anything to insert, and
then use scare brackets.

See TeX-parse-macro for details."
  
  (cond ((stringp arg)
	 (TeX-arg-string optional arg))
	((numberp arg)
	 (if (< arg 1)
	     ()
	   (TeX-parse-argument optional t)
	   (while (> arg 1)
	     (TeX-parse-argument optional nil)
	     (setq arg (- arg 1)))))
	((null arg)
	 (insert < >))
	((eq arg t)
	 (insert  < )
	 (set-marker exit-mark (point))
	 (insert >))
	((symbolp arg)
	 (funcall arg optional))
	((listp arg)
	 (let ((head (car arg))
	       (tail (cdr arg)))
	   (cond ((stringp head)
		  (apply 'TeX-arg-string optional arg))
		 ((symbolp head)
		  (apply head optional tail))
		 (t (error "Unknown list argument type %s."
			   (prin1-to-string head))))))
	(t (error "Unknown argument type %s." (prin1-to-string arg)))))

(defun TeX-argument-insert (name optional &optional prefix)
  "Insert NAME surrounded by curly braces.

If OPTIONAL, only insert it if not empty, and then use scuare brackets."
  (if (and optional (string-equal name ""))
      (setq last-optional-rejected t)
    (insert <)
    (if prefix
	(insert prefix))
    (if (and (string-equal name "")
	     (null (marker-position exit-mark)))
	(set-marker exit-mark (point))
      (insert name))
    (insert >)))

(defun TeX-argument-prompt (optional prompt default &optional complete)
  "Return a argument prompt.

If OPTIONAL is not nil then the prompt will start with ``(Optional) ''.

PROMPT will be used if not nil, otherwise use DEFAULT.

Unless optional argument COMPLETE is non-nil, ``: '' will be appended."
  (concat (if optional "(Optional) " "")
	  (if prompt prompt default)
	  (if complete "" ": ")))

;;; Argument Hooks

(defun TeX-arg-conditional (optional expr then else)
  "Implement if EXPR THEN ELSE.

If EXPR evaluate to true, parse THEN as an argument list, else parse
ELSE as an argument list."
  (TeX-parse-arguments (if (eval expr) then else)))

(defun TeX-arg-free (optional &optional &rest args)
  "Parse its arguments but use no braces when they are inserted."
  (let ((< "")
	(> ""))
    (if (equal (length args) 1)
	(TeX-parse-argument optional (car args))
      (TeX-parse-argument optional args))))

(defun TeX-arg-literal (optional &optional &rest args)
  "Insert its arguments into the buffer.  
Used for specifying extra syntax for a macro."
  (apply 'insert args))

(defun TeX-arg-string (optional &optional prompt input)
  "Prompt for a string."
  (TeX-argument-insert
   (read-string (TeX-argument-prompt optional prompt "Text") input)
   optional))

(defun TeX-arg-eval (optional &rest args)
  "Evaluate args and insert value in buffer."
  (insert (eval args)))

(defun TeX-arg-label (optional &optional prompt definition)
  "Prompt for a label completing with known labels."
  (let ((label (completing-read (TeX-argument-prompt optional prompt "Key")
				(LaTeX-label-list))))
    (if (and definition (not (string-equal "" label)))
	(LaTeX-add-labels label))
    (TeX-argument-insert label optional optional)))

(defun TeX-arg-macro (optional &optional prompt definition)
  "Prompt for a TeX macro with completion."
  (let ((macro (completing-read (TeX-argument-prompt optional prompt
						     (concat "Macro: "
							     TeX-esc)
						     t)
				(TeX-symbol-list))))
    (if (and definition (not (string-equal "" macro)))
	(TeX-add-symbols macro))
    (TeX-argument-insert macro optional TeX-esc)))

(defun TeX-arg-environment (optional &optional prompt definition)
  "Prompt for a LaTeX environment with completion."
  (let ((environment (completing-read (TeX-argument-prompt optional prompt
							   "Environment" t)
				      (TeX-symbol-list))))
    (if (and definition (not (string-equal "" environment)))
	(LaTeX-add-environments environment))

    (TeX-argument-insert environment optional)))

(defun TeX-arg-cite (optional &optional prompt definition)
  "Prompt for a BibTeX citation with completion."
  (let ((bibitem (completing-read (TeX-argument-prompt optional prompt "Key")
				(LaTeX-bibitem-list))))
    (if (and definition (not (string-equal "" bibitem)))
	(LaTeX-add-bibitems bibitem))
    (TeX-argument-insert bibitem optional optional)))

(defun TeX-arg-counter (optional &optional prompt definition)
  "Prompt for a LaTeX counter."
  ;; Completion not implemented yet.
  (TeX-argument-insert
   (read-string (TeX-argument-prompt optional prompt "Counter"))
   optional))

(defun TeX-arg-savebox (optional &optional prompt definition)
  "Prompt for a LaTeX savebox."
  ;; Completion not implemented yet.
  (TeX-argument-insert
   (read-string (TeX-argument-prompt optional prompt
				     (concat "Savebox: " TeX-esc)
				     t))
   optional TeX-esc))

(defun TeX-arg-file (optional &optional prompt)
  "Prompt for a filename in the current directory."
  (TeX-argument-insert (read-file-name (TeX-argument-prompt optional
							    prompt "File")
				       "" "" nil)
		       optional))

(defun TeX-arg-define-label (optional &optional prompt)
  "Prompt for a label completing with known labels."
  (TeX-arg-label optional prompt t))

(defun TeX-arg-define-macro (optional &optional prompt)
  "Prompt for a TeX macro with completion."
  (TeX-arg-macro optional prompt t))

(defun TeX-arg-define-environment (optional &optional prompt)
  "Prompt for a LaTeX environment with completion."
  (TeX-arg-environment optional prompt t))

(defun TeX-arg-define-cite (optional &optional prompt)
  "Prompt for a BibTeX citation."
  (TeX-arg-cite optional prompt t))

(defun TeX-arg-define-counter (optional &optional prompt)
  "Prompt for a LaTeX counter."
  (TeX-arg-counter optional prompt t))

(defun TeX-arg-define-savebox (optional &optional prompt)
  "Prompt for a LaTeX savebox."
  (TeX-arg-savebox optional prompt t))

(defvar LaTeX-style-list '(("book")
			   ("article")
			   ("letter")
			   ("slides")
			   ("report"))
  "*List of document styles.")

  (make-variable-buffer-local 'LaTeX-style-list)

(defun TeX-arg-document (optional &optional ignore)
  "Insert arguments to documentstyle and documentclass."
  (let ((style (completing-read 
		(concat "Document style: (default " LaTeX-default-style ") ")
		LaTeX-style-list))
	(options (read-string "Options: "
			      (if (stringp LaTeX-default-options)
				  LaTeX-default-options
				(mapconcat 'identity
					   LaTeX-default-options
					   ",")))))
    (if (zerop (length style))
	(setq style LaTeX-default-style))
    (if (not (zerop (length options)))
	(insert LaTeX-optop options LaTeX-optcl))
    (insert TeX-grop style TeX-grcl))

  ;; remove old information
  (TeX-remove-style)

  ;; defined in individual style hooks
  (TeX-update-style))

(defun TeX-arg-input-file (optional &optional prompt local)
  "Prompt for a tex or sty file.
First optional argument is the promt, the second is a flag.  If the
flag is set, only complete with local files."
  (message "Searching for files...")
  (let ((file (if TeX-check-path
		  (completing-read
		   (TeX-argument-prompt optional prompt "File")
		   (mapcar 'list (TeX-search-files TeX-check-path
						   TeX-file-extensions t t)))
		(read-file-name
		 (TeX-argument-prompt optional prompt "File")))))
    (if (null file)
	(setq file ""))
    (if (not (string-equal "" file))
	(TeX-run-style-hooks file))
    (TeX-argument-insert file optional)))

(defvar BibTeX-global-style-files nil
  "*Association list of BibTeX style files.

If nil, AUC TeX will search for them.")

(defun TeX-arg-bibstyle (optional &optional prompt)
  "Prompt for a BibTeX style file."
  (message "Searching for BibTeX styles...")
  (or BibTeX-global-style-files
      (setq BibTeX-global-style-files
	    (mapcar 'list
		    (TeX-search-files (append TeX-macro-private
					      TeX-macro-global)
				      BibTeX-style-extensions t t))))

  (TeX-argument-insert
   (completing-read (TeX-argument-prompt optional prompt "BibTeX style")
		    (append (mapcar 'list
				    (TeX-search-files '(".")
						      BibTeX-style-extensions
						      t t))
			    BibTeX-global-style-files))
   optional))

(defvar BibTeX-global-files nil
  "*Association list of BibTeX files.

If nil, AUC TeX will search for them.")

(defun TeX-arg-bibligraphy (optional &optional prompt)
  "Prompt for a BibTeX database file."
  (message "Searching for BibTeX files...")
  (or BibTeX-global-files
      (setq BibTeX-global-files
	    (mapcar 'list (TeX-search-files nil BibTeX-file-extensions t t))))
  
  (let ((file (completing-read
	       (TeX-argument-prompt optional prompt "BibTeX file")
	       (append (mapcar 'list
				    (TeX-search-files '(".")
						      BibTeX-file-extensions
						      t t))
		      BibTeX-global-files))))
    (TeX-argument-insert file optional)
    (if (not (string-equal file ""))
	(LaTeX-add-bibliographies (TeX-split-string "," file)))))

(defun TeX-arg-corner (optional &optional prompt)
  "Prompt for a LaTeX side or corner position with completion."
  (TeX-argument-insert
   (completing-read (TeX-argument-prompt optional prompt "Position")
		    '(("") ("l") ("r") ("t") ("b") ("tl") ("tr") ("bl") ("br"))
		    nil t)
   optional))

(defun TeX-arg-lr (optional &optional prompt)
  "Prompt for a LaTeX side with completion."
  (TeX-argument-insert
   (completing-read (TeX-argument-prompt optional prompt "Position")
		    '(("") ("l") ("r"))
		    nil t)
   optional))

(defun TeX-arg-tb (optional &optional prompt)
  "Prompt for a LaTeX side with completion."
  (TeX-argument-insert
   (completing-read (TeX-argument-prompt optional prompt "Position")
		    '(("") ("t") ("b"))
		    nil t)
   optional))

(defun TeX-arg-pagestyle (optional &optional prompt)
  "Prompt for a LaTeX pagestyle with completion."
  (TeX-argument-insert
   (completing-read (TeX-argument-prompt optional prompt "Pagestyle")
		    '(("plain") ("empty") ("headings") ("myheadings")))
   optional))

(defun TeX-arg-verb (optional &optional ignore)
  "Prompt for delimiter and text."
  (let ((del (read-quoted-char "Delimiter: "))
	(text (read-from-minibuffer "Text: ")))
    (insert del text del)))

(defun TeX-arg-pair (optional first second)
  "Insert a pair of number, prompted by FIRST and SECOND.

The numbers are surounded by parenthesizes and separated with a
comma."
  (insert "(" (read-string (concat first  ": ")) ","
	      (read-string (concat second ": ")) ")"))

(defun TeX-arg-size (optional)
  "Insert width and height as a pair."
  (TeX-arg-pair optional "Width" "Height"))

(defun TeX-arg-coordinate (optional)
  "Insert x and y coordinate as a pair."
 (TeX-arg-pair optional "X position" "Y position"))

(provide 'tex-cpl)

;;; tex-cpl.el ends here

