;;; tex-init.el - Support for TeX and LaTeX documents.

;; $Id: tex-init.el,v 5.35 1994/01/11 23:01:07 amanda Exp $

;; Copyright (C) 1985, 1986 Free Software Foundation, Inc.
;; Copyright (C) 1987 Lars Peter Fischer
;; Copyright (C) 1991 Kresten Krab Thorup
;; Copyright (C) 1993 Per Abrahamsen 
;; 
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Code:

(require 'auc-ver)
(require 'tex-site)
(require 'easymenu)

;;; Import
;;
;; The following are autoloads for different modules, as well as
;; variables that has to be made buffer local and functions we really
;; need now.

(defvar no-doc
  "This function is part of AUC TeX, but has not yet been loaded.
Full documentation will be available after autoloading the function."
  "Documentation for autoload functions.")

(autoload 'LaTeX-bibliography-list "ltx-misc" no-doc nil)
(autoload 'LaTeX-section "ltx-sec" no-doc t)
(autoload 'TeX-help-error "tex-buf" no-doc nil)
(autoload 'TeX-complete-symbol "tex-cpl" no-doc t)
(autoload 'TeX-insert-macro "tex-cpl" no-doc t)
(autoload 'TeX-electric-macro "tex-cpl" no-doc t)
(autoload 'TeX-auto-write "tex-auto" no-doc nil)
(autoload 'TeX-auto-find-files "tex-auto" no-doc nil)
(autoload 'TeX-auto-store-self "tex-auto" no-doc t)

(autoload 'TeX-comment-region "tex-init" no-doc t)
(autoload 'TeX-comment-paragraph "tex-init" no-doc t)
(autoload 'TeX-comment-out-region "tex-init" no-doc t)
(autoload 'TeX-comment-out-paragraph "tex-init" no-doc t)
(autoload 'TeX-un-comment "tex-init" no-doc t)
(autoload 'TeX-un-comment-region "tex-init" no-doc t)

;;; Buffer

(defvar TeX-command-BibTeX "BibTeX"
  "*The name of the BibTeX entry in TeX-command-list.")
  (make-variable-buffer-local 'TeX-command-BibTeX)

(defvar TeX-command-Show "View"
  "*The default command to show (view or print) a TeX file.
Must be the car of an entry in TeX-command-list.")
  (make-variable-buffer-local 'TeX-command-Show)

(defvar TeX-command-Print "Print"
  "The name of the Print entry in TeX-command-Print.")

(defvar TeX-command-Queue "Queue"
  "The name of the Queue entry in TeX-command-Queue.")

(autoload 'TeX-home-buffer "tex-buf" no-doc t)
(autoload 'TeX-command-region "tex-buf" no-doc t)
(autoload 'TeX-command-master "tex-buf" no-doc t)
(autoload 'TeX-command "tex-buf" no-doc nil)
(autoload 'TeX-kill-job "tex-buf" no-doc t)
(autoload 'TeX-recenter-output-buffer "tex-buf" no-doc t)
(autoload 'TeX-next-error "tex-buf" no-doc t)
(autoload 'TeX-toggle-debug-boxes "tex-buf" no-doc t)

(defvar TeX-trailer-start nil
  "Regular expression delimiting start of trailer in a TeX file.")

 (make-variable-buffer-local 'TeX-trailer-start)

(defvar TeX-header-end nil
  "Regular expression delimiting end of header in a TeX file.")

 (make-variable-buffer-local 'TeX-header-end)

(defvar TeX-command-default nil
  "The default command for TeX-command in the current major mode.")

 (make-variable-buffer-local 'TeX-command-default)


;;; Master File

(defvar TeX-one-master "\\.tex$"
  "*Regular expression matching ordinary TeX files.

You should set this variable to match the name of all files, where
automatically adding a file variable with the name of the master file
is a good idea.  When AUC TeX add the name of the master file as a
file variable, it does not need to ask next time you edit the file.  

If you dislike AUC TeX automatically modifying your files, you can set
this variable to \"<none>\".")

(defun TeX-master-file (&optional extension)
  "Return the name of the master file for the current document.

If optional argument EXTENSION is non-nil, add that file extension to
the name.  Special value `t' means use `TeX-default-extension'.

Currently is will check for the presence of a ``Master:'' line in
the beginning of the file, but that feature will be phased out."
  (if (eq extension t)
      (setq extension TeX-default-extension))
  (let ((my-name (if (buffer-file-name)
                     (TeX-strip-extension nil (list TeX-default-extension) t)
                   "<none>")))
    (save-excursion
      (goto-char (point-min))
      (cond
       ;; Special value 't means it is own master (a free file).
       ((equal TeX-master my-name) 
	(setq TeX-master t))

       ;; For files shared between many documents.
       ((eq 'shared TeX-master)
        (setq TeX-master
              (TeX-strip-extension
               (read-file-name "Master file: (default this file) "
                               nil "///")
               (list TeX-default-extension)
               t))
        (if (or (string-equal TeX-master "///")
                (string-equal TeX-master ""))
            (setq TeX-master t)))
       
       ;; We might already know the name.
       (TeX-master)
       
       ;; Support the ``Master:'' line (under protest!)
       ((re-search-forward
         "^%% *[Mm]aster:?[ \t]*\\([^ \t\n]+\\)" 500 t)
        (setq TeX-master
              (TeX-strip-extension (TeX-match-buffer 1)
                                   (list TeX-default-extension)))
        (if TeX-convert-master
            (progn
              (beginning-of-line)
              (kill-line 1)
              (TeX-add-local-master))))
       
       ;; Is this a master file?
       ((re-search-forward TeX-header-end 10000 t)
        (setq TeX-master my-name))
       
       ;; Ask the user (but add it as a local variable).
       (t
        (setq TeX-master
              (TeX-strip-extension
	       (condition-case name
		   (read-file-name "Master file: (default this file) "
				   nil "<default>")
		 (quit "<quit>"))
               (list TeX-default-extension)
               t))
        (cond ((string-equal TeX-master "<quit>")
	       (setq TeX-master t))
	      ((or (string-equal TeX-master "<default>")
		   (string-equal TeX-master ""))
	       (setq TeX-master t)
	       (TeX-add-local-master))
	      (t
	       (TeX-add-local-master))))))
  
    (let ((name (cond ((eq TeX-master t)
		       my-name)
		      ;; For files using the "tex" extension,
		      ;; TeX-master will be the filename without the
		      ;; extension.  We assume in the following that
		      ;; such files contain no additional dots.
		      ((and (string-match "^\\([^.].*\\)\\.[^.]*$" TeX-master)
			    (equal extension TeX-default-extension))
		       (substring TeX-master
				  (match-beginning 1) (match-end 1)))
		      (t
		       TeX-master))))
      (if extension
	  (concat name "." extension)
	name))))

(defvar TeX-master t
  "*The master file associated with the current buffer.
If the file being edited is actually included from another file, you
can tell AUC TeX the name of the master file by setting this variable.
If there are multiple levels of nesting, specify the top level file. 

If this variable is nil, AUC TeX will query you for the name.

If the variable is t, AUC TeX will assume the file is a master file
itself.

If the variable is 'shared, AUC TeX will query for the name, but not
change the file.  

It is suggested that you use the File Variables (see the info node in
the Emacs manual) to set this variable permanently for each file.")

 (make-variable-buffer-local 'TeX-master)

(defvar TeX-convert-master t
  "*If not nil, automatically convert ``Master:'' lines to file variables.
This will be done when AUC TeX first try to use the master file.")

(defun TeX-add-local-master ()
  "Add local variable for TeX-master."

  (if (and (buffer-file-name)
           (string-match TeX-one-master
                         (file-name-nondirectory (buffer-file-name)))
           (not buffer-read-only))
      (progn
        (goto-char (point-max))
        (if (re-search-backward (concat "^\\([^\n]+\\)Local " "Variables:")
                                (- (point-max) 3000) t)
            (let ((prefix (TeX-match-buffer 1)))
              (re-search-forward (regexp-quote (concat prefix
                                                        "End:")))
              (beginning-of-line 1)
              (insert prefix "TeX-master: " (prin1-to-string TeX-master) "\n"))
          (forward-line 1)
          (insert "% Local " "Variables: \n"
                  "% mode: " (substring (symbol-name major-mode) 0 -5)
		  "\n"
                  "% TeX-master: " (prin1-to-string TeX-master) "\n"
                  "% End: \n")))))

;;; Style Paths

(defvar TeX-format-directory (concat TeX-lisp-directory "format/")
  "*Directory containing information about TeX format packages.
Must end with a slash.")

(defvar TeX-auto-global (concat TeX-lisp-directory "auto/")
  "*Directory containing automatically generated information.
Must end with a slash.

For storing automatic extracted information about the TeX macros
shared by all users of a site.")  

(defvar TeX-style-global (concat TeX-lisp-directory "style/")
  "*Directory containing hand generated TeX information.
Must end with a slash.

These correspond to TeX macros shared by all users of a site.")

(defvar TeX-auto-local "auto/"
  "*Directory containing automatically generated TeX information.
Must end with a slash.

This correspond to TeX macros found in the current directory.")

(defvar TeX-style-local "style/"
  "*Directory containing hand generated TeX information.
Must end with a slash.

These correspond to TeX macros found in the current directory.")

(defvar TeX-macro-private (list (expand-file-name "~/TeX/"))
  "*Directories where you store your personal TeX macros.
Each must end with a slash.")

(defvar TeX-auto-private (concat (car TeX-macro-private) TeX-auto-local)
  "*Directory containing automatically generated information.
Must end with a slash.

These correspond to the personal TeX macros.")

(defvar TeX-style-private (concat (car TeX-macro-private) TeX-style-local)
  "*Directory containing hand generated information.
Must end with a slash.

These correspond to the personal TeX macros.")

(defvar TeX-style-path
  (let ((path))
    (mapcar (function (lambda (file) (if file (setq path (cons file path)))))
	    (list TeX-auto-global
		  TeX-style-global
		  TeX-auto-private
		  TeX-style-private
		  TeX-auto-local
		  TeX-style-local))
    path)
  "*List of directories to search for AUC TeX style files.")

(defvar TeX-check-path (append (list "./") TeX-macro-private TeX-macro-global)
  "*Directory path to search for dependencies.

If nil, just check the current file.
Used when checking if any files have changed.")

;;; Style Files

(defvar TeX-style-hook-list nil
  "List of TeX style hooks currently loaded.

Each entry is a list where the first element is the name of the style,
and the remaining elements are hooks to be run when that style is
active.")

(defvar TeX-byte-compile t
  "*Not nil means try to byte compile auto files before loading.")

(defun TeX-load-style (style)
  "Search for and load each definition for style in TeX-style-path."
  (if (assoc style TeX-style-hook-list)
      ;; We already found it
      ()
    ;; Insert empty list to mark the fact that we have searched.
    (setq TeX-style-hook-list (cons (list style) TeX-style-hook-list))
    ;; Now check each element of the path
    (mapcar (function
             (lambda (name)
               (let* ((name (if (string-match "/$" name)
                                name
                              (concat name "/")))
                      (el (concat name style ".el"))
                      (elc (concat name style ".elc")))
                 (cond ((and (null TeX-byte-compile)
			     (file-readable-p el))
			(load-file el))
		       ((file-newer-than-file-p el elc)
			(if (not (file-writable-p elc))
			    (load-file el)
			  (byte-compile-file el)
			  (load-file elc)))
		       ((file-readable-p elc)
			(load-file elc))
		       ((file-readable-p el)
			(load-file el))))))
            TeX-style-path)))

(defun TeX-add-style-hook (style hook)
  "Give STYLE yet another HOOK to run."
  (let ((entry (assoc style TeX-style-hook-list)))
    (if (null entry)
        (setq TeX-style-hook-list (cons (list style hook) TeX-style-hook-list))
      (setcdr entry (cons hook (cdr entry))))))

(defun TeX-unload-style (style)
  "Forget that we once loaded STYLE."
  (cond ((null (assoc style TeX-style-hook-list)))
        ((equal (car (car TeX-style-hook-list)) style)
         (setq TeX-style-hook-list (cdr TeX-style-hook-list)))
        (t
         (let ((entry TeX-style-hook-list))
           (while (not (equal (car (car (cdr entry))) style))
             (setq entry (cdr entry)))
           (setcdr entry (cdr (cdr entry)))))))

(defvar TeX-virgin-style (if (and TeX-auto-global
				  (file-directory-p TeX-auto-global))
			     "VIRTEX"
			   "NoVirtexSymbols")
  "Style all documents use, independent on their format package.")

(defvar TeX-format-package nil
  "Format package loaded by VirTeX.  
Set this to \"VIRTEX\" to get predefined macros.")

 (make-variable-buffer-local 'TeX-format-package)

(defvar TeX-active-styles nil
  "List of styles currently active in the document.")

 (make-variable-buffer-local 'TeX-active-styles)

(defvar TeX-inhibit-style-hooks nil
  "If not nil, never execute TeX-run-style-hooks.")
 (make-variable-buffer-local 'TeX-inhibit-style-hooks)

(defvar TeX-inhibited-style-hooks nil
  "Style hooks not run because they were inhibited.")
 (make-variable-buffer-local 'TeX-inhibited-style-hooks)

(defun TeX-run-style-hooks (&rest styles)
  "Run the TeX following style hooks."
  (if TeX-inhibit-style-hooks
      (setq TeX-inhibited-style-hooks
            (append TeX-inhibited-style-hooks styles))
    (mapcar (function
             (lambda (style)
               (if (TeX-member style TeX-active-styles 'string-equal) 
                   ()                   ;Avoid recursion.
                 (setq TeX-active-styles
                       (cons style TeX-active-styles))
                 (TeX-load-style style)
                 (mapcar 'funcall
                         (cdr-safe (assoc style TeX-style-hook-list))))))
            styles)))

(defun TeX-run-format-hooks (&rest formats)
  "Run the VirTeX FORMATS packages, but not style hooks."
  (let ((TeX-inhibit-style-hooks t)
	(TeX-style-path (list TeX-format-directory)))
    (mapcar (function
             (lambda (format)
               (if (TeX-member format TeX-active-styles 'string-equal) 
                   ()                   ;Avoid recursion.
                 (setq TeX-active-styles
                       (cons format TeX-active-styles))
                 (TeX-load-style format)
                 (mapcar 'funcall
                         (cdr-safe (assoc format TeX-style-hook-list))))))
            formats)))

(defvar TeX-parse-self nil
  "*Parse file after loading it if no style hook is found for it.")

(defvar TeX-style-hook-applied-p nil
  "Nil, unless the style specific hooks have been applied.")
 (make-variable-buffer-local 'TeX-style-hook-applied-p)

(defun TeX-update-style (&optional force)
  "Run style specific hooks for the current document.

Only do this if it has not been done before, or if optional argument
FORCE is not nil."

  (if (or (eq TeX-auto-update 'BibTeX)	; Not a real TeX buffer
	  (and (not force) TeX-style-hook-applied-p))
      ()
    (setq TeX-style-hook-applied-p t)
    (message "Applying style hooks...")
    (apply 'TeX-run-style-hooks (append TeX-inhibited-style-hooks
					(list (TeX-strip-extension nil nil t)
					      (TeX-master-file))))
    (if (and TeX-parse-self
	     (null (cdr-safe (assoc (TeX-strip-extension nil nil t)
				    TeX-style-hook-list))))
	(apply 'TeX-run-style-hooks (TeX-auto-find-files)))
    
    (message "Applying style hooks... done")))

(defvar TeX-remove-style-hook nil
  "List of hooks to call when we remove the style specific information.")
 (make-variable-buffer-local 'TeX-remove-style-hook)

(defun TeX-remove-style ()
  "Remnove all style specific information."
  (setq TeX-style-hook-applied-p nil)
  (run-hooks 'TeX-remove-style-hooks)
  (setq TeX-active-styles (list TeX-virgin-style TeX-format-package)))

(defun TeX-style-list ()
  "Return a list of all styles (subfils) use by the current document."
  (TeX-update-style)
  TeX-active-styles)

;;; Special Characters

(defvar TeX-esc "\\" "The TeX escape character.")
 (make-variable-buffer-local 'TeX-esc)

(defvar TeX-grop "{" "The TeX group opening character.")
 (make-variable-buffer-local 'TeX-grop)

(defvar TeX-grcl "}" "The TeX group closing character.")
 (make-variable-buffer-local 'TeX-grcl)

;;; Symbols
;;
;; Must be before keymaps.

(defvar TeX-complete-word (key-binding "\e\t")
  "*Function to call if M-TAB is invoked on a word which does not start 
with a backslash.  Default is the meaning of M-TAB when latex-mode was
first invoked.")

(defvar TeX-symbol-list nil
  "List of known TeX macros.")

 (make-variable-buffer-local 'TeX-symbol-list)

(defvar TeX-symbol-list-changed nil
  "True if the symbol list has changed since last time it was sorted.")

 (make-variable-buffer-local 'TeX-symbol-list-changed)

(defun TeX-add-symbols (&rest symbols)
  "Add SYMBOLS (any number of arguments) to the list of known symbols."
  (if TeX-symbol-list-changed
      (setq TeX-symbol-list (cons symbols TeX-symbol-list))
    (setq TeX-symbol-list-changed t)
    (setq TeX-symbol-list (list symbols TeX-symbol-list))))

(defun TeX-symbol-list ()
  "Return list of symbols TeX knows about.
Each element is a list with the symbol (a string) as the first entry."
  (TeX-update-style)
  (if (null TeX-symbol-list-changed)
      ()
    (setq TeX-symbol-list-changed nil)
    ;; Sort it
    (message "Sorting symbols...")
    (setq TeX-symbol-list
          (sort (mapcar 'TeX-listify (apply 'append TeX-symbol-list))
                'TeX-car-string-lessp))
    ;; Make it unique
    (message "Removing duplicates...") 
    (let ((entry TeX-symbol-list))
      (while (and entry (cdr entry))
        (let ((this (car entry))
              (next (car (cdr entry))))
          (if (not (string-equal (car this) (car next)))
              (setq entry (cdr entry))
            ;; We have to equal symbols.  Use the one with
            ;; most arguments.
            (if (> (length next) (length this))
                (setcdr this (cdr next)))
            (setcdr entry (cdr (cdr entry)))))))
    (message "Removing duplicates... done"))
  TeX-symbol-list)

(add-hook 'TeX-remove-style-hook
	  (function (lambda () (setq LaTeX-symbol-list nil))))

;;; The Mode

(defvar TeX-format-list
  '(("JSLITEX" japanese-slitex-mode
     "\\\\documentstyle[^%\n]*{jslides}")
    ("SLITEX" slitex-mode "\\\\documentstyle[^%\n]*{slides}")
    ("FOILTEX" foiltex-mode "\\\\documentstyle[^%\n]*{foils}")
    ("AMSLATEX" ams-latex-mode "\\\\documentstyle[^%\n]*{ams")
    ("JLATEX" japanese-latex-mode "\\\\documentstyle[^%\n]*{j")
    ("LATEX2E" latex2e-mode "\\\\documentclass\\b")
    ("LATEX" latex-mode 
     "\\\\\\(begin\\|section\\|chapter\\|documentstyle\\)\\b")
    ("AMSTEX" ams-tex-mode "-- string likely in AmS TeX --")
    ("JTEX" japanese-plain-tex-mode "-- string likely in Japanese TeX --")
    ("TEX" plain-tex-mode "."))
  "*List of format packages to consider when choosing a TeX mode.

A list with a entry for each format package available at the site.

Each entry is a list with three elements.

1. The name of the format package.
2. The name of the major mode.
3. A regexp typically matched in the beginning of the file.

When entering tex-mode, each regexp is tried in turn in order to find
when major mode to enter.")

(defvar TeX-default-mode 'latex-mode
  "*Mode to enter for a new file when it can't be determined whether
the file is plain TeX or LaTeX or what.")

(defvar TeX-force-default-mode nil
  "*If set to nil, try to infer the mode of the file from its content.")

;;;###autoload
(defun tex-mode ()
  "Major mode for editing files of input for TeX or LaTeX.
Tries to guess whether this file is for plain TeX or LaTeX.

The algorithm is as follows:

   1) if the file is empty or TeX-force-default-mode is not set to nil, 
      TeX-default-mode is chosen 
   2) If \\documentstyle or \\begin{, \\section{, \\part{ or \\chapter{ is
      found, latex-mode is selected.
   3) Otherwise, use plain-tex-mode "
  (interactive)

  (funcall (if (or (equal (buffer-size) 0)
                   TeX-force-default-mode)
               TeX-default-mode
             (save-excursion
               (goto-char (point-min))
               (let ((comment-start-skip ;Used by TeX-in-comment
		      (concat
		       "\\(\\(^\\|[^\\]\\)\\("
		       (regexp-quote TeX-esc)
		       (regexp-quote TeX-esc)
		       "\\)*\\)\\(%+ *\\)"))
		     (entry TeX-format-list)
                     answer)
                 (while (and entry (not answer))
                   (if (re-search-forward (nth 2 (car entry))
                                          10000 t)
		       (if (not (TeX-in-comment))
			   (setq answer (nth 1 (car entry))))
		     (setq entry (cdr entry))))
                 (if answer
                     answer
                   TeX-default-mode))))))

(defun VirTeX-mode (format)
  "Initialize major mode for VirTeX FORMAT package."
  (interactive "s")
  
  (kill-all-local-variables)

  (setq TeX-format-package format)
  (setq local-abbrev-table text-mode-abbrev-table)
  (setq indent-tabs-mode nil)
  (setq words-include-escapes t)
  (make-local-variable 'ispell-parser)
  (setq ispell-parser 'tex)
  (make-local-variable 'ispell-tex-p)
  (setq ispell-tex-p t)

  ;; Redefine some standard varaibles
  (make-local-variable 'paragraph-start)
  (make-local-variable 'paragraph-separate)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip
	(concat
	 "\\(\\(^\\|[^\\]\\)\\("
	 (regexp-quote TeX-esc)
	 (regexp-quote TeX-esc)
	 "\\)*\\)\\(%+ *\\)"))
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'TeX-comment-indent)
  (make-local-variable 'compile-command)
  (if (boundp 'compile-command)
      ()
    (setq compile-command "make"))

  ;; We want this to be early in the list, so we do not add it before
  ;; we enter TeX mode  the first time.
  (add-hook 'write-file-hooks 'TeX-auto-write)
  (make-local-variable 'TeX-auto-update)
  (setq TeX-auto-update t)

  ;; Run common hooks
  (TeX-run-format-hooks TeX-virgin-style TeX-format-package)
  (make-local-variable 'ispell-tex-major-modes)
  (setq ispell-tex-major-modes (list major-mode)))

;;;###autoload
(defun plain-tex-mode ()
  "Major mode for editing files of input for plain TeX.
Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.

See info under AUC TeX for full documentation.

Special commands:
\\{TeX-mode-map}
 
Entering plain-tex-mode calls the value of text-mode-hook,
then the value of TeX-mode-hook, and then the value
of plain-TeX-mode-hook."
  (interactive)
  (VirTeX-mode "TEX"))

;;;###autoload
(defun ams-tex-mode ()
  "Major mode for editing files of input for AmS TeX.
Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.

See info under AUC TeX for full documentation.

Special commands:
\\{TeX-mode-map}
 
Entering ams-tex mode calls the value of text-mode-hook,
then the value of TeX-mode-hook, the value of plain-TeX-mode-hook, and
then the value of AmS-TeX-mode-hook."
  (interactive)
  (VirTeX-mode "AMSTEX"))

;;;###autoload
(defun latex-mode ()
  "Major mode for editing files of input for LaTeX.

Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.
LFD and TAB indent lines as with programming modes.

See under AUC TeX for full documentation.

Special commands:
\\{LaTeX-mode-map}

Entering LaTeX mode calls the value of text-mode-hook,
then the value of TeX-mode-hook, and then the value
of LaTeX-mode-hook."
  (interactive)
  (VirTeX-mode "LATEX"))

;;;###autoload
(defun latex2e-mode ()
  "Major mode for editing files of input for LaTeX2e.

Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.
LFD and TAB indent lines as with programming modes.

See under AUC TeX for full documentation.

Special commands:
\\{LaTeX-mode-map}

Entering LaTeX2e mode calls the value of text-mode-hook,
then the value of TeX-mode-hook, and then the value
of LaTeX-mode-hook, and finally the value of LaTeX2e-mode-hook."
  (interactive)
  (VirTeX-mode "LATEX2E"))

;;;###autoload
(defun slitex-mode ()
  "Major mode for editing files of input for SliTeX.

Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.
LFD and TAB indent lines as with programming modes.

See under AUC TeX for full documentation.

Special commands:
\\{LaTeX-mode-map}

Entering LaTeX mode calls the value of text-mode-hook,
then the value of TeX-mode-hook, and then the value
of LaTeX-mode-hook. and then the value of SliTeX-mode-hook."
  (interactive)
  (VirTeX-mode "SLITEX"))

;;;###autoload
(defun ams-latex-mode ()
  "Major mode for editing files of input for AmS-LaTeX.

Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.
LFD and TAB indent lines as with programming modes.

See under AUC TeX for full documentation.

Special commands:
\\{LaTeX-mode-map}

Entering AmS LaTeX mode calls the value of text-mode-hook, then the
value of TeX-mode-hook, then the value of AmS-TeX-mode-hook, and then
the value of LaTeX-mode-hook. and then the value of
AmS-LaTeX-mode-hook." 
  (interactive)
  (VirTeX-mode "AMSLATEX"))

;;;###autoload
(defun foiltex-mode ()
  "Major mode for editing files of input for FoilTeX.

Makes $ and } display the characters they match.
Makes \" insert `` when it seems to be the beginning of a quotation,
and '' when it appears to be the end; it inserts \" only after a \\.
LFD and TAB indent lines as with programming modes.

FoilTeX is a latex macropackage for creating foils.

FoilTeX has been devoped by IBM, and is available for non-commercial
experimental use.  See the file fltplain.tex of the FoilTeX distribution
for the correct and complete license agreement.

FoilTeX can be copied from many major ftp archives, for example at
ymir.claremont.edu in tex/inputs/foiltex.

Special commands:
\\{LaTeX-mode-map}

Entering FoilTeX mode calls the value of text-mode-hook,
then the value of TeX-mode-hook, and then the value
of LaTeX-mode-hook. and then the value of FoilTeX-mode-hook."
  (interactive)
  (VirTeX-mode "FOILTEX"))

(if (featurep 'hilit19)
    (let ((latex-paterns (cdr (assq 'latex-mode hilit-patterns-alist)))
	  (plain-tex-paterns (cdr (assq 'plain-tex-mode
					hilit-patterns-alist))))
      (setq hilit-patterns-alist
	    (append (list (cons 'ams-tex-mode plain-tex-paterns)
			  (cons 'latex2e-mode latex-paterns)
			  (cons 'slitex-mode latex-paterns)
			  (cons 'ams-latex-mode latex-paterns)
			  (cons 'foiltex-mode latex-paterns))
		    hilit-patterns-alist))))

;;; Utilities
;;
;; Some of these functions has little to do with TeX, but nonetheless we
;; should use the "TeX-" prefix to avoid name clashes.

(defvar TeX-auto-regexp-list 'TeX-auto-full-regexp-list
  "*List of regular expresions used for parsing the current file.")
  (make-variable-buffer-local 'TeX-auto-regexp-list)

(defvar TeX-file-extensions '("tex" "sty" "texi")
  "*File extensions used by manually generated TeX files.")

(defvar TeX-all-extensions '("[^.\n]+")
  "All possible file extensions.")

(defvar TeX-default-extension "tex"
  "*Default extension for TeX files.")

  (make-variable-buffer-local 'TeX-default-extension)

(defvar BibTeX-file-extensions '("bib")
  "Valid file extensions for BibTeX files.")

(defvar BibTeX-style-extensions '("bst")
  "Valid file extensions for BibTeX styles.")

(defvar TeX-ignore-file "\\(^\\|/\\)\\(\\.\\|\\.\\.\\|RCS\\|SCCS\\|CVS\\)$"
  "*Regular expression matching file names to ignore.

These files or directories will not be considered when searching for
TeX files in a directory.")

(defvar TeX-file-recurse t
  "*If not nil, search TeX directories recursivly.")

(defun TeX-match-extension (file &optional extensions)
  "Return non-nil if FILE has an one of EXTENSIONS.

If EXTENSIONS is not specified or nil, the value of
TeX-file-extensions is used instead."

  (if (null extensions)
      (setq extensions TeX-file-extensions))

  (let ((regexp (concat "\\.\\("
                        (mapconcat 'identity extensions "\\|")
                        "\\)$")))
    (string-match regexp file)))

(defun TeX-strip-extension (&optional string extensions nodir nostrip)
  "Return STRING without any trailing extension in EXTENSIONS.
If NODIR is set, also remove directory part of STRING. 
If NOSTRIP is set, do not remove extension after all.
STRING defaults to the name of the current buffer.
EXTENSIONS defaults to TeX-file-extensions."
  
  (if (null string)
      (setq string (or (buffer-file-name) "<none>")))
  
  (if (null extensions)
      (setq extensions TeX-file-extensions))
  
  (let ((strip (if (and (not nostrip)
                        (TeX-match-extension string extensions))
                   (substring string 0 (match-beginning 0))
                 string)))
    (if nodir
        (file-name-nondirectory strip)
      strip)))

(defun TeX-search-files (&optional directories extensions nodir strip)
  "Return a list of all reachable files in DIRECTORIES ending with EXTENSIONS.
If optional argument NODIR is set, remove directory part.
If optional argument STRIP is set, remove file extension.
If optional argument DIRECTORIES is set, search in those directories. 
Otherwise, search in all TeX macro directories.
If optional argument EXTENSIONS is not set, use TeX-file-extensions"

  (if (null extensions)
      (setq extensions TeX-file-extensions))
  
  (if (null directories)
      (setq directories
	    (cons "./" (append TeX-macro-private TeX-macro-global))))
  
  (let (match)
    
    (while directories
      (let* ((directory (car directories))
             (content (and directory
			   (file-readable-p directory)
			   (file-directory-p directory)
			   (directory-files directory))))
        
        (setq directories (cdr directories))
	
        (while content
          (let ((file (concat directory (car content))))
	    
            (setq content (cdr content))
            (cond ((string-match TeX-ignore-file file))
		  ((not (file-readable-p file)))
                  ((file-directory-p file)
		   (if TeX-file-recurse
		       (setq directories
			     (cons (concat file "/") directories))))
                  ((TeX-match-extension file extensions)
                   (setq match (cons (TeX-strip-extension file
							  extensions
							  nodir
							  (not strip))
                                     match))))))))
    
    match))

(defun TeX-split-string (char string)
  "Returns a list of strings. given REGEXP the STRING is split into 
sections which in string was seperated by REGEXP.

Examples:

      (TeX-split-string \"\:\" \"abc:def:ghi\")
          -> (\"abc\" \"def\" \"ghi\")

      (TeX-split-string \" *\" \"dvips -Plw -p3 -c4 testfile.dvi\")

          -> (\"dvips\" \"-Plw\" \"-p3\" \"-c4\" \"testfile.dvi\")

If CHAR is nil, or \"\", an error will occur."

  (let ((regexp char)
        (start 0)
        (result '()))
    (while (string-match regexp string start)
      (let ((match (string-match regexp string start)))
        (setq result (cons (substring string start match) result))
        (setq start (match-end 0))))
    (setq result (cons (substring string start nil) result))
    (nreverse result)))

(defun TeX-car-string-lessp (a b)
  (string-lessp (car a) (car b)))

(defun TeX-listify (a)
  (if (listp a) a (list a)))

(defun TeX-member (elt list how)
  "Returns the member ELT in LIST.  Comparison done with HOW.

Return nil if ELT is not a member of LIST."
  (while (and list (not (funcall how elt (car list))))
    (setq list (cdr list)))
  (car-safe list))

(defun TeX-assoc (elem list)
  "Like assoc, except case incentive."
  (let ((case-fold-search t))
    (TeX-member elem list
		(function (lambda (a b)
		  (string-match (concat "^" (regexp-quote a) "$")
				(car b)))))))

(defun TeX-match-buffer (n)
  "Return the substring corresponding to the N'th match.

See match-data for details."
  (if (match-beginning n)
      (buffer-substring (match-beginning n) (match-end n))
    ""))

(defun TeX-function-p (arg)
  "Return non-nil if ARG is collable as a function."
  (or (and (fboundp 'byte-code-function-p)
	   (byte-code-function-p arg))
      (and (listp arg)
	   (eq (car arg) 'lambda))
      (and (symbolp arg)
	   (fboundp arg))))

;;; Syntax Table

(defvar TeX-mode-syntax-table (make-syntax-table)
  "Syntax table used while in TeX mode.")

 (make-variable-buffer-local 'TeX-mode-syntax-table)

(progn ; Define TeX-mode-syntax-table.
  (modify-syntax-entry (string-to-char TeX-esc)
		           "\\" TeX-mode-syntax-table)
  (modify-syntax-entry ?\f ">"  TeX-mode-syntax-table)
  (modify-syntax-entry ?\n ">"  TeX-mode-syntax-table)
  (modify-syntax-entry (string-to-char TeX-grop)
		           (concat "(" TeX-grcl)
		                TeX-mode-syntax-table)  
  (modify-syntax-entry (string-to-char TeX-grcl)
		           (concat ")" TeX-grop)
			        TeX-mode-syntax-table)  
  (modify-syntax-entry ?%  "<"  TeX-mode-syntax-table)
  (modify-syntax-entry ?\" "."  TeX-mode-syntax-table)
  (modify-syntax-entry ?&  "."  TeX-mode-syntax-table)
  (modify-syntax-entry ?_  "."  TeX-mode-syntax-table)
  (modify-syntax-entry ?@  "_"  TeX-mode-syntax-table)
  (modify-syntax-entry ?~  " "  TeX-mode-syntax-table)
  (modify-syntax-entry ?'  "w"  TeX-mode-syntax-table))

;;; Keymap

(defvar TeX-electric-escape nil
  "If this is non-nil when AUC TeX is loaded, the TeX escape
character ``\\'' will be bound to `TeX-electric-macro'.")

(defvar TeX-mode-map nil
  "Keymap for common TeX and LaTeX commands.")

(if TeX-mode-map 
    ()
  (setq TeX-mode-map (make-sparse-keymap))

  ;; Standard
  (define-key TeX-mode-map "\177"     'backward-delete-char-untabify)
  (define-key TeX-mode-map "\C-c}"    'up-list)
  (define-key TeX-mode-map "\C-c#"    'TeX-normal-mode)
  (define-key TeX-mode-map "\C-c\C-n" 'TeX-normal-mode)
  (define-key TeX-mode-map "\C-c?"    'describe-mode)
  
  ;; From tex-init.el
  (define-key TeX-mode-map "\""       'TeX-insert-quote)
  (define-key TeX-mode-map "$"        'TeX-insert-dollar)
  (define-key TeX-mode-map "\C-c{"    'TeX-insert-braces)
  (define-key TeX-mode-map "\C-c\C-f" 'TeX-font)

  (define-key TeX-mode-map "\C-c;"    'TeX-comment-region)
  (define-key TeX-mode-map "\C-c%"    'TeX-comment-paragraph)

  (define-key TeX-mode-map "\C-c'"    'TeX-comment-out-paragraph) ;*** Old way
  (define-key TeX-mode-map "\C-c:"    'TeX-un-comment-region) ;*** Old way
  (define-key TeX-mode-map "\C-c\""   'TeX-un-comment) ;*** Old way

  ;; From tex-cpl.el
  (define-key TeX-mode-map "\C-c\C-m" 'TeX-insert-macro)
  (if TeX-electric-escape
      (define-key TeX-mode-map "\\" 'TeX-electric-macro))

  ;; From tex-buf.el
  (define-key TeX-mode-map "\C-c\C-d" 'TeX-save-document)
  (define-key TeX-mode-map "\C-c\C-r" 'TeX-command-region)
  (define-key TeX-mode-map "\C-c\C-c" 'TeX-command-master)
  (define-key TeX-mode-map "\C-c\C-k" 'TeX-kill-job)
  (define-key TeX-mode-map "\C-c\C-l" 'TeX-recenter-output-buffer)
  (define-key TeX-mode-map "\C-c^" 'TeX-home-buffer)

  ;; From tex-dbg.el
  (define-key TeX-mode-map "\C-c`"    'TeX-next-error)
  (define-key TeX-mode-map "\C-c\C-w" 'TeX-toggle-debug-boxes)
  ;;(define-key TeX-mode-map "\C-c@" 'TeX-toggle-debug-boxes) ;*** Stupid alias

  ;; From tex.cpl.el
  (define-key TeX-mode-map "\e\t"   'TeX-complete-symbol) ;*** Emacs 19 way
  (define-key TeX-mode-map "\C-c\t"   'TeX-complete-symbol))
  
(defvar plain-TeX-mode-map (copy-keymap TeX-mode-map)
  "Keymap used in plain TeX mode.")

(defun TeX-command-menu (name file)
  ;; Execute TeX-command-list NAME on FILE from a menu.
  (if (nth 4 (assoc name TeX-command-list))
      (if (eq file 'TeX-region-file)
	  (if (mark)
	      (let ((begin (min (point) (mark)))
		    (end (max (point) (mark))))
		(TeX-region-create (TeX-region-file "tex")
				   (buffer-substring begin end)
				   (file-name-nondirectory (buffer-file-name))
				   (count-lines (point-min) begin))))
	(TeX-save-document (TeX-master-file))))
  (TeX-command name file))

(defun TeX-command-menu-print (printer command name file)
  ;; On PRINTER print FILE from a menu.
  (let ((TeX-printer-default printer)
	(TeX-printer-list nil)
	(TeX-print-command command))
    (TeX-command name file)))

(defun TeX-command-menu-printer-entry (entry)
  ;; Return TeX-printer-list ENTRY as a menu item.
  (vector (nth 0 entry)
	  (list 'TeX-command-menu-print
		(nth 0 entry)
		(or (nth lookup entry) command)
		name
		(list 'quote file))
	  t))

(defun TeX-command-menu-entry (entry)
  ;; Return TeX-command-list ENTRY as a menu item.
  (let ((name (car entry)))
    (cond ((and (string-equal name TeX-command-Print)
		TeX-printer-list)
	   (let ((command TeX-print-command)
		 (lookup 1))
	     (append (list TeX-command-Print)
		     (mapcar 'TeX-command-menu-printer-entry TeX-printer-list))))
	  ((and (string-equal name TeX-command-Queue)
		TeX-printer-list)
	   (let ((command TeX-queue-command)
		 (lookup 2))
	     (append (list TeX-command-Queue)
		     (mapcar 'TeX-command-menu-printer-entry TeX-printer-list))))
	  (t
	   (vector name (list 'TeX-command-menu name (list 'quote file)) t)))))

(defun TeX-command-create-menu (name file)
  ;; Create menu NAME for each entry TeX-command-list for FILE.
  (append (list name)
	  (mapcar 'TeX-command-menu-entry TeX-command-list)))

(easy-menu-define plain-TeX-mode-menu
    plain-TeX-mode-map
    "Menu used in plain TeX mode."
  (list "AUC TeX"
	["Macro..." TeX-insert-macro t]
	["Complete" TeX-complete-symbol t]
	["Save Document" TeX-save-document t]
	(TeX-command-create-menu "Command on Master File" 'TeX-master-file)
	(TeX-command-create-menu "Command on Region" 'TeX-region-file)
	["Next Error" TeX-next-error t]
	["Kill Job" TeX-kill-job t]
	["Toggle debug of boxes" TeX-toggle-debug-boxes t]
	["Switch to original file" TeX-home-buffer t]
	["Recenter Output Buffer" TeX-recenter-output-buffer t]
	;; ["Uncomment" TeX-un-comment t]
	["Uncomment Region" TeX-un-comment-region t]
	;; ["Comment Out Paragraph" TeX-comment-out-paragraph t]
	;; ["Comment Paragraph" TeX-comment-paragraph t]
	["Comment Region" TeX-comment-region t]
	["Switch to Master file" TeX-home-buffer t]
	["Reset Buffer" TeX-normal-mode t]
	["Reset AUC TeX" (TeX-normal-mode t) "C-u C-c C-n"]))

;; Seems to be a Lucid convention
(if (fboundp 'plain-TeX-mode-menu)
    (define-key plain-TeX-mode-map 'button3 'plain-TeX-mode-menu))

;;; Comments

(defun TeX-comment-out-region (level start end)
  "Insert LEVEL comment characters before each line in the region."
  (interactive "*p\nr")
  (save-excursion
    (goto-char start)
    (beginning-of-line)
    (setq start (point))
    (goto-char end)
    (end-of-line)
    (setq end (point))
    (save-restriction 
      (narrow-to-region end start)
      (goto-char (point-min))
      (while (not (eobp))
	(let ((count level))
	  (while (> count 0)
	    (insert comment-start)
	    (setq count (- count 1))))
	(forward-line 1)))))

(defun TeX-un-comment-region (level start end)
  "Remove up to LEVEL comment characters from each line in the region."
  (interactive "*p\nr") 
  (save-excursion
    (goto-char start)
    (beginning-of-line)
    (setq start (point))
    (goto-char end)
    (end-of-line)
    (setq end (point))
    (save-restriction 
      (narrow-to-region start end)
      (goto-char (point-min))
      (while (not (eobp))
	(let ((count level))
	  (while (and (> count 0)
		      (not (eobp))
		      (looking-at (regexp-quote comment-start)))
	    (delete-char 1)
	    (setq count (- count 1))))
	(forward-line 1)))))

(defun TeX-un-comment (level)
  "Delete up to LEVEL %'s from the beginning of each line in a comment."
  (interactive "*p")
  (save-excursion
    ; Find first comment line
    (re-search-backward (concat "^[^" comment-start "]") nil 'limit)
    (let ((beg (point)))
      (forward-line 1)
      ; Find last comment line
      (re-search-forward (concat "^[^" comment-start "]") nil 'limit)
      ; Uncomment region
      (TeX-un-comment-region level beg (point)))))

(defun TeX-comment-out-paragraph (level)
  "Inserts LEVEL %'s at the beginning of every line in the current paragraph."
  (interactive "*p")
  (save-excursion
    (end-of-line)
    (start-of-paragraph-text)
    (beginning-of-line)
    (let ((beg (point)))
      (end-of-paragraph-text)
      ;; (forward-line -1)
      (beginning-of-line)
      (TeX-comment-out-region level beg (point)))))

(defun TeX-comment-region (level begin end)
  "Remove or insert comments from region.

Without an argument, insert one level of comments in all lines in the
current region.

With a C-u argument, uncomment one level of comments from the
current region.

With a positive numeric argument, insert that many comment characters
in all lines in the current region.

With a negative numeric argument, remove that many comment character
from the current region."
  (interactive "*P\nr")
  (cond ((null level)
	 (TeX-comment-out-region 1 begin end))
	((listp level)
	 (TeX-un-comment-region 1 begin end))
	((eq level '-)
	 (TeX-un-comment-region 1 begin end))
	((> level 0)
	 (TeX-comment-out-region level begin end))
	((< level 0)
	 (TeX-un-comment-region (- level) begin end))))

(defun TeX-comment-paragraph (level)
  "Remove or insert comments from paragraph.

Without an argument, insert one level of comments in all lines in the
current paragraph.

With a C-u argument, uncomment one level of comments from the
surrounding comment lines.

With a positive numeric argument, insert that many comment characters
in all lines in the current paragraph.

With a negative numeric argument, remove that many comment character
from the surrounding comment lines."
  (interactive "*P")
  (cond ((null level)
	 (TeX-comment-out-paragraph 1))
	((listp level)
	 (TeX-un-comment 1))
	((eq level '-)
	 (TeX-un-comment 1))
	((> level 0)
	 (TeX-comment-out-paragraph level))
	((< level 0)
	 (TeX-un-comment (- level)))))



(defun TeX-in-comment ()
  ;; Return non-nil if point is in a comment.
  (if (or (bolp)
	  (null comment-start-skip)
	  (eq (preceding-char) ?\r))
      nil
    (save-excursion
      (let ((pos (point)))
	(re-search-backward "^\\|\r" nil t)
	(or (looking-at comment-start-skip)
	    (re-search-forward comment-start-skip pos t))))))

;;; Indentation

(defun TeX-brace-count-line ()
  "Count number of open/closed braces."
  (save-excursion
    (save-restriction
      (let ((count 0))
	(narrow-to-region (point)
			  (save-excursion
			    (re-search-forward "[^\\\\]%\\|\n\\|\\'")
			    (backward-char)
			    (point)))

	(if (looking-at "\\({\\|}\\)")
	  (if (string= "{" (TeX-match-buffer 1))
	      (setq count (+ TeX-brace-indent-level count))
	    (setq count (- count TeX-brace-indent-level))))

	(while (re-search-forward "[^\\\\]\\({\\|}\\)" nil t)
	  (if (string= "{" (TeX-match-buffer 1))
	      (setq count (+ TeX-brace-indent-level count))
	    (setq count (- count TeX-brace-indent-level)))
	  (if (looking-at "\\({\\|}\\)")
	      (if (string= "{" (TeX-match-buffer 1))
		  (setq count (+ TeX-brace-indent-level count))
		(setq count (- count TeX-brace-indent-level)))))
	count))))

(defvar TeX-brace-indent-level 2
  "*The level of indentation produced by a open brace.")

(defun TeX-comment-indent ()
  (if (looking-at "%%%")
      (current-column)
    (skip-chars-backward " \t")
    (max (if (bolp) 0 (1+ (current-column)))
	 comment-column)))

;;; Fonts

(defvar TeX-font-list '((?\C-b "{\\bf " "}")
			(?\C-c "{\\sc " "}")
			(?\C-e "{\\em " "\\/}")
			(?\C-i "{\\it " "\\/}")
			(?\C-r "{\\rm " "}")
			(?\C-s "{\\sl " "\\/}")
			(?\C-t "{\\tt " "}")
			(?\C-d "" "" t))
  "*List of fonts used by TeX-font.

Each entry is a list with three elements.  The first element is the
key to active the font.  The secont element is the string to insert
before point, and the third element is the string to insert after
point.  An optional fourth element means always replace if not nil.")

(defun TeX-font (replace what)
  "Insert template for font change command.

If REPLACE is not nil, replace current font.  WHAT determines the font
to use, as specified by TeX-font-list."
  (interactive "*P\nc")
  (let* ((entry (assoc what TeX-font-list)))
    (setq replace (or replace (nth 3 entry)))
    (cond ((null entry)
	   (error "No font for defined `%c'." what))
	  ((not replace)
	   (insert (nth 1 entry))
	   (save-excursion
	     (insert (nth 2 entry))))
	  (t
	   (TeX-font-replace (nth 1 entry) (nth 2 entry))))))

(defun TeX-font-replace (start end)
  "Replace font specification around point with START and END."
  (save-excursion
    (while (not (looking-at "{\\\\[a-zA-Z]+ "))
      (up-list -1))
    (forward-sexp)
    (save-excursion
      (replace-match start t t))
    (if (save-excursion
	  (backward-char 3)
	  (if (looking-at (regexp-quote "\\/}"))
	      (progn
		(delete-char 3)
		nil)
	    t))
	(delete-backward-char 1))
    (insert end)))

;;; Dollars
;;
;; Originally stolen from VorTeX.
;; Copyright (C) 1986, 1987, 1988 Pehong Chen (phc@renoir.berkeley.edu)

(defvar TeX-dollar-sign ?$
  "*Character user to enter and leaver math mode in TeX.")

(defconst TeX-dollar-string (char-to-string TeX-dollar-sign))

(defconst TeX-dollar-regexp 
  (concat "^" (regexp-quote TeX-dollar-string) "\\|[^" TeX-esc "]"
	  (regexp-quote TeX-dollar-string)))
  
(defvar TeX-dollar-list nil)
  (make-variable-buffer-local 'TeX-match-dollar-on)

(defvar TeX-par-start nil)
  (make-variable-buffer-local 'TeX-par-start)

(defvar TeX-par-end nil)
  (make-variable-buffer-local 'TeX-par-end)

(defvar TeX-symbol-marker nil)

(defvar TeX-symbol-marker-pos 0)

(defun TeX-bouncing-point (m)
  (save-excursion
    (if (pos-visible-in-window-p)
	(sit-for 1)
      (let* ((pos1 (point))
             (pos2 (+ pos1 m))
             (sym (buffer-substring pos1 pos2))
             (msg1 (progn (beginning-of-line) (buffer-substring (point) pos1)))
             (msg2 (progn (end-of-line) (buffer-substring pos2 (point)))))
        (message "%s`%s'%s" msg1 sym msg2)))))

(defun TeX-locate-delimiter (pos sym symlst)
  (let ((marker nil)
        (marker-pos 0)
        (pair t)
        (head nil))
    (catch 'loop
      (while symlst
        (setq marker (car symlst))
        (setq marker-pos (1- (marker-position marker)))
        (if (and (/= pos marker-pos) (= (char-after marker-pos) sym))
	    (if (> pos marker-pos)
		(progn
		  (setq TeX-symbol-marker-pos marker-pos)
		  (setq TeX-symbol-marker marker) 
		  (setq head (cons marker head))
		  (setq pair (not pair)))
	      (if pair (setq TeX-symbol-marker nil))
	      (throw 'loop (append (reverse head)
				   (cons (set-marker (make-marker) (1+ pos)) 
					 symlst)))))
        (setq symlst (cdr symlst)))
      (if pair (setq TeX-symbol-marker nil))
      (reverse (cons (set-marker (make-marker) (1+ pos)) head)))))

(defun TeX-dollar-verify ()
  ;; Verify if the current paragraph is the same as last.
  ;; If so, do nothing, otherwise reset TeX-par-start and TeX-par-end and
  ;; reconstruct the symbol-list.
  (let ((start (save-excursion
                 (if (re-search-backward paragraph-separate nil t)
		     (point)
                   1)))
        (end (save-excursion
               (if (re-search-forward paragraph-separate nil t)
		   (1+ (point))
                 (1+ (point-max)))))
        (init nil))
    (if (null TeX-par-start)
	(setq TeX-par-start (set-marker (make-marker) 1)))
    (if (/= (marker-position TeX-par-start) start)
	(progn
	  (set-marker TeX-par-start start)
	  (setq init t)))
    (if (null TeX-par-end)
	(setq TeX-par-end (set-marker (make-marker) 1)))
    (if (/= (marker-position TeX-par-end) end)
	(progn
	  (set-marker TeX-par-end end)
	  (setq init t)))
    (if init
	(save-excursion
	  (setq TeX-dollar-list nil)
	  (goto-char start)
	  (while (re-search-forward TeX-dollar-regexp end t)
	    (setq TeX-dollar-list
		  (append TeX-dollar-list
			  (list (set-marker (make-marker)
					    (if (= (following-char)
						   TeX-dollar-sign)
						(progn
						  (forward-char 1)
						  (point))
					      (point)))))))))))

(defun TeX-insert-dollar (&optional arg)
  "Insert dollar sign.  

Show matching dollar sign if this dollar sign end the TeX math mode.  
Ensure double dollar signs match up correctly by inserting extra
dollar signs when needed.

With optional ARG, insert that many dollar signs."
  (interactive "P")
  (if arg
      (let ((count (prefix-numeric-value arg)))
	(if (listp arg)
	    (self-insert-command 1)	;C-u always inserts just one
	  (self-insert-command count)))
    (let ((pc (preceding-char))
	  (pos (point))
	  (pt (point))
	  (single t))
      (TeX-dollar-verify)
      (if (= pc (string-to-char TeX-esc))
	  (insert TeX-dollar-sign)
	(if (= pc TeX-dollar-sign)
	    (progn
	      (setq single nil)
	      (if (and (> pos 2) (= (char-after (- pos 2)) TeX-dollar-sign))
		  (setq pt (1- pos))	; Doesn't echo 3rd $, if $$ already
		(backward-char 1) 
		(insert TeX-dollar-sign)
		(goto-char (1+ pos))))
	  (insert TeX-dollar-sign))
	(setq TeX-dollar-list
	      (TeX-locate-delimiter pt TeX-dollar-sign TeX-dollar-list))
	(if TeX-symbol-marker
	    (save-excursion
	      (goto-char TeX-symbol-marker-pos)
	      (if (and (= (preceding-char) TeX-dollar-sign)
		       (/= (char-after (- (point) 2)) TeX-dollar-sign))
		  (progn
		    (backward-char 1)
		    (if single
			(save-excursion
			  (goto-char pos)
			  (insert TeX-dollar-sign))))	; $$foo$`$'
		(if (not single)
		    (progn
		      (insert TeX-dollar-sign) ; `$'$foo$$
		      (backward-char 1))))
	      (TeX-bouncing-point (if single 1 2))))))))

;;; Simple Commands

(defun TeX-normal-mode (arg)
  "Remove all information about this buffer, and apply the style hooks again.
Save buffer first including style information.
With optional argument, also reload the style hooks."
  (interactive "*P")
  (if arg
      (setq TeX-style-hook-list nil))
  (require 'tex-auto)
  (let ((TeX-auto-save t))
    (if (buffer-modified-p)
	(save-buffer)
      (TeX-auto-write)))
  (normal-mode)
  (TeX-update-style))

(defvar TeX-open-quote "``"
  "*String inserted by typing \\[TeX-insert-quote] to open a quotation.")

(defvar TeX-close-quote "''"
  "*String inserted by typing \\[TeX-insert-quote] to close a quotation.")

;;;###autoload
(defun TeX-insert-quote (arg)
  "Insert the appropriate quote marks for TeX.
Inserts the value of `TeX-open-quote' (normally ``) or `TeX-close-quote'
\(normally '') depending on the context.  With prefix argument, always
inserts \" characters."
  (interactive "*P")
  (if arg
      (let ((count (prefix-numeric-value arg)))
	(if (listp arg)
	    (self-insert-command 1)	;C-u always inserts just one
	  (self-insert-command count)))
    (TeX-update-style)
    (insert (cond ((bobp)
		   TeX-open-quote)
		  ((= (preceding-char) (string-to-char TeX-esc))
		   ?\")
		  ((= (preceding-char) ?\")
		   ?\")
		  ((save-excursion
		     (forward-char (- (length TeX-open-quote)))
		     (looking-at (regexp-quote TeX-open-quote)))
		   (delete-backward-char (length TeX-open-quote))
		   ?\")
		  ((save-excursion
		     (forward-char (- (length TeX-close-quote)))
		     (looking-at (regexp-quote TeX-close-quote)))
		   (delete-backward-char (length TeX-close-quote))
		   ?\")
		  ((save-excursion
		     (forward-char -1)
		     (looking-at "[ \t\n]\\|\\s("))
		   TeX-open-quote)
		  (t
		   TeX-close-quote)))))

;; For the sake of BibTeX...
;;;###autoload
(fset 'tex-insert-quote 'TeX-insert-quote)

(defun TeX-insert-braces ()
  "Make a pair of braces and be poised to type inside of them."
  (interactive "*")
  (insert TeX-grop)
  (save-excursion
    (insert TeX-grcl)))

;;;###autoload
(defun TeX-submit-bug-report ()
  "Submit via mail a bug report on AUC TeX"
  (interactive)
  (require 'reporter)
  (and (y-or-n-p "Do you really want to submit a report on AUC TeX? ")
       (reporter-submit-bug-report
	"auc-tex_mgr@iesd.auc.dk"
	(concat "AUC TeX " AUC-TeX-version " of " AUC-TeX-date)
	(list 'window-system
	      'TeX-style-path
	      'TeX-auto-save
	      'TeX-parse-self
	      'TeX-master))))

;;; Ispell Support

;; The FSF ispell.el use this.
(defun ispell-tex-buffer-p ()
  (and (boundp 'ispell-tex-p) ispell-tex-p))

;; The FSF ispell.el might one day use this.
(setq ispell-enable-tex-parser t)

(defun TeX-run-ispell (command string file)
  "Run ispell on current TeX buffer."
  (cond ((and (string-equal file (TeX-region-file))
	      (fboundp 'ispell-region))
	 (call-interactively 'ispell-region))
	((string-equal file (TeX-region-file))
	 (call-interactively 'spell-region))
	((fboundp 'ispell-buffer)
	 (ispell-buffer))
	((fboundp 'ispell)
	 (ispell))
	(t 
	 (spell-buffer))))

(provide 'tex-init)

;;; tex-init.el ends here
