/*
 *		  mewencode, mewdecode, and mewcat
 *
 *	     Copyright (C) 1994, 1995  Kazuhiko Yamamoto
 *
 * This software conforms GNU GENERAL PUBLIC LICENSE Version 2.
 *
 * Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
 * Created: December 8, 1994
 *
 */

/*
  ****************************************************************
  Copyright (c) 1991 Bell Communications Research, Inc. (Bellcore)

  Permission to use, copy, modify, and distribute this material 
  for any purpose and without fee is hereby granted, provided 
  that the above copyright notice and this permission notice 
  appear in all copies, and that the name of Bellcore not be 
  used in advertising or publicity pertaining to this 
  material without the specific, prior written permission 
  of an authorized representative of Bellcore.  BELLCORE 
  MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY 
  OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", 
  WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
  ****************************************************************
*/

static char version_message[] = "mewencode version 0.04 02/06/95 Kazuhiko Yamamoto";

#include "getopt.h"
#include <stdio.h>
#include <string.h>

#define ENCODE_NAME "mewencode"
#define DECODE_NAME "mewdecode"
#define CAT_NAME    "mewcat"

#define SUCCESS 0
#define ERROR 1

#define FILESEP    '/'
#define FILESEPLEN 1

#define LINELEN 71

#define BASE64	'b'
#define QP     	'q'
#define GZIP64	'z'

#define ON  1
#define OFF 0

#define CR 13
#define LF 10

int Incr   = OFF;
int Ineof  = OFF;
int Outcr  = OFF;
int Outeof = OFF;

/*
 * long name convention for option
 */

struct option longopts [] = {
    {"decode", 	         0, 0, 'd'},
    {"base64",           0, 0, 'b'},
    {"quoted-printable", 0, 0, 'q'},
    {"gzip64",		 0, 0, 'z'},
    {"length", 		 1, 0, 'l'},
    {"text",		 0, 0, 't'},
    {"help",		 0, 0, 'h'},
    {"version",		 0, 0, 'v'},    
    {0, 0, 0, 0}
};


usage(progname)
	char* progname;
{
    fprintf(stderr, "usage: %s [-d] [-b|-q|-g] [-l length] [-t] [infile [outfile]]\n", progname);
}

char *help_message[] = {
    " -d  --decode           Decoding <infile> instead of encoding",
    "                        Decoding is default", 
    "                        when called with decoding program name.", 
    " -b  --base64           MIME base64 en/decoding.",
    " -q  --quoted-printable MIME quoted-printable en/decoding.",
    " -g  --gzip64           MIME gzip64 en/decoding(not yet specified).",
    " -z                     Same as -g.",
    " -l  --length           Line length into which base/gzip64 encoding truncate.",
    "                        Default value is 71.",
    " -t  --text             On base/gzip64 encoding,",
    "                        local newline is treated as CRLF.",
    "                        On base/gzip64 decoding,",
    "                        any newline is translated into local newline.",
    "                        Specify this option only when",
    "                        Content-Type: is text/plain.",
    " -h  --help             Display this help message.",
    " -v  --version          Display the version.",
    "",
    "Default is Encoding, Base64, Length = 71, Binary.",
    0
};

help(progname)
	char *progname;
{
    char **p = help_message;

    fprintf(stderr, "help: %s\n", progname);
    usage(progname);
    while (*p) fprintf(stderr, "%s\n", *p++);
}

version(progname)
	char *progname;
{
    fprintf(stderr, "version of %s: %s\n", progname, version_message);
}

/*
 * utils
 */

char *basename(filename)
	char *filename;
{
    char *p;

    if ((p = strrchr(filename, FILESEP)) != NULL) filename = p + FILESEPLEN;
    return filename;
}

/*
 * basic input/output
 */

/*
 * 8bit input with line canonicalization
 */

int Getc(stream, text)
	FILE *stream;
	int text;
{
    int c;
    
    if (!text) return(getc(stream));
    
    if (Ineof) {
	return(EOF);

    }
    c = getc(stream);
    
    if (Incr) {
	Incr  = OFF;	
	switch (c) {
	case EOF:
	    Ineof = ON;
	    return(LF);
	    break;
	case LF:
	    return(LF);
	    break;
	default:
	    ungetc(c, stream);
	    return(LF);
	    break;
	}
    }
    if (c == CR) {
	Incr = ON;
	return(CR);
    }
    if (c == LF) {
	ungetc(c, stream);
	Incr = ON;
	return(CR);
    }
    if (c == EOF) Ineof = ON;
    return(c);
}

/*
 * lineless 'ascii' input
 */

int GetChar(stream)
	FILE *stream;
{
    int c;

    if (Ineof) return EOF;
    
    do {
	c = getc(stream);
    } while ( c == CR || c == LF);

    if ( c == EOF ) Ineof = ON;
    
    return(c);
}
	
int PutChar(c, stream, text)
	int c;
	FILE *stream;
	int text;
{
    if (!text) {
	if (c != EOF) putc(c, stream);
	return;
    }

    /* text */
    
    if (Outeof) return;
    
    if (c == EOF) {
	Outeof = ON;
	Outcr = OFF; /* xxx */
	return;
    }

    if (Outcr) {
	Outcr = OFF;
	switch (c) {
	case LF : 
	    break;
	case CR : 
	    putc(LF, stream);
	    Outcr = ON;
	    break;
	default:
	    putc(c, stream);
	    break;
	}
	return;
    }

    switch (c) {
    case CR : 
	putc(LF, stream);
	Outcr = ON;
	break;
    default:
	putc(c, stream);
	break;
    }
    
}

/*
 * Base 64
 */

static char base64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

#define PADDING '='

base256to64 (c1, c2, c3, padding, outfile)
	int c1, c2, c3, padding;
	FILE *outfile;
{
    putc(base64[c1>>2], outfile);
    putc(base64[((c1 & 0x3)<< 4) | ((c2 & 0xF0) >> 4)], outfile);
    switch (padding) {
    case 0:
	putc(base64[((c2 & 0xF) << 2) | ((c3 & 0xC0) >>6)], outfile);
	putc(base64[c3 & 0x3F], outfile);
	break;
    case 1:
	putc(base64[((c2 & 0xF) << 2) | ((c3 & 0xC0) >>6)], outfile);
	putc(PADDING, outfile);
	break;
    case 2:
	putc(PADDING, outfile);
	putc(PADDING, outfile);
	break;
    }
}

base64_encode(infile, outfile, text, length)
	FILE *infile;
	FILE *outfile;
	int text, length;
{
    int c1, c2, c3, len = 0;
    
    while ((c1 = Getc(infile, text)) != EOF) {
	if ((c2 = Getc(infile, text)) == EOF)
	    base256to64(c1, 0, 0, 2, outfile);
	else if ((c3 = Getc(infile, text)) == EOF)
	    base256to64(c1, c2, 0, 1, outfile);
	else
	    base256to64(c1, c2, c3, 0, outfile);
	len += 4;
	if (len > length) {
	    putc('\n', outfile);
	    len = 0;
	}
    }
    if (len) putc('\n', outfile);
    fflush(outfile);
    
}

#define OOB -1
#define PAD -2	

static char base256[] = {
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    
  /*                                                -                /*/
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB, 62, OOB,OOB,OOB, 63,
      
  /*  0   1   2   3    4   5   6   7    8   9                =        */
     52, 53, 54, 55,  56, 57, 58, 59,  60, 61,OOB,OOB, OOB,PAD,OOB,OOB,
      
  /*      A   B   C    D   E   F   G    H   I   J   K    L   M   N   O*/
    OOB,  0,  1,  2,   3,  4,  5,  6,   7,  8,  9, 10,  11, 12, 13, 14,
      
  /*  P   Q   R   S    T   U   V   W    X   Y   Z                     */      
     15, 16, 17, 18,  19, 20, 21, 22,  23, 24, 25,OOB, OOB,OOB,OOB,OOB,
      
  /*      a   b   c    d   e   f   g    h   i   j   k    l   m   n   o*/
    OOB, 26, 27, 28,  29, 30, 31, 32,  33, 34, 35, 36,  37, 38, 39, 40,
      
  /*  p   q   r   s    t   u   v   w    x   y   z                     */
      41, 42, 43, 44,  45, 46, 47, 48,  49, 50, 51,OOB, OOB,OOB,OOB,OOB, 
};


base64_decode(infile, outfile, text, length)
	FILE *infile;
	FILE *outfile;
	int text, length;	/* length is ignored */
{
    int c1, c2, c3, c4;
    
    while ((c1 = GetChar(infile)) != EOF) {
	if ((c1 = base256[c1]) == OOB) {
	    fprintf(stderr,
		    "Warning: Base64 decoder saw an illegal character.\n");
	    exit(ERROR);
	}


	if ((c2 = GetChar(infile)) == EOF) {
	    fprintf(stderr, "Warning: Base64 decoder saw premature EOF.\n");
	    exit(ERROR);
	} else if ((c2 = base256[c2]) == OOB) {
	    fprintf(stderr,
		    "Warning: Base64 decoder saw an illegal character.\n");
	    exit(ERROR);
	}

	PutChar(((c1 << 2) | ((c2 & 0x30) >> 4)), outfile, text);

	if ((c3 = GetChar(infile)) == EOF) {
	    fprintf(stderr, "Warning: Base64 decoder saw premature EOF.\n");
	    exit(ERROR);
	} else if ((c3 = base256[c3]) == OOB) {
	    fprintf(stderr,
		    "Warning: Base64 decoder saw an illegal character.\n");
	    exit(ERROR);
	}
	
	
	if( c3 == PAD) {
	    break;
	} else 
	    PutChar((((c2 & 0XF) << 4) | ((c3 & 0x3C) >> 2)), outfile, text);
	    
	if ((c4 = GetChar(infile)) == EOF) {
	    fprintf(stderr, "Warning: Base64 decoder saw premature EOF.\n");
	    exit(ERROR);
	} else if ((c4 = base256[c4]) == OOB) { 
	    fprintf(stderr,
		    "Warning: Base64 decoder saw an illegal character.\n");
	    exit(ERROR);
	}
	    	    
	if( c4 == PAD) {
	    break;
	} else 
	    PutChar((((c3 & 0x03) << 6) | c4), outfile, text);
    }

    PutChar(EOF);
}

/*
 * Quoted_Printable
 */

static char base16[] = "0123456789ABCDEF";

#define EQ   '='
#define TAB   9
#define SP   32
#define DEL 127
#define softbreak(stream) {putc(EQ, stream); putc(LF, stream);}

quoted_printable_encode(infile, outfile, text, length)
	FILE *infile;
	FILE *outfile;
	int text, length;
{
    int c, len = 0, sp = OFF;
    while ((c = getc(infile)) != EOF) {
	if ((c == TAB) || (c == SP)) {
	    sp = ON;
	    putc(c, outfile);
	    if ((++len) > length) {
		sp = OFF;
		len = 0;
		softbreak(outfile);
	    }
	    continue;
	}
	if (c == LF) {
	    if (sp) softbreak(outfile);
	    len = 0;
	    sp = OFF;    
	    putc(LF, outfile);
	    continue;
	}
	if ((c < SP) || (c == EQ) || (c >= DEL)) { /* exclusive TAB, SP */
	    sp = OFF;
	    putc(EQ, outfile);
	    putc(base16[c >> 4], outfile);
	    putc(base16[c & 0x0f], outfile);
	    len += 3;
	    if (len > length) {
		len = 0;
		softbreak(outfile);
	    }
	    continue;
	}
	sp = OFF;
	putc(c, outfile);
	if ((++len) > length) {
	    len = 0;
	    softbreak(outfile);
	}
    }
    if (len > 0) softbreak(outfile);  /* ignored by decoder */
}

static char base128[] = {
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
      0,  1,  2,  3,   4,  5,  6,  7,   8,  9,OOB,OOB, OOB,OOB,OOB,OOB,
    OOB, 10, 11, 12,  13, 14, 15,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    OOB, 10, 11, 12,  13, 14, 15,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
    OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB, OOB,OOB,OOB,OOB,
};

puthexchar(c1, c2, stream)
	int c1, c2;
	FILE *stream;
{
    if ((c1 = base128[c1]) == OOB) {
	fprintf(stderr, "Error: can't translate hex to character.\n");
	exit(ERROR);
    }
    if ((c2 = base128[c2]) == OOB) {
	fprintf(stderr, "Error: can't translate hex to character.\n");
	exit(ERROR);
    }
    putc(((c1 << 4) | c2), stream);
       
}

quoted_printable_decode(infile, outfile, text, length)
	FILE *infile;
	FILE *outfile;
	int text, length;	/* length is ignored. */
{
    int c1, c2, c3;

    while((c1 = getc(infile)) != EOF) {
	if (c1 == EQ) {
	    if ((c2 = getc(infile)) == EOF) {
		fprintf(stderr, "Error End of file after =.\n");
		exit(ERROR);
	    }
	    if (c2 == LF) continue;
	    if ((c3 = getc(infile)) == EOF) {
		fprintf(stderr, "Error End of file after =.\n");
		exit(ERROR);
	    }
	    puthexchar(c2, c3, outfile);
	    continue;
	}
	putc(c1, outfile);
    }
}

/*
 * Gzip 64
 */

#define READ  0
#define WRITE 1

gzip64_encode(infile, outfile, text, length)
	FILE *infile;
	FILE *outfile;
	int text, length;
{
    int pipes[2];
    int childpid;
    
    if (pipe(pipes) != 0) {
	fprintf(stderr, "Can't open pipe\n");
	exit(ERROR);
    }

    childpid = fork ();

    if (childpid < 0) {
	fprintf(stderr, "Can't fork.\n");
	exit(ERROR);
    }

    if (childpid > 0) { /* I'm the parent. */
	close(pipes[WRITE]);
	if ((infile = fdopen(pipes[READ], "r")) == NULL) {
	    fprintf(stderr, "Can't open read pipe\n");
	    exit(ERROR);
	}

	base64_encode(infile, outfile, OFF, length);

	exit(SUCCESS);
    }

    /* I'm the child. */

    close(WRITE);
    dup(pipes[WRITE]);
    close(pipes[READ]);
	
    if (text == OFF) {
	
	if (fileno(infile) != READ) {
	    close(READ);
	    dup(fileno(infile));
	}
	
	execlp("gzip", "gzip", (char *) 0);
	
	exit(SUCCESS);

    } else {

	int pipes2[2];
	int childpid2;
	
	if (pipe(pipes2) != 0) {
	    fprintf(stderr, "Can't open pipe\n");
	    exit(ERROR);
	}
	
	childpid2 = fork ();
	
	if (childpid2 < 0) {
	    fprintf(stderr, "Can't fork.\n");
	    exit(ERROR);
	}
	
	if (childpid2 > 0) { /* I'm the child. */
	    close(pipes2[WRITE]);
	    close(READ);
	    dup(pipes2[READ]);

	    execlp("gzip", "gzip", (char *) 0);
	    
	    exit(SUCCESS);
	}

	/* I'm the grandchild */

	close(WRITE);
	dup(pipes2[WRITE]);
	close(pipes2[READ]);
	
	{
	    int c;
	    while ((c = Getc(infile, ON)) != EOF)
		putchar(c);
	}

	exit(SUCCESS);
	
    }
    
}

gzip64_decode(infile, outfile, text, length)
	FILE *infile;
	FILE *outfile;
	int text, length;	/* length is ignored. */
{
    int pipes[2];
    int childpid;
    
    if (pipe(pipes) != 0) {
	fprintf(stderr, "Can't open pipe\n");
	exit(ERROR);
    }

    childpid = fork ();

    if (childpid < 0) {
	fprintf(stderr, "Can't fork.\n");
	exit(ERROR);
    }

    if (childpid > 0) { /* I'm the parent. */
	close(READ);
	dup(pipes[READ]);

	close(pipes[WRITE]);

	if (text == ON) {
	    int c;
	    while ((c = getchar()) != EOF) PutChar(c, outfile, ON);
	} else {
	    if (fileno(outfile) != WRITE) {
		close(WRITE);
		dup(fileno(outfile));
	    }
	    execlp ("gzip", "gzip", "-d", (char *) 0);
	}
	
	exit(SUCCESS);

    }

    /* I'm the child. */
    
    close(pipes[READ]);

    if (text == ON) {
	int pipes2[2];
	int childpid2;
	
	if (pipe(pipes2) != 0) {
	    fprintf(stderr, "Can't open pipe\n");
	    exit(ERROR);
	}

	childpid2 = fork ();

	if (childpid2 < 0) {
	    fprintf(stderr, "Can't fork.\n");
	    exit(ERROR);
	}

	if (childpid2 > 0) { /* I'm the child. */
	    close(pipes2[WRITE]);
	    close(WRITE);
	    dup(pipes[WRITE]);
	    close(READ);
	    dup(pipes2[READ]);

	    execlp("gzip", "gzip", "-d", (char *) 0);
	
	    exit(SUCCESS);	    
	}

	/* I'm the grandchild */

	close(pipes2[READ]);

	if ((outfile = fdopen(pipes2[WRITE], "w")) == NULL) {
	    fprintf(stderr, "Can't open write pipe\n");
	    exit(ERROR);
	}
	
	base64_decode(infile, outfile, OFF, length);

	exit(SUCCESS);

    } else {

	if ((outfile = fdopen(pipes[WRITE], "w")) == NULL) {
	    fprintf(stderr, "Can't open write pipe\n");
	    exit(ERROR);
	}
	
	base64_decode(infile, outfile, OFF, length);
	
	exit(SUCCESS);
    }
}

/*
 * main
 */

int main (argc, argv)
	int argc;
	char **argv;
{
    int optc;
    FILE *infile;
    FILE *outfile;
    int file_count = 0;
    char *progname = basename(argv[0]);

    /*
     * default option values
     */

    int  decode = OFF;			/* -d */
    char encode = BASE64;		/* -b -q -g */
    int  length = LINELEN;		/* -l num */
    int  text   = OFF;			/* -t */


    while((optc = getopt_long(argc, argv, "dbqgzl:thv", longopts,
			      (int *)0)) != EOF) {
	switch (optc) {
	case 'd':
	    decode = ON;
	    break;
	case 'b':
	    encode = BASE64;
	    break;
	case 'q':
	    encode = QP;
	    break;
	case 'g':
	case 'z':
	    encode = GZIP64;
	    break;
	case 'l':
	    length = atoi(optarg);
	    break;
	case 't':
	    text = ON;
	    break;
	case 'h':
	    help(progname);
	    exit(SUCCESS);
	    break;
	case 'v':
	    version(progname);
	    exit(SUCCESS);
	    break;
	default:
	    usage(progname);
	    exit(ERROR);
	}
    }

    file_count = argc - optind;

    switch(file_count) {
    case 0:
	infile  = stdin;
	outfile = stdout;
	break;
    case 1:
	if ((infile  = fopen(argv[optind], "r")) == NULL) {
	    fprintf(stderr, "Can't open file %s.\n", argv[optind]);
	    exit(ERROR);
	}
	outfile = stdout;
	break;
    case 2:
	if ((infile  = fopen(argv[optind], "r")) == NULL) {
	    fprintf(stderr, "Can't open file %s.\n", argv[optind]);
	    exit(ERROR);
	}
	optind++;
	if ((outfile = fopen(argv[optind], "w")) == NULL) {
	    fprintf(stderr, "Can't open file %s.\n", argv[optind]);
	    exit(ERROR);
	}
	break;
    default:
	usage(progname);
	exit(ERROR);
	break;
    }

    /* Override argments by progname. */
    
    if (strcmp(progname, DECODE_NAME) == 0) decode = ON;
    if (strcmp(progname, CAT_NAME) == 0) {
	decode = ON;
	outfile = stdout;
    }


    if (decode) {
	switch (encode) {
	case BASE64:
	    base64_decode(infile, outfile, text, length);
	    break;
	case QP:
	    quoted_printable_decode(infile, outfile, text, length);
	    break;
	case GZIP64:
	    gzip64_decode(infile, outfile, text, length);
	    break;
	}
    } else {
	switch (encode) {
	case BASE64:
	    base64_encode(infile, outfile, text, length);
	    break;
	case QP:
	    quoted_printable_encode(infile, outfile, text, length);
	    break;
	case GZIP64:
	    gzip64_encode(infile, outfile, text, length);
	    break;
	}
    }

    exit(SUCCESS);
    
}
