;;;			      mew-pem.el
;;;
;;;  Copyright (C) 1994, 1995  Mine Sakurai and Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Mine Sakurai     <m-sakura@ccs.mt.nec.co.jp>
;;;          Kazuiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: June  13, 1994
;;; Revised: March  4, 1995
;;;

(defconst mew-pem-version "mew-pem.el version 0.07")

(provide 'mew-pem)
(require 'mew)

(defvar mew-pem-keyserv "keyserv.fujitsu.co.jp"
  "*Key server name")

(defmacro mew-pem-getenv (env default)
  (` (let ((val (getenv (, env))))
       (if val val (expand-file-name (, default) mew-pem-home-dir)))))

(defvar mew-pem-home-dir (getenv "HOME"))
(defvar mew-pem-cert-key    (mew-pem-getenv "PEMKEY" ".key"))
(defvar mew-pem-cert-cert   (mew-pem-getenv "PEMCERT" ".cert"))
(defvar mew-pem-cert-certs  (mew-pem-getenv "PEMCERTS" ".certs"))
(defvar mew-pem-cert-issuer (mew-pem-getenv "PEMISSUER" ".issuer"))

(defvar mew-prog-enpem   "/usr/local/lib/pem/enpem")
(defvar mew-prog-depem   "/usr/local/lib/pem/depem")
(defvar mew-prog-whois	 "whois")

(defconst mew-pem-boundary-begin
  "-----BEGIN PRIVACY-ENHANCED MESSAGE-----")
(defconst mew-pem-boundary-end
  "-----END PRIVACY-ENHANCED MESSAGE-----")
(defconst mew-pem-report-begin
  "----- FJPEM INTEGRITY REPORT -----")
(defconst mew-pem-report-end 
  "----- FJPEM REPORT END -----")
(defconst mew-pem-content-type
  "application/pem-1421")


(cond 
 ((boundp 'MULE) 
  (define-program-coding-system nil mew-prog-enpem *iso-2022-jp*)
  (define-program-coding-system nil mew-prog-depem *iso-2022-jp*)
  )
 )

;;;
;;; PEM
;;;

;;
;; public key oparations
;;

(defun mew-pem-put-pkey (from)
  "Save public key at the point to file mew-pem-cert-certs/FROM"
  (if (string= from (format "%s@%s"(user-login-name) mew-mail-domain))
      ()
    (let ((case-fold-search t)
	  (p nil) 
	  (cert "")
	  (buf nil))
      (re-search-forward "^Originator-Certificate:")
      (forward-line 1)
      (while (looking-at "^ ")
	(setq p (point))
	(end-of-line)
	(forward-char) ; for \n
	(setq cert (concat cert (buffer-substring p (point)))))
      (save-excursion
	(setq buf (find-file-noselect 
		   (expand-file-name from mew-pem-cert-certs)))
	(set-buffer buf)
	(erase-buffer)
	(insert cert)
	(save-buffer)
	)
      (and buf (kill-buffer buf))
      )
    ))

(defun mew-get-user-pkeys (ul)
  "Retrieves public keys of e-mail address list UL."
  (if (null (car ul))
      ()
    (if (not (file-exists-p (expand-file-name (car ul) mew-pem-cert-certs)))
	(mew-get-pkey (car ul)))
    (mew-get-user-pkeys (cdr ul))))


(defun mew-get-pkey (name)
  "Retrieves NAME's public key and save in file mew-pem-cert-certs/NAME
if not exists."
    (if (mew-whois mew-pem-keyserv name)
	(let ((buf))
	  (save-excursion
	    (setq buf (find-file-noselect 
		       (expand-file-name name mew-pem-cert-certs)))
	    (set-buffer buf)
	    (erase-buffer)
	    (insert-buffer mew-buffer-tmp)
	    (save-buffer)
	    )
	  (or buf (kill-buffer buf))
	  )
      (error "Can't get %s's public key. Try signup (C-cC-s)." name)
      )
    )


;;
;; WHOIS interface to Key Server
;;

(defun mew-whois (host key)
  "Send whois command to HOST with key KEY.
It returns t if success, otherwise reterns nil."
  (save-excursion
    (if (and host (not (string= host "")))
	(progn
	  (set-buffer mew-buffer-tmp)
	  (erase-buffer)
	  (call-process mew-prog-whois nil t nil 
			"-h" mew-pem-keyserv
			(concat "to:" key))
	  (goto-char (point-min))
	  (if (re-search-forward "ERROR" nil t) nil t)
	  )
      nil
      )
    ))

;;
;; Decription, encription, and sign.
;;

(defun mew-pem-decrypt-region (beg end)
  "Decrypt region between BEG and END."
  (interactive "r")
  (if (file-exists-p mew-prog-depem)
      (save-excursion
	(let ((from (mew-header-canform 
		     (mew-header-extract-addr 
		      (mew-field-get-value "From:")))))
	  (goto-char beg) ; move where pulic key is contained
	  (mew-pem-put-pkey from)
	  (message "PEM decrypting ...")
	  (undo-boundary)
	  (call-process-region beg end
			       mew-prog-depem t t nil
			       "-i"
			       "-c" mew-pem-cert-cert 
			       "-k" mew-pem-cert-key 
			       "-I" mew-pem-cert-issuer
			       "-m" from
			       )
	  (undo-boundary) ;this is necessary.
	  (message "PEM decrypting ... done")
;	  (if (mew-pem-decrypt-error-check)
;	      (message "PEM decrypting ... done")
;	    (message "PEM decrypting ... failed"))
	  ))))

;(defun mew-pem-decrypt-error-check ()
;  "If PEM decryption is success, it returns t.
;Otherwise, it does undo and returns nil."
;    (goto-char (point-max))
;    (previous-line 1)
;    (push-mark (point))
;    (end-of-line)
;    (let ((lastline (buffer-substring (mark) (point)))
;	  (errormessage nil))
;      (if (string= mew-pem-report-end lastline)
;	  t
;	(push-mark (point))
;	(re-search-backward "^$" nil t)
;	(forward-line 1)
;	(setq errormessage (buffer-substring (point) (mark)))
;	(undo)
;	(goto-char (point-max))
;	(insert "\n<<<<< PEM error message is below. >>>>>\n")
;	(insert errormessage)
;	nil
;	)
;      ))


(defun mew-pem-sign-region (beg end)
  "Sign region between BEG and END.
Only your secret key is necessary."
  (interactive "r")
  (if (file-exists-p mew-prog-enpem)
      (call-process-region beg end
			   mew-prog-enpem t t nil
			   "-m" 
			   "-k" mew-pem-cert-key 
			   "-c" mew-pem-cert-cert 
			   "-i" mew-pem-cert-issuer)))


(defun mew-pem-encrypt-region (beg end)
  "Encrypt region between BEG and END.
Retrieve public keys from key server if you don't have."
  (interactive "r")
  (let ((users))
    (cond ((file-exists-p mew-prog-enpem)
	   (setq users 
		 (mew-header-canform-list
		  (mew-header-expand-alias-list
		   (mew-header-address-collect '("To:" "Cc:")))))
	   (mew-get-user-pkeys users)
	   (apply (function call-process-region)
		  beg end
		  mew-prog-enpem t t nil
		  "-e" 
		  "-k" mew-pem-cert-key 
		  "-c" mew-pem-cert-cert 
		  "-i" mew-pem-cert-issuer
		  (mapcar 
		   (function 
		    (lambda (list) (expand-file-name list mew-pem-cert-certs)))
		   users)
		  )))
    ))

(defun mew-pem-decrypt-letter ()
  (interactive)
  (if (file-exists-p mew-prog-depem)
      (save-excursion
	(set-buffer (mew-buffer-message))
	(let ((buffer-read-only nil)
	      (from (mew-header-canform 
		     (mew-header-extract-addr 
		      (mew-field-get-value "From:")))))
	  (goto-char (point-min)) ; move where pulic key is contained
	  (mew-pem-put-pkey from)
	  (goto-char (point-min))
	  (re-search-forward "^-*$" nil t)
	  (forward-line 1)
	  (message "PEM decrypting ...")
	  (undo-boundary)
	  (call-process-region (point) (point-max)
			       mew-prog-depem t t nil
			       "-i"
			       "-c" mew-pem-cert-cert 
			       "-k" mew-pem-cert-key 
			       "-I" mew-pem-cert-issuer
			       "-m" from
			       )
	  (undo-boundary) ;this is necessary.
	  (message "PEM decrypting ... done")
	  )
	)
    ))

(defun mew-pem-sign-letter ()
  "Sign the draft letter."
  (interactive)
  (if (null (file-exists-p mew-prog-enpem))
      ()
    (goto-char (marker-position mew-draft-buffer-header))
    (forward-line 1)
    (message "PEM signing ...")
    (mew-pem-sign-region (point) (point-max))
    (message "PEM signing ... done")
    (goto-char (marker-position mew-draft-buffer-header))
    (mew-field-insert-here "Content-Type:" mew-pem-content-type)
    (beginning-of-line)
    )
  )

(defun mew-pem-encrypt-letter ()
  "Encrypt the draft letter."
  (interactive)
  (if (null (file-exists-p mew-prog-enpem))
      ()
    (goto-char (marker-position mew-draft-buffer-header))
    (forward-line 1)
    (message "PEM encrypting ...")
    (mew-pem-encrypt-region (point) (point-max))
    (message "PEM encrypting ... done")
    (goto-char (marker-position mew-draft-buffer-header))
    (mew-field-insert-here "Content-Type:" mew-pem-content-type)
    (beginning-of-line)
    )
  )

