;;;			      mew-pgp.el
;;;
;;;	     Copyright (C) 1994, 1995  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: August 17, 1994
;;; Revised: March   4, 1995
;;;

(defconst mew-pgp-version "mew-pgp.el version 0.12")

(provide 'mew-pgp)
(require 'mew)

(defvar mew-prog-pgp   "pgp")
(defvar mew-prog-shell "/bin/sh")

(defvar mew-pgp-encrypt-only-message
  "\nThis is encrypted without his/her signature.\n")
(defvar mew-pgp-buffer-tmp "*Mew PGP*")

(defvar mew-pgp-content-type
  "application/pgp")

(defvar mew-pgp-encode-switch 
  '((signature  . "-staf +clearsig=on +language=en")
    (encryption . "-etaf +language=en")
    (sign-enc   . "-estaf +language=en"))
  )

(defvar mew-pgp-decode-args "-f +language=en")

(defvar mew-pgp-report-boundary1 "---- PGP Message --")
;; \n is not necessary since the message starts with \n.
(defvar mew-pgp-report-boundary2 "---- PGP Message ----\n")

(defvar mew-pgp-decode-message nil)
(defvar mew-pgp-tmp-read-file nil)
(defvar mew-pgp-tmp-write-file nil)
(defvar mew-pgp-tmp-begin nil)
(defvar mew-pgp-tmp-end nil)

(defun mew-read-passwd (prompt)
  (let ((pass)
	(c 0)
	(echo-keystrokes 0)
	(ociea cursor-in-echo-area))
    (unwind-protect
	(progn
	  (setq cursor-in-echo-area 1)
	  (while (and (/= c ?\r) (/= c ?\n) (/= c ?\e) (/= c 7)) ;; ^G
	    (message "%s%s"
		     prompt
		     (make-string (length pass) ?.))
	    (setq c (read-char))
	    (if (= c ?\C-u)
		(setq pass "")
	      (if (and (/= c ?\b) (/= c ?\177))
		  (setq pass (concat pass (char-to-string c)))
		(if (> (length pass) 0)
		    (setq pass (substring pass 0 -1))))))
	  (setq cursor-in-echo-area -1)
	  )
      (setq cursor-in-echo-area ociea)
      nil)
    (message "")
    (sit-for 0)
    (substring pass 0 -1)))

;;;
;;; Draft mode
;;;

;;
;; PGP encoding
;;

(defvar mew-pgp-encode-failure nil)

(defun mew-pgp-sign-letter ()
  (interactive)
  (if (mew-field-get-value "Content-Type:")
      (message "You can't sign this draft due to Content-Type:")
    (mew-pgp-encode-letter 'signature))
  )

(defun mew-pgp-encrypt-letter ()
  (interactive)
  (if (mew-field-get-value "Content-Type:")
      (message "You can't encrypt this draft due to Content-Type:")
    (mew-pgp-encode-letter 'encryption))
  )

(defun mew-pgp-sign-encrypt-letter ()
  (interactive)
  (if (mew-field-get-value "Content-Type:")
      (message "You can't sign then ecrypt this draft due to Content-Type:")
    (mew-pgp-encode-letter 'sign-enc))
  )

(defun mew-pgp-encode-letter (action)
  (if (null (mew-which mew-prog-pgp exec-path))
      ()
    (goto-char (marker-position mew-draft-buffer-header))
    (forward-line 1)
    (if (mew-pgp-encode-region (point) (point-max) action)
	(progn
	  (goto-char (marker-position mew-draft-buffer-header))
	  (mew-field-insert-here "Content-Type:" mew-pgp-content-type)
	  )
      )
    ))

(defun mew-pgp-encode-region (beg end action)
  (let (users process
	(call-process-hook nil) ;; you aren't expected to understand this.
	(default-process-coding-system
	  (if (boundp 'MULE)
	      (cons mew-mule-charset-integrity
		    mew-mule-charset-integrity)
	    nil))
	)
    (if (null (mew-which mew-prog-pgp exec-path))
	(message "pgp is not found")
      (cond
       ((or (equal action 'encryption) (equal action 'sign-enc))
	(if (setq users 
		  (mew-header-canform-list
		   (mew-header-expand-alias-list
		    (mew-header-address-collect '("To:" "Cc:")))))
	    (setq users 
		  (concat (mapconcat (function (lambda (x) x)) users " ")
			  ;; list to string
			  " " mew-mail-address ;; to decrypt by myself
			  ))
	  (error "Specify To: or Cc:")
	  ))
       ((equal action 'signature)
	(setq users "")
	)
       (t (error "No such action."))
       )
      (setq mew-pgp-tmp-read-file (make-temp-name "/tmp/mew"))
      (write-region beg end mew-pgp-tmp-read-file)
      ;; touch file before write-file to make them unique
      (setq mew-pgp-tmp-write-file (make-temp-name "/tmp/mew"))
      (setq mew-pgp-tmp-begin beg)
      (setq mew-pgp-tmp-end end)
      (message "PGP encoding ...")
      (setq process
	    (start-process "PGP encode"
			    (current-buffer)
			    mew-prog-shell "-c"
			    (format "%s %s %s < %s > %s"
				    mew-prog-pgp
				    (cdr (assoc action mew-pgp-encode-switch))
				    users
				    mew-pgp-tmp-read-file
				    mew-pgp-tmp-write-file)))
      ;; I seems to me that PGP on UNIX can't handle ^D from stdio.
      ;; So, tmpfile1.
      (set-process-filter process 'mew-pgp-encode-filter1)
      (set-process-sentinel process 'mew-pgp-encode-sentinel)
      ;; Wait for the termination of PGP.
      ;; Emacs doesn't provide synchronize mechanism with
      ;; an asynchronous process. So, take this way. 
      (while (not (equal (process-status process) 'exit))
	(sit-for 1) ;; give CPU time to the process filter 
	;; accept-process-output or sleep-for is not enough
	)
      (if mew-pgp-encode-failure
	  (progn
	    (setq mew-pgp-encode-failure nil)
	    nil) ;; return
	(message "PGP encoding ... done")
	t) ;; return
      )
    ))

(defun mew-pgp-encode-sentinel (process event)
  (save-excursion
    (let ((file-coding-system-for-read
	   (if mew-mule-p mew-mule-charset-integrity)))
      (set-buffer (process-buffer process))
      (if mew-pgp-encode-failure
	  (cond
	   ((eq mew-pgp-encode-failure 'nofile)
	    (message "secring doesn't exist."))
	   ((eq mew-pgp-encode-failure 'nouserid)
	    (message "userid %s is not found in secring." (user-login-name)))
	   ((eq mew-pgp-encode-failure 'nopass)
	    (message "Pass phrase is wrong"))
	   (t (message "Encoding failed for some reasons"))
	   )
	(delete-region mew-pgp-tmp-begin mew-pgp-tmp-end)
	(insert-file-contents mew-pgp-tmp-write-file))
      ;;
      (if (file-exists-p mew-pgp-tmp-read-file)
	  (delete-file mew-pgp-tmp-read-file))
      (if (file-exists-p mew-pgp-tmp-write-file)
	  (delete-file mew-pgp-tmp-write-file))
      (setq mew-pgp-tmp-read-file nil)
      (setq mew-pgp-tmp-write-file nil)
      (setq mew-pgp-tmp-begin nil)
      (setq mew-pgp-tmp-end nil)
      )
    ))

  
(defun mew-pgp-encode-filter1 (process string)
  (cond
   ((string-match "Enter pass phrase: $" string) ;; sign or sign-enc
    (process-send-string
     process
     (format "%s\n" (mew-read-passwd "Enter pass phrase : ")))
    (set-process-filter process 'mew-pgp-encode-filter2))
   ((string-match "Keyring file" string) ;; no secring
    (setq mew-pgp-encode-failure 'nofile)
    (set-process-filter process 'mew-pgp-encode-filter3))
   ((string-match "Key matching userid" string) ;; no userid in secring
    (setq mew-pgp-encode-failure 'nouserid)
    (set-process-filter process 'mew-pgp-encode-filter3))
   (t ()) ;; enc
   )
  )

(defun mew-pgp-encode-filter2 (process string)
  (cond
   ((string-match "Pass phrase is good." string)
    (message "PGP encoding ... ")
    (set-process-filter process 'mew-pgp-encode-filter3)
    )
   ((string-match "No passphrase" string)
    ;; pass phrases are wrong three times
    (setq mew-pgp-encode-failure 'nopass)
    (set-process-filter process 'mew-pgp-encode-filter3))
   ((string-match "Bad pass phrase" string)
    ;; gee, I don't know why some computer doesn't match to
    ;; "Error:  Bad pass phrase.\n\nEnter pass phrase: $"?
    (process-send-string 
     process
     (format "%s\n" (mew-read-passwd "Re-enter pass phrase : ")))
    ;; never change filter
    )
   )
  )

(defun mew-pgp-encode-filter3 (process string)
  () ;; just ignore
  )

;;
;; Distribute your public key
;;

(defun mew-pgp-insert-public-key ()
  (interactive)
  (call-process mew-prog-shell 
		nil t nil 
		"-c"
		(format "%s -kxaf %s 2> /dev/null"
			mew-prog-pgp
			mew-mail-address)
		)
  )

      
;;;
;;; PGP auto decode
;;;

(defvar mew-pgp-decode-failure nil)

(defun mew-pgp-decode-region (beg end)
  (interactive "r")
  (if (mew-which mew-prog-pgp exec-path)
      (let ((process)
	    (call-process-hook nil)
	    ;; you aren't expected to understand this.
	    (default-process-coding-system
	      (if mew-mule-p
		  (cons mew-mule-charset-integrity
			mew-mule-charset-integrity)
		nil))
	    (file-coding-system
	     (if mew-mule-p mew-mule-charset-integrity))
	    )
	(setq mew-pgp-tmp-read-file (make-temp-name "/tmp/mew"))
	(write-region beg end mew-pgp-tmp-read-file)
	;; touch file before write-file to make them unique
	(setq mew-pgp-tmp-write-file (make-temp-name "/tmp/mew"))
	(message "PGP decoding ... ")
	(setq process
	      (start-process "PGP decode"
			     (current-buffer) 
			     mew-prog-shell "-c"
			     (format "%s %s < %s > %s"
				     mew-prog-pgp
				     mew-pgp-decode-args
				     mew-pgp-tmp-read-file
				     mew-pgp-tmp-write-file)))
	;; I seems to me that PGP on UNIX can't handle ^D from stdio.
	;; So, read-file.
	(set-process-filter process 'mew-pgp-decode-filter1)
	(set-process-sentinel process 'mew-pgp-decode-sentinel)
	(delete-region beg end)
	;; Wait for the termination of PGP.
	;; Emacs doesn't provide synchronize mechanism with
	;; an asynchronous process. So, take this way. 
	(while (not (equal (process-status process) 'exit))
	  (sit-for 1) ;; give CPU time to the process filter 
	  ;; accept-process-output or sleep-for is not enough
	  )
	(message "PGP decoding ... done")
	)
    ))

(defun mew-pgp-decode-sentinel (process event)
  (save-excursion
    (let ((file-coding-system-for-read
	   (if mew-mule-p mew-mule-charset-integrity)))
      (set-buffer (process-buffer process))
      (if mew-pgp-decode-failure
	  (cond
;;	   ((eq mew-pgp-decode-failure 'nofile)
;;	    (setq mew-pgp-decode-message "pubring doesn't exist."))
	   ((eq mew-pgp-decode-failure 'nouserid)
	    (setq mew-pgp-decode-message
		  "\nThe userid not found in pubring\n"))
	   ((eq mew-pgp-decode-failure 'nopass)
	    (setq mew-pgp-decode-message "\nPass phrase is wrong.\n"))
	   (t (setq mew-pgp-decode-message
		    "Decoding failed sor some reasons."))
	   )
	(insert-file-contents mew-pgp-tmp-write-file)
	(goto-char (point-max))
	)
      (insert "\n")
      (insert mew-pgp-report-boundary1)
      (if mew-pgp-decode-message
	  (insert mew-pgp-decode-message)
	(insert mew-pgp-encrypt-only-message))
      (insert mew-pgp-report-boundary2)
      ;;
      (setq mew-pgp-decode-message nil)
      (if (file-exists-p mew-pgp-tmp-read-file)
	  (delete-file mew-pgp-tmp-read-file))
      (if (file-exists-p mew-pgp-tmp-write-file)
	  (delete-file mew-pgp-tmp-write-file))
      (setq mew-pgp-tmp-read-file nil)
      (setq mew-pgp-tmp-write-file nil)
      (if mew-pgp-decode-failure
	  (setq mew-pgp-decode-failure nil))
      )
    ))
  
(defun mew-pgp-decode-filter1 (process string)
  (cond
   ((string-match "Enter pass phrase: $" string)
    ;; decrypt, decrypt then verfy
    (process-send-string
     process
     (format "%s\n" (mew-read-passwd "Enter pass phrase : ")))
    (set-process-filter process 'mew-pgp-decode-filter2))
   ((string-match "You do not have the secret key" string)
    ;; this includes not secring.pgp
    (setq mew-pgp-decode-failure 'nouserid)
    (set-process-filter process 'mew-pgp-decode-filter4))
   ;; how about "secring.pgp doesn't exist"?  'nofile
   ((string-match "File has signature" string) ;; just verfy
    (setq mew-pgp-decode-message (concat mew-pgp-decode-message string))
    (set-process-filter process 'mew-pgp-decode-filter3))
   )
  )

(defun mew-pgp-decode-filter2 (process string)
  (cond
   ((string-match "No passphrase" string)
    ;; pass phrases are wrong three times
    (setq mew-pgp-decode-failure 'nopass)
    (set-process-filter process 'mew-pgp-decode-filter4))
   ((string-match "Pass phrase is good." string) 
    ;; decrypt, decrypt then verfy
    (message "PGP decoding ... ")
    (set-process-filter process 'mew-pgp-decode-filter3)
    )
   ((string-match "Error:  Bad pass phrase.\n\nEnter pass phrase: $" string)
    (process-send-string 
     process
     (format "%s\n" (mew-read-passwd "Re-enter pass phrase : ")))
    ;; never change filter
    )
   )
  )

(defun mew-pgp-decode-filter3 (process string)
  (save-excursion
    (set-buffer (process-buffer process))
    (cond
     ((string-match "Just a moment" string)
      ;; decrypt, decrypt then verfy
      ());; do nothing
     ((string-equal "." string) ;; decrypt
      ()) ;; do nothing
     (t ;; decrypt then verfy
      (setq mew-pgp-decode-message (concat mew-pgp-decode-message string))
      )
     )
    ))

(defun mew-pgp-decode-filter4 (process string)
  () ;; do nothing
  )

(defun mew-pgp-decode-letter ()
  (interactive)
  (if (mew-which mew-prog-pgp exec-path)
      (save-excursion
	(set-buffer (mew-buffer-message))
	(let ((buffer-read-only nil))
	  (goto-char (point-min))
	  (re-search-forward "^-*$" nil t)
	  (forward-line 1)
	  (mew-pgp-decode-region (point) (point-max))
	  ))
    ))

;;;
;;; Summary mode 
;;;

(defun mew-pgp-add-key ()
  (interactive)
  (mew-summary-display 'notforce)
  (setq mew-pgp-tmp-read-file (make-temp-name "/tmp/mew"))
  (save-excursion
    (set-buffer (mew-buffer-message))
    (save-restriction
      (widen)
      (let ((file-coding-system-for-read
	     (if mew-mule-p mew-mule-charset-local)))
	(write-region (point-min) (point-max) mew-pgp-tmp-read-file)
	(message "") ;; erase write-region message
	)))
  (mew-current-set 'window (current-window-configuration))
  (mew-window-configure mew-pgp-buffer-tmp 'summary)
  (mew-terminal-emulator mew-pgp-buffer-tmp
			 mew-prog-pgp
			 (list "-ka" mew-pgp-tmp-read-file))
  )

(defun mew-terminal-emulator (buffer program args &optional width height)
  (let ((terminal-more-break-insertion nil))
    (require 'terminal)
    (switch-to-buffer buffer)
    (if (null width) (setq width (- (window-width (selected-window)) 1)))
    (if (null height) (setq height (- (window-height (selected-window)) 1)))
    (terminal-mode)
    (setq te-width width te-height height)
    (setq mode-line-buffer-identification
	  (list (format "Emacs terminal %dx%d: %%b  " te-width te-height)
		'te-pending-output-info))
    (let ((buffer-read-only nil))
      (te-clear-screen))
    (let (process)
      (while (setq process (get-buffer-process (current-buffer)))
	(if (y-or-n-p (format "Kill process %s? " (process-name process)))
	    (delete-process process)
	  (error "Process %s not killed" (process-name process)))))
    (condition-case err
	(let ((termcap
	       (concat (format "emacs-virtual:co#%d:li#%d:%s"
			       te-width te-height (if terminal-scrolling
						      "" "ns:"))
		       "cm=^p=%+ %+ :cr=^p^a:le=^p^b:nd=^p^f:"
		       "nw=^j:ce=^pc:cd=^pC:cl=^p^l:bl=^p^g:"
		       "IC=^p_%+ :DC=^pd%+ :AL=^p^o%+ :DL=^p^k%+ :"
		       "LP:NF:"
		       "ic=^p_!:dc=^pd!:al=^p^o!:dl=^p^k!:ho=^p=  :"
		       "im=:ei=:dm=:ed=:mi:do=^p^j:nl=^p^j:bs:")))
	  (let ((process-environment
		 (cons "TERM=emacs-virtual"
		       (cons (concat "TERMCAP=" termcap)
			     process-environment))))
	    (setq te-process
		  (start-process "terminal-emulator" (current-buffer)
				 "/bin/sh" "-c"
				 (format "%s; exec %s"
					 te-stty-string
					 (mapconcat 
					  'te-quote-arg-for-sh
					  (cons program args) " ")))))
	  (set-process-filter te-process 'te-filter)
	  (set-process-sentinel te-process 'mew-te-sentinel))
      (error (fundamental-mode)
	     (signal (car err) (cdr err))))
    (setq inhibit-quit t)			;sport death
    (use-local-map terminal-map)
    ))

(defun mew-te-sentinel (process message)
  (cond ((eq (process-status process) 'run))
	((null (buffer-name (process-buffer process)))) ;deleted
	(t (save-excursion
	     (set-buffer (process-buffer process))
	     (setq buffer-read-only nil)
	     (fundamental-mode)
	     (goto-char (point-max))
	     (delete-blank-lines)
	     (delete-horizontal-space)
	     (insert "\n*******\n" message "*******\n"))
	   (if (file-exists-p mew-pgp-tmp-read-file)
	       (delete-file mew-pgp-tmp-read-file))
	   (if (mew-current-get 'window)
	       (if (null (mew-y-or-n-p "Get back to summay mode "))
		   ()
		 (set-window-configuration (mew-current-get 'window))
		 (if (get-buffer mew-pgp-buffer-tmp)
		     (kill-buffer mew-pgp-buffer-tmp))))
	   )
	)
  )

;;
;; End of file
;;
