;;;			    mew-refile.el
;;;
;;;		 Copyright (C) 1994  Yoshinari NOMURA
;;;
;;;		   This emacs lisp library confirms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Yoshinari NOMURA <nomsun@csce.kyushu-u.ac.jp>
;;; Created: Jun.  11, 1994
;;; Revised: Feb.  28, 1995
;;;

(defconst mew-refile-version "mew-refile.el version 0.19")

(provide 'mew-refile)
(require 'mew)

(defvar mew-auto-folder-alist nil
  "*If non nil, mew guesses destination folder by using this hint.
The format is like this:

        (setq mew-auto-folder-alist
              '((\"To:\"   (\"wide@wide\"    .  \"+wide/wide\")
                         (\"adam\"         .  \"+labo/adam\"))
                (\"Newsgroups:\"
                         (\"^nifty\\\\.\\\\([^ ]+\\\\)\" . \"+Nifty/\\\\1\"))
                (\"From:\" (\"uucp\"         .  \"+adm/uucp\" )
                         (\".*\"           .  \"+misc\"     ))))
")

(defvar mew-msg-id-alist nil)
(defvar mew-msg-id-file-name ".mew-id-alist")

(defvar mew-from-alist nil)
(defvar mew-from-file-name ".mew-from-alist")

(defvar mew-refile-folder-guess-functions '(
					    mew-guess-by-alist
					    mew-guess-by-folder
					    mew-guess-by-message-id
					    mew-guess-by-from
					    mew-guess-by-default
					    ))

(defvar mew-summary-refile-last-destination nil)

(defvar mew-alist-max-length 1000
  "*Max length of mew-from-alist and mew-msg-id-alist.")

;;
;; mew-summary-refile
;;
(defun mew-summary-refile (&optional last-folder)
  (interactive)
  (let (msg-buffer msg hit learn-info tofolder todir)
    (cond 
     ((mew-summary-part-number)
      (mew-summary-message-up t) ;; no direction change
      (mew-summary-display))
     )
    (setq msg (mew-summary-message-number))
    (if (mew-summary-marked-p)
	(cond
	 ((not (mew-summary-message-number))
	  (message "No message"))
	 ((setq hit (cdr (assoc msg mew-summary-buffer-refile)))
	  (message (format "Already bound to %s" hit)))
	 (t (message "Already marked"))
	 )
      (setq msg-buffer (or (mew-summary-display t)
			   ;; notforce
			   ;; to avoid scroll up
			   (mew-buffer-message)))
      (save-excursion
	(set-buffer msg-buffer)
	(setq learn-info (mew-refile-folder-guess)))
      (setq tofolder (or last-folder (mew-input-folder (car learn-info)))
	    todir (mew-folder-to-dir tofolder))
      (if (and (not (mew-member todir mew-folder-list))
	       (or
		(file-exists-p (concat mew-path "/" todir))
		(mew-y-or-n-p 
		 (format "No folder %s exists. Create it? " tofolder))))
	  (progn
	    (setq 
	     mew-refile-alist (cons (mew-refile-pair todir) mew-refile-alist)
	     mew-folder-alist (cons (mew-folder-pair todir) mew-folder-alist)
	     mew-folder-list (cons todir  mew-folder-list))
	    (if (file-readable-p mew-folders-file)
		(save-excursion
		  (set-buffer (get-buffer-create mew-buffer-tmp))
		  (erase-buffer)
		  (insert todir)
		  (insert "\n")
		  (append-to-file (point-min) (point-max) mew-folders-file)))
	    ))
      (if (mew-member todir mew-folder-list)
	  (progn 
	    (mew-summary-mark mew-mark-refile)
	    (setq mew-summary-refile-last-destination tofolder)
	    (setq mew-summary-buffer-refile 
		  (cons (cons msg tofolder) mew-summary-buffer-refile))
	    (save-excursion
	      (set-buffer msg-buffer)
	      (mew-refile-folder-guess-learn (cons tofolder (cdr learn-info))))
	    (mew-summary-multipart-delete)
	    (mew-summary-display-next)))
      )))

(defun mew-summary-refile-again ()
  (interactive)
  (mew-summary-refile mew-summary-refile-last-destination))

;;
;; guess function dispatcher.
;;
(defun mew-refile-folder-guess ()
  (let ((ret nil))
    (catch 'match
      (mapcar
       (function
	(lambda (arg)
	  (setq ret (funcall arg))
	  (if (car ret)
	      (throw 'match ret))))
       mew-refile-folder-guess-functions))
    (if (listp ret)
	ret
      (list ret))))

;;
;; inform guess functions result.
;;   result: (user-chosen-folder learning-function-depend-params ..)
;;
(defun mew-refile-folder-guess-learn (result)
  ; XXX: call always mew-guess-by-from-learn for mew-alias-alist
  ;(if (mew-member 'mew-guess-by-from mew-refile-folder-guess-functions)
      (mew-guess-by-from-learn result)
    ;)
  (if (mew-member 'mew-guess-by-message-id mew-refile-folder-guess-functions)
      (mew-guess-by-message-id-learn result))
  )

;;
;; save all alists
;;
(defun mew-refile-folder-guess-save ()
  (if (and
       mew-from-alist
       (mew-member 'mew-guess-by-from mew-refile-folder-guess-functions))
      (mew-alist-save mew-from-file-name mew-from-alist)
    )
  (if (and 
       mew-msg-id-alist
       (mew-member 'mew-guess-by-message-id mew-refile-folder-guess-functions))
      (mew-alist-save mew-msg-id-file-name mew-msg-id-alist)))

;;
;; guess functions (private)
;;
;;
;; if the guess-function has learning function, guess-function must
;;    return: (folder guess-function-name learning-function-depenent-params..)
;;
;; only first member of the returned list will be used for completion.

;;
;; guess refile folder by mew-auto-folder-alist.
;;
;; return: (folder 'mew-guess-by-alist)
(defun mew-guess-by-alist ()
  (let (ret)
    (if mew-auto-folder-alist
	(list 
	 (catch 'match
	   (car
	    (mapcar 
	     (function 
	      (lambda (arg)
		(if (setq ret (mew-guess-by-alist2 (car arg) (cdr arg)))
		    (throw 'match ret))))
	     mew-auto-folder-alist)))
	 'mew-guess-by-alist))
    ))

; This function is normally called by 'mew-guess-by-alist'.
(defun mew-guess-by-alist2 (field regexp-folder-alist)
  (let ((case-fold-search t)
	(field-contents))

    (if (and (stringp field)
	     (setq field-contents (mew-field-get-value field)))
	(catch 'match
	  (car 
	   (mapcar 
	    (function (lambda (arg)
			(if (and (stringp (car arg))
				 (stringp (cdr arg))
				 (string-match (car arg) field-contents))
			    (throw 'match 
				   (mew-guess-by-alist3
				    (car arg) field-contents (cdr arg))
				   ))))
	    regexp-folder-alist))))))

(defun mew-guess-by-alist3 (regexp field string)
  (let ((p 1) (match-list nil)
	start end)
    (string-match regexp field)
    (while (<= p 9)
      (if (and (integerp (setq start (match-beginning p)))
	       (integerp (setq end (match-end p))))
	  (setq match-list (cons (substring field start end) match-list))
	(setq match-list (cons "" match-list)))
      (setq p (1+ p)))
    (setq match-list (reverse match-list)
	  p 1)
    (while (<= p 9)
      (if (string-match (concat "\\\\" (int-to-string p)) string)
	  (setq string
		(concat (substring string 0 (match-beginning 0))
			(nth (1- p) match-list)
			(substring string (match-end 0))))
	(setq p (1+ p))))
    string))

;;
;; guess refile folder by existing folder names.
;;
;; return: (folder 'mew-guess-by-folder)
(defun mew-guess-by-folder ()
  (let ((to-cc (mew-header-user-collect '("To:" "Cc:" "Apparently-To:")))
	(ret nil)
	(from nil))
    (catch 'to-cc-guess
      (while to-cc
	(if (setq ret (or (cdr (assoc (car to-cc) mew-refile-alist))
			  (cdr (assoc 
				(concat (car to-cc) "/") mew-refile-alist))))
	    (throw 'to-cc-guess ret))
	(setq to-cc (cdr to-cc))))
    (list ret 'mew-guess-by-folder)))


;;
;; guess refile folder by message-id
;;
;; return: (folder 'mew-guess-by-message-id)
(defun mew-guess-by-message-id ()
  (let ((str (or (mew-field-get-value "References:")
		 (mew-field-get-value "In-Reply-To:"))))
    ; load message id alist
    (if (not mew-msg-id-alist)
	(setq mew-msg-id-alist
	      (mew-alist-load mew-msg-id-file-name)))
    (list 
     (car
      (mew-alist-search 
       mew-msg-id-alist 
       (and str (string-match "\\(<[^ \t\n]*>\\)[^>]*\0" (concat str "\0"))
	    (substring str (match-beginning 1) (match-end 1)))))
     'mew-guess-by-message-id)
    ))


(defun mew-guess-by-message-id-learn (result)
  (let ((str (mew-field-get-value "Message-Id:")))
    (if (and str (string-match "<[^ \n]*>" str) (car result))
	(progn 
	  (setq str 
		(substring
		 str 
		 (string-match "<[^ \n]*>" str) (match-end 0)))
	  (if (not mew-msg-id-alist)
	      (setq mew-msg-id-alist
		    (mew-alist-load mew-msg-id-file-name)))
	  (setq mew-msg-id-alist
		(mew-alist-add 
		 mew-msg-id-alist 
		 str
;		 (substring str (match-beginning 0) (match-end 0))
		 (list (car result) "??")))))
    ))

;;
;; guess refile folder by From: field.
;;
;; return: (folder 'mew-guess-by-from)
(defun mew-guess-by-from ()
  (let ((from (mew-header-extract-addr 
	       (or (mew-field-get-value "From:") "")))
	(tocc (mew-header-extract-addr
	       (or (mew-field-get-value "To:")
		   (mew-field-get-value "Apparently-To:")
		   "")))
	(myaddr (mew-header-delete-at mew-mail-address))
	(folder nil))
    (if (equal myaddr (mew-header-delete-at from))
	(setq from tocc))
    ;load from alist
    (if (not mew-from-alist)
	    (setq mew-from-alist
		  (mew-alist-load mew-from-file-name)))
       ; search from alist
    (if (setq folder (mew-alist-search mew-from-alist from))
	(list folder 'mew-guess-by-from from)
      (list nil 'mew-guess-by-from))
    ))


; result: (user-chosen-folder 'mew-guess-by-from &optoinal from)
(defun mew-guess-by-from-learn (result)
  (let ((folder (nth 0 result))
	(id     (nth 1 result))
	(from   (or (nth 2 result)
		    (mew-header-extract-addr
		     (or (mew-field-get-value "From:") "")))))
    (if (and (or (equal id 'mew-guess-by-from)
		 (equal id 'mew-guess-by-default)
		 (equal id 'mew-guess-by-message-id))
	     from)
	(progn
	  (if (not mew-from-alist)
	      (setq mew-from-alist
		    (mew-alist-load mew-from-file-name)))
	  (setq mew-from-alist
		(mew-alist-add mew-from-alist from folder))
	  ))
    ; XXX: move to other place !!
    (if (not (assoc (mew-header-delete-at from) mew-alias-alist))
	(setq mew-alias-alist
	      (append mew-alias-alist
		      (list (cons (mew-header-delete-at from)
				  from)))))
    ))

;; offer default folder.
;;
;; return: (folder 'mew-guess-by-from)
;;
(defun mew-guess-by-default ()
   ;not found in alist .. return default
  (let ((from (mew-header-extract-addr 
	       (or (mew-field-get-value "From:") "")))
	(tocc (mew-header-extract-addr
	       (or (mew-field-get-value "To:") 
		   (mew-field-get-value "Apparently-To:")
		   "")))
	(myaddr (mew-header-delete-at mew-mail-address)))
    (if (equal myaddr (mew-header-delete-at from))
	(setq from tocc))
    (if (mew-member 
	 (concat mew-folders-default-folder "/")
	 mew-folder-list)
	(list (concat "+" mew-folders-default-folder "/" 
		      (mew-header-delete-at from))
	      'mew-guess-by-default from)
      (list (concat "+" (mew-header-delete-at from))
	    'mew-guess-by-default from))))

;;
;; common routines
;;

;;
;; common routines for alists
;;
(defun mew-alist-search (alist key)
  (car (cdr (assoc key alist))))

(defun mew-alist-add (alist key val)
  (cons (list key val)
	(delq (assoc key alist) alist)))

(defun mew-alist-del (alist key)
  (delq (assoc key alist) alist))

(defun mew-alist-load (filename)
  (let ((alist nil)
	(fullname (expand-file-name filename mew-path)))
    (save-excursion
      (if (not (file-readable-p fullname))
	  ()
	(find-file-read-only fullname)
	(setq alist 
	      (condition-case err
		  (read (current-buffer)) (error nil nil))
	      alist
	      (reverse
	       (nthcdr (- (length alist) mew-alist-max-length)
		       (reverse alist))))
	(kill-buffer (current-buffer))
	alist))))

(defun mew-alist-save (filename alist)
  (save-excursion
    (let* ((fullname (expand-file-name filename mew-path))
	   (tmp-buf  (set-buffer (create-file-buffer fullname))))
      (setq alist
	    (reverse
	     (nthcdr (- (length alist) mew-alist-max-length)
		     (reverse alist))))
      (set-visited-file-name fullname)
      (erase-buffer)
      (prin1 alist tmp-buf)
      (princ "\n" tmp-buf)
      (save-buffer 0)
      (kill-buffer tmp-buf))))

;;
;; common routine for lists
;;
(defun mew-member (a list)
  (catch 'member
    (while list
      (if (or (equal (car list) a)
	      (equal (car list) (and (stringp a)
				     (concat a "/"))))
	  (throw 'member t))
      (setq list (cdr list)))))

