;;;			      mew-url.el
;;;
;;;   Copyright (C) 1994, 1995  Mine Sakurai  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Mine Sakurai <m-sakura@ccs.mt.nec.co.jp>
;;;          Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: June     8, 1994
;;; Revised: February 7, 1995
;;;

(defconst mew-url-version "mew-url.el version 0.17")

(provide 'mew-url)
(require 'mew)

(if mew-emacs19-p
    (eval-when-compile 
     (require 'ange-ftp)))

(defvar mew-buffer-extb " *mew extb*")
(defvar mew-url-default-filename "a.extb")

(defvar mew-mime-external-body-switch
  '(("anon-ftp" . mew-mime-external-ftp)
    ("ftp" . mew-mime-external-ftp)
    ("local-file" . mew-mime-external-get-file)
    ("afs" . mew-mime-external-get-file)
    ("mail-server" . mew-mime-external-send-mail)
    ("tftp" . mew-mime-external-tftp)
    ("url" . mew-mime-external-url)
    )
  "MIME Content-Type: message/external-body switches.
(mew-mime-external-body-type . (program))"
  )

;; URL processor  (ex. Mosaic, chimera, lynx?)
(defvar mew-mime-external-url-process "Mosaic")

;; FTP program (ex. ftp, rftp (socksified ftp))
(defvar mew-mime-external-ftp-program "ftp")

;;
;; If you use CSTC socksified ftp, we must skip "CSTC version ..." message.
;; This message is processed as error message on ange-ftp, 
;; and ftp-processing is stopped. 
;;
(defvar mew-ftp-skip-msgs nil)
(if (string-match "rftp" mew-mime-external-ftp-program)
    (setq mew-ftp-skip-msgs "^CSTC \\|^4.2beta \\|"))

;;;
;;;
;;;

(defvar mew-external-body-parameter-set
;; the list of
;;  (access-type  (parameter-name meaning selections-list mandatory )...)
;;
  '(("anon-ftp" 
     ("site" "Ftp host" mew-extb-ftp-site-list t)
     ("directory" "Directory path of the file" () nil)
     ("name" "File name" () t)
     ("mode" "Transfer mode" mew-extb-ftp-mode-list nil))
    ("ftp" 
     ("site" "Ftp host" mew-extb-ftp-site-list t)
     ("directory" "Directory path of the file" () nil)
     ("name" "File name" () t)  
     ("mode" "Transfer mode" mew-extb-ftp-mode-list nil))
    ("mail-server" 
      ("server" "Mail address" () t) 
      ("subject" "Subject" () nil)) 
    ("local-file" 
      ("name" "File path" () t))
    ("afs" 
      ("name" "File path" () t) 
      ("site" "Site" () nil))
    ("url" 
      ("name" "URL" () t))
    ))

(defvar mew-extb-ftp-site-list 
  '("pr.aist-nara.ac.jp" "ftp.csce.kyushu-u.ac.jp" "sh.wide.ad.jp")
  )

(defvar mew-extb-ftp-mode-list 
  '("binary" "ascii" "image" "none")
  )

;; 
;; external-body common optional parameters
;;
(defvar mew-external-body-optional-parameter
  '(("expiration" "Expiration of the external-body" 
     mew-extb-expiration-selections nil)
    ("size" "Size (in bytes) of data" () nil) 
    ;;("permission") not used yet
    )
  )

(defvar mew-extb-month-year-alist 
  '(("1month" . 1) ("3month" . 3) ("6month" . 6) 
    ("1year" . 12)
    ("none" . nil))
  )

(defvar mew-extb-expiration-selections (mapcar 'car mew-extb-month-year-alist))

;;
;; selections are too limited now. It's prototype.
;;

;;;
;;; for expiration 
;;; 
(defconst mew-time-stamp-month-numbers
  '(("Jan" . 1) ("Feb" . 2) ("Mar" . 3) ("Apr" . 4) ("May" . 5) ("Jun" . 6)
    ("Jul" . 7) ("Aug" . 8) ("Sep" . 9) ("Oct" . 10) ("Nov" . 11) ("Dec" . 12))
  "Alist of months and their number.")

;;;
;;;  VIEW
;;;

(defun mew-mime-external-body (begin end params)
;; message-buffer
  (let* ((access-type
	 (mew-get-external-body-parameter "access-type=" params))
	 (program (cdr (assoc (downcase access-type) 
			      mew-mime-external-body-switch)))
	)
    (if (and (symbolp program) (fboundp program))
	(funcall program begin end params))))

(defun mew-mime-external-ftp (begin end params)
;;
;; Using ange-ftp and getting file.
;;
  (require 'ange-ftp)
  (let* ((access-type (mew-get-external-body-parameter "access-type=" params))
      (site (mew-get-external-body-parameter "site=" params))
      (directory (mew-get-external-body-parameter "directory=" params))
      (name (mew-get-external-body-parameter "name=" params))
      (filepath (if directory 
		    (if (equal (substring directory -1) "/")
			(format "%s%s" directory name)
		      (format "%s/%s" directory name))
		  name))
      (mode (mew-get-external-body-parameter "mode=" params))
      ;; common parameters
      (expiration (mew-get-external-body-parameter "expiration=" params));
      (size (mew-get-external-body-parameter "size=" params))
      ;;
      (file-content-type nil)
      (ange-file-expression nil)
      (ange-ftp-generate-anonymous-password mew-mail-address)
      (ange-ftp-ftp-program-name mew-mime-external-ftp-program)
      (ange-ftp-skip-msgs (if mew-ftp-skip-msgs 
			      (concat mew-ftp-skip-msgs ange-ftp-skip-msgs) 
			    ange-ftp-skip-msgs))
      (username "")
      (buffer-read-only nil)
      )
    (cond ((string= "anon-ftp" access-type)
	   (setq username "anonymous"))
	  ((string= "ftp" access-type)
	   (setq username "YOU")))
    (erase-buffer)
    (insert " ####### ####### ######  \n"
	    " #          #    #     # \n"
	    " #          #    #     # \n"
	    " #####      #    ######  \n"
	    " #          #    #       \n"
	    " #          #    #       \n"
	    " #          #    #       \n"
            "\n\n")
    (insert "You could get the message contents by FTP\n\n"
            (format "SITE:\t%s\n" site)
            (format "USER:\t%s\n" username)
            (format "FILE:\t%s\n" filepath))
    (if size
	(insert (format "SIZE:\t%s bytes\n" size)))
    ;;
    ;; expiration
    ;;
    (cond ((and expiration (mew-extb-expired-p expiration))
	   (insert "\n"
		   "But this information is EXPIRED at \""
		   (format "%s" expiration)
		   "\"\n"
		   ))
	  ((not (mew-y-or-n-p (format "get %s from %s? " 
				      name site )))
	   ())
	  (t
	   (if (string= "ftp" access-type)
	       (setq username
		     (read-string (format "User name at %s: " site)
				  (user-login-name))))
	   (setq ange-file-expression
		 (format "/%s@%s:%s"
			 username site filepath))
	   ;; decide to look the file contents or not 
	   (erase-buffer)
	   (cond ((and mode (string= "ascii" mode))
		  (if (or (null (insert-file-contents ange-file-expression))
			  (not (mew-y-or-n-p "Save the message buffer?")))
		      ()
		    (write-region (point-min) (point-max)
				  (read-file-name 
				   "Save to : "  
				   (format "%s%s"
					   default-directory name))
				  )))
		 (t
		  (copy-file ange-file-expression 
			     (read-file-name "Save to : " 
					     (format "%s%s" 
						     default-directory name)))
		  )
		 )
	   )
	  )))

(defun mew-mime-external-get-file (begin end params)
;;
;; We need read the file. we display it into the message buffer or
;; call the program (e.g. "xv") based on the content-type of the 
;; external-body. 
;;
  (let ((name (mew-get-external-body-parameter "name=" params))
	;; common parameters
	(expiration (mew-get-external-body-parameter "expiration=" params));
	(size (mew-get-external-body-parameter "size=" params))
	;;
	(buffer-read-only nil))
    (erase-buffer)
    (insert "#######   ###   #       #######\n"
	    "#          #    #       #\n"
	    "#          #    #       #\n"
	    "#####      #    #       #####\n"
	    "#          #    #       #\n"
	    "#          #    #       #\n"
	    "#         ###   ####### #######\n"
	    "\n\n")
    (insert "You could get information from local host (or by using afs)\n"
	    "\n"
	    (format "FILE:\t%s\n" name)
	    "\n")
    (if size
	(insert (format "SIZE:\t%s bytes\n" size)))
    (cond ((and expiration (mew-extb-expired-p expiration))
	   (insert "\n"
		   "But this information is EXPIRED at \""
		   (format "%s" expiration)
		   "\"\n"
		   ))
          ((not (file-exists-p name))
	   (insert (format "But file \"%s\" not found" name)))
	  ((not (file-readable-p name))
	   (insert (format "But You can't read the file \"%s\""  name)))
	  (t
	   (let* ((buffer-read-only nil)
		  (real-ct (or (mew-mime-extb-get-ct begin end) "text/plain"))
		  (attr (mew-content-attr real-ct  mew-mime-content-type))
		  (program (mew-content-program attr))
		  (options (mew-content-options attr))
		  (async (mew-content-async attr)))
	     (cond ((null program)
		    (erase-buffer)
		    (insert-file-contents name)
		    (message (format "Reading file \"%s\"...done" name))
		    )
		   ((and (symbolp program) (fboundp program))
		    (erase-buffer)
		    (insert-file-contents name)  ;is it enough??
		    (message (format "Reading file \"%s\"...done" name))
		    )
		   (t
		    (save-excursion
		      (set-buffer (mew-current-get 'cache))
		      (let ((selective-display nil)
			    (mc-flag nil))
			(if async
			    (mew-mime-extb-start-process name program options)
			  (mew-mime-extb-call-process name program options)
			  )))
		    ))
	     )))
    ))
      

(defun mew-mime-extb-get-ct (begin end)
  (let (real-ct)
    (save-excursion
      (set-buffer (mew-current-get 'cache))
      (narrow-to-region begin end)
      (goto-char (point-min))
      ;;
      ;; find the content-type of the external-body
      ;;
      ;(re-search-forward "^$" nil t)
      ;(forward-line 1)
      ;(narrow-to-region (point) end)
      (setq real-ct (mew-field-get-value "Content-Type:"))
      ;(widen)
      (widen))
    (if (equal real-ct "") nil real-ct)
    )
  )

(defun mew-mime-external-send-mail (begin end params)
;;
;; This enters draft-mode and sends mail to mail-server
;;
  (let ((server (mew-get-external-body-parameter "server=" params))
	(subject-param (mew-get-external-body-parameter "subject=" params))
	;; common parameters
	(expiration (mew-get-external-body-parameter "expiration=" params));
	(size (mew-get-external-body-parameter "size=" params))
	;;
	(p nil)
	(buffer-read-only nil))
    (erase-buffer)
    (insert "#     #    #      ###   #\n"
            "##   ##   # #      #    #\n"
            "# # # #  #   #     #    #\n"
            "#  #  # #     #    #    #\n"
            "#     # #######    #    #\n"
            "#     # #     #    #    #\n"
            "#     # #     #   ###   #######\n"
            "\n\n")
    (insert "You could get the information by sending mail\n"
            "\n"
            (format "SERVER:\t\t%s\n" server))
    (if size
	(insert (format "SIZE:\t%s bytes\n" size)))
    (cond ((and expiration (mew-extb-expired-p expiration))
	   	   (insert "\n"
		   "But this information is EXPIRED at \""
		   (format "%s" expiration)
		   "\"\n"
		   ))
	  ((null server)(message "\"server=\" parameter is required."))
	  ((mew-y-or-n-p (format "mail to %s ? " server))
	   (save-excursion
	     (set-buffer (mew-current-get 'cache))
	     (narrow-to-region begin end)
	     (goto-char (point-min))
	     ;;
	     ;; find the phantom body (in RFC1521)
	     ;;
	     (re-search-forward "^$" nil t)
	     (forward-line 1)
	     (setq p (point))
	     (widen))
	   ;;
	   ;; pickd up source from 'mew-send
	   ;;
	   (save-excursion
	     (unwind-protect
		 (let ((to server)
		       (cc "")
		       (subject (if subject-param subject-param 
				  "Automated Mail Server Request"))
		       (file (mew-draft-get-new))
		       (buffer-read-only nil))
		   (switch-to-buffer (find-file-noselect file))
		   (mew-draft-mode)
		   (mew-draft-rename file)
		   (mew-draft-header to cc subject)
		   (insert-buffer-substring (mew-current-get 'cache) p end)
		   )
	       (save-buffer))
	     ;;
	     ;; send
	     ;;
	     (if (mew-field-get-value "Content-Type:")
		 (mew-draft-real-send-letter)
	       (message "Making a MIME letter ...")
	       (undo-boundary)
	       (mew-draft-make-mime)
	       (undo-boundary)
	       (message "Making a MIME letter ... done")
	       (if (mew-y-or-n-p "Do you really send this letter? ")
		   (mew-draft-real-send-letter)
		 ;; draft killed
		 (set-buffer-modified-p t) ;; for delete recovery file
		 (save-buffer)
		 (if (file-exists-p (buffer-file-name))
		     (delete-file (buffer-file-name)))
		 (set-buffer-modified-p nil)
		 (kill-buffer (buffer-name)))
	       ))))))

(defun mew-mime-external-tftp (begin end params)
  (message "access-type TFTP Not supported  yet")
  (let ((buffer-read-only nil))
    (insert-buffer-substring (mew-current-get 'cache) begin end)))


(defun mew-mime-external-url (begin end params)
;;
;; This starts "mew-mime-external-url-proces" asynchronously
;;
  (let ((url (mew-get-external-body-parameter "name=" params))
	;; common parameters
	(expiration (mew-get-external-body-parameter "expiration=" params));
	(size (mew-get-external-body-parameter "size=" params))
	;;
	(buffer-read-only nil))
    (erase-buffer)
    (insert "#     # ######  #\n"
            "#     # #     # #\n"
            "#     # #     # #\n"
            "#     # ######  #\n"
            "#     # #   #   #\n"
            "#     # #    #  #\n"
            " #####  #     # #######\n"
            "\n\n")
    (insert (format "You could get the information by executing \"%s\"\n" 
		    mew-mime-external-url-process)
	    "This is executed asynchronously,\n"
	    (format "so you could do another job on this editor while executing \"%s\"\n"
		    mew-mime-external-url-process)
	    "\n"
	    (format "URL:\t\t%s\n" url))
    (if size
	(insert (format "SIZE:\t%s bytes\n" size)))
    (cond ((and expiration (mew-extb-expired-p expiration))
	   	   (insert "\n"
		   "But this information is EXPIRED at \""
		   (format "%s" expiration)
		   "\"\n"
		   ))
	  ((mew-y-or-n-p (format "get %s ? " url))
	   (save-excursion
	     (set-buffer (mew-current-get 'cache))
	     (let ((pro nil)
		   (program mew-mime-external-url-process))
	       (setq pro (apply (function start-process)
				(format "*mew %s*" program)
				mew-buffer-tmp
				program
				(list url)))
	       (message "Sending data to %s ..." program)
	       (process-send-region pro begin end)
	       (message "Sending data to %s ... done" program)))
	   ))
;    (if (mew-y-or-n-p (format "get %s ? " url))
;	(w3-fetch url))
      ))

;(defun mew-get-external-body-parameter (field params)
;  (let* ((param1 (car params))
;	 (p (and param1 (string-match field param1))))
;    (cond ((null params) nil)
;	  ((null p)
;	   (mew-get-external-body-parameter field (cdr params)))
;	  (t
;	   (substring param1 (match-end 0) (length param1))))))

(defun mew-get-external-body-parameter (field params)
  (let ((ret))
    (catch 'parameter
      (while params
	(if (string-match field (car params))
	    (progn
	      (setq ret (substring (car params) (match-end 0) nil))
	      (throw 'parameter nil)))
	(setq params (cdr params))
	))
    ret
    ))
	  

;;;
;;; COMPOSE
;;;
;;;
;;; Message/external-body composing
;;; 

(defun mew-multi-external-body ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't insert external-body here")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Copy to %s(%s): " 
				   subdir
				   mew-url-default-filename))
	   (efile nil)
	   (filepath))
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (setq filepath (expand-file-name efile mimedir))
      (if (and (file-exists-p filepath)
	       (null (mew-y-or-n-p
		      (message "File %s exsists. Overwrite it? " efile))))
	  ()
	(mew-multi-extb-make-file filepath)
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       nums
	       (list file
		     (mew-file-content 
		      (mew-file-attr file mew-mime-content-type))
		     ""
		     nil)
	       mew-multi-syntax))
	(mew-multi-display)
	))
    ))


(defun mew-multi-extb-make-file (filename)
  (let ((buf))
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-extb))
      (erase-buffer)
      (mew-multi-extb-construct-content-type)
      (write-file filename)
      (setq buf (current-buffer))
      )
    (kill-buffer buf)
    ))

(defun mew-multi-extb-construct-content-type ()
;;
;;query to user and construct "Content-Type:" field etc.
;;
  (let* ((access-type 
	  (mew-completing-read "External body type"
			 mew-external-body-parameter-set
			 nil t "anon-ftp"))
	 (params (cdr (assoc access-type mew-external-body-parameter-set)))
	)
    (insert "Content-Type: Message/external-body")
    (insert (format ";\n\taccess-type=%s" access-type))
    (if (string= access-type "anon-ftp")
	(mew-multi-extb-anon-ftp)
      (mew-multi-extb-putparam params)
      ;; make phantombody
      (if (string= access-type "mail-server")
	  (mew-multi-extb-construct-phantombody))
      )
    )
  )

(defun mew-multi-extb-putparam (params)
  "PARAMS is a list of..
    (parameter meaning (value-selections) mandatory)
mandatory is t if the parameter is mandatory for the access-type specific."
  (let ((value "")
	(guessed-content-type "Application/Octet-stream")
	)
    (setq params (append params mew-external-body-optional-parameter))
    (while params 
      (let* ((a-param (car (car params)))
	     (query (car (cdr (car params))))
	     (selections (car (cdr (cdr (car params)))))
	     (selections-alist (if selections 
				   (mapcar 'list (symbol-value 
						  selections))
				 nil))
	     (mandatory (car (cdr (cdr (cdr (car params))))))
	     (default (car (car selections-alist)))
	     (query-form (if mandatory 
			     (format "[Required] %s" query)
			   (format "[Optional] %s" query))))
	(if default
	    (setq value (mew-completing-read query-form
					 selections-alist
					 nil nil default))
	  (setq value (read-string (format "%s : " query-form))))
	(cond ((and (string= value "") mandatory) nil)
	      ((or (string= value "") (string= value "none"))
	       (setq params (cdr params)))
	      (t
	       (if (string= (downcase a-param) "expiration")
		   ;; create RFC822 type date-time
		   (setq value (mew-extb-set-expiration-date value)))
	       (insert (format ";\n\t%s=\"%s\"" a-param value))
	       ;;
	       ;; guess content-type of specified filename
	       ;; but, we can't guess charset because the file is
	       ;; in external place.
	       ;;
	       (if (string= (downcase a-param) "name")
		   (setq guessed-content-type
			 (mew-file-content (mew-file-attr 
					    value 
					    mew-mime-content-type)))
		   )
	       (setq params (cdr params))))
	))
    ;;
    (insert "\n")
    ;;
    ;; content type of file specified by the external-body
    ;; This is guessed from filename (parameter "name").
    ;;
    (insert "\n")
    (insert (format "Content-Type: %s\n" 
		    (mew-completing-read 
		     "Content type of the external file"
		     (mapcar 
		      (function (lambda (x) (cons (mew-file-content x)
						  (mew-file-content x))))
		      mew-mime-content-type)
		     nil t guessed-content-type
		     )))
    (insert "\n")
    ))



(defun mew-multi-extb-construct-phantombody ()
;; only for mail-server type
  (insert (read-string "Input message to the mail-server: " ))
  (insert "\n")
  )

(defun mew-completing-read (prompt alist &optional predicate require-match initial )
;;
;; prompt is bit more complicated than "completing-read".
;; 
  (let* ((completion-ignore-case t)
	 (question (if initial (format "%s (%s) : " prompt initial)
		       (format "%s : " prompt)))
	 (readval (completing-read 
		   question alist predicate require-match nil))
	 )
    (if (and initial (string= readval "")) 
	(setq readval initial)
      readval)
    )
  )

;;;
;;; similar functions...
;;; 

;;;
;;; modified mew-mime-start-process
;;;
(defun mew-mime-extb-start-process (filename program options)
  (if (mew-y-or-n-p (format "Start %s? " program))
      (let ((pro nil)
	    (mc-flag nil)
	    (file-coding-system (if mew-mule-p *noconv*)))
	(message "Starting %s ..." program)
	(setq pro (apply (function start-process)
			 (format "*mew %s*" program)
			 mew-buffer-tmp ;; xxx
			 program
			 (append options (list filename))))
	(set-process-sentinel pro 'mew-mime-extb-start-process-sentinel)
	(message "Sending %s ... done" program)
	(setq mew-process-file-alist
	      (cons (cons pro filename) mew-process-file-alist))
	))
  t ;; to next part
  )

;;;
;;; modified mew-mime-start-process-sentinel
;;; (not to delete file)
;;;
(defun mew-mime-extb-start-process-sentinel (process event)
  (let ((al (assoc process mew-process-file-alist)))
    (setq mew-process-file-alist (mew-delq al mew-process-file-alist))
    ))

;;;
;;; modified mew-mime-call-process
;;;	    
(defun mew-mime-extb-call-process (filename program options)
  (if (mew-y-or-n-p (format "Call %s? " program))
      (let ((mc-flag nil)
	    (file-coding-system (if mew-mule-p *noconv*)))
	(message "Calling %s ..." program)
	(apply (function call-process)
	       program filename nil nil options)
	(message "Calling %s ... done" program)
	)
    )
  t ;; to next part
  )

;;;
;;; This function is originally "time-stamp-dd-mon-yy", which is
;;;  one of "time-stamp.el".
;;;      Maintainer: Stephen Gildea <gildea@lcs.mit.edu>
;;;      Time-stamp: <94/02/14 15:02:07 gildea>
;;;
;;;
;;; RFC 822 date (and extended by RFC1123)
;;; e.g. "28 Dec 1994 13:37:50"
;;;      "01 Dec 1994 10:55:35"
;;; 

(defun mew-time-stamp-rfc822 ()
  "Return the current date as a string in \"DD Mon YY hh:mm:ss\" form."
  (let ((date (current-time-string)))
    (format "%02d %s %s %s"
            (string-to-int (substring date 8 10))
            (substring date 4 7)
	    ;;
	    ;; 4digit is permitted in the year field (in RFC1123)
	    ;;
            (substring date -4)
	    (substring (current-time-string) 11 19)
	    )))

;;;
;;; "01 Dec 1994 10:55:35" -> ("01" "Dec" "1994" "10:55:35")
;;;
;;; using by (mew-header-split "01 Dec 1994 10:55:35" " ")
;;;
;;; TIME-LIST is assumed to list of (day Mon yy hh:mm:yy)
;;; eg. ("01" "Dec" "1994" "10:55:35")
;;; note: "day" is maybe "01" or " 1" or "1".
;;; 

(defun mew-time-stamp-year (time-list)
  (let ((year-num (string-to-int (car (cdr (cdr time-list)))))
	)
    (cond ((< year-num 100)  ; 2digit
	   (+ year-num 1900)); we adjust 4digit (hard coding :<)
	  (t year-num))      ; 4digit
    ))
;;;
;;; month number is returned. (not string)
;;; 
;(defmacro mew-time-stamp-month (time-string)
;  (` (cdr (assoc (substring (, time-string) 3 6) 
;		 mew-time-stamp-month-numbers)))
;  )
(defmacro mew-time-stamp-month (time-list)
  (` (cdr (assoc (car (cdr (, time-list))) mew-time-stamp-month-numbers))
     ))

(defmacro mew-time-stamp-day (time-list)
  (` (string-to-int (car (, time-list))))
  )

;;;
;;;  see if the external-body is expired. (bit primitive??)
;;;
(defun mew-extb-expired-p (expiration-times)
  (let* ((cur-timel (mew-header-split 
		     (mew-time-stamp-rfc822) ? )); split by space
	 (cur-year  (mew-time-stamp-year cur-timel))
	 (cur-month (mew-time-stamp-month cur-timel))
	 (cur-date  (mew-time-stamp-day cur-timel))
	 (exp-timel (mew-header-split expiration-times ? ))
	 (exp-year  (mew-time-stamp-year exp-timel))
	 (exp-month (mew-time-stamp-month exp-timel))
	 (exp-date  (mew-time-stamp-day exp-timel))
	 )
    (cond ((> cur-year exp-year) t)
	  ((< cur-year exp-year) nil)
	  ((> cur-month exp-month) t)
	  ((< cur-month exp-month) nil)
	  (t (> cur-date exp-date)))
    ))

;(defun mew-extb-put-expiration ()
;  (let ((value (mew-completing-read 
;		"expiration of the external file[optional]"
;		mew-extb-month-year-alist
;		nil t "6month")))
;    (if (not (string= value "none"))
;	(insert (format ";\n\texpiration=\"%s\""
;			(mew-extb-set-expiration-date value))
;		))
;	))

;;;
;;; 
;;; argument VALUE is a string (number of months)
;;; 
(defun mew-extb-set-expiration-date (value)
  (let* ((cur-timel (mew-header-split 
		     (mew-time-stamp-rfc822) ? )) ; split by space
	 (cur-year (mew-time-stamp-year cur-timel))
	 (cur-month (mew-time-stamp-month cur-timel))
	 (cur-date (mew-time-stamp-day cur-timel))
	 (forward-month (cdr (assoc value mew-extb-month-year-alist)))
	 (macro-y (+ (* cur-year 12) cur-month -1 forward-month))
	 (exp-year (/ macro-y 12))
	 (exp-month-str (car (rassq (1+ (% macro-y 12))
				 mew-time-stamp-month-numbers)))
	 )
    ;; time is ignored (set to 00:00:00).... sigh 
    (format "%02d %s %d 00:00:00" cur-date exp-month-str exp-year)
  ))


;;;
;;; remote-file support for anon-ftp
;;;

(defvar mew-ebody-host "")
(defvar mew-ebody-map nil)

(if mew-ebody-map
    ()
  (setq mew-ebody-map (make-sparse-keymap))
  (define-key mew-ebody-map "\t"     'mew-ebody-rfile)
  (define-key mew-ebody-map "\r"     'exit-minibuffer)
  (define-key mew-ebody-map "\n"     'exit-minibuffer)
  (define-key mew-ebody-map "\C-g"   'abort-recursive-edit)
  )

(defun mew-input-rfile (prompt) ;; prompt="To:"
  (read-from-minibuffer (concat prompt " ") "" mew-ebody-map nil)
  )

(defun mew-ebody-rfile ()
  (interactive)
  (let (file path comp all)
    (if (re-search-backward "/" nil t)
	(forward-char 1)
      (beginning-of-line))
    (setq file (buffer-substring (point) (point-max)))
    (delete-region (point) (point-max))
    (setq path (concat "/"
			mew-ebody-host
			":"
			(buffer-substring (point-min) (point))))
    (setq comp (ange-ftp-file-name-completion file path))
    (setq all (ange-ftp-file-name-all-completions file path))
    (cond
     ((eq comp t) 
      (and (get-buffer mew-buffer-completions)
	   (kill-buffer mew-buffer-completions))
      )
     ((equal 1 (length all))
      (insert (car all))
      (and (get-buffer mew-buffer-completions)
	   (kill-buffer mew-buffer-completions))
      )
     ((stringp comp)
      (insert comp)
      (and (> (length all) 1)
	   (with-output-to-temp-buffer mew-buffer-completions
	     (display-completion-list all))))
     (t (insert file))
     )
    )
  )

(defun mew-multi-extb-anon-ftp ()
  (require 'ange-ftp)
  (let ((ange-ftp-default-user "anonymous")
	(ange-ftp-generate-anonymous-password mew-mail-address)
	(mew-ebody-host (mew-completing-read
			 "[required] ftp hostname"
			 (if mew-extb-ftp-site-list
			     (mapcar 'list mew-extb-ftp-site-list))
			 nil nil
			 (car mew-extb-ftp-site-list)))
	file)
    (insert (format ";\n\tsite=\"%s\"" mew-ebody-host))
    ;; name of mew-ebody-host is important
    (setq file (mew-input-rfile "[required] filename :"))
    (cond
     ((string-match "\\(.*\\)/\\(.*\\)" file)
      (insert (format ";\n\tdirectory=\"%s\""(substring file 0 (match-end 1))))
      (insert (format ";\n\tname=\"%s\""
			 (substring file (match-beginning 2) nil))))
     (t (insert (format ";\n\tname=\"%s\"" file)))
     )
    (mew-multi-extb-putparam
     '(("mode" "Transfer mode" mew-extb-ftp-mode-list nil)))
    )
  )

