;;;	     mew.el :: Message interface to Emacs Window
;;;
;;;	     Copyright (C) 1994, 1995  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: March 22, 1994
;;; Revised: March  4, 1995
;;;

(defconst mew-version "Mew beta version 0.89")

(defvar mew-debug nil)
;(setq mew-debug nil)
;(setq mew-debug t)

(defvar mew-emacs18-p (string-match "^18" emacs-version))
(defvar mew-emacs19-p (string< "19" emacs-version))
(defvar mew-mule-p    (boundp 'MULE))
(defvar mew-xemacs-p  (string-match "XEmacs\\|Lucid" emacs-version))

(provide 'mew)
(require 'mew-refile)
(require 'mew-url)
(require 'mew-pem)
(require 'mew-pgp)
(require 'mew-fib)


;;;
;;; user customizable variables
;;;

(defvar mew-demo t
  "*Mew demo displays at boot time when non-nil.")

(defvar mew-analysis t
  "*MIME syntax analysis and cache when non-nil.
   Skip syntax analysis when nil.")

(defvar mew-cache-size 10
  "*Number of buffer for message cache.")

(defvar mew-summary-cache-file ".mew-cache"
  "*Cache file for summary mode contents.")

(defvar mew-summary-cache-use t
  "*If non-nil, insert mew-summary-cache-file into summary mode buffer 
when visits the summary that does not exsist in buffer. 
mew-summary-cache-file is automatically saved on inc and scan.")

(defvar mew-window-use-full nil
  "*Dynamic window configuration when nil. This variable will be obsolated
because mew will support mew-window-percent(e.g 50, 100, etc.).")

(defvar mew-summary-show-direction 'down
  "*Direction for SPC at end of message. 
'up 'down 'next(current direction) 'stop")

(defvar mew-mime-compose-folder-delete 'ask
 "*Action to delete MIME draft directory after sending a multipart letter.
Proper value is 'ask', 'delete', and 'retain'.")


(defvar mew-mail-domain-list nil
  "*Your e-mail address domain list like 
(\"is.aist-nara.ac.jp\" \"csce.kyushu-u.ac.jp\").
They are made use of mail-domain completion in draft mode(C-cTAB)")

(defvar mew-mail-domain (or (car mew-mail-domain-list) "")
  "*Your e-mail address domain.")

(defvar mew-mail-address-list nil
  "*The addresses included in this list never appear on the Cc:
field on a draft buffer.")


(defvar mew-redist-needs-full-contents nil)
  ; if you are using MH/post speaks SMTP, leave this as "nil".
  ; if you are using MH/post spawns sendmail and passes message with PIPE,
  ; change this as t.

(defvar mew-eof-string nil
  "*A value inserted on the end of message buffer if non-nil.")

(defvar mew-loop-depth 20
  "*A value to decide loop depth for header field syntax analysis")

;; Draft fields

(defvar mew-fcc nil
  "*A value inserted into Fcc: field in draft mode if non-nil.")
(defvar mew-cc nil
  "*A value inserted into Cc: field in draft mode if non-nil.")
(defvar mew-dcc nil
  "*A value inserted into Dcc: field in draft mode if non-nil.")
(defvar mew-from nil
  "*A value inserted into From: field in draft mode if non-nil.")
(defvar mew-x-mailer
  (concat mew-version " on Emacs " emacs-version
	  (and mew-mule-p
	       (string-match "^[0-9.]*" mule-version)
	       (concat ", Mule " (substring mule-version
					    (match-beginning 0) 
					    (match-end 0)))))
  "*A value inserted into From: field in draft mode if non-nil.")

(defvar mew-emacs-y-or-n-p-use nil)
(defvar mew-highlight-lines-use t
  "*Use highlight line in summary mode.")

(defvar mew-signature-file "~/.signature")

(defvar mew-file-append-p nil
  "If non-nil, a message or a part is appended to the existing file
on mew-summary-save(\"y\"). Otherwise overwrited.")
mew-file-append-p


;; Citation

(defvar mew-cite-prefix "> "
  "*Prefix of citation.")
(defvar mew-cite-hook nil
  "*If you want to use super-cite, (setq mew-cite-hook 'sc-cite-original).")
(defvar mew-cite-format
  '(format "From: %s\nSubject: %s\nDate: %s\n\n" from subject date)
  "*Format of citetaion string. 
Keywords 'from', 'subject', and 'date' indicates each header field value.")


;; Hooks

(defvar mew-init-hook nil
  "*Hook called at initialize time.")
(defvar mew-summary-mode-hook nil
  "*Hook called in summary mode.")
(defvar mew-draft-mode-hook nil
  "*Hook called in draft mode.")
(defvar mew-message-mode-hook nil
  "*Hook called in message mode.")
(defvar mew-message-hook nil
  "*Hook called whenever message displayed.")
(defvar mew-send-hook nil
  "*Hook called before sending a message")

(defvar mew-summary-recenter-p t)
(defvar mew-break-pages t)
(defvar mew-page-delimiter "^\^L")

;; Character code

(cond
 (mew-mule-p
  (defvar mew-mule-charset-local     *iso-2022-jp*) ;; Set your local charset
  (defvar mew-mule-charset-integrity *iso-2022-jp*)
  )
 )

(defvar mew-inbox "+inbox")
(defvar mew-folders-file ".folders")
(defvar mew-folders-ignore '("from"))
(defvar mew-folders-default-folder "from")

(defvar mew-print-function (function lpr-buffer))

(defvar mew-default-boundary "--%s(%s)--")

(cond 
 ((string= mew-mail-domain "")
  (defvar mew-mail-address (user-login-name))
  (defvar mew-reply-to nil))
 (t
  (defvar mew-mail-address (concat (user-login-name) "@" mew-mail-domain))
  (defvar mew-reply-to mew-mail-address))
 )

(defvar mew-window-configuration 
  '((summary (1  0))
    (message (3 10))))

(defvar mew-summary-message-regex "^ *\\([0-9]+\\)")
(defvar mew-summary-part-regex "^\t+\\([0-9.]+\\)")
(defvar mew-address-separator ":, \t\n")

;; X face

(defvar mew-x-face-filter '("uncompface" "ikon2xbm"))
(defvar mew-x-face-prog "xv")
(defvar mew-x-face-args nil)
(defvar mew-x-face-file "~/.xface"
  "*If non-nil and the file exists, X-Face: fields is inserted.")


(defvar mew-draft-mode-syntax-table nil
  "*Syntax table used while in mew letter mode.")

(defvar mew-field-visible
  '("Subject:" "From:" "To:" "Apparently-To:"
    "Cc:" "Newsgroups:" "Date:" "Reply-To:"
    "Resent-From:" "Resent-To:" "Resent-Cc:"
    "Mime-Version:" "Content-Type:")
  "*Visible fields for mail header.")

(defvar mew-field-invisible
  '("Received:" "Return-Path:" "Sender:" "Errors-To:"
    "Resent-Date:" "Message-Id:" "Resent-Message-Id:"
    "Resent-Sender:" "Resent-Reply-To:" "Delivery-Date:")
  "*Invisible fields for mail header.")

;; xxx If you change the order or add new fields, change mew-part-*.
(defconst mew-mime-fields
  '("Content-Type:" "Content-Transfer-Encoding:"
    "Content-ID:"   "Content-Description:"))

(defvar mew-mime-fields-number (length mew-mime-fields))
(defconst mew-part-magic (+ (length [type beg end]) (length mew-mime-fields)))

(defvar mew-prog-inc         "inc")
(defvar mew-prog-scan        "scan")
(defvar mew-prog-ali         "ali")
(defvar mew-prog-folders     "folders")
(defvar mew-prog-folder      "folder")
(defvar mew-prog-pack        "folder")
(defvar mew-prog-rmm         "rmm")
(defvar mew-prog-refile      "refile")
(defvar mew-prog-send        "send")
(defvar mew-prog-sortm       "sortm")
(defvar mew-prog-pick        "pick")
(defvar mew-prog-mime-encode "mewencode")
(defvar mew-prog-mime-decode "mewdecode")

(defvar mew-prog-unshar   "unshar")
(defvar mew-prog-uumerge  "uumerge")
(defvar mew-prog-tar      "tar")
(defvar mew-prog-compress "compress")
(defvar mew-prog-gzip     "gzip")


(defvar mew-prog-mime-encode-quoted '("-q"))
(defvar mew-prog-mime-decode-quoted '("-q"))

(defvar mew-prog-mime-encode-base64 '("-b"))
(defvar mew-prog-mime-decode-base64 '("-b"))
(defvar mew-prog-mime-encode-base64-text '("-b" "-t"))
(defvar mew-prog-mime-decode-base64-text '("-b" "-t"))

(defvar mew-prog-mime-encode-gzip64 '("-g"))
(defvar mew-prog-mime-decode-gzip64 '("-g"))
(defvar mew-prog-mime-encode-gzip64-text '("-g" "-t"))
(defvar mew-prog-mime-decode-gzip64-text '("-g" "-t"))


(defvar mew-prog-audio '("cat" "/dev/audio" mew-buffer-tmp nil))
;(defvar mew-prog-audio '("arecord" nil nil nil "-silentlevel" "-100"))

(defvar mew-prog-pick-argalist
  '(("-from") ("-date") ("-cc") ("-to") ("-search") ("-before")
    ("-after") ("-subject") ("-or") ("-and")
    ("-lbrace") ("-rbrace")))

(defvar mew-prog-pick-arglogic
  '("-or" "-and"))

(defvar mew-prog-pick-argdate
  '("-before" "-after"))

(defvar mew-prog-pick-argdatealist
  '(("yesterday") ("today") ("tomorrow")
    ("sunday") ("monday") ("tuesday") ("wednesday")
    ("thursday") ("friday") ("saturday")))

(defvar mew-mark-rmm    ?D)
(defvar mew-mark-mark   ?@)
(defvar mew-mark-hop    ?*)
(defvar mew-mark-refile ?o) ;; do you like "^"?

(defvar mew-mark-switch
  (list 
   (cons mew-mark-rmm    'mew-delete-line)
   (cons mew-mark-refile 'mew-delete-line)
   (cons mew-mark-mark   'mew-summary-unmark)
   (cons mew-mark-hop    'mew-summary-mark-letitbe)
   )
  )

(defvar mew-undo-switch
  (list 
   (cons mew-mark-rmm    'mew-summary-unmark)
   (cons mew-mark-refile 'mew-summary-refile-undo)
   (cons mew-mark-mark   'mew-summary-unmark)
   (cons mew-mark-hop    'mew-summary-unmark)
   )
  )

;(defvar mew-draft-mime-folder (format "+%s/mime" mew-draft-folder))
(defvar mew-draft-mime-folder "+drafts/mime")
(defvar mew-draft-multipart-begin "---- multipart --")
(defvar mew-draft-multipart-end   "---- multipart  ----")

(defvar mew-summary-mode-map    nil)
(defvar mew-message-mode-map    nil)
(defvar mew-draft-mode-map      nil)
(defvar mew-draft-header-map    nil)
(defvar mew-draft-body-map      nil)
(defvar mew-draft-multipart-map nil)
(defvar mew-minibuffer-map      nil)
(defvar mew-pick-map            nil)

;;;
;;; filename "^$" never match to anything
;;;

(defvar 
 mew-mime-content-type
 '(
   ("audio/basic"
    "\\.au$"	"base64"	("sh" ("-c" "cat - > /dev/audio") nil))
;   ("audio/basic"
;    "\\.au$"	"base64"	("aplay" () t))
   ("image/gif"
    "\\.gif$" 	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/jpeg"
    "\\.jpe?g$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/x-xwd"
    "\\.xwd$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/x-xbm"
    "\\.xbm$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/x-bmp"
    "\\.bmp$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/.*"
    "^$"        "base64"	("xv" ("-geometry" "+0+0") t))
   ("video/mpeg"
    "\\.mpe?g$"	"base64"	("mpeg_play" () t))
   ("application/postscript"
    "\\.ps$"	"quoted-printable"	("ghostview" ("-geom" "+0+0") t))
   ("application/octet-stream"
    "\\.tar\\.?g?z?$" "base64"	(mew-mime-text/plain () nil))
   ("application/x-graph"
    "\\.xg$"	"quoted-printable"	("xgraph" () t))
   ("message/external-body" 
    "\\.extb$"	nil		(mew-mime-external-body () nil))
   ("message/rfc822"
    "^[0-9]+$"	nil		(mew-mime-message/rfc822 () nil))
   ("text/richtext"
    "\\.rtf$"	nil		(mew-mime-text/plain () nil))
   ("text/plain"
	".*"	nil		(mew-mime-text/plain () nil))
   )
 "(content-type filename encoding (program (options ...) async))"
 )

(defvar mew-mime-content-type-list
  '("Text/Plain"
    "Message/Rfc822" "Message/External-body" ;; "Message/Partial"
    "Application/Postscript" "Application/Octet-Stream"
    "Image/Jpeg" "Image/Gif"
    "Audio/Basic" "Video/Mpeg"
    "Application/X-Graph")
  )

(defvar mew-mime-content-type-multipart-list
  '("Multipart/Mixed" "Multipart/Alternative"
    ;;"Multipart/Digest" "Multipart/Parallel"
    )
  )
     
(defvar mew-mime-default-function (function mew-mime-text/plain))

;;
(defmacro mew-file-attr (filename alist)
  (` (mew-assoc (, filename) (, alist) 1 t)))

;; for mapcar
(defun mew-file-content (attr)
  (car attr))

(defmacro mew-file-encoding (attr)
  (` (car (cdr (cdr (, attr))))))

;;
(defmacro mew-content-attr (content alist)
  (` (mew-assoc (, content) (, alist) 0 t)))

(defmacro mew-content-program (attr)
  (` (car (car (cdr (cdr (cdr (, attr))))))))

(defmacro mew-content-options (attr)
  (` (car (cdr (car (cdr (cdr (cdr (, attr)))))))))


(defmacro mew-content-async (attr)
  (` (car (cdr (cdr (car (cdr (cdr (cdr (, attr))))))))))

;;;
;;;
;;;

(if mew-mule-p
    (defvar mew-mule-character-set
      (list 
       (if (boundp '*iso-8859-1*)	;; Latin-1
	   (list lc-ltn1	*iso-8859-1*		"iso-8859-1")
	 (if (boundp '*ctext*) ;; for old mule
	     (list lc-ltn1	*ctext*			"iso-8859-1")))
       (if (boundp '*iso-8859-2*)	;; Latin-2
	   (list lc-ltn2	*iso-8859-2*		"iso-8859-2"))
       (if (boundp '*iso-8859-3*)	;; Latin-3
	   (list lc-ltn3	*iso-8859-3*		"iso-8859-3"))
       (if (boundp '*iso-8859-4*)	;; Latin-4
	   (list lc-ltn4	*iso-8859-4*		"iso-8859-4"))
       (if (boundp '*iso-8859-5*)	;; Cyrillic
	   (list lc-crl		*iso-8859-5*		"iso-8859-5"))
       (if (boundp '*iso-8859-6*)	;; Arabic
	   (list lc-arb		*iso-8859-6*		"iso-8859-6"))
       (if (boundp '*iso-8859-7*)	;; Greek
	   (list lc-grk		*iso-8859-7*		"iso-8859-7"))
       (if (boundp '*iso-8859-8*)	;; Hebrew
	   (list lc-hbw		*iso-8859-8*		"iso-8859-8"))
       (if (boundp '*iso-8859-9*)	;; Latin-5
	   (list lc-ltn5	*iso-8859-9*		"iso-8859-9"))
       (if (boundp '*iso-2022-jp*)	;; Japanese
	   (list lc-jp		*iso-2022-jp*		"iso-2022-jp"))
       (if (boundp '*iso-2022-kr*)	;; Korean
	   (list lc-kr		*iso-2022-kr*		"iso-2022-kr"))
;;       (if (boundp '*euc-korea*)
;;	   (list lc-kr		*euc-korea*		"euc-korea"))
       (if (boundp '*iso-2022-ss2-7*)
	   (list t		*iso-2022-ss2-7*	"iso-2022-jp-2"))
       )
      )
  )

(defmacro mew-mule-lc-attr (lc alist)
  (` (mew-assoc (, lc) (, alist) 0 nil)))

(defmacro mew-mule-lc-content (attr)
  (` (car (cdr (cdr (, attr))))))

(defmacro mew-mule-lc-symbol (attr)
  (` (car (cdr (, attr)))))

(defmacro mew-mule-content-attr (content alist)
  (` (mew-assoc2 (, content) (, alist) 2 t)))

(defmacro mew-mule-content-coding (attr)
  (` (car (cdr (, attr)))))


;;;
;;; constant
;;;

(defconst mew-profile (expand-file-name "~/.mh_profile"))
(defconst mew-buffer-message     "*Mew message*")
(defconst mew-buffer-mime       " *mew mime*")
(defconst mew-buffer-hello      " *mew hello*")
(defconst mew-buffer-cache      " *mew cache*")
(defconst mew-buffer-watch       "*Mew watch*")
(defconst mew-buffer-tmp        " *mew tmp*")
(defconst mew-buffer-completions "*Mew completions*")
(defconst mew-buffer-whom        "*Mew whom*")
(defconst mew-mime-version "1.0")

(defun mew-buffer-message ()
  (if (and window-system mew-emacs19-p)
      (concat
       mew-buffer-message
       (int-to-string
	(mew-member-match 
	 (cdr (assoc 
	       'window-id
	       (if mew-xemacs-p
		   (screen-parameters (window-screen (selected-window)))
		 (frame-parameters (window-frame (selected-window))))))
	 (sort
	  (mapcar
	   (function 
	    (lambda (frame)
	      (cdr (assoc
		    'window-id
		    (if mew-xemacs-p
			(screen-parameters frame)
		      (frame-parameters frame))))))
	   (if mew-xemacs-p
	       (screen-list)
	     (frame-list)))
	  (function string<))
	 )))
    mew-buffer-message
    ))

(defconst mew-hello-message 
"

Welcome to Mew world.

Mew -- Message interface to Emacs Window

%s

Copyright (C) 1994, 1995  Kazuhiko Yamamoto

Please send comments to kazu@is.aist-nara.ac.jp.

/\\\\ - \\\\/"
)

(defconst mew-error-message 
"

    Sorry, your configuration is *NOT* appropriate for Mew. 
    Please check following items.

        (1) Does \"~/.mh_profile\" exist?
        (2) Does \"~/.mh_profile\" contain \"Path: Mail\" field?
        (3) Does \"~/.mh_profile\" contain \"Draft-Folder: drafts\" field?
        (4) Is the path for MH included in exec-path?
              Put (setq exec-path (cons \"/usr/local/bin/mh\" exec-path))
              in .emacs.
")

;;;
;;; initialize global variables
;;;

(setplist 'mew-current-state 
	  (list
	   'message    nil 
	   'cache      nil
	   'part       nil
	   'window     nil
	   ))

(defun mew-current-get (key)
  (get 'mew-current-state key))

(defun mew-current-set (key value)
  (put 'mew-current-state key value))

(defvar mew-current-marker (make-marker))
(defvar mew-current-marker2 (make-marker))
(defvar mew-decode-marker (make-marker))

(defvar mew-path nil) ;; should be nil

(setq mew-draft-folder nil)
(setq mew-alias-alist  ())
(setq mew-folder-list  ())
(setq mew-folder-alist ())
(setq mew-refile-alist ())
(setq mew-folders (list mew-inbox))

(setq mew-message-citation nil)
(setq mew-message-citation-buffer nil) ;; should be nil
(defvar mew-last-shell-command "")

(setq mew-process-file-alist nil)

(setq mail-header-separator "----")

;;
;; buffer local
;;

(defvar mew-summary-buffer-process nil)
(defvar mew-summary-buffer-string nil)
(defvar mew-summary-buffer-direction nil)
(defvar mew-watch-buffer-process nil)
(defvar mew-draft-buffer-header nil)
(defvar mew-draft-buffer-multipart nil)
(defvar mew-mime-syntax ()
  "Variable to save MIME syntax list")
(defvar mew-summary-buffer-refile nil)
(defvar mew-cache-attribute nil)
(defvar mew-summary-buffer-disp-msg t)
(defvar mew-summary-buffer-left-msgs "-")
(defvar mew-multi-syntax nil)

(mapcar
 (function make-variable-buffer-local)
 (list 'mew-summary-buffer-process
       'mew-summary-buffer-string
       'mew-summary-buffer-direction
       'mew-mime-syntax
       'mew-summary-buffer-refile
       'mew-cache-attribute
       'mew-watch-buffer-process
       'mew-draft-buffer-header
       'mew-draft-buffer-multipart
       'mew-summary-buffer-disp-msg
       'mew-summary-buffer-left-msgs
       'mew-multi-syntax
       ))

;;;
;;; environment setting
;;;

(cond 
 (mew-emacs18-p
  (defun mew-menu-define (symbol maps doc menu)))
 (mew-xemacs-p
  (defmacro mew-menu-define (symbol maps doc menu)
    (` (progn
	 (defvar (, symbol) nil (, doc))
	 (mew-menu-do-define
	  (quote (, symbol)) (, maps) (, doc) (, menu)))))
  (defun mew-menu-do-define (symbol maps doc menu)
    (set symbol menu)
    (fset symbol (list 'lambda '(e)
		       doc
		       '(interactive "@e")
		       '(run-hooks 'activate-menubar-hook)
		       '(setq zmacs-region-stays 't)
		       (list 'popup-menu symbol)))
    (define-key maps 'button3 symbol)))
 (t ;; mew-emacs19-p
  (require 'easymenu)
  (fset 'mew-menu-define 'easy-menu-define))
 )

(if (< max-lisp-eval-depth 1000) (setq max-lisp-eval-depth 1000))

(if mew-summary-mode-map
    ()
  (setq mew-summary-mode-map (make-sparse-keymap))
  (define-key mew-summary-mode-map " "    'mew-summary-show)
  (define-key mew-summary-mode-map "."    'mew-summary-display)
  (define-key mew-summary-mode-map "<"    'mew-summary-display-top)
  (define-key mew-summary-mode-map ">"    'mew-summary-display-bottom)
  (define-key mew-summary-mode-map "\177" 'mew-summary-prev-page)
  (define-key mew-summary-mode-map "\r"   'mew-summary-scroll-up)
  (define-key mew-summary-mode-map "\e\r" 'mew-summary-scroll-down)
  (define-key mew-summary-mode-map "g"    'mew-summary-goto-folder)
  (define-key mew-summary-mode-map "j"    'mew-summary-jump-message)
  (define-key mew-summary-mode-map "o"    'mew-summary-refile)
  (define-key mew-summary-mode-map "O"    'mew-summary-pack)
  (define-key mew-summary-mode-map "i"    'mew-summary-inc)
  (define-key mew-summary-mode-map "s"    'mew-summary-rescan)
  (define-key mew-summary-mode-map "S"    'mew-summary-sort)
  (define-key mew-summary-mode-map "d"    'mew-summary-rmm)
  (define-key mew-summary-mode-map "x"    'mew-summary-exec)
  (define-key mew-summary-mode-map "a"    'mew-summary-reply)
  (define-key mew-summary-mode-map "A"    'mew-summary-reply-with-citation)
  (define-key mew-summary-mode-map "E"    'mew-summary-reedit)
  (define-key mew-summary-mode-map "f"    'mew-summary-forward)
  (define-key mew-summary-mode-map "F"    'mew-summary-multi-forward)
  (define-key mew-summary-mode-map "r"    'mew-summary-redist)
  (define-key mew-summary-mode-map "@"    'mew-summary-mark-mark)
  (define-key mew-summary-mode-map "*"    'mew-summary-mark-hop)
  (define-key mew-summary-mode-map "y"    'mew-summary-save)
  (define-key mew-summary-mode-map "u"    'mew-summary-undo)
  (define-key mew-summary-mode-map "U"    'mew-summary-undo-all)
  (define-key mew-summary-mode-map "n"    'mew-summary-display-down)
  (define-key mew-summary-mode-map "p"    'mew-summary-display-up)
  (define-key mew-summary-mode-map "N"    'mew-summary-display-hop-down)
  (define-key mew-summary-mode-map "P"    'mew-summary-display-hop-up)
  (define-key mew-summary-mode-map "w"    'mew-summary-send)
  (define-key mew-summary-mode-map "B"    'mew-summary-burst)
  (define-key mew-summary-mode-map "/"    'mew-summary-search)
  (define-key mew-summary-mode-map "?"    'mew-summary-search-mark)
  (define-key mew-summary-mode-map "!"    'mew-summary-refile-again)
  (define-key mew-summary-mode-map "#"    'mew-summary-print)
  (define-key mew-summary-mode-map "|"    'mew-summary-pipe-message)
  (define-key mew-summary-mode-map "q"    'mew-summary-suspend)
  (define-key mew-summary-mode-map "Q"    'mew-summary-quit)
  (define-key mew-summary-mode-map "\C-c\C-s" 'mew-summary-isearch-forward)
  (define-key mew-summary-mode-map "\C-c\C-r" 'mew-summary-isearch-backward)
  (define-key mew-summary-mode-map "\el"  'mew-summary-recenter)
  (define-key mew-summary-mode-map "\et"  'mew-summary-uudecode)
  (define-key mew-summary-mode-map "\C-c\C-d" 'mew-pem-decrypt-letter)
  (define-key mew-summary-mode-map "\C-cd"    'mew-pgp-decode-letter)
  (define-key mew-summary-mode-map "\C-ca"    'mew-pgp-add-key)
  (define-key mew-summary-mode-map "\C-c\C-l"
    'mew-summary-convert-local-charset)
  (define-key mew-summary-mode-map "\C-c\177" 'mew-summary-convert-header)
  (define-key mew-summary-mode-map "\es"  'mew-summary-unshar)
  (define-key mew-summary-mode-map "v"    'mew-summary-toggle-disp-msg)
  (define-key mew-summary-mode-map "\ea"  'mew-summary-toggle-analysis)
  (define-key mew-summary-mode-map "\C-c\C-x" 'mew-summary-x-face)
  (define-key mew-summary-mode-map "\C-c\C-q" 'mew-kill-buffer)
  (cond
   (mew-emacs19-p
    (mew-menu-define mew-summary-mode-menu
		     mew-summary-mode-map
		     "Menu used in Mew summary mode."
		     '("Mew"
		       ["Show" mew-summary-show t]
		       ["Next part" mew-summary-display-down t]
		       ["Previous part" mew-summary-display-up t]
		       ["Top" mew-summary-display-top t]
		       ["Bottom" mew-summary-display-bottom t]
		       ["Jump" mew-summary-jump-message t]
		       "----"
		       ["Delete" mew-summary-rmm t]
		       ["Refile" mew-summary-refile t]
		       ["Mark mark" mew-summary-mark-mark t]
		       ["Mark hop" mew-summary-mark-hop t]
		       ["Undo" mew-summary-undo t]
		       ["Undo all" mew-summary-undo-all t]
		       ["Execute" mew-summary-exec t]
		       ["Suspend" mew-summary-suspend t]
		       ["Quit" mew-summary-quit t]
		       "----"
		       ("Manipulate folder"
			["Inc" mew-summary-inc t]
			["Rescan" mew-summary-rescan t]
			["Pack" mew-summary-pack t]
			["Sort" mew-summary-sort t]
			["Burst" mew-summary-burst t]
			["Go to folder" mew-summary-goto-folder t]
			)

		       ("Manipulate file"
			["Save" mew-summary-save t]
			["PEM decrypt" mew-pem-decrypt-letter t]
			["PGP decode" mew-pgp-decode-letter t]
			["Convert to local charset" mew-summary-convert-local-charset t]
			["Display X-Face" mew-summary-x-face t]
			)
		       
		       ("Write/Reply/Forward"
			["Write a mail" mew-summary-send t]
			["Reedit" mew-summary-reedit t]
			["Reply" mew-summary-reply t]
			["Reply with citation" mew-summary-reply-with-citation t]
			["Forward" mew-summary-forward t]
			["Multi forward" mew-summary-multi-forward t]
			)
		       
		       ("Misc"
			["Recenter" mew-summary-recenter t]
			["Uudecode" mew-summary-uudecode t]
			["Unshar" mew-summary-unshar t]
			["Print" mew-summary-print t]
		        ["Pipe message" mew-summary-pipe-message t]
			["Isearch forward" mew-summary-isearch-forward t]
			["Isearch backward" mew-summary-isearch-backward t]
			["PGP add key" mew-pgp-add-key t]
			["Toggle disp msg" mew-summary-toggle-disp-msg t]
                        ["Toggle analysus" mew-summary-toggle-analysis t]
			["Convert header" mew-summary-convert-header t]
			)
		       ))
    (if mew-xemacs-p
	(define-key mew-summary-mode-map 'button2 'mew-summary-mouse-show)
      (define-key mew-summary-mode-map [mouse-2] 'mew-summary-mouse-show))
    )
   )
  )


(if mew-message-mode-map
    ()
  (setq mew-message-mode-map (make-sparse-keymap))
  (define-key mew-message-mode-map " "    'mew-message-next-page)
  (define-key mew-message-mode-map "\177" 'mew-message-prev-page)
  (define-key mew-message-mode-map "n"    'mew-message-next-msg)
  (define-key mew-message-mode-map "p"    'mew-message-prev-msg)
  (define-key mew-message-mode-map "o"    'mew-message-goto-summary)
  (define-key mew-message-mode-map "h"    'mew-message-goto-summary)
  (cond
   (mew-emacs19-p
    (mew-menu-define mew-message-mode-menu
		     mew-message-mode-map
		     "Menu used in Mew message mode."
		     '("Mew/message"
		       ["Next part" mew-message-next-msg t]
		       ["Prev part" mew-message-prev-msg t]
		       ["Next page" mew-message-next-page t]
		       ["Prev page" mew-message-prev-page t]
		       ["Goto summary" mew-message-goto-summary t]))
    )
   )
  )


(if mew-draft-mode-map
    ()
  (setq mew-draft-mode-map (make-sparse-keymap))
  (let ((begin ?\40) (end ?\177))
    (while (<= begin end)
      (define-key mew-draft-mode-map 
	(char-to-string begin) 'mew-draft-keyswitch)
      (setq begin (1+ begin))))
  (define-key mew-draft-mode-map "\C-l"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\t"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-c\t"   'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-d"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-o"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-q"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-t"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-w"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-k"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\r"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\n"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-c\C-y" 'mew-draft-cite)
  (define-key mew-draft-mode-map "\C-cy"    'mew-draft-yank)
  (define-key mew-draft-mode-map "\C-c\C-w" 'mew-draft-check-whom)
  (define-key mew-draft-mode-map "\C-c\C-m" 'mew-draft-make-mime)
  (define-key mew-draft-mode-map "\C-c\C-c" 'mew-draft-send-letter)
  (define-key mew-draft-mode-map "\C-c\C-s" 'mew-pem-sign-letter)
  (define-key mew-draft-mode-map "\C-c\C-e" 'mew-pem-encrypt-letter)
  (define-key mew-draft-mode-map "\C-cs"    'mew-pgp-sign-letter)
  (define-key mew-draft-mode-map "\C-ce"    'mew-pgp-encrypt-letter)
  (define-key mew-draft-mode-map "\C-cb"    'mew-pgp-sign-encrypt-letter)
  (define-key mew-draft-mode-map "\C-cp"    'mew-pgp-insert-public-key)
  (define-key mew-draft-mode-map "\C-c\C-q" 'mew-draft-kill)
  (define-key mew-draft-mode-map "\C-cM"    'mew-draft-prepare-multipart)
  (define-key mew-draft-mode-map "\C-c\C-n" 'mew-fib-next-item)
  (define-key mew-draft-mode-map "\C-c\C-p" 'mew-fib-previous-item)
  (define-key mew-draft-mode-map "\C-c\C-u" 'mew-fib-flush-input)
  (define-key mew-draft-mode-map "\C-c\C-f" 'mew-fib-fill-default)
  (define-key mew-draft-mode-map "\C-c\C-k" 'mew-fib-delete-frame)
  (define-key mew-draft-mode-map "\C-c\177" 'mew-header-mime-decode)
  (cond
   (mew-emacs19-p
    (mew-menu-define
     mew-draft-mode-menu
     mew-draft-mode-map
     "Menu used in Mew draft mode."
     '("Mew/draft"
       ["Citation" mew-draft-cite t]
       ["Yank" mew-draft-yank t]
       ["Prepare multi part message" mew-draft-prepare-multipart t]
       ["Make mime message" mew-draft-make-mime t]
       ["Insert signature" mew-draft-insert-signature t]
       ["Send letter" mew-draft-send-letter t]
       ["Kill draft" mew-draft-kill t]
       "----"
       ("PEM"
	["PEM sign" mew-pem-sign-letter t]
	["PEM encrypt" mew-pem-encrypt-letter t]
	)

       ("PGP"
	["PGP sign" mew-pgp-sign-letter t]
	["PGP encrypt" mew-pgp-encrypt-letter t]
	["PGP sign then encrypt" mew-pgp-sign-encrypt-letter t]
	["PGP insert public key" mew-pgp-insert-public-key t]
	)

       ("FIB"
	["FIB next item" mew-fib-next-item t]
	["FIB previous item" mew-fib-previous-item t]
	["FIB flush input" mew-fib-flush-input t]
	["FIB fill default" mew-fib-fill-default t]
	["FIB delete frame" mew-fib-delete-frame t]
	)

       ))
    )
   )
  )


(if mew-draft-header-map
    ()
  (setq mew-draft-header-map (make-sparse-keymap))
  (define-key mew-draft-header-map "\t"     'mew-draft-header-alias)
  (define-key mew-draft-header-map "\C-c\t" 'mew-draft-header-domain)
  )

(if mew-draft-body-map
    ()
  (setq mew-draft-body-map (make-sparse-keymap))
  (define-key mew-draft-body-map "\t"     'tab-to-tab-stop)
  (define-key mew-draft-body-map "\C-c\t" 'mew-draft-insert-signature)
  )

(if mew-draft-multipart-map
    ()
  (setq mew-draft-multipart-map (make-keymap))
  (let ((begin ?\0) (end ?\177))
    (while (<= begin end)
      (define-key mew-draft-multipart-map 
	(char-to-string begin) 'mew-draft-null-function)
      (setq begin (1+ begin))))
  (define-key mew-draft-multipart-map "a"    'mew-multi-audio)
  (define-key mew-draft-multipart-map "c"    'mew-multi-copy)
  (define-key mew-draft-multipart-map "d"    'mew-multi-delete)
  (define-key mew-draft-multipart-map "D"    'mew-multi-description)
  (define-key mew-draft-multipart-map "e"    'mew-multi-external-body)
  (define-key mew-draft-multipart-map "f"    'mew-multi-find-file)
  (define-key mew-draft-multipart-map "F"    'mew-multi-find-mew-file)
  (define-key mew-draft-multipart-map "l"    'mew-multi-link)
  (define-key mew-draft-multipart-map "m"    'mew-multi-multipart)
  (define-key mew-draft-multipart-map "r"    'mew-multi-rename)
  (define-key mew-draft-multipart-map "T"    'mew-multi-type)
  )

(if mew-minibuffer-map
    ()
  (setq mew-minibuffer-map (make-sparse-keymap))
  (define-key mew-minibuffer-map "\t"     'mew-minibuffer-alias)
  (define-key mew-minibuffer-map "\C-c\t" 'mew-minibuffer-domain)
  (define-key mew-minibuffer-map "\r"     'exit-minibuffer)
  (define-key mew-minibuffer-map "\n"     'exit-minibuffer)
  (define-key mew-minibuffer-map "\C-g"   'abort-recursive-edit)
  )

(if mew-pick-map
    ()
  (setq mew-pick-map (make-sparse-keymap))
  (define-key mew-pick-map "\t"   'mew-pick-complete)
  (define-key mew-pick-map "\r"   'exit-minibuffer)
  (define-key mew-pick-map "\n"   'exit-minibuffer)
  (define-key mew-pick-map "\C-g" 'abort-recursive-edit)
  )

;; should remove?
(cond 
 (mew-mule-p
  (define-program-coding-system nil mew-prog-inc  mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-scan mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-send mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-pick mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-mime-encode *noconv*)
  (define-program-coding-system nil mew-prog-mime-decode *noconv*)
  )
 )

(cond 
 ((null mew-draft-mode-syntax-table)
  (setq mew-draft-mode-syntax-table
	(make-syntax-table text-mode-syntax-table))
  (set-syntax-table mew-draft-mode-syntax-table)
  (modify-syntax-entry ?% "." mew-draft-mode-syntax-table))
 )

;;;
;;; Basic functions
;;;

(defun mew-version-show ()
  (interactive)
  (message "%s" mew-version))


(defun mew-replace-character (string from to)
  (let ((len (length string))
	(cnt 0))
    (while (< cnt len)
      (if (equal (aref string cnt) from)
	  (aset string cnt to))
      (setq cnt (1+ cnt)))
    string
    ))

(defun mew-y-or-n-p (string)
  (interactive)
  (if mew-emacs-y-or-n-p-use
      (y-or-n-p string)
    (let ((char nil) (ociea cursor-in-echo-area))
      (unwind-protect
	  (progn
	    (setq cursor-in-echo-area t)
	    (catch 'break
	      (while t
		(message (concat string "(y=RET or n=SPC) "))
		(setq char (read-char))
		(cond
		 ((equal char ?y)
		  (setq char t)
		  (throw 'break nil))
		 ((equal char ?\r)
		  (setq char t)
		  (throw 'break nil))
		 ((equal char ?n)
		  (setq char nil)
		  (throw 'break nil))
		 ((equal char 32)
		  (setq char nil)
		  (throw 'break nil))
		 (t (ding))
		 )
		))
	    )
	(setq cursor-in-echo-area ociea)
	)
      char ;; return value
      ))
  )

(defun mew-member-match (str list)
  (let ((n 0))
    (catch 'member
      (while list
	(if (equal (downcase (car list)) (downcase str))
	    (throw 'member n))
	(setq list (cdr list))
	(setq n (1+ n))
	))
    ))

(defun mew-delq (key list)
  (cond 
   ((null list) ())
   ((equal key (car list)) (cdr list))
   (t (cons (car list) (mew-delq key (cdr list))))))

(defun mew-folder-to-dir (folder)
  (if (equal (aref folder 0) ?+)
      (substring folder 1 nil)
    folder)
  )

(defun mew-dir-to-folder (dir)
  (if (equal (aref dir 0) ?+)
      dir
    (concat "+" dir))
  )

;(defmacro mew-draft-to-mime (draft)
;  (` (expand-file-name (file-name-nondirectory (, draft)) 
;		       mew-draft-mime-folder))
;  )

(defmacro mew-draft-to-mime (draft)
  (` (concat mew-draft-mime-folder "/" (file-name-nondirectory (, draft)))))

(defun mew-make-folder (folder)
  (mew-make-directory (mew-folder-to-dir folder)))

(defun mew-make-directory (path)
  (let ((parent (directory-file-name (file-name-directory path))))
  (if (null (file-directory-p parent))
    (mew-make-directory parent))
  (if (and (file-exists-p path) (not (file-directory-p path)))
      (delete-file path))
  (call-process "mkdir" nil nil nil path)
  ))

;(defun mew-rassoq (a alist)
;  (cond 
;   ((null alist) nil)
;   ((equal (cdr (car alist)) a) (car alist))
;   (t (mew-rassoq a (cdr alist)))
;   ))

;; I don't like such a procedural programming, but for max depth safety.
(defun mew-rassoq (a alist)
  (catch 'conscell
    (while alist
      (if (equal (cdr (car alist)) a)
	  (throw 'conscell (car alist)))
      (setq alist (cdr alist)))))

(defun mew-match (pos &optional string)
  (cond 
   ((stringp string)
    (substring string (match-beginning pos) (match-end pos)))
   (t 
    (buffer-substring (match-beginning pos) (match-end pos)))
   ))

;(defun mew-assoc-match (string alist)
;  (cond ((null alist) ())
;	((string-match (car (car alist)) string) (car alist))
;	(t (mew-assoc-match string (cdr alist)))))

(defun mew-assoc (key alist nth match)
  (let ((case-fold-search t))
    (cond
     ((null alist) ())
     ((and match
	   (stringp (nth nth (car alist)))
	   (string-match (nth nth (car alist)) key))
      (car alist))
     ((equal (nth nth (car alist)) key)
      (car alist))
     ((eq (nth nth (car alist)) t)
      (car alist))
     (t (mew-assoc key (cdr alist) nth match))
     )
    ))

;; gee

(defun mew-assoc2 (key alist nth exact)
  (let ((case-fold-search t))
    (cond
     ((null alist) ())
     ((and exact 
	   (stringp (nth nth (car alist)))
	   (equal (downcase key) (downcase (nth nth (car alist)))))
      (car alist))
     ((equal (nth nth (car alist)) key)
      (car alist))
     ((eq (nth nth (car alist)) t)
      (car alist))
     (t (mew-assoc key (cdr alist) nth exact))
     )
    ))

(defmacro mew-directory-empty-p (dir)
  (` (null (car (cdr (cdr (directory-files (, dir)))))))
  )

(defun mew-which(file path)
  (catch 'loop
    (while path
      (if (file-exists-p (expand-file-name file (car path)))
	  (throw 'loop (expand-file-name file (car path)))
	(setq path (cdr path)))
      )
    ))

;;;
;;; Window configuration stack
;;;

(defvar mew-window-stack nil)

(defun mew-window-push ()
  (interactive)
  (let* ((key (cond
	       ((and mew-emacs19-p (not mew-xemacs-p))
		(selected-frame))
	       (t 'dummy)
	       ))
	 (stack (mew-alist-search mew-window-stack key)))
    (setq stack (cons (current-window-configuration) stack))
    (setq mew-window-stack
	  (mew-alist-add mew-window-stack key stack))
    ))

(defun mew-window-pop ()
  (interactive)
  (let* ((key (cond
	       ((and mew-emacs19-p (not mew-xemacs-p))
		(selected-frame))
	       (t 'dummy)
	       ))
	 (stack (mew-alist-search mew-window-stack key)))
    (if stack
	(set-window-configuration (car-safe stack))
      (if (and mew-emacs19-p
	       (not mew-xemacs-p)
	       (nth 1 (frame-list))
	       (mew-y-or-n-p "Delete this frame ? "))
	  (delete-frame)
	(message "Window Stack is empty.")))
    (setq stack (cdr-safe stack))
    (setq mew-window-stack
	  (mew-alist-add mew-window-stack key stack))
    ))


;;;
;;; Bootstrap --- mew, mew-read, and mew-send
;;;

(defun mew (&optional arg)
  (interactive "P")
  (mew-window-push)
  (if (null mew-path) (mew-init))
  (if arg
      (mew-summary-goto-folder) ;; C-u
    (if (and mew-summary-cache-use (null (get-buffer mew-inbox)))
	(let ((cache (mew-expand-file-name mew-summary-cache-file mew-inbox)))
	  (switch-to-buffer (get-buffer-create mew-inbox))
	  (if (file-exists-p cache) (insert-file-contents cache))
	  (mew-summary-mode) ;; gee-dirty
	  ))
    (mew-summary-inc))
  )

(defun mew-read (&optional arg)
  (interactive "P")
  (mew-window-push)
  (if (null mew-path) (mew-init))
  (if arg
      (mew-summary-goto-folder) ;; C-u
    (mew-summary-inc))
  )

(defun mew-send (&optional to cc subject)
  (interactive)
;  (mew-window-push)
  (mew-current-set 'window (current-window-configuration))
  (if (null mew-path) (mew-init))
  (mew-summary-send t to cc subject)
  )

;;
;; Functions for boot time
;;

(defun mew-init ()
  (mew-hello)
  (if mew-demo
      (mew-demo mew-demo-string))
  (message "Setting mew world ...")
  (mew-set-environment)
  (run-hooks 'mew-init-hook)
  (message "Setting mew world ... done")
  )

(defun mew-center-line ()
  (save-excursion
    (end-of-line)
    (let ((ll (current-column)))
      (beginning-of-line)
      (indent-to (/ (- (window-width) ll) 2))
      )
    ))

(defun mew-center-region (from to)
  (interactive "r")
  (if (> from to)
      (let ((tem to))
        (setq to from from tem)))
  (save-excursion
    (save-restriction
      (narrow-to-region from to)
      (goto-char from)
      (while (not (eobp))
        (mew-center-line)
        (forward-line 1)))))

(defun mew-hello ()
  (let ((nbuf (get-buffer-create mew-buffer-hello)))
    (mew-window-configure nbuf '(1 0))
    (erase-buffer)
    (insert (format mew-hello-message mew-version))
    (mew-center-region (point-min) (point-max))
    (end-of-line)
    (insert (make-string (1- (- (window-width) (current-column))) 32))
    (sit-for 0) ;; to redraw
    ))

(defun mew-set-environment ()
  (let ((xerr nil))
    (condition-case err
	(progn
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (erase-buffer)
	  (if (file-exists-p mew-profile)
	      (insert-file-contents mew-profile)
	    (setq xerr (format "No %s" mew-profile))
	    (error ""))
	  (if (setq mew-path (mew-field-get-value "Path:"))
	      (setq mew-path
		    (expand-file-name 
		     (mew-header-delete-ws mew-path)
		     (expand-file-name "~")))
	    (setq xerr "No Path:")
	    (error ""))
	  (if (setq mew-draft-folder (mew-field-get-value "Draft-Folder:"))
	      (setq mew-draft-folder
		    (format "+%s" (mew-folder-to-dir
				   (mew-header-delete-ws		  
				    mew-draft-folder))))
	    (setq xerr "No Draft-Folder:")
	    (error ""))
	    ;;
	    (if (null mew-alias-alist)
		(setq mew-alias-alist (mew-alias-make-alist)))
	    ;;
	    (setq mew-folders-file 
		  (expand-file-name mew-folders-file mew-path))
	    (cond 
	     ((null mew-folder-list)
	      (setq mew-folder-list (mew-folder-make-list))
	      (setq mew-folder-alist (mew-folder-make-alist mew-folder-list))
	      (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
	      ))
	    )
      (error (set-buffer mew-buffer-hello)
	     (insert mew-error-message)
	     (set-buffer-modified-p nil)
	     (setq buffer-read-only t)
	     (if xerr (error xerr)
	       (error "If you have ~/Mail/.mew-from-alist, remove it. Or maybe exec-path error")))
      )
    ))

;;;
;;; window configuration
;;;

(defun mew-window-configure (nbuf action)
;;; action : summary, message or list
;;; list for action (1 0)  for Summary only
;;; list for action (3 10) for Summary and Message
  (if (equal action 'summary)
      (mew-current-set 'message nil))
  (let* ((windows
	  (if (listp action) 
	      action
	    (car (cdr (assq action mew-window-configuration)))))
	 (obufwin (get-buffer-window (current-buffer)))
	 (msgwin  (get-buffer-window (mew-buffer-message)))
	 (height nil) (winsum nil) (sumheight 0) (msgheight 0))
    (cond 
     (mew-window-use-full
      ;; Delete other windows and use full emacs window.
      (delete-other-windows)
      (setq height (window-height (selected-window))))
     (t
      (setq height
	    (+ (if obufwin (window-height obufwin) 0)
	       (if msgwin  (window-height msgwin)  0)
	       ))))
    (if (get-buffer (mew-buffer-message))
	(delete-windows-on (mew-buffer-message))
      (save-excursion
	(set-buffer (get-buffer-create (mew-buffer-message)))
	(mew-message-mode)))
    (setq winsum (apply (function +) windows))
    (if (not (zerop (nth 0 windows)))
	(setq sumheight (max window-min-height
			     (/ (* height (nth 0 windows)) winsum))))
    (if (not (zerop (nth 1 windows)))
	(setq msgheight (max window-min-height
			     (- height sumheight))))
    (setq height (+ sumheight msgheight))
    (enlarge-window (max 0 (- height (window-height (selected-window)))))
    (if (null (zerop sumheight))
	(switch-to-buffer nbuf 'norecord))
    (if (zerop msgheight)
	()
      (split-window nil sumheight)
      (other-window 1)
      (switch-to-buffer (mew-buffer-message) 'norecord))
    ))

;;;
;;; Input method
;;;

(defun mew-input-folder (default)
  (let ((folder))
    (setq folder (completing-read (format "Folder name (%s): " default)
				  mew-folder-alist
				  nil
				  nil  ;; not require match
				  "+"))
    (directory-file-name 
     (if (or (string= folder "") (string= folder "+"))
 	 default
       folder))))

(defun mew-input-range (folder)
  "return (range erase 'update)"
  (mew-summary-multipart-delete)
  ;; for the case when parts is expanded in the bottom of the folder
  (let ((range (read-string "Range (update): " "")))
    (if (not (or (string= range "") (string= range "update")))
	(list range t nil) ;; erase
      (cond
       ((get-buffer folder)
	(save-excursion
	  (set-buffer folder)
	  (if (equal (point-min) (point-max))
	      (list "all" nil 'update) ;; no need to erase
	    (goto-char (point-max))
	    (forward-line -1)
	    (list (concat (int-to-string 
			   (1+ (string-to-int (mew-summary-message-number))))
			  "-" 
			  "last")
		  nil 'update) ;; not erase
	    ))
	)
       (t (list "all" nil 'update)) ;; no need to erase
       )
      )))


(defun mew-input-address (prompt) ;; prompt="To:"
  (read-from-minibuffer (concat prompt " ") "" mew-minibuffer-map nil)
  )


(defun mew-input-filename (&optional prompt default)
  (expand-file-name (read-file-name (or prompt "File : ") 
				    nil 
				    (or default "~")
				    nil))
  )

(defun mew-input-directory-name ()
  (let ((dir (expand-file-name (read-file-name "Directory : " nil "~" t))))
    (if (file-directory-p dir)
	dir
      (mew-input-directory-name)
      )
    ))

(defun mew-input-string (prompt subdir default)
  (let ((input (read-string (format prompt subdir default) "")))
    (if (string= input "") default input))
  )

(defun mew-input-type (filename type-list default)
  (let ((completion-ignore-case t)
	(type))
    (setq type (completing-read
		(format "Type for %s (%s): " filename default)
		(mapcar (function (lambda (x) (cons x x)))
			type-list)
		nil
		t  ;; not require match
		""))
    (if (string= type "") default type)
    ))


;;;
;;; Misc
;;;

(defun mew-boundary-get (&optional string)
  (if (null string) (setq string "Next_Part"))
  (format mew-default-boundary
	  string
	  (mew-replace-character (current-time-string) 32 ?_) ; 32 == " "
	  )
  )

(defun mew-expand-file-name (msg &optional buffer-or-name)
  (if (bufferp buffer-or-name)
      (setq buffer-or-name (buffer-name buffer-or-name)))
  (and buffer-or-name 
       (setq buffer-or-name (directory-file-name buffer-or-name)))
  (if buffer-or-name
      (expand-file-name 
       (concat (mew-folder-to-dir buffer-or-name) "/" (mew-folder-to-dir msg))
       mew-path)
    (expand-file-name (mew-folder-to-dir msg) mew-path))
  )
  

;;;
;;; Message header oparation
;;;

(defun mew-field-get-value (field)
  "currently, when no match, it returns nil."
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(match nil)
	(ret nil)) ;; (concat nil "foo") -> "foo"
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s[ \t]*" field) nil t)
	  (setq key (mew-match 0))
	  (setq start (match-end 0))
	  (if (string-match "^-*$" key) (throw 'header ret))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (setq match (buffer-substring start (1- (point))))
	  (if (null ret)
	      (if (null (string= "" match))
		  (setq ret match))
	    (if (null (string= "" match))
		(setq ret (concat ret "," match))))
	)))
    ret
    ))
	  
(defun mew-field-get-line (field)
;; return  with ^J
;; currently, when no match, it returns nil.
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(ret nil)) ;; (concat nil "foo") -> "foo"
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s" field) nil t)
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header ret))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (setq ret (concat ret (buffer-substring start (point))))
	  ))
      )
    ret
    ))

(defun mew-field-delete-line (field)
  (let ((case-fold-search t)
	(start nil)
	(key nil))
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s" field) nil t)
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (delete-region start (point))
	  ))
      )))

(defun mew-field-resent-line (field)
  (let ((case-fold-search t)
	(start nil)
	(key nil))
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s" field) nil t)
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (beginning-of-line)
	  (insert "Prev-")
	  ))
      )))


(defun mew-field-insert-last (field value)
  (save-excursion
    (let ((case-fold-search t))
      (if (string= value "")
	  ()
	(goto-char (point-min))
	(re-search-forward 
	 (format "^-*$\\|^%s" field))
	(let ((match (mew-match 0)))
	  (if (string= (downcase match) (downcase field))
	      (error "Field %s already exists." field)
	    (beginning-of-line)
	    (insert-before-markers (format "%s %s\n" field value)))
	  )
	))))

(defun mew-field-insert-here (field &optional value)
  (if value
      (insert (format "%s %s\n" field value))
    (insert (format "%s \n" field))))

;;;
;;; field parser
;;;

(defun mew-header-syntax (string)
  (mew-header-split
   (mew-header-concat-list
    (mew-header-delete-ws-list 
     (mew-header-separate-quote string)))
   ?;))

(defun mew-header-split (string sep)
 (let ((quote nil)
       (len (length string))
       (ret nil)
       (start 0)
       (n 0))
   (while (< n len)
     (cond
      ((equal (aref string n) ?\")
       (setq quote (not quote)))
      ((equal (aref string n) sep)
       (if (null quote)
	   (progn
	     (setq ret (append ret (list (substring string start n))))
	     (setq start (1+ n))
	     )
	 ))
      )
     (setq n (1+ n))
     )
   (append ret (list (substring string start n)))
   ))

(defmacro mew-header-concat-list (list)
  (` (mapconcat (function (lambda (x) x)) (, list) "")))

(defmacro mew-header-delete-ws-list (list)
  (` (mapcar (function mew-header-delete-ws) (, list))))

(defmacro mew-header-delete-ws2-list (list)
  (` (mapcar (function mew-header-delete-ws2) (, list))))

(defun mew-header-delete-ws (string)
  (if (eq (aref string 0) ?\")
      (substring string 1 (1- (length string)))
    (while (string-match "[ \t]+\\|\n[ \t]+\\|([^()]*)" string)
      ;; regular expression cannot express recursive paretheses.
      ;; delete () again and again.
      (setq string (concat (substring string 0 (match-beginning 0))
			   (substring string 
				      (match-end 0)
				      (length string)))))
    string))

(defun mew-header-delete-ws2 (string)
  (if (eq (aref string 0) ?\")
      string
    (while (string-match "[ \t]+\\|\n[ \t]+\\|([^()]*)" string)
      (setq string (concat (substring string 0 (match-beginning 0))
			   (substring string 
				      (match-end 0)
				      (length string)))))
    string))

;(defmacro mew-header-separate-quote (string)
;  "Return list"
;  (` (mew-header-separate-quote-loop (, string) ())))
;
;(defun mew-header-separate-quote-loop (string list)
;  "Return list"
;  (cond
;   ((string= "" string) list)
;   ((string-match "\"[^\"]*\"" string)
;    (mew-header-separate-quote-loop
;     (substring string (match-end 0) (length string))
;     (append list 
;	     (list (substring string 0 (match-beginning 0)))
;	     (list (mew-match 0 string)))))
;   (t (append list (list string)))
;   ))

(defun mew-header-separate-quote (string)
  (let (ret begin (end 0) (lastend 0))
    (while (string-match "\"[^\"]*\"" string end)
      (setq lastend end)
      (setq begin (match-beginning 0))
      (setq end (match-end 0))
      (if (equal begin lastend)
	  (setq ret (append ret (list (substring string lastend end))))
	(setq ret (append ret
			  (list (substring string lastend begin)
				(substring string begin end)))))
      )
    (if (equal (length string) end)
	ret
      (append ret (list (substring string end nil))))
    ))

;;;
;;; extract address
;;;

(defmacro mew-header-extract-addr-list (list)
  (` (mapcar (function mew-header-extract-addr) (, list))))

;(defun mew-header-extract-addr (str)
;  "Extracts a real e-mail address from STR and returns it.
;e.g. \"Mine Sakurai <m-sakura@ccs.mt.nec.co.jp>\" 
;  ->  \"m-sakura@ccs.mt.nec.co.jp\"."
;  (let ((n (string-match "<[^>]*>" str)))
;    (if (null n) 
;        str
;      (substring str (1+ n) (1- (match-end 0))))
;    ))

(defun mew-header-extract-addr (str)
  "Extracts a real e-mail address from STR and returns it.
e.g. \"Mine Sakurai <m-sakura@ccs.mt.nec.co.jp>\" 
  ->  \"m-sakura@ccs.mt.nec.co.jp\"."
  (if (string-match ".*<\\([^>]*\\)>" str) ;; .* to extract last <>
      (mew-match 1 str)
    str
    )
  )

(defun mew-header-delete-nullstring-list (list)
  (if (null list)
      ()
    (cond 
     ((string= (car list) "") 
      (mew-header-delete-nullstring-list (cdr list)))
     (t 
      (cons (car list) (mew-header-delete-nullstring-list (cdr list)))))))

(defun mew-header-user-collect (list)
  (let ((fields (mapcar (function mew-field-get-value) list))
	(users "") (count 0) (begin 0))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-loop-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (equal count mew-loop-depth) (setq users (substring users 0 begin)))
    (mapcar (function downcase)
	    (mew-header-delete-nullstring-list
	     (mew-header-delete-at-list
	      (mew-header-extract-addr-list
	       (mew-header-split
		(mew-header-concat-list
		 (mew-header-delete-ws2-list
		  (mew-header-separate-quote users)))
		?,)))))
    ))


(defun mew-header-address-collect (list)
  (let ((fields (mapcar (function mew-field-get-value) list))
	(users "") (count 0) (begin 0))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-loop-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (equal count mew-loop-depth) (setq users (substring users 0 begin)))
    (mapcar (function downcase)
	    (mew-header-delete-nullstring-list
	     (mew-header-extract-addr-list
	      (mew-header-split
	       (mew-header-concat-list
		(mew-header-delete-ws2-list
		 (mew-header-separate-quote users)))
		?,))))
    ))


(defmacro mew-header-delete-at-list (list)
  (` (mapcar (function mew-header-delete-at) (, list))))

;; never macro for mapcar
;; by Motonori NAKAMURA <motonori@cs.ritsumei.ac.jp> on 17 Aug 1994
(defun mew-header-delete-at (string)
  (if (string-match "@.*:" string)
      (setq string (substring string (match-end 0) (length string)))
    (setq string (substring string 0 (string-match "%" string)))
    (setq string (substring string 0 (string-match "@" string)))
    (substring string 0 (string-match ":;" string))
    ))

;;
;; e-mail address canonical form 
;;

(defmacro mew-header-canform-list (list)
  (` (mapcar (function mew-header-canform) (, list))))

(defun mew-header-canform (string)
  "Complete STRING with mew-mail-domain."
  (if (string= mew-mail-domain "")
      string
    (if (string-match "^.*@\\(.*\\)$" string)
	(let ((last-dom nil)
	      (dom (mew-match 1 string))
	      (rest-dom nil)
	      (len (length mew-mail-domain)))
	  (if (string-match "^.*\\.\\([^.]*\\)$" dom)
	      (setq last-dom (mew-match 1 dom))
	    (setq last-dom dom))
	  (if (string-match (concat "\\(" last-dom "\\)") mew-mail-domain)
	      ;; \\(?
	      (setq rest-dom (substring mew-mail-domain (match-end 0) len)))
	  (if rest-dom
	      (concat string rest-dom)
	    string))
      (concat string "@" mew-mail-domain))))

;;
;; e-mail address alias
;;

(defun mew-alias-expand (alias)
  "The function of MH alias expantion. If ALIAS can be expanded, it 
returns the expansion. Otherwise, it returns ALIAS itself."
  (let ((expn (cdr (assoc alias mew-alias-alist))))
    ;assoc returns nil if alist is nil
    (if expn expn alias)))

(defun mew-alias-make-alist ()
  "Make alias alist with association of (alias . expantion) from
MH \"ali\" command. Currently, only \"user: user@domain\" syntax
is supported."
  (save-excursion
    (let ((case-fold-search t)
	  (alias nil)
	  (expn nil)
	  (alist nil)
	  (buf (get-buffer-create mew-buffer-tmp)))
      (set-buffer buf)
      (erase-buffer)
      (call-process mew-prog-ali nil t nil "-nolist" "-nonormalize" "-nouser")
      ;; concat separated lines by comma
      (goto-char (point-min))
      (while (re-search-forward ",$" nil t)
	(end-of-line)
	(forward-char 1)
 	(delete-backward-char 1))
      ;;
      (goto-char (point-min))
      (while (re-search-forward "^\\([-\\+a-z0-9_]+\\):\\s-+\\(.*\\)$" nil t)
	(setq alias (mew-match 1)
	      expn (mew-match 2))
	;; append for first assoc comes first
	(setq alist (append alist (list (cons alias expn)))))
      ;; load from-alist
      (if (not mew-from-alist)
	  (setq mew-from-alist
		(mew-alist-load mew-from-file-name)))
      (mapcar
       '(lambda (arg)
	  (if (and (car arg)
		   (not (assoc (mew-header-delete-at (car arg)) alist)))
	      (setq alist 
		    (append alist
			    (list 
			     (cons (mew-header-delete-at (car arg)) (car arg)))
			    ))))
       mew-from-alist)
      alist ; return value
      )))

(defun mew-header-expand-alias-list (list)
  "The function of MH alias expansion. One \"ali\" command is executed
   for all the members of LIST. And, list is returned."
  (let ((field-value nil)
	(multi-line nil))
    (if (null list)
	()
      (save-excursion
	;; alias extraction
        (set-buffer mew-buffer-tmp)
        (erase-buffer)
	(apply 'call-process 
	       mew-prog-ali nil mew-buffer-tmp nil "-list" list)
	(goto-char (point-min))
        (setq list nil)
        (while (not (eobp))
          (let ((p (point))
                (theline nil))
            (end-of-line)
            (setq theline (buffer-substring p (point)))
            (delete-region p (point))
            (setq list (append list (list theline)))
            (forward-line)))
        list))))

;;
;; folders
;;


(defmacro mew-folder-make-alist (list)
  (` (mapcar (function mew-folder-pair) (, list))))

;; never defmacro for (mapcar)
(defun mew-folder-pair (string)
  (cons (format "+%s" string) (format "+%s" string)))

(defmacro mew-refile-make-alist (list)
  (` (mapcar (function mew-refile-pair) (, list))))


(defun mew-refile-pair (string)
  (if (mew-folders-ignore-p string)
      nil
    (cons (downcase (file-name-nondirectory (directory-file-name string)))
	  (format "+%s" (directory-file-name string)))

    ))

(defun mew-folders-ignore-p (string)
  (let ((ilist mew-folders-ignore))
    (catch 'ignore
      ;; while always returns nil
      (while ilist
	(if (string-match (concat "^" (car ilist))
			  string)
	    (throw 'ignore t))
	(setq ilist (cdr ilist))))))

(defun mew-folder-make-list ()
  (save-excursion
    (let ((case-fold-search t)
	  (folders ())
	  (folder nil)
	  (start nil))
      (set-buffer (get-buffer-create mew-buffer-tmp))
      (erase-buffer)
      (if (file-exists-p mew-folders-file)
	  (insert-file-contents mew-folders-file)
	(call-process mew-prog-folders nil t nil "-fast" "-recurs")
	)
      (goto-char (point-min))
      (while (not (eobp))
        (setq start (point))
        (forward-line 1)
	(setq folder (buffer-substring start (1- (point))))
	(if (string-match (concat "^" (car folders) "/") folder)
	    (setq folders 
		  (cons folder 
			(cons (concat (car folders) "/"	)
			      (cdr folders))))
	  (setq folders (cons folder folders)))
	)
      folders ;; return value
      )))

;;;
;;; Summary mode
;;;

(defun mew-summary-mode ()
  "Major mode for reading messages.
The keys that are defined for this mode are:

SPC	Read through messages. See mew-summary-show-direction to set 
	'up,'down,'next(current direction) or 'stop. Default is 'down.
DEL	Back scroll this message.
	Unnecessary header fields are hidden over the window. Type DEL
	to see them when message are displayed.
.	Display this message or part. 
	If without MIME analysis, force to analyze this message.

RET	1 line scroll up this message.
ESC RET 1 line scroll down this message.

C-n	Go to the next line.
C-p	Go to the previous line.
n	Display below message or part.
p	Display above message or part.
N	Jump to below * marked message or display below message
	around multipart.
P	Jump to above * marked message or display above message
	around multipart.
j	Jump to a message according to inputed number.

i	Incorporate +inbox asynchronously.
s	Scan this folder asynchronously. 
g	Go to inputed folder.

w	Prepare draft to send a message.
a	Reply this message or this part.
A	Reply this message and insert it in draft buffer.
f	Forward this message as MIME format.
F	Forward @ marked messages as MIME format.
E	Edit this message again to send. In a draft folder, it just edit
	the message. Otherwise, copy the message to draft folder, then
	edit.
r	Re-distribute this message with Resent-To: and Resent-Cc:.

v	Toggle summary window only and summary & message windows.
ESC a	Toggle with/out MIME analysis.
ESC l	Recenter this folder.

o	Mark o(refile) this message or this part's message. If already
	marked as o, it prints which folder this message is refiled.
!	Refile this message or part to the previous refile folder.
d	Mark D(delete) on this message.
*	Mark * on this message. Use N or P to jump to * marked message.
@	Mark @ on this message for F, ESC s, and ESC t.
u	Cancel mark on this message.
U	Cancel all inputed mark.
x	Process marked messages. To cancel * mark, use u or U.

C-cC-s	Incremental search forward on the message buffer.
C-cC-r	Incremental search backward on the message buffer.

ESC s	Apply unshar on @ marked messages.
ESC t	Apply uudecode on @ marked messages.

/	Pick messages according to inputed condition, then scan.
?	Pick messages according to inputed condition, then mark *.

y	Copy this message or save this part as inputed file name.
#	Print this message or this part.
|	Pipe this message.

S	Sort messages in this folder.
O	Pack messages in this folder.

B	Burst messages encapsulated in MIME.

q	Switch to other buffer.
Q	Exit mew.
C-cC-q	Kill this buffer.

C-cC-l	Localize the transformed charset.
C-cd	Manual PGP decryption/vrfy.
C-ca	Add PGP public key in this message to pubring.pgp.
C-cC-d	Manual PEM decryption/vrfy.
C-cC-x	Display xface.

Range means as follows;
	<num1>-<num2>, <num>:+N, <num>:-N
	first:N, prev:N, next:N, last:N, update

'update' means the range between the last message included summary
mode + 1 and the real last message on the folder.

Pick condition means as follews;
	<cond1> -and <cond2>, <cond1> -or <cond2>
	-from <pat>, -to <pat>, -search <pat> 
	-date <date>, -before <date>, -after <date>
	--xxx <pat>
	-lbrace or \"(\", -rbrace or \")\"
"
  (interactive)
  (setq major-mode 'mew-summary-mode)
  (setq mode-name "Summary")
  (use-local-map mew-summary-mode-map)
  (setq buffer-read-only t)
  (setq truncate-lines t)
  ;; xxx
  (if (equal (nth (- (length mode-line-format) 2) mode-line-format)
	     '(-3 . "%p"))
      (setq mode-line-format
	    (reverse
	     (append '("-%-" " more]" mew-summary-buffer-left-msgs "[")
		     (cdr (cdr (reverse mode-line-format)))))
	    ))
  (if mew-xemacs-p
      (setq mode-motion-hook 'mode-motion-highlight-line))
  (run-hooks 'mew-summary-mode-hook)
  )

(defun mew-summary-message-number ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at mew-summary-message-regex)
	(mew-match 1)
      nil
      )
    ))

(defun mew-summary-part-number ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at mew-summary-part-regex)
	(mew-match 1)
      nil
      )
    ))

;;
;; process control
;;

;; mew-summary-buffer-process is a key to see if exclusive

(defun mew-summary-exclusive-p ()
  (cond
   ((processp mew-summary-buffer-process)
    (message "%sing now. Try again later." (process-name mew-summary-buffer-process))
    nil) ;; not exclusive
   (t t)) ;; exclusive
  )

;;
;; inc
;; 

(defvar mew-summary-inc-start nil)

;; drop folder is meaningless at present.
(defun mew-summary-inc ()
  (interactive)
  (set-buffer (get-buffer-create mew-inbox))
  (mew-summary-mode)
  (mew-window-configure (current-buffer) 'summary)
  (mew-current-set 'message nil)
  (mew-current-set 'part nil)
  (mew-current-set 'cache nil)
  (setq mew-summary-buffer-direction 'down)
  (mew-summary-multipart-delete)
  (if (null (mew-summary-exclusive-p))
      ()
    (condition-case err
	(progn
	  (message "Incing ...")
	  (goto-char (point-max))
	  (setq mew-summary-inc-start (point))
	  (setq mew-summary-buffer-process
		(start-process "Inc" (current-buffer) 
			       mew-prog-inc
			       "-width"
			       (if (< (window-width) 80)
				   "80"
				 (int-to-string (window-width)))
			       ))
	  (set-process-filter mew-summary-buffer-process
			      'mew-summary-inc-filter)
	  (set-process-sentinel mew-summary-buffer-process
				'mew-summary-inc-sentinel)
	  (process-kill-without-query mew-summary-buffer-process)
	  )
      (quit
       (set-process-sentinel mew-summary-buffer-process nil)
       (setq mew-summary-inc-start nil)
       (setq mew-summary-buffer-process nil)
       (setq mew-summary-buffer-string nil)
       )
      )
    ))

(defun mew-summary-inc-filter (process string)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (setq mew-summary-buffer-string 
	  (concat mew-summary-buffer-string string)) ;; nil can concat
    (let ((buffer-read-only nil))
      (if (string-match "^Incorpo.*\n\n" mew-summary-buffer-string)
	  (setq mew-summary-buffer-string
		(substring mew-summary-buffer-string (match-end 0))))
      (while (string-match "^ *[0-9]+\\([ +]\\).*\n" mew-summary-buffer-string)
	;; leave mew-summary-buffer-string if "inc: no messages"
	(goto-char (point-max))
	(if (equal " " (mew-match 1 mew-summary-buffer-string))
	    (insert (substring mew-summary-buffer-string 0 (match-end 0)))
	  (insert (substring mew-summary-buffer-string 0 (match-beginning 1)))
	  (insert " ")
	  (insert (substring mew-summary-buffer-string
			     (match-end 1)
			     (match-end 0)))
	  )
	(setq mew-summary-buffer-string
	      (substring mew-summary-buffer-string (match-end 0)))
	)
      )))

(defun mew-summary-inc-sentinel (process event)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (if mew-summary-cache-use
	(write-region
	 (point-min)
	 (point-max)
	 (mew-expand-file-name mew-summary-cache-file (buffer-name))))
    (set-buffer-modified-p nil)
    (message "Incing ... done")
    (if mew-summary-buffer-string
	(cond 
	 ((string-match "^inc: no mail" mew-summary-buffer-string)
	  (message "No new mail"))
	 ((string-match "^inc: " mew-summary-buffer-string)
	  (message "Inc error"))
	 (t 
	  (let ((buffer-read-only nil))
	    (goto-char (point-max))
	    (insert mew-summary-buffer-string)
	    (goto-char mew-summary-inc-start)
	    (keep-lines mew-summary-message-regex))
	  )
	 ))
    (mew-summary-highlight-lines)
    (set-process-sentinel mew-summary-buffer-process nil)
    (setq mew-summary-inc-start nil)
    (setq mew-summary-buffer-process nil)
    (setq mew-summary-buffer-string nil)
    ))

;;
;; scan
;;

(defun mew-summary-scan (folder range-erase)
  (let ((range (car range-erase)) 
	(erase (car (cdr range-erase))))
    (set-buffer (get-buffer-create folder))
    (if (null (mew-member folder mew-folders))
	(setq mew-folders (cons folder mew-folders)))
    (mew-summary-mode)
    (mew-window-configure (current-buffer) 'summary)
    (mew-current-set 'message nil)
    (mew-current-set 'part nil)
    (mew-current-set 'cache nil)
    (setq mew-summary-buffer-direction 'down)
    (if (null (mew-summary-exclusive-p))
	()
      (condition-case err
	  (progn
	    (message "Scanning %s ..." folder)
	    (let ((buffer-read-only nil))
	      (if erase
		  (erase-buffer)
		(goto-char (point-max))))
	    (if (null (listp range)) (setq range (list range)))
	    (setq mew-summary-buffer-process
		  (apply (function start-process) 
			 "Scann" 
			 (current-buffer) 
			 mew-prog-scan
			 folder
			 "-noclear" "-noheader"
			 "-width"
			 (if (< (window-width) 80)
			     "80"
			   (int-to-string (window-width)))
			 range
			 ))
	    (set-process-filter mew-summary-buffer-process
				'mew-summary-scan-filter)
	    (set-process-sentinel mew-summary-buffer-process
				  'mew-summary-scan-sentinel)
	    (process-kill-without-query mew-summary-buffer-process)
	    )
	(quit
	 (set-process-sentinel mew-summary-buffer-process nil)
	 (setq mew-summary-buffer-process nil)
	 (setq mew-summary-buffer-string nil)
	 ))
      )
    ))

(defun mew-summary-scan-filter (process string)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (setq mew-summary-buffer-string 
	  (concat mew-summary-buffer-string string)) ;; nil can concat
    (let ((buffer-read-only nil))
      (while (string-match "^ *[0-9]+\\([ +]\\).*\n" 
			   mew-summary-buffer-string)
	;; leave mew-summary-buffer-string if "scan: no messages"
	(goto-char (point-max))
	(if (equal " " (mew-match 1 mew-summary-buffer-string))
	    (insert (substring mew-summary-buffer-string 0 (match-end 0)))
	  (insert (substring mew-summary-buffer-string 0 (match-beginning 1)))
	  (insert " ")
	  (insert (substring mew-summary-buffer-string
			     (match-end 1)
			     (match-end 0)))
	  )
	(setq mew-summary-buffer-string
	      (substring mew-summary-buffer-string (match-end 0)))
	)
      )))

(defun mew-summary-scan-sentinel (process event)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (if mew-summary-cache-use
	(write-region
	 (point-min)
	 (point-max)
	 (mew-expand-file-name mew-summary-cache-file (buffer-name))))
    (set-buffer-modified-p nil)
    (message "Scanning %s ... done" (buffer-name))
    (if mew-summary-buffer-string
	(if (string-match "^scan: no messages" mew-summary-buffer-string)
	    (message "No messages in %s" (buffer-name))
	  ;; need to hack
	  (let ((ppoint (process-mark process))
		(buffer-read-only nil))
	    (goto-char (point-max))
	    (insert mew-summary-buffer-string)
	    (goto-char (point-min))
	    (keep-lines mew-summary-message-regex))
	  ))
    (mew-summary-highlight-lines)
    (set-process-sentinel mew-summary-buffer-process nil)
    (setq mew-summary-buffer-process nil)
    (setq mew-summary-buffer-string nil)
    ))

(defun mew-summary-rescan (&optional folder)
  (interactive)
  (let ((range (mew-input-range (or folder (buffer-name)))))
    (if (and (mew-summary-mark-exist-p
	      (list mew-mark-rmm mew-mark-refile mew-mark-hop mew-mark-mark))
	     (not (equal (car (cdr (cdr range))) 'updage)))
	(if (mew-y-or-n-p "Marked messages exist. Process mark before scan? ")
	    (mew-summary-exec)
	  ))
    (mew-summary-scan (or folder (buffer-name)) range)
    ))

;;
;;
;;


(defun mew-pick-complete ()
  (interactive)
  (let ((word nil) (comp nil) (all nil))
    (if (null (setq word (mew-delete-backward-char)))
	()
      (setq comp (try-completion word mew-prog-pick-argalist))
      (setq all (all-completions word mew-prog-pick-argalist))
      (cond
       ((eq comp t)
	(insert (car (assoc word mew-prog-pick-argalist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp)
	(and (> (length all) 1)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (ding))
       )
      )
    ))


(defun mew-pick-input-pattern (&optional prompt)
  (let ((pat nil))
    (setq pat (read-from-minibuffer 
	       (format "Pattern%s (%s) : "
		       (if prompt prompt "")
		       (car (car mew-prog-pick-argalist)))
		       "-" mew-pick-map nil))
    (setq pat (downcase pat))
    (cond
     ((string-match " " pat)
      (mapcar (function mew-summary-search-blace)
	      (mew-header-delete-nullstring-list
	       (mew-header-split pat 32))));; 32 is " "
     (t 
      (if (or (string= pat "") (string= pat "-"))
	  (setq pat (car (car mew-prog-pick-argalist))))
      (cond
       ((mew-member pat mew-prog-pick-arglogic)
	(mew-pick-input-logic pat))
       ((mew-member pat mew-prog-pick-argdate)
	(cons pat (mew-pick-input-value (format "Date for %s" pat)
					mew-prog-pick-argdatealist)))
       (t (cons pat (mew-pick-input-value (format "Value for %s" pat))))
       )
      )
    )))

;; pattern = key value | logic

(defun mew-pick-input-value (prompt &optional alist)
  (let ((ret nil))
    (cond
     (alist (setq ret (completing-read (format "%s : " prompt)
				       alist
				       nil
				       nil  ;; not require match
				       "")))
     (t (setq ret (read-string (format "%s : " prompt) "")))
     )
    (list ret)
    ))

(defun mew-pick-input-logic (logic)
  (append (mew-pick-input-pattern (format "1 for %s" logic))
	  (cons logic (mew-pick-input-pattern (format "2 for %s" logic))))
  )

    
(defun mew-summary-search-blace (string)
  (cond
   ((equal string "(") "-lbrace")
   ((equal string ")") "-rbrace")
   (t string)))

(defun mew-summary-search ()
  (interactive)
  (let ((folder (mew-input-folder (buffer-name)))
	(pattern nil)
	(range nil))
    (if (null (file-directory-p (mew-expand-file-name folder)))
	(message "No such folder %s" folder)
      (setq pattern (mew-pick-input-pattern))
      (message "Picking messages in %s ..." folder)
      (setq range (mew-summary-pick folder pattern))
      (message "Picking messages in %s ... done" folder)
      (if range (mew-summary-scan folder (list range t))) ;; erase
      )
    ))

(defun mew-summary-search-mark ()
  (interactive)
  (if (equal (point-min) (point-max))
      (message "No messages in this buffer.")
    (let ((folder (buffer-name))
	  (pattern nil)
	  (first nil)
	  (last nil)
	  (range nil))
      (setq pattern (mew-pick-input-pattern))
      (message "Picking messages in %s ..." folder)
      (goto-char (point-min))
      (setq first (mew-summary-message-number))
      (goto-char (point-max))
      (forward-line -1)
      (setq last (mew-summary-message-number))
      (setq range (mew-summary-pick folder pattern (concat first "-" last)))
      (message "Picking messages in %s ... done" folder)
      (if (null range)
	  ()
	(message "Marking messages ... ")
	(goto-char (point-min))
	(while (not (eobp))
	  (if (and (null (mew-summary-marked-p))
		   (mew-member (mew-summary-message-number) range))
	      (mew-summary-mark mew-mark-hop))
	  (forward-line))
	(message "Marking messages ... done")
	)
      )))

(defun mew-summary-pick (folder pattern &optional range)
  (let ((start nil)
	(point nil)
	(msgs nil))
    (setq range (or range "all"))
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-tmp))
      (erase-buffer)
      (apply (function call-process)
	     mew-prog-pick nil t nil "-list" folder range pattern)
      (goto-char (point-min))
      (cond 
       ((looking-at "pick: no messages") (message "No such messages") nil)
       ((looking-at "pick: ") (message "Illegal pattern") nil)
       (t 
	(while (not (eobp))
	  (setq start (point))
	  (forward-line 1)
	  (setq msgs (cons (buffer-substring start (1- (point))) msgs))
	  )
	(reverse msgs))
       )
      )))

;;
;; show
;;

(defun mew-summary-highlight-lines ()
  (if mew-highlight-lines-use
      (if mew-emacs19-p
	  (save-excursion
	    (let ((p (point))
		  (buffer-read-only nil))
	      (goto-char (point-min))
	      (while (not (eobp))
		(put-text-property (point)
				   (progn (end-of-line) (point))
				   'mouse-face
				   'highlight
				   )
		(forward-line 1))
	      (goto-char p))
	    ))
    ))

(defun mew-summary-toggle-analysis ()
  (interactive)
  (cond 
   (mew-analysis
    (message "Skip MIME analysis")
    (setq mew-analysis nil))
   (t
    (message "Display message with MIME analysis")
    (setq mew-analysis t))
   ))


(defun mew-summary-show ()
  (interactive)
  (let* ((fld (buffer-name))
	 (msg (mew-summary-message-number))
	 (ofld-msg (mew-current-get 'message))
	 (part (mew-summary-part-number))
	 (opart (mew-current-get 'part))
	 (next nil)
	 (mew-summary-buffer-disp-msg-current t))
    (if (not (file-exists-p (mew-expand-file-name (concat fld "/" msg))))
	(message "File not exists.")
      (mew-summary-toggle-disp-msg 'on)
      (unwind-protect
	  (progn
	    ;; message buffer
	    (cond 
	     (msg 
	      (mew-window-configure fld 'message)
	      (if (or (null ofld-msg)
		      (null (equal (cons fld msg) ofld-msg))
		      (and (equal (cons fld msg) ofld-msg) 
			   (null (equal opart part))))
		  (mew-summary-display-message fld msg)
		(if (mew-message-next-page)
		    (setq next t)))
	      )
	     (part
	      (mew-window-configure fld 'message)
	      (if (or (null opart) (null (equal opart part)))
		  (mew-summary-display-part 
		   (mew-cache-mime-syntax (mew-cache-hit ofld-msg)) part)
		(if (mew-message-next-page)
		    (setq next t)))
	      )
	     (t
	      (message "No message")
	      nil
	      )
	     ))
	(if (or msg part)
	    (progn
	      (pop-to-buffer fld)
	      (mew-summary-recenter)
	      (if next
		  (cond 
		   ((equal mew-summary-show-direction 'down)
		    (mew-summary-display-down))
		   ((equal mew-summary-show-direction 'up)
		    (mew-summary-display-up))
		   ((equal mew-summary-show-direction 'next)
		    (mew-summary-display-next))
		   (t ())
		   )
		)
	      )
	  )
	))
    ))

(defun mew-summary-toggle-disp-msg (&optional arg)
  (interactive)
  (cond 
   ((equal arg 'on)
    (setq mew-summary-buffer-disp-msg t))
   ((equal arg 'off)
    (setq mew-summary-buffer-disp-msg nil))
   (t
    (setq mew-summary-buffer-disp-msg (not mew-summary-buffer-disp-msg))
    (if mew-summary-buffer-disp-msg
	(mew-summary-display)
      (mew-window-configure (current-buffer) 'summary)))))

(defun mew-summary-display (&optional notforce)
  (interactive)
  (let* ((fld (buffer-name))
	 (msg (mew-summary-message-number))
	 (fld-msg (cons fld msg))
	 (ofld-msg (mew-current-get 'message))
	 (part (mew-summary-part-number))
	 (opart (mew-current-get 'part))
	 (cache nil)
	 (mew-summary-buffer-disp-msg-current
	  (or (interactive-p) mew-summary-buffer-disp-msg)))
    (if (not (file-exists-p (mew-expand-file-name (concat fld "/" msg))))
	(message "File not exists.")
      (unwind-protect
	  (progn
	    (if (or (interactive-p) mew-summary-buffer-disp-msg)
		(progn 
		  (mew-summary-toggle-disp-msg 'on)
		  (mew-window-configure fld 'message))
	      (mew-window-configure (current-buffer) 'summary)
	      (set-buffer (mew-buffer-message)))
	    (cond
	     (msg
	      (if (or (null notforce)
		      (null ofld-msg)
		      (null (equal (cons fld msg) ofld-msg)))
		  (if (or (interactive-p) mew-analysis)
		      (setq cache
			    (mew-summary-display-message fld msg 'analysis))
		    (setq cache
			  (mew-summary-display-message fld msg))))
	      )
	     (part
	      (if (or (null opart) (null (equal opart part)))
		  (mew-summary-display-part 
		   (mew-cache-mime-syntax (mew-cache-hit ofld-msg)) part)))
	     (t (message "No message"))
	     ))
	(progn
	  (pop-to-buffer fld)
	  (mew-summary-recenter)
	  )
	)
      (or cache (mew-cache-hit ofld-msg)) ;; xxx
      )
    ))


(defun mew-summary-multipart-delete ()
  (let ((buf (current-buffer)))
    (if (null (marker-position mew-current-marker))
	()
      (set-buffer (marker-buffer mew-current-marker))
      (set-marker mew-current-marker2 (point))
      (goto-char (marker-position mew-current-marker))
      (let ((start (point))
	    (buffer-read-only nil))
	(while (looking-at "^\t")
	  (forward-line))
	(delete-region start (point)))
      (set-marker mew-current-marker nil)
      (goto-char (marker-position mew-current-marker2))
      (set-buffer buf)
      )
    ))

(defun mew-summary-multipart-print (fld syntax)
  (save-excursion
    (set-buffer fld)
    ;; summary buffer
    (forward-line 1)
    (set-marker mew-current-marker (point) (current-buffer))
    (let ((buffer-read-only nil))
      (save-excursion (mew-mime-syntax-print syntax)))
    (forward-line -1)
    (mew-summary-highlight-lines)
    ))


(defun mew-summary-mode-line (fld)
  (save-excursion
    (set-buffer fld)
    ;; if not running process in this buffer
    ;; display how many messages are unread
    (if (null mew-summary-buffer-process)
	(setq mew-summary-buffer-left-msgs ;; local variable
	      (int-to-string (1- (count-lines (point) (point-max)))))
      )
    ))


(defun mew-summary-display-message (fld msg &optional analysis)
  ;; message buffer
  (let ((hit nil)
	(zmacs-regions nil)) ;; for XEmacs
    (let ((buffer-read-only nil))
      (erase-buffer)) ;; for PGP pass phase input
    (set-marker (mark-marker) nil) ;; kill mark for cite
    (if (or mew-analysis analysis)
	(progn
	  (setq hit (mew-cache-message (cons fld msg)))
	  ;; copy local variable
	  (setq mew-mime-syntax (mew-cache-mime-syntax hit))
	  (mew-current-set 'cache hit)
	  )
      (setq mew-mime-syntax nil)
      (mew-current-set 'cache nil)
      )
    (mew-current-set 'message (cons fld msg))
    (mew-current-set 'part nil)
    ;;
    (mew-summary-multipart-delete)
    (mew-summary-mode-line fld)
    ;; message buffer
    (let ((buffer-read-only nil))
;;      (erase-buffer) ;; xxx
      (if (equal fld mew-draft-folder)
	  (insert-file-contents
	   (mew-expand-file-name (concat fld "/" msg)))
	(if (null (or mew-analysis analysis))
	    (mew-header-insert-from-file
	     (mew-expand-file-name (concat fld "/" msg)))
	  (if (mew-message-multipart-p mew-mime-syntax)
	      (mew-summary-multipart-print fld mew-mime-syntax))
	  (mew-mime-message/rfc822 mew-mime-syntax)
	  )
	)
      )
    (run-hooks 'mew-message-hook)
    hit ;; return value
    ))

(defun mew-summary-display-part (fullpart num &optional non-erase)
;; called in message buffer
;; return t to next part
  (let* ((part    (mew-mime-part-syntax fullpart num))
	 (begin   (mew-part-get-begin  part))
	 (end     (mew-part-get-end    part))
	 (type    (or (mew-part-get-type part) "text/plain"))
	 (params  (mew-part-get-params part))
	 (attr    (mew-content-attr type mew-mime-content-type))
	 (program (mew-content-program attr))
	 (options (mew-content-options attr))
	 (async   (mew-content-async   attr))
	 (zmacs-regions nil))  ;; for XEmacs
    (if (null non-erase)
	(let ((buffer-read-only nil)) (erase-buffer)))
    (set-marker (mark-marker) nil) ;; kill mark for cite
    (setq mew-message-citation 'noheader)
    ;;
    (if (null program)
	(setq program mew-mime-default-function))
    (if (symbolp program)
	(if (fboundp program)
	    (if (equal program 'mew-mime-message/rfc822)
		(funcall program part) ;; for recursive MIME
	      (funcall program begin end params)))
      (save-excursion
	(set-buffer (mew-current-get 'cache))
	(let ((selective-display nil)
	      (mc-flag nil))
	  (if async
	      (mew-mime-start-process begin end program options)
	    (mew-mime-call-process begin end program options)
	    ))))
    (mew-current-set 'part num) ;; should be after funcall
    (run-hooks 'mew-message-hook)
    ))

(defun mew-summary-recenter ()
  (interactive)
  (if (or mew-summary-recenter-p
	  (interactive-p))
      (recenter (/ (- (window-height) 2) 2))
    ))

;;
;;
;;

(defun mew-mime-start-process (begin end program options)
  (if (mew-which program exec-path)
      (if (mew-y-or-n-p (format "Start %s? " program))
	  (let ((pro nil)
		(tmp (make-temp-name "/tmp/mew"))
		(mc-flag nil)
		(file-coding-system (if mew-mule-p *noconv*)))
	    (write-region begin end tmp)
	    (message "Starting %s ..." program)
	    (setq pro (apply (function start-process)
			     (format "*mew %s*" program)
			     mew-buffer-tmp ;; xxx
			     program
			     (append options (list tmp))))
	    (set-process-sentinel pro 'mew-mime-start-process-sentinel)
	    (message "Sending %s ... done" program)
	    (setq mew-process-file-alist
		  (cons (cons pro tmp) mew-process-file-alist))
	    )))
  t ;; to next part
  )

(defun mew-mime-start-process-sentinel (process event)
  (let ((al (assoc process mew-process-file-alist)))
    (if (cdr al) (delete-file (cdr al)))
    (setq mew-process-file-alist (mew-delq al mew-process-file-alist))
    ))

(defun mew-mime-call-process (begin end program options)
  (if (mew-y-or-n-p (format "Call %s? " program))
      (let ((mc-flag nil)
	    (file-coding-system (if mew-mule-p *noconv*)))
	(message "Calling %s ..." program)
	(apply (function call-process-region) 
	       begin end program nil nil nil options)
	(message "Calling %s ... done" program)
	)
    )
  t ;; to next part
  )

(defun mew-mime-prev-page ()
  (interactive)
  (let ((buf (current-buffer)))
    (unwind-protect
	(progn
	  (pop-to-buffer (mew-buffer-message))
	  (scroll-down))
      (pop-to-buffer buf)))
  )



(defun mew-message-header-end (begin end)
  (save-excursion
    (set-buffer (mew-current-get 'cache))
    (narrow-to-region begin end)  
    (goto-char (point-min))
    (re-search-forward "^-*$")
    (save-restriction ;; for no body
      (forward-line 1))
    (prog1 
	(point)
      (widen))
    ))


;; change
(defun mew-mime-text/plain (begin end &optional params)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (let ((buffer-read-only nil))
      (insert-buffer-substring (mew-current-get 'cache) begin end)
      (if mew-eof-string
	  (progn
	    (goto-char (point-max)) ;; necessary?
	    (insert mew-eof-string)))
      )
    (cond (mew-break-pages 
	   (goto-char (point-min))
	   (mew-message-narrow-to-page)
	   ))
    ))

(defun mew-mime-message/rfc822 (part)
  (let* ((hbegin (mew-message-get-begin part))
	 (hend   (mew-message-get-end   part))
	 (cache  (mew-current-get 'cache))
	 (buffer-read-only nil))
    (mew-header-insert-from-cache cache hbegin hend)
    (if (mew-message-singlepart-p part)
	(mew-summary-display-part part nil 'non-erase)) ;; nil is single
    ;; display-part sets citation noheader
    (setq mew-message-citation 'header)
    ))

;;
;;
;;

(defun mew-summary-display-top ()
  (interactive)
  (goto-char (point-min))
  (mew-summary-display))

(defun mew-summary-display-bottom ()
  (interactive)
  (goto-char (point-max))
  (if (not (bobp)) (forward-line -1))
  (mew-summary-display))


(defun mew-summary-mouse-show (e)
  (interactive "e")
  (mouse-set-point e)
  (beginning-of-line)
  (mew-summary-show))

;;
;; direction
;;

(defun mew-summary-next ()
  (if (equal mew-summary-buffer-direction 'up)
      (mew-summary-up)
    (mew-summary-down))
  )

(defun mew-summary-message-next ()
  (if (equal mew-summary-buffer-direction 'up)
      (mew-summary-message-up)
    (mew-summary-message-down))
  )


(defun mew-summary-down ()
  (forward-line)
  (cond 
   ((re-search-forward 
     (concat mew-summary-message-regex 
	     "[ " (char-to-string mew-mark-hop) "]\\|"
	     mew-summary-part-regex)
     nil t nil)
    (beginning-of-line)
    (setq mew-summary-buffer-direction 'down)
    t)
   (t 
    (mew-summary-multipart-delete)
    (mew-current-set 'message nil)
    (forward-line -1)
    (mew-window-configure (current-buffer) 'summary)
    (message "No more message")
    nil)
   )
  )


(defun mew-summary-message-down ()
  (forward-line)
  (cond 
   ((re-search-forward
     (concat mew-summary-message-regex "[ " (char-to-string mew-mark-hop) "]")
     nil t nil)
    (beginning-of-line)
    (setq mew-summary-buffer-direction 'down)
    t)
   (t 
    (forward-line -1)
    (message "No more message")
    nil)
   )
  )


(defun mew-summary-up ()
  (interactive)
  (cond 
   ((re-search-backward
     (concat mew-summary-message-regex 
	     "[ " (char-to-string mew-mark-hop) "]\\|"
	     mew-summary-part-regex)
     nil t nil)
    (setq mew-summary-buffer-direction 'up)
    t)
   (t 
    (mew-summary-multipart-delete)
    (mew-window-configure (current-buffer) 'summary)
    (message "No more message")
    nil)
   ))

(defun mew-summary-message-up (&optional nochange)
  (interactive)
  (cond 
   ((re-search-backward
     (concat mew-summary-message-regex "[ " (char-to-string mew-mark-hop) "]")
     nil t nil)
    (if (null nochange)
	(setq mew-summary-buffer-direction 'up))
    t)
   (t 
    (message "No more message")
    nil)
   ))


(defun mew-summary-display-next ()
  (interactive)
  (cond
   ((and (mew-summary-next) mew-summary-buffer-disp-msg)
    (mew-summary-display)
    ))
  )

(defun mew-summary-display-down ()
  (interactive)
  (if (and (mew-summary-down) mew-summary-buffer-disp-msg)
      (mew-summary-display))
  )

(defun mew-summary-down-mark (mark)
  (interactive)
  (forward-line)
  (cond 
   ((re-search-forward (concat mew-summary-message-regex
			       (regexp-quote (char-to-string mark)))
		       nil t nil)
    (beginning-of-line)
    t)
   (t 
    (forward-line -1)
    (message "No more marked message")
    nil)))


(defun mew-summary-display-hop-down ()
  (interactive)
  (let ((part nil))
    (if (looking-at mew-summary-part-regex)
	(setq part t)
      (save-excursion
	(forward-line)
	(if (looking-at mew-summary-part-regex)
	    (setq part t))))
    (if (null part)
	(if (mew-summary-down-mark mew-mark-hop)
	    (mew-summary-display))
      (mew-summary-message-down)
      (mew-summary-display))
    ))

(defun mew-summary-display-up ()
  (interactive)
  (if (and (mew-summary-up) mew-summary-buffer-disp-msg)
      (mew-summary-display))
  )

(defun mew-summary-up-mark (mark)
  (interactive)
  (cond 
   ((re-search-backward (concat mew-summary-message-regex
			       (regexp-quote (char-to-string mark)))
			nil t nil)
    t)
   (t 
    (message "No more marked message")
    nil)
   ))

(defun mew-summary-display-hop-up ()
  (interactive)
  (let ((part nil))
    (if (looking-at mew-summary-part-regex)
	(setq part t))
    (if (null part)
	(if (mew-summary-up-mark mew-mark-hop)
	    (mew-summary-display))
      (mew-summary-message-up)
      (mew-summary-display))
    ))

(defun mew-summary-prev-page ()
  (interactive)
  (if (or (mew-summary-message-number) (mew-summary-part-number))
      (let ((buf (current-buffer)))
	(unwind-protect
	    (progn
	      (pop-to-buffer (mew-buffer-message))
	      (mew-message-prev-page))
	  (pop-to-buffer buf))
	)
    (message "No message or part here")
    ))


(defun mew-summary-scroll-up ()
  (interactive)
  (let ((buf (current-buffer))
	(msg (mew-summary-message-number))
	(ofld-msg (mew-current-get 'message))
	(part (mew-summary-part-number))
	(opart (mew-current-get 'part)))
    (cond ((or (and msg (null part) (string= msg (cdr ofld-msg)))
	       (and part (string= part opart)))
	   (unwind-protect
	       (progn
		 (mew-window-configure buf 'message)
		 (mew-message-next-page 1))
	     (pop-to-buffer buf)))
	  ((or msg part)
	   (mew-summary-show))
	  (t
	   (message "No message or part here")))))

;(defun mew-summary-scroll-up ()
;  (interactive)
;  (if (or (mew-summary-message-number) (mew-summary-part-number))
;      (let ((buf (current-buffer)))
;	(unwind-protect
;	    (progn
;	      (mew-window-configure buf 'message)
;	      (mew-message-next-page 1)
;	      )
;	  (pop-to-buffer buf))
;	)
;    (message "No message or part here")
;    ))

(defun mew-summary-scroll-down ()
  (interactive)
  (if (or (mew-summary-message-number) (mew-summary-part-number))
      (let ((buf (current-buffer)))
	(unwind-protect
	    (progn
	      (mew-window-configure buf 'message)
	      (mew-message-prev-page 1)
	      )
	  (pop-to-buffer buf))
	)
    (message "No message or part here")
    ))

    

;;
;;
;;

(defun mew-summary-goto-folder ()
  (interactive)
  (let* ((folder (mew-input-folder mew-inbox))
	 (range nil))
    (cond
     ((file-directory-p (mew-expand-file-name folder))
      (if (and mew-summary-cache-use (null (get-buffer folder)))
	  (let ((cache (mew-expand-file-name
			mew-summary-cache-file folder)))
	    (switch-to-buffer (get-buffer-create folder))
	    (if (file-exists-p cache) (insert-file-contents cache))
	    (mew-summary-mode) ;; gee dirty
	    )
	(switch-to-buffer folder))
      (goto-char (point-max))
      (setq range (mew-input-range folder))
      (if (and (mew-summary-mark-exist-p
		(list mew-mark-rmm mew-mark-refile mew-mark-hop mew-mark-mark))
	       (not (equal (car (cdr (cdr range))) 'update)))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)
	    ))
      (mew-summary-scan folder range)
      )
     (t (message "No such folder %s" folder))
     )
    ))
    

(defun mew-summary-rmm ()
  (interactive)
  (if (mew-summary-part-number)
      (mew-summary-message-up t)) ;; no direction change
  (cond
   ((null (mew-summary-marked-p))
    (mew-summary-mark mew-mark-rmm)
    (mew-summary-multipart-delete)
    (mew-summary-display-next))
   (t (message "Already marked")) ;;; summary-mark-mark check !!
   ))
   
(defun mew-summary-exec ()
  (interactive)
  (mew-window-configure (current-buffer) 'summary)
  (setq mew-summary-buffer-process t)
  (message "Refiling and deleting ...")
  (let ((msgs (mew-summary-mark-collect mew-mark-refile))
	(src (buffer-name))
	refal folder tmp msg)
    (while msgs
      (setq msg (car msgs))
      (setq msgs (cdr msgs))
      (if (setq folder (cdr (assoc msg mew-summary-buffer-refile)))
	  (if (setq tmp (assoc folder refal))
	      (setq refal (cons (append tmp (list msg)) (mew-delq tmp refal)))
	    (setq refal (cons (list folder msg) refal)))
	)
      )
    (while refal
      (apply (function call-process)
	     mew-prog-refile nil nil nil
	     "-src" src
	     (car (car refal))
	     (cdr (car refal)))
      (setq refal (cdr refal))
      ))
  (setq mew-summary-buffer-refile nil)
  (let ((dels (mew-summary-mark-collect mew-mark-rmm)))
    (if dels
	(progn
	  (apply (function call-process)
		 mew-prog-rmm
		 nil
		 nil
		 nil
		 (buffer-name)
		 dels)))
    )
  (mew-current-set 'message nil) ;;; hack
  (mew-refile-folder-guess-save)
  (let ((next))
    (if (mew-summary-marked-p)
	(if (mew-summary-message-down) ;; point moves
	   (setq next (mew-summary-message-number))
	  (setq next t))
      (setq next (mew-summary-message-number)))
    (mew-summary-multipart-delete)
    (mew-summary-mark-process)
    (mew-summary-jump-message next)
    )
  (if mew-summary-cache-use
      (write-region
       (point-min)
       (point-max)
       (mew-expand-file-name mew-summary-cache-file (buffer-name))))
  (setq mew-summary-buffer-process nil)
  (set-buffer-modified-p nil)
  (message "Refiling and deleting ... done")
  )

(defun mew-summary-reply ()
  (interactive)
  (cond 
   ((or (mew-summary-message-number)
	(mew-summary-part-number))
    (mew-current-set 'window (current-window-configuration))
    (let ((fld (buffer-name))
	  (file (mew-draft-get-new))
	  from to cc subject in-reply-to references
	  cbuf cache)
      (unwind-protect
	  (progn
	    (if (get-buffer (mew-buffer-message))
		(delete-windows-on (mew-buffer-message)))
	    (if (< (window-height) 25) (delete-other-windows))
	    (split-window-vertically)
	    (switch-to-buffer-other-window (find-file-noselect file))
	    (mew-draft-rename file)
	    (setq cbuf (current-buffer)) ;; draft
	    (pop-to-buffer fld) 
	    ;;
	    (setq cache (mew-summary-display)) ;; force to display
	    (if cache
		(set-buffer cache)
	      (set-buffer (mew-buffer-message)))
	    ;; now cache buffer
	    (setq from (mew-header-address-collect '("From:")))
	    (cond 
	     ((and from (string-match mew-mail-address (car from)))
	      (setq to (mew-header-address-collect '("To:" "Apparently-To:")))
 	      (if (and (null (cdr to)) (string-match ":;" (car to)))
		  (setq to (or (mew-header-address-collect '("Reply-To:"))
			       from)))
	      (setq cc (mew-header-address-collect '("Cc:"))))
	     (t
	      (setq to (or (mew-header-address-collect '("Reply-To:")) from))
	      (setq cc (mew-header-address-collect
			'("To:" "Cc:" "Apparently-To:"))))
	     )
	    (setq subject (mew-field-get-value "Subject:"))
	    (if (and subject (not (string-match "^Re:" subject)))
		(setq subject (concat "Re: " subject)))
	    (setq in-reply-to (mew-field-get-value "Date:"))
	    (setq references (mew-field-get-value "Message-ID:"))
	    ;;
	    (pop-to-buffer cbuf) ;; draft
	    (mew-draft-header to cc subject in-reply-to references)
	    (mew-draft-mode) ;; for hilight
	    )
	(save-buffer))) ;; to make sure not to use this draft again
    (message "Draft is prepared"))
   (t (message "No message"))
   )
  )

(defun mew-summary-reply-with-citation ()
  "Reply to current message and insert it in draft-buffer."
  (interactive)
  (mew-summary-reply)
  (mew-draft-cite))

(defun mew-summary-reedit ()
  (interactive)
  (mew-current-set 'window (current-window-configuration))
  (mew-window-configure (current-buffer) 'summary)
  (let ((msg (mew-summary-message-number)) ;; must get msg here
	(fld (buffer-name))
	(rename nil))
    (unwind-protect
	(progn
	  (if (equal fld mew-draft-folder)
	      (progn
		(switch-to-buffer 
		 (find-file-noselect 
		  (mew-expand-file-name msg fld)))
		(setq rename msg))
	    (setq rename (mew-draft-get-new))
	    (switch-to-buffer 
	     (find-file-noselect 
	      (mew-expand-file-name rename mew-draft-folder)))
	    (insert-file-contents (mew-expand-file-name msg fld)))
	  (mew-draft-mode)
	  (mew-draft-rename rename))
      (goto-char (point-min)) 
      (re-search-forward "^----$" nil t)
      (beginning-of-line)
      (setq mew-draft-buffer-header (point-marker))
      (forward-line 1)
      (save-buffer))
    )
  (message "Draft is prepared")
  )

(defun mew-summary-sort ()    
  (interactive)
  (let ((folder (buffer-name)))
    (if (null (mew-summary-exclusive-p))
	()
      (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)))
      (if (null (mew-y-or-n-p (format "Sort %s ? " folder)))
	  ()
	(setq mew-summary-buffer-process t)
	(message "Sorting %s ... " folder)
	(call-process mew-prog-sortm nil nil nil folder)
	(message "Sorting %s ... done" folder)
	(setq mew-summary-buffer-process nil)
	(let ((buffer-read-only nil)) (erase-buffer))  ;; for update
	(mew-summary-scan folder (mew-input-range folder))
	)
      )
    ))

(defun mew-summary-pack ()    
  (interactive)
  (let ((folder (buffer-name)))
    (if (null (mew-summary-exclusive-p))
	()
      (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)))
      (if (null (mew-y-or-n-p (format "Pack %s? " folder)))
	  ()
	(setq mew-summary-buffer-process t)
	(message "Packing %s ... " folder)
	(call-process mew-prog-pack nil nil nil folder "-pack")
	(message "Packing %s ... done" folder)
	(setq mew-summary-buffer-process nil)
	(let ((buffer-read-only nil)) (erase-buffer)) ;; for update
	(mew-summary-scan folder (mew-input-range folder))
	)
      )
    ))

(defun mew-summary-forward ()
  (interactive)
  (cond 
   ((mew-summary-message-number)
    (mew-current-set 'window (current-window-configuration))
    (let* ((fld (buffer-name)) ;; fld == "+inbox" e.g.
	   (msg (mew-summary-message-number))
	   (to (mew-input-address "To:"))
	   (cc (mew-input-address "Cc:"))
	   (subject "")
	   (cbuf nil)
	   (file (mew-draft-get-new)))
      (unwind-protect
	  (progn
	    (delete-other-windows)
	    (split-window-vertically)
	    (switch-to-buffer-other-window (find-file-noselect file))
	    (mew-draft-rename file)
	    (setq cbuf (current-buffer)) ;; draft
	    (pop-to-buffer fld)
	    ;;
	    (set-buffer (mew-summary-display)) ;; force to display
	    ;; now cache buffer
	    (setq subject (concat "Forward: " (mew-field-get-value "Subject:")))
	    (pop-to-buffer cbuf) ;;; draft
	    ;;
	    (mew-draft-header to cc subject)
	    (mew-draft-mode)
	    (mew-draft-multi-copy file fld (list msg))
	    (save-excursion
	      (insert "\n")
	      (mew-draft-prepare-multipart))
	    )
	(save-buffer))) ;; to make sure no to use this draft again
    (message "Draft is prepared"))
   (t (message "No message")))
  )

(defun mew-summary-multi-forward ()
  (interactive)
  (let ((msglist (mew-summary-mark-collect mew-mark-mark)))
    (cond (msglist
	   (mew-current-set 'window (current-window-configuration))
	   (let ((fld (buffer-name))
		 (to (mew-input-address "To:"))
		 (cc (mew-input-address "Cc:"))
		 (subject (read-string "Subject: " ""))
		 (file (mew-draft-get-new)))
	     (unwind-protect
		 (progn
		   (delete-other-windows)
		   (split-window-vertically)
		   (switch-to-buffer-other-window (find-file-noselect file))
		   (mew-draft-rename file)
		   (mew-draft-header to cc subject)
		   (mew-draft-mode)
		   (mew-draft-multi-copy file fld msglist)
		   (save-excursion
		     (insert "\n")
		     (mew-draft-prepare-multipart))
		     )
	       (save-buffer))) ;; to make sure no to use this draft again
	   (message "Draft is prepared"))
	  (t (message "No marks"))
	  )
    ))

(defun mew-summary-redist ()
  (interactive)
  (cond 
   ((mew-summary-message-number)
    (mew-current-set 'window (current-window-configuration))
    (let* ((file (mew-expand-file-name (mew-summary-message-number)
				       (buffer-name)))
	   (to (mew-input-address "Resent-To:"))
	   (cc (mew-input-address "Resent-Cc:"))
	   (draft (mew-expand-file-name (mew-draft-get-new) mew-draft-folder)))
      (save-excursion
	(set-buffer (find-file-noselect draft))
	(if mew-redist-needs-full-contents
	  (progn
	    (insert-file-contents file)
	    (mew-field-resent-line "Resent-From:")
	    (mew-field-resent-line "Resent-Sender:")
	    (mew-field-resent-line "Resent-To:")
	    (mew-field-resent-line "Resent-Cc:")
	    (mew-field-resent-line "Resent-Date:")
	    (mew-field-resent-line "Resent-Message-Id:")
	    (mew-field-resent-line "Resent-Reply-To:")
	    (mew-field-delete-line "Received:")
	    (mew-field-delete-line "Return-Path:")
	    (goto-char (point-min))))
	(if (or (null to) (equal to ""))
	    ()
	  (mew-field-insert-here "Resent-To:" to))
	(if (or (null cc) (equal cc ""))
	    ()
	  (mew-field-insert-here "Resent-Cc:" cc))
	(and mew-cc (mew-field-insert-here "Resent-Cc:" mew-cc))
	(save-buffer) 
	(message "Redistributing ... ")
	(if mew-redist-needs-full-contents
	  (call-process "/bin/sh" nil nil nil "-c"
		         (format "mhdist=1 %s %s"
				  mew-prog-send
				  (buffer-file-name)))
	  (call-process "/bin/sh" nil nil nil "-c"
		         (format "mhdist=1 mhaltmsg=%s %s %s"
				  file
				  mew-prog-send
				  (buffer-file-name))))
	(message "Redistributing ... done")
	(kill-buffer (current-buffer))
	))
    )
   (t (message "No message"))
   )
  )

(defun mew-draft-multi-copy (draft folder msglist)
  (let* ((mimefolder (mew-draft-to-mime draft))
	 (mimedir (mew-expand-file-name mimefolder)))
    (if (null (file-directory-p mimedir))
	(mew-make-directory mimedir)
      (if (null (mew-directory-empty-p mimedir))
	  (if (mew-y-or-n-p (format "Mime folder %s is not empty. Delete it? "
				mimefolder))
	      (progn
		(call-process "rm" nil nil nil "-rf" mimedir)
		(mew-make-directory mimedir))
	    )))
    (while msglist
      (call-process "ln" nil nil nil "-s"
		    (mew-expand-file-name (car msglist) folder)
		    (mew-draft-get-new mimefolder))
      (setq msglist (cdr msglist)))
    ))

(defun mew-summary-send (&optional unconfig to cc subject)
  (interactive)
  (let ((file (mew-draft-get-new)))
    (or to (setq to (mew-input-address "To:")))
    (or cc (setq cc (mew-input-address "Cc:")))
    (or subject (setq subject (read-string "Subject: " "")))
    (if (null unconfig)
	(mew-current-set 'window (current-window-configuration)))
    (mew-window-configure (current-buffer) 'summary)
    (unwind-protect
	(progn
	  (switch-to-buffer (find-file-noselect file))
	  (mew-draft-rename file)
	  (mew-draft-header to cc subject)
	  (mew-draft-mode) ;; hack hack mew-draft-buffer-header
	  )
      (save-buffer))) ;; to make sure no to use this draft again
  (message "Draft is prepared")
  )

(defun mew-summary-save ()
  (interactive)
  (let ((msg (mew-summary-message-number))
	(part (mew-summary-part-number))
	(action "Save")
	(append-p nil)
	(cbuf (current-buffer))
	file)
    (if (not (or msg part))
	(message "No message or part here")
      (setq file (mew-input-filename))
      (if (file-exists-p file)
	  (if (null mew-file-append-p)
	      (setq action "Overwrite")
	    (setq action "Append")
	    (setq append-p t)))
      (cond
       (msg
	(if (mew-y-or-n-p (format "%s message %s to %s? " action msg file))
	    (save-excursion
	      (set-buffer (get-buffer-create mew-buffer-tmp))
	      (widen)
	      (erase-buffer)
	      (let ((mc-flag nil)
		    (file-coding-system-for-read (if mew-mule-p *noconv*))
		    (file-coding-system (if mew-mule-p *noconv*)))
		(insert-file-contents (mew-expand-file-name msg cbuf))
		(write-region (point-min) (point-max) file append-p))
	      (message "Wrote to %s" file)
	      )
	  )
	)
       (part
	(if (mew-y-or-n-p (format "%s part %s to %s? " action part file))
	    (save-excursion
	      (set-buffer (mew-current-get 'cache))
	      (let* ((case-fold-search t)
		     (syntax (mew-mime-part-syntax
			      (mew-cache-mime-syntax
			       (mew-current-get 'cache)) part))
		     (begin (mew-part-get-begin syntax))
		     (end (mew-part-get-end syntax))
		     (type (mew-part-get-type syntax)))
		(if (string-match "message/rfc822" type)
		    (setq end (mew-part-get-end
			       (mew-message-get-part syntax))))
		(if (string-match "application/octet-stream" type)
		    (let ((mc-flag nil)
			  (file-coding-system (if mew-mule-p *noconv*)))
		      (write-region begin end file append-p))
		  (let ((file-coding-system
			(if mew-mule-p mew-mule-charset-local)))
		    (write-region begin end file append-p)
		    ))
		)
	      (message "Wrote to %s" file)
	      ))
	)
       )
      )
    ))

(defun mew-summary-burst ()
  (interactive)
  (let (target dir)
    (save-excursion
      (cond 
       ((mew-summary-part-number)
	(mew-summary-message-up t)
	(mew-summary-display)
	)
       )
      (if (null (mew-summary-message-number))
	  (message "No message here")
	(setq target (mew-expand-file-name 
		      (mew-summary-message-number) (current-buffer)))
	(setq dir (mew-input-directory-name))
	(message "Bursting ... ")
	(set-buffer mew-buffer-tmp)
	(widen)
	(erase-buffer)
	(let ((selective-display nil)
	      (mc-flag nil)
	      (file-coding-system-for-read (if mew-mule-p *noconv*))
	      (file-coding-system (if mew-mule-p *noconv*))
	      (syntax) (n 1))
	  (insert-file-contents target)
	  (setq syntax (mew-mime-syntax))
	  (if (mew-message-singlepart-p syntax)
	      ()
	    (setq syntax (mew-multi-get-parts (mew-part-get-part syntax)))
	    (while syntax
	      (if (null (mew-part-message-p (car syntax)))
		  ()
		(write-region (mew-part-get-begin (car syntax))
			      (mew-part-get-end (mew-part-get-part (car syntax)))
			      ;; xxx
			      (expand-file-name (int-to-string n) dir))
		(setq n (1+ n)))
	      (setq syntax (cdr syntax))
	      ))
	  )
	  (message "Bursting ... done")
	  )
      )
    ))

      
(defun mew-summary-print ()
  (interactive)
  (mew-summary-display)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (save-restriction
      (widen)
      (if (mew-y-or-n-p "Print this message? ")
	  (funcall mew-print-function))
      )
    ))

(defun mew-summary-unshar ()
  (interactive)
  (let ((msglist (reverse (mew-summary-mark-collect mew-mark-mark)))
	(files nil)
	(dir nil))
    (if (null msglist)
	(message "No marked messages")
      (if (null (mew-y-or-n-p (format "Execute unshar for these messages? "
				   mew-prog-unshar)))
	  ()
	(setq dir (mew-input-directory-name))
	(while msglist
	  (setq files (cons (mew-expand-file-name (car msglist) (current-buffer))
			    files))
	  (setq msglist (cdr msglist)))
	(message "Executing %s ... " mew-prog-unshar)
	(apply 'call-process mew-prog-unshar nil nil nil "-d" dir files)
	(message "Executing %s ... done" mew-prog-unshar)
	))
    ))

(defun mew-summary-uudecode ()
  (interactive)
  (let ((msglist (reverse (mew-summary-mark-collect mew-mark-mark)))
	(files nil)
	(dir nil)
	(tarfile nil)
	(case-fold-search nil))
    (if (null msglist)
	(message "No marked messages")
      (if (null (mew-y-or-n-p (format "Execute %s for these messages? "
				   mew-prog-uumerge)))
	  ()
	(setq dir (mew-input-directory-name))
	(while msglist
	  (setq files (cons (mew-expand-file-name (car msglist)
						  (current-buffer))
			    files))
	  (setq msglist (cdr msglist)))
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (let ((buffer-read-only nil)) (erase-buffer))
	  (message "Executing %s ..." mew-prog-uumerge)
	  (apply 'call-process mew-prog-uumerge
		 nil t nil "-d" dir files)
	  (message "Executing %s ... done" mew-prog-uumerge)
	  (goto-char (point-min))
	  (if (looking-at "^uumerge:")
	      (message "Failed to executing %s" mew-prog-uumerge)
	    (forward-line)
	    (setq tarfile (buffer-substring (point-min) (1- (point))))
	    (setq tarfile (mew-summary-prog-exec
			   mew-prog-compress "-df" "Z" dir tarfile))
	    (setq tarfile (mew-summary-prog-exec
			   mew-prog-gzip "-df" "gz" dir tarfile))
	    (if (string-match "^\\(.*\\)\\.tar$" tarfile)
		(if (null (mew-y-or-n-p (format "Execute %s for %s? " 
					    mew-prog-tar tarfile)))
		    ()
		  (message (format "Executing %s for %s ... "
				    mew-prog-tar tarfile))
		  (call-process "sh" nil nil nil "-c"
				(concat "cd " dir " ; cat " 
					tarfile " | tar -xf -"))
		  (message (format "Executing %s for %s ... done"
				    mew-prog-tar tarfile))
		  ))
	    ))
	))
    ))


 
(defun mew-summary-prog-exec (prog opts suf dir tarfile)
  (if (string-match (format "^\\(.*\\)\\.%s$" suf) tarfile)
      (if (null (mew-y-or-n-p (format "Execute %s for %s? " prog tarfile)))
	  tarfile
	(message (format "Executing %s for %s ... " prog tarfile))
	(call-process prog nil nil nil opts (expand-file-name tarfile dir))
	(message (format "Executing %s for %s ... done" prog tarfile))
	(mew-match 1 tarfile))
    tarfile
    ))


(defun mew-summary-jump-message (&optional msg)
;; if msg is t, jump last
  (interactive)
  (let ((here (point)))
    (if (null msg) (setq msg (read-string "Message No. : " "")))
    (cond 
     ((equal msg "") ())
     ((equal msg t)
      (goto-char (point-max))
      (forward-line -1))
     (t 
      (goto-char (point-min))
      (if (re-search-forward (format "^[ ]*%s[^0-9]+" msg) nil t)
	  (beginning-of-line)
	(goto-char here))))
    ))

;;
;; mark
;;

(defun mew-summary-marked-p ()
  (save-excursion
      (beginning-of-line)
      (cond 
       ((looking-at (concat mew-summary-message-regex " ")) nil)
       (t t))))

(defun mew-summary-mark-exist-p (marklist)
  (let ((regex 
	 (concat
	  mew-summary-message-regex
	  "["
	  (mapconcat (function char-to-string) marklist "\\|")
	  "]"
	   )))
    (save-excursion
      (goto-char (point-min))
      (re-search-forward regex nil t)
      )
    ))

(defun mew-summary-mark (mark)
  (let ((buffer-read-only nil))
    (cond 
     ((null (mew-summary-marked-p))
      (beginning-of-line)
      (re-search-forward mew-summary-message-regex)
      (delete-char 1)
      (insert (char-to-string mark)))
     (t (message "Already marked")))
    ))

(defun mew-summary-unmark ()
  (save-excursion
    (let ((buffer-read-only nil))
      (beginning-of-line)
      (re-search-forward mew-summary-message-regex)
      (delete-char 1)
      (insert " ")
      )))

(defun mew-summary-mark-letitbe ()
  () ;; do nothing
  )

(defun mew-summary-mark-process ()
  (let ((buffer-read-only nil))
    (goto-char (point-min))
    (while (re-search-forward 
	    (concat mew-summary-message-regex "\\([^ +0-9]\\)") nil t)
      (funcall (cdr (assoc (string-to-char (mew-match 2)) mew-mark-switch)))
      )
    (set-buffer-modified-p nil)
    ))

(defun mew-summary-mark-mark ()
  (interactive)
  (cond 
   ((null (mew-summary-marked-p))
    (mew-summary-mark mew-mark-mark)
    (mew-summary-display-next))))

(defun mew-summary-mark-hop ()
  (interactive)
  (cond 
   ((null (mew-summary-marked-p))
    (mew-summary-mark mew-mark-hop)
    (mew-summary-display-next))))

(defun mew-summary-mark-collect (mark)
  (save-excursion
    (let ((re (format (concat mew-summary-message-regex "%s") 
		      (regexp-quote (char-to-string mark))))
	  (msglist nil))
      (goto-char (point-min))
      (while (re-search-forward re nil t)
	(setq msglist (append msglist (list (mew-summary-message-number)))))
      msglist)))
	
    

;;
;; undo
;;

(defun mew-summary-undo ()
  (interactive)
  (cond 
   ((mew-summary-marked-p)
    (save-excursion
      (beginning-of-line)
      (looking-at (concat mew-summary-message-regex "\\([^ +0-9]\\)"))
      (funcall (cdr (assoc (string-to-char (mew-match 2)) mew-undo-switch)))
      )
    (mew-window-configure (current-buffer) 'summary)
;    (forward-line)
    )
   (t 
    (message "No mark"))
   )
  )

(defun mew-summary-undo-all ()
  (interactive)
  (let ((char nil) (ociea cursor-in-echo-area) (regex nil))
    (unwind-protect
	(progn
	  (message "Input mark : ")
;	  (setq cursor-in-echo-area 1)
	  (setq cursor-in-echo-area t)
	  (setq char (read-char))
;	  (setq cursor-in-echo-area -1)
	  (message "Input mark : %s" (char-to-string char))
	  )
      (setq cursor-in-echo-area ociea))
    (if (null (assoc char mew-mark-switch))
	(message "Mark %s is not supported" (char-to-string char))
      (mew-summary-multipart-delete)
      (let ((buffer-read-only nil))
	(save-excursion
	  (goto-char (point-min))
	  (setq regex (concat mew-summary-message-regex
			      (regexp-quote (char-to-string char))))
	  (while (re-search-forward regex nil t)
	    (delete-char -1)
	    (insert " "))
	  ))
      )
    ))


(defun mew-summary-refile-undo ()
  (setq mew-summary-buffer-refile
	(mew-delq (assoc (mew-summary-message-number) 
			 mew-summary-buffer-refile)
		  mew-summary-buffer-refile))
  (mew-summary-unmark)
  )

(defun mew-summary-convert-local-charset ()
  (interactive)
  (if mew-mule-p
      (save-excursion
	(set-buffer (mew-buffer-message))
	(goto-char (point-min))
	(if (eq mew-message-citation 'header)
	    (progn
	      (re-search-forward "^-*$" nil t)
	      (forward-line 1)
	      ))
	(let ((buffer-read-only nil))
	  (code-convert-region (point) (point-max) 
			       mew-mule-charset-local *internal*)
	  )
	))
  )


(defun mew-summary-convert-header ()
  (interactive)
  (if mew-mule-p
      (save-excursion
	(set-buffer (mew-buffer-message))
	(goto-char (point-min))
	(let ((buffer-read-only nil))
	  (mew-header-mime-decode)
	  )
	)
    )
  )

;;

(defun mew-summary-isearch-forward ()
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if mwin 
	(progn
	  (select-window mwin)
	  (unwind-protect
	      (isearch-forward)
	    (select-window cwin))
	  ))
    ))

(defun mew-summary-isearch-backward ()
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if mwin 
	(progn
	  (select-window mwin)
	  (unwind-protect
	      (isearch-backward)
	    (select-window cwin))
	  ))
    ))

;;

(defun mew-summary-x-face ()
  (interactive)
  (save-excursion
    (let (xface)
      (set-buffer (mew-buffer-message))
      (if (null (setq xface (mew-field-get-value "X-Face:")))
	  ()
	(set-buffer mew-buffer-tmp)
	(widen)
	(erase-buffer)
	(insert xface)
	(let ((selective-display nil)
	      (mc-flag nil)
	      (call-process-hook nil)
	      (default-process-coding-system 
		(if mew-mule-p (cons *noconv* *noconv*)))
	      (filters mew-x-face-filter))
	  (while filters
	    (call-process-region (point-min) (point-max) 
				 (car filters)
				 'delete t nil)
	    (setq filters (cdr filters))
	    )
	  (mew-mime-start-process (point-min) (point-max)
				  mew-x-face-prog
				mew-x-face-args)
	  )
	))
    ))


;;
;; quiting
;;

(defun mew-summary-suspend ()
  (interactive)
  (mew-window-pop))

(defun mew-kill-buffer (&optional buf)
  (interactive)
  (if buf
      (if (get-buffer buf) (kill-buffer buf))
    (kill-buffer (current-buffer)))
  )

(defun mew-summary-quit ()
  "Exit summary mode."
  (interactive)
  (if (null (mew-y-or-n-p "Quit mew? "))
      ()
    (setq mew-path nil)
    (setq mew-draft-folder nil)
    (setq mew-alias-alist ())
    (setq mew-folder-list ())
    (setq mew-folder-alist ())
    (setq mew-refile-alist ())
    (setq mew-cache nil)
    (if (and window-system mew-emacs19-p)
	(let ((x (buffer-list)))
	  (while x
	    (if (and (buffer-name (car x))
		     (string-match mew-buffer-message (buffer-name (car x))))
		(mew-kill-buffer (car x))
	      (setq x (cdr x))
	      )))
      (mew-kill-buffer (mew-buffer-message)))
    (mew-kill-buffer mew-buffer-hello)
    (mew-kill-buffer mew-buffer-mime)
    (mew-kill-buffer mew-buffer-tmp)
    (let ((x (buffer-list))
	  (regex (concat "^"
			 (regexp-quote mew-buffer-watch)
			 "\\|^"
			 (regexp-quote mew-buffer-cache))))
      (while x
	(if (and (buffer-name (car x))
		 (string-match regex (buffer-name (car x))))
	    (mew-kill-buffer (car x))
	  (setq x (cdr x))
	  )))
    (while mew-folders
      (mew-kill-buffer (car mew-folders))
      (setq mew-folders (cdr mew-folders))
      )
    (setq mew-folders (list mew-inbox))
    )
  )

;; kill-emacs-hook should not be list of functios but a single function
;; since kill-emacs not use run-hooks but c subroutine.
;; This means that you can't add-hook kill-emacs-hook.
;;
;;So, set
;; (setq mew-init-hook
;;       (function
;;        (lambda ()
;; 	 (setq kill-emacs-hook (function mew-mark-process-all-folders)))))
;;
;;or
;;
;; (setq mew-init-hook
;;       (function
;;        (lambda ()
;;         (your job1)
;;         (your job2)
;;	   ...
;;	   (mew-mark-process-all-folders))))


(defun mew-mark-process-all-folders ()
  (let ((folders mew-folders))
    (save-excursion
      (while folders
	(if (bufferp (car folders))
	    (progn
	      (set-buffer (car folders))
	      (if (mew-summary-mark-exist-p
		   (list mew-mark-rmm mew-mark-refile))
		  (if (mew-y-or-n-p
		       (format "Marks exists in %s. Process marks? "
			       (car folders)))
		      (mew-summary-exec)))
	      ))
	(setq folders (cdr folders))
	))
    ))

;;
;;

(defun mew-summary-pipe-message (prefix command)
  (interactive
   (list current-prefix-arg 
	 (read-string "Shell command on message: " mew-last-shell-command)))
  (mew-summary-display)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (widen)
    (if (string-equal command "")
	(setq command mew-last-shell-command))
    (goto-char (point-min)) ; perhaps this line won't be necessary
    (if prefix
	(search-forward "\n\n"))
    (shell-command-on-region (point) (point-max) command nil)
    (setq mew-last-shell-command command))
  (mew-message-narrow-to-page)
  )

;;;
;;; Message mode
;;;

(defun mew-message-mode ()
  "Major mode for display a message.
The keys that are defined for this mode are:

SPC	Scroll up this message.
DEL	Back scroll this message.
n	Display below message or part.
p	Display above message or part.
h, o	Get back to summary mode.
"
  (interactive)
  (setq major-mode 'mew-message-mode)
  (setq mode-name "Message")
  (use-local-map mew-message-mode-map)
  (setq buffer-read-only t)
  (setq mew-message-citation 'header)
  (make-local-variable 'page-delimiter)
  (setq page-delimiter mew-page-delimiter)
  (run-hooks 'mew-message-mode-hook)
  )

(defun mew-message-next-page (&optional lines)
;; return t   if no more pages
;;        nil if more pages
  (interactive)
  (move-to-window-line -1)
  (if (save-excursion
        (end-of-line)
        (and (pos-visible-in-window-p)
             (eobp)))
      ;; Nothing in this page.
      (if (or (null mew-break-pages)
	      (save-excursion
		(save-restriction
		  (widen) (forward-line 1) (eobp)))) ;Real end-of-buffer?
;;	  (progn (message "End of message") t) ;Nothing more.
	  t
	(mew-message-narrow-to-page 1)		;Go to next page.
	nil
	)
    ;; More in this page.
    (condition-case ()
	(scroll-up lines)
      (end-of-buffer
       ;; Long lines may cause an end-of-buffer error.
       (goto-char (point-max))))
    nil
    ))

(defun mew-message-narrow-to-page (&optional arg)
  (interactive "P")
  (setq arg (if arg (prefix-numeric-value arg) 0))
  (save-excursion
    (forward-page -1) ;;Beginning of current page.
    (widen)
    (cond
     ((> arg 0)	(forward-page arg))
     ((< arg 0) (forward-page (1- arg)))
     )
    (forward-page)
    (narrow-to-region (point)
		      (progn
			(forward-page -1)
			(if (and (eolp) (not (bobp)))
			    (forward-line 1))
			(point)))
    ))

(defun mew-message-prev-page (&optional lines)
  (interactive)
  (move-to-window-line 0)
  (if (and mew-break-pages
	   (bobp)
	   (not (save-restriction (widen) (bobp)))) ;Real beginning-of-buffer?
      (progn
	(mew-message-narrow-to-page -1) ;Go to previous page.
	(goto-char (point-max))
	(recenter -1))
    (scroll-down lines)))

(defun mew-message-goto-summary ()
  (interactive)
  (pop-to-buffer (car (mew-current-get 'message))))

(defun mew-message-next-msg (&optional arg)
  (interactive "p")
  (let ((obuf (current-buffer))
	(fld-msg (mew-current-get 'message))
	(fld (car (mew-current-get 'message)))
	(buffer-read-only nil)
	(mew-summary-buffer-disp-msg-current t)
	msg part)
    (set-buffer fld)
    (forward-line arg)
    (setq msg (mew-summary-message-number))
    (setq part (mew-summary-part-number))
    (set-buffer obuf)
    (cond (msg
	   (erase-buffer)
	   (mew-summary-display-message fld msg))
	  (part
	   (erase-buffer)
	   (mew-summary-display-part
	    (mew-cache-mime-syntax (mew-cache-hit fld-msg)) part)))))

(defun mew-message-prev-msg (&optional arg)
  (interactive "p")
  (mew-message-next-msg (- arg)))

(defmacro mew-cache-mime-syntax (buf)
  (` (save-excursion
       (set-buffer (, buf))
       mew-mime-syntax)))

;;
;;  Prepare new message --- caching
;;

(defvar mew-cache nil
      "Mail cache. (old ... new) order alist with association
 (("+folder" . "message") . cache-buffer)"
      )

(defmacro mew-cache-buffer-get (entry)
  (` (cdr (, entry))))

(defmacro mew-cache-folder-get (entry)
  (` (car (car (, entry)))))

(defmacro mew-cache-message-get (entry)
  (` (cdr (car (, entry)))))

(defmacro mew-cache-entry-make (fld-msg buf)
  (` (cons (, fld-msg) (, buf))))

(defmacro mew-cache-hit (fld-msg)
  "Return value assosiated with key."
  (` (mew-cache-buffer-get (assoc (, fld-msg) mew-cache))))

(defmacro mew-cache-sort (entry)
  (` (setq mew-cache (append (mew-delq (, entry) mew-cache)
			     (list (, entry))))))

(defun mew-cache-add (fld-msg)
  "Rutern associating buffer"
  (let ((len (length mew-cache))
	(buf nil))
    (if (>= len mew-cache-size)
	(prog1
	    (setq buf (mew-cache-buffer-get (car mew-cache)))
	  (setq mew-cache (append (cdr mew-cache)
				  (list (mew-cache-entry-make fld-msg buf))))
	  )
      (prog1
	  (setq buf (get-buffer-create 
		     (concat mew-buffer-cache (int-to-string len))))
	(setq mew-cache (append mew-cache
				(list (mew-cache-entry-make fld-msg buf))))
	))
    ))

(defun mew-cache-attribute-get (file)
  ;; 5th is modified time
  ;; 7th is file size
  (let ((attr (file-attributes file)))
    (list (nth 5 attr)
	  (nth 7 attr))))
       
(defun mew-cache-message (fld-msg)
  (let ((hit (mew-cache-hit fld-msg))
	(file (mew-expand-file-name (concat (car fld-msg) "/" (cdr fld-msg)))))
    (save-excursion
      (if hit
	  (progn
	    (mew-cache-sort (mew-cache-entry-make fld-msg hit))
	    (set-buffer hit)
	    ;; in cache buffer
	    (if (null (equal mew-cache-attribute 
			     (mew-cache-attribute-get file)))
		;; cache is invalid
		(mew-cache-message-new file)))
	;; no cache hit
	(setq hit (mew-cache-add fld-msg))
	(set-buffer hit)
	;; in cache buffer
	(mew-cache-message-new file)
	)
      hit ;; retrun value
      )))

(defun mew-cache-message-new (file)
  ;; in cache buffer
  (widen)
  (erase-buffer)
  (if mew-mule-p
      (let ((file-coding-system-for-read *noconv*))
	(insert-file-contents file))
    (insert-file-contents file))
  (if mew-debug
      (let ((debug-on-error t))
	(mew-mime-localform)
	(setq mew-mime-syntax (mew-mime-syntax-spice (mew-mime-syntax)))
	)
    (condition-case err
	(progn
	  (mew-mime-localform)
	  ;; buffer local variables
	  (setq mew-mime-syntax (mew-mime-syntax-spice (mew-mime-syntax)))
	  )
      (error
       (widen)
       (goto-char (point-min))
       (re-search-forward "^-*$")
       (beginning-of-line)
       (setq mew-mime-syntax (mew-rfc822-default-type))
       )
      ))
  (setq mew-cache-attribute (mew-cache-attribute-get file))
  )

;; use to debug
(defun mew-cache-flush ()
  (interactive)
  (let ((n 0))
    (while (< n mew-cache-size)
      (mew-kill-buffer (concat mew-buffer-cache (int-to-string n)))
      (setq n (1+ n))
      ))
  (mew-current-set 'cache nil)
  (setq mew-cache nil)
  )

(defun mew-alias-update ()
  (interactive)
  (setq mew-alias-alist (mew-alias-make-alist)))


;;
;; Cache header coordination
;;

(defun mew-cache-delete-field (field begin)
  (let ((case-fold-search t)
	(start nil))
    (save-excursion
      (goto-char (or begin (point-min)))
      (if (re-search-forward (format "^$\\|^%s" field) nil t)
	  (if (not (string= (downcase (mew-match 0)) (downcase field)))
	      nil
	    (setq start (match-beginning 0))
	    (forward-line 1)
	    (while (looking-at "[ \t]")
	      (forward-line 1))
	    (prog1
		(buffer-substring start (point))
	      (delete-region start (point)))
	    )
	nil)
      )))

;;
;; MIME local form creator
;;

(defun mew-charset-7bit-p (charset)
  (cond 
   ((string-match "iso-8859" charset) nil)
   ((string-match "iso-2022" charset) t)
   ((string-match "us-ascii" charset) t)
   (t t) ;; not safe
   )
  )

(defun mew-mime-decode (boundary encode charset &optional text)
  (let ((beg (point))
	(end (mew-mime-boundary-end boundary))
	(option nil)
	(selective-display nil)
	(mc-flag nil))
    (if encode (setq encode (downcase encode)))
    (cond 
     ((string= encode "quoted-printable") 
      (setq option mew-prog-mime-decode-quoted))
     ((string= encode "base64")
      (if text
	  (setq option mew-prog-mime-decode-base64-text)
	(setq option mew-prog-mime-decode-base64)))
     ((string= encode "x-gzip64")
      (if text
	  (setq option mew-prog-mime-decode-gzip64-text)
	(setq option mew-prog-mime-decode-gzip64)))
     )
    (if boundary (set-marker mew-decode-marker (1+ end)))
    ;; boundary exists, so set marker after region
    (if (null option)
	();; 7bit, 8bit, and binary
      (apply 'call-process-region beg end
	     mew-prog-mime-decode t t nil option)
      )
    (if (and mew-mule-p charset)
	(code-convert-region 
	 beg
	 (if boundary
	     (1- (marker-position mew-decode-marker)) ;; tricky
	   (point-max)) ;; singlepart stuff
	 (mew-mule-content-coding
	  (mew-mule-content-attr charset mew-mule-character-set))
	 *internal*
	 )
      )
    ))

(defun mew-mime-boundary-end (boundary)
  (if (null boundary)
      (point-max)
    (save-excursion
      (re-search-forward (concat "^--" boundary))
      (beginning-of-line)
      (1- (point)) ;; boundary includes preceding CRLF
      )
    ))

(defun mew-mime-localform ()
  (if (null (mew-field-get-value "Mime-Version:"))
      (if mew-mule-p
	  (code-convert-region (point-min) (point-max)
			       mew-mule-charset-local *internal*))
    (goto-char (point-min))
    ;; message/rfc822
;    (mew-header-mime-decode)
    (mew-mime-localform-singlepart))
  )

(defun mew-mime-localform-singlepart (&optional boundary digest)
  ;;Called just after boundary. So, just before Content-Type:.
  (mew-header-mime-decode)
  (let* ((case-fold-search t)
	 (begin  (point)) ;; xxx
	 (part   (mew-part-make));; move point after content-*
	 (type   (mew-part-get-type   part))
	 (param1 (mew-part-get-param1 part))
	 (encode (mew-part-get-encode part))
	 (charset nil))
    (if (and (null type)
	     (or (null encode) (string-match "7bit" encode))
	     (null digest))
	() ;; text/plain; charset=us-ascii
      (if (and (stringp param1) (string-match "^Charset=\\(.*\\)$" param1))
	  (setq charset (mew-match 1 param1)))
      (cond 
       ;; message/rfc822
       ;; no decoding is required
       ;; type may be null. be carefull
       ((or digest (string-match "message/rfc822" type))
	(while (looking-at "^$") (forward-line)) ;; skip null mime header
	(save-restriction
	  (narrow-to-region (point) (mew-mime-boundary-end boundary))
	  (mew-mime-localform))
	)
       ;; multipart
       ;; no decoding is required
       ((string-match "^Multipart/\\(.*\\)$" type)
	(if (string-match "digest" (mew-match 1 type))
	    (mew-mime-localform-multipart part 'digest)
	  (mew-mime-localform-multipart part))
	)
       ;; aplication switch. make general
       ((string-match mew-pem-content-type type)
	(mew-cache-delete-field "Content-Type:" begin)
	(delete-char -1) ;; xxx delete unnecessary \n
	(mew-pem-decrypt-region (point) (mew-mime-boundary-end boundary))
	(if (null (looking-at "Content-"))
	    (insert "\n"))
	;; xxx eval single part again after decode
	)
       ;; support only PGPed text.
       ((string-match mew-pgp-content-type type)
	(mew-cache-delete-field "Content-Type:" begin)
	(mew-pgp-decode-region (point) (mew-mime-boundary-end boundary))
	)
       ;; param1 = charset=iso-2022-jp
       ((or encode charset)
	(if (or (null type) (string-match "text/" type))
	    (mew-mime-decode boundary encode charset 'text)
	  (mew-mime-decode boundary encode charset))
	)
       )
      )
    ))


(defun mew-mime-localform-multipart (part &optional digest)
  (let ((case-fold-search nil) ;; boundary is case sensitive
	(param1 (mew-part-get-param1 part))
	(boundary nil))
    (let ((case-fold-search t))
      (if (string-match "^Boundary=\\(.*\\)$" param1)
	  (setq boundary (regexp-quote (mew-match 1 param1)))
	(error "No boundary parameter for multipart"))
      )
    (re-search-forward (concat "^--" boundary ))
    (forward-line) 
    (mew-mime-localform-singlepart boundary digest)
    (catch 'multipart
      (while (re-search-forward (concat "^--" boundary "-?-?$"))
	(forward-line)
	(if (string= (regexp-quote (mew-match 0)) (concat "--" boundary "--"))
	    (throw 'multipart nil)
	  )
	(mew-mime-localform-singlepart boundary digest)
	))
    ))

;;;
;;; MIME local form syntax analizer
;;;

;; <message> = [ 'message hbed hend
;;                ("message/rfc822") (CTE:) (CI:) (CD:) <part> ]
;; <single>  = [ 'single beg end (CT:) (CTE:) (CI:) (CD:) ]
;; <multi>   = [ 'multi nil nil
;;               ("multipart/mixed" |
;;                "multipart/digest" |
;;                "multipart/alternative" |
;;                "multipart/parallel " )
;;               (CTE:) (CI:) (CD:)
;;                1*<part> ]
;; <part>    = <message>|<single>|<multi>


(defmacro mew-message-singlepart-p (message)
  (` (eq (aref (aref (, message) mew-part-magic) 0) 'single)))

(defmacro mew-message-multipart-p (message)
  (` (eq (aref (aref (, message) mew-part-magic) 0) 'multi)))

(defmacro mew-message-get-begin (message)
  (` (aref (, message) 1)))

(defmacro mew-message-set-begin (begin message)
  (` (progn (aset (, message) 1 (, begin)) (, message))))

(defmacro mew-message-get-end (message)
  (` (aref (, message) 2)))

(defmacro mew-message-set-end (end message)
  (` (progn (aset (, message) 2 (, end)) (, message))))

(defmacro mew-message-get-part (message)
  (` (aref (, message) mew-part-magic)))

(defmacro mew-message-set-part (body message)
  (` (progn (aset (, message) mew-part-magic (, body)) (, message))))

(defmacro mew-message-part-set-end (end message)
  (` (progn (aset (aref (, message) mew-part-magic) 2 (, end)) (, message))))

;;

(defmacro mew-part-message-p (part)
  (` (eq (aref (, part) 0) 'message)))

(defmacro mew-part-singlepart-p (part)
  (` (eq (aref (, part) 0) 'single)))

(defmacro mew-part-multipart-p (part)
  (` (eq (aref (, part) 0) 'multi)))

(defmacro mew-part-get-begin (part)
  (` (aref (, part) 1)))

(defmacro mew-part-set-begin (begin part)
  (` (progn (aset (, part) 1 (, begin)) (, part))))

(defmacro mew-part-get-end (part)
  (` (aref (, part) 2)))

(defmacro mew-part-set-end (end part)
  (` (progn (aset (, part) 2 (, end)) (, part))))

(defmacro mew-part-get-type (part)
  (` (car (aref (, part) 3))))

(defmacro mew-part-get-params (part)
  (` (cdr (aref (, part) 3))))

(defmacro mew-part-get-param1 (part)
  (` (car (cdr (aref (, part) 3)))))

(defmacro mew-part-get-encode (part)
  (` (car (aref (, part) 4))))

(defmacro mew-part-get-id (part)
  (` (car (aref (, part) 5))))

(defmacro mew-part-get-desc (part)
  (` (car (aref (, part) 6))))

(defmacro mew-part-get-part (part)
  (` (aref (, part) mew-part-magic )))

(defun mew-multi-get-parts (multi) 
  (let ((ret nil)
	(n (1- (length multi)))
	(sentinel (1- mew-part-magic)))
    (while (> n  sentinel)
      (setq ret (cons (aref multi n) ret))
      (setq n (1- n))
      )
    ret
    ))

(defmacro mew-part-to-multi (part)
  (` (progn (aset (, part) 0 'multi) (, part))))

(defmacro mew-part-get-head (part)
  (` (vector (aref (, part) 0) (aref (, part) 1) (aref (, part) 2)
	     (aref (, part) 3) (aref (, part) 4)
	     (aref (, part) 5)
	     (aref (, part) 6))))

(defmacro mew-part-only-one-part-p (multi)
  (` (equal (1+ mew-part-magic) (length (, multi)))))

(defmacro mew-single-make (fields)
  (` (vconcat (vector 'single (point) nil) (, fields))))

(defmacro mew-syntax-cat (parts part)
  (` (vconcat (, parts) (vector (, part)))))

;;
;;
;;

(defun mew-mime-default-type (&optional type)
  (forward-line) ;; skip null line due to "^-*$"
  (vector 'single (point) (point-max) ;; end is dummy
	  (if (null type)
	      '("text/plain" "charset=us-ascii")
	    (list type))
	  nil nil nil)
  )

(defun mew-rfc822-default-type ()
  (vector 'message (point-min) (point) ;; end is dummy
	  (list "message/rfc822") nil nil nil (mew-mime-default-type)))


(defun mew-part-make ()
  (let ((fields (mew-fields-get-as-list mew-mime-fields)))
    ;; point moves after content-* and \n
    (mew-single-make fields)
    ))

(defun mew-position (list a)
  (let ((n 0)
	(ret nil))
    (catch 'loop 
      (while list 
	(if (equal (downcase (car list)) (downcase a))
	    (throw 'loop (setq ret n)))
	(setq n (1+ n))
	(setq list (cdr list))
	)
      )
    ret
    ))

(defun mew-fields-get-as-list (fields)
  (let* ((case-fold-search t)
	 (start nil)
	 (end nil)
	 (key nil)
	 (position nil)
	 (len (length fields))
	 (vec (make-vector len nil))
	 (n (1- len))
	 (ret))
    (catch 'loop
      (while (re-search-forward "^-*$\\|^[^:\n]+:" nil t)
	;; Emacs 19's regex is not line based, so \n is necessary.
	;; \n is not harmless to Emacs 18.
	(setq key (mew-match 0))
	(setq start (match-end 0))
	(if (string-match "^-*$" key) (throw 'loop (forward-line)))
	(forward-line 1)
	(while (looking-at "[ \t]")
	  (forward-line 1))
	(setq end (1- (point)))
	(if (setq position (mew-position fields key))
	    (aset vec position (buffer-substring start end)))
	))
    (while (> n -1)
      (if (aref vec n)
	  (setq ret (cons (mew-header-syntax (aref vec n)) ret))
	(setq ret (cons nil ret)))
      (setq n (1- n))
      )
    ret
    ))

;;
;;
;;

(defun mew-mime-syntax ()
  (let* ((ver (mew-field-get-value "Mime-Version:"))
	 (mime (and ver (mew-header-delete-ws ver)))
	 (case-fold-search t))
    (goto-char (point-min))
    (re-search-forward "^-*$")
    (beginning-of-line)
    (cond
     ((not mime)  ;; RFC822
      (mew-rfc822-default-type) ;; min, point, point+1, point-max
      )
     (t ;; MIME
      (if (null (string-match (concat "^" mew-mime-version) mime))
	  (error "Unsupport MIME version %s." ver)
	(let ((skel (mew-rfc822-default-type))) ;; cursor after "^-*$"
	  ;; virtual message/rfc822 header
	  ;; mew-mime-default-type in mew-rfc822-default-type is meaningless
	  ;; min, point, point+1, point-max
	  (goto-char (point-min))
	  (mew-message-set-part
	   (mew-mime-syntax-singlepart nil 'toplevel) skel)
	  )
	)
      )
     )
    ))

(defun mew-mime-syntax-singlepart (&optional digest toplevel)
  ;;called just after boundary. So, just before Content-Type:.
  (if (null (looking-at "^$"))
      (let* ((case-fold-search t)
	     (part (mew-part-make));; move point after content-* and \n
	     (type (mew-part-get-type part))
	     (subpart))
	(if (null type) (setq type "text/plain"))
	(cond
	 ((string-match "^Multipart/\\(.*\\)$" type)
	  (if (string-match "digest" (mew-match 1 type))
	      (mew-mime-syntax-multipart part 'digest)
	    (mew-mime-syntax-multipart part))
	  )
	 ((string-match "^message/rfc822" type)
	  (save-restriction
	    (narrow-to-region (point) (point-max))
	    (mew-mime-syntax)
	    )
	  )
	 (t (if toplevel (mew-part-set-end (point-max) part) part))
	 )
	)
    (if digest
	(progn
	  (while (looking-at "^$") (forward-line))
	  (save-restriction
	    (narrow-to-region (point) (point-max))
	    (mew-mime-syntax)
	    )
	  )
      (if toplevel
	  (mew-part-set-end (point-max) (mew-mime-default-type))
	(mew-mime-default-type)) ;; set end in multipart
      )
    )
  )

(defun mew-mime-syntax-multipart (part &optional digest)
  (let ((case-fold-search nil) ;; boundary is case sensitive
	(param1 (mew-part-get-param1 part))
	(boundary nil)
	(last nil)
	(multi [ ]))
    (let ((case-fold-search t))
      (if (string-match "^Boundary=\\(.*\\)$" param1)
	  (setq boundary (regexp-quote (mew-match 1 param1)))
	(error "No boundary parameter for multipart"))
      )
    (re-search-forward (concat "^--" boundary "$"))
    (forward-line) 
    (setq last (mew-mime-syntax-singlepart digest))
    (catch 'multipart
      (while (re-search-forward (concat "^--" boundary "-?-?$"))
	(beginning-of-line) ;; for end
	;; boundary includes preceding CRLF, so (1- )
	(if (mew-part-message-p last)
	    (if (mew-message-multipart-p last)
		(setq multi (mew-syntax-cat multi last))
	      (setq multi (mew-syntax-cat multi 
					  (mew-message-part-set-end
					   (1- (point)) last))))
	  (setq multi (mew-syntax-cat multi
				      (mew-part-set-end (1- (point)) last))))
	(forward-line) ;; skip boundary
	(if (string= (regexp-quote (mew-match 0)) (concat "--" boundary "--"))
	    (throw 'multipart
		   (vconcat (mew-part-set-end
			     (point) (mew-part-to-multi part)) multi))
	  (setq last (mew-mime-syntax-singlepart digest)))
	))
    ))

;;;
;;; Make mew-mime-syntax appropriate to Mew MIME viewer
;;;

(defconst mew-multi-skel [multi nil nil ("multipart/mixed") nil nil nil])

(defun mew-mime-syntax-spice (syntax)
  (cond 
   ((mew-part-singlepart-p syntax) syntax)
   ((mew-part-message-p syntax)
    (let* ((head (mew-part-get-head syntax))
	   (part (mew-message-get-part syntax))
	   (hotpart (mew-mime-syntax-spice part)))
      (if (mew-part-message-p part)
	  (mew-syntax-cat head (mew-syntax-cat mew-multi-skel hotpart))
	(mew-syntax-cat head hotpart))
      )
    )
   ((mew-part-multipart-p syntax)
    (let ((head (mew-part-get-head syntax))
	  (part (mew-part-get-part syntax)))
      (if (mew-part-only-one-part-p syntax)
	  (if (mew-part-message-p part)
	      (mew-syntax-cat head (mew-mime-syntax-spice part))
	    (mew-mime-syntax-spice part))
	(let* ((parts (mew-multi-get-parts syntax))
	       (len (length parts))
	       (vec (make-vector len nil))
	       (n 0))
	  (while (< n len)
	    (aset vec n (mew-mime-syntax-spice (nth n parts)))
	    (setq n (1+ n))
	    )
	  (vconcat head vec)
	  )
	))
    )
   ))

;;
;;
;;

(defun mew-mime-syntax-print-loop (parts pref)
  (cond
   ((mew-part-message-p parts)
    (insert (format "%s\t%s\t%s\n" 
		    pref
		    (mew-part-get-type parts)
		    (or (mew-part-get-desc parts) "")))
    (if (mew-part-multipart-p (mew-message-get-part parts))
	(mew-mime-syntax-print-loop
	 (mew-message-get-part parts)
	 (concat "\t" pref))))
   ((mew-part-singlepart-p parts)
    (insert (format "%s\t%s\t%s\n" 
		    pref
		    (mew-part-get-type parts)
		    (or (mew-part-get-desc parts) ""))))
   ((mew-part-multipart-p parts)
    (setq parts (mew-multi-get-parts parts))
    (let ((count 1))
      (while parts
	(mew-mime-syntax-print-loop 
	 (car parts)
	 (concat pref "." (int-to-string count)))
	(setq count (1+ count))
	(setq parts (cdr parts))
	))
    )
   )
  )

(defun mew-mime-syntax-print (parts)
  ;; message is always multipart.
  (setq parts (mew-multi-get-parts (mew-message-get-part parts)))
  (let ((pref "\t")
	(count 1))
    (while parts
      (mew-mime-syntax-print-loop 
       (car parts)
       (concat pref (int-to-string count)))
      (setq count (1+ count))
      (setq parts (cdr parts))
      )
    ))


(defun mew-mime-part-syntax (syntax num)
  (if (null num)
      (mew-message-get-part syntax)
    (mew-mime-part-syntax-loop syntax (mew-multi-split-number num))
    )
  )

(defun mew-mime-part-syntax-loop (syntax nums)
  (cond 
   ((null nums)	syntax) ;; singlepart mutch here
   ((mew-part-message-p syntax)
    (mew-mime-part-syntax-loop (mew-message-get-part syntax) nums))
   ((mew-part-multipart-p syntax)
    (mew-mime-part-syntax-loop
     (nth (1- (car nums)) (mew-multi-get-parts syntax)) (cdr nums)))
   )
  )

;;;
;;;
;;;

(defun mew-header-insert-from-file (file)
  (insert-file-contents file) ;; whole message
  ;;(insert "\n")
  (mew-header-arrange)
  )

(defun mew-header-insert-from-cache (cache begin end)
  (insert-buffer-substring cache begin end)
  (mew-header-arrange)
  (insert "\n")
  )

(defun mew-header-arrange ()
  (let* ((len (length mew-field-visible))
	 (visible-regex (mapconcat 
			 (function (lambda (x) (concat "^" x "$")))
			 mew-field-visible "\\|"))
	 (invisible-regex (mapconcat 
			   (function (lambda (x) (concat "^" x "$")))
			   mew-field-invisible "\\|"))
	 (visible (make-vector len nil))
	 (others nil) (key "") (line "") (start nil) (n nil))
    (goto-char (point-min))
    (catch 'header
      (while (re-search-forward "^-*$\\|^[^ \t\n]*:")
	(setq key (mew-match 0))
	(setq start (match-beginning 0))
	; XXX: mew-summary-buffer-disp-msg-current is taken over from 
	; mew-summary-display or mew-summary-show.
	; Because (recenter 0) destroies top of the summary window
	; when mew-summary-buffer-disp-msg is nil. I don't know why.
	(if (string-match "^-*$" key)
	    (throw 'header (and mew-summary-buffer-disp-msg-current
				(recenter 0))))
	(forward-line 1)
	(while (looking-at "[ \t]")
	  (forward-line 1))
	(if (string-match invisible-regex key)
	    ()
	  (setq line (buffer-substring start (point)))
	  (delete-region start (point))
	  (if (null (string-match visible-regex key))
	      (setq others (concat others line))
	    (setq n (mew-member-match key mew-field-visible))
	    (aset visible n (concat line (aref visible n)))
	    ))
	))
    (setq n 0)
    (while (< n len)
      (if (aref visible n) (insert (aref visible n)))
      (setq n (1+ n))
      )
    (and others (insert others))
    )
  )

;;;
;;; Draft mode
;;;

(defvar mew-draft-window-config nil)

(defun mew-draft-mode ()
  "Major mode for composing MIME message.
Key action is diffrent in each region -- Header, Body, Multipart.

To send a draft, type C-cC-m and C-cC-c.
To make multipart, type C-cM, edit multipart, and type C-cC-m and C-cC-c.

*Whole buffer key assignment:

C-cC-y	Cite from *mew message* buffer.
C-cC-m	Make MIME message. Charset guess, maps directory structure to 
	multipart, and so on.
C-cC-c	Send this message. If you skipped C-cC-m, Content-Type: is added
	to header and you are asked, 
	\"Content-Type: was automatically added. Send this mail? \".
	Type y or n. 
	Mew sends the message in background. So, when you exit Emacs, you
	may be asked, 
	\"Active processes exist; kill them and exit anyway? (yes or no)\".
	In this case, check if *mew watch* buffer exist. If so, never 
	exit Emacs because Mew is still sending the message.

C-cC-q	Kill this draft.

(Following functions automatically insert an appropriate Content-Type:. 
So, C-cC-m is not necessary.)
C-cC-e	Encrypt this message by PEM. Currently only FJPEM is supported.
C-cC-s	Sign this message by PEM. Currently only FJPEM is supported.
C-ce	Encrypt this message by PGP.
C-cs	Sign this message by PGP.
C-cb	Sign then encrypt this message by PGP.
C-cp	Insert your PGP public key here.

C-cM	Prepare multipart region.

*Header region key assignment:

TAB	Complete and expand alias on To: or Cc: line.
C-cTAB	Complete your mail domain on To: or Cc: line.


*Body region key assignment:

C-cTAB	Insert signature file.


*Multipart region Key assignment:

c	Copy a file in +drafts/mime/<num>/.
d	Delete this file.	
l	Link inputed file from +drafts/mime/<num>/.
r	Rename this file.
m	Make sub-multipart.
f	Read this file into a buffer.
F	Read a new file into a buffer.
e	Input external-body.
a	Sampling voice and insert as audio file.
T	Change its Content-Type:.
D	Input its Content-Description:.

* Fill blanks

Prepare ~/.mew-fib like;

	name:  Kazuhiko Yamamoto
	email: kazu@is.aist-nara.ac.jp

If you receive a message like;

	Your name : |>name<|
	Your e-mail address: |>email<|

Type a (in summary mode), C-cy, C-cC-f, and C-cC-k makes following
draft.

	Your name : Kazuhiko Yamamoto
	Your e-mail address: kazu@is.aist-nara.ac.jp

In this way, mew-fil fills up items quoted like |> <| from .mew-fib.

C-cy	Yank from *mew message* buffer without citation label.
C-cC-f	Fill |>item<| from .mew-fib.
C-cC-k	Delete all quotations, i.e. |> <|.
C-cC-n	Jump to the next fib item.
C-cC-p	Jump to the previous fib item.
C-cC-u	Flush input from .mew-fib.

Moreover, mew-fib supports aliases like;

	email: kazu@is.aist-nara.ac.jp
	e-mail:

"
  (interactive)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^[ \t]*[-_][-_][-_]+$\\|" paragraph-start))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate
        (concat "^[ \t]*[-_][-_][-_]+$\\|" paragraph-separate))
  (setq major-mode 'mew-draft-mode)
  (setq mode-name "Draft")
  (use-local-map mew-draft-mode-map)
  (set-syntax-table mew-draft-mode-syntax-table)
  (cd (expand-file-name "~"))
  (run-hooks 'text-mode-hook 'mew-draft-mode-hook)
  (if mew-emacs19-p
      (if (featurep 'hilit19)
          (hilit-rehighlight-buffer)
        (redraw-display))
    (redraw-display))
  ;; xxx how about xemacs ?
  (cond
   (mew-mule-p
    (make-local-variable 'file-coding-system)
    (make-local-variable 'file-coding-system-for-read)
    (make-local-variable 'default-process-coding-system)
    (make-local-variable 'mew-draft-window-config)
    (setq file-coding-system mew-mule-charset-integrity)
    (setq file-coding-system-for-read *autoconv*)
    (setq default-process-coding-system 
	  (cons mew-mule-charset-integrity mew-mule-charset-integrity))
    )
   )
  (cond
   (mew-emacs18-p ;; never user (and )
    (if auto-fill-hook 
	(progn
	  (make-local-variable 'auto-fill-hook)
	  (setq auto-fill-hook (function mew-draft-auto-fill))
	  )
      )
    )
   ((not mew-xemacs-p)
    (if auto-fill-function
	(progn
	  (make-local-variable 'auto-fill-function)
	  (setq auto-fill-function (function mew-draft-auto-fill))
	  )
      )
    )
   )
  )

(defun mew-draft-auto-fill ()
  (do-auto-fill)
  (if (mew-draft-in-header-p)
      (save-excursion
	(beginning-of-line)
	(if (not (looking-at "[^ \t]+:\\|[ \t]"))
	    (insert "\t"))
	)))

(defun mew-draft-get-new (&optional folder)
  (save-excursion
    (set-buffer mew-buffer-tmp)
    (erase-buffer)
    (call-process "mhpath" nil (current-buffer) nil
		   (or folder mew-draft-folder)
		   "new")
    (goto-char (point-min))
    (forward-line)
    (buffer-substring (point-min) (1- (point)))
    ))

(defun mew-draft-rename (file)
  (rename-buffer (concat mew-draft-folder 
			 "/" 
			 (file-name-nondirectory file)))
  )

;; I'm sorry. This code is very much procedural.

(defun mew-draft-header (to cc subject &optional in-reply-to references)
;; to -- string or list
;; cc -- string or list
  (let ((del (cons (concat "^" (regexp-quote (user-login-name)) "$")
		   (cons mew-mail-address mew-mail-address-list)))
	;; deleting list for Cc:
	(c nil))
    (goto-char (point-min))
    ;; Insert To: at first.
    ;; All addresses inserted on To: are appended to del.
    (if (null to)
	(insert "To: \n")
      (cond
       ;; To: inputed from the mini-buffer.
       ((stringp to)
	(setq del (cons to del))
	(mew-field-insert-here "To:" to)
	)
       ;; To: collected by reply
       ((listp to)
	(setq del (append to del))
	(insert (format "To: %s" (car to)))
	(setq to (cdr to))
	(while to
	  (insert (format ",\n\t%s" (car to)))
	  (setq to (cdr to)))
	(insert "\n")
	)
       ))
    (if (null cc)
	(insert "Cc: \n")
      (cond 
       ;; Cc: inputed from the mini-buffer.
       ((stringp cc)
	(mew-field-insert-here "Cc:" cc))
       ;; Cc: collected by reply.
       ((listp cc)
	(insert "Cc: ")
	;; find the first Cc: value since this is preceding by Cc:
	(catch 'first
	  (while cc
	    (setq c (car cc) cc (cdr cc))
	    (if (or 
		 (catch 'match 
		   (car (mapcar 
			 (function 
			  (lambda (arg)
			    (and (string-match arg c) (throw 'match t))))
			 del)))
		 (string-match ":;" c))
		()
	      (insert c)
	      (throw 'first nil))
	    ))
	;; insert the second or more Cc: preceding ",\n\t"
	(while cc
	  (setq c (car cc) cc (cdr cc))
	  (if (or 
	       (catch 'match 
		 (car (mapcar 
		       (function 
			(lambda (arg)
			  (and (string-match arg c) (throw 'match t))))
		       del)))
	       (string-match ":;" c))
	      ()
	    (insert (format ",\n\t%s" c)))
	  )
	(insert "\n")
	)
       ))
    )
  (and mew-cc   (mew-field-insert-here "Cc:" mew-cc))
  (and subject  (mew-field-insert-here "Subject:" subject))
  (and mew-from (mew-field-insert-here "From:" mew-from))
  (and mew-fcc  (mew-field-insert-here "Fcc:" mew-fcc))
  (and mew-dcc  (mew-field-insert-here "Dcc:" mew-dcc))
  (and mew-reply-to (mew-field-insert-here "Reply-To:" mew-reply-to))
  ;; xxx date to in-reply-to ?
  (and in-reply-to (mew-field-insert-here 
		    "In-Reply-To:"
		    (concat "Your message of \"" in-reply-to "\"")))
  (and references (mew-field-insert-here "References:" references))
  (if (and mew-x-face-file
	   (file-exists-p (expand-file-name mew-x-face-file)))
      (let ((xface))
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (erase-buffer)
	  (insert-file-contents (expand-file-name mew-x-face-file))
	  (setq xface (buffer-substring (point-min) (max (buffer-size) 1)))
	  )
	(mew-field-insert-here "X-Face:" xface)
	))
  (and mew-x-mailer (mew-field-insert-here "X-Mailer:" mew-x-mailer))
  (mew-field-insert-here "Mime-Version:" mew-mime-version)
  (setq mew-draft-buffer-header (point-marker))
  (insert "----\n")
  )

(defun mew-draft-send-letter ()
  (interactive)
  (run-hooks 'mew-send-hook)
  (save-excursion
    (if (mew-field-get-value "Content-Type:")
	(mew-draft-real-send-letter)
      (message "Making a MIME letter ...")
      (mew-draft-make-mime)
      (message "Making a MIME letter ... done")
      (condition-case err
	  (if (mew-y-or-n-p
	       "Content-Type: was automatically added. Send this mail? ")
	      (mew-draft-real-send-letter)
	    (mew-draft-delete-content-type))
	(quit
	 (mew-draft-delete-content-type)))
      )
    ))

(defun mew-draft-delete-content-type ()
  (goto-char (point-min))
  (save-restriction
    (narrow-to-region (point) (marker-position mew-draft-buffer-header))
    (let ((case-fold-search t)
	  (begin nil))
      (re-search-forward "^Content-Type:")
      (beginning-of-line) ;; for safty
      (setq begin (point))
      (if (looking-at "Content-Type: multipart")
	  ()
	(forward-line)
	(while (looking-at "[ \t]")
	  (forward-line 1))
	(delete-region begin (point))
	)
      )
    ))

(defun mew-draft-real-send-letter ()
  (let* ((mimefolder (mew-draft-to-mime (buffer-name)))
	 (mimedir (mew-expand-file-name mimefolder))
	 (msg (file-name-nondirectory (buffer-file-name)))
	 (fcc nil)
	 (delete nil))
;;    (run-hooks 'mew-send-hook)
    (set-buffer-modified-p t)		; Make sure buffer is written
    (save-buffer)
    ;; make backup folder
    (if (setq fcc (mew-field-get-value "Fcc:"))
	(if (file-exists-p (mew-expand-file-name fcc))
	    () ;; do nothing
	  (call-process mew-prog-folder nil nil nil
			"-create"
			(mew-dir-to-folder fcc))
	  ))
    (kill-buffer (current-buffer))
    ;;
    (if (mew-current-get 'window)
	(progn
	  (set-window-configuration (mew-current-get 'window))
	  (mew-current-set 'window nil)))
    (set-buffer (generate-new-buffer mew-buffer-watch))
    ;; watch buffer
    (setq mew-watch-buffer-process
	  (start-process "Send" (current-buffer)
			 mew-prog-send
			 "-draftfolder" mew-draft-folder
			 "-draftmessage" msg
			 "-watch" "-verbose"))
    (set-process-sentinel mew-watch-buffer-process
			  'mew-watch-sentinel)
    (message "Sending a message in background")
    (if (null (file-directory-p mimedir))
	()
      (cond
       ((equal mew-mime-compose-folder-delete 'ask)
	(setq delete (mew-y-or-n-p
		      (format "Folder %s exists. Remove it? " mimefolder))))
       ((equal mew-mime-compose-folder-delete 'delete)
	(setq delete t))
       ((equal mew-mime-compose-folder-delete 'retain)
	(setq delete nil))
       (t
	(setq delete (mew-y-or-n-p
		      (format "Folder %s exists. Remove it? " mimefolder))))
       )
      (if (null delete )
	  (format "Folder %s remains" mimefolder)
	(message "Removing folder %s ... " mimefolder)
	(call-process "rm" nil nil nil "-rf" mimedir)
	(message "Removing folder %s ... done" mimefolder))
      )
    ))

(defun mew-watch-sentinel (process event)
  (let ((cbuf (current-buffer)) (kbuf (process-buffer process)))
    (set-buffer kbuf)
    (goto-char (point-min))
    (if (null (re-search-forward "^send:" nil t))
	(progn
	  (set-buffer cbuf)  ;; to avoid cursor-in-echo-area bug
	  (kill-buffer kbuf) ;; set-buffer before kill-buffer
	  )
      (ding)
      (message "Send failed")
      (beginning-of-line)
      (switch-to-buffer (process-buffer process))
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )
    ))

(defun mew-draft-yank ()
  (interactive)
  (let ((cite))
    (save-excursion
      (save-restriction
	(set-buffer (mew-buffer-message))
	(widen)
	(setq cite
	      (cond
	       ((marker-position (mark-marker)) ;;; (mark)
		(buffer-substring (region-beginning)
				  (region-end)))
	       ((equal mew-message-citation 'noheader)
		(buffer-substring (point-min) (point-max)))
	       (t 
		(goto-char (point-min))
		(re-search-forward (format "^-*$"))
		(forward-line)
		(buffer-substring (point) (point-max)))
	       )
	      )
	))
    (insert-string cite)
    ))


(defun mew-draft-cite (&optional arg)
  (interactive "P")
  (let ((from "") (subject "") (date "") (cite "") (fields "")
	(nonmewbuf mew-message-citation-buffer)) ;; buffer local
    (save-excursion
      (if nonmewbuf
	  (set-buffer nonmewbuf)
	(set-buffer (mew-cache-hit (mew-current-get 'message))))
      (save-restriction
	(widen)
	(setq from (or (mew-field-get-value "From:") ""))
	(setq date (or (mew-field-get-value "Date:") ""))
	(setq subject (or (mew-field-get-value "Subject:") ""))
	)
      (save-restriction
	(set-buffer (or nonmewbuf (mew-buffer-message)))
	(widen)
	(setq cite
	      (cond
	       (arg  ;; C-u C-c C-y cites body with header.
		(buffer-substring (point-min) (point-max)))
	       ((marker-position (mark-marker)) ;;; (mark)
		(buffer-substring (region-beginning)
				  (region-end)))
	       ((equal mew-message-citation 'noheader)
		(buffer-substring (point-min) (point-max)))
	       (t 
		(goto-char (point-min))
		(re-search-forward (format "^-*$"))
		(forward-line)
		(buffer-substring (point) (point-max)))
	       )
	      )
	))
    (if mew-cite-hook
	;; use super-cite
	(progn
	  (narrow-to-region (point) (point))
	  (save-excursion
	    (cond
	     (nonmewbuf (set-buffer nonmewbuf))
	     ((mew-current-get 'cache) (set-buffer (mew-current-get 'cache)))
	     (t (set-buffer (mew-buffer-message)))
	     )
	    (beginning-of-buffer)
	    (re-search-forward "^$")
	    (setq fields (buffer-substring (point-min) (point))))
	  (insert fields)
	  (insert "\n")
	  (insert cite)
	  (beginning-of-buffer)
	  (push-mark nil)
	  (end-of-buffer)
 	  (run-hooks 'mew-cite-hook)
	  (pop-mark))
      ;; use mew original cite
      (insert (eval-expression mew-cite-format))
      (narrow-to-region (point) (point))
      (set-mark (point)) ;; for C-x C-x
      (insert cite)
      (goto-char (point-min))
      (and (bolp) (insert mew-cite-prefix))
      (while (equal 0 (forward-line 1))
	(or (equal (point) (point-max))
	    (insert mew-cite-prefix)))
	)
    (widen)
    ))

(defun mew-draft-check-whom ()
  "Verify recipients of the draft."
   (interactive)
  (let* ((to-cc (mew-header-user-collect
		'("To:" "Cc:" "Bcc:" "Apparently-To:")))
	 (exp-to-cc (mew-header-expand-alias-list to-cc))
	 (buf (current-buffer)))
    (message "Checking recipients ... ")
    (get-buffer-create mew-buffer-whom)
    (switch-to-buffer-other-window mew-buffer-whom)
    (erase-buffer)
    (mapcar
     (function (lambda (x) (insert (format "%s\n" x))))
     exp-to-cc
     )
    (pop-to-buffer buf)
    (message "Checking recipients ... done")
    ))

(defun mew-draft-kill ()
  (interactive)
  (let* ((mimefolder (mew-draft-to-mime (buffer-name))) ;; buffer will kill
	 (mimedir (mew-expand-file-name mimefolder))
	 (delete nil))
    (if (mew-y-or-n-p "Kill draft message? ")
	(progn
	  (set-buffer-modified-p t) ;; for delete recovery file
	  (save-buffer)
	  (if (file-exists-p (buffer-file-name))
	      (delete-file (buffer-file-name)))
	  (set-buffer-modified-p nil)
	  (and (get-buffer (mew-draft-to-mime (buffer-name)))
	       (kill-buffer (mew-draft-to-mime (buffer-name))))
	  (kill-buffer (buffer-name))
	  (if (mew-current-get 'window)
	      (progn
		(set-window-configuration (mew-current-get 'window))
		(mew-current-set 'window nil)))
	  (message "Draft was killed")
	  (if (null (file-directory-p mimedir))
	      ()
	    (cond
	     ((equal mew-mime-compose-folder-delete 'ask)
	      (setq delete (mew-y-or-n-p
			    (format "Folder %s exists. Remove it? "
				    mimefolder))))
	     ((equal mew-mime-compose-folder-delete 'delete)
	      (setq delete t))
	     ((equal mew-mime-compose-folder-delete 'retain)
	      (setq delete nil))
	     (t
	      (setq delete (mew-y-or-n-p
			    (format "Folder %s exists. Remove it? "
				    mimefolder))))
	     )
	    (if (null delete )
		(format "Folder %s remains" mimefolder)
	      (message "Removing folder %s ... " mimefolder)
	      (call-process "rm" nil nil nil "-rf" mimedir)
	      (message "Removing folder %s ... done" mimefolder))
	    )
	  )
      (message "Draft was not killed"))
    ))

(defun mew-draft-insert-signature ()
  (interactive)
  (insert-file-contents (expand-file-name mew-signature-file))
  )

;;;
;;; draft mime operation
;;;

(defun mew-draft-make-mime ()
  (interactive)
  (if (null (mew-field-get-value "Content-Type:"))
      (if (mew-multi-p)
	  (mew-draft-make-multi)
	(mew-draft-make-single))
    (ding)
    (message "Content-Type: already exists!")
    )
  )

(defun mew-draft-make-single ()
  (let ((case-fold-search t)
	(charset (mew-draft-charset-guess-letter)))
    (goto-char (marker-position mew-draft-buffer-header)) ;; E?
    ;; the beginning of "----"
    (forward-line 1)
    ;; cursor is just after ----
    (mew-field-insert-last "Content-Type:"
			   (format "Text/Plain; charset=%s" charset))
    ;; cursor is just after ----
    (if (mew-charset-7bit-p charset)
	(forward-line -2) ;; go to content-type
      (save-excursion
	(mew-mime-encode-region (point) (point-max) "quoted-printable")
	)
      (mew-field-insert-last "Content-Transfer-Encoding:"
			     "quoted-printable")
      (forward-line -3) ;; go to content-type
      )
    ))

(defun mew-draft-make-multi ()
  (goto-char (marker-position mew-draft-buffer-multipart))
  (delete-region (point) (point-max))
  ;; disable marker
  (set-marker mew-draft-buffer-multipart nil)
  (setq mew-draft-buffer-multipart nil)
  ;;
  (let ((boundary (mew-boundary-get))
	(case-fold-search t)
	(syntax mew-multi-syntax)
	(charset nil))
    (mew-field-insert-last "Content-Type:" 
			    (concat 
			     (mew-multi-get-type (car syntax))
			     ";\n\tboundary=\"" boundary "\""))
    ;; body stuff
    (setq syntax (cdr syntax)) ;; xxx skip dir name
    (goto-char (marker-position mew-draft-buffer-header))
    (forward-line)
    ;; cursor is just after "----"
    ;; see if cover page is empty or not
    (while (and (looking-at "^$") (not (eobp)))
      (forward-line))
    (if (eobp)
	(progn
	  ;; cover page is empty
	  (goto-char (marker-position mew-draft-buffer-header))
	  (end-of-line)
	  ;; cursor is on the end of "----" since boundary is
	  ;; preceded by CRLF
	  (delete-region (point) (point-max))
	  )
      ;; cover page exists
      (goto-char (marker-position mew-draft-buffer-header))
      (forward-line)
      ;; cursor is just after "----"
      (insert (concat "--" boundary "\n")) 
      ;; first boundary, no preceding CRLF is required
      (setq charset (mew-draft-charset-guess-region (point) (point-max))) 
      (mew-field-insert-here "Content-Type:" 
			     (format "Text/Plain; charset=%s" charset))
      (if (not (equal (mew-multi-get-desc (car syntax)) ""))
	  (mew-field-insert-here
	   "Content-Description:"
	   (concat "\""
		   (mew-header-encode-string
		    (mew-multi-get-desc (car syntax)) "B")
		   "\"")))
      (if (mew-charset-7bit-p charset)
	  (insert "\n") ;; demiliter for content header and content body
	(mew-field-insert-here "Content-Transfer-Encoding:"
			       "quoted-printable")
	(insert "\n")
	;; demiliter for content header and content body
	(mew-mime-encode-region (point) (point-max) "quoted-printable")
	)
      (goto-char (point-max))
      )
    ;; if cover page exist, cursor on after "--boundary--\n".
    ;; if not, cursor after "----" that is end of buffer.
    (let* ((fullname (mew-expand-file-name
		      (mew-folder-to-dir (mew-draft-to-mime (buffer-name)))))
	   (syntax (cdr (cdr mew-multi-syntax)))) ;; dir and 00CoverPage
      (while syntax
	(insert (concat "\n--" boundary "\n"))
	(if (listp (car (car syntax)))
	    (mew-draft-make-multipart (car syntax) fullname 2)
	  (mew-insert-file
	   ;; file
	   (mew-expand-file-name 
	    (mew-multi-get-name (car syntax)) fullname)
	   ;; type
	   (mew-multi-get-type (car syntax))
	   ;; description
	   (mew-header-encode-string (mew-multi-get-desc (car syntax)) "B")
	   ))
	(setq syntax (cdr syntax)) ;; xxx
	)
      )
    (insert (concat "\n--" boundary "--\n")) ;; last boundary
    )
  ;; move cursor to content-type
  (goto-char (point-min))
  (re-search-forward "^Content-Type:")
  (beginning-of-line)
  )

(defun mew-draft-make-multipart (syntax path depth)
  (let ((boundary (mew-boundary-get
		   (format "BOUNDARY%s" (int-to-string depth))))
	(fullname))
    (mew-field-insert-here "Content-Type:" 
			   (concat 
			    (mew-multi-get-type (car syntax))
			    ";\n\tboundary=\"" boundary "\""))
    ;; demiliter is not necessary
    (setq fullname
	  (mew-expand-file-name (mew-multi-get-name (car syntax)) path))
    (setq syntax (cdr syntax))
    (while syntax
      (insert (concat "\n--" boundary "\n"))
      (if (listp (car (car syntax)))
	  (mew-draft-make-multipart (car syntax) fullname (1+ depth))
	(mew-insert-file
	 ;; file
	 (mew-expand-file-name 
	  (mew-multi-get-name (car syntax)) fullname)
	 ;; type
	 (mew-multi-get-type (car syntax))
	 ;; description
	 (mew-header-encode-string (mew-multi-get-desc (car syntax)) "B")
	 )
	)
      (setq syntax (cdr syntax)) ;; xxx
      )
    (insert (concat "\n--" boundary "--\n"))
    ))

(defun mew-insert-file (file type desc)
  (let ((encoding 
	 (mew-file-encoding (mew-content-attr type mew-mime-content-type))))
    (if (null encoding)
	(cond 
	 ((string= (downcase type) "text/plain")
	  (let (here charset)
	    (setq here (point))
	    (insert "\n") 
	    ;; xxxx
	    ;; maybe *noconv* is right
	    (let ((file-coding-system-for-read
		   (if mew-mule-p mew-mule-charset-local))) 
	      (insert-file-contents file)
	      )
	    (setq charset 
		  (mew-draft-charset-guess-region (point) (point-max)))
	    ;; xxxx this code doesn't work well for Mule, I guess
	    (if (not (mew-charset-7bit-p charset))
		(save-excursion
		  (mew-mime-encode-region (point)
					  (point-max) "quoted-printable")))
	    (goto-char here)
	    (insert 
	     (format "Content-Type: %s; charset=%s\n" type charset))
	    (if (not (equal desc ""))
		(mew-field-insert-here "Content-Description:" 
				       (concat "\"" desc "\"")))
	    ))
	 ((string= (downcase type) "message/external-body")
	  (insert-file-contents file))
	 (t
	  (mew-field-insert-here "Content-Type:" type)
	  (if (not (equal desc ""))
	      (mew-field-insert-here "Content-Description:" 
				     (concat "\"" desc "\"")))
	  (insert "\n")
	  (insert-file-contents file))
	 )
      (save-excursion
	(set-buffer mew-buffer-tmp)
	(widen)
	(let ((file-coding-system-for-read
	       (and mew-mule-p *noconv*)) ;; Mule
	      (buffer-read-only nil)
	      (mc-flag nil))
	  (erase-buffer)
	  (insert-file-contents file)
	  (mew-mime-encode-region (point-min) (point-max) encoding)
	  ))
      (mew-field-insert-here "Content-Type:" type)
      (if (not (equal desc ""))
	  (mew-field-insert-here "Content-Description:" 
				 (concat "\"" desc "\"")))
      (mew-field-insert-here "Content-Transfer-Encoding:" encoding)
      (insert "\n")
      (insert-buffer mew-buffer-tmp)
      )
    (goto-char (point-max))
    ))

(defun mew-mime-encode-region (beg end encode)
  (let ((option nil)
	(selective-display nil)
	(mc-flag nil))
    (setq encode (downcase encode))
    (cond ((string= encode "quoted-printable") 
	   (setq option mew-prog-mime-encode-quoted))
	  ((string= encode "base64")
	   (setq option mew-prog-mime-encode-base64))
	  )
    (if (null option)
	();; maybe 7bit encoding
      (message "MIME encoding ...")
      (apply 'call-process-region beg end
	     mew-prog-mime-encode t t nil option)
      (message "MIME encoding ... done")
      )
    ))


(defun mew-draft-charset-guess-letter ()
  (interactive)
  (mew-draft-charset-guess-region 
   (if (markerp mew-draft-buffer-header)
       (marker-position  mew-draft-buffer-header)
     (point-min))
   (point-max)
   ))

(defun mew-draft-charset-guess-region (beg end)
  (interactive "r")
  "Guess minimum character set name."
  (cond 
   (mew-mule-p
    (let ((lc (car (find-charset-region beg end))))
      (if (null lc)
	  "us-ascii"
	(mew-mule-lc-content (mew-mule-lc-attr lc mew-mule-character-set))
	)))
   (mew-emacs19-p
    (save-excursion
      (goto-char beg)
      (if (re-search-forward "[\200-\377]" end t)
	  "iso-8859-1"
	"us-ascii")))
   (t "us-ascii") ;; Emacs
   ))

;;;
;;; Draft multipart stuff
;;;

;; 3 for mark
;; 1 for space
;; 4 for number
;; 16 for filename (chopped)
;; 1 for space
;; 26 for type (chopped)
;; 1 for space
;; 26 + 2 for despcription (chopped)
;; chopped ".."
;;
;;---- multipart --
;;    0   1/               Multipart/Mixed           
;;SE  1     00CoverPage      Text/Plain              "Hello World"
;;se  2     file2            Text/Plain              "0123456789012345678901"
;;    3.0   dir3/            Multipart/Mixed
;;    3.1     822              Message/Rfc822        "0123456789012345678901"
;;C   4     file4            Text/Plain
;;---- multipart  ----

;(setq syntax
;      '(("1/" "Multipart/Mixed" "" ())
;	("00CoverPage" "Text/Plain" "Hello World" '(SE))
;	("file2" "Text/Plain" "0123456789012345678901" '(se))
;	(("dir3/" "Multipart/Mixed" "" nil)
;	 ("822" "Message/Rfc822" "0123456789012345678901xxxx"))
;	("file4" "Text/Plain" "" nil)
;	)
;      )


;;
;; prepare multipart region
;;

(defvar mew-multi-initial-syntax
  '(("1/" "Multipart/Mixed" "" ())
    ("00CoverPage" "Text/Plain" "Cover Page" ()))
  )

(defun mew-draft-prepare-multipart ()
  (interactive)
  (save-excursion
    (let* ((fullname (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (cp (concat fullname "/00CoverPage"))
	   (dir (file-name-nondirectory fullname))
	   (path (file-name-directory fullname)))
      (if (mew-multi-p)
	  ()
	(mew-multi-insert-boundary)
	(mew-make-directory fullname)
	(if (null (file-exists-p cp))
	    (call-process "ln" nil nil nil
			   "-s"
			    (mew-expand-file-name (buffer-name))
			    cp))
	(setq mew-multi-syntax
	      (mew-multi-ls path dir))
	(mew-multi-print "" 0 mew-multi-syntax)
	))
    )
  (goto-char (point-max))
  (beginning-of-line)
  )


(defun mew-multi-insert-boundary ()
   (goto-char (point-max))
   (if (null (bolp))(insert "\n"))
   (if (markerp mew-draft-buffer-multipart)
       (set-marker mew-draft-buffer-multipart (point))
     (setq mew-draft-buffer-multipart (point-marker)))
   (insert mew-draft-multipart-begin)
   (insert "\n")
   (insert mew-draft-multipart-end)
   (beginning-of-line)
   )

;;
;; print stuff
;;

(defun mew-multi-ls (path dir)
  (save-excursion
    (set-buffer mew-buffer-tmp)
    (widen)
    (goto-char (point-min))
    (let ((buffer-read-only nil))
      (erase-buffer)
      (mew-multi-lsdir (expand-file-name dir path))
      (goto-char (point-min))
      (mew-multi-ls-list (concat dir "/" ))
      )
    ))

(defun mew-multi-lsdir (cwd)
  (save-excursion
    (call-process "sh" nil t nil 
		  "-c"
		  (format "ls -1F %s | sort -n" cwd))
    ;; "ls -1fF" acts differently in each machine :<
    (insert "\n") ;; sentinel
    )
  (while (not (looking-at "^$"))
    (if (looking-at "^\\./$\\|^\\.\\./$\\|.*~$\\|#")
	(let ((begin (point)))
	  (forward-line)
	  (delete-region begin (point)))
      (if (looking-at ".*/$")
	  (progn
	    (forward-line)
	    (mew-multi-lsdir (expand-file-name (mew-match 0) cwd))
	    (forward-line -1)
	    ))
      (if (looking-at "^\\(.*\\)\\*$")
	  (replace-match "\\1"))
      (if (looking-at "\\(.*\\)@$")
	  (replace-match "\\1"))
      (forward-line))
    )
  (forward-line)
  )

(defun mew-multi-ls-list (tag)
  (let ((files nil))
    (while (not (looking-at "^$"))
      (setq files (append files (list (mew-multi-ls-listdir)))))
    (forward-line)
    (cons (list tag "Multipart/Mixed" "" nil) files)
    ))

(defun mew-multi-ls-listdir ()
  (if (not (looking-at ".*/$"))
    (let ((start (point))
	  (ret nil))
      (forward-line)
      (setq ret (buffer-substring start (1- (point))))
      (list ret ;; filename
	    (mew-file-content (mew-file-attr ret mew-mime-content-type))
	    ""  ;; description
	    nil ;; marks
	    ))
    (forward-line)
    (mew-multi-ls-list (mew-match 0))
    )
  )

(defun mew-multi-print (number level syntax)
  (let ((buffer-read-only nil)
	(count 0))
    (while syntax
      (if (listp (car (car syntax)))
	  (mew-multi-print (concat number (int-to-string count) ".")
			   (1+ level)
			   (car syntax))
	(insert 
	 (mew-multi-format (concat number (int-to-string count))
			   level 
			   (mew-multi-get-name (car syntax))
			   (mew-multi-get-type (car syntax))
			   (mew-multi-get-desc (car syntax)))
	 ))
      (setq syntax (cdr syntax))
      (setq count (1+ count))
      )
    ))
	  

(defun mew-multi-format (number level name type desc)
  ;; level starts 1
  (let* ((marks "    ") (space " ") 
	 (quote "\"") (space-char 32)
	 (cnt "..") (lcnt (length cnt))
	 (ln 16) (lnc (- ln lcnt))
	 (lt 22) (ltc (- lt lcnt))
	 (ld 24) (ldc (- ld lcnt))
	 (dirp (string-match "/$" name)))
    (concat
     marks
     number
     space
     (make-string (if dirp 2 4) space-char)

     (if (< ln (length name))
	 (concat (substring name 0 lnc) cnt)
       (concat name (make-string (- ln (length name)) space-char)))
     space

     (let* ((tmp (* 2 (if dirp (- level 2) (1- level))))
	    (vlt (- lt tmp))
	    (vltc (- ltc tmp))
	    (ltype (length type)))
       (if (< vlt ltype)
	   (concat (substring type 0 vltc) cnt)
	 (concat type (make-string (- vlt ltype) space-char))))
     space

     (if (equal 0 (length desc))
	 ""
	(if (< ld (length desc))
	    (concat quote (substring desc 0 ldc) cnt quote)
	  (concat quote desc quote)))
     "\n"
     )
    ))
  

(defun mew-multi-display ()
  (interactive)
  (let ((end nil))
    (cond
   ((mew-multi-p)
    (goto-char (point-max))
    (re-search-backward (concat "^" mew-draft-multipart-end "$"))
    (setq end (point))
    (re-search-backward (concat "^" mew-draft-multipart-begin "$"))   
    (forward-line)
    (delete-region (point) end)
    (mew-multi-print "" 1 mew-multi-syntax)
    ))
  ))

;;
;; some goodies
;;

(defmacro mew-multi-p ()
  (` (markerp mew-draft-buffer-multipart))
  )


(defun mew-multi-part-number ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at "^....\\([\.0-9]+\\) *")
	(mew-match 1)
      nil
      )
    ))

(defun mew-multi-line ()
  (save-excursion
    (beginning-of-line)
    (count-lines
     (marker-position mew-draft-buffer-multipart)
     (point))
    ))



(defmacro mew-multi-get-name (list)
 (` (car (, list))))

(defmacro mew-multi-get-type (list)
 (` (car (cdr (, list)))))

(defmacro mew-multi-get-desc (list)
 (` (car (cdr (cdr (, list))))))

(defmacro mew-multi-get-mark (list)
 (` (car (cdr (cdr (cdr (, list)))))))

(defmacro mew-multi-set-name (name list)
  (` (mew-setnth 0 (, name) (, list))))

(defmacro mew-multi-set-type (type list)
  (` (mew-setnth 1 (, type) (, list))))

(defmacro mew-multi-set-desc (desc list)
  (` (mew-setnth 2 (, desc) (, list))))

(defmacro mew-multi-set-mark (mark list)
  (` (mew-setnth 3 (, mark) (, list))))


(defun mew-multi-split-number (num-str &optional plus1)
  (let ((start 0)
	(end 0)
	(nums nil))
    (while (setq end (string-match "\\." num-str start))
      (setq nums (cons (string-to-int (substring num-str start end)) nums))
      (setq start (1+ end))
      )
    (if (null plus1)
	(reverse (cons (string-to-int (substring num-str start nil)) nums))
      (reverse (cons (1+ (string-to-int (substring num-str start nil))) nums))
      )
    ))

(defun mew-setnth (num value list)
  (let ((n (1- num))
	(ret nil))
    (while (< -1 n)
      (setq ret (cons (nth n list) ret))
      (setq n (1- n))
      )
    (append ret (list value) (nthcdr (1+ num) list))
    ))


;; ignore "1/"
(defun mew-multi-expand-path (nums syntax)
  (let ((path ""))
    (while (cdr nums)
      (setq syntax (nth (car nums) syntax))
      (setq path (concat path (mew-multi-get-name (car syntax))))
      (setq nums (cdr nums))
      )
    path
    ))


;;
;; Get the entry
;;

(defun mew-multi-get (nums syntax)
  "Get the entry for NUMS from SYNTAX.
e.g. entry == (name type desc mark)"
  (while nums
    (setq syntax (nth (car nums) syntax))
    (setq nums (cdr nums))
    )
  syntax
  )

;;
;; Set entry, entry should be prepared before call this.
;;

(defun mew-multi-set (nums entry syntax)
  (if (null nums)
      entry
    (let ((n (1- (car nums)))
	  (ret nil))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret
	      (list (mew-multi-set (cdr nums)
				   entry
				   (nth (car nums) syntax)))
	      (nthcdr (1+ (car nums)) syntax))
      )
    ))

;;
;; Insert entry
;;


(defun mew-multi-insert (nums entry syntax)
  (let ((n (1- (car nums)))
	(ret nil))
    (cond 
     ;; insert diretory
     ((and (null (cdr (cdr nums))) (equal (car (cdr nums)) 0))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (list (list entry)) (nthcdr (car nums) syntax))
      )
     ;; insert file
     ((null (cdr nums))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (list entry) (nthcdr (car nums) syntax))
      )
     (t
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret
	      (list (mew-multi-insert (cdr nums)
				      entry
				      (nth (car nums) syntax)))
	      (nthcdr (1+ (car nums)) syntax))
      )
     )
    ))

;;
;; remove
;;

(defun mew-multi-remove (nums syntax)
  (let ((n (1- (car nums)))
	(ret nil))
    (cond 
     ;; insert diretory
     ((and (null (cdr (cdr nums))) (equal (car (cdr nums)) 0))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (nthcdr (1+ (car nums)) syntax))
      )
     ;; insert file
     ((null (cdr nums))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (nthcdr (1+ (car nums)) syntax))
      )
     (t
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret
	      (list (mew-multi-remove (cdr nums)
				      (nth (car nums) syntax)))
	      (nthcdr (1+ (car nums)) syntax))
      )
     )
    ))


;;
;; commands
;;

(defun mew-multi-nums (&optional previous plus1)
  (save-excursion
    (if previous (forward-line -1))
    (if (mew-multi-part-number)
	(mew-multi-split-number (mew-multi-part-number) plus1)
      nil)
    ))

(defun mew-multi-copy ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't copy here")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (fromfile (mew-input-filename "Copy from : " "~"))
	   (tofile (mew-input-string "Copy to %s(%s): " 
				     subdir
				     (file-name-nondirectory fromfile)))
	   (efile nil))
      (setq efile (if (equal subdir "") tofile (concat subdir tofile)))
      (if (and (file-exists-p (expand-file-name efile mimedir))
	       (null (mew-y-or-n-p (message "File %s exists. Overwrite it? " 
					    efile))))
	  ()
	(save-excursion
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (widen)
	  (erase-buffer)
	  (let ((mc-flag nil)
		(file-coding-system-for-read (if mew-mule-p *noconv*))
		(file-coding-system (if mew-mule-p *noconv*)))
	    (insert-file-contents fromfile)
	    (write-region (point-min) (point-max)
			  (expand-file-name efile mimedir))
	    ))
;;	(call-process "cp" nil nil nil
;;		      fromfile 
;;		      (expand-file-name efile mimedir))
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       nums
	       (list tofile
		     (mew-file-content 
		      (mew-file-attr tofile mew-mime-content-type))
		     ""
		     nil)
	       mew-multi-syntax))
	(mew-multi-display)
	)
      )
    ))
  

(defun mew-multi-link ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't link here.")
    (let* ((doit t)
	   (nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (fromfile (mew-input-filename "Link from : " "~"))
	   (tofile (mew-input-string "link to %s(%s): " 
				     subdir
				     (file-name-nondirectory fromfile)))
	   (efile nil))
      (setq efile (if (equal subdir "") tofile (concat subdir tofile)))
      (if (file-exists-p (expand-file-name efile mimedir))
	  (if (null (mew-y-or-n-p (message "File %s exists. Overwrite it? "
					   efile)))
	      (setq doit nil)
	    (call-process "rm" nil nil nil 
			  (expand-file-name efile mimedir))))
      (if (null doit)
	  ()
	(call-process "ln" nil nil nil "-s" fromfile 
		      (expand-file-name efile mimedir))
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       nums
	       (list tofile
		     (mew-file-content 
		      (mew-file-attr tofile mew-mime-content-type))
		     ""
		     nil)
	       mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-rename ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't rename here.")
    (if (null (mew-multi-part-number))
	(message "No part here.")
      (let* ((insert-default-directory nil)
	     (nums (mew-multi-nums))
	     (subdir (mew-multi-expand-path nums mew-multi-syntax))
	     (mimedir (mew-expand-file-name
		       (mew-draft-to-mime (buffer-name))))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (fromfile (mew-multi-get-name entry))
	     (efrom (concat subdir fromfile))
	     (tofile nil)
	     (efile nil))
	(setq tofile
	      (read-file-name 
	       (format "From %s%s to (%s): " subdir fromfile fromfile)
	       (expand-file-name subdir mimedir)
	       nil nil))
	(if (equal tofile "") (setq tofile fromfile))
	(setq efile (concat subdir tofile))
	(if (and (file-exists-p (expand-file-name efile mimedir))
		 (null (mew-y-or-n-p
			(message "File %s exists. Overwrite it? " efile))))
	    ()
	  (call-process "mv" nil nil nil 
			(expand-file-name efrom mimedir)
			(expand-file-name efile mimedir))
	(setq mew-multi-syntax
	      (mew-multi-set nums
			     (mew-multi-set-name tofile entry)
			     mew-multi-syntax))
	(mew-multi-display)
	))
      )))
	

(defun mew-multi-delete ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't delete here.")
    (if (null (mew-multi-part-number))
	(message "No part here.")
      (let* ((nums (mew-multi-nums))
	     (subdir (mew-multi-expand-path nums mew-multi-syntax))
	     (mimedir (mew-expand-file-name
		       (mew-draft-to-mime (buffer-name))))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (name (mew-multi-get-name entry))
	     (ename nil)
	     (doit nil))
	(cond
	 ((string-match ".*/$" name)
	  (setq ename subdir)
	  (if (mew-y-or-n-p
	       (format "Delete %s with contained files? " ename))
	      (setq doit t)
	    (setq doit nil)))
	 (t 
	  (setq ename (if (equal subdir "") name (concat subdir name)))
	  (if (mew-y-or-n-p (format "Delete %s? " ename))
	      (setq doit t)
	    (setq doit nil)))
	 )
	(if (null doit)
	    ()
	  (message "Deleting %s ... " ename)
	  (call-process "rm" nil nil nil
			"-rf"
			(expand-file-name ename mimedir))
	  (message "Deleting %s ... done" ename)
	  (setq mew-multi-syntax (mew-multi-remove nums mew-multi-syntax))
	  (mew-multi-display)
	  )
	))
    ))



;; subpart
(defun mew-multi-multipart ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't creat sub-multipart here.")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (insert-default-directory nil)
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (dir (read-file-name 
		 (format "Directory name (%s): " subdir) mimedir nil nil));; xxx
	   (edir nil))
      ;; mimedir{/subdir}/dir
      ;; edir = {/subdir}/dir
      (setq edir (if (equal subdir "") dir (concat subdir dir)))
      (if (and (file-exists-p (expand-file-name edir mimedir))
	       (null (mew-y-or-n-p
		      (message 
		       "File or directory %s exists. Overwrite it? " 
		       edir))))
	  ()
	(mew-make-directory (expand-file-name edir mimedir))
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       (append nums (list 0)) ;; directory
	       (list 
		(if (string-match "/$" dir) dir (concat dir "/"))
		"Multipart/Mixed"
		""
		nil)
	       mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-find-file ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't find a file here.")
    (if (null (mew-multi-part-number))
	(message "No part here.")
      (let* ((nums (mew-multi-nums))
	     (subdir (mew-multi-expand-path nums mew-multi-syntax))
	     (mimedir (mew-expand-file-name
		       (mew-draft-to-mime (buffer-name))))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (file (mew-multi-get-name entry))
	     (efile nil))
	(setq efile (if (equal subdir "") file (concat subdir file)))
	(find-file (expand-file-name efile mimedir))
	;; buffer switched
	(local-set-key "\C-c\C-q" 'mew-kill-buffer)
	)
      )))

(defun mew-multi-find-mew-file ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't find a new file here.")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name
		     (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Open file as %s(%s): " 
				    subdir
				    "a.txt"))
	   (efile nil))
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (setq mew-multi-syntax
	    (mew-multi-insert 
	     nums
	     (list file "Text/Plain" "" nil)
	     mew-multi-syntax))
      (mew-multi-display)
      (find-file (expand-file-name efile mimedir))
      ;; buffer switched
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )
    ))

(defun mew-multi-description ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't tag description here")
    (if (null (mew-multi-part-number))
	(message "Can't tag description here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (description)
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (name (mew-multi-get-name entry)))
	(if (string-match "/$" name)
	    (message "Can't tag description to Multipart.")
	  (setq description
		(read-string (format "Description (%s): " name) ""))
	  (if (equal description "") (setq description name))
	  (setq mew-multi-syntax
		(mew-multi-set nums
			       (mew-multi-set-desc description entry)
			       mew-multi-syntax))
	  (mew-multi-display)
	  ))
      )
    ))

(defmacro mew-multi-directory-p (nums)
  (` (equal 0 (nth (1- (length (, nums))) (, nums)))))

(defun mew-multi-type ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't change Content-Type: here")
    (if (null (mew-multi-part-number))
	(message "Can't change Content-Type: here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry)))
	(if (mew-multi-directory-p nums)
	    (setq type
		  (mew-input-type
		   name mew-mime-content-type-multipart-list type))
	  (setq type (mew-input-type name mew-mime-content-type-list type)))
	(setq mew-multi-syntax
	      (mew-multi-set nums
			     (mew-multi-set-type type entry)
			     mew-multi-syntax))
	(mew-multi-display)
	)
      )
    ))

(defun mew-multi-audio ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't insert voice here.")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Create as %s(%s): " 
				    subdir
				    "a.au"))
	   (efile nil)
	   (buf nil))
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (save-excursion
	(set-buffer mew-buffer-tmp)
	(erase-buffer)
	(condition-case errorcode
	    (let ((selective-display nil)
		  (mc-flag nil)
		  (call-process-hook nil)
		  (default-process-coding-system 
		    (if mew-mule-p (cons *noconv* *noconv*))))
	      (while (not (mew-y-or-n-p "Are you ready? ")) ())
	      (message "Type C-g to finish recording...")
	      (apply 'call-process mew-prog-audio)
	      )
	  (quit (message "Type C-g to finish recording... done.")))
	(let ((mc-flag nil)
	      (file-coding-system (if mew-mule-p *noconv* nil)))
	  (write-region (point-min) (point-max)
			(expand-file-name efile mimedir)))
	)
      (kill-buffer mew-buffer-tmp)
      (setq mew-multi-syntax
	    (mew-multi-insert 
	     nums
	     (list file "Audio/Basic" "" nil)
	     mew-multi-syntax))
      (mew-multi-display)
      )))

;;;
;;; Demo
;;;

(defun mew-demo (list)
  ;; 9 is "/\\ - \\/"
  (let* ((wl (window-width))
	 (pl (/ (- wl 9) 2))
	 (pre (make-string pl 32))
	 (suf (make-string (1- (- (- wl pl) 9)) 32)))
    (save-window-excursion
      (select-window (get-buffer-window (get-buffer mew-buffer-hello)))
      (while list
	(mew-demo-print (car list) pre suf)
	(mew-demo-loop)
	(setq list (cdr list)))
      )))


(setq mew-demo-string
'(
 "/\\\\ - \\\\/"

 "-\\\\ - \\\\/" "\\\\\\ - \\\\/" "|\\\\ - \\\\/" "/\\\\ - \\\\/"
 "-\\\\ - \\\\/" "\\\\\\ - \\\\/" "|\\\\ - \\\\/" "/\\\\ - \\\\/"

 "/|\\ - \\\\/"  "//\\ - \\\\/" "/-\\ - \\\\/" "/\\\\ - \\\\/"
 "/|\\ - \\\\/"  "//\\ - \\\\/"  "/-\\ - \\\\/" "/\\\\ - \\\\/"

 "/\\| - \\\\/" "/\\/ - \\\\/" "/\\- - \\\\/" "/\\\\ - \\\\/" 
 "/\\| - \\\\/" "/\\/ - \\\\/" "/\\- - \\\\/" "/\\\\ - \\\\/"

 "/\\\\ - |\\/" "/\\\\ - /\\/" "/\\\\ - -\\/" "/\\\\ - \\\\/"
 "/\\\\ - |\\/" "/\\\\ - /\\/" "/\\\\ - -\\/" "/\\\\ - \\\\/"

 "/\\\\ - \\|/" "/\\\\ - \\//" "/\\\\ - \\-/" "/\\\\ - \\\\/"
 "/\\\\ - \\|/" "/\\\\ - \\//" "/\\\\ - \\-/" "/\\\\ - \\\\/"

 "/\\\\ - \\\\-" "/\\\\ - \\\\\\" "/\\\\ - \\\\|" "/\\\\ - \\\\/"
 "/\\\\ - \\\\-" "/\\\\ - \\\\\\" "/\\\\ - \\\\|" "/\\\\ - \\\\/"
 )
)


(defun mew-delete-line ()
  (beginning-of-line)
  (let ((start (point)))
    (forward-line)
    (delete-region start (point))
    ))

(defun mew-demo-print (string prefix suffix)
  (goto-char (point-max))
  (mew-delete-line)
  (insert (concat prefix string suffix))
  )


(cond
 (mew-emacs18-p
  (defun mew-demo-loop ()
    (let ((i 30))
      (while (> i 1)
	(sit-for 0)
	(message "")
	(setq i (1- i))))))
 (t
  (defun mew-demo-loop ()
    (sit-for (string-to-int "0.02"))))
 )


;;;
;;;
;;;
;;;

(defun mew-draft-in-header-p ()
  (if (markerp mew-draft-buffer-header)
      (<= (point) (marker-position mew-draft-buffer-header))
    nil)
  )

(defun mew-draft-in-multipart-p ()
  (if (markerp mew-draft-buffer-multipart)
      (>= (point) (marker-position mew-draft-buffer-multipart))
    nil)
  )

(defun mew-command-key-sequence (key command)
  "Remove initial sequence of universal arguments from KEY and return
the real key sequence that invoked COMMAND.  If COMMAND is invoked by
`execute-extended-command', nil is returned."
  (while (not (or (zerop (length key))
                  (eq command (key-binding key))))
    (setq key (substring key 1)))
  (and (not (zerop (length key))) key))

(defun mew-draft-keyswitch ()
  (interactive)
  (let ((func) (key (this-command-keys)))
    (cond
     ((and (markerp mew-draft-buffer-multipart) (mew-draft-in-multipart-p))
      (setq func (lookup-key mew-draft-multipart-map key)))
     ((and (markerp mew-draft-buffer-header) (mew-draft-in-header-p))
      (setq func (lookup-key mew-draft-header-map key)))
     (t 
      (setq func (lookup-key mew-draft-body-map key)) ;; \e\t -> 1
      (if (not (symbolp func)) (setq func nil)))
     )
    (if (null func)
	(setq func (lookup-key (current-global-map)
			       (mew-command-key-sequence key 'mew-draft-keyswitch))))
    (if func
	(call-interactively func)
      (insert key)) ;; for safety
    ))

(defun mew-draft-null-function ()
  (interactive)
  ())

(defun mew-draft-header-alias ()
  (interactive)
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (mew-draft-on-to-p))
	(tab-to-tab-stop) ;; default key binding
      (if (null (setq word (mew-delete-backward-char)))
	  (tab-to-tab-stop)
	(setq comp (try-completion word mew-alias-alist))
	(setq all (all-completions word mew-alias-alist))
	(setq len (length word))
	(cond
	 ((eq comp t)
	  (insert (cdr (assoc word mew-alias-alist)))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (if mew-draft-window-config
	      (progn
		(set-window-configuration mew-draft-window-config)
		(setq mew-draft-window-config nil)
		))
;;	  (message "Expanded")
	  )
	 ((and (equal (aref word (1- len)) ?@)
	       (assoc (substring word 0 (1- len)) mew-alias-alist))
	  (insert (cdr (assoc (substring word 0 (1- len)) mew-alias-alist)))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (if mew-draft-window-config
	      (progn
		(set-window-configuration mew-draft-window-config)
		(setq mew-draft-window-config nil)
		))
;;	  (message "Expanded")
	  )
	 ((stringp comp)
	  (insert comp)
	  (if (> (length all) 1)
	      (progn
		(or mew-draft-window-config
		    (setq mew-draft-window-config
			  (current-window-configuration)))
		(with-output-to-temp-buffer mew-buffer-completions
		  (display-completion-list all))
		)
	    )
	  )
	 (t (insert word)
	    (message "No match alias"))
	 )
	))
    ))

(defun mew-draft-on-to-p ()
  (save-excursion
    (beginning-of-line)
    (while (and (< (point-min) (point))	(looking-at "[ \t]"))
      (forward-line -1)
      )
    (looking-at "To:\\|Cc:\\|Bcc:")
    ))

(defun mew-delete-backward-char ()
  (interactive)
  (let ((case-fold-search t)
	(start nil)
	(end (point))
	(regex (concat "[^" mew-address-separator "]")))
    (save-excursion
      (while (and (not (bobp))
		  (string-match regex
				(buffer-substring (1- (point)) (point))))
	(forward-char -1))
      (setq start (point))
      (if (= start end)
	  nil
	(prog1
	    (buffer-substring start end)
	  (delete-region start end)))
      )
    ))

(defun mew-draft-header-domain ()
  (interactive)
  (let ((word nil))
    (if (null (mew-draft-on-to-p))
	(lisp-complete-symbol) ;; default key binding
      (setq word (mew-delete-backward-char-at))
      ;; word is strings between @ and cursor
      (cond
       ((equal word nil) ;; @ doesn't exist.
	(if (null mew-mail-domain-list)
	    ()
	  (insert "@")
	  (or mew-draft-window-config
	      (setq mew-draft-window-config (current-window-configuration)))
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list mew-mail-domain-list))))
       ((equal word t) ;; just after @
	(if mew-mail-domain-list
	    (progn
	      (or mew-draft-window-config
		  (setq mew-draft-window-config
			(current-window-configuration)))
	      (with-output-to-temp-buffer mew-buffer-completions
		(display-completion-list mew-mail-domain-list))
	      )
	  )
	)
       (t
	(let ((comp nil) (all nil) (len nil)
	      (dalist (mew-slide-pair mew-mail-domain-list)))
	  (setq comp (try-completion word dalist))
	  (setq all (all-completions word dalist))
	  (setq len (length word))
	  (cond
	   ;; already completed
	   ((eq comp t) 
	    (insert (cdr (assoc word dalist))) ;; next candidate
	    (and (get-buffer mew-buffer-completions)
		 (kill-buffer mew-buffer-completions))
	    (if mew-draft-window-config
		(progn
		  (set-window-configuration mew-draft-window-config)
		  (setq mew-draft-window-config nil)
		  ))
	    )
	   ;; just one candidate
	   ((equal 1 (length all))
	    (insert (car all))
	    (and (get-buffer mew-buffer-completions)
		 (kill-buffer mew-buffer-completions))
	    (if mew-draft-window-config
		(progn
		  (set-window-configuration mew-draft-window-config)
		  (setq mew-draft-window-config nil)
		  ))
	    (message "Sole completion"))
	   ;; one or more candidate
	   ((stringp comp)
	    (insert comp)
	    (if (> (length all) 1)
		(progn
		  (or mew-draft-window-config
		      (setq mew-draft-window-config
			    (current-window-configuration)))
		  (with-output-to-temp-buffer mew-buffer-completions
		    (display-completion-list all))
		  )
	      )
	    )
	   ;; no candidate
	   (t (insert word)
	      (message "No match alias"))
	   )
	  ))
       )
      )
    ))


(defun mew-delete-backward-char-at ()
  (interactive)
  (let ((case-fold-search t)
	(start nil)
	(end (point))
	(regex (concat "[^" mew-address-separator "]")))
    (save-excursion
      (while (and (not (bobp))
		  (string-match regex
				(buffer-substring (1- (point)) (point))))
	(forward-char -1))
      (if (null (re-search-forward "@" end t))
	  nil ;; @ doesn't exist.
	(setq start (point))
	(if (= start end)
	    t ;; just after @
	  (prog1
	      (buffer-substring start end)
	    (delete-region start end)))
	)
      )
    ))

(defun mew-slide-pair (x)
  (let ((ret nil)
	(len (length x))
	(first (car x)))
    (cond 
     ((eq x 0) nil)
     ((eq x 1) (cons first first))
     (t
      (while (cdr x)
	(setq ret (cons (cons (nth 0 x) (nth 1 x)) ret))
	(setq x (cdr x))
	)
      (setq ret (cons (cons (car x) first) ret))
      (reverse ret)
      )
     )
    )
  )

;;;
;;; mini buffer
;;;

(defun mew-temp-minibuffer-message (m)
  "A Lisp version of `temp_minibuffer_message' from minibuf.c."
  (if (fboundp 'temp-minibuffer-message)
      (temp-minibuffer-message m)
    (let ((savemax (point-max)))
      (save-excursion
	(goto-char (point-max))
	(insert (format " (%s)" m)))
      (let ((inhibit-quit t))
	(sit-for 2)
	(delete-region savemax (point-max))
	(if quit-flag
	    (setq quit-flag nil
		  unread-command-char 7))
	))
    ))


(defun mew-minibuffer-alias ()
  (interactive)
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop) ;; default key binding
      (setq comp (try-completion word mew-alias-alist))
      (setq all (all-completions word mew-alias-alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((and (equal (aref word (1- len)) ?@)
	     (assoc (substring word 0 (1- len)) mew-alias-alist))
	(insert (cdr (assoc (substring word 0 (1- len)) mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp)
	(and (> (length all) 1)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (mew-temp-minibuffer-message "No match alias"))
       )
      )
    ))


(defun mew-minibuffer-domain ()
  (interactive)
  (let ((word (mew-delete-backward-char-at)))
;;;;    (if (null (setq word (mew-delete-backward-char-at)))
;;;;	(lisp-complete-symbol) ;; default key binding)
    ;; word is strings between @ and cursor
    (cond
     ((equal word nil) ;; @ doesn't exist.
      (if (null mew-mail-domain-list)
	  ()
	(insert "@")
	(with-output-to-temp-buffer mew-buffer-completions
	  (display-completion-list mew-mail-domain-list))))
     ((equal word t) ;; just after @
      (if mew-mail-domain-list
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list mew-mail-domain-list)))
      )
     (t
      (let ((comp nil) (all nil) (len nil)
	    (dalist (mew-slide-pair mew-mail-domain-list)))
	(setq comp (try-completion word dalist))
	(setq all (all-completions word dalist))
	(setq len (length word))
	(cond
	 ;; already completed
	 ((eq comp t) 
	  (insert (cdr (assoc word dalist))) ;; next candidate
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  )
	 ;; just one candidate
	 ((equal 1 (length all))
	  (insert (car all))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (mew-temp-minibuffer-message "Sole completion")
	  )
	 ;; one or more candidate
	 ((stringp comp)
	  (insert comp)
	  (and (> (length all) 1)
	       (with-output-to-temp-buffer mew-buffer-completions
		 (display-completion-list all))))
	 ;; no candidate
	 (t (insert word)
	    (mew-temp-minibuffer-message "No match domain"))
	 ))
      )
     )
    ))

;;;
;;; MIME encoding and decoding
;;;

(defconst mew-base64-mask 63)
(defconst mew-base64-boundary1 26)
(defconst mew-base64-boundary2 52)
(defconst mew-base64-boundary3 62)
(defconst mew-base64-boundary4 63)

(defvar mew-encode-switch
  (list
   (cons "B" 'mew-base64-encode-string)
   (cons "Q" 'mew-quoted-printable-encode-string)
   )
  )

(defvar mew-decode-switch
  (list
   (cons "B" 'mew-base64-decode-string)
   (cons "Q" 'mew-quoted-printable-decode-string)
   )
  )

(defmacro mew-decode-func-attr (encode alist)
  (` (mew-assoc (, encode) (, alist) 0 t)))

(defmacro mew-decode-func-func (attr)
  (` (cdr (, attr))))

(defvar mew-decode-header
  (list
   (cons "To:" nil)
   (cons "Cc:" nil)
   (cons "From:" t)
   (cons "Subject" t)
   (cons "Content-Description:" t)
   )
  )

(defmacro mew-decode-attr (key alist)
  (` (mew-assoc (, key) (, alist) 0 t)))

(defmacro mew-decode-delp (attr)
  (` (cdr (, attr))))

;;
;; kick off routine
;;

(defun mew-header-mime-decode ()
  (interactive)
  (let ((case-fold-search t)
	(key   "")
	(start nil)
	(value nil)
	(attr  nil))
    (save-excursion
      (catch 'header
	(while (re-search-forward "^-*$\\|^[^ \t\n]*:")
	  (setq key (mew-match 0))
	  (setq start (match-end 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (if (setq attr (mew-decode-attr key mew-decode-header))
	      (progn
		(setq value (buffer-substring start (1- (point))))
		(delete-region start (point))
		(insert (mew-header-decode value (mew-decode-delp attr)))
		(insert "\n")
		))
	  ))
      )
    ))


(defun mew-header-decode (str &optional del-ret)
  (if del-ret
      (while (string-match "\n[ \t]*" str)
	(setq str (concat (substring str 0 (match-beginning 0))
			  (substring str (match-end 0) (length str)))
	      )
	)
    )
  (while (string-match "=\\?\\([^?]+\\)\\?\\(.\\)\\?\\([^?]+\\)\\?= ?" str)
    ;; last " ?" is for mhn. (I don't like it.)
    (let ((charset (mew-match 1 str))
	  (encode (mew-match 2 str))
	  (enstr (mew-match 3 str))
	  (head (substring str 0 (match-beginning 0)))
	  (tail (substring str (match-end 0) (length str)))
	  (destr ""))
      (setq destr 
	    (funcall 
	     (mew-decode-func-func
	      (mew-decode-func-attr encode mew-decode-switch))
	     enstr))
      (if mew-mule-p
	  (setq destr
		(code-convert-string
		 destr
		 (mew-mule-content-coding
		  (mew-mule-content-attr charset mew-mule-character-set))
		 *internal*
		 ))
	)
      (setq str (concat head destr tail))
      ))
  str
  )

;;
;; Base 64 encode
;;


(defun mew-base64-encode-string (str256)
  (let* ((len (length str256))
	 (pad (- 3 (% len 3)))
	 (count 0)
	 (ret ""))
    (setq str256 (concat str256 (make-string pad 0)))
    (while (< count len)
      (let ((char0 (aref str256 count))
	    (char1 (aref str256 (1+ count)))
	    (char2 (aref str256 (+ 2 count))))
	(setq ret
	      (concat
	       ret
	       (char-to-string (mew-base64-char64 (lsh char0 -2)))
	       (char-to-string
		(mew-base64-char64 (logior (lsh char0 4) (lsh char1 -4))))
	       (char-to-string
		(mew-base64-char64 (logior (lsh char1 2) (lsh char2 -6))))
	       (char-to-string (mew-base64-char64 char2))
	       )
	      )
	(setq count (+ count 3))
	))
    (concat (substring ret 0 (- 0 pad)) (make-string pad ?=))
    ))

	      
(defun mew-base64-char256 (ch64)
  (cond
   ((null ch64) 0)
   ((and (<= ?A ch64) (<= ch64 ?Z)) (- ch64 ?A))
   ((and (<= ?a ch64) (<= ch64 ?z)) (+ (- ch64 ?a) mew-base64-boundary1))
   ((and (<= ?0 ch64) (<= ch64 ?9)) (+ (- ch64 ?0) mew-base64-boundary2))
   ((= ch64 ?+) mew-base64-boundary3)
   ((= ch64 ?/) mew-base64-boundary4)
   ((= ch64 ?=) 0)
   )
  )


;;
;; Base 64 decode
;;

(defun mew-base64-decode-string (str64)
  (let* ((len (length str64))
	 (count 0)
	 (ret ""))
    (while (< count len)
      (let ((char0 (mew-base64-char256 (aref str64 count)))
            (char1 (mew-base64-char256 (aref str64 (1+ count))))
            (char2 (mew-base64-char256 (aref str64 (+ 2 count))))
            (char3 (mew-base64-char256 (aref str64 (+ 3 count)))))
	(setq ret
	      (concat
	       ret
	       (char-to-string
		(logand (logior (lsh char0 2) (lsh char1 -4)) 255))
	       (char-to-string 
		(logand (logior (lsh char1 4) (lsh char2 -2)) 255))
	       (char-to-string
		(logand (logior (lsh char2 6) char3) 255))
	       )
	      )
	(setq count (+ count 4))
	))
    (if (string-match "=+$" str64)
	(substring ret 0 (- (match-beginning 0) (match-end 0)))
      ret
      )
    ))


(defun mew-base64-char64 (ch256)
  (let ((ch64 (logand ch256 mew-base64-mask)))
    (cond ((< ch64 mew-base64-boundary1)
	   (+ ch64 ?A))
	  ((< ch64 mew-base64-boundary2)
	   (+ (- ch64 mew-base64-boundary1) ?a))
	  ((< ch64 mew-base64-boundary3)
	   (+ (- ch64 mew-base64-boundary2) ?0))
	  ((= ch64 mew-base64-boundary3) 
	   ?+)
	  (t ?/))
    ))


;;
;; Quoted-printable decode
;;

(defun mew-hexstring-to-int (hex)
  (let ((len (length hex))
	(count 0)
	(ch nil)
	(ret 0))
    (while (< count len)
      (setq ch (aref hex count))
      (cond
       ((and (<= ?0 ch) (<= ch ?9))
	(setq ret (+ (* ret 16) (- ch ?0))))
       ((and (<= ?A ch) (<= ch ?F))
	(setq ret (+ (* ret 16) (+ (- ch ?A) 10))))
       ((and (<= ?a ch) (<= ch ?f))
	(setq ret (+ (* ret 16) (+ (- ch ?a) 10))))
       )
      (setq count (1+ count))
      )
    ret
    ))

(defun mew-quoted-printable-decode-string (qpstr)
  (let ((start) (end))
    (while (string-match "_" qpstr)
      (aset qpstr (match-beginning 0) 32)) ;; 32 = space
    (while (string-match "=[0-9A-Z][0-9A-Z]" qpstr)
      (setq start (match-beginning 0))
      (setq end (match-end 0))
      (setq qpstr
	    (concat (substring qpstr 0 start)
		    (char-to-string (mew-hexstring-to-int 
				     (substring qpstr (1+ start) end)))
		    (substring qpstr end nil))))
    qpstr
    ))


(defun mew-char-to-qhex (char)
  (concat "="
	  (char-to-string (aref "0123456789ABCDEF" (lsh char -4)))
	  (char-to-string (aref "0123456789ABCDEF" (logand char 15)))))

(defun mew-quoted-printable-encode-string (str)
  (let ((len (length str))
	(count 0)
	(ch nil)
	(ret))
    (while (< count len)
      (setq ch (aref str count))
      (cond
       ((and (> ch 32)
	     (< ch 126)
	     (not (equal ch ?=))
	     (not (equal ch ??))
	     (not (equal ch ?_)) ;; space
	     )
	(setq ret (concat ret (char-to-string ch)))
	)
       ((equal ch 32)
	(setq ret (concat ret "_"))
	)
       (t 
	(setq ret (concat ret (mew-char-to-qhex ch)))
	)
       )
      (setq count (1+ count))
      )
    ret
    ))

(defun mew-header-encode-string (str encode)
  (let ((fun (mew-decode-func-func
	      (mew-decode-func-attr encode mew-encode-switch)))
	lc)
    (cond
     (mew-mule-p
      (if (setq lc (car (find-charset-string str)))
	  (let* ((attr (mew-mule-lc-attr lc mew-mule-character-set))
		 (charset (mew-mule-lc-content attr))
		 (symbol (mew-mule-lc-symbol attr))
		 (txstr (code-convert-string str *internal* symbol)))
	    (concat "=?" charset "?" encode "?" (funcall fun txstr) "?=")
	    )
	str
	)
      )
     (mew-emacs19-p
      (if (string-match "[\200-\377]" str)
	  (concat "=?iso-8859-1?" encode "?" (funcall fun str) "?=")
	str)
      )
     (t str)
     )
    ))

;;;
;;; End of Mew
;;;
