;;;	             mhc.el -- MH calendar.
;;;
;;;		 Copyright (C) 1994  Yoshinari NOMURA
;;;
;;;		   This emacs lisp library confirms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Yoshinari NOMURA <nomsun@csce.kyushu-u.ac.jp>
;;; Created: Jul    4, 1994
;;; Revised: Dec   17, 1994

;;;
;;; Constants.
;;;

(defconst mhc-date-field "X-SC-Date:")
(defconst mhc-subject-field "X-SC-Subject:")
(defconst mhc-mode-version "mhc.el version 0.08")
(defconst mhc-rfc-date-re
  (concat 
   "^ *"
   "\\(\\(Mon\\|Tue\\|Wed\\|Thu\\|Fri\\|Sat\\|Sun\\) *, *\\)?"
   "\\([0-9][0-9]?\\) +"
   "\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun"
   "\\|Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\) +"
   "\\([0-9]+\\) +"
   "\\([0-9][0-9]\\):\\([0-9][0-9]\\)\\(:\\([0-9][0-9]\\)\\)? *"
   "\\([-+][0-9][0-9][0-9][0-9]\\|[A-Z]+\\)? *$"))


;;;
;;; User customize variables.
;;;
(defvar mhc-summary-msgno-re "^ +\\([0-9]+\\)")
(defvar mhc-summary-line-re "\\([^|]+\\)| +\\([0-9]+\\)/\\([0-9]+\\)")
(defvar mhc-guess-date-re '("\\([$B#0(B-$B#9(B0-9]+\\) *[$B!?(B/$B7n(B] *\\([$B#0(B-$B#9(B0-9]+\\)"))
(defvar mhc-guess-time-re '("\\([$B#0(B-$B#9(B0-9]+\\) *[$B!'(B:$B;~(B] *\\([$B#0(B-$B#9(B0-9]+\\)"))
(defvar mhc-mailer-package 'mh-e)
(defvar mhc-scan-function 'mh-visit-folder)
(defvar mhc-mail-path (expand-file-name "~/Mail"))
(defvar mhc-schedule-folder "schedule")
(defvar mhc-window-stack nil)
(defvar mhc-draft-mode-hook nil)
(defvar mhc-mode-hook nil)
(defvar mhc-mode-map nil)
(defvar mhc-draft-mode-map nil)
(defvar mhc-tmp-string-buffer " *mhc-tmp-string*")
(defvar mhc-tmp-message-buffer " *mhc-tmp-message*")
(defvar mhc-prog-refile "refile")

;;;
;;; Key maps
;;; 
(if mhc-draft-mode-map
    nil
  (setq mhc-draft-mode-map (make-sparse-keymap))
  (define-key mhc-draft-mode-map "\C-c\C-c" 'mhc-entry-this-buffer))

;;;
;;; mhc-mode, mhc-draft-mode
;;;
(defun mhc-mode (&optional args)
  (interactive)
  (cond ((equal mhc-mailer-package 'mew)
	 (setq mhc-scan-function 'mew-summary-scan))
	((equal mhc-mailer-package 'mh-e)
	 (setq mhc-scan-function 'mh-visit-folder))
	(t
	 (message (format "Mailer %s is not support." mhc-mailer-package))))
  (define-key (current-local-map) "\C-c>" 'mhc-goto-next-month)
  (define-key (current-local-map) "\C-c." 'mhc-goto-this-month)
  (define-key (current-local-map) "\C-c<" 'mhc-goto-previous-month)
  (define-key (current-local-map) "\C-cf" 'mhc-goto-today)
  (define-key (current-local-map) "\C-c|" 'mhc-entry-message-buffer)
  (define-key (current-local-map) "\C-cm" 'mhc-modify)
  (define-key (current-local-map) "\C-ce" 'mhc-entry)
  (or (assq 'mhc-mode minor-mode-alist)
      (setq minor-mode-alist
	    (cons '(mhc-mode " Schedule") minor-mode-alist)))
  (run-hooks 'mhc-mode-hook))

(defun mhc-draft-minor-mode ()
  (interactive)
  (define-key (current-local-map) "\C-cd" 'mhc-insert-date-field)
  (or (assq 'mhc-draft-minor-mode minor-mode-alist)
      (setq minor-mode-alist
	    (cons '(mhc-draft-minor-mode " mhc") minor-mode-alist))))

(defun mhc-draft-mode ()
  (interactive)
  (setq major-mode 'mhc-draft-mode)
  (setq mode-name "MHC-Draft")
  (use-local-map mhc-draft-mode-map)
  (run-hooks 'text-mode-hook 'mhc-draft-mode-hook))

;;;
;;; Mhc user interface
;;;

;;
;; Goto previous mon, next mon, this mon, today.
;;
(defun mhc-goto-today ()
  (interactive)
  (beginning-of-buffer)
  (re-search-forward 
   (format "%02d/%02d" 
	   (nth 1 (mhc-get-digit-date-today))
	   (nth 2 (mhc-get-digit-date-today)))
   nil t
   ))

(defun mhc-goto-this-month ()
  (interactive)
  (let ((yy (nth 0 (mhc-get-digit-date-today)))
	(mm (nth 1 (mhc-get-digit-date-today))))
    (funcall mhc-scan-function
	     (format "+%s/%02d/%02d" mhc-schedule-folder yy mm) 
	     (list "all" t))))

(defun mhc-goto-next-month ()
  (interactive)
  (let ((yy (nth 0 (mhc-get-digit-date-cursor)))
	(mm (nth 1 (mhc-get-digit-date-cursor))))
    (if (= mm 12)
	(setq mm 1 yy (1+ yy))
      (setq mm (1+ mm)))
    (if (mhc-parse-digit-date (list yy mm '1 '0 '0))
	(funcall mhc-scan-function
		 (format "+%s/%02d/%02d" mhc-schedule-folder yy mm) 
		 (list "all" t)))
    ))

(defun mhc-goto-previous-month ()
  (interactive)
  (let ((yy (nth 0 (mhc-get-digit-date-cursor)))
	(mm (nth 1 (mhc-get-digit-date-cursor))))
    (if (= mm 1)
	(setq mm 12 yy (1- yy))
      (setq mm (1- mm)))
    (if (mhc-parse-digit-date (list yy mm '1 '0 '0))
	(funcall mhc-scan-function
	       (format "+%s/%02d/%02d" mhc-schedule-folder yy mm)
	       (list "all" t)))
    ))

;;
;; Entry, modify
;;
(defun mhc-entry ()
  (interactive)
  (let* ((yy (nth 0 (mhc-get-digit-date-cursor)))
	 (mm (nth 1 (mhc-get-digit-date-cursor)))
	 (dd (nth 2 (mhc-get-digit-date-cursor)))
	 (buf (concat "schedule " (format "%2d/%2d %2d" mm dd yy)))
	 (digit-date
	  (mhc-input-slash-date "Date: " (format "%02d/%02d/00:00" mm dd)))
	 (subject (read-from-minibuffer "Subject: ")))
    (mhc-window-push)
    (setq buf (buffer-name (generate-new-buffer buf)))
    (switch-to-buffer-other-window buf)
    (insert (concat mhc-date-field " " 
		    (mhc-parse-digit-date digit-date) "\n"))
    (insert (concat mhc-subject-field " " subject))
    (insert "\n----\n")
    (mhc-draft-mode)
    ))

(defun mhc-modify ()
  (interactive)
  (let ((yy (nth 0 (mhc-get-digit-date-cursor)))
	(mm (nth 1 (mhc-get-digit-date-cursor)))
	(no (mhc-get-message-no)))
    (mhc-window-push)
    (find-file-other-window 
     (concat mhc-mail-path "/" mhc-schedule-folder
	     (format "/%02d/%02d/%d" yy mm no)))
    (mhc-draft-mode)))

(defun mew-mime-schedule (begin end &optional params)
  (mhc-window-push)
  (mew-mime-text/plain begin end params)
  (delete-other-windows)
  (condition-case ()
      (if (not (y-or-n-p "Do you want to entry this buffer ? "))
	  (mhc-window-pop)
	(copy-to-buffer 
	 (get-buffer-create mhc-tmp-message-buffer) (point-min) (point-max))
	(switch-to-buffer mhc-tmp-message-buffer t)
	(mhc-entry-this-buffer t))
    (quit (mhc-window-pop))))


(defun mhc-entry-message-buffer ()
  (interactive)
    (mhc-window-push)
    (other-window 1)
    (delete-other-windows)
    (condition-case ()
	(if (not (y-or-n-p "Do you want to entry this buffer ? "))
	    (mhc-window-pop)
	  (copy-to-buffer 
	   (get-buffer-create mhc-tmp-message-buffer) (point-min) (point-max))
	  (switch-to-buffer mhc-tmp-message-buffer t)
	  (mhc-entry-this-buffer t))
      (quit (mhc-window-pop))))

(defun mhc-entry-this-buffer (&optional query)
  (interactive)
  (let (folder digit-date
        (tmp-file (or (buffer-file-name)
		      (expand-file-name (make-temp-name "mhc-temp.")
					mhc-mail-path))))
    (setq digit-date 
	  (mhc-parse-rfc-date (mhc-get-field-value mhc-date-field)))
    (if (or (not (mhc-parse-digit-date digit-date))
	    (and query 
		 (not (y-or-n-p "X-SC-Date already exists. Use ? "))))
	(setq digit-date 
	      (mhc-input-slash-date "Please input date " 
				    (mhc-guess-slash-date))))
    (mhc-add-field mhc-date-field (mhc-parse-digit-date digit-date))
    (mhc-add-field mhc-subject-field 
		   (or (mhc-get-field-value mhc-subject-field)
		       (read-from-minibuffer 
			"Subject : "
			(mhc-get-field-value "Subject:"))))
    (mhc-delete-field-separator)
    (setq folder (mhc-digit-date-to-folder digit-date))
    (if (y-or-n-p (format "I will refile to %s ... " folder))
	(progn
	  (write-region (point-min) (point-max) tmp-file nil 'silence)
	  (message "Refiling ...")
	  (call-process mhc-prog-refile nil nil nil "-file" tmp-file folder)
	  (message "Refiling ... Done.")
	  (set-buffer-modified-p nil)
	  ;(mhc-cache-header)
	  (kill-buffer (buffer-name))
	  ))
    (mhc-window-pop)
    ))

;;
;; Insert X-SC-Date, X-SC-Subject fields into a mail draft.
;;
(defun mhc-insert-date-field ()
  (interactive)
  (save-excursion
    (let ((digit-date (mhc-input-slash-date 
		       "Please input date " 
		       (mhc-guess-slash-date))))
      (mhc-add-field mhc-date-field (mhc-parse-digit-date digit-date))
      (mhc-add-field mhc-subject-field 
			 (read-from-minibuffer 
			  "Subject : "
			  (or (mhc-get-field-value mhc-subject-field)
			      (mhc-get-field-value "Subject:"))))
      )))


;;; mhc date format handler
;;;
;;; I use three date format for mhc.
;;;
;;; 1. RFC   date format
;;;        ex. Tue, 12 Jul 1994 21:11:28 +0900
;;; 2. Slash date format
;;;        ex. 94/7/12/21:11 or 7/12/21:00 or 21:00
;;; 3. Digit date format
;;;        ex (94 7 12 21 11 2) ;2 is tuesday
;;;
;;;                       input      output      syntax-check semantics-check
;;; mhc-parse-rfc-date  : rfc-date   digit-date  yes          no
;;; mhc-parse-slash-date: slash-date digit-date  yes          no
;;; mhc-parse-digit-date: digit-date rfc-date    yes          yes

;;
;; RFC date format handler
;;
; Check validity of a RFC date format (only) and convert to a digit date.
(defun mhc-parse-rfc-date (rfc-date)
  (let (yy mm dd hh ss)
    (if (not (and (stringp rfc-date) (string-match mhc-rfc-date-re rfc-date)))
	nil
      (setq
       yy (substring rfc-date (match-beginning 5) (match-end 5)) ; year
       mm (substring rfc-date (match-beginning 4) (match-end 4)) ; month
       dd (substring rfc-date (match-beginning 3) (match-end 3)) ; day
       hh (substring rfc-date (match-beginning 6) (match-end 6)) ; hour
       ss (substring rfc-date (match-beginning 7) (match-end 7)) ; minute
       mm (+ (/ (string-match 
		 mm 
		 "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec") 4) 1)
       yy (% (string-to-int yy) 100))
      (list  yy mm (string-to-int dd) (string-to-int hh) (string-to-int ss)))
    ))

;;
;; Slash date format handler
;;
; Check validity of a Slash date format (only) and convert to a digit date.
(defun mhc-parse-slash-date (slash-date)
  (let ((yy nil) (mm nil) (dd nil) (hh nil) (ss nil)
	(now (mhc-get-digit-date-today))
	(n  "\\([0-9]+\\)")
	(ns "\\([0-9]+\\)/")
	(nc "\\([0-9]+\\):"))
    (if (not (stringp slash-date))
	nil
      (cond
       ((string-match (concat "^" ns ns ns nc n "$") slash-date)
	(setq yy (substring slash-date (match-beginning 1) (match-end 1))
	      mm (substring slash-date (match-beginning 2) (match-end 2))
	      dd (substring slash-date (match-beginning 3) (match-end 3))
	      hh (substring slash-date (match-beginning 4) (match-end 4))
	      ss (substring slash-date (match-beginning 5) (match-end 5))))
       ((string-match (concat "^" ns ns nc n "$") slash-date)
	(setq mm (substring slash-date (match-beginning 1) (match-end 1))
	      dd (substring slash-date (match-beginning 2) (match-end 2))
	      hh (substring slash-date (match-beginning 3) (match-end 3))
	      ss (substring slash-date (match-beginning 4) (match-end 4))))
       ((string-match (concat "^" ns nc n "$") slash-date)
	(setq dd (substring slash-date (match-beginning 1) (match-end 1))
	      hh (substring slash-date (match-beginning 2) (match-end 2))
	      ss (substring slash-date (match-beginning 3) (match-end 3))))
       ((string-match (concat "^" nc n "$") slash-date)
	(setq hh (substring slash-date (match-beginning 1) (match-end 1))
	      ss (substring slash-date (match-beginning 2) (match-end 2)))))
      (if (or yy mm dd hh ss)
	  (progn
	    (setq yy (or (and (stringp yy) (string-to-int yy)) (nth 0 now))
		  mm (or (and (stringp mm) (string-to-int mm)) (nth 1 now))
		  dd (or (and (stringp dd) (string-to-int dd)) (nth 2 now))
		  hh (or (and (stringp hh) (string-to-int hh)) (nth 3 now))
		  ss (or (and (stringp ss) (string-to-int ss)) (nth 4 now)))
	    (list yy mm dd hh ss)))
      )))

;;
;; Digit date format handler
;;

; Check validity of a digit date syntax and semantics,
; then convert to a RFC date.
(defun mhc-parse-digit-date (digit-date)
  (let (yy mm dd hh ss
	   (mon-str-alist
	    '(( 1 . "Jan") ( 2 . "Feb") ( 3 . "Mar") ( 4 . "Apr")
	      ( 5 . "May") ( 6 . "Jun") ( 7 . "Jul") ( 8 . "Aug")
	      ( 9 . "Sep") (10 . "Oct") (11 . "Nov") (12 . "Dec"))))
    (if (or (not (listp digit-date)) (not digit-date))
	nil
      (setq yy (nth 0 digit-date)
	    mm (nth 1 digit-date)
	    dd (nth 2 digit-date)
	    hh (nth 3 digit-date)
	    ss (nth 4 digit-date))
      (and (>= yy 0) (<= yy 99) 
	   (>= mm 1) (<= mm 12)
	   (>= hh 0) (<= hh 23)
	   (>= ss 0) (<= ss 59)
	   (>= dd 1) (<= dd (mhc-day-of-month yy mm))
	   (format "%d %s %d %02d:%02d" 
		     dd (cdr (assoc mm mon-str-alist)) yy hh ss)))))

(defun mhc-day-of-month (yy mm)
  (if (or (= (% yy 400) 0)
	  (and (= (% yy 4) 0)
	       (not (= (% yy 100) 0))))
      (nth (1- mm) (list 31 29 31 30 31 30 31 31 30 31 30 31))
    (nth (1- mm) (list 31 28 31 30 31 30 31 31 30 31 30 31))))

(defun mhc-get-digit-date-today ()
  (let (week mon day hour min sec year (now (current-time-string)))
  ; current-time-string is "Thu Jun 30 01:58:16 1994"
  (string-match (concat 
		 "^\\([^ ]+\\) +\\([^ ]+\\) +\\([^ ]+\\) +"
		 "\\([^: ]+\\):\\([^: ]+\\):[^ ]+ +"
		 "\\([^ ]+\\)$")
		 (current-time-string))
  (setq 
   week (substring now (match-beginning 1) (match-end 1))
   mon  (substring now (match-beginning 2) (match-end 2))
   day  (substring now (match-beginning 3) (match-end 3))
   hour (substring now (match-beginning 4) (match-end 4))
   min  (substring now (match-beginning 5) (match-end 5))
   year (substring now (match-beginning 6) (match-end 6))

   week (/ (string-match week "Sun Mon Tue Wed Thu Fri Sat") 4)
   mon (1+ (/ (string-match 
	       mon 
	       "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec") 4))
   year (% (string-to-int year) 100))
  (list  year
	 mon
	 (string-to-int day)
	 (string-to-int hour)
	 (string-to-int min)
	 week)))


; Get destination folder name from digit date
(defun mhc-digit-date-to-folder (digit-date)
  (if (not (mhc-parse-digit-date digit-date))
      nil
    (concat
     "+" mhc-schedule-folder "/"
      (format "%02d/%02d" (nth 0 digit-date) (nth 1 digit-date)))))

; Get cursor line's digit date 
(defun mhc-get-digit-date-cursor ()
  (let ((yy (string-to-int (file-name-nondirectory 
			    (directory-file-name 
			     (file-name-directory (buffer-name))))))
	(mm (string-to-int (file-name-nondirectory (buffer-name))))
	(dd nil))
    (save-excursion
      (beginning-of-line)
      (while (not (looking-at mhc-summary-line-re))
	(previous-line 1)))
    (setq dd (string-to-int 
	      (buffer-substring (match-beginning 3) (match-end 3))))
    (list yy mm dd '0 '0)))

;;
;; Input.
;;

; Input slash-date, return digit-date
; ret-null: when non-nil  , if user's input is "", returns "".
;           when nil      , if user's input is "", prompt again.
(defun mhc-input-slash-date (prompt &optional default ret-null)
  (interactive)
  (let (slash-date)
    (setq slash-date 
	  (read-from-minibuffer (concat prompt " ([yy/][mm/][dd/]hh:mm): ")
				(or default "")))
    (while (and 
	    (not (and ret-null (equal slash-date "")))
	    (not (mhc-parse-digit-date (mhc-parse-slash-date slash-date))))
      (setq slash-date 
	    (read-from-minibuffer
	     "Bad date format. again ([yy/][mm/][dd/]hh:mm): "
	     (or slash-date ""))))
    (cond ((equal slash-date "")
	   "")
	  (t
	   (mhc-parse-slash-date slash-date)))))

;;;
;;; Fields handler.
;;;

; Get field value as one line.
(defun mhc-get-field-value (field)
  (let ((ret nil))
    (save-excursion
      (if (mhc-have-fields-p)
	  (progn
	    (goto-char (point-min))
	    (re-search-forward "^-*$" nil t)
	    (beginning-of-line)
	    (narrow-to-region (point-min) (point))
	    (goto-char (point-min))
	    (if (re-search-forward
		 (format  "^%s +\\([^\n]*\\(\n[ \t]+.*\n\\)*\\)" field) nil t)
		(setq ret 
		      (mhc-replace-string
		       (buffer-substring (match-beginning 1) (match-end 1))
		       "\n *" " "))
	      (setq ret nil))
	    (widen))))
    ret))


; Delete fields
(defun mhc-delete-field (field)
  (save-excursion
    (if (mhc-have-fields-p)
	(progn
	  (goto-char (point-min))
	  (re-search-forward "^-*$" nil t)
	  (beginning-of-line)
	  (narrow-to-region (point-min) (point))
	  (goto-char (point-min))
	  (delete-matching-lines 
	   (format  "^%s[^\n]*\\(\n[ \t]+.*\n\\)*" field))
	  (widen)))))

; Add field and value to a buffer, if already exists the field,
; erase it before the addtion.
(defun mhc-add-field (field value)
  (save-excursion
    (if (mhc-have-fields-p)
	(progn
	  (mhc-delete-field field)
	  (goto-char (point-min))
	  (re-search-forward "^-*$" nil t)
	  (beginning-of-line)
	  (insert (concat field " " value "\n")))
      (goto-char (point-min))
      (beginning-of-line)
      (insert (concat field " " value "\n\n")))))

; replace strig like s/foo/var/
(defun mhc-replace-string (string from to)
  (save-excursion
    (get-buffer-create mhc-tmp-string-buffer)
    (set-buffer mhc-tmp-string-buffer)
    (erase-buffer)
    (insert string)
    (beginning-of-buffer)
    (replace-regexp from to nil)
    (buffer-substring (point-min) (point-max))))

; return as if the buffer has a mail fields.
(defun mhc-have-fields-p ()
  (save-excursion
    (goto-char (point-min))
    (and
     (looking-at "^[-a-zA-Z0-9]+: ")
     (re-search-forward "^-*$" nil t))))

; delete field separator.
(defun mhc-delete-field-separator ()
  (save-excursion
    (goto-char (point-min))
    (if (and (mhc-have-fields-p) (re-search-forward "^-*$" nil t))
	(progn
	  (beginning-of-line)
	  (kill-line 1)
	  (insert "\n")
	  t))))

;;;
;;; folder, filename, message no handler.
;;;
; Get cursor line's mh message no
(defun mhc-get-message-no ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at mhc-summary-msgno-re)
	(string-to-int (buffer-substring (match-beginning 1) (match-end 1))))
    ))

;;;
;;; Guess slash-date.
;;;

; return "09/15/11:25"
(defun mhc-guess-slash-date ()
  (concat 
   (if (equal (mhc-guess-date) "")
       ""
     (concat (mhc-guess-date) "/"))
   (if (equal (mhc-guess-time) "")
       "00:00"
     (mhc-guess-time))))

; return "09/15" or ""
(defun mhc-guess-date ()
  (let ((ret nil)
	(regexp-list mhc-guess-date-re))
    (while (car regexp-list)
      (setq ret (append ret (mhc-guess-workhorse (car regexp-list))))
      (setq regexp-list (cdr regexp-list)))
    (or 
     (catch 'match
       (while (car ret)
	 (if (not (string-lessp
	      (format "%02d/%02d"
		      (car (car ret)) (car (cdr (car ret))))
	      (format "%02d/%02d"
		      (nth 1 (mhc-get-digit-date-today))
		      (nth 2 (mhc-get-digit-date-today)))))
	     (throw 'match 
		    (format "%02d/%02d"
			    (car (car ret)) (car (cdr (car ret))))))
	 (setq ret (cdr ret))))
     "")))

; return "11:25" or ""
(defun mhc-guess-time ()
  (let ((ret nil)
	(regexp-list mhc-guess-time-re))
    (while (car regexp-list)
      (setq ret (append ret (mhc-guess-workhorse (car regexp-list))))
      (setq regexp-list (cdr regexp-list)))
    (if (not (equal ret nil))
	(format "%02d:%02d" (car (car ret)) (car (cdr (car ret))))
      "")))

; return ((9 10) (11 22) ..) or nil
(defun mhc-guess-workhorse (regexp)
  (let ((list nil))
    (save-excursion
      (beginning-of-buffer)
      (re-search-forward "^-*$" nil t)

      (while (re-search-forward regexp nil t)
	(setq list 
	      (cons 
	       (list (mhc-string-to-int
		      (buffer-substring (match-beginning 1) (match-end 1)))
		     (mhc-string-to-int
		      (buffer-substring (match-beginning 2) (match-end 2))))
	       list)))
      (reverse list))))

;;
;; routines.
;;
(defun mhc-string-to-int (string)
  (let ((char "")
	(ret "")
	(data (match-data)))
    (while (string-match "^." string)
      (setq char (substring string (match-beginning 0) (match-end 0)))
      (setq ret (concat ret
			(cond ((equal char "$B#0(B") 0)
			      ((equal char "$B#1(B") 1)
			      ((equal char "$B#2(B") 2)
			      ((equal char "$B#3(B") 3)
			      ((equal char "$B#4(B") 4)
			      ((equal char "$B#5(B") 5)
			      ((equal char "$B#6(B") 6)
			      ((equal char "$B#7(B") 7)
			      ((equal char "$B#8(B") 8)
			      ((equal char "$B#9(B") 9)
			      (t              char))))
      (setq string (substring string (match-end 0))))
    (store-match-data data)
    (string-to-int ret)))

;;
;; Window stack
;;

(defun mhc-window-push ()
  (interactive)
  (setq mhc-window-stack
	(cons (current-window-configuration) mhc-window-stack)))

(defun mhc-window-pop ()
  (interactive)
  (if mhc-window-stack
      (set-window-configuration (car-safe mhc-window-stack))
    (message "Window Stack is empty."))
  (setq mhc-window-stack (cdr-safe mhc-window-stack)))
