;; Regular expression compiler
;; Copyright (C) 1992 Free Software Foundation, Inc.
;; This file is part of Mule (MULtilingual Enhancement of GNU Emacs).
;; This file contains Japanese characters

;; Mule is free software distributed in the form of patches to GNU Emacs.
;; You can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; Mule is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;; Coded by S.Tomura <tomura@etl.go.jp>

;;;;
;;;; regexp.el
;;;; 

(defvar regexp-version "0.37")
;;; Last modified date: Thu Dec 26 20:51:36 1991
;;; Last modified date: Mon Mar  9 09:45:11 JST 1992
;;; 92.5.16 by K.Handa -- buffer name preceded by space.
;;; 92.7.17  modified for Mule Ver.0.9.5 by K.Handa <handa@etl.go.jp>
;;;	regexp-compile-word is added to avoid a bug in regexp-compile.
;;; 92.10.26 modified for Mule Ver.0.9.6
;;;				 by T.Saneto <sanewo@pdp.crl.sony.co.jp>
;;;	Type fixed.

(if (fboundp 'syntax-spec-code) nil
  (defun syntax-spec-code (ch) ch))

(if (fboundp 'syntax-code-spec) nil
  (defun syntax-code-spec (code) code))

;;;
;;; utilities for ml
;;;

(defun read-hexa (str)
  (let ((result 0) (i 0) (max (length str)))
    (while (< i max)
      (let ((ch (aref str i)))
	(cond((and (<= ?0 ch) (<= ch ?9))
	      (setq result (+ (* result 16) (- ch ?0))))
	     ((and (<= ?a ch) (<= ch ?f))
	      (setq result (+ (* result 16) (+ (- ch ?a) 10))))
  	     ((and (<= ?A ch) (<= ch ?F))
	      (setq result (+ (* result 16) (+ (- ch ?A) 10)))))
	(setq i (1+ i))))
    result))

;;;/* Special characters */
(defconst ESC (read-hexa "1B") "escape character")
(defconst SS2 (read-hexa "8E") "leading char of Kana in EUC")

(defun string-forward-anychar (str start)
  (and (stringp str) (numberp start)
       (let ((max (length str)))
	 (and (<= 0 start) 
	      (< start max)
	      (+ start (char-bytes (sref str start)))))))

(defun member (elm list pred)
  (while (and list (not (funcall pred elm (car list))))
    (setq list (cdr list)))
  list)

(defun memequal (elm list)
  (while (and list (not (equal elm (car list))))
    (setq list (cdr list)))
  list)

(defun find (elm list)
  (let ((pos 0))
    (while (and list (not (equal elm (car list))))
      (setq list (cdr list))
      (setq pos  (1+ pos)))
    (if list
	pos
      nil)))

(defun delete-duplicate (list)
  (let ((result nil))
    (while list
      (if (not (memequal (car list) result))
	  (setq result (cons (car list) result)))
      (setq list (cdr list)))
    (reverse result)))

(defun delete (elm list)
  (cond((null list) nil)
       ((equal elm (car list))
	(delete elm (cdr list)))
       (t 
	(cons (car list) (delete elm (cdr list))))))

(defmacro define-enum (&rest list)
  (list 'define-enum* (list 'quote list)))

(defun define-enum* (list)
  (let ((i 0))
    (while list
      (set (car list) i)
      (setq i (1+ i))
      (setq list (cdr list)))))

(define-enum 
  UNUSED
  EXACTN
  BEGLINE
  ENDLINE
  JUMP     
  ON_FAILURE_JUMP
  FINALIZE_JUMP
  MAYBE_FINALIZE_JUMP
  DUMMY_FAILURE_JUMP
  ANYCHAR
  CHARSET
  CHARSET_NOT
;;;  MLCHARSET
;;;  MLCHARSET_NOT
  START_MEMORY
  STOP_MEMORY
  DUPLICATE
  BEFORE_DOT
  AT_DOT
  AFTER_DOT
  BEGBUF
  ENDBUF
  WORDCHAR
  NOTWORDCHAR
  WORDBEG
  WORDEND
  WORDBOUND
  NOTWORDBOUND
  SYNTAXSPEC
  NOTSYNTAXSPEC
;;; extended instructions
  EXACT1
  EXACT2
  EXACT3
  CHARSET_M
  CHARSET_M_NOT
  CASE
  SUCCESS_SHORT
  SUCCESS
  POP
  EXECPT0 ;;; ALLCHAR
  EXECPT1
  CATEGORYSPEC
  NOTCATEGORYSPEC
  )

;;; ANYCHAR = EXECPT1 \n
;;; ALLCHAR = EXECPT0
;;;;
;;;  UNUSED
;;;  EXACTN n ch1 ch2 ... chn
;;;  BEGLINE
;;;  ENDLINE
;;;  JUMP disp[2]
;;;  ON_FAILURE_JUMP disp[2]
;;;  FINALIZE_JUMP disp[2]
;;;  MAYBE_FINALIZE_JUMP disp[2]
;;;  DUMMY_FAILURE_JUMP disp[2]
;;;  ANYCHAR
;;;  CHARSET n b1 b2 ... bn
;;;**CHARSET 0xff l1 l2 cf1 ct1 cf2 ct2 ... cfn ctn
;;;  CHARSET_NOT n b1 b2 ... bn
;;;**CHARSET_NOT 0xff l1 l2 cf1 ct1 cf2 ct2 ... cfn ctn
;;; $B0J2<$O$($J$_;a$NDs0F$K$h$k?7$?$J%;%^%s%F%#%C%/%9(B
;;;$B!J$^$@(B regexp.el $B$KH?1G$5$l$F$$$J$$!K(B
;;
;;	CHARSET n b1 b2 ... bn lh lo cf1 ct1 cf2 ct2 ... cfn ctn
;;	         |<- bitmap ->|     |<-     range table       ->|
;;	CHARSET_NOT n b1 b2 ... bn lh lo cf1 ct1 cf2 ct2 ... cfn ctn
;;	CHARSETM m n b1 b2 ... bn lh lo cf1 ct1 cf2 ct2 ... cfn ctn
;;	CHARSETM_NOT m n b1 b2 ... bn lh lo cf1 ct1 cf2 ct2 ... cfn ctn
;;
;;	  o cfx, ctx $B0J30$O$9$Y$F(B 1byte.  cfx, ctx $B$O(B multi byte
;;	    character.
;;
;;	  o CHARSET(_NOT) $B$H(B CHARSETM(_NOT) $B$H$N0c$$$O(B, CHARSETM(_NOT) 
;;	    $B$N>l9g$K$O(B bitmap $B$N@hF,$N(B m bytes $B$,>J$+$l$F$$$kE@(B.
;;
;;	  o b1 ... bn ($B$D$^$j(B bitmap$B$ND9$5(B)$B$O(B, (n & 0x7f) bytes.  n $B$N(B
;;	    $BJ,(B 1byte $B$O4^$^$J$$(B.
;;
;;	  o lh $B0J2<$O(B n & 0x80 $B$,(B 0 $B$J$iB8:_$7$J$$(B.
;;
;;	  o lh $B$+$i(B ctn $B$^$G$ND9$5(B($B$D$^$j(B range table $B$ND9$5(B) $B$O(B ((lh
;;	    << 8) + lo) byte.  lh $B$H(B lo $B$N(B 2byte $B$r4^$`(B.  ($B>e$N(B n $B$N>l(B
;;	    $B9g$H0c$$$^$9$,(B, $BE}0l$7$?$[$&$,$$$$$+$J(B?).
;;
;; 	  o cfx $B$O(B multi byte character $B$G(B, cfx $B$H(B ctx $B$N(B leading char 
;;	    $B$OF1$8$G$J$$$H$$$1$J$$(B.  $B$^$?(B, cfx $B$N(B leading char $B$O(B 0 $B$G(B
;;	    $B$"$C$F$O$$$1$J$$(B(range table $B$K(B leading char $B$,(B 0 (ASCII$B$H(B
;;	    $B$+(B) $B$NJ8;z$,$"$C$F$b(B, $B8=:_$O(B fastmap $B$KH?1G$5$l$J$$$+$i(B).
;;
;;;  START_MEMORY regno
;;;  STOP_MEMORY regno
;;;  DUPLICATE regno
;;;  BEFORE_DOT
;;;  AT_DOT
;;;  AFTER_DOT
;;;  BEGBUF
;;;  ENDBUF
;;;  WORDCHAR
;;;  NOTWORDCHAR
;;;  WORDBEG
;;;  WORDEND
;;;  WORDBOUND
;;;  NOTWORDBOUND
;;;  SYNTAXSPEC ch
;;;  NOTSYNTAXSPEC ch

;;;<<added instructions>>
;;;  EXACT1 ch
;;;  EXACT2 ch1 ch2
;;;  EXACT3 ch1 ch2 ch3
;;;  CHARSETM m n b1 b2 .. bn
;;;    charset $B$N(B bitmaps $B$N$&$A@hF,$N(B m bytes $B$r>J$$$?$b$N(B
;;;  CHARSETM_NOT m n b1 b2 .. bn
;;;    charset_not $B$N(B bitmaps $B$N$&$A@hF,$N(B m bytes $B$r>J$$$?$b$N(B
;;;  CASE n disp[1] disp[2] ... disp[n] l u indl ... indu
;;;    $B:G=i$K(B n $B8D$N(B jump relative address(2bytes) $B$,B3$-!$(B
;;;    $B<!$K(Bcharacter code l $B$+$i(B m $B$^$G$NJ,$N(Bindex(1byte)$B$,B3$/!%(B
;;;  SUCCESS_SHORT
;;;       short_mode $B$G$J$$$H$-$O!$(Balternative stack $B$r6u$K$7!$(Bpend $B$r(B push $B$9$k!%(B
;;;       short_mode $B$N;~$O!$(B success $B$HF1$8!%(B
;;;  SUCCESS
;;;       pend $B$X%8%c%s%W$9$k!%(B
;;;  POP
;;;       alternative stack $B$r(B pop $B$9$k!%(B

;;;
;;; regexp-parse
;;;

;;;  .    single character except a newline
;;;  REG* more than zero
;;;  REG+ at least once
;;;  REG? once or not at all
;;;  [...] character set
;;;  [^...]  character not set
;;;  ^    beginning of line
;;;  $    end of line
;;;  \    quote
;;;  \|   alternative
;;;  \( ... \) group and mark
;;;  \DIGIT  
;;;  \`   beginning of buffer
;;;  \'   end of buffer
;;;  \b   beginning of word or end of word
;;;  \B   not \b
;;;  \<   beginning of word
;;;  \>   end of word
;;;
;;;  \w   word-constituent character
;;;  \W   not \w
;;;  \sCODE  syntax CODE character
;;;  \SCODE  not \sCODE

;;; REG0 ::= REG1 |
;;;          REG1 "\|" REG0
;;;
;;; REG1 ::= REG2 |
;;;          REG2 REG1
;;;
;;; REG2 ::= REG3 "*" |
;;;          REG3 "+" |
;;;          REG3 "?"
;;;
;;; REG3 ::= "." |
;;;          "[" ... "]" |
;;;          "[" "^" ... "]" |
;;;          "^" |
;;;          "$" |
;;;          "\"" DIGIT |
;;;          "\(" REG0 "\)"

(defvar *regexp-parse-index*  nil)
(defvar *regexp-parse-end*    nil)
(defvar *regexp-parse-str*    nil)
(defvar *regexp-parse-regno*  0)

(defun regexp-error ()
  (error "Illegal regexp:\n%s\n%s\n" (substring *regexp-parse-str* 0 *regexp-parse-index*)  (substring *regexp-parse-str* *regexp-parse-index*)))

(defun regexp-parse (str)
  (setq *regexp-parse-str* str
	*regexp-parse-index*  0
	*regexp-parse-end*    (length str)
	*regexp-parse-regno* 0
	)
  (let ((result (regexp-parse-0)))
    (if (<= *regexp-parse-end* *regexp-parse-index*)
	result
      (regexp-error))))

(defun regexp-parse-0 ()
  (let ((result (regexp-parse-1)))
    (cond((<= *regexp-parse-end* *regexp-parse-index*)
	  result)
	 ((and (< (1+ *regexp-parse-index*) *regexp-parse-end*)
	       (= (aref *regexp-parse-str* *regexp-parse-index*) ?\\)
	       (= (aref *regexp-parse-str* (1+ *regexp-parse-index*)) ?|))
	  (setq *regexp-parse-index* (+ *regexp-parse-index* 2))
	  (list ':or result (regexp-parse-0)))
	 (t result))))

(defun regexp-parse-1 ()
  (let ((results nil)
	(result2 nil))
    (while (setq result2 (regexp-parse-2))
      (setq results (cons result2 results)))
    (if results
	(if (cdr results)
	    (cons ':seq (reverse results))
	  (car results))
      nil)))

(defun regexp-parse-2 ()
  (let ((result (regexp-parse-3)))
    (cond((and (< *regexp-parse-index* *regexp-parse-end*)
	       (= ?* (aref *regexp-parse-str* *regexp-parse-index*)))
	  (setq *regexp-parse-index* (1+ *regexp-parse-index*))
	  (if (and (consp result)
		   (eq (car result) ':mark))
	      (list ':mark (car(cdr result))
		    (list ':star (car (cdr (cdr result)))))
	    (list ':star result)))
	 ((and (< *regexp-parse-index* *regexp-parse-end*)
	       (= ?+ (aref *regexp-parse-str* *regexp-parse-index*)))
	  (setq *regexp-parse-index* (1+ *regexp-parse-index*))
	  (if (and (consp result)
		   (eq (car result) ':mark))
	      (list ':mark (car(cdr result))
		    (list ':plus (car (cdr (cdr result)))))
	    (list ':plus result)))

	 ((and (< *regexp-parse-index* *regexp-parse-end*)
	       (= ?? (aref *regexp-parse-str* *regexp-parse-index*)))
	  (setq *regexp-parse-index* (1+ *regexp-parse-index*))
	  (if (and (consp result)
		   (eq (car result) ':mark))
	      (list ':mark (car(cdr result))
		    (list ':optional (car(cdr(cdr result)))))
	    (list ':optional result)))
	 (t result))))

(defun regexp-parse-3 ()
  (if (<= *regexp-parse-end* *regexp-parse-index*)
      nil
    (let* ((start *regexp-parse-index*)
	   (i *regexp-parse-index*)
	   (end *regexp-parse-end*)
	   (ch (aref *regexp-parse-str* i)))
      (setq *regexp-parse-index* (1+ *regexp-parse-index*))
      (cond ((= ch ?.) 'ANYCHAR)
	    ((= ch ?^) 'BEGLINE)
	    ((= ch ?$) 'ENDLINE)
	    ((= ch ?\[)
	     (regexp-parse-charset))
	    ((= ch ?\])
	     (setq *regexp-parse-index* start)
	     nil
	     )
	    ((= ch ?*)
	     (setq *regexp-parse-index* start)
	     nil
	     )
	    ((= ch ?+)
	     (setq *regexp-parse-index* start)
	     nil
	     )
	    ((= ch ??)
	     (setq *regexp-parse-index* start)
	     nil
	     )
	    ((and (= ch ?\\) (< (1+ i) end))
	     (setq ch (aref *regexp-parse-str* (1+ i)))
	     (setq i (1+ i))
	     (setq *regexp-parse-index* (1+ *regexp-parse-index*))
	     (cond ((= ch ?| )
		    (setq *regexp-parse-index* start)
		    nil
		    )
		   ((= ch ?\( )
		    (let ((result (regexp-parse-0)))
		      (cond((and (< (1+ *regexp-parse-index*) *regexp-parse-end*)
				 (= (aref *regexp-parse-str* *regexp-parse-index*) ?\\ )
				 (= (aref *regexp-parse-str* (1+ *regexp-parse-index*)) ?\) ))
			    (setq *regexp-parse-index* (+ *regexp-parse-index* 2))
			    (setq *regexp-parse-regno* (1+ *regexp-parse-regno*))
			    (list ':mark *regexp-parse-regno* result))
			   (t
			    (regexp-error))))
		    )
		   ((= ch ?\) )
		    (setq *regexp-parse-index* start)
		    nil
		    )
		   ((= ch ?` ) 'BEGBUF)
		   ((= ch ?' ) 'ENDBUF)
		   ((= ch ?b ) 'WORDBOUND)
		   ((= ch ?B ) 'NOTWORDBOUND)
		   ((= ch ?< ) 'WORDBEG)
		   ((= ch ?> ) 'WORDEND)
		   ((= ch ?w ) 'WORDCHAR)
		   ((= ch ?W ) 'NOTWORDCHAR)
		   ((= ch ?=)  'ATDOT)
		   ((and (<= ch ?1)
			 (<= ?9 ch))
		    (list 'DUPLICATE (- ch ?0)))
		   ((= ch ?0)
		    (regexp-error))
		   ((and (= ch ?s )
			 (< (1+ i) end))
		    (setq *regexp-parse-index* (1+ *regexp-parse-index*))
		    (list 'SYNTAXSPEC (syntax-spec-code (aref *regexp-parse-str* (1+ i)))))
		   ((and (= ch ?S )
			 (< (1+ i) end))
		    (setq *regexp-parse-index* (1+ *regexp-parse-index*))
		    (list 'SYNTAXSPEC (syntax-spec-code (aref *regexp-parse-str* (1+ i))))
		    )
		   ((and (= ch ?c )
			 (< (1+ i) end))
		    (setq *regexp-parse-index* (1+ *regexp-parse-index*))
		    (list 'CATEGORYSPEC (aref *regexp-parse-str* (1+ i))))
		   ((and (= ch ?C )
			 (< (1+ i) end))
		    (setq *regexp-parse-index* (1+ *regexp-parse-index*))
		    (list 'NOTCATEGORYSPEC (aref *regexp-parse-str* (1+ i))))

		   (t (substring *regexp-parse-str* (1+ i) (+ i 2)))))
	    (t
	     (let ((nextpos (string-forward-anychar *regexp-parse-str* i)))
	       (cond(nextpos
		     (setq *regexp-parse-index* nextpos)
		     (substring *regexp-parse-str* i nextpos))
		    (t
		     (regexp-error)))))))))


(defun regexp-parse-charset ()
  (if (< *regexp-parse-index* *regexp-parse-end*)
      (cond((eq (aref *regexp-parse-str* *regexp-parse-index*) ?^)
	    (setq *regexp-parse-index* (1+ *regexp-parse-index*))
	    (regexp-parse-charset0 'CHARSET_NOT nil)
	    )
	   (t (regexp-parse-charset0 'CHARSET nil)))
    (regexp-error)))

(defun regexp-parse-charset0 (op list)
  (if (< *regexp-parse-index* *regexp-parse-end*)
      (cond ((eq (aref *regexp-parse-str* *regexp-parse-index*) ?\])
	     (setq *regexp-parse-index* (1+ *regexp-parse-index*))
	     (regexp-parse-charset1 op '("\]"))
	     )
	    (t 
	     (regexp-parse-charset1 op nil)))
    (regexp-error)
    ))

(defun regexp-parse-charset1 (op list)
  (if (< *regexp-parse-index* *regexp-parse-end*)
      (let* ((pos0 *regexp-parse-index*)
	     (pos1 (string-forward-anychar *regexp-parse-str* pos0))
	     (pos2 (string-forward-anychar *regexp-parse-str* pos1))
	     (pos3 (string-forward-anychar *regexp-parse-str* pos2)))
	(if pos0
	    (cond((eq (aref *regexp-parse-str* pos0) ?\])
		  (setq *regexp-parse-index* pos1)
		  (cons op list))
		 ((and pos1 pos2 pos3
		       (eq (aref *regexp-parse-str* pos1) ?-)
		       (not (eq (aref *regexp-parse-str* pos2) ?\])))
		  (setq list (cons (list ':range (substring *regexp-parse-str* pos0 pos1)
					 (substring *regexp-parse-str* pos2 pos3))
				   list))
		  (setq *regexp-parse-index* pos3)
		  (regexp-parse-charset1 op list))
		 ((and pos1
		       (eq (aref *regexp-parse-str* pos1) ?-))
		  (regexp-error))
		 (t
		  (setq list (cons (substring *regexp-parse-str* pos0 pos1)
				   list))
		  (setq *regexp-parse-index* pos1)
		  (regexp-parse-charset1 op list)))
	  (regexp-error)))
    (regexp-error))
  )
	  
;;;
;;; regexp-inverse
;;;

(defun string-reverse (str)
  (if (<= (length str) 1) str
    (let ((result (make-string (length str) 0))
	  (i 0)
	  (j (1- (length str))))
      (while (<= 0 j)
	(aset result i (aref str j))
	(setq i (1+ i)
	      j (1- j)))
      result)))
      
(defun regexp-inverse (form)
  (cond((stringp form) (string-reverse form))
       ((symbolp form) form)
       ((numberp form) form)
       (t 
	(cond((eq (car form) ':mark)
	      (list ':mark (car(cdr form))
		    (regexp-inverse (car(cdr(cdr form))))))
	     ((eq (car form) 'DUPLICATE)
	      form)
	     ((eq (car form) ':or)
	      (cons ':or (mapcar 'regexp-inverse (cdr form))))
	     ((eq (car form) ':seq)
	      (cons ':seq (reverse (mapcar 'regexp-inverse (cdr form)))))
	     ((eq (car form) ':optional)
	      (list ':optional (regexp-inverse (car(cdr form)))))
	     ((eq (car form) ':star)
	      (list ':star (regexp-inverse (car (cdr form)))))
	     ((eq (car form) ':plus)
	      (list ':plus (regexp-inverse (car (cdr form)))))
	     (t form)))))

;;;
;;; regexp-dump
;;;

(defun regexp-dump-0 (op)
  (princ op) (terpri) (setq i (1+ i)))

(defun regexp-dump-1 (op)
  (princ (format "%s %d\n" op (aref str (1+ i))))
  (setq i (+ i 2)))

(defun regexp-dump-1ch (op)
  (princ (format "%s %c\n" op (aref str (1+ i))))
  (setq i (+ i 2)))


(defun regexp-get-absolute-address (point b1 b2)
  (cond ((< b2 128)
	 (+ point (+ (* 256 b2) b1)))
	(t
	 (+ point (logior (logxor -1 (+ (* 255 256) 255)) (* 256 b2) b1)))))
  

(defun regexp-dump-jump (op)
  (let* ((b1 (aref str (1+ i)))
	 (b2 (aref str (+ i 2)))
	(p (regexp-get-absolute-address (+ i 3) b1 b2)))
    (princ (format "%s %d\n" op p)))
  (setq i (+ i 3)))

(defun regexp-dump-charset (op)
  (princ (format "%s %d " op (aref str (1+ i))))
  (let ((j (+ i 2)) (max (+ i 2 (aref str (1+ i)))))
    (while (< j max)
      (princ (format "%2x" (aref str j)))
      (setq j (1+ j)))
    (setq i j))
  (terpri)
  )
  
(defun regexp-dump-charset-m (op)
  (princ (format "%s %d %d " op (aref str (1+ i))
		 (aref str (+ i 2))))
  (let ((j (+ i 3)) (max (+ i 3 (aref str (+ i 2)))))
    (while (< j max)
      (princ (format "%2x" (aref str j)))
      (setq j (1+ j)))
    (setq i j))
  (terpri)
  )


(defun regexp-dump (str)
  (terpri)
  (let ((i 0)
	(max (length str)))
    (while (< i max)
      (princ (format "%4d:" i))
      (let((op (aref str i)))
	(cond((= op UNUSED) (regexp-dump-0 "unused"))
	     ((= op EXACTN) 
	      (princ (format "exactn(%d) " (aref str (1+ i))))
	      (let ((j (+ i 2)) (max (+ i 2 (aref str (1+ i)))))
		(while (< j max)
		  (princ (format "%c" (aref str j)))
		  (setq j (1+ j)))
		(setq i j))
	      (terpri)
	      )
	     ((= op BEGLINE) (regexp-dump-0 "begline"))
	     ((= op ENDLINE) (regexp-dump-0 "endline"))
	     ((= op JUMP) (regexp-dump-jump "jump"))
	     ((= op ON_FAILURE_JUMP ) (regexp-dump-jump "on_failure_jump"))
	     ((= op MAYBE_FINALIZE_JUMP) (regexp-dump-jump "maybe_finalize_jump"))
	     ((= op DUMMY_FAILURE_JUMP) (regexp-dump-jump "dummy_failure_jump"))
	     ((= op ANYCHAR) (regexp-dump-0 "anychar"))
	     ((= op CHARSET) (regexp-dump-charset "charset"))
	     ((= op CHARSET_NOT) (regexp-dump-charset "charset_not"))
	     ((= op START_MEMORY) (regexp-dump-1 "start_memory"))
	     ((= op STOP_MEMORY) (regexp-dump-1 "stop_memory"))
	     ((= op DUPLICATE) (regexp-dump-1 "duplicate"))
	     ((= op BEFORE_DOT) (regexp-dump-0 "before_dot"))
	     ((= op AT_DOT) (regexp-dump-0 "at_dot"))
	     ((= op AFTER_DOT) (regexp-dump-0 "after_dot"))
	     ((= op BEGBUF) (regexp-dump-0 "begbuf"))
	     ((= op ENDBUF) (regexp-dump-0 "endbuf"))
	     ((= op WORDCHAR) (regexp-dump-0 "wordchar"))
	     ((= op NOTWORDCHAR) (regexp-dump-0 "notwordchar"))
	     ((= op WORDBEG) (regexp-dump-0 "wordbeg"))
	     ((= op WORDEND) (regexp-dump-0 "wordend"))
	     ((= op WORDBOUND) (regexp-dump-0 "wordbound"))
	     ((= op NOTWORDBOUND) (regexp-dump-0 "notwordbound"))
	     ((= op SYNTAXSPEC) (regexp-dump-1 "syntaxspec"))
	     ((= op NOTSYNTAXSPEC) (regexp-dump-1 "notsyntaxspec"))
	     ((= op EXACT1) (regexp-dump-1ch "exact1"))
	     ((= op EXACT2)
	      (princ (format "exact2 %c%c\n" (aref str (1+ i))
			     (aref str (+ i 2))))
	      (setq i (+ i 3)))
	     ((= op EXACT3)
	      (princ (format "exact3 %c%c%c\n" 
			     (aref str (1+ i))
			     (aref str (+ i 2))
			     (aref str (+ i 3))))
	      (setq i (+ i 4)))
	     ((= op CHARSET_M) (regexp-dump-charset-m "CHARSET_M"))
	     ((= op CHARSET_M_NOT) (regexp-dump-charset-m "CHARSET_M_NOT"))
	     ((= op CASE)
	      (princ (format "case %d\n" (aref str (1+ i))))
	      (let ((j (+ i 2)) (max (+ i 2 (* 2 (aref str (1+ i))))))
		(while (< j max)
		  (princ (format "[%d]::%d\n" (1+ (/ (- j (+ i 2)) 2))
				  (regexp-get-absolute-address
				   (+ j 2) (aref str j)
				   (aref str (1+ j)))))
		  (setq j (+ j 2)))
		(let ((ch (aref str j)) (chmax (aref str (1+ j))))
		  (princ (format "%c::%c\n" ch chmax))
		  (setq j (+ j 2))
		  (while (<= ch chmax)
		    (princ (format "%c=>[%d]\n" ch (aref str j)))
		    (setq j (1+ j))
		    (setq ch (1+ ch))))
		(setq i j)))
	     ((= op SUCCESS_SHORT) (regexp-dump-0 "SUCCESS_SHORT"))
	     ((= op SUCCESS) (regexp-dump-0 "SUCCESS"))
	     ((= op POP) (regexp-dump-0 "POP"))
	     ((= op EXECPT0) (regexp-dump-0 "EXECPT0"))
	     ((= op EXECPT1) (regexp-dump-1ch "EXECPT1"))
	     ((= op CATEGORYSPEC) (regexp-dump-1 "categoryspec"))
	     ((= op NOTCATEGORYSPEC) (regexp-dump-1 "notcategoryspec"))
	     (t (princ (format "unknown op=%d\n" op)) (setq i (1+ i))))))
    (princ (format "%4d:\n" i)))
  nil
 )

;;;
;;;  regexp-*-lessp
;;;

;;; nil < number < string < symbol < cons

(defun regexp-lessp (exp1 exp2)
  (cond((equal exp1 exp2)
	nil)
       ((null exp1) t)
       ((numberp exp1)
	(cond((null exp2) nil)
	     ((numberp exp2)
	      (< exp1 exp2))
	     (t t)))
       ((stringp exp1)
	(cond((or (null exp2)
		  (numberp exp2))
	      nil)
	     ((stringp exp2)
	      (string< exp1 exp2))
	     (t t)))
       ((symbolp exp1)
	(cond((or (null exp2)
		  (numberp exp2)
		  (stringp exp2))
	      nil)
	     ((symbolp exp2)
	      (string< exp1 exp2))
	     (t t)))
       ((consp exp1)
	(cond ((not (consp exp2))
	       nil)
	      ((< (length exp1) (length exp2))
	       t)
	      ((= (length exp1) (length exp2))
	       (regexp-lessp-list exp1 exp2)
	       )
	      (t nil)))))

(defun regexp-lessp-list (exp1 exp2)
  (cond((null exp1) nil)
       ((regexp-lessp (car exp1) (car exp2))
	t)
       ((equal (car exp1) (car exp2))
	(regexp-lessp-list (cdr exp1) (cdr exp2)))
       (t nil)))

;;; item = list of regexps
;;; nil < cons

(defun regexp-item-lessp (item1 item2)
  (cond((equal item1 item2)
	nil)
       ((null item2) t)
       ((consp item1)
	(cond((consp item2)
	      (cond ((regexp-key-lessp (car item1) (car item2))
		     t)
		    ((equal (car item1) (car item2))
		     (regexp-item-lessp (cdr item1) (cdr item2)))
		    (t nil))
	      )
	     (t nil)))))

(defun regexp-alist-lessp (pair1 pair2)
  (regexp-key-lessp (car pair1) (car pair2)))

(defvar *regexp-key-class0* '(START_MEMORY STOP_MEMORY))
(defvar *regexp-key-class1* '(BEGLINE ENDLINE 
				    BEFORE_DOT AT_DOT AFTER_DO 
				    BEGBUF ENDBUF 
				    WORDBEG WORDEND WORDBOUND NOTWORDBOUND))
(defvar *regexp-key-class2* '(ANYCHAR WORDCHAR NOTWORDCHAR))
(defvar *regexp-key-class3* '(DUPLICATE SYNTAXSPEC NOTSYNTAXSPEC CATEGORYSPEC NOTCATEGORYSPEC))
                             ;;; CHARSET CHARSET_NOT

(defun regexp-key-class0 (key)
  (and (consp key) (memequal (car key) *regexp-key-class0*)))

(defun regexp-key-class1 (key)
  (and (symbolp key)
	      (memequal key *regexp-key-class1*)))

(defun regexp-key-class2 (key)
  (and (symbolp key) (memequal key *regexp-key-class2*)))

(defun regexp-key-class3 (key)
  (and (consp key)
       (memequal (car key) *regexp-key-class3*)))

(defun regexp-key-class4 (key)
  (numberp key))

(defun regexp-key-lessp-list (sym1 sym2 list)
  (< (find sym1 list) (find sym2 list)))

(defun regexp-key-lessp (key1 key2)
  (cond ((regexp-key-class0 key1)
	 (cond((regexp-key-class0 key2)
	       (regexp-key-lessp-list (car key1) (car key2) *regexp-key-class0*))
	      (t t)))
	((regexp-key-class1 key1)
	 (cond((regexp-key-class1 key2)
	       (regexp-key-lessp-list key1 key2 *regexp-key-class1*))
	      ((or (regexp-key-class2 key2)
		   (regexp-key-class3 key2)
		   (regexp-key-class4 key2)
		   (null key2)
		   ) t)))
	((regexp-key-class2 key1)
	 (cond((regexp-key-class2 key2)
	       (regexp-key-lessp-list key1 key2 *regexp-key-class2*))
	      ((or (regexp-key-class3 key2)
		   (regexp-key-class4 key2)
		   (null key2)
		   )t)))
	((regexp-key-class3 key1)
	 (cond((regexp-key-class3 key2)
	       (regexp-key-lessp-list (car key1) (car key2) *regexp-key-class3*))
	      ((or (regexp-key-class4 key2)
		   (null key2)) t)))
	((regexp-key-class4 key1)
	 (or (null key2)
	     (and (regexp-key-class4 key2) (< key1 key2))))
	(t nil)))

(defvar *regexp-sort-flag* t)
(defvar *regexp-debug* nil)

(defun regexp-sort (list pred)
  (if *regexp-sort-flag* 
      (progn
	(if *regexp-debug* (princ (format "(regexp-sort %s %s)\n" list pred)))
	(let ((result (sort list pred)))
	  (if *regexp-debug* (princ (format "<== %s\n" result)))
	  result))
    list))

;;;
;;; regexp-expand
;;;

(defvar *regexp-ignore-mark-and-duplicate* nil)

(defun regexp-expand (regexp)
  ;;; returns island
  (cond((symbolp regexp) (list (list regexp)))
       ((numberp regexp) (list (list regexp)))
       ((stringp regexp)
	(let ((result nil))
	  (let ((i 0) (max (length regexp)))
	    (while (< i max)
	      (setq result (cons (aref regexp i) result))
	      (setq i (1+ i)))
	    (list (reverse result)))))
       (t
	(cond ((eq (car regexp) 'CHARSET)
	       (regexp-expand-charset t (cdr regexp)))
	      ((eq (car regexp) 'CHARSET_NOT)
	       (regexp-expand-charset nil (cdr regexp)))
	      ((eq (car regexp) ':or)
	       (regexp-expand-or (cdr regexp)))
	      ((eq (car regexp) ':seq)
	       (regexp-expand-seq (cdr regexp)))
	      ((eq (car regexp) ':star)
	       (append  (regexp-expand-seq (list (car(cdr regexp)) regexp)) '(nil)))
	      ((eq (car regexp) ':plus)
	       (regexp-expand-seq (list (car(cdr regexp)) (list ':star (car(cdr regexp))))))
	      ((eq (car regexp) ':optional)
	       (cons nil
		     (regexp-expand (car(cdr regexp)))))
	      ((eq (car regexp) ':mark)
	       (if *regexp-ignore-mark-and-duplicate*
		   (regexp-expand (nth 2 regexp))
		 (list (list (list 'START_MEMORY (car (cdr regexp)))
			     (car (cdr (cdr regexp)))
			     (list 'STOP_MEMORY  (car (cdr regexp)))))))
	      ((and *regexp-ignore-mark-and-duplicate* (eq (car regexp) ':duplicate))
	       (regexp-expand-seq nil))
	      (t (list (list regexp)))))))

(defconst charset-vector (make-vector 256 nil))


(defun string-to-list (str)
  (let ((result nil)(i 0) (max (length str)))
    (while (< i max)
      (setq result (cons (aref str i) result))
      (setq i (1+ i)))
    (reverse result)))

(defun regexp-expand-charset-set-mark (chars alist)
  (if (null chars) t
    (let ((place (assoc (car chars) alist)))
      (cond((null place)
	    (cons 
	     (cons (car chars) (regexp-expand-charset-set-mark (cdr chars) nil))
	     alist))
	   (t
	    (setcdr place (regexp-expand-charset-set-mark (cdr chars) (cdr place)))
	    alist)))))

(defun regexp-expand-charset (mode charsets)
  (let ((i 0))
    (while (< i 256)
      (aset charset-vector i nil)
      (setq i (1+ i))))
  (while charsets
    (cond((numberp (car charsets))
	  (aset charset-vector (car charsets) t))
	 ((stringp (car charsets))
	  (if (= (length (car charsets)) 1)
	      (aset charset-vector (aref (car charsets) 0) t)
	    (let ((list (string-to-list (car charsets))))
	      (aset charset-vector (car list)
		    (regexp-expand-charset-set-mark (cdr list)
						    (aref charset-vector (car list)))))))
	 ((and (consp (car charsets))
	       (eq (car (car charsets)) ':range))
	  (let ((from (aref (nth 1 (car charsets)) 0))
		(to   (aref (nth 2 (car charsets)) 0)))
	    (if (<= from to)
		(if (< to 128)
		    (let ((char from))
		      (while (<= char to)
			(aset charset-vector char t)
			(setq char (1+ char))))
		  (let ((from-list (string-to-list (nth 1 (car charsets))))
			(to-list   (string-to-list (nth 2 (car charsets)))))
		    

		    ))))))
    (setq charsets (cdr charsets)))
  (let ((result nil)
	(i 0))
    (while (< i 256)
      (if (eq (aref charset-vector i) mode)
	  (setq result (cons (list i) result)))
      (setq i (1+ i)))
    (reverse result)))

(defun regexp-expand-or (regexps)
  (let ((result nil)
	(list (reverse regexps)))
    (while list
      (setq result (append (regexp-expand (car list)) result))
      (setq list (cdr list)))
    result))

(defun regexp-expand-items (items)
  (let ((result nil))
    (while items
      (setq result (append result (regexp-expand-seq (car items))))
      (setq items (cdr items)))
    result))

;;; regexp
;;; item == list of regexps
;;; island == list of items
;;; closure == island and alist

(defun regexp-expand-seq (item)
  (if (null item)
      (list nil)
    (let ((result (regexp-expand (car item))))
      (if (memequal nil result)
	  (let ((newresult (regexp-expand-seq (cdr item))))
	    (setq result (delete nil result))
	    (while result
	      (setq newresult (cons (append (car result) (cdr item))
				    newresult))
	      (setq result (cdr result)))
	    newresult)
	(let ((newresult nil))
	  (while result
	    (setq newresult (cons (append (car result) (cdr item))
				  newresult))
	    (setq result (cdr result)))
	  newresult)))))

;;;
;;; regexp-
;;;

(defun regexp-make-island (items)
  (regexp-sort (delete-duplicate (regexp-expand-items items))
	'regexp-item-lessp))

(defvar *regexp-number-to-alist* nil)
(defvar *regexp-island-to-number* nil)

;;;
;;; Table structure
;;;  (key . next)
;;;  (key .(then else))
;;;; (:true . next )

(defun regexp-make-table (regexp)
  (let ((*regexp-number-to-alist* nil)
	(*regexp-island-to-number*  nil)
	(*regexp-counter* 0))
    (let ((island (regexp-sort (delete-duplicate (regexp-expand regexp))
			'regexp-item-lessp)))
      (regexp-make-closure island)
      (reverse *regexp-number-to-alist*))))

;;; class2 , class3, class4 only
(defun regexp-make-pre-alist (items)
  (let ((pre-alist nil))
    (while items
      (cond((or (regexp-key-class2 (car (car items)))
		(regexp-key-class3 (car (car items))))
	    (let ((key (car (car items)))
		  (newitems nil))
	      (while (and items (equal key (car (car items))))
		(setq newitems (cons (cdr (car items)) newitems))
		(setq items (cdr items)))
	      (setq newitems (reverse newitems))
	      (setq pre-alist (cons (cons key newitems) pre-alist))))
	   ((regexp-key-class4 (car (car items)))
	    (let((alist nil))
	      (while (and items (regexp-key-class4 (car (car items))))
		(let ((place (assoc (car (car items)) alist)))
		  (if place
		      (setcdr place
			      (cons (cdr (car items)) (cdr place)))
		    (setq alist (cons (cons (car (car items)) (list(cdr (car items)))) alist))))
		(setq items (cdr items)))
	      (let ((list alist))
		(while list
		  (setcdr (car list) (reverse (cdr (car list))))
		  (setq list (cdr list)))
		(setq pre-alist (append alist pre-alist))
		)))
	   ((null (car (car items)))
	    (setq pre-alist (cons '(nil) pre-alist))
	    (setq items (cdr items)))
	   (t 
	      (setq items (cdr items)))))
    (reverse pre-alist)))

(defun regexp-make-pre-alist-to-island (pre-alist)
  (let ((island nil))
    (while pre-alist
      (cond((equal '(nil) (car pre-alist))
	    (setq island (cons nil island)))
	   (t
	    (let ((key (car (car pre-alist)))
		  (items (cdr (car pre-alist))))
	      (while items
		(setq island (cons (cons key (car items)) island))
		(setq items (cdr items))))))
      (setq pre-alist (cdr pre-alist)))
    (reverse island)))

(defun regexp-find-items (items pred)
  (let ((no 0))
    (while (and items (not (funcall pred (car (car items)))))
      (setq no (1+ no))
      (setq items (cdr items)))
    (if items
	no
      nil)))

(defun regexp-make-closure (island)
;;;
  (if *regexp-debug*  (princ (format "regexp-make-closure %s\n" island)))
  (if (null island) nil
    (let ((place (assoc island *regexp-island-to-number*)))
      (cond(place
	    (cdr place))
	   ((regexp-find-items island 'regexp-key-class0)
	    (let ((items island) (result nil) (key nil))
	      (while (and items (not (regexp-key-class0 (car (car items)))))
		(setq result (cons (car items) result))
		(setq items (cdr items)))
	      (setq key (car (car items)))
	      (setq result (append (reverse result) (list (cdr (car items))) (cdr items)))
	      (let* ((number (setq *regexp-counter* (1+ *regexp-counter*)))
		     (pair (cons key nil))
		     (alist (list pair))
		     (place (cons number alist)))
		(setq *regexp-island-to-number*
		      (cons (cons island number) *regexp-island-to-number*))
		(setq *regexp-number-to-alist* (cons place *regexp-number-to-alist*))
		(setcdr pair 
			(regexp-make-closure 
			 (regexp-make-island  result)))
		number)))
	   ((regexp-find-items island 'regexp-key-class1)
	    (let ((items island) (result-true nil) (result-false nil) (key nil))
	      (while (and items (not (regexp-key-class1 (car (car items)))))
		(setq result-true (cons (car items) result-true))
		(setq result-false (cons (car items) result-false))
		(setq items (cdr items)))
	      (setq key (car (car items)))
	      (setq result-true (cons (cdr (car items)) result-true))
	      (setq items (cdr items))
	      (while items
		(if (equal key (car (car items)))
		    (setq result-true (cons (cdr (car items)) result-true))
		  (setq result-true (cons (car items) result-true)
			result-false (cons (car items) result-false)))
		(setq items (cdr items)))
	      (setq result-true (reverse result-true)
		    result-false (reverse result-false))
	      (let* ((number (setq *regexp-counter* (1+ *regexp-counter*)))
		     (pair (cons key nil))
		     (alist (list pair))
		     (place (cons number alist)))
		(setq *regexp-island-to-number*
		      (cons (cons island number) *regexp-island-to-number*))
		(setq *regexp-number-to-alist* (cons place *regexp-number-to-alist*))
		(setcdr pair 
			(cons (regexp-make-closure (regexp-make-island result-true))
			      (regexp-make-closure (regexp-make-island result-false))))
		number)))
	   (t
	    (regexp-make-closure* (regexp-make-pre-alist island)))))))

(defun regexp-make-closure* (pre-alist)
  (let* ((pre-alist pre-alist)
	 (alist nil))
    (cond
     ((and(consp (car pre-alist))
	  (or (regexp-key-class2 (car (car pre-alist)))
	      (regexp-key-class3 (car (car pre-alist)))))
      (let* ((number (setq *regexp-counter* (1+ *regexp-counter*)))
	     (pair (cons (car (car pre-alist)) nil))
	     (alist (list pair))
	     (place (cons number alist)))
	(setq *regexp-island-to-number*
	      (cons (cons island number) *regexp-island-to-number*))
	(setq *regexp-number-to-alist* (cons place *regexp-number-to-alist*))
	(let ((then (regexp-make-closure 
		     (regexp-make-island (cdr (car pre-alist)))))
	      (else (regexp-make-closure 
		     (regexp-make-island (regexp-make-pre-alist-to-island (cdr pre-alist))))))
	  (setcdr pair 
		  (if (cdr pre-alist)
		      (list then else)
		    (list then else))))
	number))
     (t
      (while (and pre-alist
		  (or (null (car (car pre-alist)))
		      (numberp (car (car pre-alist)))))
	(setq alist (cons (car pre-alist) alist))
	(setq pre-alist (cdr pre-alist)))
      (let* ((number (setq *regexp-counter* (1+ *regexp-counter*)))
	     (place (cons number alist)))
	(setq *regexp-island-to-number*
	      (cons (cons island number) *regexp-island-to-number*))
	(setq *regexp-number-to-alist* (cons place *regexp-number-to-alist*))
	(if pre-alist
	    (setq alist (cons(cons ':true
				   (regexp-make-closure* pre-alist)) alist)))
	(setq alist (reverse alist))
	(setq alist (regexp-sort alist 'regexp-alist-lessp))
	(setcdr place alist)
	(while alist
	  (let ((pair (car alist)))
	    (cond ((and (car pair)
			(not (eq (car pair) ':true)))
		   (setcdr pair
			   (regexp-make-closure (regexp-make-island (cdr pair)))))))
	  (setq alist (cdr alist)))
	number)))))



(defun regexp-compile-string (str)
  (regexp-make-table (regexp-parse str)))

(defun regexp-compile-forward (str)
  (regexp-make-table (regexp-parse str)))

(defun regexp-compile-backward (str)
  (regexp-make-table (regexp-inverse (regexp-parse str))))

;; 92.5.16 by K.Handa -- buffer name preceded by space.
(defvar *regexp-code-buffer* (get-buffer-create " *regexp-code-buffer*"))

(defvar *regexp-dump* nil)

(defun regexp-code-gen (table)
  (let ((*table* table) (*labels* nil) (*final* nil) (*counter* 0))
    (let ((list table))
      (while (and list (null *final*))
	(if (equal '((nil)) (cdr (car list)))
	    (setq *final* (car (car list))))
	(setq list (cdr list))))
    (cond((null *final*)
	  (setq *final* (1+ (length table)))
	  (setq *counter* (1+ *final*)))
	 (t 
	  (setq *counter* (1+ (length table)))))
    (save-excursion
      (set-buffer *regexp-code-buffer*)
      (let ((kanji-flag nil)
	    (mc-flag nil))
	(erase-buffer)
	(regexp-code-gen* 1)
	(buffer-substring (point-min) (point-max)))
      )))

(defvar *regexp-case-table* (make-vector 256 nil))

(defun regexp-code-gen* (node)
  (cond((= node *final*)
	(if (null (assoc node *labels*))
	    (setq *labels* (cons (cons node (point)) *labels*)))
	(insert SUCCESS))
       ((null (assoc node *labels*))
	(setq *labels* (cons (cons node (point)) *labels*))
	(let ((alist (cdr (assoc node *table*))))
	  (cond((equal '((nil)) alist)
		(insert SUCCESS))
	       (t (regexp-code-gen-alist alist)))))
       (t
	(let ((disp (- (cdr (assoc node *labels*)) (+ (point) 3))))
	  (insert JUMP
		  (logand disp 255)
		  (/ (logand disp (* 255 256)) 256))))))

(defun regexp-code-gen-alist (alist)
  (if (eq (car (car alist)) nil)
      nil
    (let ((nextalist alist)
	  (numberkey nil)
	  (point nil)
	  (min 256) (max -1) (nexts nil) (nodealist nil))
      (cond((numberp (car (car alist)))
	    (setq numberkey t)
	    (let ((i 0))
	      (while (< i 256)
		(aset *regexp-case-table* i nil)
		(setq i (1+ i))))

	    (while (and nextalist
			(numberp (car (car nextalist))))
	      (let ((ch (car (car nextalist)))
		    (next (cdr (car nextalist))))
		(let ((place (assoc next nodealist)))
		  (if place
		      (setcdr place
			      (cons ch (cdr place)))
		    (setq nodealist (cons (cons ch (list next)) nodealist))))
		(aset *regexp-case-table* ch next)
		(if (< ch min) (setq min ch))
		(if (< max ch) (setq max ch))
		(if (not (memequal next nexts))
		    (setq nexts (cons next nexts)))
		)
	      (setq nextalist (cdr nextalist))))
	   (t (setq nextalist (cdr alist))))

      (if nextalist
	  (cond((eq (car (car nextalist)) nil)
		(insert SUCCESS_SHORT))
	       (t
		(insert ON_FAILURE_JUMP 0 0)
		(setq point (point)))))

      (cond(numberkey
	    (cond((= min max)
                ;;; exact1
		  (regexp-code-gen-exact (list min) (car nexts)))

		 ((= (length nexts) 1)
                ;;; charset or charset_not
		  (if (= (length alist) 256)
		      (insert EXECPT0)	;92.10.26 by T.Saneto
		    (let ((not_min 256)
			  (not_max -1)
			  (ch 0)
			  (mode (car nexts)))
		      (while (< ch 256)
			(cond((null (aref *regexp-case-table* ch))
			      (if (< ch not_min) (setq not_min ch))
			      (if (< not_max ch) (setq not_max ch))))
			(setq ch (1+ ch)))
		      (if (<= (- not_max not_min) (- max min))
			  (setq min not_min
				max not_max
				mode nil))
		      (let ((minb (/ min 8))
			    (maxb (1+ (/ max 8))))
			(insert (if mode CHARSET_M CHARSET_M_NOT) minb (- maxb minb))
			(let ((b minb))
			  (while (< b maxb)
			    (let ((i 7) (bits 0))
			      (while (<= 0 i)
				(setq bits (* bits 2))
				(if (eq (aref *regexp-case-table* (+ (* 8 b) i))
					mode)
				    (setq bits (1+ bits)))
				(setq i (1- i)))
			      (insert bits))
			    (setq b (1+ b)))))))
		  (regexp-code-gen* (car nexts)))
		 (t
                ;;; case
		  (let ((point nil))
		    (insert CASE)
		    (insert (length nexts))
		    (setq point (point))
		    (let ((list nexts))
		      (while list
			(insert 0 0)
			(setq list (cdr list))))
		    (insert min max)
		    (let ((ch min))
		      (while (<= ch max)
			(if (aref *regexp-case-table* ch)
			    (insert (1+ (find (aref *regexp-case-table* ch) nexts)))
			  (insert 0))
			(setq ch (1+ ch))))
		    (let ((list nexts))
		      (while list
			(if (null (assoc (car list) *labels*))
			    (regexp-code-gen* (car list)))
			(setq list (cdr list))))
		    (save-excursion
		      (goto-char point)
		      (let ((list nexts))
			(while list
			  (delete-char 2)
			  (let ((disp (- (cdr (assoc (car list) *labels*)) (+ (point) 2))))
			    (insert (logand disp 255)
				    (/ (logand disp (* 255 256)) 256)))
			  (setq list (cdr list)))))
		    ))))
	   ((eq (car (car alist)) ':true)
	    (regexp-code-gen* (cdr (car alist))))
	   (t
	    (let ((key (car (car alist)))
		  (next (cdr (car alist))))
	      (let ((point nil))
		(cond((and (consp next)
			   (or (numberp (cdr next))
			       (and (consp (cdr next))
				    (numberp (car (cdr next))))))
		      (insert ON_FAILURE_JUMP 0 0)
		      (setq point (point))))

		(cond ((symbolp key)
		       (insert (eval key)))
		      (t
		       (insert (eval (car key)) (car (cdr key)))))
		(cond((numberp next)
		      (regexp-code-gen* next))
		     (point
		      (let ((else))
			(cond((numberp (cdr next))
			      (insert POP)
			      (setq else (cdr next)))
			     (t
			      (setq else (car (cdr next)))))
			(regexp-code-gen* (car next))
			(save-excursion
			  (let ((disp (- (point) point)))
			    (goto-char point)
			    (delete-char -2)
			    (insert (logand disp 255)
				    (/ (logand disp (* 255 256)) 256))
			    ))
			(regexp-code-gen* else)))
		     (t
		      (regexp-code-gen* (car next))))))))
      (if point
	  (let ((disp (- (point) point)))
	    (save-excursion
	      (goto-char point)
	      (delete-char -2)
	      (insert (logand disp 255)
		      (/ (logand disp (* 255 256)) 256)))
	    (regexp-code-gen-alist nextalist))))))

(defun regexp-code-gen-exact (chars node)
  (let ((alist (cdr (assoc node *table*))))
    (cond((and (null (assoc node *labels*))
	       (= (length alist) 1)
	       (numberp (car (car alist))))
	  (regexp-code-gen-exact (cons (car (car alist)) chars)
				 (cdr (car alist))))
	 (t
	  (regexp-code-gen-exact* (reverse chars))
	  (regexp-code-gen* node)))))
    
(defun regexp-code-gen-exact* (chars)
  (cond((= (length chars) 1)
	(insert EXACT1 (car chars)))
       ((= (length chars) 2)
	(insert EXACT2 (car chars) (car (cdr chars))))
       ((= (length chars) 3)
	(insert EXACT3 (car chars) (car (cdr chars)) (car (cdr (cdr chars)))))
       (t
	(insert EXACTN (length chars))
	(let ((list chars))
	  (while list
	    (insert (car list))
	    (setq list (cdr list)))))))

(defun regexp-compile-regexp (regexp &optional *regexp-sort-flag*)
  (regexp-code-gen (regexp-make-table regexp)))

(defun regexp-compile-pattern (pattern &optional *regexp-sort-flag*)
  (regexp-code-gen (regexp-compile-string pattern)))

(defun regexp-compile-pattern-bothward (pattern &optional *regexp-sort-flag*)
  (let ((regexp (regexp-parse pattern)))
    (list
     (regexp-code-gen (regexp-make-table regexp))
     (let ((START_MEMORY STOP_MEMORY)
	   (STOP_MEMORY START_MEMORY))
       (regexp-code-gen (regexp-make-table (regexp-inverse regexp)))))))

(defun regexp-compile-pattern-forward (pattern &optional *regexp-sort-flag*)
  (regexp-code-gen (regexp-compile-forward pattern)))

(defun regexp-compile-pattern-backward (pattern &optional *regexp-sort-flag*)
  (let ((START_MEMORY STOP_MEMORY)
        (STOP_MEMORY START_MEMORY))
    (regexp-code-gen (regexp-compile-backward pattern))))

(defun regexp-split (regexp)
  (cond((symbolp regexp) 
	(list (list nil (list regexp)) (list (list regexp) nil)))
       ((numberp regexp)
	(list (list nil (list regexp)) (list (list regexp) nil)))
       ((stringp regexp)
	(let ((result nil))
	  (let ((i 0) (max (length regexp)))
	    (while (< i max)
	      (setq result (cons (aref regexp i) result))
	      (setq i (1+ i))))
	  (regexp-split-seq (reverse result))))
       ((memequal (car regexp) '(CHARSET CHARSET_NOT))
	(list (list nil (list regexp)) (list (list regexp) nil)))
       ((eq (car regexp) ':or)
	(regexp-split-or (cdr regexp)))
       ((eq (car regexp) ':seq)
	(regexp-split-seq (cdr regexp)))
       ((eq (car regexp) ':star)
	(let ((result (list (list (list regexp) (list regexp))))
	      (splits (regexp-split (nth 1 regexp))))
	  (while splits
	    (if (and (nth 0 (car splits)) (nth 1 (car splits)))
		(setq result
		      (cons (list (cons regexp (nth 0 (car splits)))
				  (append (nth 1 (car splits)) (list regexp)))
			    result)))
	    (setq splits (cdr splits)))
	  (delete-duplicate result)))
       ((eq (car regexp) ':plus)
	(let ((result nil)
	      (splits (regexp-split (nth 1 regexp))))
	  (while splits
	    (setq result
		  (cons (list (append (list (list ':star (nth 1 regexp)))
				      (nth 0 (car splits)))
			      (append (nth 1 (car splits))
				      (list (list ':star (nth 1 regexp)))))
			result))
	    (setq splits (cdr splits)))
	  (delete-duplicate result)))
       ((eq (car regexp) ':optional)
	(let ((result (list (list nil nil))))
	  (setq result (cons (list nil nil)
			     (regexp-split (nth 1 regexp))))
	  (delete-duplicate result)))
       ((eq (car regexp) ':mark)
	(regexp-split (nth 2 regexp)))
       ((eq (car regexp) ':duplicate)
	(list (list nil nil)))
       (t
	(list (list (list regexp) nil)
	      (list nil (list regexp)))
	)))

(defun regexp-split-seq (regexps)
  (if (null regexps)
      (list (list nil nil))
    (let ((result nil) (pre nil)(splits nil))
      (while regexps
	(setq result
	      (append (list
		       (list (append pre (list (car regexps))) (cdr regexps))
		       (list (append pre regexps)))
		      result))
	(setq splits (regexp-split (car regexps)))
	(while splits
	  (if (and (nth 0 (car splits)) (nth 1 (car splits)))
	      (setq result
		    (cons (list (append pre (car (car splits)))
				(append (nth 1 (car splits)) (cdr regexps)))
			  result)))
	  (setq splits (cdr splits)))
	(setq pre (append pre (list (car regexps))))
	(setq regexps (cdr regexps))
	)
      (delete-duplicate result))))

(defun regexp-split-or (regexps)
  (let ((result nil))
    (while regexps
      (setq result (append (regexp-split (car regexps)) result))
      (setq regexps (cdr regexps)))
    result))

(defun regexp-equal (regexp1 regexp2)
  (or(and (stringp regexp1)
	  (= (length regexp1) 1)
	  (numberp regexp2)
	  (eq (aref regexp1 0) regexp2))
     (and (stringp regexp2)
	  (= (length regexp2) 1)
	  (numberp regexp1)
	  (eq (aref regexp2 0) regexp1))
     (and (consp regexp1)
	  (consp regexp2)
	  (= (length regexp1) (length regexp2))
	  (regexp-equal-list regexp1 regexp2))
     (equal regexp1 regexp2)))

(defun regexp-equal-list (regexp1 regexp2)
  (or (null regexp1)
      (and (regexp-equal (car regexp1) (car regexp2))
	   (regexp-equal-list (cdr regexp1) (cdr regexp2)))))

(defun regexp-normalize-plus (regexp)
  (cond((null regexp) regexp)
       ((numberp regexp) regexp)
       ((stringp regexp) regexp)
       ((symbolp regexp) regexp)
       ((memequal (car regexp) '(CHARSET CHARSET_NOT)) regexp)
       ((eq (car regexp) ':or)
	(cond((= (length regexp)1) nil)
					;92.10.26 by T.Saneto
	     ((= (length regexp)2) (regexp-normalize-plus (nth 1 regexp)))
	     (t
	      (cons ':or  (mapcar 'regexp-normalize-plus (cdr regexp))))))
       ((eq (car regexp) ':seq)
	(cond((= (length regexp) 1) nil)
	     ((= (length regexp) 2) (regexp-normalize-plus (nth 1 regexp)))
	     (t
	      (let ((list (mapcar 'regexp-normalize-plus (cdr regexp)))
		    (result nil))
		(while list
		  (cond ((and (consp (nth 0 list))
			      (eq (nth 0 (nth 0 list)) ':star)
			      (regexp-equal (nth 1 (nth 0 list)) (nth 1 list)))
			 (setq result (cons (list ':plus (nth 1 list)) result))
			 (setq list (cdr (cdr list))))
			((and (consp (nth 1 list))
			      (eq (nth 0 (nth 1 list)) ':star)
			      (regexp-equal (nth 0 list) (nth 1 (nth 1 list))))
			 (setq result (cons (list ':plus (nth 0 list)) result))
			 (setq list (cdr (cdr list))))
			(t
			 (setq result (cons (car list) result))
			 (setq list (cdr list)))))
		(if (= (length result) 1)
		    (nth 0 result)
		  (cons ':seq (reverse result)))))))
       ((eq (car regexp) ':star) (list ':star (regexp-normalize-plus (nth 1 regexp))))
       ((eq (car regexp) ':plus) (list ':plus (regexp-normalize-plus (nth 1 regexp))))
       ((eq (car regexp) ':optional) (list ':optional (regexp-normalize-plus (nth 1 regexp))))
       ((eq (car regexp) ':mark) (list ':mark (nth 1 regexp)
				       (regexp-normalize-plus (nth 2 regexp))))
       (t regexp)))
  


(defun regexp-split-nonempty (regexp)
  (let ((splits (regexp-split regexp))
	(result nil))
    (while splits
      (let* ((a (nth 0 (car splits)))
	     (b (nth 1 (car splits)))
	     (u (if (= (length a) 1) (nth 0 a) (cons ':seq a)))
	     (v (if (= (length b) 1) (nth 0 b) (cons ':seq b)))
	     (x (regexp-remove-null u))
	     (y (regexp-remove-null v))
	     (pre (regexp-normalize-plus x))
	     (post (regexp-normalize-plus y)))
	(if (and pre post)
	    (setq result (cons (list pre post) result))))
      (setq splits (cdr splits)))
    (delete-duplicate result)))

(defun regexp-compile-word-pattern (regexp &optional *regexp-sort-flag*)
  (let ((*regexp-ignore-mark-and-duplicate* t))
    (regexp-compile-word-regexp (regexp-parse regexp))))

(defun regexp-compile-split-pair (pair)
  (cons (regexp-compile-regexp (regexp-inverse (nth 0 pair)))
	(regexp-compile-regexp (nth 1 pair))))

(defun regexp-compile-word-regexp (regexp)
    (cons (regexp-compile-regexp regexp)
	  (cons (regexp-compile-regexp (regexp-inverse regexp))
		(mapcar 'regexp-compile-split-pair
			(regexp-merge-pairlist (regexp-split-nonempty regexp))))))

(defun every (pred list)
  (let ((success t))
    (while (and list success)
      (if (funcall pred (car list))
	  nil
	(setq success nil))
      (setq list (cdr list)))
    success))

(defun some (pred list)
  (let ((success nil))
    (while (and list (null success))
      (if (funcall pred (car list))
	  (setq success t))
      (setq list (cdr list)))
    success))

(defun regexp-nullable (regexp)
  (cond((null regexp) t)
       ((numberp regexp) nil)
       ((stringp regexp) (equal regexp ""))
       ((symbolp regexp) nil)
       ((memequal (car regexp) '(CHARSET CHARSET_NOT))
	nil)
       ((eq (car regexp) ':or)
	(some 'regexp-nullable (cdr regexp)))
       ((eq (car regexp) ':seq)
	(every 'regexp-nullable (cdr regexp)))
       ((eq (car regexp) ':star) t)
       ((eq (car regexp) ':plus) (regexp-nullable (nth 1 regexp)))
       ((eq (car regexp) ':optional) t)
       ((eq (car regexp) ':mark) (regexp-nullable (nth 2 regexp)))
       (t nil)))
	
(defun regexp-remove-null (regexp)
  (cond((null regexp) nil)
       ((numberp regexp) regexp)
       ((stringp regexp) 
	(if (equal regexp "") nil
	  regexp))
       ((symbolp regexp) regexp)
       ((memequal (car regexp) '(CHARSET CHARSET_NOT)) regexp)
       ((eq (car regexp) ':or)
	(let ((result nil) )
	  (setq regexp (cdr regexp))
	  (while regexp
	    (let ((regexp1 (regexp-remove-null (car regexp))))
	      (if regexp1 
		  (setq result (cons regexp1 result))))
	    (setq regexp (cdr regexp)))
	  (cond((null result) nil)
	       ((= (length result) 1) (nth 1 result))
	       (t (cons ':or (reverse result))))))
       ((eq (car regexp) ':seq)
	(cond((= (length regexp) 1) nil)
	     ((= (length (cdr regexp)) 1)
	      (regexp-remove-null (nth 1 regexp)))
	     ((every 'regexp-nullable (cdr regexp))
	      (let ((result nil) (pre nil) (post (cdr regexp)))
		(while post
		  (setq result (cons (append '(:seq) pre (list (regexp-remove-null (car post)))(cdr post))
				     result))
		  (setq pre (append pre (list (car post))))
		  (setq post (cdr post)))
		(cond((null result) nil)
		     ((= (length result) 1)
		      (car result))
		     (t (cons ':or (reverse result))))))
	      (t regexp)))
       ((eq (car regexp) ':star)
	(list ':plus (regexp-remove-null (nth 1 regexp))))
       ((eq (car regexp) ':plus)
	(list ':plus (regexp-remove-null (nth 1 regexp))))
       ((eq (car regexp) ':optional)
	(regexp-remove-null (nth 1 regexp)))
       ((eq (car regexp) ':mark)
	(regexp-remove-null (nth 2 regexp)))
       (t regexp)))
	    
(defun regexp-first-equal (pair1 pair2)
  (equal (nth 0 pair1) (nth 0 pair2)))

(defun regexp-second-equal (pair1 pair2)
  (equal (nth 1 pair1) (nth 1 pair2)))

(defun regexp-merge-pairlist (list)
  (let ((result nil) (place nil))
    (while list
      (cond((setq place (member (car list) (cdr list) 'regexp-first-equal))
	    (setq result (cons (list (car (car list))
				     (regexp-merge-regexp (nth 1 (car list))
							  (nth 1 (car place))))
			       result))
	    (setq list (delete (car place) (cdr list)))
	    )
	   ((setq place (member (car list) (cdr list) 'regexp-second-equal))
	    (setq result (cons (list (regexp-merge-regexp (nth 0 (car list))
							  (nth 0 (car place)))
				     (nth 1 (car list)))
			       result))
	    (setq list (delete (car place) (cdr list)))
	    )
	   (t (setq result (cons (car list) result))
	      (setq list (cdr list)))))
    (reverse result)))

(defun regexp-merge-regexp (regexp1 regexp2)
  (cond((and (consp regexp1)
	     (eq (car regexp1) ':or)
	     (consp regexp2)
	     (eq (car regexp2) ':or))
	(cons ':or (delete-duplicate (append (cdr regexp1) (cdr regexp2)))))
       ((and (consp regexp1)
	     (eq (car regexp1) ':or))
	(cons ':or (delete-duplicate (cons regexp2 (cdr regexp1)))))
       ((and (consp regexp2)
	     (eq (car regexp2) ':or))
	(cons ':or (delete-duplicate (cons regexp1 (cdr regexp2)))))
       (t 
	(list ':or regexp1 regexp2))))
	     

(defun regexp-compile (pat &optional word par)
  (if word
      (regexp-compile-word-pattern pat par)
    (regexp-compile-pattern-bothward pat par)))

;; 92.7.17 by K.Handa
;; Since regexp-compile does not generate correct code when
;; optional argument WORD is t, we should use, for the moment, the
;; following function to generate code for 'word'.
(defun regexp-compile-word (regexp split)
  (let ((*regexp-ignore-mark-and-duplicate* t)
	(pat (regexp-parse regexp))
	code)
    (while split
      (setq code (cons (cons (regexp-compile-regexp
			      (regexp-parse (car (car split))))
			     (regexp-compile-regexp
			      (regexp-parse (cdr (car split)))))
		       code))
      (setq split (cdr split)))
    (cons (regexp-compile-regexp pat)
	  (cons (regexp-compile-regexp (regexp-inverse pat))
		code))))
;; end of patch

(defun regexp-compile-fastmap (table)
  )

;;; Finite State Automaton
;;;

(defvar *fna1* nil)
(defvar *fna2* nil)
(defvar *fna3* nil)
(defvar *fna1-s* nil)
(defvar *fna2-s* nil)

(defun fna:equal (fna1 fna2)
  (let ((*fna1* fna1)
	(*fna2* fna2)
	(*fna1-s* 0)
	(*fna2-s* 0)
	(*fna3* nil))
    (fna:equal* (assoc 1 *fna1*)
		(assoc 1 *fna2*))))

(defun fna:equal* (state1 state2)
  (let ((state (cons (car state1) (car state2))))
  (if (assoc state *fna3*)
      nil
    (let ((place (cons state nil))
	  (alist nil))
      (setq *fna3* (cons place *fna3*))
      (cond((and (null (car state1))
		 (null (car state2)))
	    )
	   ((null (car state1))
	    (setq *fna1-s* (1+ *fna1-s*)))
	   ((null (car state2))
	    (setq *fna2-s* (1+ *fna2-s*))))
      ))))
      
      
