;;;   fort-header.el - code for producing a comment header in FORTRAN routines
;;;   Copyright (C) 1992 Free Software Foundation, Inc.
;;;
;;;   This program is free software; you can redistribute it and/or modify
;;;   it under the terms of the GNU General Public License as published by
;;;   the Free Software Foundation; either version 1, or (at your option)
;;;   any later version.
;;;
;;;   This program is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;   GNU General Public License for more details.
;;;
;;;   You should have received a copy of the GNU General Public License
;;;   along with this program; if not, write to the Free Software
;;;   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;   Title:  fortran header
;;;
;;;   Description:  The following function produces a fortran heading at the 
;;;                 top of the current buffer. It is based on ada-header found
;;;                 in the standard emacs distribution. It can also add in the
;;;                 RCS (revision control system) and last edit date strings
;;;                 tags with a prefix argument. Will attempt to fill the
;;;                 description using fortran-auto-fill if the version number
;;;                 of fortran-mode is greater than 1.28.5. Bound to C-c h.
;;;                 
;;;                 Also included at bottom is blink-or-goto-paren-match which
;;;                 will move the cursor backwards or forwards in the current
;;;                 buffer to a matching parenthesis of the type (,[,or {.
;;;                 Useful for complicated fortran expressions. Both bound to 
;;;                 C-c { and C-c }.
;;;
;;;   Created:  Thu May 28 10:59:30 1992
;;;   Author:   Lawrence R. Dodd
;;;             <dodd@roebling.poly.edu>
;;;   
;;;   Bug Reports, Comments, Suggestions, Smart Remarks: 
;;;         dodd@roebling.poly.edu 
;;;
;;;   $LastEditDate:     "Sat Nov 14 14:03:24 1992"$
;;;   $Date: 1992/11/14 19:09:04 $
;;;
;;;   $Author: dodd $
;;;   $Id: fort-header.el,v 1.29 1992/11/14 19:09:04 dodd Exp $
;;;   $Source: /home/dodd/lisp/RCS/fort-header.el,v $ 
;;;   $Revision: 1.29 $ 

;;;   INSTALLATION: 
;;; 
;;;     o  save as fort-header.el in the GNU emacs load-path.
;;;  
;;;     o  stick (autoload 'fortran-header "fort-header"
;;;   		     "Code for producing a comment header in FORTRAN routines."
;;;    		     t)
;;;        inside your .emacs, preferably inside fortran-mode-hook:
;;;
;;;        (setq fortran-mode-hook 
;;;           '(lambda () 
;;;                    .
;;;                    . other stuff
;;;                    .
;;;              (define-key fortran-mode-map "\C-ch" 'fortran-header)
;;;
;;;              (autoload 'fortran-header "fort-header"
;;;                  "Code for producing a comment header in FORTRAN 
;;;                  routines." t)))

;;;   USAGE INSTRUCTIONS: 
;;; 
;;;     o type M-x fortran-header or simply C-c h
;;; 
;;;     o type M-1 M-x fortran-header or simply M-1 C-c h to get string tags:
;;;        oo will insert standard RCS version string tags (RCS can be 
;;;           obtained via anonymous ftp at 
;;;           /prep.ai.mit.edu:/pub/gnu/rcs-*.tar.Z).
;;;        oo will insert standard LastEditDate string tags (written by Randal 
;;;           L. Schwartz (merlyn@iWarp.intel.com)).
;;;        oo if the variable fortran-header-rcs is set to true this is done 
;;;           by default.

;;;   KEY BINDINGS:

;;;     o fortran-header is bound to C-c h
;;;     o blink-or-goto-paren-match is bound to C-c { and C-c }
;;;       (I bind it globally to M-{ and M-} as well).

;;;   SEE ALSO:
;;;
;;;     /hallc1.cebaf.gov:/emacs for the most recent version of fortran-mode
;;;     and also for Ralph Finch's fortran-beautifier stored in
;;;     fortran-misc.el
;;;
;;;     /archive.cis.ohio-state.edu:/pub/gnu/emacs/elisp-archive/misc for
;;;     fort-clean.el code for cleaning and stylizing FORTRAN code

;;;     LCD Archive Entry:
;;;     fort-header|Lawrence R. Dodd|dodd@roebling.poly.edu
;;;     |code for producing a comment header in FORTRAN routines
;;;     |$Date: 1992/11/14 19:09:04 $|$Revision: 1.29 $|

(defconst fortran-header-version "$Revision: 1.29 $"
  "$Id: fort-header.el,v 1.29 1992/11/14 19:09:04 dodd Exp $")

(defvar fortran-header-rcs nil
  "Flag for fortran-header.  If t will include RCS and last edit date flags in
header by default.")

;;; bind fortran-header to C-c h
(define-key fortran-mode-map "\C-ch" 'fortran-header)

(defun fortran-header (&optional arg)
  "Insert a fortran comment block at the top of the current buffer containing
the module title, description, author, date. With prefix arg, will also
include update-last-edit-date tag and RCS tags."
  (interactive "P")
  (save-excursion
    (goto-char (point-min)) ; go to top of buffer
    (open-line 1)           ; open a line
    
    ;; read in title string
    (insert "c--\t\nc--   Title:  ")
    (insert (read-string "Brief Title: "))
    
    ;; read in (possibly long) description string
    (insert "\nc--\t\nc--   Description:  ")
    (insert (read-string "Description (use C-x ^ if needed): "))
    
    ;; write creation time and author
    (insert "\nc--   ")
    (insert "\nc--   Created:\t" (current-time-string))
    (insert "\nc--   Author: \t" (user-full-name))
    (insert "\nc--\t\t<" (user-login-name) "@" (system-name) ">\nc     ")
    
    ;; if arg is non-nil, then add last edit date and rcs tags
    (if (or arg fortran-header-rcs)
	(progn
	  
	  ;; To prevent update-last-edit and RCS from substituting values into
	  ;; the keywords in this defun, I have placed a double quote after
	  ;; the dollar signs and broke last-edit-date into three separate
	  ;; strings
	  
	  ;; insert update-last-edit-date tag
	  (insert "\nc     $" "Last" "Edit" "Date:\t\"" (current-time-string) "\"$")
	  
	  ;; insert reversion-control (rcs) tags
	  (insert "\nc     $" "Id$")
	  (insert "\nc     $" "Revision$")
	  (insert "\nc     $" "Log$\n")))

    ;; remove tabs and extraneous blank lines
    (untabify (point-min) (point))
    (delete-blank-lines)
    
    ;; now fill description provided that the version of fortran-mode is
    ;; greater than 1.28.5 so that we may use fortran-auto-fill..

    (if (and (string< "1.28.5" fortran-mode-version)
	     (re-search-backward "^c--   Description:  " nil t))

	;; fill description
	(let ((comment-line-start "c--                "))
	  (if (not auto-fill-hook)
	      ;; not on
	      (progn (fortran-auto-fill-mode nil)  ; turn on
		     (fortran-indent-line)     ; fill
		     (fortran-auto-fill-mode nil)) ; turn off
	    ;; on already
	    (fortran-indent-line))))))


;;; blink on different types of matching parentheses

(define-key fortran-mode-map "\C-c{" 'blink-or-goto-paren-match)
(define-key fortran-mode-map "\C-c}" 'blink-or-goto-paren-match)

(defun blink-or-goto-paren-match (&optional arg)
  "Blink the matching parenthesis if on parenthesis. With argument, go to
matching parenthesis and stay there."

  (interactive "P")

  (let

      ;; VARLIST
      (opoint)

    ;; BODY

    (setq opoint (point))

    (cond

     ((looking-at "[[({]")
      (forward-sexp 1)
      (backward-char)
      (if (not arg) (progn (sit-for 1) (goto-char opoint))))    ; blink?

     ;; note `\]' must be the first character in the regular expression 
     ;; character set that is passed to `looking-at'

     ((looking-at "[])}]")
      (forward-char)
      (backward-sexp 1)
      (if (not arg) (progn (sit-for 1) (goto-char opoint))))))) ; blink?
