/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


#include <stdio.h>
#include "defs.h"
#include "dvi-com.h"
#include "units.h"
#include "dvitps.h"
#include "extfil.h"
#include "fontd.h"
#include "emit.h"

extern EX_FILES ExDvi;
extern EX_FILES Ex_PsOutput;
extern int Count[10];
extern int Verbose;
extern int PrevPP;
extern FE_P Fonts[MAX_FONTS];
extern int CurFontNumber;
extern FE_P CurFontPointer;
extern int CurFontSelected;
extern int DviFileFontMap[MAX_FONTS];
extern int ConformingDocument;
extern int StderrOutputPageNumber;
extern int StderrOutputPageNumberMod;
extern int FromPageNumber; /* -f option of driver */
extern int ToPageNumber;   /* -t option of driver */

int CharactersDownloadedThisPage;

/*
 * Pass0
 * *****
 * This is pass 0 through a page. The intial position of the dvi file is at a command,
 * which is typically BOP (but also FNT_DEF, NOP or POST).
 *
 * reposition_control:
 *    TRUE: before returning from this procedure reposition to the
 *       beginning of the current page, in other words to the position
 *       when this procedure was started!
 *    FALSE: before returning do NOT reposition. Should lead to reading
 *      in the BOP of the next page (or POST).
 *
 * RET: One of PASS0_*
 *      PASS0_NORMAL_EOP       Regular end of page
 *	PASS0_POSTAMBLE_FOUND  Postamble beginning found
 * Side effects:
 *      PrevPP, the pointer to the previous page, is set when the BOP command
 *      is executed.
 */
int
Pass0(reposition_control)
     int reposition_control;
{
  int i;
  int val;
  int command;
  int file_pos_beginning; /* Offset of file when we started. */
  int k;
  int output_this_page; /* Output this page? -f/-t options of the driver
			   may suppress output! */

  CharactersDownloadedThisPage = 0;

  SwitchPsOutput(1);
  PssChangeFormat (PSS_FOR_CONT);

  /* So we can reposition to the beginning of the file after finishing
     this pass. */
  file_pos_beginning = FExTell(&ExDvi);

  /* Get the very first command in the dvi file now. */
  switch (command = NoSignExtend (EX_FP(ExDvi), 1)) {
    case DVI_POST:
      FExSeek (&ExDvi, -1, FSEEK_REL); /* Position backwards so DVI_POST can be read again. */
      return (PASS0_POSTAMBLE_FOUND);
    case DVI_BOP:
      break;
    default:
      Fatal2 ("Pass0(): illegal command %d", command);
      break;
  }

  FExSeek (&ExDvi, -1, FSEEK_REL); /* Position backwards to read command again. */
  CurFontSelected = FALSE;

  /*
   * Loop to go through one page in the dvi file. Many commands are
   * simply skipped because the main purpose of pass0 is to collect
   * character definitions.
   */
  while (TRUE)    {
    command = NoSignExtend (EX_FP(ExDvi), 1);
#ifdef DEBUG
    fprintf (stderr, "%% Pass0: command = %d\n", command);
#endif
    switch (command)    {
      case DVI_PUT_RULE:
      case DVI_SET_RULE:
        FExSeek (&ExDvi, 8, FSEEK_REL);
	break;

      case DVI_SET1: case DVI_SET2: case DVI_SET3: case DVI_SET4:
	val = NoSignExtend (EX_FP(ExDvi), command - DVI_SET1+1);
	if (! output_this_page)
	  break;
	if (! CurFontPointer->f_ch[val].c_dl)
	  HandleCharacterPass0 (val);
	break;
      case DVI_PUT1: case DVI_PUT2: case DVI_PUT3: case DVI_PUT4:
	val = NoSignExtend(EX_FP(ExDvi),command - DVI_PUT1+1);
	if (! output_this_page)
	  break;
	if (! CurFontPointer->f_ch[val].c_dl)
	  HandleCharacterPass0 (val);
	break;

      case DVI_NOP:
	break;
	
      case DVI_BOP:
	for (i=0; i<=9; i++) /* \count0 to \count9 from TeX */
	  Count[i] = NoSignExtend(EX_FP(ExDvi), 4);
	PrevPP = NoSignExtend(EX_FP(ExDvi), 4); /* Previous page pointer */
	output_this_page = (FromPageNumber <= Count[0] &&
			    Count[0] <= ToPageNumber);
	if (! output_this_page)
	  break;
	EMIT_2 ("\n%d @bop0\n", Count[0]);
	switch (Verbose) {
          case V_QUIET:
	    break;
          case V_SOME:
	    StderrOutputPageNumber++;
	    fprintf(stderr, "[%d", Count[0]);
	    fflush(stderr);
	    break;
          case V_A_LOT:
	    StderrOutputPageNumber++;
	    fprintf(stderr, "[P-0: %d", Count[0]);
	    fflush(stderr);
	    break;
        }
	break;

      case DVI_PUSH:
      case DVI_POP:
	break;

      case DVI_RIGHT1: case DVI_RIGHT2: case DVI_RIGHT3: case DVI_RIGHT4:
	FExSeek (&ExDvi, command - DVI_RIGHT1+1, FSEEK_REL);
	break;

      case DVI_W0:
	break;

      case DVI_W1: case DVI_W2: case DVI_W3: case DVI_W4:
	FExSeek (&ExDvi, command - DVI_W1+1, FSEEK_REL);
	break;

      case DVI_X0:
	break;

      case DVI_X1: case DVI_X2: case DVI_X3: case DVI_X4:
	FExSeek (&ExDvi, command - DVI_X1+1, FSEEK_REL);
	break;

      case DVI_DOWN1: case DVI_DOWN2: case DVI_DOWN3: case DVI_DOWN4:
	FExSeek (&ExDvi, command - DVI_DOWN1+1, FSEEK_REL);
	break;

      case DVI_Y0:
	break;

      case DVI_Y1: case DVI_Y2: case DVI_Y3: case DVI_Y4:
	FExSeek (&ExDvi, command - DVI_Y1+1, FSEEK_REL);
	break;

      case DVI_Z0:
	break;

      case DVI_Z1: case DVI_Z2: case DVI_Z3: case DVI_Z4:
	FExSeek (&ExDvi, command - DVI_Z1+1, FSEEK_REL);
	break;

      /* Change font instructions */
      case DVI_FNT1: case DVI_FNT2: case DVI_FNT3: case DVI_FNT4:
	k = NoSignExtend(EX_FP(ExDvi), command - DVI_FNT1+1);
	if (! output_this_page)
	  break;
	CurFontNumber = DviFileFontMap[k];
	if (CurFontNumber > MAX_FONTS)
	  Fatal ("Pass0(): MAX_FONTS");
	CurFontPointer = Fonts[CurFontNumber];
	CurFontSelected = FALSE;
	SendNewFontInstruction();
	break;

      case DVI_XXX1: case DVI_XXX2: case DVI_XXX3: case DVI_XXX4:
	DoSpecial (command, 0, output_this_page);
	break;

      /* Font definitions: just skip them */
      case DVI_FNT_DEF1: case DVI_FNT_DEF2: case DVI_FNT_DEF3: case DVI_FNT_DEF4:
	k = NoSignExtend(EX_FP(ExDvi), command - DVI_FNT_DEF1+1);
	k = DviFileFontMap[k];
	FExSeek (&ExDvi, Fonts[k]->f_fnt_def_length, FSEEK_REL);
	break;

      case DVI_PRE:
	Fatal ("Pass0(): PRE");
	break;

      case DVI_POST:
	Fatal ("Pass0(): DVI_POST");
	break;

      /* EOP: you are done */
      case DVI_EOP:
	if (output_this_page) {
	  switch (Verbose) {
            case V_QUIET:
	      break;
	    case V_SOME:
	      if (ConformingDocument) {
		fprintf(stderr, "] ");
		if (StderrOutputPageNumber >= StderrOutputPageNumberMod) {
		  fprintf(stderr, "\n");
		  StderrOutputPageNumber = 0;
		}
	      }
	      fflush(stderr);
	      break;
	    case V_A_LOT:
	      fprintf (stderr, "]");
	      fflush(stderr);
	      break;
	    }
	  EMIT_2 ("\n%d @eop0\n", Count[0]);
	}

	if (reposition_control)
	  FExSeek (&ExDvi, file_pos_beginning, FSEEK_ABS);
	return(PASS0_EOP_FOUND);

      case DVI_POST_POST:
	Fatal ("Pass0(): DVI_POST_POST");
	break;

      /*
       * Three left over possibilities of what we find in dvi file:
       *	(a) set a character
       *	(b) change font
       *	(c) illegal code
       */
      default:
	if (! output_this_page)
	  break;
	if (command >= DVI_SETC_000 && command <= DVI_SETC_127) {
	  if (! CurFontPointer->f_ch[command].c_dl)
	    HandleCharacterPass0 (command);
	  break;
	}
	if (command >= DVI_FONT_00 && command <= DVI_FONT_63)	{
	  k = command - DVI_FONT_00;
	  if (k > MAX_FONTS)
	    Fatal ("Pass0(): MAX_FONTS");
	  CurFontNumber = DviFileFontMap[k];
	  CurFontPointer = Fonts[CurFontNumber];
	  CurFontSelected = FALSE;
	  SendNewFontInstruction();
	  break;
	}
	Fatal2 ("Pass0(): command %d undefined", command);
      }   /* switch */
  }	    /* while */
}
