/*
	External procedures for use with dviimp.
	Written by H. Trickey, 6/25/85.
	
	Implement the following procedures (as declared in Pascal)

	function gfbyte: integer;
	function gftwobytes: integer;
	function gfthreebytes: integer;
	function gfsignedquad: integer;
	function getbyte: integer;
	function signedbyte: integer;
	function gettwobytes: integer;
	function signedpair: integer;
	function getthreebytes: integer;
	function signedtrio: integer;
	function signedquad: integer;
	function curpos(var f:bytefile):integer;
	procedure imbyte(w:integer);
	procedure imhalfword(w:integer);
	procedure setpos(var f:bytefile;n:integer);
	procedure setpaths;
	function testaccess(accessmode:integer; filepath:integer):boolean;
	procedure setusername;
*/

#include <stdio.h>
#include <strings.h>
#include <pwd.h>
#include "h00vars.h" /* pc runtime structures */
#include "texpaths.h"

extern struct iorec dvifile,gffile,imfile; /* pc file structs */
extern int eofdvifile,eofgffile; /* need to be set true when these happens */
extern int curloc; /* needs to be incremented when a dvi byte is read */
extern int curgfloc; /* needs to be incremented when a gf byte is read */
extern int imbyteno; /* needs to be incremented when an im byte is written */

/* return unsigned version of next byte in gffile */
int gfbyte()
{
	register int c;
	
	if ((c = getc(gffile.fbuf))==EOF) {
		eofgffile = -1;
		return(0);
		}
	curgfloc++;
	return(c);
}

/* return unsigned version of next two bytes in gffile */
int gftwobytes()
{
	register int a,b;
	register FILE *iop = gffile.fbuf;
	a = getc(iop);
	eofgffile = ((b = getc(iop))==EOF) ? -1 : 0;
	curgfloc += 2;
	return((a << 8) | b);
}

/* return unsigned version of next three bytes in gffile */
int gfthreebytes()
{
	register int a,b,c;
	register FILE *iop = gffile.fbuf;
	a = getc(iop);
	b = getc(iop);
	eofgffile = ((c = getc(iop))==EOF) ? -1 : 0;
	curgfloc +=3;
	return((a << 16) | (b << 8) | c);
}

/* return signed version of next four bytes in gffile */
int gfsignedquad()
{
	register int a,b,c,d;
	register FILE *iop = gffile.fbuf;
	a = getc(iop);
	b = getc(iop);
	c = getc(iop);
	eofgffile = ((d = getc(iop))==EOF) ? -1 : 0;
	curgfloc += 4;
	return((a << 24) | (b << 16) | (c << 8) | d);
}

/* return unsigned version of next byte in dvifile */
int getbyte()
{
	register int c;
	
	if ((c = getc(dvifile.fbuf))==EOF) {
		eofdvifile = -1;
		return(0);
		}
	curloc++;
	return(c);
}

/* return signed version of next byte in dvifile */
int signedbyte()
{	register int c;
	if ((c = getc(dvifile.fbuf))==EOF) {
		eofdvifile = -1;
		return(0);
		}
	curloc++;
	if (c>127) return(c-256);
	return(c);
}

/* return unsigned value of next two bytes (high order first) */
int gettwobytes()
{
	register int a,b;
	register FILE *iop = dvifile.fbuf;
	a = getc(iop);
	eofdvifile = ((b = getc(iop))==EOF) ? -1 : 0;
	curloc += 2;
	return((a << 8) | b);
}

/* return signed value of next two bytes (high order first) */
int signedpair()
{
	register int a,b;
	register FILE *iop = dvifile.fbuf;
	if ( (a = getc(iop)) > 127) a -= 256; /* sign extend */
	eofdvifile = ((b = getc(iop))==EOF) ? -1 : 0;
	curloc += 2;
	return((a << 8) | b);
}

/* return unsigned value of next three bytes */
int getthreebytes()
{
	register int a,b,c;
	register FILE *iop = dvifile.fbuf;
	a = getc(iop);
	b = getc(iop);
	eofdvifile = ((c = getc(iop))==EOF) ? -1 : 0;
	curloc +=3;
	return((a << 16) | (b << 8) | c);
}

/* return signed value of next three bytes */
int signedtrio()
{
	register int a,b,c;
	register FILE *iop = dvifile.fbuf;
	if ( (a = getc(iop)) > 127) a -= 256;
	b = getc(iop);
	eofdvifile = ((c = getc(iop))==EOF) ? -1 : 0;
	curloc +=3;
	return((a << 16) | (b << 8) | c);
}

/* return value of next four bytes */
int signedquad()
{
	register int a,b,c,d;
	register FILE *iop = dvifile.fbuf;
	a = getc(iop);
	b = getc(iop);
	c = getc(iop);
	eofdvifile = ((d = getc(iop))==EOF) ? -1 : 0;
	curloc += 4;
	return((a << 24) | (b << 16) | (c << 8) | d);
}

/* put the low order byte of w on imfile */
imbyte(w)
	int w;
{
	putc(w, imfile.fbuf);
	imbyteno++;
}

/* put the two low order bytes of w on imfile, high order first */
imhalfword(w)
	int w;
{
	putc(w >> 8, imfile.fbuf);
	putc(w, imfile.fbuf);
	imbyteno += 2;
}

/* seek to byte n of file f; maintain eof's if f is dvifile or gffile */
setpos(f,n)
	register struct iorec *f;
	int n;
{
	if (n>=0) {
	    fseek(f->fbuf,(long) n,0); 
	    if (f==&dvifile) eofdvifile = 0;
	    else if (f==&gffile) eofgffile =0;
	    }
	else {
	    fseek(f->fbuf, (long) n, 2);
	    if (f==&dvifile) eofdvifile = -1;
	    else if (f==&gffile) eofgffile = -1;
	    }
}

/* return current byte offset in file f */
int curpos(f)
	struct iorec *f;
{
	return((int) ftell(f->fbuf));
}

char *fontpath;

char *getenv();

/*
 * setpaths is called to set up the pointer fontpath
 * as follows:  if the user's environment has a value for TEXFONTS
 * then use it;  otherwise, use defaultfontpath.
 */
setpaths()
{
	register char *envpath;
	
	if ((envpath = getenv("TEXFONTS")) != NULL)
	    fontpath = envpath;
	else
	    fontpath = defaultfontpath;
}

#define namelength 100   /* should agree with dvitype.ch */
extern char curname[],realnameoffile[]; /* these have size namelength */

/*
 *	testaccess(amode,filepath)
 *
 *  Test whether or not the file whose name is in the global curname
 *  can be opened for reading (if mode=READACCESS)
 *  or writing (if mode=WRITEACCESS).
 *
 *  The filepath argument is one of the ...FILEPATH constants defined below.
 *  If the filename given in curname does not begin with '/', we try 
 *  prepending all the ':'-separated areanames in the appropriate path to the
 *  filename until access can be made, if it ever can.
 *
 *  The realnameoffile global array will contain the name that yielded an
 *  access success.
 */

#define READACCESS 4
#define WRITEACCESS 2

#define NOFILEPATH 0
#define FONTFILEPATH 3

bool
testaccess(amode,filepath)
    int amode,filepath;
{
    register bool ok;
    register char *p;
    char *curpathplace;
    int f;
    
    switch(filepath) {
	case NOFILEPATH: curpathplace = NULL; break;
	case FONTFILEPATH: curpathplace = fontpath; break;
	}
    if (curname[0]=='/')	/* file name has absolute path */
	curpathplace = NULL;
    do {
	packrealnameoffile(&curpathplace);
	if (amode==READACCESS)
	    /* use system call "access" to see if we could read it */
	    if (access(realnameoffile,READACCESS)==0) ok = TRUE;
	    else ok = FALSE;
	else {
	    /* WRITEACCESS: use creat to see if we could create it, but close
	    the file again if we're OK, to let pc open it for real */
	    f = creat(realnameoffile,0666);
	    if (f>=0) ok = TRUE;
	    else ok = FALSE;
	    if (ok)
		close(f);
	    }
    } while (!ok && curpathplace != NULL);
    if (ok) {  /* pad realnameoffile with blanks, as Pascal wants */
	for (p = realnameoffile; *p != '\0'; p++)
	    /* nothing: find end of string */ ;
	while (p < &(realnameoffile[namelength]))
	    *p++ = ' ';
	}
    return (ok ? -1 : 0);
}

/*
 * packrealnameoffile(cpp) makes realnameoffile contain the directory at *cpp,
 * followed by '/', followed by the characters in curname up until the
 * first blank there, and finally a '\0'.  The cpp pointer is left pointing
 * at the next directory in the path.
 * But: if *cpp == NULL, then we are supposed to use curname as is.
 */
packrealnameoffile(cpp)
    char **cpp;
{
    register char *p,*realname;
    
    realname = realnameoffile;
    if ((p = *cpp)!=NULL) {
	while ((*p != ':') && (*p != '\0')) {
	    *realname++ = *p++;
	    if (realname == &(realnameoffile[namelength-1]))
		break;
	    }
	if (*p == '\0') *cpp = NULL; /* at end of path now */
	else *cpp = p+1; /* else get past ':' */
	*realname++ = '/';  /* separate the area from the name to follow */
	}
    /* now append curname to realname... */
    p = curname;
    while (*p != ' ') {
	if (realname >= &(realnameoffile[namelength-1])) {
	    fprintf(stderr,"! Full file name is too long\n");
	    break;
	    }
	*realname++ = *p++;
	}
    *realname = '\0';
}

extern struct passwd * getpwuid();
extern char username[];
extern int lenusername;
/*
 * find out the caller's name, putting it into username[1..namelength]
 * and put the length used into lenusername
 */
setusername()
{
	int i;
	char *nam;
	
	nam = getpwuid(getuid())->pw_name;
	for (i=0; *nam!=0 && *nam!='(' && i<namelength; )
		username[i++] = *nam++;
	lenusername = i;
}

extern char hostname[];
extern int lenhostname;
sethostnam()
{
	int i;
	char nam[namelength];
	
	lenhostname = 0;
	if (gethostname(nam,255) != 0) return;
	for (i=0; nam[i]!=0 && nam[i]!='.' && i<namelength; ) {
		hostname[i] = nam[i];
		i++;
		}
	lenhostname = i;
}
