/* System V interface to the terminal for use by DVItoVDU.
 
   Marc Hadley.  Kernel Technology.  June 1989.
 
   Procedures for setting and resetting UNIX tty characteristics.
   Interesting functions are:
      savetty();
      restoretty();
      echoon();
      echooff();
      singlecharon();
      singlecharoff();
      buffercount();
      suspend();
   A side effect of calling savetty() is to set up signal handling to reset
   the terminal characteristics appropriately for the various interrupt
   signals.
*/
 
#include "def.h"
#include <termio.h>
#include <signal.h>
#include <fcntl.h>
#include <setjmp.h>
 
static char *sccsid[] = "@(#)usgio.c	1.1";
 
extern jmp_buf commprompt_env;		/* jump buffer for handleint() */
					/* (defined in dvitovdu.c) */
 
struct termio initchars; /* store initial terminal characteristics */
 
/* Save the original tty characteristics and set up the signalling. */
void savetty()
{
  (void) ioctl(0, TCGETA, &initchars);
  setsignals();
}
 
/* Restore the original tty characteristics. */
void restoretty()
{
  (void) ioctl(0, TCSETA, &initchars);
}
 
/* Set driver to read characters as they are typed without waiting for a
   terminator. Echo remains unchanged. */
void singlecharon()
{
  struct termio s;
  (void) ioctl(0, TCGETA, &s);
  /* s.c_iflag = IGNPAR | IGNBRK | ICRNL; */
  s.c_lflag &= ~ICANON;
  s.c_cc[VEOF] = 1;
  s.c_cc[VEOL] = 1;
  (void) ioctl(0, TCSETA, &s);
  /* fflush(stdin); */
}
 
/* Turn off single character read mode. */
void singlecharoff()
{
  struct termio s;
  (void) ioctl(0, TCGETA, &s);
  s.c_iflag = initchars.c_iflag;
  s.c_lflag |= ICANON;
  s.c_cc[VEOF] = initchars.c_cc[VEOF];
  s.c_cc[VEOL] = initchars.c_cc[VEOL];
  (void) ioctl(0, TCSETA, &s);
  (void) fflush(stdin);
}
 
/* Turn character echoing on. */
void echoon()
{
  struct termio s;
  (void) ioctl(0, TCGETA, &s);
  s.c_lflag |= ECHO;
  (void) ioctl(0, TCSETA, &s);
}
 
/* Turn character echoing off. */
void echooff()
{
  struct termio s;
  (void) ioctl(0, TCGETA, &s);
  s.c_lflag &= ~ECHO;
  (void) ioctl(0, TCSETA, &s);
}
 
/* Catch signals from tty and do nothing */
 
handleint(sig)
     int sig;
{
  char inbuf[512];
  extern Void (*ResetVDU)();
 
  (void) fflush(stdin);
  setsignals();				/* do this first, before longjmp! */
  if(sig == SIGINT)
      longjmp(commprompt_env, 1);	/* longjmp to NextCommand() loop */
}
 
/* Signal initialization. */
setsignals()
{
  (void) signal(SIGINT, handleint);
}
 
/* set terminal to no blocking read mode , read a char and if one available
push it back then return after setting back to blocking read mode */
 
int getcharnowait()
{
    int success;
    unsigned char ch;
 
    no_block_on();
 
    success = read(0,(char *)(&ch),1);
 
    no_block_off();
 
    if (success > 0) return ((int) ch);
    else return (EOF);
}
 
static int s;
 
no_block_on()   /* set reads to be knob locking */
{
    s = fcntl( 0,F_GETFL,0);
    (void) fcntl( 0, F_SETFL, s | O_NDELAY );
}
 
no_block_off()  /* turn off knob locking reads */
{
    extern int errno;
 
    errno = 0;
    if( fcntl( 0, F_SETFL, s ) == -1 )
        (void) printf( "fcntl failed, errno = %d\n", errno );
}
 
/* Suspend the process */
void suspend()
{
  (void) signal(SIGINT, SIG_DFL);
#ifdef SIGTSTP
  kill(0, SIGTSTP);
#endif
  /* resumed again, goody! */
  setsignals();
}
 
