/*
 * Copyright (c) Matsushita Electric Industrial Co.,Ltd. 1994
 *
 * $Id: dvi.c,v 1.14 1994/03/30 09:50:56 kakiuchi Exp $
 */

static char rcsid[] = "$Id: dvi.c,v 1.14 1994/03/30 09:50:56 kakiuchi Exp $";

/*
 * DVI file functions.
 */

#include "dvi.h"
#include "font_public.h"
#include "dvi_public.h"
#include <sys/stat.h>

/*
 * Constants and macro functions.
 */
#define move_down(a)	v += (a)
#define move_over(b)	h += (b)

#define AppFontDesc	(app_info->fontdesc)
#define AppEHandler	(*app_info->error_handler)
#define AppInitDVI(f,n)	(app_info->init_dvi?(*app_info->init_dvi)(f,n):True)
#define AppCheckSheet(s) (app_info->check_sheet?(*app_info->check_sheet)(s):True)
#define AppBOP		(app_info->bop)
#define AppEOP()	(app_info->eop?(*app_info->eop)():True)
#define AppSetFntNum	(app_info->set_fnt_num)
#define AppSetChar	(app_info->set_char)
#define AppSetRule	(app_info->set_rule)
#define AppDoSpecial	(app_info->do_special)

/*
 * Global variables
 */
DVI dvi_info = {
  /* filename		*/	"",
  /* directory		*/	"",
  /* dvifp		*/	NULL,
  /* mtime		*/	0,
  /* num		*/	0,
  /* den		*/	0,
  /* mag		*/	0,
  /* hpd		*/	0,
  /* wpw		*/	0,
  /* stack_size		*/	0,
  /* NumberOfPage	*/	0,
  /* last_known_sheet	*/	0,
  /* pages		*/	NULL,
  /* curPage		*/	NULL,
  /* curFont		*/	NULL,
};

/*
 * Private functions
 */
static bool ReadPostamble();
static bool CheckDVIFile();
static void set_font_num();
static void read_font_def();
#if NeedVarargsPrototypes
static void error_handler(char *fmt, ...);
#else
static void error_handler();
#endif
static u32 SetChar(), SetRule();

/*
 * Private variables
 */
typedef struct		/* stack entry */
{
  i32 h, v, w, x, y, z;	/* what's on stack */
#ifdef PTEX
  u8 d;			/* value of dir register */
#endif /* PTEX */
} Stack;

static i32 h;			/* current horizontal position */
static i32 v;			/* current vertical position */
static Stack *stack = NULL;     /* stack */
static AppInfo *app_info;
static bool callSetFntNum;

/*
 * error_handler:
 *	If any error was found, this function would be called.
 *	It cleans up DVI file pointer and calls an error handler provided
 *	by applications.
 */

static void
#if NeedVarargsPrototypes
error_handler(char *fmt, ...)
#else
error_handler(fmt, va_alist)
register char *fmt;
va_dcl
#endif
{
  va_list argptr;
  static char msg[256];

  VA_START(argptr, fmt);
  (void) vsprintf(msg, fmt, argptr);
  va_end(argptr);

  if (dvi_info.dvifp) (void) fclose(dvi_info.dvifp);
  dvi_info.dvifp = NULL;
  dvi_info.mtime = 0;
  AppEHandler(msg);
}

/*
 * InitDVIFile:
 *	Opens the dvi file, and checks for valid codes etc.
 *	Reads the postamble (if enabled)
 *	Leaves the file pointer at the start of the first page.
 */

#if NeedFunctionPrototypes
bool
InitDVIFile(
    register char *filename,
    register int resolution,
    register u32 user_mag,
    register char *font_path,
    register char *tfm_path,
    register AppInfo *appinfo
)
#else
bool
InitDVIFile(filename, resolution, user_mag, font_path, tfm_path, appinfo)
register char *filename, *font_path, *tfm_path;
register int resolution;
register u32 user_mag;
register AppInfo *appinfo;
#endif
{
  register int i;
  register char *extension;
  char fullname[MAXPATHLEN], directory[MAXPATHLEN], name[MAXPATHLEN];

  if (appinfo) app_info = appinfo;
  else return(False);

  /*
   * If the filename has no extension, or if it
   * has an extension and it is not '.dvi' then
   * cat .dvi onto the filename.
   */
  if (filename == NULL) {
    filename = "stdin";
    (void) strcpy(name, filename);
    (void) strcpy(fullname, filename);
  } else {
    /* get directory */
    if (extension = rindex(filename, '/')) {
      bcopy(filename, directory, extension - filename);
      directory[extension - filename] = '\0';
      (void) strcpy(name, extension + 1);
    } else {
      (void) getwd(directory);
      (void) strcpy(name, filename);
    }
    if ((extension = rindex(name, '.')) && !strcmp(extension, ".dvi"))
      *extension = '\0';

    (void) strcpy(fullname, directory);
    (void) strcat(fullname, "/");
    (void) strcat(fullname, name);
    (void) strcat(fullname, ".dvi");

    /*
     * Open the file
     */
#ifdef DEBUG
    prerror("Opening file \"%s\"\n", fullname);
#endif
    if (access(fullname, 04) != 0) {
      error_handler("%s: Cant access DVI file", fullname);
      return(False);
    }
  }

  /*
   * We must free all file descriptors used by font libraries before opening
   * new DVI file, because all of file descriptors could be exhausted.
   */
  if (!AppInitDVI(fullname, name)) {
    error_handler("The application's init_dvi failed");
    return(False);
  }
  if (dvi_info.dvifp) {
    /*
     * close the dvifile.
     */
    (void) fclose(dvi_info.dvifp);
    dvi_info.mtime = 0;

    /*
     * Close the fonts and free up memory.
     */
    CloseFonts();
  }

  if (!strcmp(fullname, "stdin")) dvi_info.dvifp = stdin;
  else if ((dvi_info.dvifp = fopen(fullname, "r")) == NULL) {
    error_handler("%s: Cant open DVI file", fullname);
    return(False);
  }

  if (!CheckDVIFile()) return(False);
  /*
   * Read the magic number and version number
   */
  if ((i = get_unsigned1(dvi_info.dvifp)) != PRE) {
    error_handler("%s: not a dvi file (bad magic number %d)", fullname, i);
    return(False);
  }
  if ((i = get_signed1(dvi_info.dvifp)) != DVIFORMAT) {
    error_handler("%s: dvi format %d not supported", fullname, i);
    return(False);
  }

  /*
   * Load font information from postable.
   */
  if (!ReadPostamble(dvi_info.dvifp, resolution, user_mag, font_path, tfm_path))
    return(False);

  /* Return to start of first page. */
  (void) fseek(dvi_info.dvifp, 14L, 0);

  /*
   * Skip i more bytes of preamble.
   */
  (void) fseek(dvi_info.dvifp, (long) get_unsigned1(dvi_info.dvifp), 1);

  /*
   * Set up the page fseek pointer table.
   * We are now at page 0.
   */
  if (dvi_info.pages) free((char *) dvi_info.pages);
  if (!(dvi_info.pages =
	(DVIPages *) malloc((size_t)(sizeof(DVIPages) * dvi_info.NumberOfPage)))) {
    error_handler("Cannot allocate page maps");
    return(False);
  }
  for (i = dvi_info.NumberOfPage; i-- > 0; ) {
    dvi_info.pages[i].sheet_table = 0;
    dvi_info.pages[i].sheet_page = 0;
  }
  dvi_info.last_known_sheet = (u16) 0;
  dvi_info.pages[0].sheet_table = ftell(dvi_info.dvifp);

  if (stack) free((char *) stack);
  if (!(stack = (Stack *) malloc((size_t) (sizeof(Stack) * dvi_info.stack_size)))) {
    error_handler("Cannot allocate stack");
    return(False);
  }

#ifdef DEBUG
  prerror("sheet_table[%d] = %d\n",
	  (int) dvi_info.last_known_sheet, ftell(dvi_info.dvifp));
#endif

  (void) strcpy(dvi_info.filename, fullname);
  (void) strcpy(dvi_info.directory, directory);

  return(True);
}

/*
 * CheckDVIFile:
 *	Checks that this is the same file -- has not been modified since
 *	it was opened.
 */

static bool
CheckDVIFile()
{
  struct stat stat_buf; /* For checking file changes. */

  if (dvi_info.dvifp == NULL) {
    error_handler("No DVI file open");
    return(False);
  }

  if (fstat(fileno(dvi_info.dvifp), &stat_buf) != 0) {
    error_handler("%s: dvifile fstat failed", dvi_info.filename);
    return(False);
  }

  if (!dvi_info.mtime) dvi_info.mtime = stat_buf.st_mtime;
  else if (stat_buf.st_mtime != dvi_info.mtime) {
    error_handler("%s: dvifile modified", dvi_info.filename);
    return(False);
  }

  return(True);
}

/*
 * ReadPostamble:
 *	This  routine  is  used  to  read  in  the  postamble  values.	It
 *	initializes the magnification and checks  the stack height prior  to
 *	starting printing the document.
 */

static bool
ReadPostamble(fp, resolution, user_mag, font_path, tfm_path)
register FILE *fp;
register int resolution;
register u32 user_mag;
register char *font_path, *tfm_path;
{
  register u8 byte, i;

  /*
   * find_postamble_ptr
   *	Move to the end of the dvifile and find the start of the postamble.
   */
  /* Get postamble pointer */
  (void) fseek(fp, -1L, 2);
  while ((i = get_unsigned1(fp)) == DVI_PAD) {
    if (fseek(fp, -2L, 1) == -1) {
      error_handler("Bad DVI file: can't seek postamble");
      return(False);
    }
  }
  (void) fseek(fp, -6L, 1);
#ifdef PTEX
  if (get_unsigned1(fp) != POST_POST || (i != DVIFORMAT && i != PTEX_DVIFORMAT)) {
#else
  if (get_unsigned1(fp) != POST_POST || i != DVIFORMAT) {
#endif /* PTEX */
    error_handler("Bad DVI file: bad end of file");
    return(False);
  }
  (void) fseek(fp, (long) get_unsigned4(fp), 0);

  if (get_unsigned1(fp) != POST) {
    error_handler("Bad DVI file: no POST at head of postamble");
    return(False);
  }

  (void) fseek(fp, 4L, 1); /* discard last page pointer */
  dvi_info.num = get_unsigned4(fp);
  dvi_info.den = get_unsigned4(fp);
  dvi_info.mag = get_unsigned4(fp);
  if (user_mag > 0) dvi_info.mag = user_mag;

  /*
   * Init font libraries
   */
  InitFont(dvi_info.num, dvi_info.den, dvi_info.mag,
	   resolution, font_path, tfm_path, AppFontDesc);

  dvi_info.hpd = VConv(get_unsigned4(fp));
  /* height-plus-depth of tallest page */
  dvi_info.wpw = HConv(get_unsigned4(fp));
  /* width of widest page */
  dvi_info.stack_size = get_unsigned2(fp);
  dvi_info.NumberOfPage = get_unsigned2(fp);

/*
 *	Read the font  definitions as they  are in the  postamble of the  DVI
 *	file.  Note that the font directory  is not yet loaded.  In order  to
 *	adapt ourselves to the existing "verser" the following font paramters
 *	are  copied   onto   output   fontno  (4   bytes),  chksum,  fontmag,
 *	fontnamelength (1 byte), fontname.  At the end, a -1 is put onto  the
 *	file.
 */

  while (((byte = get_unsigned1(fp)) >= FNT_DEF1) && byte <= FNT_DEF4)
    read_font_def(fp, (u32) get_unsigned(fp, byte - FNT_DEF1 + 1));

  if (byte != POST_POST) {
    error_handler("Bad DVI file: no postpostamble after fontdefs");
    return(False);
  } else return(True);
}

/*
 * ProcessPage:
 *	Rasterises the page specified as new_sheet in the dvifile
 *      into page_pmem.
 */

#if NeedFunctionPrototypes
bool
ProcessPage(
    int new_sheet
)
#else
bool
ProcessPage(new_sheet)
int new_sheet;
#endif
{
  register bool skip_mode = True;
  register u8 cmd;		/* current command */
  register i32 w;		/* current horizontal spacing */
  register i32 x;		/* current horizontal spacing */
  register i32 y;		/* current vertical spacing */
  register i32 z;		/* current vertical spacing */
  register i32 sp;		/* stack pointer */
  register u32 val, val2;	/* temporarys */
  register u16 file_sheet;
#ifdef PTEX
  register u8 dir;		/* DIR register */
#endif /* PTEX */
  i32 counter[10];

  if (!app_info || !app_info->error_handler || !CheckDVIFile())
    return(False);

  /* Check against page limits. */
  if (new_sheet < 0) new_sheet = 0;
  else if (new_sheet >= (int) dvi_info.NumberOfPage)
    new_sheet = dvi_info.NumberOfPage - 1;

  if (!AppCheckSheet(new_sheet)) return(False);

  /* Do we already know where the page is ? */
  if ((u16) new_sheet <= dvi_info.last_known_sheet) {
    (void) fseek(dvi_info.dvifp, dvi_info.pages[new_sheet].sheet_table, 0);
    file_sheet = (u16) new_sheet;
  } else {
    file_sheet = dvi_info.last_known_sheet;
    (void) fseek(dvi_info.dvifp, dvi_info.pages[file_sheet].sheet_table, 0);
  }

  while (file_sheet <= (u16) new_sheet) {
    if (file_sheet == (u16) new_sheet) skip_mode = False;
    dvi_info.curPage = &dvi_info.pages[file_sheet];
#ifdef DEBUG
    prerror("sheet %d starts at %d\n",
	    (int) file_sheet, (int) dvi_info.curPage->sheet_table);
#endif

    while ((cmd = get_unsigned1(dvi_info.dvifp)) != EOP) {
      switch(cmd) {

      case SET1: case SET2: case SET3: case SET4:
	if (skip_mode) (void) fseek(dvi_info.dvifp, (long)(cmd - SET1 + 1), 1);
#ifdef PTEX
	else if (dir) move_down(SetChar(get_unsigned(dvi_info.dvifp, cmd - SET1 + 1), dir));
	else move_over(SetChar(get_unsigned(dvi_info.dvifp, cmd - SET1 + 1), dir));
#else
	else move_over(SetChar(get_unsigned(dvi_info.dvifp, cmd - SET1 + 1)));
#endif /* PTEX */
	break;

      case SET_RULE:
	if (skip_mode) (void) fseek(dvi_info.dvifp, 8L, 1);
	else {
	  val = get_unsigned4(dvi_info.dvifp);
	  val2 = get_unsigned4(dvi_info.dvifp);
#ifdef PTEX
	  if (dir) move_down(SetRule(val, val2, dir));
	  else move_over(SetRule(val, val2, dir));
#else
	  move_over(SetRule(val, val2));
#endif /* PTEX */
	}
	break;

      case PUT1: case PUT2: case PUT3: case PUT4:
	if (skip_mode) (void) fseek(dvi_info.dvifp, (long) (cmd - PUT1 + 1), 1);
#ifdef PTEX
	else (void) SetChar(get_unsigned(dvi_info.dvifp, cmd - PUT1 + 1), dir);
#else
	else (void) SetChar(get_unsigned(dvi_info.dvifp, cmd - PUT1 + 1));
#endif /* PTEX */
	break;

      case PUT_RULE:
	if (skip_mode) (void) fseek(dvi_info.dvifp, 8L, 1);
	else {
	  val = get_unsigned4(dvi_info.dvifp);
	  val2 = get_unsigned4(dvi_info.dvifp);
#ifdef PTEX
	  (void) SetRule(val, val2, dir);
#else
	  (void) SetRule(val, val2);
#endif /* PTEX */
	}
	break;

      case NOP:
	break;

      case BOP:
	/*
	 * These are the 10 counters.
	 * Discard previous page pointer.
	 */
	for (val = 0; val < 10; val++)
	  counter[val] = get_signed4(dvi_info.dvifp);
	(void) fseek(dvi_info.dvifp, 4L, 1);

	/*
	 * The first counter is the page number.
	 */
	dvi_info.curPage->sheet_page = counter[0];

	/*
	 * Show what is happening.
	 */
#ifdef DEBUG
	prerror("File \"%s\"   Page %d   %s\n", dvi_info.filename,
		dvi_info.curPage->sheet_page, skip_mode ? "Skipping" : "Processing");
#endif

	if (!skip_mode && AppBOP) (*AppBOP)();

	h = v = w = x = y = z = 0;
#ifdef PTEX
	dir = 0;
#endif /* PTEX */
	sp = 0;
	dvi_info.curFont = NULL;
	callSetFntNum = False;
	break;

      case PUSH:
	if (sp >= (i32) dvi_info.stack_size) {
	  error_handler("Bad DVI file: stack overflow");
	  return(False);
	}
	stack[sp].h = h; stack[sp].v = v; stack[sp].w = w;
	stack[sp].x = x; stack[sp].y = y; stack[sp].z = z;
#ifdef PTEX
	stack[sp].d = dir;
#endif /* PTEX */
	sp++;
	break;

      case POP:
	if (--sp < 0) {
	  error_handler("Bad DVI file: stack underflow");
	  return(False);
	}
	h = stack[sp].h; v = stack[sp].v; w = stack[sp].w;
	x = stack[sp].x; y = stack[sp].y; z = stack[sp].z;
#ifdef PTEX
	dir = stack[sp].d;
#endif /* PTEX */
      break;

      case RIGHT1: case RIGHT2: case RIGHT3: case RIGHT4:
	if (skip_mode) (void) fseek(dvi_info.dvifp, (long) (cmd - RIGHT1 + 1), 1);
#ifdef PTEX
	else if (dir) move_down(get_signed(dvi_info.dvifp, cmd - RIGHT1 + 1));
#endif /* PTEX */
	else move_over(get_signed(dvi_info.dvifp, cmd - RIGHT1 + 1));
	break;

      case W1: case W2: case W3: case W4:
	w = get_signed(dvi_info.dvifp, cmd - W1 + 1);

      case W0:
#ifdef PTEX
	if (!skip_mode) {
	  if (dir) move_down(w); else move_over(w);
	}
#else
	if (!skip_mode) move_over(w);
#endif /* PTEX */
	break;

      case X1: case X2: case X3: case X4:
	x = get_signed(dvi_info.dvifp, cmd - X1 + 1);

      case X0:
#ifdef PTEX
	if (!skip_mode) {
	  if (dir) move_down(x); else move_over(x);
	}
#else
	if (!skip_mode) move_over(x);
#endif /* PTEX */
	break;

      case DOWN1: case DOWN2: case DOWN3: case DOWN4:
	if (skip_mode) (void) fseek(dvi_info.dvifp, (long) (cmd - DOWN1 + 1), 1);
#ifdef PTEX
	else if (dir) move_over(-get_signed(dvi_info.dvifp, cmd - DOWN1 + 1));
#endif
	else move_down(get_signed(dvi_info.dvifp, cmd - DOWN1 + 1));
	break;

      case Y1: case Y2: case Y3: case Y4:
	y = get_signed(dvi_info.dvifp, cmd - Y1 + 1);

      case Y0:
#ifdef PTEX
	if (!skip_mode) {
	  if (dir) move_over(-y); else move_down(y);
	}
#else
	if (!skip_mode) move_down(y);
#endif /* PTEX */
	break;

      case Z1: case Z2: case Z3: case Z4:
	z = get_signed(dvi_info.dvifp, cmd - Z1 + 1);

      case Z0:
#ifdef PTEX
	if (!skip_mode) {
	  if (dir) move_over(-z); else move_down(z);
	}
#else
	if (!skip_mode) move_down(z);
#endif /* PTEX */
	break;

      case FNT1: case FNT2: case FNT3: case FNT4:
	if (skip_mode) (void) fseek(dvi_info.dvifp, (long) (cmd - FNT1 + 1), 1);
	else set_font_num(get_unsigned(dvi_info.dvifp, cmd - FNT1 + 1));
	break;

      case XXX1: case XXX2: case XXX3: case XXX4:
	val = get_unsigned(dvi_info.dvifp, cmd - XXX1 + 1);
	if (skip_mode) (void) fseek(dvi_info.dvifp, (long) val, 1);
	else {
	  register char *SpecialStr;	/* "\special" strings */

	  SpecialStr = (char *) malloc((size_t) (val + 1));
	  if (SpecialStr == NULL)
	    error_handler("Cannot allocate string for specials");
	  else {
	    if ((int) val==fread(SpecialStr, 1, (int) val, dvi_info.dvifp)) {
	      SpecialStr[val] = NULL;
	      if (AppDoSpecial)
		(*AppDoSpecial)(SpecialStr, val, HConv(h), VConv(v));
	    } else error_handler("Bad DVI file: cannot read special string");
	    free(SpecialStr);
	  }
	}
	break;

      case FNT_DEF1: case FNT_DEF2: case FNT_DEF3: case FNT_DEF4:
	(void) fseek(dvi_info.dvifp, (long) (12 + cmd - FNT_DEF1 + 1), 1);
	(void) fseek(dvi_info.dvifp,
	      (long) (get_unsigned1(dvi_info.dvifp) + get_unsigned1(dvi_info.dvifp)),
	      1);
	break;

      case PRE:
	error_handler("Bad DVI file: preamble found within main section");
	return(False);

      case POST:
	(void) fseek(dvi_info.dvifp, -1L, 1);
	return(False);

      case POST_POST:
	error_handler("Bad DVI file: postpostamble found within main section");
	return(False);

#ifdef PTEX
      case DIR:
	dir = get_unsigned1(dvi_info.dvifp);
	break;
#endif /* PTEX */

      default:
	if (cmd >= FONT_00 && cmd <= FONT_63) {
	  if (!skip_mode) set_font_num((u32) (cmd - FONT_00));
	} else if (cmd >= SETC_000 && cmd <= SETC_127) {
#ifdef PTEX
	  if (!skip_mode) {
	    if (dir) move_down(SetChar((u32)(cmd - SETC_000), dir));
	    else move_over(SetChar((u32)(cmd - SETC_000), dir));
	  }
#else
	  if (!skip_mode) move_over(SetChar((u32)(cmd - SETC_000)));
#endif /* PTEX */
	} else {
	  error_handler("Bad DVI file: undefined command (%d) found", cmd);
	  return(False);
	}
      }
    }

    /*
     * End of page.
     */

    /*
     * The file is now at the start of the next page.
     */
    file_sheet++;
    if (file_sheet < dvi_info.NumberOfPage &&
	file_sheet > dvi_info.last_known_sheet) {
      dvi_info.last_known_sheet = file_sheet;
      dvi_info.pages[file_sheet].sheet_table = ftell(dvi_info.dvifp);
    }
  }

  return(AppEOP());
}


/*
 * set_font_num:/
 *	This routine is used to specify the font to be used in printing future
 *	chars.
 */

static void
set_font_num(k)
register u32 k;
{
  if (!(dvi_info.curFont = GetFont(k, False))) {
    dvi_info.curFont = NULL;
    error_handler("Bad DVI file: font %d undefined", k);
  }
  /* Call set_font_num later, because a font file will not be loaded before
     a glyph is extracted at the first time. */
  callSetFntNum = True;
}


/*
 * Font functions
 * ==============
 */

/*
 * read_font_def:
 */

static void
read_font_def(fp, k)
register FILE *fp;
register u32 k;
{
  register FontEntry *fe = GetFont(k, True);
  register u32 c, s, d;
  register u8 a, l;
  char n[MAXPATHLEN];

  c = get_unsigned4(fp); /* checksum */
  s = get_unsigned4(fp); /* space size */
  d = get_unsigned4(fp); /* design size */
  a = get_unsigned1(fp); /* area length for font name */
  l = get_unsigned1(fp); /* device length */
  (void) fread(n, 1, (int) (a + l), fp);
  if (!fe->common.font_ops) OpenFont(fe, c, s, d, a, l, n);
}

#ifdef PTEX
static u32
SetChar(c, dir)
register u32 c;
u8 dir;
{
  register CharEntry *ptr;
  register i32 hh = HConv(h), vv = VConv(v);

  ptr = GetGlyph(dvi_info.curFont, c);
  if (callSetFntNum) {
    if (AppSetFntNum) (*AppSetFntNum)(dvi_info.curFont);
    callSetFntNum = False;
  }
  if (!ptr) return(0);
  if (AppSetChar) (*AppSetChar)(ptr, c, h, v, hh, vv, dir);
  return(ptr->tfmw);
}
#else
static u32
SetChar(c)
register u32 c;
{
  register CharEntry *ptr;
  register int hh = HConv(h), vv = VConv(v);

  ptr = GetGlyph(dvi_info.curFont, c);
  if (callSetFntNum) {
    if (AppSetFntNum) (*AppSetFntNum)(dvi_info.curFont);
    callSetFntNum = False;
  }
  if (!ptr) return(0);
  if (AppSetChar) (*AppSetChar)(ptr, c, h, v, hh, vv);
  return(ptr->tfmw);
}
#endif /* PTEX */

#ifdef PTEX
static u32
SetRule(a, b, dir)
register u32 a, b;
u8 dir;
{
  register i32 hh, vv;
  register i32 ehh, evv;

  if (dir) {
    hh = HConv(h); vv = VConv(v);
    ehh = HConv(h + (double) a); evv = VConv(v + (double) b);
    if (AppSetRule) (*AppSetRule)(b, a, hh, vv, ehh, evv);
  } else {
    hh = HConv(h); vv = VConv(v - (double) a);
    ehh = HConv(h + (double) b); evv = VConv(v);
    if (AppSetRule) (*AppSetRule)(a, b, hh, vv, ehh, evv);
  }
  return(b);
}
#else
static u32
SetRule(a, b)
register u32 a, b;
{
  register i32 hh = HConv(h), vv = VConv(v - (double) a);
  register i32 ehh = HConv(h + (double) b), evv = VConv(v);

  if (AppSetRule) (*AppSetRule)(a, b, hh, vv, ehh, evv);
  return(b);
}
#endif /* PTEX */
