/*
 * Copyright (c) Matsushita Electric Industrial Co.,Ltd. 1994
 *
 * $Id: jdvi2.c,v 1.20 1994/03/30 09:50:41 kakiuchi Exp $
 */

static char rcsid[] = "$Id: jdvi2.c,v 1.20 1994/03/30 09:50:41 kakiuchi Exp $";

#include <ctype.h>
#include "jdvi2.h"

OptionCore core_option;

static char version[] = "2.1";
static char progName[] = "jdvi2";
static char usage[] = "\
	[-Llanguage]		Set language.\n\
	[-ooutput_filename]	Specify output file name.\n\
	[-f start_page]		Specify start page.\n\
	[-t end_page]		Specify end page.\n\
	[-pe]			Print even pages only.\n\
	[-po]			Print odd pages only.\n\
	[-Ppages_to_print]	Specify pages to print. (not supported yet)\n\
	[-ppages_to_print]	Specify pages to print. (not supported yet)\n\
	[-r]			Print reverse order.\n\
	[-l]			Print landscape mode.\n\
	[-pa paper_type]	Set paper size.\n\
	[-mf]			Set manual feed mode on.\n\
	[-ccopies]		Specify number of copies.\n\
	[-m[0|h|1|2|3|4|5]]	Specify mag step.\n\
	[-s]			Silent.\n\
	[-x left_margin]	Specify left margin.\n\
	[-y top_margin]		Specify top margin.\n\
	[-d resolution]		Specify resolution.\n\
	[-v]			Show version.\n\
	[-help]			Show usage.\n\
	[dvifile]		DVI file.";

static PDLInfo *pdl_info;

static OptionDesc pdl_desc[] = {
  {"-L", StickyArg, STRING, "unknown", NULL, (Pointer) &CoreLanguage},
};

static OptionDesc core_desc[] = {
  {"-o", StickyArg, STRING, NULL, NULL, (Pointer) &CoreOutputFilename},
  {"-f", SepArg, INTEGER, "1", NULL, (Pointer) &CorePageFrom},
  {"-t", SepArg, INTEGER, "65535", NULL, (Pointer) &CorePageTo},
  {"-P", SepArg, STRING, NULL, NULL, (Pointer) &CoreTeXPagesString},
  {"-p", SepArg, STRING, NULL, NULL, (Pointer) &CorePagesString},
  {"-r", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreReverse},
  {"-l", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreLandscape},
  {"-pa", SepArg, STRING, "a4", NULL, (Pointer) &CorePaper},
  {"-mf", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreManualFeed},
  {"-s", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreSilent},
  {"-c", StickyArg, INTEGER, "1", NULL, (Pointer) &CoreCopies},
  {"-m", StickyArg, MAGSTEP, "*", NULL, (Pointer) &CoreMagStep},
  {"-x", SepArg, DIMENSION, "1.0in", NULL, (Pointer) &CoreLeftMargin},
  {"-y", SepArg, DIMENSION, "1.0in", NULL, (Pointer) &CoreTopMargin},
  {"-d", SepArg, INTEGER, NULL, NULL, (Pointer) &CoreResolution},
  {"-v", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreShowVersion},
  {"-po", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreOddPage},
  {"-pe", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreEvenPage},
  {"-help", NoArg, BOOLEAN, "False", "True", (Pointer) &CoreShowHelp},
};

static PaperType paper_type[] =
  {{"a3", 11.75, 16.5}, {"a4", 8.25, 11.75}, {"a4extra", 9.26, 12.69},
   {"a5", 5.875, 8.25}, {"b4", 10.0, 14.0}, {"b5", 7.0, 10.0},
   {"letter", 8.5, 11.0}, {"note", 8.5, 11.0}, {"legal", 8.5, 14.0},
   {"ledger", 11.0, 17.0}, {"executive", 7.25, 10.5}};

#define	NumberOfPaperTypes	(sizeof(paper_type)/sizeof(PaperType))

#define	pt2in(value)	((value) / 72.27)
#define	pc2in(value)	pt2in((value) * 12.)
#define	bp2in(value)	((value) / 72.)
#define	cm2in(value)	((value) / 2.54)
#define	mm2in(value)	cm2in((value) / 10.)
#define	dd2in(value)	pt2in((value) * 1238. / 1157.)
#define	cc2in(value)	dd2in((value) * 12.)
#define	sp2in(value)	pt2in((value) / 65536.)

static void
ConvString(type, value, ret_value)
OptionType type;
char *value;
Pointer ret_value;
{
  char *p;
  double atof(), strtod();

  if (!value) return;
  switch (type) {
  case INTEGER:
    *((int *) ret_value) = atoi(value);
    break;
  case FLOAT:
    *((float *) ret_value) = atof(value);
    break;
  case DIMENSION:
    *((float *) ret_value) = strtod(value, &p);
    if (!strcmp(p, "pt"))
      *((float *) ret_value) = pt2in(*((float *) ret_value));
    else if (!strcmp(p, "pc"))
      *((float *) ret_value) = pc2in(*((float *) ret_value));
    else if (!strcmp(p, "bp"))
      *((float *) ret_value) = bp2in(*((float *) ret_value));
    else if (!strcmp(p, "cm"))
      *((float *) ret_value) = cm2in(*((float *) ret_value));
    else if (!strcmp(p, "mm"))
      *((float *) ret_value) = mm2in(*((float *) ret_value));
    else if (!strcmp(p, "dd"))
      *((float *) ret_value) = dd2in(*((float *) ret_value));
    else if (!strcmp(p, "cc"))
      *((float *) ret_value) = cc2in(*((float *) ret_value));
    else if (!strcmp(p, "sp"))
      *((float *) ret_value) = sp2in(*((float *) ret_value));
    break;
  case MAGSTEP:
    switch (*value) {
    case '*': *((u32 *) ret_value) = 0; break;
    case '0': *((u32 *) ret_value) = 1000; break;
    case 'h': *((u32 *) ret_value) = 1095; break;
    case '1': *((u32 *) ret_value) = 1200; break;
    case '2': *((u32 *) ret_value) = 1440; break;
    case '3': *((u32 *) ret_value) = 1728; break;
    case '4': *((u32 *) ret_value) = 2074; break;
    case '5': *((u32 *) ret_value) = 2488; break;
    default: prerror("%c is a bad mag step\n", *value); break;
    }
    break;
  case STRING:
    *((char **) ret_value) = value;
    break;
  case BOOLEAN:
    if (!strcmp(value, "True")) *((bool *) ret_value) = True;
    else *((bool *) ret_value) = False;
    break;
  default:
    prerror("Illegal type conversion.\n");
    break;
  }
}

#ifdef DEBUG
static void
PrintOption(option, option_num)
register OptionDesc option[];
register long option_num;
{
  register int i;

  for (i = 0; i < option_num; ++i) {
    prerror("Value of option [%s] is ", option[i].option);
    switch (option[i].value_type) {
    case INTEGER:
      prerror("%d\n", *((int *) option[i].value));
      break;
    case MAGSTEP:
      prerror("%d\n", *((u32 *) option[i].value));
      break;
    case FLOAT: case DIMENSION:
      prerror("%f\n", *((float *) option[i].value));
      break;
    case STRING:
      if (*((char **) option[i].value))
	prerror(*((char **) option[i].value));
      prerror("\n");
      break;
    case BOOLEAN:
      prerror(*((bool *) option[i].value) == True ? "True\n" : "False\n");
      break;
    default:
      prerror("Illegal type.\n");
      break;
    }
  }
}
#endif /* DEBUG */

static void
DecodeArgs(option_desc, option_desc_num, option_check_func, argc, argv)
OptionDesc option_desc[];
int option_desc_num, *argc;
char **argv;
void (*option_check_func)();
{
  register int i, opt_len, consumed;
  register int cur_argc = *argc;
  register char **cur_arg = &argv[1], **last_arg = &argv[cur_argc], **arg;

  for (i = 0; i < option_desc_num; ++i)
    ConvString(option_desc[i].value_type,
	       option_desc[i].initial_value,
	       option_desc[i].value);
  while (cur_arg < last_arg) {
    for (i = 0; i < option_desc_num; ++i) {
      switch (option_desc[i].type) {
      case NoArg:
	if (!strcmp(option_desc[i].option, *cur_arg)) {
	  ConvString(option_desc[i].value_type,
		     option_desc[i].default_value,
		     option_desc[i].value);
	  consumed = 1;
	} else continue;
	break;
      case StickyArg:
	opt_len = strlen(option_desc[i].option);
	if (!strncmp(option_desc[i].option, *cur_arg, opt_len)) {
	  ConvString(option_desc[i].value_type,
		     cur_arg[0] + opt_len,
		     option_desc[i].value);
	  consumed = 1;
	} else continue;
	break;
      case SepArg:
	if (!strcmp(option_desc[i].option, *cur_arg)) {
	  ConvString(option_desc[i].value_type,
		     cur_arg[1],
		     option_desc[i].value);
	  consumed = 2;
	} else continue;
	break;
      default:
	prerror("Illegal option type.\n");
	continue;
      }
      for (arg = cur_arg; &arg[consumed] < last_arg; ++arg)
	arg[0] = arg[consumed];
      cur_argc -= consumed;
      --cur_arg;
      last_arg = &argv[cur_argc];
      if (option_check_func) (*option_check_func)(&option_desc[i]);
      break;
    }
    ++cur_arg;
  }
  *argc = cur_argc;
#ifdef DEBUG
  PrintOption(option_desc, option_desc_num);
#endif /* DEBUG */
}

static void
Usage(version, usage)
register char *version, *usage;
{
  prerror("This is %s version %s.\n", progName, version);
  prerror("usage : %s\\\n%s\n\n", progName, usage);
}

static void
PDLUsage(info)
register PDLInfo *info;
{
  prerror("%s driver version %s.\n", info->language, info->version);
  if (info->usage)
    prerror("extensions : %s\\\n%s\n\n", CoreProgName, info->usage);
  else
    prerror("no extensions.\n");
}

static PDLInfo *
GetPDLInfo()
{
#include "pdl_def.h"
  register int i;

  /* Set default language by argv[0] */
  for (i = 0; i < pdl_info_num; ++i) {
    if (!strcmp(pdl_info_list[i]->prog_name, CoreProgName)) {
      CoreLanguage = pdl_info_list[i]->language;
      break;
    }
  }

  /* Set language by options */
  for (i = 0; i < pdl_info_num; ++i) {
    if (!strcmp(pdl_info_list[i]->language, CoreLanguage))
      return(pdl_info_list[i]);
  }
  prerror("Can not find a driver for language [%s].\n", CoreLanguage);
  Usage(version, usage);
  exit(1);
}

static void
CoreCheckOption(option_desc)
register OptionDesc *option_desc;
{
  if (CoreCopies <= 0) {
    prerror("Invalid copies parameter.  Ignore it.\n");
    CoreCopies = 1;
  }
  if (CoreResolution <= 0) {
    prerror("Invalid resolution [%d].\nUse default value.\n", CoreResolution);
    CoreResolution = pdl_info->default_resolution;
  }
  if (CoreEvenPage) CoreOddPage = False;
  if (CoreOddPage) CoreEvenPage = False;
}

static PaperType *
CheckPaperType(paper)
register char **paper;
{
  register char *p;
  register int i;

  for (p = *paper; *p != NULL; ++p)
    if (isupper(*p)) *p = tolower(*p);
 retry:
  for (i = 0; i < NumberOfPaperTypes; ++i)
    if (!strcmp(*paper, paper_type[i].name)) return(&paper_type[i]);
  prerror("Invalid paper size [%s].\nUse default (a4).\n", *paper);
  *paper = "a4";
  goto retry;
}

static void
PagePrint(i)
register int i;
{
  static int ndone = 0;
  register bool even = ((i / 2) * 2 != i);

  if (CoreEvenPage && even || CoreOddPage && !even ||
      (!CoreEvenPage && !CoreOddPage)) {
    if (!CoreSilent) prerror("[%d", i + 1);
    ProcessPage(i);
    if (!CoreSilent) {
      prerror("] ");
      if ((++ndone % 10) == 0) prerror("\n");
    }
  }
}

void
#if NeedVarargsPrototypes
FatalExit(char *fmt, ...)
#else
FatalExit(fmt, va_alist)	/* issue a fatal error message */
char *fmt;			/* format */
va_dcl				/* arguments */
#endif
{
  va_list argptr;

  prerror("\n");
  prerror( "%s: FATAL--", CoreProgName);
  VA_START(argptr, fmt);
  vfprintf(stderr, fmt, argptr);
  va_end(argptr);
  prerror( "\n\n");
  exit(2);
}

#define BITOF(d)			(0x80 >> (d))
#define isBitOn3(data,scanline,x)	(data[scanline+(x>>3)]&BITOF(x&7))

void
RotateImage(data, w, h, pdest, degree)
register u8 *data, *pdest;
register u16 w, h;
int degree;
{
  register int scanlen = UWIDTH((int) w), col, row;

  h *= scanlen;
  switch (degree) {
  case 90:
    for (col=0; col<(int)w; col++) {
      register int val = 0, mod = 0;
      for (row=h-scanlen; row>=0; row -= scanlen) {
	if (isBitOn3(data, row, col))
	  val |= BITOF(mod);
	if (mod == 7 || row == 0) {
	  *pdest++ = (u8)val; val = 0; mod = 0;
	} else mod++;
      }
    }
    break;
 case 180:
    for (row=h-scanlen; row>=0; row -= scanlen) {
      register int val = 0, mod = 0;
      for (col=w-1; col>=0; col--) {
	if (isBitOn3(data, row, col))
	  val |= BITOF(mod);
	if (mod == 7 || col == 0) {
	  *pdest++ = (u8)val; val = 0; mod = 0;
	} else mod++;
      }
    }
    break;
 case 270:
    for (col=w-1; col>=0; col--) {
      register int val = 0, mod = 0;
      for (row=0; row<(int)h; row += scanlen) {
	if (isBitOn3(data, row, col))
	  val |= BITOF(mod);
	if (mod == 7 || row == h - scanlen) {
	  *pdest++ = (u8)val; val = 0; mod = 0;
	} else mod++;
      }
    }
    break;
  }
}

main(argc, argv)
int argc;
char **argv;
{
  register int i;
  register char *cp;

  /*
   * force stderr to non buffering mode
   */
  (void) setbuf(stderr, NULL);

  if ((cp = rindex(argv[0], '/')) != NULL) cp++;
  else cp = argv[0];

  CoreProgName = cp;

  DecodeArgs(pdl_desc, (int) NumArray(pdl_desc), (void (*)()) NULL, &argc, argv);
  pdl_info = GetPDLInfo();
  CoreResolution = pdl_info->default_resolution;

  if (pdl_info->initialize) pdl_info->initialize();
  DecodeArgs(pdl_info->option_desc, pdl_info->option_desc_num,
	     pdl_info->option_check_func, &argc, argv);

  DecodeArgs(core_desc, (int) NumArray(core_desc), CoreCheckOption, &argc, argv);
  CorePaperType = CheckPaperType(&CorePaper);
  if (CoreLandscape) {	/* Swap width & height */
    register float width = CorePaperType->width;

    CorePaperType->width = CorePaperType->height;
    CorePaperType->height = width;
  }

#ifdef DEBUG
  prerror("language is %s\n", CoreLanguage);
#endif /* DEBUG */
  if (CoreShowVersion) {
    prerror("This is %s version %s.\n", progName, version);
    prerror("%s driver version %s.\n", pdl_info->language, pdl_info->version);
    exit(0);
  }
  if (CoreShowHelp) {
    Usage(version, usage);
    PDLUsage(pdl_info);
    exit(0);
  }
  if (argc == 2) CoreDVIFilename = argv[1];
  else if (argc == 1) CoreDVIFilename = NULL;
  else {
    Usage(version, usage);
    exit(1);
  }

  if (!CoreOutputFilename || !(CoreOutFp = fopen(CoreOutputFilename, "w")))
    CoreOutFp = stdout;

  InitDVIFile(CoreDVIFilename, CoreResolution, (u32) CoreMagStep,
	      NULL, NULL, pdl_info->app_info);

  --CorePageFrom;
  --CorePageTo;
  CorePageFrom = CorePageFrom < 0 ? 0 : CorePageFrom;
  CorePageTo = CorePageTo < 0 ? 0 :
    (CorePageTo < (int) dvi_info.NumberOfPage ? CorePageTo : dvi_info.NumberOfPage - 1);
  if (CoreReverse)
    for (i = CorePageTo; i >= CorePageFrom; --i) PagePrint(i);
  else
    for (i = CorePageFrom; i <= CorePageTo; ++i) PagePrint(i);

  if (pdl_info->postprocess) pdl_info->postprocess();
  return(0);
}
