/*
 * VFlib.c  -  Vector font library
 *
 *  Programmmed by Hirotsugu Kakugawa, Hiroshima University
 *  E-Mail:  kakugawa@se.hiroshima-u.ac.jp
 *
 *  Edition History
 *  13 Mar. 1993
 *  20 Mar. 1993  Changed font alias functions
 *  22 Mar. 1993  Added VF_close_font_all().
 *  10 June 1993  Added JG Font feature.
 */

/*
 * Copyright (C) 1993 Hirotsugu KAKUGAWA.   All rights reserved.
 *
 * VFlib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
 * to anyone for the consequences of using it or for whether it serves any
 * particular purpose or works at all, unless he says so in writing.  Refer
 * to the GNU General Public License for full details.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * VFlib, but only under the conditions described in the GNU
 * General Public License.  A copy of this license is supposed to have been
 * given to you along with VFlib so you can know your rights and
 * responsibilities.  It should be in a file named COPYING.  Among other
 * things, the copyright notice and this notice must be preserved on all
 * copies. 
 */

#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <ctype.h>
#include  <sys/types.h>

#include  "VF.h"
#include  "config.h"

#ifdef  USE_SONY_FONT
static Method   method_sony;
#endif
static Method   method_zeit;
static Method   method_jg;

static FontInfo  FontTable[FONT_TABLE_SIZE];
static inited = 0;


/*
 * VF_init
 */
int
VF_init(conf)
  char  *conf;
{
  int         i;
  static int  ReadConf();

  ReadConf(conf);
  for (i = 0; i < FONT_TABLE_SIZE; i++)
    FontTable[i].font_class = VF_NOT_USED;

  inited = 1;

#ifdef USE_SONY_FONT
  method_sony.open_font   = FS_open_font;
  method_sony.close_font  = FS_close_font;
  method_sony.get_outline = FS_get_outline;
  method_sony.scan_conv   = FS_scan_conv;
  method_sony.get_bitmap  = FS_get_bitmap;
  method_sony.fast_xform  = FS_fast_xform;
  method_sony.mat_xform   = FS_mat_xform;
#endif

  method_zeit.open_font   = ZF_open_font;
  method_zeit.close_font  = ZF_close_font;
  method_zeit.get_outline = ZF_get_outline;
  method_zeit.scan_conv   = ZF_scan_conv;
  method_zeit.get_bitmap  = ZF_get_bitmap;
  method_zeit.fast_xform  = ZF_fast_xform;
  method_zeit.mat_xform   = ZF_mat_xform;

  method_jg.open_font     = JG_open_font;
  method_jg.close_font    = JG_close_font;
  method_jg.get_outline   = JG_get_outline;
  method_jg.scan_conv     = JG_scan_conv;
  method_jg.get_bitmap    = JG_get_bitmap;
  method_jg.fast_xform    = JG_fast_xform;
  method_jg.mat_xform     = JG_mat_xform;

}



/*
 * VF_open_font
 */
int
VF_open_font(fontpath)
  char *fontpath;
{
  int         i;
  static int  FontClass(); 

  if (inited == 0) 
    VF_init(NULL);

  for (i = 0; i < FONT_TABLE_SIZE; i++){
    if (FontTable[i].font_class == VF_NOT_USED){
      FontTable[i].font_class = FontTypeByPath(fontpath);
      if ((FontTable[i].font_name = malloc(strlen(fontpath)+2)) == NULL)
	goto Err;
      strcpy(FontTable[i].font_name, fontpath);
      switch(FontTable[i].font_class){

#ifdef   USE_SONY_FONT
      case VF_FONT_SONY:
	FontTable[i].method = &method_sony;
	break;
#endif

      case VF_FONT_ZEIT:
	FontTable[i].method = &method_zeit;
	break;

      case VF_FONT_JG:
	FontTable[i].method = &method_jg;
	break;

      default:
	fprintf(stderr, "Unknown font class\n");
	exit(-1);
      }
      FontTable[i].fd = (*FontTable[i].method->open_font)(fontpath);

      return i;
    }
  }
Err:
  return -1;
}


/*
 * VF_close_font
 */
int
VF_close_font(fd)
  int fd;
{
  FontTable[fd].font_class = VF_NOT_USED;
  return (*FontTable[fd].method->close_font)(FontTable[fd].fd);
}


/*
 * VF_close_font_all
 */
void
VF_close_font_all()
{
  int  fd, junk;

  for (fd = 0; fd < FONT_TABLE_SIZE; fd++){
    if (FontTable[fd].font_class != VF_NOT_USED){
      junk =  VF_close_font(fd);
    }
  }
}


/*
 * VF_get_outline
 */
int
VF_get_outline(c, fd, buf)
  int   c;
  int   fd;
  long  **buf;
{
  return (*FontTable[fd].method->get_outline)(c, FontTable[fd].fd, buf);
}


/*
 * VF_scan_conv
 */
int
VF_scan_conv(vk_buf, w, h, bw, bo, bm_buf, thick, frame, smooth)
  long   *vk_buf;
  int  w;
  int  h;
  int  bw;
  int  bo;
  char *bm_buf;
  int  thick;
  int  frame;
  int  smooth;
{
  fprintf(stderr, "VF_scan_conv: not supported. sorry....\n");
  exit(-1);
}


/*
 * VF_get_bitmap
 */
int
VF_get_bitmap(c, fd, w, h, bw, bo, bm_buf, thick, frame, smooth)
  int   c;
  int   fd;
  int   w;
  int   h;
  int   bw;
  int   bo;
  char  *bm_buf;
  int   thick;
  int   frame;
  int   smooth;
{
  return 
    (*FontTable[fd].method->get_bitmap)
      (c, FontTable[fd].fd, w, h, bw,  bo, bm_buf, thick, frame, smooth);
}


/*
 * VF_fast_xform
 */
void
VF_fast_xform(vk, rotate, refx, refy)
  long  *vk;
  int   rotate;
  int   refx;
  int   refy;
{
  fprintf(stderr, "VF_fast_xform: not supported. sorry....\n");
  exit(-1);
}


/*
 * VF_mat_xform
 */
void
VF_mat_xform(x, y, t1, t2, t3, t4, t5, t6)
  int     *x, *y;
  double  t1, t2, t3, t4, t5, t6;
{
  fprintf(stderr, "VF_mat_xform: not supported. sorry....\n");
  exit(-1);
}



/** FONT ALIAS MANEGER **/

struct _FontMan {
  char              *alias;
  char              *path;
  int               type;
  struct _FontMan   *next;
};
typedef struct _FontMan   FontMan;

static FontMan  FontAliasHead;
static FontMan  *FontAliasTail;
static int      FMinited = -1;

int
InternFontAlias(alias, path, type)
  char  *alias, *path;
  int   type;
{
  int      i;
  FontMan  *fm; 
  static int ReadConf();

  if (FMinited == -1)
    ReadConf(NULL);

  if (   ((fm        = (FontMan*)malloc(sizeof(FontMan))) == NULL)
      || ((fm->alias =           malloc(strlen(alias)+4)) == NULL)
      || ((fm->path  =           malloc(strlen(path)+4))  == NULL) ){
    fprintf(stderr, "No mem in InternAlias()\n");
    exit(-1);
  }
  strcpy(fm->alias, alias);
  strcpy(fm->path,  path);
  fm->type = type;
  fm->next = NULL;
  FontAliasTail->next = fm;
  FontAliasTail       = fm;
  return 0;
}

dumpfontman()
{
  FontMan *fm;

  printf("------\n");
  for (fm = FontAliasHead.next; fm != NULL; fm = fm->next)
    printf("++%s,%s,%d.\n", fm->alias, fm->path, fm->type);
  printf("------\n");
}

char*
FontPath(ali)
  char  *ali;
{
  FontMan *fm;
  char    *p;
  int     l, max_l;

  if (FMinited == -1)
    return NULL;

  max_l = 0;
  p = NULL;
  for (fm = FontAliasHead.next; fm != NULL; fm = fm->next){
    l = strlen(fm->alias);
    if (strncmp(fm->alias, ali, l) == 0){
      if (max_l < l){                          /** LONGEST MATCH **/
	max_l = l;
	p = fm->path;
      }
    }
  }
  return p;
}

int
FontTypeByPath(fpath)
  char *fpath;
{
  FontMan *fm;
  int     l;

  if (FMinited == -1)
    return VF_FONT_DEFAULT;

  for (fm = FontAliasHead.next; fm != NULL; fm = fm->next){
    if (strcmp(fm->path, fpath) == 0)
      return fm->type;
  }
  return VF_FONT_DEFAULT;
}

int
FontType(ali)
  char *ali;
{
  FontMan *fm, *fmt;
  int     l, max_l;

  if (FMinited == -1)
    return VF_FONT_DEFAULT;

  max_l = 0;
  fmt = NULL;
  for (fm = FontAliasHead.next; fm != NULL; fm = fm->next){
    l = strlen(fm->alias);
    if (strncmp(fm->alias, ali, l) == 0){
      if (max_l < l){                          /** LONGEST MATCH **/
	max_l = l;
	fmt = fm;
      }
    }
  }
  if (fmt == NULL)
    return VF_FONT_DEFAULT;
  return fmt->type;
}


int
FontOpened(fontpath)
  char *fontpath;
{
  int  i;

  for (i = 0; i < FONT_TABLE_SIZE; i++){
    if (FontTable[i].font_class != VF_NOT_USED){
      if (strcmp(FontTable[i].font_name, fontpath) == 0)
	return i;
    }
  }
  return -1;
}

int
FontOpenedByAlias(ali)
  char *ali;
{
  return FontOpened(FontPath(ali));
}

static int
ReadConf(conf)
  char *conf;
{
  int        i, type;
  FILE       *fp;
  char       buff[512], *alias, *path;
  static int parseline();

  if (FMinited != -1)
    return 0;

  FMinited = 0;
  FontAliasHead.next == (FontMan*) NULL;
  FontAliasTail = &FontAliasHead;

  if (conf == NULL)
    return 0;
  if ((fp = fopen(conf, "r")) == (FILE*)NULL){
    fprintf(stderr, "Font configuration file %s not found.\n", conf);
    return -1;
  }
  while (! feof(fp)){
    if (fgets(buff, 512, fp) == NULL)
      break;
    if (parseline(buff, &alias, &path, &type, conf) == 0){
      InternFontAlias(alias, path, type);
    }
  }
  fclose(fp);
  /*dumpfontman();*/

  return 0;
}

static int 
parseline(s, aliasp, pathp, typep, conf)
  char  *s, **aliasp, **pathp, *conf;
  int   *typep;
{
  int   i, c;
  char  *typename;

  i = 0;
  for (;;){
    switch (s[i]){
    case ' ':
    case '\t':
      continue;
    case '%':
      return 1;
    case '\n':
      return 2;
    default:
      goto L1;
    }
  }
L1:
  *aliasp = &s[i];
  while ((s[i] != ' ') && (s[i] != '\t'))
    i++;
  s[i++] = '\0';

  while ((s[i] == ' ') || (s[i] == '\t'))
    i++;
  *pathp = &s[i];
  while ((s[i] != ' ') && (s[i] != '\t'))
    i++;
  s[i++] = '\0';

  while ((s[i] == ' ') || (s[i] == '\t'))
    i++;
  typename = &s[i];
  while ((s[i] != ' ') && (s[i] != '\t') && (s[i] != '\n'))
    i++;
  s[i++] = '\0';

  if (strcmp(typename, "SONY") == 0){
    *typep = VF_FONT_SONY;
  } else if (strcmp(typename, "ZEIT") == 0){
    *typep = VF_FONT_ZEIT;
  } else if (strcmp(typename, "JG") == 0){
    *typep = VF_FONT_JG;
  } else {
    fprintf(stderr, "Unknown font type in %s: %s\n", typename, conf);
    exit(-1);
  }
  return 0;
}
