/*
 * print-out DVI
 *
 * Copyright (c) 1993
 *      MATSUURA Syun           syun@fuka.info.waseda.ac.jp
 *      HIRAHARA Atsushi        hirahara@fuka.info.waseda.ac.jp
 *      ONO Kouichi             onono@fuka.info.waseda.ac.jp
 * All rights reserved.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Cardinals.h>

#include "xdvi.h"
#include "print.h"
#ifdef MARKPAGE
#include "markpage.h"
#endif /* MARKPAGE */

enum PRINTSTYLE
#ifdef MARKPAGE
{PRINTALL, PRINTCUR, PRINTMARKED, PRINTREGION, PRINTALLMARKED};
#else
{PRINTALL, PRINTCUR};
#endif /* MARKPAGE */

static String print_style[] =
#ifdef MARKPAGE
{LBLPRINTALL,  LBLPRINTCUR,  LBLPRINTMRK,  LBLPRINTRGN,  LBLPRINTAMK};
#else
{LBLPRINTALL,  LBLPRINTCUR};
#endif /* MARKPAGE */

static String print_command[] =
#ifdef MARKPAGE
{PRCMDALL,  PRCMDCUR,  PRCMDMRK,  PRCMDRGN,  PRCMDLST};
#else
{PRCMDALL,  PRCMDCUR};
#endif /* MARKPAGE */

static char     printer_name[PRINTERNAMELEN];

static Boolean  PrintMenuEnd = False;

static Widget   popup, dialog, print_button, print_menu;

static void
  GetBaseName(pathname, basename)
char pathname[STRLENGTH], basename[STRLENGTH];
{
  int len,i;
  int found = -1;
  
  len = strlen(pathname);
  
  for(i=len-1; i>found; i--) {
    if(pathname[i]==PATH_DELIMITER) /* found it */
      found = i;
  }
  if (found <= -1) /* not found */
    found = -1;
  strcpy(basename, pathname + found + 1);
}

static void
  GetDirName(pathname, dirname)
char pathname[STRLENGTH], dirname[STRLENGTH];
{
  int len,i;
  int found = -1;
  char copypathname[STRLENGTH];

  strcpy(copypathname, pathname);
  
  len = strlen(copypathname);
  
  for(i=len-1; i>found; i--) {
    if(copypathname[i]==PATH_DELIMITER) /* found it */
      found = i;
  }

  if (found==0) {
    /* found, and directory is "/" */
    copypathname[0] = PATH_DELIMITER;
    copypathname[1] = NULL;
  }
  else if (found > 0) {
    /* found, and directory is "/xxx/yyy/zz..." */
    copypathname[found] = NULL;
  }
  else {
    /* not found, pathname may be plain filename */
    /* copy current directory */
    copypathname[0] = '.';
    copypathname[1] = NULL;
  }
  strcpy(dirname, copypathname);
}

static void
  SubstitutePrintVar(ResultStr, SourceStr)
char ResultStr[STRLENGTH], SourceStr[STRLENGTH];
{
  char SourceChar[STRLENGTH], DestStr[STRLENGTH], StrBuff[STRLENGTH];
  int charpos, strlength;
  PAGE          PageList[LISTSIZE];
  register      LIST_LOCATION LocList, EndLocList;
  
  strcpy(SourceChar, SourceStr);
  strlength=strlen(SourceChar);
  DestStr[0] = NULL;
  
  for(charpos=0; charpos<strlength; charpos++) {
    if (SourceChar[charpos]==VARMARK) { /* Is it a variable ? */
      /* Probably it is a variable */
      charpos++;
      if (charpos<strlength) { /* Is the source string end ? */
	/* No, the source string has more characters */
	switch (SourceChar[charpos]) {
	case VARMARK:
	  StrBuff[0] = VARMARK;
	  StrBuff[1] = NULL;
	  strcat(DestStr, StrBuff);
	  break;
	case 'f':
	  strcat(DestStr, dvi_name);
	  break;
	case 'F':
	  GetBaseName(dvi_name,StrBuff);
	  strcat(DestStr, StrBuff);
	  break;
	case 'd':
	  GetDirName(dvi_name,StrBuff);
	  strcat(DestStr, StrBuff);
	  break;
	case 'p':
	  strcat(DestStr, printer_name);
	  break;
	case 'r':
	  sprintf(StrBuff, "%d", getpid());
	  strcat(DestStr, StrBuff);
	  break;
	case 'c':
	  sprintf(StrBuff, "%d", current_page + 1);
	  strcat(DestStr, StrBuff);
	  break;
#ifdef MARKPAGE
	case 'm':
	  sprintf(StrBuff, "%d", LastMarkPage() + 1);
	  strcat(DestStr, StrBuff);
	  break;
	case 's':
	  sprintf(StrBuff, "%d", MIN(current_page + 1, LastMarkPage() + 1));
	  strcat(DestStr, StrBuff);
	  break;
	case 'e':
	  sprintf(StrBuff, "%d", MAX(current_page + 1, LastMarkPage()+ 1));
	  strcat(DestStr, StrBuff);
	  break;
	case 'l':
	  GetMarkedPageListAsString(StrBuff, NULL, NULL, ',');
	  strcat(DestStr, StrBuff);
	  break;
	case 'L':
	  GetMarkedPageListAsString(StrBuff, NULL, NULL, ' ');
	  strcat(DestStr, StrBuff);
	  break;
	case 't':
	  GetSortedMarkedPageListAsString(StrBuff, NULL, NULL, ',');
	  strcat(DestStr, StrBuff);
	  break;
	case 'T':
	  GetSortedMarkedPageListAsString(StrBuff, NULL, NULL, ' ');
	  strcat(DestStr, StrBuff);
	  break;
	case 'k':
	  GetMarkedPageListAsString(StrBuff, '=', NULL, ',');
	  strcat(DestStr, StrBuff);
	  break;
	case 'K':
	  GetMarkedPageListAsString(StrBuff, '=', NULL, ' ');
	  strcat(DestStr, StrBuff);
	  break;
	case 'a':
	  GetSortedMarkedPageListAsString(StrBuff, '=', NULL, ',');
	  strcat(DestStr, StrBuff);
	  break;
	case 'A':
	  GetSortedMarkedPageListAsString(StrBuff, '=', NULL, ' ');
	  strcat(DestStr, StrBuff);
	  break;
#endif /* MARKPAGE */
	default:
	  StrBuff[0] = VARMARK;
	  StrBuff[1] = SourceChar[charpos];
	  StrBuff[2] = NULL;
	  strcat(DestStr, StrBuff);
	  break;
	}
      }
      else { /* The source string is end */
	StrBuff[0] = VARMARK;
	StrBuff[1] = NULL;
	strcat(DestStr, StrBuff);
	charpos--;
      }
    }
    else { /* It is not a variable */
      StrBuff[0] = SourceChar[charpos];
      StrBuff[1] = NULL;
      strcat(DestStr, StrBuff);
    }
  }
  strcpy(ResultStr, DestStr);
}

static void
  close_all_font_file()
{
  register struct font *fontp;
  
  for (fontp = font_head; fontp != NULL; fontp = fontp->next)
    if (fontp->file != NULL) {
      Fclose(fontp->file);
    }
}

static void
  PrintDVI(widget, client_data, call_data)
Widget  widget;
XtPointer client_data, call_data;
{
  int   PrintStyle, pid;
  
  PrintStyle = (int)client_data;

  strcpy(printer_name, (char *)XawDialogGetValueString(dialog));
  if (*printer_name==NULL) {
    /* When the input of dialog widget as printer name is null string, */
    /* Let it the default printer name */
#ifdef DVIPRINTER
    strcpy(printer_name, DVIPRINTER);
#else
    strcpy(printer_name, DEFAULTPRINTER);
#endif /* DVIPRINTER */
  }

  if ((pid = fork ()) < 0) {
    perror("fork");
    return;
  }
  /* Actual print-out work should be done with the child process */
  /* Because the work takes long time */
  
  if (pid == 0) {
    /* The following is the work for the child process */
    /* It is the actual print-out work */
    int  i;
    char buff[STRLENGTH];
    
    close_all_font_file();
    /* Forcedly close all font files */
    
    SubstitutePrintVar(buff, print_command[PrintStyle]);
    system(buff);
    exit(0);      
  }

  PrintMenuEnd = True;
  /* Print-out pull down menu should be closed */
}

static void 
  DestroyPopupPrompt(widget, client_data, call_data)
Widget  widget;
XtPointer client_data, call_data;
{
  PrintMenuEnd = True;
  /* Print-out pull down menu should be closed */
}

static char *
  StrDup(str)
char *str;
{
  char *ptr = xmalloc(strlen(str) + 1, "strdup");
  return(strcpy(ptr, str));
}

static char *
  get_printer_name()
{
  char  *p;
  
  if( p = getenv(ENVVARPRINTER) )
    return((char *)StrDup(p));
  else
#ifdef DVIPRINTER
    return((char *)StrDup(DVIPRINTER));
#else
    return((char *)StrDup(DEFAULTPRINTER));
#endif /* DVIPRINTER */
}

void
  InitPrintMenu()
{
  strcpy(printer_name, get_printer_name());
}

void
  print_DVI_file(top_level)
Widget top_level;
{
  Widget        entry;
  Position      x, y;
  Dimension     width, height;
  int           i;
  XtAppContext  PrintApp;
  XEvent        event;
  
  XtVaGetValues(top_level,
		XtNwidth, &width,
		XtNheight, &height,
		NULL);
  XtTranslateCoords(top_level,
		    (Position) (width / 3), (Position) (height / 3),
		    &x, &y);
  

  popup = XtVaCreatePopupShell(LBLPRINTSH,
			       transientShellWidgetClass,
			       top_level,
			       XtNx, x,
			       XtNy, y,
			       NULL);
  /* 
   * The popup will contain a dialog box, prompting the user for input. 
   */
  
  dialog = XtVaCreateManagedWidget(LBLPRINTER,
				   dialogWidgetClass,
				   popup,
				   XtNlabel,    (String)LBLPRINTER,
				   XtNvalue,    (String)printer_name,
				   NULL);

  /*
   * The prompting message's size is dynamic; allow it to request resize. 
   */
  
  print_button = XtVaCreateManagedWidget(LBLPRINTOUT,
					 menuButtonWidgetClass,
					 dialog,
					 NULL);
  XtVaSetValues(print_button,
		XtNmenuName, LBLPRINTMENU,
		NULL);
  
  XawDialogAddButton(dialog,
		     LBLCANCEL,
		     DestroyPopupPrompt,
		     (XtPointer)dialog);
  
  print_menu = XtVaCreatePopupShell(LBLPRINTMENU,
				    simpleMenuWidgetClass,
				    print_button,
				    NULL);
  
  for(i = 0; (int)XtNumber(print_style) > i; i++) {
    switch (i) {
    case PRINTALL:
    case PRINTCUR:
      entry = XtVaCreateManagedWidget(print_style[i],
				      smeBSBObjectClass,
				      print_menu,
				      NULL);
      XtAddCallback(entry, XtNcallback, PrintDVI, (XtPointer)i);
      break;
#ifdef MARKPAGE
    case PRINTMARKED:
    case PRINTREGION:
    case PRINTALLMARKED:
      if (EmptyPageMarkRing()) {
	entry = XtVaCreateManagedWidget(print_style[i],
					smeBSBObjectClass,
					print_menu,
					XtNsensitive, False,
					NULL);
      }
      else {
	entry = XtVaCreateManagedWidget(print_style[i],
					smeBSBObjectClass,
					print_menu,
					NULL);
      }
      XtAddCallback(entry, XtNcallback, PrintDVI, (XtPointer)i);
      break;
#endif
    default:
      break;
    }
  }

  XtPopup(popup, XtGrabNone);

  PrintMenuEnd = False;

  XtAddGrab(popup, True, True);

  PrintApp = XtWidgetToApplicationContext(popup);

  while(! PrintMenuEnd) {
    XtAppNextEvent(PrintApp, &event);
    XtDispatchEvent(&event);
  };

  XtDestroyWidget(popup);
}
