/* 
 * Tpic 2.2 routines are drived from SeeTeX/Xtex/DviPage2.c.
 * Followings are original Copyright.
 */

/*
 * Copyright 1989 Dirk Grunwald
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of Dirk Grunwald or M.I.T.
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  Dirk
 * Grunwald and M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 * DIRK GRUNWALD AND M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL M.I.T.  BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * Author:
 * 	Dr. Dirk Grunwald
 * 	Dept. of Computer Science
 * 	Campus Box 430
 * 	Univ. of Colorado, Boulder
 * 	Boulder, CO 80309
 * 
 * 	grunwald@colorado.edu
 * 	
 */ 

/***********************************************************************
 *
 *	Tpic/tpic2 specials
 *
 ***********************************************************************/

/*
 * Improvements by Martin Jourdan, INRIA (jourdan@minos.inria.fr),
 * April 1990.
 *
 * 1. Accomodation of Tpic 2 output language; most of the stuff was
 *    written by Tim Morgan, but I modified it a little.
 * 2. Shading is now completely effective.  There are five levels of
 *    shading:
 *	fillval < 0.1:		white
 *	0.1 <= fillval < 0.35:	lightgray 25% density
 *	0.35 <= fillval <= 0.65: midgray 50% density
 *	0.65 < fillval <= 0.9:	darkgray 75% density
 *	fillval > 0.9:		black
 *    (see functions "getTile", "fill_path" and "fill_arc")
 *    As requested by the "specification" of Tpic 2, default fillval
 *    is 0.5, except if textures are used (see below).
 *    Shading applies to *complete* ellipses/circles ("fill_arc") and
 *    to arbitrary closed but *convex* polygons ("fill_path").  So
 *    triangular solid arrowheads come out nicely.  However the
 *    specification of Tpic2 does not *guarantee* that polygons are
 *    actually convex, it's just an assumption I make for sake of
 *    speed (?); cross your fingers!
 *    In addition to being shaded, objects can be invisible or dotted
 *    or dashed (in the source text); however I assume, as is
 *    presently the case, that, *in the output*, only solid or
 *    invisible objects can be shaded; more precisely a shaded
 *    dotted/dashed object is output twice, first as an invisible
 *    shaded object then as a non-shaded dotted/dashed object.  Beware
 *    of changes by Tim Morgan!
 * 3. Textures are supported (for poor ole guys who only have tpic1):
 *    the patterns are analyzed to compute their density (ratio of
 *    "black" bits to total number of bits), which is then used as a
 *    valid fillval that overrides the default of 0.5.
 * 4. Other changes:
 *  - I stole from "xdvi" a fast routine to draw complete ellipses and
 *    circles, because XDrawArc is awfully slow on my machine.
 *  - I took care of the very small arcs generated by tpic2 for dotted
 *    ellipses, which put too heavy a strain on my X server.
 *  - I made changes to pen size work (see "set_pen_size").
 *  - I added dotted/dashed decoding for splines but did not do more
 *    because I can't figure out what "flush_spline" does...
 * Remains to be done:
 *  - make better use of updateRegion
 *  - move some initializations (wDepth, GCArcMode) to DviPage.c, at
 *    window/GC creation time
 *  - clean up code; for instance, some routines work on
 *    Tpic-coordinates while some others work on X-coordinates...
 *  - implement dotted/dashed splines?
 */

/*
 * Support drawing routines for TeXsun and TeX
 *
 *      Copyright, (C) 1987, 1988 Tim Morgan, UC Irvine
 *
 * At the time these routines are called, the values of hh and vv should
 * have been updated to the upper left corner of the graph (the position
 * the \special appears at in the dvi file).  Then the coordinates in the
 * graphics commands are in terms of a virtual page with axes oriented the
 * same as the Imagen and the SUN normally have:
 *
 *                      0,0
 *                       +-----------> +x
 *                       |
 *                       |
 *                       |
 *                      \ /
 *                       +y
 *
 * Angles are measured in the conventional way, from +x towards +y.
 * Unfortunately, that reverses the meaning of "counterclockwise"
 * from what it's normally thought of.
 *
 * A lot of floating point arithmetic has been converted to integer
 * arithmetic for speed.  In some places, this is kind-of kludgy, but
 * it's worth it.
 */

#include <math.h>
#include <ctype.h>
#include <X11/bitmaps/light_gray>
#include <X11/bitmaps/flipped_gray>
#include <X11/bitmaps/gray>

#include "xdvi.h"

#if	!defined(X10) && XlibSpecificationRelease >= 4
#define	HAVE_XGETGCVALUES
#endif

EXTERN	Display *DISP;
EXTERN	GC	ruleGC;
EXTERN	GC	tpicGC;
EXTERN	int	min_x, max_x, min_y, max_y;


#ifndef	X_NOT_STDC_ENV
#include <stdlib.h>
#endif

#define	MAXPOINTS	300	/* Max points in a path */
#define	TWOPI		(3.14159265359*2.0)
#define	MAX_PEN_SIZE	7	/* Max pixels of pen width */


static	int	xx[MAXPOINTS], yy[MAXPOINTS];	/* Path in milli-inches */
static	int	path_len = 0;	/* # points in current path */
static	int	pen_size = 1;	/* Pixel width of lines drawn */

static	double	shading = -1.0;
static	double	txshading = -1.0;

/* Unfortunately, these values also appear in dvisun.c */
#define	xRESOLUTION	(pixels_per_inch/shrink_factor)
#define	yRESOLUTION	(pixels_per_inch/shrink_factor)


/*
 * Issue warning messages
 */
static	void
Warning(fmt, msg)
	char	*fmt, *msg;
{
	Fprintf(stderr, fmt, msg);
	(void) fputc('\n', stderr);
}


/********************************
 *	  tpic routines		*
 *******************************/

/* Things we need from spec_draw, unfortunately */

/* (ignored for now)
extern int pen_size, blacken, whiten, shade;
*/

#define	toint(x)	((int) ((x) + 0.5))
#define	conv(x)		(toint(specialConv*(x)/shrink_factor))
#define	xconv(x)	(toint(specialConv*(x)/shrink_factor) + PXL_H)
#define	yconv(y)	(toint(specialConv*(y)/shrink_factor) + PXL_V)
#define	vxconv(x)	(-toint(specialConv*(x)/shrink_factor) + PXL_H)

/*
 *	Draw a line from (fx,fy) to (tx,ty).
 *	Right now, we ignore pen_size.
 */
void
line_btw(fx, fy, tx, ty)
	int	fx, fy, tx, ty;
{
        register int    fcx = xconv(fx),
                        tcx = xconv(tx),
                        fcy = yconv(fy),
                        tcy = yconv(ty);

	if ((fcx < max_x || tcx < max_x) && (fcx >= min_x || tcx >= min_x) &&
	    (fcy < max_y || tcy < max_y) && (fcy >= min_y || tcy >= min_y))
#ifndef X10
		XDrawLine(DISP, WINDOW(currwin), tpicGC,
		    fcx - currwin.base_x, fcy - currwin.base_y,
		    tcx - currwin.base_x, tcy - currwin.base_y);
#else
		XLine(WINDOW(currwin),
		    fcx - currwin.base_x, fcy - currwin.base_y,
		    tcx - currwin.base_x, tcy - currwin.base_y,
		    1, 1, ruleGC, GXcopy, AllPlanes);
#endif
}

/*
 *	Draw a dot at (x,y)
 */
void
dot_at(x, y)
	int	x, y;
{
	register int	cx = xconv(x),
                        cy = yconv(y);

	if (cx < max_x && cx >= min_x && cy < max_y && cy >= min_y)
#ifndef X10
	    XDrawPoint(DISP, WINDOW(currwin), tpicGC,
		cx - currwin.base_x, cy - currwin.base_y);
#else
	    XPixSet(WINDOW(currwin), cx - currwin.base_x, cy - currwin.base_y,
		1, 1, ruleGC);
#endif
}


/*
 * Return a tile for shading
 * Three levels of shading (in addition to black and white) are
 * available, because I use standard bitmaps found in the X distribution.
 */
static Pixmap
getTile(shading)
	double shading;
{
	static Pixmap	shadeTiles[3];  
	static int	wDepth = 0;
	Pixmap		the_tile;
	XGCValues	tpicGC_values;

#ifdef	HAVE_XGETGCVALUES
	XGetGCValues(DISP, tpicGC, GCForeground|GCBackground, &tpicGC_values);
#else
	tpicGC_values = tpicGC->values;
#endif

	if (shading < 0.35) {	/* lightgray 25% density */
	    the_tile = shadeTiles[0];
	    if (!the_tile) {
	        if (wDepth == 0) { /* this code belongs to DviPage.c */
		/* the depth won't change during a session */
		    XWindowAttributes wAttrs;
		    XGetWindowAttributes(DISP, WINDOW(currwin), &wAttrs);
		    wDepth = wAttrs.depth;
	        }
	        shadeTiles[0] = the_tile = 
		      XCreatePixmapFromBitmapData(DISP, WINDOW(currwin),
						  light_gray_bits,
						  light_gray_width,
						  light_gray_height,
						  tpicGC_values.foreground,
						  tpicGC_values.background,
						  wDepth);
	    }
	} else if (shading > 0.65) { /* darkgray 75% density */
	    the_tile = shadeTiles[2];
	    if (!the_tile) {
	        if (wDepth == 0) {
		    XWindowAttributes wAttrs;
		    XGetWindowAttributes(DISP, WINDOW(currwin), &wAttrs);
		    wDepth = wAttrs.depth;
	        }
	        shadeTiles[2] = the_tile =
		    XCreatePixmapFromBitmapData(DISP, WINDOW(currwin),
						flipped_gray_bits,
						flipped_gray_width,
						flipped_gray_height,
						tpicGC_values.foreground,
						tpicGC_values.background,
						wDepth);
	    }
	} else {	/* midgray 50% density */
	    the_tile = shadeTiles[1];
	    if (!the_tile) {
	        if (wDepth == 0) {
		    XWindowAttributes wAttrs;
		    XGetWindowAttributes(DISP, WINDOW(currwin), &wAttrs);
		    wDepth = wAttrs.depth;
	        }
	        shadeTiles[1] = the_tile =
		    XCreatePixmapFromBitmapData(DISP, WINDOW(currwin),
						gray_bits,
						gray_width,
						gray_height,
						tpicGC_values.foreground,
						tpicGC_values.background,
						wDepth);
	    }
	}
	return the_tile;
}


/*
 * Shade the given polygon -- X-coordinates.  Reset shading.
 * The polygon is assumed to be convex!
 */
static void
fill_path(xxyy, path_len)
	XPoint	*xxyy;
	int	path_len;
{  
	unsigned long fore, back;
	XGCValues	tpicGC_values;

#ifdef	HAVE_XGETGCVALUES
	XGetGCValues(DISP, tpicGC, GCForeground|GCBackground, &tpicGC_values);
#else
	tpicGC_values = tpicGC->values;
#endif

	if (shading >= 0.0) {
	    if (shading < 0.1) {	/* white */
	        fore = tpicGC_values.foreground;
	        back = tpicGC_values.background;
		XSetForeground(DISP, tpicGC, back);
		XSetBackground(DISP, tpicGC, fore);
		XFillPolygon(DISP, WINDOW(currwin), tpicGC,
			     xxyy, path_len, Complex, CoordModeOrigin);
		XSetForeground(DISP, tpicGC, fore);
		XSetBackground(DISP, tpicGC, back);
	    } else if (shading > 0.9) { /* black */
	        XFillPolygon(DISP, WINDOW(currwin), tpicGC,
			     xxyy, path_len, Complex, CoordModeOrigin);
	    } else {		/* gray -- one of three levels */
	        XGCValues	values;
		/* temporarily set fill_style to Tiled */
		values.fill_style = FillTiled;
		values.tile = getTile(shading);
		XChangeGC(DISP, tpicGC, (unsigned)(GCFillStyle | GCTile),
			  &values);
		/* fill */
		XFillPolygon(DISP, WINDOW(currwin), tpicGC,
			     xxyy, path_len, Complex, CoordModeOrigin);
		/* reset fill_style to Solid */
		values.fill_style = FillSolid;
		XChangeGC(DISP, tpicGC, (unsigned)GCFillStyle, &values);
	    }
	}
	shading = -1.0;		/* reset shading */
}

/*
 * Shade the given ellipse or circle -- X-coordinates.  Reset shading.
 * The ellipse or circle is assumed to be complete!
 */
static void
fill_arc(x, y, wd, ht)
	int	x, y;
	unsigned int	wd, ht;
{
	static arcModeInit = 0;
	unsigned long fore, back;
	XGCValues	tpicGC_values;

#ifdef	HAVE_XGETGCVALUES
	XGetGCValues(DISP, tpicGC, GCForeground|GCBackground, &tpicGC_values);
#else
	tpicGC_values = tpicGC->values;
#endif

	if (!arcModeInit) {
	/* This code belongs to DviPage.c */
	    XGCValues	values;

	    values.arc_mode = ArcPieSlice;
	    XChangeGC(DISP, tpicGC, (unsigned)GCArcMode, &values);

	    arcModeInit = 1;
	}
  
	if (shading >= 0.0) {
	    if (shading < 0.1) {
	        fore = tpicGC_values.foreground;
	        back = tpicGC_values.background;
		XSetForeground(DISP, tpicGC, back);
		XSetBackground(DISP, tpicGC, fore);
	        XFillArc(DISP, WINDOW(currwin), tpicGC,
			 x, y, wd, ht, 0, 360*64);
		XSetForeground(DISP, tpicGC, fore);
		XSetBackground(DISP, tpicGC, back);
	    } else if (shading > 0.9) {
	        XFillArc(DISP, WINDOW(currwin), tpicGC, 
			 x, y, wd, ht, 0, 360*64);
	    } else {
	        XGCValues	values;

		values.fill_style = FillTiled;
		values.tile = getTile(shading);
		XChangeGC(DISP, tpicGC, (unsigned)(GCFillStyle | GCTile),
			  &values);

		/* XFillArc is awfully slow but it's worth it */
		XFillArc(DISP, WINDOW(currwin), tpicGC,
			 x, y, wd, ht, 0, 360*64);

		values.fill_style = FillSolid;
		XChangeGC(DISP, tpicGC, (unsigned)GCFillStyle, &values);
	    }
        }
	shading = -1.0;
}


/*
 * Set the size of the virtual pen used to draw in milli-inches
 */
/* ARGSUSED */
static	void
set_pen_size(cp)
	char *cp;
{
	int	ps;
        XGCValues       values;

	if (sscanf(cp, " %d ", &ps) != 1) {
	    Warning("illegal .ps command format: %s", cp);
	    return;
	}
	pen_size = (ps * (xRESOLUTION + yRESOLUTION) + 1000) / 2000;
	if (pen_size < 1) pen_size = 1;
	else if (pen_size > MAX_PEN_SIZE) pen_size = MAX_PEN_SIZE;

	/* update paintGC -> line_width */
	values.line_width = pen_size;
	XChangeGC(DISP, tpicGC, (unsigned) GCLineWidth, &values);
}


/*
 * Print the line defined by previous path commands
 */
static	void
flush_path(invis)
	Boolean	invis;
{
	register int i;
	static XPoint	xxyy[MAXPOINTS];

	for (i = 1; i <= path_len; i++) {
	  xxyy[i-1].x = xconv(xx[i]) - currwin.base_x;
	  xxyy[i-1].y = yconv(yy[i]) - currwin.base_y;
	}

	fill_path(xxyy, path_len);

	if (!invis) {
	    if (path_len > 2) {
	        XDrawLines(DISP, WINDOW(currwin), tpicGC, 
			   xxyy, path_len, CoordModeOrigin);
	    } else {
	        for (i = 1; i < path_len; i++)
		    line_btw(xx[i], yy[i], xx[i+1], yy[i+1]);
	    }
	}
	path_len = 0;
}


/*
 * Print a dashed line along the previously defined path, with
 * the dashes/inch defined.
 */

static	void
flush_dashed(cp, dotted)
	char	*cp;
	int	dotted;
{
	int	i;
	int	numdots;
	int	lx0, ly0, lx1, ly1;
	int	cx0, cy0, cx1, cy1;
	float	inchesperdash;
	double	d, spacesize, a, b, dx, dy, milliperdash;

	if (sscanf(cp, " %f ", &inchesperdash) != 1) {
	    Warning("illegal format for dotted/dashed line: %s", cp);
	    return;
	}
	if (path_len <= 1 || inchesperdash <= 0.0) {
	    Warning("illegal conditions for dotted/dashed line", "");
	    return;
	}
	milliperdash = inchesperdash * 1000.0;
	lx0 = xx[1];	ly0 = yy[1];
	lx1 = xx[2];	ly1 = yy[2];
	dx = lx1 - lx0;
	dy = ly1 - ly0;
	if (dotted) {
	    numdots = sqrt(dx*dx + dy*dy) / milliperdash + 0.5;
	    if (numdots == 0) numdots = 1;
	    for (i = 0; i <= numdots; i++) {
		a = (float) i / (float) numdots;
		cx0 = lx0 + a * dx + 0.5;
		cy0 = ly0 + a * dy + 0.5;
		dot_at(cx0, cy0);
	    }
	}
	else {
	    d = sqrt(dx*dx + dy*dy);
	    numdots = d / (2.0 * milliperdash) + 1.0;
	    if (numdots <= 1)
		line_btw(lx0, ly0, lx1, ly1);
	    else {
		spacesize = (d - numdots * milliperdash) / (numdots - 1);
		for (i = 0; i < numdots - 1; i++) {
		    a = i * (milliperdash + spacesize) / d;
		    b = a + milliperdash / d;
		    cx0 = lx0 + a * dx + 0.5;
		    cy0 = ly0 + a * dy + 0.5;
		    cx1 = lx0 + b * dx + 0.5;
		    cy1 = ly0 + b * dy + 0.5;
		    line_btw(cx0, cy0, cx1, cy1);
		    b += spacesize / d;
		}
		cx0 = lx0 + b * dx + 0.5;
		cy0 = ly0 + b * dy + 0.5;
		line_btw(cx0, cy0, lx1, ly1);
	    }
	}
	path_len = 0;
}


/*
 * Add a point to the current path
 */
static	void
add_path(cp)
	char	*cp;
{
	int	pathx, pathy;

	if (++path_len >= MAXPOINTS) oops("Too many points");
	if (sscanf(cp, " %d %d ", &pathx, &pathy) != 2)
	    oops("Malformed path command");
	xx[path_len] = pathx;
	yy[path_len] = pathy;
}


/*
 * Draw to a floating point position
 */
static void
im_fdraw(x, y)
	double	x,y;
{
	if (++path_len >= MAXPOINTS) oops("Too many arc points");
	xx[path_len] = x + 0.5;
	yy[path_len] = y + 0.5;
}


/*
 * Draw an ellipse with the indicated center and radices.
 */
static	void
draw_ellipse(xc, yc, xr, yr)
	int	xc, yc, xr, yr;
{
	double	angle, theta;
	int	n;
	int	px0, py0, px1, py1;

	angle = (xr + yr) / 2.0;
	theta = sqrt(1.0 / angle);
	n = TWOPI / theta + 0.5;
	if (n < 12) n = 12;
	else if (n > 80) n = 80;
	n /= 2;
	theta = TWOPI / n;

	angle = 0.0;
	px0 = xc + xr;		/* cos(0) = 1 */
	py0 = yc;		/* sin(0) = 0 */
	while ((angle += theta) <= TWOPI) {
	    px1 = xc + xr*cos(angle) + 0.5;
	    py1 = yc + yr*sin(angle) + 0.5;
	    line_btw(px0, py0, px1, py1);
	    px0 = px1;
	    py0 = py1;
	}
	line_btw(px0, py0, xc + xr, yc);
}

/*
 * Draw an arc
 */
static	void
arc(cp, invis)
	char	*cp;
	Boolean	invis;
{
	int	xc, yc, xrad, yrad, n;
	float	start_angle, end_angle, angle, theta, r;
	double	xradius, yradius, xcenter, ycenter;

	if (sscanf(cp, " %d %d %d %d %f %f ", &xc, &yc, &xrad, &yrad,
		&start_angle, &end_angle) != 6) {
	    Warning("illegal arc specification: %s", cp);
	    return;
	}

	if (shading >= 0.0) {
	    if (start_angle <= 0.0 && end_angle >= 6.282) {
		fill_arc(xconv(xc)-conv(xrad)-currwin.base_x,
			 yconv(yc)-conv(yrad)-currwin.base_y,
			 (unsigned)2*conv(xrad), (unsigned)2*conv(yrad));
	    }
	}

	if (invis) return;

	/* We have a specialized fast way to draw closed circles/ellipses */
	if (start_angle <= 0.0 && end_angle >= 6.282) {
	    draw_ellipse(xc, yc, xrad, yrad);
	    return;
	}
	xcenter = xc;
	ycenter = yc;
	xradius = xrad;
	yradius = yrad;
	r = (xradius + yradius) / 2.0;
	theta = sqrt(1.0 / r);
	n = 0.3 * TWOPI / theta + 0.5;
	if (n < 12) n = 12;
	else if (n > 80) n = 80;
	n /= 2;
	theta = TWOPI / n;
	flush_path(False);
	im_fdraw(xcenter + xradius * cos(start_angle),
	    ycenter + yradius * sin(start_angle));
	angle = start_angle + theta;
	if (end_angle < angle) end_angle += TWOPI;
	while (angle < end_angle) {
	    im_fdraw(xcenter + xradius * cos(angle),
		ycenter + yradius * sin(angle));
	    angle += theta;
	}
	im_fdraw(xcenter + xradius * cos(end_angle),
	    ycenter + yradius * sin(end_angle));
	flush_path(False);
}


/*
 * APPROXIMATE integer distance between two points
 */
#define	dist(x0, y0, x1, y1)	(abs(x0-x1)+abs(y0-y1))


/*
 * Draw a spline along the previously defined path
 */
static	void
flush_spline(cp)
	char	*cp;
{
	int	xp, yp;
	int	N;
	int	lastx, lasty;
	Boolean	lastvalid = False;
	int	t1, t2, t3;
	int	steps;
	int	j;
	register int i, w;
	float inchesperdash;
	int spStyle;

	if (cp && *cp) {
	    if (sscanf(cp, " %f ", &inchesperdash) != 1) {
		Warning("illegal format for dotted/dashed line: %s", cp);
		spStyle = LineSolid;
	    } else if (inchesperdash < 0.0) {
	        spStyle = LineOnOffDash;
		inchesperdash = -inchesperdash;
	    } else if (inchesperdash > 0.0)
	        spStyle = LineOnOffDash;
	    else spStyle = LineSolid;
	} else spStyle = LineSolid;
	XSetLineAttributes(DISP, tpicGC, pen_size, spStyle, CapButt, JoinMiter);

#ifdef	lint
	lastx = lasty = -1;
#endif
	N = path_len + 1;
	xx[0] = xx[1];
	yy[0] = yy[1];
	xx[N] = xx[N-1];
	yy[N] = yy[N-1];
	for (i = 0; i < N - 1; i++) {	/* interval */
	    steps = (dist(xx[i], yy[i], xx[i+1], yy[i+1]) +
		dist(xx[i+1], yy[i+1], xx[i+2], yy[i+2])) / 80;
	    for (j = 0; j < steps; j++) {	/* points within */
		w = (j * 1000 + 500) / steps;
		t1 = w * w / 20;
		w -= 500;
		t2 = (750000 - w * w) / 10;
		w -= 500;
		t3 = w * w / 20;
		xp = (t1*xx[i+2] + t2*xx[i+1] + t3*xx[i] + 50000) / 100000;
		yp = (t1*yy[i+2] + t2*yy[i+1] + t3*yy[i] + 50000) / 100000;
		if (lastvalid) line_btw(lastx, lasty, xp, yp);
		lastx = xp;
		lasty = yp;
		lastvalid = True;
	    }
	}
	path_len = 0;
}


/*
 * Process textures (.tx command): analyse density and set txshading
 */
static void
Texture(cp)
	register char	*cp;
{
	int blackbits = 0, totalbits = 0;

	while (*cp) {
	    switch (*cp) {
		case '0':
		    totalbits += 4;
		    break;
		case '1': case '2': case '4': case '8':
		    blackbits += 1;
		    totalbits += 4;
		    break;
		case '3': case '5': case '6': case '9':
		case 'a': case 'A': case 'c': case 'C':
		    blackbits += 2;
		    totalbits += 4;
		    break;
		case '7': case 'b': case 'B': case 'd':
		case 'D': case 'e': case 'E':
		    blackbits += 3;
		    totalbits += 4;
		    break;
		case 'f': case 'F':
		    blackbits += 4;
		    totalbits += 4;
		    break;
		case ' ':
		    break;
		default:
		    Warning("Invalid character in .tx pattern: \"%c\"", *cp);
		    break;
	    }
	    cp++;
	}
	txshading = (double) blackbits / (double) totalbits;
}                               /* end of Texture */


/*
 * Shade the last box, circle, or ellipse
 */
static	void
shade_last(cp)
	char	*cp;
{
    if (cp && *cp) {
	if (sscanf(cp, "%lf", &shading) != 1)
	    Warning("illegal format for shading value: %s", cp);
	else if (shading < 0.0 || shading > 1.0) {
	    Warning("bad shading value: %f", shading);
	    shading = 0.5;
	}
    }
    else if (txshading >= 0.0)	/* account for textures */
	shading = txshading;
    else shading = 0.5;		/* default value as requested by tpic2 */
}


/*
 *	The following copyright message applies to the rest of this file.  --PV
 */

/*
 *	This program is Copyright (C) 1987 by the Board of Trustees of the
 *	University of Illinois, and by the author Dirk Grunwald.
 *
 *	This program may be freely copied, as long as this copyright
 *	message remaines affixed. It may not be sold, although it may
 *	be distributed with other software which is sold. If the
 *	software is distributed, the source code must be made available.
 *
 *	No warranty, expressed or implied, is given with this software.
 *	It is presented in the hope that it will prove useful.
 *
 *	Hacked in ignorance and desperation by jonah@db.toronto.edu
 */

/*
 *      The code to handle the \specials generated by tpic was modified
 *      by Dirk Grunwald using the code Tim Morgan at Univ. of Calif, Irvine
 *      wrote for TeXsun.
 */

#define	COMLEN	4

void
applicationDoSpecial(cp)
	char	*cp;
{
	char	command[COMLEN], *orig_cp;
	register int len;

	orig_cp = cp;
	while (isspace(*cp)) ++cp;
	len = 0;
	while (!isspace(*cp) && *cp && len < COMLEN-1) command[len++] = *cp++;
	command[len] = '\0';
	if (strcmp(command, "pn") == 0) set_pen_size(cp);
	else if (strcmp(command, "fp") == 0) flush_path(False);
	else if (strcmp(command, "da") == 0) flush_dashed(cp, 0);
	else if (strcmp(command, "dt") == 0) flush_dashed(cp, 1);
	else if (strcmp(command, "pa") == 0) add_path(cp);
	else if (strcmp(command, "ar") == 0) arc(cp, False);
	else if (strcmp(command, "ia") == 0) arc(cp, True);
	else if (strcmp(command, "sp") == 0) flush_spline(cp);
	else if (strcmp(command, "sh") == 0) shade_last(cp);
	else if (strcmp(command, "wh") == 0) shade_last("0");
	else if (strcmp(command, "bk") == 0) shade_last("1");
	/* throw away the path -- jansteen */
	else if (strcmp(command, "ip") == 0) flush_path(True);
	else if (strcmp(command, "tx") == 0) Texture(cp);
#ifdef	PS
	else if (psSpecial(orig_cp) == 0) /**/;
#endif	/* PS */
	else if (!hush_spec_now)
	    Fprintf(stderr, "%s:  special \"%s\" not implemented\n", prog,
		orig_cp);
}
