/*
 * ZFlib -  FSlib compatible library for Zeit Vector font
 *
 *  Programmmed by Hirotsugu Kakugawa, Hiroshima University
 *  E-Mail:  kakugawa@se.hiroshima-u.ac.jp
 *
 *  Edition History
 *  14 Mar. 1993
 *  28 Mar. 1993
 *  20 Jun. 1993  Fixed ZENKAKU-space bug.
 */

/*
 * Copyright (C) 1993 Hirotsugu KAKUGAWA.   All rights reserved.
 *
 * ZFlib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
 * to anyone for the consequences of using it or for whether it serves any
 * particular purpose or works at all, unless he says so in writing.  Refer
 * to the GNU General Public License for full details.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * ZFlib, but only under the conditions described in the GNU
 * General Public License.  A copy of this license is supposed to have been
 * given to you along with ZFlib so you can know your rights and
 * responsibilities.  It should be in a file named COPYING.  Among other
 * things, the copyright notice and this notice must be preserved on all
 * copies. 
 */

#include  <stdio.h>
#include  <stdlib.h>
#include  <sys/types.h>
#include  "ZFlib.h"
#include  "ZFlib0.h"

static ZF_inited = 0;

static BUFFER  zf_buffer;
static int     zf_width;
static int     zf_height;
static int     zf_max_width;
static int     zf_max_height;
static int     zf_rast;
static int     zf_thin;
static int     zf_frame;



#define D_X_CL	500	/* horizontal center of data */
#define D_Y_CL	472	/* vertical center of data */
#define	D_Y_BL	872	/* base line of data */

#define B_X_CL	500	/* horizontal center in BuildChar */
#define B_Y_CL	400	/* vertical center in BuildChar */
#define B_Y_BL	0	/* base line in BuildChar */


#define Zerror_no_memory    (-100)
#define Zerror_cannot_read  (-101)
#define Zerror_cannot_write (-102)
#define Zerror_end_of_file  (-103)
#define Zerror_no_font      (-104)

static int   DrawFont();
static int   OpenFont();
static Font  *Zread_font();
static void  Zfree_font();
static int   Zread_header();
static Cycle *Zread_cycle();
static void  Zseek();
static long  Zread_long();
static int   Zread_byte();
static int   Zread_10_bits();
static int   jis2c();
static Free  *Zalloc();
static Cycle *Zalloc_cycle();
static Font  *Zalloc_font();
static void  Zfree();


#define  ZF_FONT_TBL   16
typedef struct
{
  char        *font_name;
  FILE        *fd1;
  FILE        *fd2;
  Zs_Header   read_header1;
  Zs_Header   read_header2;
} ZFFont;
ZFFont  ZFFontTbl[ZF_FONT_TBL];



static 
ZF_init()
{
  register int  i;

  for (i = 0; i < ZF_FONT_TBL; i++)
    ZFFontTbl[i].font_name = NULL;
  ZF_inited = 1;
}

static 
ZF_find_slot()
{
  register int  i;

  for (i = 0; i < ZF_FONT_TBL; i++)
    if (ZFFontTbl[i].font_name == NULL)
      return i;
  return -1;
}


/*
 * ZF_open_font()
 */
int
ZF_open_font(fontname)
  char *fontname;
{
  int  fd; 

  if (ZF_inited == 0)
    ZF_init();

  if ((fd = ZF_find_slot()) == -1)
    return -1;
  OpenFont(fd, fontname);

  return fd;
}


/*
 * ZF_close_font()
 */
int
ZF_close_font(fd)
  int  fd;
{
  char       *fontname;

  if ((fontname = ZFFontTbl[fd].font_name) == NULL)
    return -1;

  free(ZFFontTbl[fd].font_name);
  ZFFontTbl[fd].font_name = NULL;
  fclose(ZFFontTbl[fd].fd1);
  fclose(ZFFontTbl[fd].fd2);
  return 0;
}


/*
 * ZF_get_outline()
 */
int
ZF_get_outline(c, fd, buf)
  int  c;     /* char code in serial code */
  int  fd;
  int  **buf;
{
  fprintf(stdout, "Not implemented yet. ZF_get_outline()\n");
  return -1;
}

int
ZF_scan_conv(vk_buf, w, h, bw, bo, bm_buf, thick, frame, smooth)
  long  *vk_buf;
  int   w;
  int   h;
  int   bw;
  int   bo;
  char  *bm_buf;
  int   thick;
  int   frame;
  int   smooth;
{
  fprintf(stdout, "Not implemented yet. ZF_scan_conv()\n");
  return -1;
}

int
ZF_get_bitmap(c, fd, w, h, bw, bo, bm_buf, thick, frame, smooth)
  int  c;
  int  fd;
  int  w;
  int  h;
  int  bw;
  int  bo;
  char *bm_buf;
  int  thick;
  int  frame;
  int  smooth;    /* ignored */
{
  char           *buff;
  int            ccode, rast, x, y;
  unsigned char  d;
  static int     scode2jis();
  
  rast = (w+7)/8;
  if ((buff = malloc(h*rast)) == NULL){
    fprintf(stderr, "No mem in zf_get_bitmap\n");
    exit(-1);
  }
  bzero(buff, rast*h);

  zf_width      = w;
  zf_height     = h;
  zf_buffer     = buff;
  zf_max_width  = MAX_VALUE;
  zf_max_height = MAX_VALUE;
  zf_rast       = rast;
  switch (thick){
  case 1:
    zf_thin = 30; break;
  case 0:
  default:
    zf_thin =  0; break;
  }

  ccode = scode2jis(c);
  if (ccode == 0x2121)    /* ZENKAKU space */
    goto End;

  if (DrawFont(fd, ccode) < 0)
    return -1;

  for (y = 0; y < h; y++){
    for (x = 0; x < rast; x++)
      bm_buf[y*bw + x] = 0;
    for (x = 0; x < rast; x++){
      d = buff[y*rast + x];
      bm_buf[y*bw + x]   |= d >> bo;
      bm_buf[y*bw + x+1] |= d << (8-bo);
    }
  }
  free(buff);

End:
  return 0;
}

void
ZF_set_f_xform(rotate, refx, refy)
  int rotate;
  int refx;
  int refy;
{
  return;
}

void
ZF_fast_xform(vk, rotate, refx, refy)
  long *vk;
  int  rotate;
  int  refx;
  int  refy;
{
  return;
}

void
ZF_set_m_xform(t1, t2, t3, t4, t5 ,t6)
  double  t1, t2, t3, t4, t5, t6;
{
  return;
}


void
ZF_mat_xform(x, y, t1, t2, t3, t4, t5, t6)
   int    *x, *y;
  double  t1, t2, t3, t4, t5, t6;
{
  return;
}


/* ----------------------- */
/* Following routines are based on or snrafed 
 *   zfzeit.c (Ghostscript 2.4.1), and
 *   vfont.c  (dviout)
 */

static int
OpenFont(fd, fontname)
  int   fd;
  char  *fontname;
{
  int         err;
  FILE        *wfd1,*wfd2;
  Zs_Header   header1,header2;
  char	      *the_name, *fn;

  if ((the_name = malloc(strlen(fontname))) == NULL){
    fprintf(stderr, "ZFlib - no memory.");
    exit( -1 );
  }
  if ((fn = malloc(strlen(fontname)+10)) == NULL){
    fprintf(stderr, "ZFlib - no memory.");
    exit( -1 );
  }
  strcpy(the_name, fontname);
  
  sprintf(fn, "%s", fontname);
  strcat(fn, ".vf1");
  if((wfd1 = fopen(fn, "rb")) == NULL){
    perror(fn);
    free(fn); free(the_name);
    return -1;
  }
  Zread_header(wfd1, &header1);
  sprintf(fn, "%s", fontname);    
  strcat(fn, ".vf2");
  if((wfd2 = fopen(fn, "rb")) == NULL){
    perror(fn);
    free(fn); free(the_name);
    return -1;
  }
  Zread_header(wfd2, &header2);
  
  ZFFontTbl[fd].font_name    = the_name;
  ZFFontTbl[fd].fd1          = wfd1;
  ZFFontTbl[fd].fd2          = wfd2;
  ZFFontTbl[fd].read_header1 = header1;
  ZFFontTbl[fd].read_header2 = header2;
  free(fn);

  return fd;
}

static int 
DrawFont(fd, jiscode)
  int fd;
  int jiscode;
{
  char        *font_name;
  Font        *ptr, *fontptr;
  Cycle       *cptr, *cptr0;
  int         x, y, xo, yo, xbeg, ybeg;
  int         func, h, j, beg;
  static void fill_edges(), trace_outline();
  static void (*DrawFuncTbl[])() = {fill_edges, trace_outline};
#ifdef TEST
  extern int  moveto(), lineto();
#endif

  if((fontptr = Zread_font(fd, jiscode)) == NULL)
    return -1;

  switch (zf_frame){
  case 0:
    h = 0;  j = 1;  break;
  case 1:
  default:
    h = 0;  j = 2;  break;
  }
  h = 0;  j = 2;  

  for (func = h; func < j; func++){
    beg = 1;
    for(ptr = fontptr; ptr; ptr = ptr->next){
      xbeg = xo = (ptr->cycle->point.x);
      ybeg = yo = (ptr->cycle->point.y);
#ifdef TEST
      moveto(xbeg, ybeg);
#endif
      cptr0 = cptr = ptr->cycle;
      do {
	x = (cptr->next->point.x);
	y = (cptr->next->point.y);
#ifdef TEST
	lineto(x, y);
#endif
	DrawFuncTbl[func](xo, yo, x, y);
	xo = x;
	yo = y;
	cptr = cptr->next;
      } while (cptr != cptr0);

    }
  }
  
  Zfree_font(fontptr);

  return 0;
}


static void 
trace_outline(x1, y1, x2, y2)
  int x1, y1, x2, y2;
{
  int     dx, dy, dx2, dy2, e, i, tmp;
  BUFFER  address;
  BUFFER  buffer     = zf_buffer;
  int     width      = zf_width;
  int     height     = zf_height;
  int     max_width  = zf_max_width;
  int     max_height = zf_max_height;
  int     thin       = zf_thin;
  int     y_factor   = zf_rast;
  long    offset_x, minus_offset_x, plus_offset_x;
  long    offset_y, minus_offset_y, plus_offset_y;
  unsigned char mask;
  
  offset_x       = (long)thin*(max_width+1)/100L/2L;
  minus_offset_x = -(max_width+1)/2L - offset_x;
  plus_offset_x  = -(max_width+1)/2L + offset_x;
  offset_y       = (long)thin*(max_height+1)/100L/2L;
  minus_offset_y = -(max_height+1)/2L - offset_y;
  plus_offset_y  = -(max_height+1)/2L + offset_y;
  
  /**printf("T.O.    (%5d,%5d)-(%5d,%5d)\n", x1,y1,x2,y2);**/
  dx = x2 - x1;
  dy = y2 - y1;
  if (dy < 0) {
    x1 = (int)(((long)x1 * width + minus_offset_x) / (max_width + 1));
    x2 = (int)(((long)x2 * width + minus_offset_x) / (max_width + 1));
  } else if (dy == 0) {
    x1 = (int)(((long)x1 * width - (max_width + 1) / 2) / (max_width + 1));
    x2 = (int)(((long)x2 * width - (max_width + 1) / 2) / (max_width + 1));
  } else {
    x1 = (int)(((long)x1 * width + plus_offset_x) / (max_width + 1));
    x2 = (int)(((long)x2 * width + plus_offset_x) / (max_width + 1));
  }
  if (dx > 0) {
    y1 = (int)( ((long)y1 * height + minus_offset_y) / (max_height + 1) );
    y2 = (int)( ((long)y2 * height + minus_offset_y) / (max_height + 1) );
  } else if (dx == 0) {
    y1 = (int)( ((long)y1 * height - (max_height + 1) / 2) 
	       / (max_height + 1) );
    y2 = (int)( ((long)y2 * height - (max_height + 1) / 2) 
	       / (max_height + 1) );
  } else {
    y1 = (int)(((long)y1 * height + plus_offset_y) / (max_height + 1));
    y2 = (int)(((long)y2 * height + plus_offset_y) / (max_height + 1));
  }
  if (x1 < 0) x1 = 0; else if (x1 >= max_width ) x1 = max_width - 1;
  if (x2 < 0) x2 = 0; else if (x2 >= max_width ) x2 = max_width - 1;
  if (y1 < 0) y1 = 0; else if (y1 >= max_height) y1 = max_height - 1;
  if (y2 < 0) y2 = 0; else if (y2 >= max_height) y2 = max_height - 1;
  
  dy = y2 - y1;
  if (dy < 0) {
    tmp = x1; x1 = x2; x2 = tmp;
    tmp = y1; y1 = y2; y2 = tmp;
    dy = -dy;
  }
  dx = x2 - x1;
  if (dx < 0) dx = -dx;
  if (dx == 0 && dy == 0) return;
  address = &buffer[y1 * y_factor + x1 / 8];
  mask = 0x80 >> (x1 & 7);
  dx2 = 2 * dx;
  dy2 = 2 * dy;
  
  if (dx < dy) {
    e = dx2 - dy;
    if (x1 < x2) {
      for (i = 0; i <= dy; i++) {
	*address |= mask;
	while (e >= 0) {
	  if ((mask >>= 1) == 0) {
	    address++;
	    mask = 0x80;
	  }
	  e -= dy2;
	}
	address += y_factor;
	e += dx2;
      }
    } else {
      for (i = 0; i <= dy; i++) {
	*address |= mask;
	while (e >= 0) {
	  if ((mask <<= 1) == 0) {
	    address--;
	    mask = 0x1;
	  }
	  e -= dy2;
	}
	address += y_factor;
	e += dx2;
      }
    }
  } else {
    e = dy2 - dx;
    if (x1 < x2) {
      for (i = 0; i <= dx; i++) {
	*address |= mask;
	while (e >= 0) {
	  address += y_factor;
	  e -= dx2;
	}
	if ((mask >>= 1) == 0) {
	  address++;
	  mask = 0x80;
	}
	e += dy2;
      }
    } else {
      for (i = 0; i <= dx; i++) {
	*address |= mask;
	while (e >= 0) {
	  address += y_factor;
	  e -= dx2;
	}
	if ((mask <<= 1) == 0) {
	  address--;
	  mask = 0x1;
	}
	e += dy2;
      }
    }
  }
}

static void 
fill_edges(x1, y1, x2, y2)
  int x1, y1, x2, y2;
{
  static unsigned char mask_pattern[8] =
      { 0xff, 0x7f, 0x3f, 0x1f, 0xf, 0x7, 0x3, 0x1 };
  int      dx, dy, dx2, dy2, sx, e, i, j, tmp;
  int      width      = zf_width;
  int      height     = zf_height;
  int      max_width  = zf_max_width;
  int      max_height = zf_max_height;
  int      y_factor   = zf_rast;
  BUFFER   buffer     = zf_buffer;
  BUFFER   address;
  int      bit, right_bytes;
  
  /**printf("F.E.    (%5d,%5d)-(%5d,%5d)\n", x1,y1,x2,y2);**/
  x1 = (int)( ((long)x1*width -(max_width+1)/2)  / (max_width+1) );
  y1 = (int)( ((long)y1*height-(max_height+1)/2) / (max_height+1) );
  x2 = (int)( ((long)x2*width -(max_width+1)/2)  / (max_width+1) );
  y2 = (int)( ((long)y2*height-(max_height+1)/2) / (max_height+1) );
  
  dy = y2 - y1;
  if (dy == 0) return;
  if (dy < 0) {
    tmp = x1; x1 = x2; x2 = tmp;
    tmp = y1; y1 = y2; y2 = tmp;
    dy = -dy;
  }
  dx = x2 - x1;
  sx = dx > 0 ? 1 : -1;
  if (dx < 0) dx = -dx;
  address = &buffer[y1 * y_factor + x1 / 8];
  right_bytes = (width - 1) / 8 - x1 / 8;
  bit = x1 & 7;
  dx2 = 2 * dx;
  dy2 = 2 * dy;
  e = dx2 - dy;
  
  for (i = 0; i < dy; i++) {
    /* Change right bits in this byte. */
    *address ^= mask_pattern[bit];
    /* Change all the bits in right bytes. */
    for (j = 1; j <= right_bytes; j++) {
      address[j] = ~address[j];
    }
    while (e >= 0) {
      bit += sx;
      if (bit & 0x8) {
	address += sx;
	right_bytes -= sx;
	bit &= 0x7;
      }
      e -= dy2;
    }
    address += y_factor;
    e += dx2;
  }
}


/*
 * scode2jis - serial code to jis code 
 */
static int 
scode2jis(scode)
  int scode;
{
  int  jis;

  jis = ((scode/94)+0x21)*0x100 + ((scode%94)+0x21);
  return jis;
}



/***************************************************
 * Zeit Font Accsess Routines
 *     Original by Ext(T.Kawamoto)
 *     modifyed by K-ras
 **************************************************/
/*
 *	zeit font file library Ver 1.00
 *		for Human68k
 *	Copyright 1991 by Ext(T.Kawamoto)
 */

static int 
Zread_header(fp, hptr)
  FILE      *fp;
  Zs_Header *hptr;
{
  int i;
  
  Zseek(fp, 0);
  hptr->dummy[0] = Zread_byte(fp);
  hptr->dummy[1] = Zread_byte(fp);
  for (i = 0; i < CODE_SIZE; ++i)
    hptr->offsets[i] = Zread_long(fp);
  
  return 0;
}

static Font* 
Zread_font(fd, code)
  int  fd;
  int  code;
{
  Font  font, *fptr;
  Cycle *cptr;
  int   code_offset;
  long  seek_offset, no_font_flag;
  FILE  *fp;
  
  code_offset = jis2c(code);
  if (code < 0x5000){
    fp = ZFFontTbl[fd].fd1;
    seek_offset  = no_font_flag = ZFFontTbl[fd].read_header1.offsets[code_offset];
    seek_offset += Zs_Header_SIZE;
  } else {
    fp = ZFFontTbl[fd].fd2;
    seek_offset  = no_font_flag = ZFFontTbl[fd].read_header2.offsets[code_offset];
    seek_offset += Zs_Header_SIZE;
  }
  if (code_offset == 0 || no_font_flag == 0xffffffff)
    return NULL;
  
  Zseek(fp, seek_offset);
  fptr = &font;
  while (1){
    cptr = Zread_cycle(fp);
    if (cptr == NULL)
      break;
    if ((fptr->next = Zalloc_font()) == NULL)
      return fptr->next;
    fptr        = fptr->next;
    fptr->cycle = cptr;
  }
  fptr->next = NULL;
  return font.next;
}

static Cycle * 
Zread_cycle(fp)
  FILE *fp;
{
  int   x0, y0, x, y;
  Cycle *cptr0, *cptr;
  
  x0 = Zread_10_bits(fp);
  y0 = Zread_10_bits(fp);
  if (x0 == 1023 && y0 == 1023)
    return NULL;
  if ((cptr0 = cptr = Zalloc_cycle()) == NULL)
    return cptr;
  cptr->point.x = x0;
  cptr->point.y = y0;
  while (1) {
    x = Zread_10_bits(fp);
    y = Zread_10_bits(fp);
    if (x == 1023 && y == 1023)
      break;
    if ((cptr->next = Zalloc_cycle()) == NULL)
      return cptr->next;
    cptr          = cptr->next;
    cptr->point.x = x;
    cptr->point.y = y;
  }
  if (cptr->point.x == cptr0->point.x && cptr->point.y == cptr0->point.y){
    cptr->next = cptr0->next;
    Zfree((Free *)cptr0);
    return cptr;
  } else {
    cptr->next = cptr0;
    return cptr0;
  }
}

static long
Zread_long(fp)
  FILE *fp;
{
  unsigned long dummy;
  
  dummy = Zread_byte(fp);
  dummy |= (unsigned long)Zread_byte(fp) << 8;
  dummy |= (unsigned long)Zread_byte(fp) << 16;
  dummy |= (unsigned long)Zread_byte(fp) << 24;
  
  return dummy;
}

static short left;
static unsigned long divide;

static void 
Zseek(fp, offset)
  FILE *fp;
  long offset;
{
  fseek(fp, offset, 0);
  left   = 0;
  divide = 0;
}

static int 
Zread_10_bits(fp)
  FILE *fp;
{
  if (left < 10) {
    divide <<= 16;
    divide |= (unsigned long)Zread_byte(fp);
    divide |= (unsigned long)Zread_byte(fp) << 8;
    left   += 16;
  }
  
  left -= 10;
  return (divide >> left) & 0x3ff;
}

static int 
Zread_byte(fp)
  FILE *fp; 
{
  return fgetc(fp);
}


static int
jis2c(code)
  int  code;
{
  int  zscode;
  
  if (code < 0x5000){
    zscode = (((code >> 8) & 0xff) - 0x21) * 0x5e + (code & 0xff) - 0x21;
  } else {
    zscode = (((code >> 8) & 0xff) - 0x50) * 0x5e + (code & 0xff) - 0x21;
  }
  
  return zscode;
}

/****************************************/
/*					*/
/*	alloc, free			*/
/*					*/
/****************************************/
static Free *free_ptr = NULL;

static Free *
Zalloc()
{
  Free *ptr;
  
  if (free_ptr == NULL){
    int i;
    
    if ((ptr = (Free *)malloc(sizeof(Free)*1024)) == NULL){
      printf("\nZalloc malloc error\n");
      exit(0);
    }
    free_ptr = ptr;
    for (i = 0; i < 1023; ++i, ++ptr)
      ptr->next = ptr + 1;
    ptr->next = NULL;
  }
  ptr      = free_ptr;
  free_ptr = free_ptr->next;
  
  return ptr;
}

static Cycle * 
Zalloc_cycle()
{
  Free *ptr;
  
  ptr = Zalloc();
  if (ptr == NULL)
    return (Cycle *)ptr;
  return &(ptr->cycle);
}

static Font * 
Zalloc_font()
{
  Free *ptr;
  
  ptr = Zalloc();
  if (ptr == NULL)
    return (Font *)ptr;
  return &(ptr->font);
}

static void 
Zfree(ptr)
  Free *ptr;
{
  if (ptr == NULL)
    return;
  ptr->next = free_ptr;
  free_ptr  = ptr;
}

static void 
Zfree_cycle(ptr)
  Cycle *ptr;
{
  Cycle *ptr0, *ptr2;
  
  if (ptr == NULL)
    return;
  ptr0 = ptr;
  do {
    ptr2 = ptr->next;
    Zfree((Free *)ptr);
    ptr = ptr2;
  } while (ptr != ptr0);
}

static void 
Zfree_font(ptr)
  Font *ptr;
{
  Font *ptr2;
  
  while (ptr != NULL){
    Zfree_cycle(ptr->cycle);
    ptr2 = ptr->next;
    Zfree((Free *)ptr);
    ptr = ptr2;
  }
}

/* EOF */
