/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/drawing.c,v 2.112 1992/06/19 18:09:08 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "align.e"
#include "animate.e"
#include "arc.e"
#include "attr.e"
#include "auxtext.e"
#include "box.e"
#include "choice.e"
#include "cmd.e"
#include "copypaste.e"
#include "cursor.e"
#include "dialog.e"
#include "dup.e"
#include "edit.e"
#include "eps.e"
#include "file.e"
#include "font.e"
#include "grid.e"
#include "group.e"
#include "mark.e"
#include "mainloop.e"
#include "menu.e"
#include "msg.e"
#include "obj.e"
#include "oval.e"
#include "pattern.e"
#include "poly.e"
#include "polygon.e"
#include "raster.e"
#include "rcbox.e"
#include "rect.e"
#include "ruler.e"
#include "scroll.e"
#include "select.e"
#include "setup.e"
#include "shortcut.e"
#include "special.e"
#include "stk.e"
#include "stretch.e"
#include "text.e"
#include "xbitmap.e"
#include "xpixmap.e"
#ifdef XI18N
#include "input.e"

#define IMK	'\134'	/* Ctrl-IMK(Ctrl-\) to connect input method. */
#endif

#define O_VIS 4
#define O_INVIS 4
#define O_GRID (O_VIS+O_INVIS)

int	numRedrawBBox=INVALID;

static int		checkBBox=TRUE;
static struct BBRec	smallArea[2];

void SetDefaultDrawWinClipRecs ()
{
   XRectangle	recs[1];

   SetRecVals (recs[0], 0, 0, ZOOMED_SIZE(drawWinW), ZOOMED_SIZE(drawWinH));
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, 1, YXBanded);
}

void SetDefaultIconWinClipRecs ()
{
   XRectangle	recs[1];

   SetRecVals (recs[0], 0, 0, iconWindowW, iconWindowH);
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, 1, YXBanded);
}

static
void DrawHorizOutline (Win, Y, X1, X2, XStart, XEnd)
   Window	Win;
   int		Y, X1, X2, XStart, XEnd;
   /* XStart and XEnd are the real place, X1 and X2 are on outline grid */
{
   register int	i;

   if (XStart-X1 < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, XStart, Y, X1+O_VIS-1, Y);
   for (i = X1+O_GRID; i < X2-O_GRID; i+= O_GRID)
      XDrawLine (mainDisplay, Win, defaultGC, i, Y, i+O_VIS-1, Y);
   if (X2-XEnd < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, X2-O_GRID, Y, XEnd, Y);
   else
      XDrawLine (mainDisplay, Win, defaultGC, X2-O_GRID, Y, X2-O_INVIS-1, Y);
}

static
void DrawVertOutline (Win, X, Y1, Y2, YStart, YEnd)
   Window	Win;
   int		X, Y1, Y2, YStart, YEnd;
   /* YStart and YEnd are the real place, Y1 and Y2 are on outline grid */
{
   register int	i;

   if (YStart-Y1 < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, X, YStart, X, Y1+O_VIS-1);
   for (i = Y1+O_GRID; i < Y2-O_GRID; i+= O_GRID)
      XDrawLine (mainDisplay, Win, defaultGC, X, i, X, i+O_VIS-1);
   if (Y2-YEnd < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, X, Y2-O_GRID, X, YEnd);
   else
      XDrawLine (mainDisplay, Win, defaultGC, X, Y2-O_GRID, X, Y2-O_INVIS-1);
}

static
void DrawSymOutline (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   int	ltx, lty, rbx, rby, x_start, x_end, y_start, y_end;

   ltx = ZOOMED_SIZE(ObjPtr->obbox.ltx - XOff - QUARTER_INCH) + 1;
   lty = ZOOMED_SIZE(ObjPtr->obbox.lty - YOff - QUARTER_INCH) + 1;
   rbx = ZOOMED_SIZE(ObjPtr->obbox.rbx - XOff + QUARTER_INCH) - 1;
   rby = ZOOMED_SIZE(ObjPtr->obbox.rby - YOff + QUARTER_INCH) - 1;

   x_start = (ltx % O_GRID == 0) ? ltx : (int)(ltx / O_GRID) * O_GRID;
   x_end = (rbx % O_GRID == 0) ? rbx : ((int)(rbx / O_GRID) + 1) * O_GRID;
   DrawHorizOutline (Win, lty, x_start, x_end, ltx, rbx);
   DrawHorizOutline (Win, rby, x_start, x_end, ltx, rbx);
   y_start = (lty % O_GRID == 0) ? lty : (int)(lty / O_GRID) * O_GRID;
   y_end = (rby % O_GRID == 0) ? rby : ((int)(rby / O_GRID) + 1) * O_GRID;
   DrawVertOutline (Win, ltx, y_start, y_end, lty, rby);
   DrawVertOutline (Win, rbx, y_start, y_end, lty, rby);
}

static
int NeedToDraw (ObjBBox)
   struct BBRec	ObjBBox;
{
   switch (numRedrawBBox)
   {
      case 0: return (BBoxIntersect (ObjBBox, drawWinBBox));
      case 1: return (BBoxIntersect (ObjBBox, drawWinBBox) &&
            BBoxIntersect (ObjBBox, smallArea[0]));
      case 2: return (BBoxIntersect (ObjBBox, drawWinBBox) &&
            (BBoxIntersect (ObjBBox, smallArea[0]) ||
            BBoxIntersect (ObjBBox, smallArea[1])));
      default: fprintf (stderr, "Warning:  Invalid numRedrawBBox.\n"); break;
   }
}

void DrawObj (Win, ObjPtr)
   Window			Win;
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         DrawPolyObj (Win, drawOrigX, drawOrigY, ObjPtr); 
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_BOX:
         DrawBoxObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_OVAL:
         DrawOvalObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_TEXT: DrawTextObj (Win, drawOrigX, drawOrigY, ObjPtr); break;
      case OBJ_POLYGON:
         DrawPolygonObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_ARC:
         DrawArcObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_RCBOX:
         DrawRCBoxObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_XBM:
         DrawXBmObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_XPM:
         DrawXPmObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;

      case OBJ_SYM:
      case OBJ_ICON:
      case OBJ_GROUP:
         obj_ptr = ObjPtr->detail.r->last;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
            if (!checkBBox || NeedToDraw (obj_ptr->bbox))
               DrawObj (Win, obj_ptr);
         if (ObjPtr->type == OBJ_ICON && ObjPtr->dirty)
         {
            attr_ptr = ObjPtr->fattr;
            for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
               UpdTextBBox (attr_ptr->obj);
            AdjObjBBox (ObjPtr);
            UpdSelBBox ();
            ObjPtr->dirty = FALSE;
         }
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         if (ObjPtr->type == OBJ_SYM) DrawSymOutline (Win, drawOrigX, drawOrigY,
               ObjPtr);
         break;
   }
}

void DrawPaperBoundary ()
{
   register int	x_end, y_end;

   if (drawOrigX+drawWinW > paperWidth)
   {
      x_end = OFFSET_X(paperWidth);
      if (drawOrigY+drawWinH > paperHeight)
      {
         y_end = OFFSET_Y(paperHeight);
         XDrawLine (mainDisplay, drawWindow, defaultGC, x_end, 0, x_end, y_end);
         XDrawLine (mainDisplay, drawWindow, defaultGC, 0, y_end, x_end, y_end);
      }
      else
         XDrawLine (mainDisplay, drawWindow, defaultGC, x_end, 0, x_end,
               ZOOMED_SIZE(drawWinH));
   }
   else if (drawOrigY+drawWinH > paperHeight)
   {
      y_end = OFFSET_Y(paperHeight);
      XDrawLine (mainDisplay, drawWindow, defaultGC, 0, y_end,
            ZOOMED_SIZE(drawWinW), y_end);
   }
}

void RedrawAnArea (BotObj, LtX, LtY, RbX, RbY)
   struct ObjRec	* BotObj;
   int			LtX, LtY, RbX, RbY;
   /* LtX, LtY, RbX, RbY are absolute coordinates */
{
   register struct ObjRec	* obj_ptr;
   XRectangle			recs[1];

   SetRecVals (recs[0], OFFSET_X(LtX), OFFSET_Y(LtY), ZOOMED_SIZE(RbX-LtX)+1,
         ZOOMED_SIZE(RbY-LtY)+1);
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, 1, YXBanded);

   XClearArea (mainDisplay, drawWindow, OFFSET_X(LtX), OFFSET_Y(LtY),
         ZOOMED_SIZE(RbX-LtX)+1, ZOOMED_SIZE(RbY-LtY)+1, FALSE);

   if (paperWidth >= LtX && paperWidth <= RbX ||
         paperHeight >= LtY && paperHeight <= RbY)
      DrawPaperBoundary ();

   DrawGridLines (drawWindow, OFFSET_X(LtX), OFFSET_Y(LtY),
         ZOOMED_SIZE(RbX-LtX)+1, ZOOMED_SIZE(RbY-LtY)+1);

   numRedrawBBox = 1;
   smallArea[0].ltx = LtX; smallArea[0].lty = LtY;
   smallArea[0].rbx = RbX; smallArea[0].rby = RbY;
   for (obj_ptr = BotObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      if (BBoxIntersect (obj_ptr->bbox, drawWinBBox) &&
            BBoxIntersect (obj_ptr->bbox, smallArea[0]))
         DrawObj (drawWindow, obj_ptr);

   SetDefaultDrawWinClipRecs ();
}

static
int Inside (BBox1, BBox2)
   struct BBRec	BBox1, BBox2;
{
   return (BBox1.ltx >= BBox2.ltx && BBox1.lty >= BBox2.lty &&
         BBox1.rbx <= BBox2.rbx && BBox1.rby <= BBox2.rby);
}

void RedrawAreas (BotObj, LtX1, LtY1, RbX1, RbY1, LtX2, LtY2, RbX2, RbY2)
   struct ObjRec	* BotObj;
   int			LtX1, LtY1, RbX1, RbY1, LtX2, LtY2, RbX2, RbY2;
   /* note:  these coordinates are absolute */
{
   register struct ObjRec	* obj_ptr;
   int				ltx, lty, rbx, rby, rec1_slot, num;
   struct BBRec			bbox1, bbox2;
   XRectangle			recs[4];

   bbox1.ltx = LtX1; bbox1.lty = LtY1;
   bbox1.rbx = RbX1; bbox1.rby = RbY1;
   bbox2.ltx = LtX2; bbox2.lty = LtY2;
   bbox2.rbx = RbX2; bbox2.rby = RbY2;

   if (Inside (bbox1, bbox2))
   {
      RedrawAnArea (BotObj, LtX2, LtY2, RbX2, RbY2);
      return;
   }
   else if (Inside (bbox2, bbox1))
   {
      RedrawAnArea (BotObj, LtX1, LtY1, RbX1, RbY1);
      return;
   }

   XClearArea (mainDisplay, drawWindow, OFFSET_X(LtX1), OFFSET_Y(LtY1),
         ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1, FALSE);

   if (BBoxIntersect (bbox1, bbox2))
   {
      ltx = min(LtX1,LtX2); lty = min(LtY1,LtY2);
      rbx = max(RbX1,RbX2); rby = max(RbY1,RbY2);
      RedrawAnArea (BotObj, ltx, lty, rbx, rby);
      return;
   }

   if (LtY1 == LtY2)
   {
      rec1_slot = (LtX1 <= LtX2) ? 0 : 1;
      SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
            ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1);
      SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
            ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY2)+1);
      num = 2;
   }
   else
   {
      if (LtY1 < LtY2)
      {
         if (RbY1 <= LtY2)
         {  /* y-extents do not intersect */
            SetRecVals (recs[0], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1);
            SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY2)+1);
            num = 2;
         }
         else if (RbY1 >= RbY2)
         {  /* box 2's y-extents is inside box 1's y-extents */
            rec1_slot = (LtX1 < LtX2) ? 0 : 1;  
            SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1);
            SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY2)+1);
            num = 2;
         }
         else
         {  
            SetRecVals (recs[0], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(LtY2-LtY1)+1);
            if (LtX1 < LtX2)
            {
               SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY2),
                     ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY2)+1);
               SetRecVals (recs[2], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                     ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY1-LtY2)+1);
            }
            else
            {
               SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                     ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY1-LtY2)+1);
               SetRecVals (recs[2], OFFSET_X(LtX1), OFFSET_Y(LtY2),
                     ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY2)+1);
            }
            SetRecVals (recs[3], OFFSET_X(LtX2), OFFSET_Y(RbY1),
                  ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-RbY1)+1);
            num = 4;
         }
      }
      else
      {
         if (RbY2 <= LtY1)
         {  /* y-extents do not intersect */
            SetRecVals (recs[0], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY2)+1);
            SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1);
            num = 2;
         }
         else if (RbY2 >= RbY1)
         {  /* box 1's y-extents is inside box 2's y-extents */
            rec1_slot = (LtX1 < LtX2) ? 0 : 1;  
            SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1);
            SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY2)+1);
            num = 2;
         }
         else
         {  
            SetRecVals (recs[0], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(LtY1-LtY2)+1);
            if (LtX1 < LtX2)
            {
               SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                     ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY2-LtY1)+1);
               SetRecVals (recs[2], OFFSET_X(LtX2), OFFSET_Y(LtY1),
                     ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY1)+1);
            }
            else
            {
               SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY1),
                     ZOOMED_SIZE(RbX2-LtX2)+1, ZOOMED_SIZE(RbY2-LtY1)+1);
               SetRecVals (recs[2], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                     ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY2-LtY1)+1);
            }
            SetRecVals (recs[3], OFFSET_X(LtX1), OFFSET_Y(RbY2),
                  ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-RbY2)+1);
            num = 4;
         }
      }
   }
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, num, YXSorted);

   if (paperWidth >= LtX1 && paperWidth <= RbX1 ||
         paperHeight >= LtY1 && paperHeight <= RbY1 ||
         paperWidth >= LtX2 && paperWidth <= RbX2 ||
         paperHeight >= LtY2 && paperHeight <= RbY2)
      DrawPaperBoundary ();

   DrawGridLines (drawWindow, OFFSET_X(LtX1), OFFSET_Y(LtY1),
         ZOOMED_SIZE(RbX1-LtX1)+1, ZOOMED_SIZE(RbY1-LtY1)+1);

   numRedrawBBox = 2;
   smallArea[0].ltx = LtX1; smallArea[0].lty = LtY1;
   smallArea[0].rbx = RbX1; smallArea[0].rby = RbY1;
   smallArea[1].ltx = LtX2; smallArea[1].lty = LtY2;
   smallArea[1].rbx = RbX2; smallArea[1].rby = RbY2;
   for (obj_ptr = BotObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      if (BBoxIntersect (obj_ptr->bbox, drawWinBBox) &&
            (BBoxIntersect (obj_ptr->bbox, bbox1) ||
            BBoxIntersect (obj_ptr->bbox, bbox2)))
         DrawObj (drawWindow, obj_ptr);
   }

   SetDefaultDrawWinClipRecs ();
}

Pixmap DrawAllOnPixmap (LtX, LtY, W, H)
   int	*LtX, *LtY, *W, *H;
{
   register struct ObjRec	* obj_ptr;
   int				ltx, lty, rbx, rby, w, h;
   int				saved_draw_orig_x, saved_draw_orig_y;
   int				saved_draw_win_w, saved_draw_win_h;
   int                          saved_zoom_scale, saved_zoomed_in;
   char				msg[MAXSTRING];
   Pixmap			pixmap;
   XGCValues			values;

   ltx = botObj->bbox.ltx; lty = botObj->bbox.lty;
   if (botObj->type == OBJ_XBM || botObj->type == OBJ_XPM)
   {
      rbx = botObj->bbox.rbx-1;
      rby = botObj->bbox.rby-1;
   }
   else
   {
      rbx = botObj->bbox.rbx;
      rby = botObj->bbox.rby;
   }
   for (obj_ptr = botObj->prev; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      if (obj_ptr->bbox.ltx < ltx) ltx = obj_ptr->bbox.ltx;
      if (obj_ptr->bbox.lty < lty) lty = obj_ptr->bbox.lty;
      if (obj_ptr->type == OBJ_XBM || obj_ptr->type == OBJ_XPM)
      {
         if (obj_ptr->bbox.rbx-1 > rbx) rbx = obj_ptr->bbox.rbx-1;
         if (obj_ptr->bbox.rby-1 > rby) rby = obj_ptr->bbox.rby-1;
      }
      else
      {
         if (obj_ptr->bbox.rbx > rbx) rbx = obj_ptr->bbox.rbx;
         if (obj_ptr->bbox.rby > rby) rby = obj_ptr->bbox.rby;
      }
   }
   *W = w = rbx - ltx + 1;
   *H = h = rby - lty + 1;
   *LtX = ltx;
   *LtY = lty;

   saved_draw_orig_x = drawOrigX; saved_draw_orig_y = drawOrigY;
   saved_draw_win_w = drawWinW; saved_draw_win_h = drawWinH;
   saved_zoom_scale = zoomScale;
   saved_zoomed_in = zoomedIn;

   drawOrigX = ltx; drawOrigY = lty;
   drawWinW = w; drawWinH = h;
   zoomScale = 0;
   zoomedIn = FALSE;

   SetDefaultDrawWinClipRecs ();

   pixmap = XCreatePixmap (mainDisplay, mainWindow, w, h, mainDepth);

   if (pixmap == None)
   {
      sprintf (msg, "Can not allocate pixmap of size %1dx%1d.", w, h);
      Msg (msg);
      return (None);
   }

   values.foreground = myBgPixel;
   values.function = GXcopy;
   values.fill_style = FillSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle, &values);
   XFillRectangle (mainDisplay, pixmap, drawGC, 0, 0, w, h);

   AdjSplineVs ();

   checkBBox = FALSE;
   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      DrawObj (pixmap, obj_ptr);
   checkBBox = TRUE;

   drawOrigX = saved_draw_orig_x; drawOrigY = saved_draw_orig_y;
   drawWinW = saved_draw_win_w; drawWinH = saved_draw_win_h;
   zoomedIn = saved_zoomed_in;
   zoomScale = saved_zoom_scale;

   AdjSplineVs ();
   SetDefaultDrawWinClipRecs ();

   return (pixmap);
}

void RedrawDrawWindow (BotObj)
   struct ObjRec	* BotObj;
{
   register struct ObjRec	* obj_ptr;

   DrawPaperBoundary ();
   RedrawGridLines ();

   numRedrawBBox = 0;
   for (obj_ptr = BotObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      if (BBoxIntersect (obj_ptr->bbox, drawWinBBox))
         DrawObj (drawWindow, obj_ptr);
}

void ClearAndRedrawDrawWindow ()
{
   XClearWindow (mainDisplay, drawWindow);
   somethingHighLighted = FALSE;
   RedrawDrawWindow (botObj);
   RedrawCurText ();
   HighLightForward ();
}

void ClearAndRedrawDrawWindowDontDrawCurText ()
{
   XClearWindow (mainDisplay, drawWindow);
   somethingHighLighted = FALSE;
   RedrawDrawWindow (botObj);
   HighLightForward ();
}

void CleanUpDrawingWindow ()
{
   SetCurChoice (NOTHING);
   if (topSel != NULL) { HighLightReverse (); RemoveAllSel (); }
   DelAllObj ();
}

static XComposeStatus	c_stat;

int ShortHand (input)
   XEvent	* input;
   /* returns BAD if the character is a <CONTROL> or a <META> character */
   /* returns INVALID if the character is a normal character */
   /* otherwise, returns the value of sub-functions, such as QuitProc () */
{
   register int		i;
   char			buf[80], * name;
   int			valid_shortcut=FALSE;
   KeySym		key_sym;
   XKeyEvent		* key_ev;

   key_ev = &(input->xkey);
#ifdef XI18N
   if (curFont == FONT_MUL) {
     if (!textIMOpen) {
	/*
	 * try to open input method.
	 */
	if (!(textIMOpen = OpenTextIM(mainDisplay, drawWindow))) {
	   textIMOn = 0;
	   Msg("Fail to connect with xwnmo.");
	}
     }
     /*
      * If failed to connect with xwnmo, skip to normal XLookupString.
      */
     if (textIMOpen && XFilterEvent(input, input->xany.window)) {
	/* 
	 * The keys are filtered by input method, and translated.
	 * So just return untill user finished.
	 * If user closed input method at this time, the textIMOpen
	 * and textIMOn will be set to FALSE by XFilterEvent's callback
	 * fuction.
	 * see _XipSetIOErrorHandler(TextIMIOError) in function
	 * OpenTextIM().
	 */
	if (textIMOpen) {
	   textIMOn = 1;		/* flag now is input method mode .*/
	   return(BAD);			/* here BAD means to continue.   */
	}
     }
   }
#endif
   XLookupString (key_ev, buf, 80-1, &key_sym, &c_stat);
   TranslateKeys (buf, &key_sym);

   if (key_sym>='\040' && key_sym<='\177' &&
         (key_ev->state & (ControlMask | Mod1Mask)))
      valid_shortcut = TRUE;
   else if ((key_sym>'\040' && key_sym<='\177' ||
         key_sym>0xa0 && key_sym<=0xff) &&
         !(key_ev->state & (ControlMask | Mod1Mask)) &&
         curChoice != DRAWTEXT)
   {
      char		code;
      unsigned int	state;

      for (i = 0; i < numExtraWins; i++)
         if (key_ev->window == extraWinInfo[i].window &&
               extraWinInfo[i].window != None)
            break;

      if (i == numExtraWins)
      {
         valid_shortcut = FetchShortCut ((char)(((char)key_sym)&0x7f),
               &code, &state, &name);
         if (valid_shortcut)
         {
            key_sym = code;
            key_ev->state = state;
         }
      }
   }

   if (valid_shortcut)
   {
      Msg ("");
      if ((key_ev->state & ControlMask) && (!(key_ev->state & Mod1Mask)))
      {
         switch (key_sym&0xff)
         {
            case 'a': /*^a*/ SelAllObj (TRUE); break;
            case 'b': /*^b*/ BackProc (); break;
            case 'c': /*^c*/ ChangeDomain (); break;
            case 'd': /*^d*/ DupSelObj (); break;
            case 'e': /*^e*/ PushCurChoice (); break;
            case 'f': /*^f*/ FrontProc (); break;
            case 'g': /*^g*/ GroupSelObj (); break;
            case 'h': /*^h*/ return (INVALID);
            case 'i': /*^i*/ Instantiate (); break;
            case 'j': /*^j*/ return (INVALID);
            case 'k': /*^k*/ PopIcon (); break;
            case 'l': /*^l*/ AlignSelObjs (); break;
            case 'm': /*^m*/ return (INVALID);
            case 'n': /*^n*/ NewProc (); break;
            case 'o': /*^o*/ OpenProc (); break;
            case 'p': /*^p*/ Dump (""); break;
            case 'q': /*^q*/ return (QuitProc ());
            case 'r': /*^r*/ ClearAndRedrawDrawWindow (); break;
            case 's': /*^s*/ SaveFile (); break;
            case 't': /*^t*/ AlignSelToGrid (); break;
            case 'u': /*^u*/ UngroupSelObj (); break;
            case 'v': /*^v*/ PushIcon (); break;
            case 'w': /*^w*/ SetCurChoice (DRAWTEXT); break;
            case 'x': /*^x*/ DelAllSelObj (); break;
            case 'y': /*^y*/ CopyToCutBuffer (); break;
            case 'z': /*^z*/ return (AnimateProc ());
            case ',': /*^,*/ ScrollLeft (NULL); break;
            case '.': /*^.*/ ScrollRight (NULL); break;
            case '-': /*^-*/ PrintWithCommand (""); break;
         }
      }
      else if ((key_ev->state & Mod1Mask) && (!(key_ev->state & ControlMask)))
      {
         switch (key_sym&0xff)
         {
            case 'a': /*#a*/ AddAttrs (); break;
            case 'b': /*#b*/ return (ProbeProc ());
            case 'c': /*#c*/ RotateCounter (); break;
            case 'd': /*#d*/ DecGrid (); break;
            case 'e': /*#e*/ AnimateSel (); break;
            case 'f': /*#f*/ FlashSelColor (); break;
            case 'g': /*#f*/ ToggleGridShown (); break;
            case 'h': /*#h*/ FlipHorizontal (); break;
            case 'i': /*#i*/ IncGrid (); break;
            case 'j': /*#j*/ HideAllAttrNames (); break;
            case 'k': /*#k*/ SetCurChoice (NOTHING); break;
            case 'l': /*#l*/ DistrSelObjs (); break;
            case 'm': /*#m*/ MoveAttr (); break;
            case 'n': /*#n*/ ShowAllAttrNames (); break;
            case 'o': /*#o*/ ZoomOut (); break;
            case 'p': /*#p*/ ImportFile (); break;
            case 'q': /*#q*/ SetCurChoice (DRAWPOLY); break;
            case 'r': /*#r*/ SetCurChoice (DRAWBOX); break;
            case 's': /*#s*/ return (SolveProc ());
            case 't': /*#t*/ DetachAttrs (); break;
            case 'u': /*#u*/ UndoCmd (); break;
            case 'v': /*#v*/ FlipVertical (); break;
            case 'w': /*#w*/ RotateClockWise (); break;
            case 'x': /*#x*/ return (EscapeProc ());
            case 'y': /*#y*/ return (SimulateProc ());
            case 'z': /*#z*/ ZoomIn (); break;
            case '9': /*#9*/ MakePreciseArc (); break;
            case '0': /*#0*/ UpdateSelObjs (); break;
            case ',': /*#,*/ ScrollUp (NULL); break;
            case '.': /*#.*/ ScrollDown (NULL); break;
            case '-': /*#-*/ ShowAllAttrs (); break;
            case '{': /*#{*/ AlignObjsTop (); break;
            case '+': /*#+*/ AlignObjsMiddle (); break;
            case '}': /*#}*/ AlignObjsBottom (); break;
            case '[': /*#[*/ AlignObjsLeft (); break;
            case '=': /*#=*/ AlignObjsCenter (); break;
            case ']': /*#]*/ AlignObjsRight (); break;
            case '"': /*#"*/ MakeRegularPolygon (); break;
            case '%': /*#%*/ SetPrintReduction (); break;
            case ':': /*#:*/ DefaultZoom (); break;
            case '`': /*#`*/ ZoomWayOut (); break;
            case '~': /*#~*/ SaveNewFile (TRUE); break;
            case ';': /*#;*/ CutMaps (); break;
            case '_': /*#_*/ AbutHorizontal (); break;
            case '|': /*#|*/ AbutVertical (); break;
            case '#': /*##*/ BreakUpText (); break;
            case '^': /*#^*/ ScrollToOrigin (); break;
            case '@': /*#@*/ ToggleMoveMode (); break;
            case '$': /*#$*/ SetCurChoice (VERTEXMODE); break;
            case '&': /*#&*/ AlignSelToPage (); break;
            case '*': /*#**/ RedoCmd (); break;
            case '(': /*#(*/ ImportEPSFile (); break;
            case ')': /*#)*/ ScaleAllSelObj (); break;
            case '<': /*#<*/ LockSelObj (); break;
            case '>': /*#>*/ UnlockSelObj (); break;
         }
      }
      else if ((key_ev->state & Mod1Mask) && (key_ev->state & ControlMask))
      {
         switch (key_sym&0xff)
         {
            case 'a': /*^#a*/ AddPoint (); break;
            case 'b': /*^#b*/ ChangeFontStyle (STYLE_BR); break;
            case 'c': /*^#c*/ ChangeFontJust (JUST_C); break;
            case 'd': /*^#d*/ DeletePoint (); break;
            case 'e': /*^#e*/ SetCurChoice (DRAWRCBOX); break;
            case 'f': /*^#f*/ InvertXBitmaps (); break;
            case 'g': /*^#g*/ ToggleSnapOn (); break;
            case 'h': /*^#h*/ HideAllAttrs (); break;
            case 'i': /*^#i*/ MakeIconic (); break;
            case 'j': /*^#j*/ UnMakeIconic (); break;
            case 'k': /*^#k*/ ToggleColorPostScript (); break;
            case 'l': /*^#l*/ ChangeFontJust (JUST_L); break;
            case 'm': /*^#m*/ MakeSymbolic (); break;
            case 'n': /*^#n*/ UnMakeSymbolic (); break;
            case 'o': /*^#o*/ ChangeFontStyle (STYLE_NR); break;
            case 'p': /*^#p*/ ChangeFontStyle (STYLE_BI); break;
            case 'q': /*^#q*/ SetCurChoice (DRAWPOLYGON); break;
            case 'r': /*^#r*/ ChangeFontJust (JUST_R); break;
            case 's': /*^#s*/ SaveNewFile (FALSE); break;
            case 't': /*^#t*/ ChangeFontStyle (STYLE_NI); break;
            case 'u': /*^#u*/ UpdateSymbols (); break;
            case 'v': /*^#v*/ SetCurChoice (DRAWCIRCLE); break;
            case 'w': /*^#w*/ ToggleAllSelLineType (); break;
            case 'x': /*^#x*/ ToggleWhereToPrint (); break;
            case 'y': /*^#y*/ PasteFromCutBuffer (); break;
            case 'z': /*^#z*/ SetCurChoice (DRAWARC); break;
            case '.': /*^#.*/ ImportXBitmapFile (); break;
            case ',': /*^#,*/ ImportXPixmapFile (); break;
            case '-': /*^#-*/ ToggleGridSystem (); break;
         }
      }
      else if (key_sym == '\0' && key_ev->state == 0)
      {
         XButtonEvent	button_ev;

         button_ev.state = ShiftMask;
         if (strcmp (name, "ScrollPageUp()") == 0)
            ScrollUp (&button_ev);
         else if (strcmp (name, "ScrollPageDown()") == 0)
            ScrollDown (&button_ev);
         else if (strcmp (name, "ScrollPageRight()") == 0)
            ScrollRight (&button_ev);
         else if (strcmp (name, "ScrollPageLeft()") == 0)
            ScrollLeft (&button_ev);
         else if (strcmp (name, "FlushUndoBuffer()") == 0)
            CleanUpCmds ();
         else if (strcmp (name, "PrintMsgBuffer()") == 0)
            PrintMsgBuffer ();
         else if (strcmp (name, "SaveOrigin()") == 0)
            SaveOrigin ();
         else if (strcmp (name, "RestoreImageWH()") == 0)
            RestoreImageWH ();
         else if (strcmp (name, "UpdateEPS()") == 0)
            UpdateEPS ();
         else if (strcmp (name, "ToggleMapShown()") == 0)
            ToggleMapShown ();
         else if (strcmp (name, "ToggleUseGrayScale()") == 0)
            ToggleUseGray ();
         else if (strcmp (name, "FreeHandMode()") == 0)
            SetCurChoice (FREEHAND);
      }
      return (BAD);
   }
   return (INVALID);
}

int SomethingDirty ()
{
   register struct ObjRec	* obj_ptr = topObj;

   for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->dirty)
         return (TRUE);
   return (FALSE);
}

int DrawingEventHandler (input)
   XEvent	* input;
{
   int		mouse_x, mouse_y, grid_x, grid_y;
   XEvent	ev;
   XButtonEvent	* button_ev;

   if (input->type == Expose)
   {
      XSync (mainDisplay, False);
      while (XCheckWindowEvent (mainDisplay, drawWindow, ExposureMask, &ev)) ;

      if (topSel != NULL || curChoice == VERTEXMODE || SomethingDirty ())
         ClearAndRedrawDrawWindow ();
      else
      {
         RedrawDrawWindow (botObj);
         RedrawCurText ();
      }
      return (INVALID);
   }
   else if (input->type == MotionNotify)
   {
      while (XCheckWindowEvent (mainDisplay,drawWindow,PointerMotionMask,&ev)) ;

      mouse_x = (input->xmotion).x;
      mouse_y = (input->xmotion).y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      MarkRulers (grid_x, grid_y);
      return (INVALID);
   }

   if (input->type == ButtonPress)
   {
      button_ev = &(input->xbutton);
      if ((button_ev->button == Button3) &&
            (button_ev->state & (ShiftMask | ControlMask)))
      {
         SetCurChoice (NOTHING);
         return (INVALID);
      }
      else if ((button_ev->button == Button2) && curChoice == NOTHING &&
            (button_ev->state & (ShiftMask | ControlMask)))
      {
         Teleport (button_ev);
         return (INVALID);
      }
      else if (button_ev->button == Button2)
         return (MainMenu ());
      else if (button_ev->button == Button3)
      {
         ModeMenu (button_ev->x_root, button_ev->y_root);
         return (INVALID);
      }
      Msg ("");
   }

   switch(curChoice)
   {
      case NOTHING: Select (input); break;
      case VERTEXMODE: Select (input); break;
      case DRAWTEXT: DrawText (input); break;
      case DRAWBOX: DrawBox (input); break;
      case DRAWCIRCLE: DrawOval (input); break;
      case DRAWPOLY: DrawPoly (input); break;
      case DRAWPOLYGON: DrawPolygon (input); break;
      case DRAWARC: DrawArc (input); break;
      case DRAWRCBOX: DrawRCBox (input); break;
      case FREEHAND: DrawPoly (input); break;
   }
   return (INVALID);
}
