/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/font.c,v 2.59 1992/03/31 07:53:16 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "auxtext.e"
#include "choice.e"
#include "color.e"
#include "cursor.e"
#include "drawing.e"
#include "mainmenu.e"
#include "mark.e"
#include "menu.e"
#include "obj.e"
#include "pattern.e"
#include "raster.e"
#include "select.e"
#include "setup.e"
#include "text.e"

#define FONTS_PER_DPI (((MAXFONTS-1)*MAXFONTSTYLES+1)*MAXFONTSIZES)
#define FONTTABLESIZE (MAXFONTDPIS*FONTS_PER_DPI)

#define COUR8R_75  FontIndex(FONT_DPI_75,FONT_COU,0,STYLE_NR)
#define COUR10R_75 FontIndex(FONT_DPI_75,FONT_COU,1,STYLE_NR)
#define COUR12R_75 FontIndex(FONT_DPI_75,FONT_COU,2,STYLE_NR)
#define COUR14R_75 FontIndex(FONT_DPI_75,FONT_COU,3,STYLE_NR)
#define COUR18R_75 FontIndex(FONT_DPI_75,FONT_COU,4,STYLE_NR)
#define COUR24R_75 FontIndex(FONT_DPI_75,FONT_COU,5,STYLE_NR)

#define COUR11R_100 FontIndex(FONT_DPI_100,FONT_COU,0,STYLE_NR)
#define COUR14R_100 FontIndex(FONT_DPI_100,FONT_COU,1,STYLE_NR)
#define COUR17R_100 FontIndex(FONT_DPI_100,FONT_COU,2,STYLE_NR)
#define COUR20R_100 FontIndex(FONT_DPI_100,FONT_COU,3,STYLE_NR)
#define COUR25R_100 FontIndex(FONT_DPI_100,FONT_COU,4,STYLE_NR)
#define COUR34R_100 FontIndex(FONT_DPI_100,FONT_COU,5,STYLE_NR)

struct MyFontRec {
#ifdef XI18N
   XFontSet	xfs;
   XFontStruct	* xfs_sym;
#else
   XFontStruct	* xfs;
#endif
   int		valid;
   int		checked;
};

#ifdef XI18N
XFontSet	canvasFontPtr;
XFontStruct	* canvasFontSymPtr;
#else
XFontStruct	* canvasFontPtr;
#endif
int	canvasFontHeight;
int	canvasFontAsc;
int	canvasFontDes;

int	canvasFontIndex;

#ifdef XI18N
XFontSet	rulerFontPtr;
#else
XFontStruct	* rulerFontPtr;
#endif
int	rulerFontWidth;
int	rulerFontHeight;
int	rulerFontAsc;
int	rulerFontDes;

#ifdef XI18N
XFontSet	defaultFontPtr;
#else
XFontStruct	* defaultFontPtr;
#endif
int	defaultFontWidth;
int	defaultFontHeight;
int	defaultFontAsc;
int	defaultFontDes;

int	* pointSize;
int	curFont = FONT_COU;
int	curSize = 4;
int	curStyle = STYLE_NR;
int	curFontDPI = FONT_DPI_75;
int	curRotate = ROTATE0;

int	pointSize75[] = { 8, 10, 12, 14, 18, 24 };
int	pointSize100[] = { 11, 14, 17, 20, 25, 34 };

/*
 * /8  ->     1
 * /4  ->     2   -   -   3
 * /2  ->     4   5   -   6   7   -   9
 * x1  ->     8  10  11  12  14  17  18  20  24  25  34
 * x2  ->    16   -  22   -  28   -  36  40  48  50  68
 * x4  ->    32   -  44   -  56   -  72  80  96 100 136
 * x8  ->    64   -  88   - 112   - 144 ...
 * x16 ->   128
 */

char	* fontMenuStr[] =
      {
#ifdef XI18N
         "Times", "Courier", "Helvetica", "NewCentury", "Multibyte", "Symbol"
#else
         "Times", "Courier", "Helvetica", "NewCentury", "Symbol"
#endif
      };
char	* sizeMenuStr[] =
      {
         "8 ",
         "10",
         "11",
         "12",
         "14",
         "17",
         "18",
         "20",
         "24",
         "25",
         "34"
      };
char	* styleMenuStr[] =
      {
         "Roman      ^#o",
         "Bold       ^#b",
         "Italic     ^#t",
         "BoldItalic ^#p",
         "--------------",
         "Left       ^#l",
         "Center     ^#c",
         "Right      ^#r"
      };

static struct MyFontRec	myFontInfo[FONTTABLESIZE];
static char	* fontDPIMenuStr[] = { "75dpi", "100dpi" };

#ifdef XI18N
#include "fontset.c"
#else
static char	* fontNameStr[] =
{
/* 75 dpi fonts */
  "-*-times-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--24-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--24-*-*-*-*-*-iso8859-1",

  "-*-courier-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--24-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--24-*-*-*-*-*-iso8859-1",

  "-*-helvetica-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--24-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--24-*-*-*-*-*-iso8859-1",

  "-*-new century schoolbook-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--24-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--24-*-*-*-*-*-iso8859-1",

  "-*-symbol-medium-r-normal--8-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--10-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--12-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--14-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--18-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--24-*-*-*-*-*-*-*",

/* 100 dpi fonts */
  "-*-times-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--34-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--34-*-*-*-*-*-iso8859-1",

  "-*-courier-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--34-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--34-*-*-*-*-*-iso8859-1",

  "-*-helvetica-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--34-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--34-*-*-*-*-*-iso8859-1",

  "-*-new century schoolbook-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--34-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--34-*-*-*-*-*-iso8859-1",

  "-*-symbol-medium-r-normal--11-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--14-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--17-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--20-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--25-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--34-*-*-*-*-*-*-*"
};
#endif /* XI18N */

static char	* charCodeToName[] =
{
/* \200 */ "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
/* \220 */ "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
/* \240 */ "x", "", "", "", "\\250", "", "x", "",
           "x", "x", "\\343", "", "x", "x", "x", "x",
/* \260 */ "x", "x", "x", "x", "\\302", "x", "", "\\264",
           "\\313", "x", "\\353", "", "x", "x", "x", "",
/* \300 */ "8#260 /Agrave", "8#265 /Aacute", "8#276 /Acircumflex",
           "8#300 /Atilde", "8#311 /Adieresis", "8#314 /Aring", "\\341",
           "8#321 /Ccedilla", "8#322 /Egrave", "8#323 /Eacute",
           "8#324 /Ecircumflex", "8#325 /Edieresis", "8#326 /Igrave",
           "8#327 /Iacute", "8#330 /Icircumflex", "8#331 /Idieresis",
/* \320 */ "x", "8#332 /Ntilde", "8#333 /Ograve", "8#334 /Oacute",
           "8#335 /Ocircumflex", "8#336 /Otilde", "8#337 /Odieresis", "x",
           "\\351", "8#340 /Ugrave", "8#342 /Uacute", "8#344 /Ucircumflex",
           "8#345 /Udieresis", "x", "x", "\\373",
/* \340 */ "8#346 /agrave", "8#347 /aacute", "8#354 /acircumflex",
           "8#355 /atilde", "8#356 /adieresis", "8#357 /aring", "\\361",
           "8#360 /ccedilla", "8#362 /egrave", "8#363 /eacute",
           "8#364 /ecircumflex", "8#366 /edieresis", "8#367 /igrave",
           "8#374 /iacute", "8#375 /icircumflex", "8#376 /idieresis",
/* \360 */ "x", "8#254 /ntilde", "8#255 /ograve", "8#256 /oacute",
           "8#257 /ocircumflex", "8#271 /otilde", "8#274 /odieresis", "x",
           "\\371", "8#275 /ugrave", "8#350 /uacute", "8#352 /ucircumflex",
           "8#370 /udieresis", "x", "x", "8#372 /ydieresis"
};

#ifndef XI18N
XFontStruct * GetFontStruct (font_index)
   register int	font_index;
{
   return (myFontInfo[font_index].xfs);
}
#endif

int FontIndex (dpi_index, font_index, size_index, style_index)
   register int	dpi_index, font_index, size_index, style_index;
{
   if (font_index == FONT_SYM)
      return (size_index+MAXFONTSIZES*(MAXFONTSTYLES*font_index) +
            dpi_index*FONTS_PER_DPI);
   else
      return (size_index+MAXFONTSIZES*(style_index+MAXFONTSTYLES*font_index) +
            dpi_index*FONTS_PER_DPI);
}

int ValidCharCode (c_ptr)
   char	* c_ptr;
{
   register int	index = (int)(*c_ptr - '\200');
   char		msg[MAXSTRING];

   if (*charCodeToName[index] == '\0' || *charCodeToName[index] == '8' ||
         *charCodeToName[index] == '\\')
      return (TRUE);
   else
   {
      sprintf (msg, "Unrecognized character code \\%o.  Character discarded!",
         (*c_ptr)&0xff);
      Msg (msg);
      return (FALSE);
   }
}

char * CharCodeTranslate (c_ptr)
   char	* c_ptr;
{
   register int	index = (int)(*c_ptr - '\200');

   if (*charCodeToName[index] == '\\' || *charCodeToName[index] == '8')
      return (charCodeToName[index]);
   else
      return (NULL);
}

static
void RemoveIllegalChars (TextPtr)
   struct TextRec	* TextPtr;
{
   register char	* c_ptr, * dest_c_ptr;
   struct StrRec	* str_ptr;

   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      for (c_ptr = dest_c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
         if (!(((*c_ptr) & 0x80) && !ValidCharCode (c_ptr)))
            *dest_c_ptr++ = *c_ptr;
      *dest_c_ptr = '\0';
   }
}

static int	encodeFont[MAXFONTS*MAXFONTSTYLES];
static short	* encodeCharFlags[MAXFONTS*MAXFONTSTYLES];
static int	encodeCharFlagsAllocated = FALSE;

static
void PrepareText (TextPtr)
   register struct TextRec	* TextPtr;
{
   register char	* c_ptr;
   struct StrRec	* str_ptr;
   int			index, byte_index;
   short		* flag_ptr;

   if (TextPtr->font == FONT_SYM) return;

   index = TextPtr->font*MAXFONTSTYLES + TextPtr->style;
   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      for (c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
         if (((*c_ptr)&0x80) && *charCodeToName[(int)(*c_ptr-'\200')]=='8')
         {
            encodeFont[index] = TRUE;
            flag_ptr = encodeCharFlags[index];
            byte_index = (int)(((*c_ptr) & 0x7f) / 8);
            flag_ptr[byte_index] |= (1<<(((*c_ptr) & 0x7f) % 8));
         }
   }
}

static
void PrepareObjFontInfo (ObjPtr)
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;

   for (obj_ptr = ObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT: PrepareText (obj_ptr->detail.t); break;

         case OBJ_GROUP:
         case OBJ_SYM: PrepareObjFontInfo (obj_ptr->detail.r->last); break;
      }
      for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
         PrepareText (attr_ptr->obj->detail.t);
   }
}

void PrepareEightBitFontInfo ()
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;
   register int			j, i;
   short			* flag_ptr;

   if (!encodeCharFlagsAllocated)
   {
      for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++)
         encodeCharFlags[i] = (short *) calloc (16, sizeof(short));
      encodeCharFlagsAllocated = TRUE;
   }

   for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++)
   {
      encodeFont[i] = FALSE;
      flag_ptr = encodeCharFlags[i];
      for (j = 0; j < 16; j++) flag_ptr[j] = 0;
   }

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT: PrepareText (obj_ptr->detail.t); break;

         case OBJ_GROUP:
         case OBJ_SYM: PrepareObjFontInfo (obj_ptr->detail.r->last); break;
      }
      for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
         PrepareText (attr_ptr->obj->detail.t);
   }
}

int NeedEncode (FontIndex, StyleIndex)
{
#ifdef XI18N
   if (FontIndex == FONT_SYM || FontIndex == FONT_MUL)
#else
   if (FontIndex == FONT_SYM)
#endif
      return (FALSE);
   else
      return (encodeFont[FontIndex*MAXFONTSTYLES+StyleIndex]);
}

void GetPSFontStr (FontIndex, StyleIndex, FontStr)
   int	FontIndex, StyleIndex;
   char	* FontStr;
{
#ifdef XI18N
   if (FontIndex == FONT_MUL) {
      if (StyleIndex != STYLE_BR)
	strcpy (FontStr, "/Ryumin-Times-H");
      else
	strcpy (FontStr, "/Gothic-Helvetica-H");
      return;
   } else
#endif
   if (FontIndex == FONT_SYM)
   {
      strcpy (FontStr, "/Symbol");
      return;
   }
   switch (FontIndex)
   {
      case FONT_TIM: strcpy (FontStr, "/Times"); break;
      case FONT_COU: strcpy (FontStr, "/Courier"); break;
      case FONT_HEL: strcpy (FontStr, "/Helvetica"); break;
      case FONT_CEN: strcpy (FontStr, "/NewCenturySchlbk"); break;
   }
   switch (StyleIndex)
   {
      case STYLE_BI:
         switch (FontIndex)
         {
            case FONT_TIM: strcat (FontStr, "-BoldItalic"); break;
            case FONT_COU: strcat (FontStr, "-BoldOblique"); break;
            case FONT_HEL: strcat (FontStr, "-BoldOblique"); break;
            case FONT_CEN: strcat (FontStr, "-BoldItalic"); break;
         }
         break;
      case STYLE_BR: strcat (FontStr, "-Bold"); break;
      case STYLE_NI:
         switch (FontIndex)
         {
            case FONT_TIM: strcat (FontStr, "-Italic"); break;
            case FONT_COU: strcat (FontStr, "-Oblique"); break;
            case FONT_HEL: strcat (FontStr, "-Oblique"); break;
            case FONT_CEN: strcat (FontStr, "-Italic"); break;
         }
         break;
      case STYLE_NR:
         switch (FontIndex)
         {
            case FONT_TIM: strcat (FontStr, "-Roman"); break;
            case FONT_COU: break;
            case FONT_HEL: break;
            case FONT_CEN: strcat (FontStr, "-Roman"); break;
         }
         break;
   }
}

void CurFontMsg ()
{
   char	msg[MAXSTRING], style[20];

   switch (curStyle)
   {
      case STYLE_BI: strcpy (style, "BoldItalic"); break;
      case STYLE_BR: strcpy (style, "Bold"); break;
      case STYLE_NI: strcpy (style, "Italic"); break;
      case STYLE_NR: strcpy (style, "Roman"); break;
   }

   sprintf (msg, "%s-%s-%1dpt-%s", fontMenuStr[curFont], style,
         pointSize[curSize], fontDPIMenuStr[curFontDPI]);
   Msg (msg);
}

void DumpEightBitFontInfo (FP)
   FILE	* FP;
{
   register int		k, j;
   register short	* flag_ptr, flag;
   int			font_index, style_index;
   char			font_str[MAXSTRING];

   for (font_index = FONT_TIM; font_index < FONT_SYM; font_index++)
   {
      for (style_index = 0; style_index < MAXFONTSTYLES; style_index++)
      {
         if (NeedEncode (font_index, style_index))
         {
            GetPSFontStr (font_index, style_index, font_str);
            fprintf (FP, "%s-vec [\n", font_str);
            flag_ptr = encodeCharFlags[font_index*MAXFONTSTYLES+style_index];
            for (j = 0; j < 16; j++, flag_ptr++)
            {
               flag = *flag_ptr;
               if ((flag & 0xff) != 0)
               {
                  for (k = 0; k < 8; k++)
                     if (flag & (1<<k))
                        fprintf (FP, " %s\n", charCodeToName[j*8+k]);
               }
            }
            fprintf (FP, " ] def\n");
            fprintf (FP, "%s %s-8 %s-vec tgifReEncodeSmall\n\n", font_str,
                  font_str, &font_str[1]);
         }
      }
   }
}

#ifdef XI18N
void SetCanvasFont ()
{
   register int	index, old_index;
   char		msg[MAXSTRING];
   int		watch_cursor;
   char		**missing_charsets;
   int		missing_count;
   char		*def_string;
   XFontSetExtents *ext;

   old_index = -1;

   index = FontIndex (curFontDPI, curFont, curSize, curStyle);
   if (!myFontInfo[index].valid)
   {
      watch_cursor = watchCursorOnMainWindow;
      if (!watch_cursor)
      {
         SetWatchCursor (drawWindow);
         SetWatchCursor (mainWindow);
      }
      myFontInfo[index].checked = TRUE;
      if ((myFontInfo[index].xfs =
            XCreateFontSet(mainDisplay, fontNameStr[index],
		&missing_charsets, &missing_count, &def_string)) == NULL)
      {
         old_index = index;
         curFontDPI = (curFontDPI==FONT_DPI_75) ? FONT_DPI_100 : FONT_DPI_75;
         index = FontIndex (curFontDPI, curFont, curSize, curStyle);
         if (!myFontInfo[index].valid)
         {
            myFontInfo[index].checked = TRUE;
            if ((myFontInfo[index].xfs =
            XCreateFontSet(mainDisplay, fontNameStr[index],
		&missing_charsets, &missing_count, &def_string)) == NULL)
            {
               sprintf (msg, "Can not open '%s' nor\n\t'%s'! (no font or memory)",
                     fontNameStr[old_index], fontNameStr[index]);
               Error ("SetCanvasFont()", msg);
            }
            fprintf (stderr, "%s '%s' font,\n\tuse '%s' instead.\n",
                  "Warning:  can not open", fontNameStr[old_index],
                  fontNameStr[index]);
            SetFileModified (TRUE);
         }
      }
      if (missing_count > 0) {
         WPRINTF("Warning: missing charsets in XCreateFontSet of ");
         if (old_index >= 0)
            WPRINTF2("'%s'\n", fontNameStr[old_index]);
         else
            WPRINTF2("'%s'\n", fontNameStr[index]);
      }
      myFontInfo[index].valid = TRUE;
      if (!watch_cursor)
      {
         SetDefaultCursor (mainWindow);
         ShowCursor ();
      }
   }

   canvasFontIndex = index;

   canvasFontPtr = myFontInfo[index].xfs;
   ext = XExtentsOfFontSet(canvasFontPtr);
   canvasFontAsc = - ext->max_logical_extent.y; /* y is negative. */
   canvasFontDes = ext->max_logical_extent.height + ext->max_logical_extent.y;
   canvasFontHeight = ext->max_logical_extent.height;

   textCursorH = canvasFontHeight;

   pointSize = (curFontDPI==FONT_DPI_75) ? pointSize75 : pointSize100;
}

/*
 * ConflictTextFont():
 *   returns true if the font (FontIndex) selected on Font menu conflicts
 *   with the font of selected text object (botSel), i.e, one of them
 *   is font set(multibyte font).
 *
 * Parameters:
 * FontIndex -- font selected on Font menu.
 */
int ConflictTextFont(FontIndex)
   int	FontIndex;
{
   register struct SelRec	* sel_ptr;
   register struct ObjRec	* obj_ptr;

   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev) {
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->type == OBJ_TEXT && 
	  (obj_ptr->detail.t->font != FontIndex) &&
	  (obj_ptr->detail.t->font == FONT_MUL || FontIndex == FONT_MUL))
	return(1);
   }
   return(0);
}

#define canvasFontPtr	canvasFontSymPtr

void SetCanvasFontSym ()
#else
void SetCanvasFont ()
#endif
{
   register int	index, old_index;
   char		msg[MAXSTRING];
   int		watch_cursor;

   index = FontIndex (curFontDPI, curFont, curSize, curStyle);
   if (!myFontInfo[index].valid)
   {
      watch_cursor = watchCursorOnMainWindow;
      if (!watch_cursor)
      {
         SetWatchCursor (drawWindow);
         SetWatchCursor (mainWindow);
      }
      myFontInfo[index].checked = TRUE;
#ifdef XI18N
      if ((myFontInfo[index].xfs_sym =
#else
      if ((myFontInfo[index].xfs =
#endif
            XLoadQueryFont (mainDisplay, fontNameStr[index])) == NULL)
      {
         old_index = index;
         curFontDPI = (curFontDPI==FONT_DPI_75) ? FONT_DPI_100 : FONT_DPI_75;
         index = FontIndex (curFontDPI, curFont, curSize, curStyle);
         if (!myFontInfo[index].valid)
         {
            myFontInfo[index].checked = TRUE;
#ifdef XI18N
            if ((myFontInfo[index].xfs_sym =
#else
            if ((myFontInfo[index].xfs =
#endif
                  XLoadQueryFont (mainDisplay, fontNameStr[index])) == NULL)
            {
               sprintf (msg, "Can not open '%s' nor\n\t'%s'!",
                     fontNameStr[old_index], fontNameStr[index]);
               Error ("SetCanvasFont()", msg);
            }
            fprintf (stderr, "%s '%s' font,\n\tuse '%s' instead.\n",
                  "Warning:  can not open", fontNameStr[old_index],
                  fontNameStr[index]);
            SetFileModified (TRUE);
         }
      }
      myFontInfo[index].valid = TRUE;
      if (!watch_cursor)
      {
         SetDefaultCursor (mainWindow);
         ShowCursor ();
      }
   }

   canvasFontIndex = index;

#ifdef XI18N
   canvasFontPtr = myFontInfo[index].xfs_sym;
#else
   canvasFontPtr = myFontInfo[index].xfs;
#endif
   canvasFontAsc = canvasFontPtr->max_bounds.ascent;
   canvasFontDes = canvasFontPtr->max_bounds.descent;
   canvasFontHeight = canvasFontAsc + canvasFontDes;

   XSetFont (mainDisplay, drawGC, canvasFontPtr->fid);

   textCursorH = canvasFontHeight;

   pointSize = (curFontDPI==FONT_DPI_75) ? pointSize75 : pointSize100;
}
#ifdef XI18N
#undef canvasFontPtr
#endif

void InitFonts ()
{
   register int	i;
   int		ruler_index, default_index;
   int		old_ruler_index, old_default_index, len, dpi;
   char		* s;
#ifdef XI18N
   char		**missing_charsets;
   int		missing_count;
   char		*def_string;
   XFontSetExtents *ext;

   old_ruler_index = old_default_index = -1;
   for (i = 0; i < FONTTABLESIZE; i++) {
	myFontInfo[i].valid = FALSE;
      myFontInfo[i].checked = FALSE;
	myFontInfo[i].xfs = NULL;
	myFontInfo[i].xfs_sym = NULL;
   }
#else

   for (i = 0; i < FONTTABLESIZE; i++)
   {
      myFontInfo[i].valid = FALSE;
      myFontInfo[i].checked = FALSE;
   }
#endif

   for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++)
      encodeCharFlags[i] = (short *) calloc (16, sizeof(short));
   encodeCharFlagsAllocated = TRUE;

   curFont = FONT_COU;
   curStyle = STYLE_NR;
   curSize = 4;
   curFontDPI = FONT_DPI_75;

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFontDPI")) != NULL)
   {
      dpi = atoi (s);
      switch (dpi)
      {
         case 75: curFontDPI = FONT_DPI_75; curSize = 4; break;
         case 100: curFontDPI = FONT_DPI_100; curSize = 2; break;
         default:
            fprintf (stderr,"Warning:  can not set InitialFontDPI to '%s'\n",s);
            break;
      }
   }

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "MsgFontSizeIndex")) != NULL)
   {
      if (atoi (s) < MAXFONTSIZES)
         curSize = atoi (s);
      else
         fprintf (stderr, "%s '%s', '%1d' is used.\n",
               "Warning:  can not set MsgFontSizeIndex to", s, curSize);
   }

   default_index = FontIndex (curFontDPI, curFont, curSize, curStyle);
   myFontInfo[default_index].checked = TRUE;
   if ((myFontInfo[default_index].xfs =
#ifdef XI18N
            XCreateFontSet(mainDisplay, fontNameStr[default_index],
		&missing_charsets, &missing_count, &def_string)) == NULL)
#else
         XLoadQueryFont (mainDisplay, fontNameStr[default_index])) == NULL)
#endif
   {
      old_default_index = default_index;
      curSize = 2;
      curFontDPI = FONT_DPI_100;
      default_index = FontIndex (curFontDPI, curFont, curSize, curStyle);
      myFontInfo[default_index].checked = TRUE;
      if ((myFontInfo[default_index].xfs =
#ifdef XI18N
            XCreateFontSet(mainDisplay, fontNameStr[default_index],
		&missing_charsets, &missing_count, &def_string)) == NULL)
      {
         fprintf (stderr, "Can not open '%s' nor\n\t'%s'!  %s aborted! (no font or memory)\n\n",
#else
            XLoadQueryFont (mainDisplay, fontNameStr[default_index])) == NULL)
      {
         fprintf (stderr, "Can not open '%s' nor\n\t'%s'!  %s aborted!\n\n",
#endif
               fontNameStr[old_default_index], fontNameStr[default_index],
               TOOL_NAME);
         exit (-1);
      }
   }
#ifdef XI18N
   if (missing_count > 0) {
      WPRINTF("Warning: missing charsets in XCreateFontSet of ");
      if (old_default_index >= 0)
         WPRINTF2("'%s'\n", fontNameStr[old_default_index]);
      else
         WPRINTF2("'%s'\n", fontNameStr[default_index]);
   }
#endif
   myFontInfo[default_index].valid = TRUE;

   ruler_index = FontIndex (FONT_DPI_75, curFont, 1, curStyle);
   myFontInfo[ruler_index].checked = TRUE;
   if ((myFontInfo[ruler_index].xfs =
#ifdef XI18N
            XCreateFontSet(mainDisplay, fontNameStr[ruler_index],
		&missing_charsets, &missing_count, &def_string)) == NULL)
#else
         XLoadQueryFont (mainDisplay, fontNameStr[ruler_index])) == NULL)
#endif
   {
      old_ruler_index = ruler_index;
      ruler_index = FontIndex (FONT_DPI_100, curFont, 0, curStyle);
      myFontInfo[ruler_index].checked = TRUE;
      if ((myFontInfo[ruler_index].xfs =
#ifdef XI18N
            XCreateFontSet(mainDisplay, fontNameStr[ruler_index],
		&missing_charsets, &missing_count, &def_string)) == NULL)
      {
         fprintf (stderr, "Can not open '%s' nor\n\t'%s'!  %s aborted! (no font or memory)\n\n",
#else
            XLoadQueryFont (mainDisplay, fontNameStr[ruler_index])) == NULL)
      {
         fprintf (stderr, "Can not open '%s' nor\n\t'%s'!  %s aborted!\n\n",
#endif
               fontNameStr[old_ruler_index], fontNameStr[ruler_index],
               TOOL_NAME);
         exit (-1);
      }
   }
#ifdef XI18N
   if (missing_count > 0) {
       WPRINTF("Warning: missing charsets in XCreateFontSet of ");
       if (old_ruler_index >= 0)
          WPRINTF2("'%s'\n", fontNameStr[old_ruler_index]);
       else
          WPRINTF2("'%s'\n", fontNameStr[ruler_index]);
   }
#endif
   myFontInfo[ruler_index].valid = TRUE;

   defaultFontPtr = myFontInfo[default_index].xfs;
#ifdef XI18N
   ext = XExtentsOfFontSet(defaultFontPtr);
   defaultFontWidth = ext->max_logical_extent.width;
   defaultFontAsc = - ext->max_logical_extent.y; /* y is negative. */
   defaultFontDes = ext->max_logical_extent.height + ext->max_logical_extent.y;
   defaultFontHeight = ext->max_logical_extent.height;

   rulerFontPtr = myFontInfo[ruler_index].xfs;
   ext = XExtentsOfFontSet(rulerFontPtr);
   rulerFontWidth = ext->max_logical_extent.width;
   rulerFontAsc = - ext->max_logical_extent.y; /* y is negative. */
   rulerFontDes = ext->max_logical_extent.height + ext->max_logical_extent.y;
   rulerFontHeight = ext->max_logical_extent.height;
#else
   defaultFontWidth = defaultFontPtr->max_bounds.width;
   defaultFontAsc = defaultFontPtr->max_bounds.ascent;
   defaultFontDes = defaultFontPtr->max_bounds.descent;
   defaultFontHeight = defaultFontAsc + defaultFontDes;

   rulerFontPtr = myFontInfo[ruler_index].xfs;
   rulerFontWidth = rulerFontPtr->max_bounds.width;
   rulerFontAsc = rulerFontPtr->max_bounds.ascent;
   rulerFontDes = rulerFontPtr->max_bounds.descent;
   rulerFontHeight = rulerFontAsc + rulerFontDes;
#endif

   pointSize = (curFontDPI==FONT_DPI_75) ? pointSize75 : pointSize100;

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFont")) != NULL)
   {
      for (i = 0; i < MAXFONTS; i++)
         if (strcmp (s, fontMenuStr[i]) == 0)
            break;
      if (i != MAXFONTS)
         curFont = i;
      else
         fprintf (stderr, "Warning:  can not set InitialFont to '%s'\n", s);
   }
   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFontStyle")) != NULL)
   {
      len = strlen (s);
      if (len < strlen (styleMenuStr[0]))
      {
         for (i = 0; i < MAXFONTSTYLES; i++)
            if (strncmp (s, styleMenuStr[i], len) == 0)
               break;
         if (i != MAXFONTSTYLES)
            curStyle = i;
         else
            fprintf (stderr, "%s '%s'\n",
                  "Warning:  can not set InitialFontStyle to", s);
      }
      else
         fprintf (stderr,"Warning:  can not set InitialFontStyle to '%s'\n",s);
   }
   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFontJust")) != NULL)
   {
      len = strlen (s);
      if (len < strlen (styleMenuStr[0]))
      {
         for (i = MAXFONTSTYLES+1; i < MAXFONTSTYLES+1+MAXJUSTS; i++)
            if (strncmp (s, styleMenuStr[i], len) == 0)
               break;
         if (i != MAXFONTSTYLES+1+MAXJUSTS)
            textJust = i-MAXFONTSTYLES-1;
         else
            fprintf (stderr, "%s '%s'\n",
                  "Warning:  can not set InitialFontJust to", s);
      }
      else
         fprintf (stderr, "Warning:  can not set InitialFontJust to '%s'\n", s);
   }
   if ((s = XGetDefault (mainDisplay,TOOL_NAME,"InitialFontSizeIndex")) != NULL)
   {
      if (atoi (s) < MAXFONTSIZES)
         curSize = atoi (s);
      else
         fprintf (stderr, "%s '%s'\n",
               "Warning:  can not set InitialFontSizeIndex to", s);
   }
}

int ChangeObjTextStyle (ObjPtr, StyleIndex)
   register struct ObjRec	* ObjPtr;
   register int			StyleIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->style != StyleIndex)
         {
            ObjPtr->detail.t->style = StyleIndex;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextStyle (obj_ptr, StyleIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontStyle (StyleIndex)
   int	StyleIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				changed = FALSE, text_cursor_shown;

   if (StyleIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curStyle = StyleIndex;
#ifdef XI18N
      if (curFont == FONT_SYM)
	SetCanvasFontSym ();
      else
#endif
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_STYLE);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextStyle (sel_ptr->obj, StyleIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int ChangeObjTextJust (ObjPtr, JustIndex)
   register struct ObjRec	* ObjPtr;
   register int			JustIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->just != JustIndex)
         {
            ObjPtr->detail.t->just = JustIndex;
            UpdTextBBox (ObjPtr);

            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextJust (obj_ptr, JustIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontJust (JustIndex)
   int JustIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby;
   int				text_obj_created, text_cursor_shown;
   int				changed = FALSE;

   if (JustIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      textJust = JustIndex;
      ShowJust ();
      UpdateSubMenu (MENU_STYLE);
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextJust (sel_ptr->obj, JustIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

void StyleSubMenu (index)
   int	index;
{
   if (index < MAXFONTSTYLES)
      ChangeFontStyle (index);
   else if (index > MAXFONTSTYLES)
      ChangeFontJust (index - MAXFONTSTYLES - 1);
}

void StyleMenu (X, Y)
   int  X, Y;
{
   register int	index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXFONTSTYLES+MAXJUSTS+1, &fore_colors, &valid,
         &init_rv);

   init_rv[curStyle] = TRUE;
   init_rv[MAXFONTSTYLES+1+textJust] = TRUE;
   activeMenu = MENU_STYLE;
   index = TextMenuLoop (X, Y, styleMenuStr, MAXFONTSTYLES+MAXJUSTS+1,
         fore_colors, valid, init_rv, SINGLECOLOR);

   if (index != INVALID) StyleSubMenu (index);
}

int ChangeObjTextSize (ObjPtr, SizeIndex)
   register struct ObjRec	* ObjPtr;
   register int			SizeIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->size != SizeIndex ||
               ObjPtr->detail.t->dpi != curFontDPI)
         {
            ObjPtr->detail.t->size = SizeIndex;
            ObjPtr->detail.t->dpi = curFontDPI;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextSize (obj_ptr, SizeIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontSize (SizeIndex)
   int	SizeIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				text_cursor_shown, changed = FALSE;

   if (SizeIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curSize = SizeIndex;
#ifdef XI18N
      if (curFont == FONT_SYM)
	SetCanvasFontSym ();
      else
#endif
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_SIZE);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextSize (sel_ptr->obj, SizeIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int GetSizeMenuIndex ()
{
   switch (curFontDPI)
   {
      case FONT_DPI_75:
         switch (curSize)
         {
            case 0: return (0);
            case 1: return (1);
            case 2: return (3);
            case 3: return (4);
            case 4: return (6);
            case 5: return (8);
         }
         break;
      case FONT_DPI_100:
         switch (curSize)
         {
            case 0: return (2);
            case 1: return (4);
            case 2: return (5);
            case 3: return (7);
            case 4: return (9);
            case 5: return (10);
         }
         break;
   }
   return (INVALID);
}

int ParseSizeMenuIndex (index, dpi)
   int	index, *dpi;
{
   switch (index)
   {
      case 0: *dpi = FONT_DPI_75; return (0);
      case 1: *dpi = FONT_DPI_75; return (1);
      case 2: *dpi = FONT_DPI_100; return (0);
      case 3: *dpi = FONT_DPI_75; return (2);
      case 4: *dpi = FONT_DPI_75; return (3);
      case 5: *dpi = FONT_DPI_100; return (2);
      case 6: *dpi = FONT_DPI_75; return (4);
      case 7: *dpi = FONT_DPI_100; return (3);
      case 8: *dpi = FONT_DPI_75; return (5);
      case 9: *dpi = FONT_DPI_100; return (4);
      case 10: *dpi = FONT_DPI_100; return (5);
   }
   return (INVALID);
}

void SizeSubMenu (index)
   int	index;
{
   int	saved_dpi=curFontDPI, new_index;

   if (topSel != NULL) saved_dpi = curFontDPI;
   new_index = ParseSizeMenuIndex (index, &curFontDPI);
   ChangeFontSize (new_index);
   if (topSel != NULL) curFontDPI = saved_dpi;
}

void SizeMenu (X, Y)
   int  X, Y;
{
   register int	index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (2*MAXFONTSIZES-1, &fore_colors, &valid, &init_rv);

   init_rv[GetSizeMenuIndex ()] = TRUE;
   activeMenu = MENU_SIZE;
   index = TextMenuLoop (X, Y, sizeMenuStr, 2*MAXFONTSIZES-1, fore_colors,
         valid, init_rv, SINGLECOLOR);

   if (index != INVALID) SizeSubMenu (index);
}

int ChangeObjTextFont (ObjPtr, FontIndex)
   register struct ObjRec	* ObjPtr;
   register int			FontIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->font != FontIndex)
         {
            if (ObjPtr->detail.t->font == FONT_SYM)
               RemoveIllegalChars (ObjPtr->detail.t);
            ObjPtr->detail.t->font = FontIndex;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextFont (obj_ptr, FontIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFont (FontIndex)
   int	FontIndex;
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE, text_cursor_shown;
   int				ltx, lty, rbx, rby, text_obj_created;

   if (FontIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curFont = FontIndex;
#ifdef XI18N
      if (curFont == FONT_SYM)
	SetCanvasFontSym ();
      else
#endif
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_FONT);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextFont (sel_ptr->obj, FontIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

void FontMenu (X, Y)
   int  X, Y;
{
   register int			index;
   int				* fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXFONTS, &fore_colors, &valid, &init_rv);

   init_rv[curFont] = TRUE;
   activeMenu = MENU_FONT;
   index = TextMenuLoop (X, Y, fontMenuStr, MAXFONTS, fore_colors, valid,
         init_rv, SINGLECOLOR);

#ifdef XI18N
   /*
    * can not exchange Multibyte font set with one font
    * (English or Symbol).
    */
   if (index != INVALID) {
      if (!ConflictTextFont(index))
          ChangeFont (index);
      else
          Msg("Conflict with multibyte font.");
  }
#else
   if (index != INVALID) ChangeFont (index);
#endif
}

int ChangeObjVSpace (ObjPtr, VSpace)
   register struct ObjRec	* ObjPtr;
   register int			VSpace;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;
   int				saved_v_space;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->v_space != VSpace)
         {
            saved_v_space = ObjPtr->detail.t->v_space;
            ObjPtr->detail.t->v_space = VSpace;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("Vertical spacing for that object not changed.");
               ObjPtr->detail.t->v_space = saved_v_space;
               UpdTextBBox (ObjPtr);
            }
            else
            {
               if (ObjPtr->detail.t->cached_bitmap != None)
                  XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

               ObjPtr->detail.t->cached_zoom = 0;
               ObjPtr->detail.t->cached_bitmap = None;

               changed = TRUE;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjVSpace (obj_ptr, VSpace))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeVSpace (VSpace)
   int	VSpace;
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE, text_cursor_shown;
   int				ltx, lty, rbx, rby, text_obj_created;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      if (textCursorH+textVSpace <= 0)
         Msg ("Text vertical spacing too small.  No change.");
      else
      {
         textVSpace = VSpace;
#ifdef XI18N
	 if (curFont == FONT_SYM)
	     SetCanvasFontSym ();
	 else
#endif
         SetCanvasFont ();
         ShowTextVSpace ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjVSpace (sel_ptr->obj, VSpace))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

static int	savedFont, savedSize, savedStyle, savedDPI;
static int	savedJust, savedRotate, savedPen, savedVSpace;

void SaveCurFont ()
{
   savedFont = curFont;
   savedSize = curSize;
   savedStyle = curStyle;
   savedDPI = curFontDPI;
   savedJust = textJust;
   savedRotate = curRotate;
   savedPen = penPat;
   savedVSpace = textVSpace;
}

void RestoreCurFont ()
{
   curFont = savedFont;
   curSize = savedSize;
   curStyle = savedStyle;
   curFontDPI = savedDPI;
   textJust = savedJust;
   curRotate = savedRotate;
   penPat = savedPen;
   textVSpace = savedVSpace;

#ifdef XI18N
   if (curFont == FONT_SYM)
      SetCanvasFontSym ();
   else
#endif
   SetCanvasFont ();
}

static int	pushedFont, pushedSize, pushedStyle, pushedDPI, pushedJust;
static int	pushedColorIndex, pushedRotate, pushedPen, pushedVSpace;

void PushCurFont ()
{
   pushedFont = curFont;
   pushedSize = curSize;
   pushedStyle = curStyle;
   pushedDPI = curFontDPI;
   pushedJust = textJust;
   pushedRotate = curRotate;
   pushedPen = penPat;
   pushedVSpace = textVSpace;
   pushedColorIndex = colorIndex;
}

void PopCurFont ()
{
   curFont = pushedFont;
   curSize = pushedSize;
   curStyle = pushedStyle;
   curFontDPI = pushedDPI;
   textJust = pushedJust;
   curRotate = pushedRotate;
   penPat = pushedPen;
   textVSpace = pushedVSpace;
   colorIndex = pushedColorIndex;

#ifdef XI18N
   if (curFont == FONT_SYM)
	SetCanvasFontSym ();
   else
#endif
   SetCanvasFont ();
}

void CleanUpFonts ()
{
   register int	i;

   for (i = 0; i < FONTTABLESIZE; i++)
   {
      if (myFontInfo[i].valid)
      {
#ifdef XI18N
	 if (myFontInfo[i].xfs != NULL) {
	     XFreeFontSet (mainDisplay, myFontInfo[i].xfs);
	     myFontInfo[i].xfs = NULL;
	 }
	 if (myFontInfo[i].xfs_sym != NULL) {
	     XFreeFont (mainDisplay, myFontInfo[i].xfs_sym);
	     myFontInfo[i].xfs_sym = NULL;
	 }
#else
         XFreeFont (mainDisplay, myFontInfo[i].xfs);
#endif
         myFontInfo[i].valid = FALSE;
      }
   }
   for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++) cfree (encodeCharFlags[i]);
   encodeCharFlagsAllocated = FALSE;
}

/* The following procedure is used to generate pdrawFontAsc[] and */
/*    pDrawFontDes[], to be used in ``prtgif.c''.  It is supposed */
/*    to be called within dbx and not tgif.                       */

/*
 * static
 * void GenerateFontInfo ()
 * {
 *    register int	i, j, num_rows;
 * 
 *    for (i = 0; i < FONTTABLESIZE; i++)
 *       if (!myFontInfo[i].valid)
 *       {
 *          if ((myFontInfo[i].xfs =
 *                XLoadQueryFont (mainDisplay, fontNameStr[i])) == NULL)
 *          {
 *             printf ("Can not open %s.  Abort.\n\n", fontNameStr[i]);
 *             exit (-1);
 *          }
 *          myFontInfo[i].valid = TRUE;
 *       }
 * 
 *    num_rows = FONTTABLESIZE / MAXFONTSIZES;
 *    printf ("short\tpDrawFontAsc[] =\n{\n");
 *    for (i = 0; i < num_rows; i++)
 *    {
 *       printf ("   ");
 *       for (j = 0; j < MAXFONTSIZES; j++)
 *          if (i == num_rows-1 && j == MAXFONTSIZES-1)
 *             printf ("%2d ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.ascent);
 *          else
 *             printf ("%2d, ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.ascent);
 *       printf ("\n");
 *    }
 *    printf ("};\n\n");
 * 
 *    printf ("short\tpDrawFontDes[] =\n{\n");
 *    for (i = 0; i < num_rows; i++)
 *    {
 *       printf ("   ");
 *       for (j = 0; j < MAXFONTSIZES; j++)
 *          if (i == num_rows-1 && j == MAXFONTSIZES-1)
 *             printf ("%2d ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.descent);
 *          else
 *             printf ("%2d, ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.descent);
 *       printf ("\n");
 *    }
 *    printf ("};\n");
 * }
 */
