/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/msg.c,v 2.26 1992/05/07 01:21:43 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "button.e"
#include "copypaste.e"
#include "cursor.e"
#include "dialog.e"
#include "file.e"
#include "font.e"
#include "mainloop.e"
#include "menu.e"
#include "raster.e"
#include "setup.e"

#define MSG_ROWS 2

char	scanFileName[MAXPATHLENGTH];
int	scanLineNum = 0;

struct MsgRec {
   char			* s;
   struct MsgRec	* next, * prev;
};

static struct MsgRec	* topMsg = NULL, * botMsg = NULL;
static struct MsgRec	* mostRecentTopMsgPtr = NULL;
static int		msgCount = 0;
static int		topMsgNumber = 0, mostRecentTopMsgNumber = INVALID;

static
void AddMsg (Msg)
   char	* Msg;
{
   char			* s;
   struct MsgRec	* msg_ptr;

   if (*Msg == '\0') { topMsgNumber = msgCount; return; }

   s = (char *) calloc (strlen (Msg) + 1, sizeof(char));
   msg_ptr = (struct MsgRec *) calloc (1, sizeof(struct MsgRec));

   strcpy (s, Msg);
   msg_ptr->s = s;

   ++msgCount;
   if (msgCount > topMsgNumber+MSG_ROWS) topMsgNumber = msgCount-MSG_ROWS;

   msg_ptr->prev = botMsg;
   msg_ptr->next = NULL;

   if (botMsg == NULL)
      topMsg = msg_ptr;
   else
      botMsg->next = msg_ptr;

   botMsg = msg_ptr;
}

void CleanUpMsg ()
{
   register struct MsgRec	* msg_ptr, * prev_msg=NULL;

   for (msg_ptr = botMsg; msg_ptr != NULL; msg_ptr = prev_msg)
   {
      prev_msg = msg_ptr->prev;
      cfree (msg_ptr->s);
      cfree (msg_ptr);
   }
   topMsg = botMsg = mostRecentTopMsgPtr = NULL;
   msgCount = topMsgNumber = 0;
   mostRecentTopMsgNumber = INVALID;
}

static
struct MsgRec * FindMsg (Number)
   int	Number;
{
   register int			i;
   register struct MsgRec	* ptr = NULL;

   if (Number >= msgCount)
      return (botMsg);
   else if (Number < 0)
      return (topMsg);
   else if (Number > (int)(msgCount/2))
      for (i = msgCount-1, ptr = botMsg; i != Number; i--, ptr = ptr->prev) ;
   else
      for (i = 0, ptr = topMsg; i != Number; i++, ptr = ptr->next) ;

   return (ptr);
}

void RedrawMsg ()
{
   int			i, x, y;
   XEvent       	ev;
   struct MsgRec	* msg_ptr;

   XClearWindow (mainDisplay, msgWindow);
   XSync (mainDisplay, False);
   while (XCheckWindowEvent (mainDisplay, msgWindow, ExposureMask, &ev)) ;

   if (topMsgNumber == msgCount) return;

   x = 2;
   y = 2 + defaultFontAsc;

   mostRecentTopMsgPtr = msg_ptr = (topMsgNumber == mostRecentTopMsgNumber) ?
         mostRecentTopMsgPtr : FindMsg (topMsgNumber);
   mostRecentTopMsgNumber = topMsgNumber;

   for (i = topMsgNumber; i < min(msgCount,topMsgNumber+MSG_ROWS); i++)
   {
#ifdef XI18N
      XmbDrawString (mainDisplay, msgWindow, defaultFontPtr, defaultGC,
	    x, y, msg_ptr->s, strlen(msg_ptr->s));
#else
      XDrawString (mainDisplay, msgWindow, defaultGC, x, y, msg_ptr->s,
            strlen(msg_ptr->s));
#endif
      msg_ptr = msg_ptr->next;
      y += defaultFontHeight;
   }
   XSync (mainDisplay, False);
}

void Msg (Message)
   char	* Message;
{
   AddMsg (Message);
   RedrawMsg ();
}

void Warning (Where, Message)
   char	* Where, * Message;
{
   char	buf[MAXSTRING];

   sprintf (buf, "Warning in %s.  %s.", Where, Message);
   Msg (buf);
}

void TwoLineMsg (Msg1, Msg2)
   char	* Msg1, * Msg2;
{
   AddMsg (Msg1);
   AddMsg (Msg2);
   RedrawMsg ();
}

void PrintMsgBuffer ()
{
   char			file_name[MAXPATHLENGTH], * rest, msg[MAXSTRING];
   int			short_name=FALSE;
   FILE			*fp;
   struct MsgRec	* msg_ptr;

   Dialog ("Please enter a file name to write the message buffer content:",
         "( \"stdout\", \"stderr\", <ESC>: cancel )", file_name);
   if (*file_name == '\0') return;

   if (strcmp (file_name, "stdout") == 0)
   {
      for (msg_ptr = topMsg; msg_ptr != NULL; msg_ptr = msg_ptr->next)
         printf ("%s\n", msg_ptr->s);
   }
   else if (strcmp (file_name, "stderr") == 0)
   {
      for (msg_ptr = topMsg; msg_ptr != NULL; msg_ptr = msg_ptr->next)
         fprintf (stderr, "%s\n", msg_ptr->s);
   }
   else
   {
      if (!OkayToCreateFile (file_name)) return;

      if (short_name = IsPrefix (bootDir, file_name, &rest)) ++rest;
      if ((fp = fopen (file_name, "w")) == NULL)
      {
         if (short_name)
            sprintf (msg, "Can not open '%s', message buffer not written out.",
                  rest);
         else
            sprintf (msg, "Can not open '%s', message buffer not written out.",
                  file_name);
         Msg (msg);
         return;
      }
      for (msg_ptr = topMsg; msg_ptr != NULL; msg_ptr = msg_ptr->next)
         fprintf (fp, "%s\n", msg_ptr->s);
      fclose (fp);

      if (short_name)
         sprintf (msg, "Message buffer saved into '%s'.", rest);
      else
         sprintf (msg, "Message buffer saved into '%s'.", file_name);
      Msg (msg);
   }
}

void MsgEventHandler (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   double	frac;

   if (input->type == Expose)
      RedrawMsg ();
   else if (input->type == ButtonPress)
   {
      button_ev = &(input->xbutton);
      if (button_ev->button == Button1)
      {
         if (topMsgNumber+1 >= msgCount) return;

         topMsgNumber++;
         RedrawMsg ();
      }
      else if (button_ev->button == Button2)
      {
         int	done=FALSE, saved_y=button_ev->y, y;
         XEvent	ev;

         frac = ((double)saved_y) / ((double)msgWindowH);
         topMsgNumber = max(0,round(msgCount * frac));
         RedrawMsg ();

         XGrabPointer (mainDisplay, msgWindow, False,
            PointerMotionMask | ButtonReleaseMask, GrabModeAsync,
            GrabModeAsync, None, handCursor, CurrentTime);

         while (!done)
         {
            XNextEvent (mainDisplay, &ev);

            if (ev.type == Expose || ev.type == VisibilityNotify)
               ExposeEventHandler (&ev, TRUE);
            else if (ev.type == ButtonRelease)
            {
               XUngrabPointer (mainDisplay, CurrentTime);
               done = TRUE;
            }
            else if (ev.type == MotionNotify)
            {
               y = ev.xmotion.y;
               if (!(y<0 && topMsgNumber<=0 || y>msgWindowH &&
                     topMsgNumber>=msgCount) && y != saved_y)
               {
                  saved_y = y;
                  frac = ((double)saved_y) / ((double)msgWindowH);
                  topMsgNumber = max(0,round(msgCount * frac));
                  RedrawMsg ();
               }
               while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
            }
         }
      }
      else if (button_ev->button == Button3)
      {
         if (topMsgNumber == 0) return;

         topMsgNumber--;
         RedrawMsg ();
      }
   }
}

static char	* scanVal = NULL;
static char	* scanSep = NULL;

void InitScan (s, pat)
   char	* s, * pat;
{
   scanVal = s;
   scanSep = pat;
}

static
char * GetString ()
{
   char	* c_ptr;

   while (*scanVal!='\0' && strchr(scanSep,*scanVal)!=NULL) *scanVal++ = '\0';
   if (*scanVal == '\0') return (NULL);

   c_ptr=scanVal;
   for ( ; *scanVal!='\0' && strchr(scanSep,*scanVal)==NULL; scanVal++) ;
   if (*scanVal != '\0') *scanVal++ = '\0';

   return (c_ptr);
}

int ScanValue (fmt, v, item, stype)
   char	* fmt, * v, * item, * stype;
{
   char	* c_ptr, msg[MAXPATHLENGTH];

#ifdef DEBUG
   printf ("get %s for %s from %s:  ", item, stype, scanVal);
#endif

   if ((c_ptr = GetString()) == NULL)
   {

#ifdef DEBUG
      printf ("nothing\n");
#endif

      (void) sprintf (msg, "%s, %d:  Missing %s in %s",
            scanFileName, scanLineNum, item, stype);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      return (INVALID);
   }

   if (sscanf (c_ptr, fmt, v) != 1)
   {

#ifdef DEBUG
      printf ("error in %s\n", c_ptr);
#endif

      (void) sprintf (msg, "%s, %d:  Bad %s in %s [%s]",
            scanFileName, scanLineNum, item, stype, c_ptr);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      return (INVALID);
   }

#ifdef DEBUG
      printf ("got %d\n", (*(int *)v));
#endif

   return (0);
}

#define MAXEMERGENCYCOUNT 5

static int	emergencyCount = 0;

void EmergencySave (sig)
   int	sig;
{
   switch (sig)
   {
      case SIGHUP:
         fprintf(stderr, "SIGHUP signal received!  %s aborted!\n", TOOL_NAME);
         break;
      case SIGFPE:
         fprintf(stderr, "SIGFPE signal received!  %s aborted!\n", TOOL_NAME);
         break;
      case SIGBUS:
         fprintf(stderr, "SIGBUS signal received!  %s aborted!\n", TOOL_NAME);
         break;
      case SIGSEGV:
         fprintf(stderr, "SIGSEGV signal received!  %s aborted!\n", TOOL_NAME);
         break;
   }

   if (++emergencyCount > MAXEMERGENCYCOUNT)
   {
      fprintf (stderr, "Error count exceeds %1d in %s!  %s aborted!\n",
            MAXEMERGENCYCOUNT, "EmergencySave()", TOOL_NAME);
      exit (-1);
   }

   if (exitNormally) return;

   signal (SIGHUP, SIG_DFL);
   signal (SIGFPE, SIG_DFL);
   signal (SIGBUS, SIG_DFL);
   signal (SIGSEGV, SIG_DFL);

   if (fileModified)
   {
      switch (SaveTmpFile ("EmergencySave"))
      {
         case OBJ_FILE_SAVED:
            fprintf (stderr, "Saved to EmergencySave.%s.\n", OBJ_FILE_EXT);
            break;
         case SYM_FILE_SAVED:
            fprintf (stderr, "Saved to EmergencySave.%s.\n", SYM_FILE_EXT);
            break;
         case INVALID:
            fprintf (stderr, "Unable to save working file.\n"); break;
      }
   }
   exitNormally = TRUE;
   exit (0);
}

static
int EmergencySaveForXCont (s)
   char	* s;
{
   if (++emergencyCount > MAXEMERGENCYCOUNT)
   {
      fprintf (stderr, "Error count exceeds %1d in %s!  %s aborted!\n",
            MAXEMERGENCYCOUNT, s, TOOL_NAME);
      exit (-1);
   }

   if (exitNormally) return (0);

   if (copyingToCutBuffer)
   {
      copyingToCutBuffer = INVALID;
      return (0);
   }

   signal (SIGHUP, SIG_DFL);
   signal (SIGFPE, SIG_DFL);
   signal (SIGBUS, SIG_DFL);
   signal (SIGSEGV, SIG_DFL);

   if (fileModified)
   {
      switch (SaveTmpFile ("EmergencySave"))
      {
         case OBJ_FILE_SAVED:
            fprintf (stderr, "Saved to EmergencySave.%s.\n", OBJ_FILE_EXT);
            break;
         case SYM_FILE_SAVED:
            fprintf (stderr, "Saved to EmergencySave.%s.\n", SYM_FILE_EXT);
            break;
         case INVALID:
            fprintf (stderr, "Unable to save working file.\n");
            break;
      }
   }
   exitNormally = TRUE;
   return (-1);
}

int EmergencySaveForX (dsp, ev)
   Display	* dsp;
   XErrorEvent	* ev;
{
   char	msg[MAXSTRING+1];

   XGetErrorText (mainDisplay, (int)(ev->error_code), msg, MAXSTRING);
   fprintf (stderr, "X ERROR:  %s.\n", msg);

   return (EmergencySaveForXCont ("EmergencySaveForX()"));
}

int IOEmergencySaveForX (dsp)
   Display	* dsp;
{
   return (EmergencySaveForXCont ("IOEmergencySaveForX()"));
}

void Error (Where, Message)
   char	* Where, * Message;
{
   fprintf (stderr, "Error in %s.  %s.\n", Where, Message);
   fprintf (stderr, "%s aborted!\n", TOOL_NAME);
   if (fileModified) EmergencySave (0);
   exit (-1);
}

void Usage (tool_name)
   char	* tool_name;
{
   fprintf (stderr, "Usage:\t%s\n", tool_name);
   fprintf (stderr, "\t[-display displayname]\n");
   fprintf (stderr, "\t[-fg <color>]\n");
   fprintf (stderr, "\t[-bg <color>]\n");
   fprintf (stderr, "\t[-bd <color>]\n");
   fprintf (stderr, "\t[-rv] [-nv]\n");
   fprintf (stderr, "\t[-bw]\n");
   fprintf (stderr, "\t[-geometry geom] [=geom]\n");
#ifdef XI18N
   fprintf (stderr, "\t[-warning]\n");
#endif
   fprintf (stderr, "\t[filename[.obj]]\n");
   exit (-1);
}
