/*
 * Copyright 1990 by Tektronix, Inc., Beaverton, Oregon, and
 * the Massachusetts Institute of Technology, Cambridge, Massachusetts.
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of Tektronix or M.I.T. not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  Tektronix and M.I.T. make no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *
 * TEKTRONIX AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL TEKTRONIX OR M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef LINT
#ifdef RCS_ID
static char *rcsid=  "$Header: /n/lacey/usr/local/src/video/MVEX/server/ddx/rasterops/RCS/roMvexInit.c,v 1.10 1991/09/26 21:22:48 toddb Exp $";
#endif /* RCS_ID */
#endif /* LINT   */

#include "X.h"
#include "scrnintstr.h"
#include "misc.h"
#include "MVEX.h"
#include "MVEXproto.h"
#include "videostr.h"
#include "vmi.h"
#include "roStruct.h"
#include "roMvex.h"

/*
 * Fraction ranges for hue, saturation, brightness and contrast.
 * It turns out that the adjustment for all of these is the same.
 * So we only declare one structure.
 */
#define AdjustmentValues						\
{									\
    0, 1, 63,		/* numerator: base, increment, limit */		\
    63,1, 63,		/* denominator: base, increment, limit */	\
    MVEXLinearRange, MVEXLinearRange					\
}

static xFractionRange RoAdjustments[] = {
    AdjustmentValues,	/* saturation */
    AdjustmentValues,	/* contrast */
    AdjustmentValues,	/* hue */
    AdjustmentValues,	/* bright */
};
#undef AdjustmentValues

/*
 * ascii port names to be interned as atoms.
 * This list is actually bogus in that it includes both ntsc and pal: in
 * 99.8% of all situations, a site will use one or the other.  Period.
 * But for now...
 */
static char *RoPortNames[] = {
    "ntsc-composite",
    "ntsc-svhs",
    "ntsc-rgb",
};

/*
 * device dependent values to map atoms to a device value.
 * In this case we just use the port in the lower word.  The 3.0
 * version of the RasterOps driver does not allow the timing to
 * be set, hence we assume it is ntsc.
 */
int RoPortDevices[] = {
    PIPIO_COMPOSITE,
    PIPIO_YC,
    PIPIO_RGB,
};
#define NumberPorts (sizeof(RoPortDevices) / sizeof(RoPortDevices[0]))

/*
 * These are initialized by the vmi layer.
 */
static Atom RoPortAtoms[ NumberPorts ];

/*
 * all are false, so don't initialize
 */
static Setup RoSetupValues[ NumberPorts ];

/*
 * Protocol clip size data
 */
static xRectangle RoClipSize = {
    0, 0, 1, 1
};

/*
 * Protocol placement data, included below in RoInattrs
 * Basically, we express the entire range of the digitization ability
 * which are two concentric boxes that look like
 *
 *    -------------------
 *    |(-102,-16)       |
 *    |  ------------   |
 *    |  |(0,0)     |   |
 *    |  |          |   |
 *    |  |          |   |
 *    |  |          |   |
 *    |  ------------   |
 *    |                 |
 *    -------------------
 *
 * The extent of the outer box is the limit of what we can
 * digitize from; the inner box is the reccommended origin.
 * Hence if the user digitizes from 0,0 using the (protocol) signal
 * width and height a good picture is pretty likely.
 *
 * According to Michael Maeta at RasterOps who designed the board,
 * The available x source can be no more than about 700 pixels selected
 * from 0-779 (780 wide); the y source can be no more than about 500
 * selected from 2-524 (525 lines ala NTSC).  The y source value must
 * be >= 2 and must be even; the height must also be even.
 */
static xPlacement RoPlacements[] = {
    {   /* first (and only) xPlacement */
        {30, 1},			/* 30 frames per second */
        {       /* src xPlacement */
            {-RO_PICT_X, -RO_PICT_Y+RO_MIN_Y, 1, 2},/* base rectangle */
            {RO_SIG_W - RO_PICT_X - 1,
	     RO_SIG_H - RO_PICT_Y - 1,
	     RO_MAX_SIG_W, RO_MAX_SIG_H},/* limit rectangle */
            1,2,			/* xInc, yInc */
            1,2,			/* widthInc, heightInc */
            MVEXLinearRange
        },
        {       /* dst xPlacement */
            {-RO_SIG_W+1, -RO_SIG_H+1, 1, 1},	/* base rectangle */
            {0, 0, RO_SIG_W, RO_SIG_H},	/* limit rectangle */
            1,1,			/* xInc, yInc */
            1,1,			/* widthInc, heightInc */
            MVEXLinearRange
        },
        {       /* xScale xFractionRange -- arbitrary scale (but only down) */
            1, 1, RO_SIG_W,		/* dst base, increment, limit */
            RO_SIG_W, 1, RO_SIG_W,	/* src base, increment, limit */
            MVEXLinearRange, MVEXLinearRange
        },
        {       /* yScale xFractionRange -- arbitrary scale (but only down) */
            1, 1, RO_SIG_H,		/* dst base, increment, limit */
            RO_SIG_H, 1, RO_SIG_H,	/* src base, increment, limit */
            MVEXLinearRange, MVEXLinearRange
        },
	FALSE /* x and y scale do not have to be identical */
    }
};

/*
 * Protocol geometry data
 */
static VideoGeometry RoInattrs= {
    {	/* xVideoGeometry */
	{2997, 100},	/* 30 frames per second */
	{5994, 100},	/* (59.94) 60 fields per second */
	RO_PICT_W,
	RO_PICT_H,	/* width, height of input (NTSC) signal */
	1,		/* Concurrent use */
	2,		/* Priority step */
	0,		/* Reference id (created at init time) */
	1		/* nPlacement (0 ==> we accept any placement) */
    },
    RoPlacements,	/* if nPlacement>0 ==> pointer to list of xPlacement */
};

/*
 * Protocol ability data
 */
static VideoAbility RoAbility = {
    {
	/* xVideoAbility */
	{ 32, 63 },	/* normal saturation: 32/63 =~ .5 */
	{ 32, 63 },	/* normal contrast:   ~.5 */
	{ 32, 63 },	/* normal hue:        ~.5 */
	{ 32, 63 },	/* normal bright:     ~.5 */
	1, 1, 1, 1,	/* 1 each of sat., contrast, hue and bright */
    },
    4,	/* helper: total count of sat., contrast, hue and bright */
    RoAdjustments,	/* if any adjustments, then pointer to list */
			/* of xFractionRange */
};

/*
 * Protocol port data
 */
static VioPortList RoPortList = {
    NumberPorts,	/* nPorts */
    RoPortAtoms,	/* ports (filled in at run time) */
    RoPortNames,	/* portNames */
    RoPortDevices,	/* devices */
    RoSetupValues,	/* setup values */
};
    
/*
 *    NAME
 *        roInit - External call for Toucan initialization.
 *
 *    SYNOPSIS
 */
/*ARGSUSED*/
Bool
roTcpInit(pScreen, pInitVideoInfo, haveRo)
    ScreenPtr pScreen;			/* in: X screen */
    VideoScreenPtr pInitVideoInfo;	/* in/out: MVEX video screen */
    Bool *haveRo;			/* out: TRUE if tc/pip present */
/*
 *    DESCRIPTION
 *        Entry point to Toucan initialization by MVEX.
 *        Do all video input device initialization
 *        Add depths/visuals to the VideoScreenRec
 *
 *    RETURNS
 *        TRUE if memory allocation does not prevent initialization,
 *        FALSE otherwise
 */
{
    RoScreenPrivatePtr roInfo;

    if ((RasterOpsScreenMask & 1<<pScreen->myNum) == 0
      || RasterOpsScreenIndex < 0
      || ((roInfo = RoGetScreenPrivate(pScreen)) == (RoScreenPrivatePtr) NULL)
      || roInfo->vidEnablePix == NULL) {
	*haveRo = FALSE;
	return(TRUE);
    }

    *haveRo = TRUE;

    /*
     * RasterOps VideoIn adds no new depths/visuals for video.
     * If it did you would do it here.
     */

    /*
     * That's it
     */
    return(TRUE);
}


/*
 *    NAME
 *        roDataInit - MVEX data initialization for RasterOps tc/pip
 *
 *    SYNOPSIS
 */
Bool
roTcpDataInit(pScreen, pInitVideoInfo)
    ScreenPtr pScreen;			/* in: X screen */
    VideoScreenPtr pInitVideoInfo;	/* in/out: MVEX video screen */
/*
 *    DESCRIPTION
 *        Initialize MVEX data structures other than depth/visuals.
 *        This routine will only be called if the tc/pip exists.
 *        The depths/visuals would have been added to the VideoScreenRec
 *        in roInit.  This routine does everything else.
 *
 *    RETURNS
 *        TRUE if memory allocation does not prevent initialization,
 *        FALSE otherwise
 *
 */
{
    VioDataPtr pvioData;
    int i,j,k;
    int nvid;
    DepthPtr pDepth;
    VisualPtr pVisual;
    VideoVisual *pAllowedDepth;
    VideoElementPtr pvElement;
    RoScreenPrivatePtr roInfo;

    /*
     * Fill in the destination rectangle maximum width and height from
     * the screen private info.
     */
    roInfo = RoGetScreenPrivate(pScreen);
    for (i = RoInattrs.xvg.nPlacement-1; i >= 0; i--) {
	RoInattrs.pPlacements[ i ].dest.limit.x = roInfo->cg8.width - 1;
	RoInattrs.pPlacements[ i ].dest.limit.y = roInfo->cg8.height - 1;
    }

    /*
     * Allocate space for the per-screen data.
     */
    pvElement = (VideoElementPtr) xalloc(sizeof(VideoElement));
    if (!pvElement)
	return FALSE;
    pvioData = (VioDataPtr) xalloc(sizeof(VioDataRec));
    if (!pvioData){
	xfree((char *)pvElement);
	return FALSE;
    }

    /*
     * protocol data
     */
    for (i = 0; i < NumberPorts; i++)
	RoPortAtoms[ i ]
	    = MakeAtom(RoPortNames[ i ], strlen(RoPortNames[ i ]), TRUE);
    pvioData->pattr = &RoInattrs;
    pvioData->pattr->xvg.referenceId = FakeClientID(0);
    pvioData->pAbility = &RoAbility;
    pvioData->pClipSize = &RoClipSize;
    pvioData->pPortList = &RoPortList;

    /*
     * vectors
     */
    pvioData->StopVideo = roStopVideo;
    pvioData->SetupVideo = roSetupVideo;
    pvioData->StartVideo = roStartVideo;
    pvioData->PollVideo = roPollVideo;
    pvioData->QuerySetup = vmiNoOp;		/* for now, it never changes */
    pvioData->RealizeWindow = vmiNoOp;		/* nothing special */
    pvioData->UnrealizeWindow = vmiUnrealizeWindow;
    pvioData->ClipNotify = vmiClipNotify;
    pvioData->GetCompositeClip = vmiGetCfbCompositeClip;
    pvioData->ValidateRequest = roValidateRequest;

    /*
     * polling data
     * Set this to zero if you don't need to be polled.
     */
    pvioData->timeout = 2000;	/* every 2 second */
    pvioData->lastCall = 0;	/* a value to start with */

    /*
     * AllowedDepths:
     *     fill with depths/vids for "video visuals".
     *
     * For tc/pip, we will use all 24-bit visuals as long as they are not
     * PseudoColor or GrayScale.  All rendermodels are marked "not opaque"
     * because tc/pip pixels are RGB (if, for example, it rendered pixels in
     * luminance-chrominance values, then it would be "opaque")..  We support
     * windows for these visuals, but do not support pixmaps for VideoIn
     * because there is not enough off-screen memory (there's only about 10
     * lines.
     */
    nvid = 0;
    pDepth = pScreen->allowedDepths;
    pvioData->pAllowedDepths = (VideoVisual *)NULL;
    for (i = 0; i < pScreen->numDepths; i++, pDepth++) {
	/*
	 * only consider the 24-bit depths
	 */
	if (pDepth->depth != 24)
	    continue;

	/*
	 * for this depth, look through all the visual ids for appropriate
	 * matches.
	 */
	for (j = 0; j < pDepth->numVids; j++) {
	    pVisual = pScreen->visuals;
	    for (k = 0; k < pScreen->numVisuals; k++, pVisual++) {
		if (pVisual->vid != pDepth->vids[ j ])
		    continue;
		if (pVisual->class == PseudoColor
		 || pVisual->class == GrayScale)
		    continue;

		pvioData->pAllowedDepths =
		    (VideoVisual *)xrealloc(pvioData->pAllowedDepths,
					    ++nvid *sizeof(VideoVisual));
		pAllowedDepth = &pvioData->pAllowedDepths[ nvid-1 ];
		pAllowedDepth->xrm.depth = pDepth->depth;
		pAllowedDepth->xrm.opaque = FALSE;
		pAllowedDepth->xrm.visualid = pDepth->vids[j];
		pAllowedDepth->xrm.redMask = pVisual->redMask;
		pAllowedDepth->xrm.greenMask = pVisual->greenMask;
		pAllowedDepth->xrm.blueMask = pVisual->blueMask;
		pAllowedDepth->model = MVEXWindowModelMask;
		break;
	    }
	}
    }

    if (nvid == 0) {
	ErrorF("No appropriate depths for tc/pip\n");
	return FALSE;
    }

    pvioData->nAllowedDepths = nvid;
    pvElement->type = Input;
    pvElement->id = pvioData->pattr->xvg.referenceId;
    pvElement->pvio = pvioData;
    pvElement->Reset = roReset;
    pvElement->devPrivate = (pointer) NULL;
    /*
     * Insert pvElement at head of the list, just cuz that's the easiest thing
     * to do.
     */
    pvElement->next = pInitVideoInfo->vElements;
    pInitVideoInfo->vElements = pvElement;
	
    pInitVideoInfo->inputOverlap = TRUE;
    pInitVideoInfo->ioOverlap = TRUE;

    return(TRUE);
}

/*
 *    NAME
 *        roReset - MVEX Reset routine for tc/pip
 *
 *    SYNOPSIS
 */
static void
roReset(pvElement)
    VideoElementPtr pvElement;  /* in/out: tc/pip vin data */
/*
 *    DESCRIPTION
 *        Delete any dynamically allocated memory on the vin data structure.
 *        Perform Toucan device-specific reset if needed (if not STAND_ALONE).
 *
 *    RETURNS
 *        void
 *
 */
{
    if (pvElement->pvio->pAllowedDepths)
	xfree(pvElement->pvio->pAllowedDepths);
}
