/*
 * Copyright IBM Corporation 1991
 *
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that 
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of IBM not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * IBM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * IBM BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
*/

#ifndef PLIPROTO_H
#define PLIPROTO_H
 
 
#include "PLIsstk.h"  /* typedef for strokes */
 
/* EXTENSION PROTOCOL FOR PAPER-LIKE INTERFACE (PLI) */
 
/* official name of extension */
#define XPLI_NAME "xpli"

/*
  The single event generated by this extension is the strokeEvent
  defined below. Because the event data structure in the X11 protocol
  is fixed in size at 32 bytes, the XY data from the tablet cannot be
  returned in the event, but rather must be queried using the
  strokeID value which is returned in the event. Some helpful
  information is returned in the event, namely, the size of the
  stroke in XY samples, the starting and ending points, and the
  extremes of the bounding box. All coordinates are relative to the
  eventWin window, and all coordinates are reported in pixel units
  (c.f. PLIStrokeXYReply below). When a stroke lies partly outside of
  a window, some coordinates may be negative
  and other coordinates may be greater than the window extremes.
 
  The initial stroke event that appears when the server is brought up
  will be routed to the window containing the starting point of the stroke.
  All strokes thereafter will be routed to the same window, until
  (a) the window is unmapped or the client connection is broken, or
  (b) the client rejects the stroke using the PLIRedirectStrokeReq
  request. In the latter cases, the stroke will be routed to the
  window containing the starting point of the stroke.
 
  Strokes are only routed to windows/clients which have expressed
  a willingness to receive stroke events. The client must indicate this
  interest by acquiring the event flag for strokes from the server
  and setting this flag for each window in which the client wishes to
  receive strokes.
 
  If a stroke cannot be routed to any window, it is discarded and the
  ink is deleted from the screen. This should probably be accompanied
  by a distinguished "BEEP".
 
  If a window containing a stroke selects for mouse events and
  not stroke events, then mouse emulation will be done in the
  extension, and the client will not see any stroke events.
 
  The actual XYs of a stroke are obtained by the PLIManageStrokeXYReq which
  contains the strokeID of the desired stroke. The stroke XYs are
  returned in a variable length message headed by the PLIManageStrokeXYReqReply
  structure. The coordinates are scaled window coordinates. See plicoord.h
  for functions to manage these coordinates. These coordinates are
  also window relative, which means that if the stroke lies partly
  outside of the window, some coordinates may be negative and others
  may be greater than the window extremes.
 
  Current displays may exceed 1024 pixels in height or width, therefore
  at least 12 bits must be allocated to hold the integer part of the
  scaled coordinates. The dynamic range of current tablets is also
  about 12 bits. Therefore 15 bits should be sufficient to hold the
  magnitude for coordinates, leaving the 16th bit for the sign.
  The scale factor for coordinates must be adjusted, however, to
  reflect the ratio of display range to tablet range. For example,
  when the tablet range is 0..2047, and the display range is 0..1023,
  the appropriate scale factor is 2**1. For the VGA, a 640x400 device,
  the dynamic range is approximately 0..512, and the tablet dynamic
  range is 0..2047, so the scale factor might be 2**2, which makes
  the range of the VGA display 2560x1600. Now the tablet may report
  in a coordinate system of equal intervals in horizontal and vertical,
  and if the tablet is rectangular, the full resolution of the tablet
  will not be realized along one axis. In the previous case, if the
  tablet reports are 2047x1350, the scale factor of 2**2 will be
  adequate. On the other hand, if the tablet coordinate system
  is not equally spaced, the tablet reports might be 2047x2047, with
  greater resolution along one axis. In this case, one might elect
  to use 2**3 as the scale factor so that the dynamic range of the
  scaled display coordinates always exceed that of the tablet.
  By the way, the alignment process will result in conversion equations
  which map the tablet coordinates into display coordinates, so that
  if the display resolution is equal in horizontal and vertical,
  the scaled coordinates will also be of equal horizontal and
  vertical resolution.
 
  This lengthy explanation is provided because in the present protocol,
  the dynamic range of scaled display coordinates MAY NOT EXCEED
  15 BITS. When implementing a server extension, one must carefully
  choose a scale factor that meets this restriction. We envision
  that future tablets may have a dynamic range in excess of 2**15,
  and a new protocol will have to be developed at that point.
 
*/
 
/*
   ERRORS - there are no asynchronous errors in this protocol.
   Where there is a device problem, it is not clear that the clients
   would be able to do anything about it. The remaining errors
   are synchronous, i.e. they are associated with particular
   requests. Immediately preceeding each request is a list of
   the error conditions which may result.
*/
 
 
/************ xStrokeEvent *********************/
 
typedef struct {
        BYTE type;
        BYTE detail;
           /* 00X - motion report (no stroke) */
           /* 01X - start of stroke report */
           /* 02X - in-stroke motion report */
           /* 03X - end of stroke report */
	   /* 04X - stroke deleted by server */
           /* 05X - 07X unused and reserved codes */
	   /* 08X - unused and reserved flag */
	   /* 3-X - entire incomplete stroke in window */
	   /* 7-X - entire stroke in window */
           /* 8-X - entire stroke outside of window */
           /* 9-X - some in, some out, start out, end out */
           /* B-X - some in, some out, start in,  end out */
           /* D-X - some in, some out, start out, end in  */
           /* F-X - some in, some out, start in,  end in  */
           /* all other codes in upper nibble are invalid */
        CARD16 sequenceNumber B16;
        Time time B32;
        Window eventWin B32;
        CARD16 strokeID B16;
        INT16 numXYs B16;
        INT16 startX B16, startY B16; /* xys are pixels, window relative */
        INT16 endX B16, endY B16;
        INT16 minX B16, minY B16;
        INT16 maxX B16, maxY B16;
      } xStrokeEvent;

/* detail type codes, must be masked by DET_TYPE_MASK to compare */
#define DET_MOTION     0x0
#define DET_START      0x1
#define DET_STROKE     0x2
#define DET_END        0x3
#define DET_WITHDRAWN  0x4 
/* mask to extract detail type codes */
#define DET_TYPE_MASK  0x7
/* detail inout flags */
#define DET_IN         0x10
#define DET_OUT        0x80
#define DET_STRT_IN    0x20
#define DET_END_IN     0x40
/* mask to extract inout flags */
#define DET_INOUT_MASK 0xF0
/* legal inout detail codes, must be masked by DET_INOUT_MASK to compare  */
   /* entire stroke in window */
#define DET_ALL_IN     0x70
   /* incomplete stroke all in window */
#define DET_ALL_IN_INC 0x30
   /* entire stroke outside of window */
#define DET_ALL_OUT    0x80
   /* some in, some out, start and end out */
#define DET_INOUT_SEOUT   0x90
   /* some in, some out, start out, end in */
#define DET_INOUT_SOUT    0xD0
#define DET_INOUT_EIN     0xD0
   /* some in, some out, start in, end out */
#define DET_INOUT_EOUT    0xB0
#define DET_INOUT_SIN     0xB0
   /* some in, some out, start in, end in */
#define DET_INOUT_SEIN    0xF0
 
/************ xPLIGrabStrokeReq ****************/
/************ xPLIUngrabStrokeReq **************/
/* Nonexclusive grab allows a window to see strokes that fall entirely
   outside its borders. Exclusive grab routes all strokes only to
   windows on grab list. Grab should be used selectively as grab
   windows occupy a permanent place in the 32 slot window cache
   in the server. */
/* Errors:
      Request - major or minor opcode is incorrect.
      Length  - request is the wrong length.
*/
typedef struct {
        CARD8  reqType;   /* request code for PLI extension */
        CARD8  extReqType;  /* X_PLIGrabStroke */
        CARD16 length B16;
	BYTE   grabType;  /* 01X exclusive grab */
	BYTE   pad0;
	CARD16 pad1 B16;
	Window window B32;
      } xPLIGrabStrokeReq;
#define sz_xPLIGrabStrokeReq sizeof(xPLIGrabStrokeReq)
 
 
 
/********** xPLIEnqueueStrokeReq ************/
 
/*
  the following request is to be used for Xlib, Xt integration test.
  The stroke is submitted to stroke dispatch as if it came from the
  tablet.
*/
/* Errors:
      Request - major or minor opcode is incorrect.
      Length  - request is the wrong length.
      Implementation - request not yet implemented in server.
*/
/* 
 * note that the Stroke Type is included in xPLIStroke and therefore does not 
 * have to be a field in the request
 */
typedef struct {
        CARD8  reqType;     /* request code for xpli extension */
        CARD8  extReqType;  /* X_PLIEnqueueStroke */
        CARD16 length B16;
        xPLIStroke stroke;
      } xPLIEnqueueStrokeReq;
/* the xys follow the fixed size part of this struct */
#define sz_xPLIEnqueueStrokeReq sizeof(xPLIEnqueueStrokeReq)
 
/********* xPLIConfigReq *******************/
 
/* Errors:
      Request - major or minor opcode is incorrect.
      Length  - request is the wrong length.
*/
typedef struct {
        CARD8  type;  /* X_Reply */
        BYTE   pad0;
        CARD16 sequenceNumber B16;
        CARD32 length B32;
        CARD32 eventMask B32;/* window mask for enabling xpli events */
        CARD32 digtype B32;  /* digitizer characteristics */
        CARD16 scalefac B16; /* coordinate scale factor being used */
        CARD16 sampsec B16;  /* samples per second */
	CARD16 resolX B16, resolY B16; /* scaled coords per cm */
	CARD32 resolTimeSec B32; /* accuracy of stroke timestamp */
	CARD32 resolTimeNsec B32;
	CARD32 pad8 B32;
      } xPLIConfigReply;

   /* digitizer reports proximity */
#define DIG_TYPE_PROX    0x00000001
   /* digitizer reports pressure */
#define DIG_TYPE_PRESS   0x00000002
   /* digitiser reports tilt (azimuth or xy skew) */
#define DIG_TYPE_TILT    0x00000004
#define DIG_TYPE_XYTILT  0x00000008
   /* digitizer has microswitch for contact detection */
#define DIG_TYPE_USW     0x00000010
 
typedef struct {
        CARD8  reqType;   /* request code for xpli extension */
        CARD8  extReqType;  /* PLIConfigReq */
        CARD16 length B16;
      } xPLIConfigReq;
#define sz_xPLIConfigReq sizeof(xPLIConfigReq)
 
 
/*  NOTE: can get display parameters via the display macros
    provided in the standard Xlib interface */
 
/********* xPLIAlignReq *******************/
/********* xPLIAlignAbortReq **************/
 
/* Errors:
      Request - major or minor opcode is incorrect.
      Length  - request is the wrong length.
      Access - PLIState machine not in correct state
*/
typedef struct {
        CARD8  reqType;   /* request code for xpli extension */
        CARD8  extReqType;  /* X_PLIAlign, X_PLIAlignAbort */
        CARD16 length B16;
      } xPLIAlignReq;
#define sz_xPLIAlignReq sizeof(xPLIAlignReq)

/************* xPLIManageStrokeXYReq ***********/
/* though not presently implemented, for digitizers that return
   pressure, tilt, etc., the numXYs values will follow all of the
   XY values that follow this structure, and the reply length
   adjusted accordingly. Present technology pretty much limits
   pressure and tilt values to 8 bits - so a likely format
   will use 4 bytes for each such sample, one for pressure, one
   for x tilt, one for y tilt, and one for whatever. */
typedef struct {
        CARD8  type;  /* X_Reply */
        BYTE   pad0;
        CARD16 sequenceNumber B16;
        CARD32 length B32;
	Window window B32; /* which window stroke data is relative to */
	/* the protocol supports the return of partial strokes.
	   the following elements define the span of the returned
	   XYs in the stroke. For a full stoke of n XYs, startIndex == 0
	   and endIndex == n. For stroke XYs k thru l (k>=l),
	   startIndex == k and endIndex == l+1. */
	CARD16 startIndex B16;
	CARD16 endIndex B16;
	CARD32 pad5 B32;
	CARD32 pad6 B32;
	CARD32 pad7 B32;
	CARD32 pad8 B32;
	xPLIStroke stroke;
      } xPLIManageStrokeXYReply;
 
/* Errors:
      Request - major or minor opcode is incorrect.
      Value   - range requested for reply is incorrect
      Length  - request is the wrong length.
      IDChoice - stroke id unknown to server
*/
typedef struct {
        CARD8  reqType;  /* request code for xpli extension */
        CARD8  extReqType;  /* X_PLIManageStrokeXY */
        CARD16 length B16;
	Window window B32; /* which window the stroke data should be */
	                  /* relative to */
        CARD16 strokeID B16;
        BYTE   subReq; /* see subreq codes below */
        BYTE   pad3_4;
	CARD16 startIndex B16, endIndex B16; /* range of XYs to reply */
	                             /* 0:0 - reply all */
	                             /* n:0 - reply all n to end */
	                             /* n:m - reply all n to m-1 */
                                     /* 0:m - reply all 0 to m-1 */
      } xPLIManageStrokeXYReq;
#define sz_xPLIManageStrokeXYReq sizeof(xPLIManageStrokeXYReq)
 
/* xPLIManageStrokeXYReq subrequest codes */
   /* RetainStroke means stroke stays pending in server for this window */
#define RETAIN_STROKE             0x00
   /* AcceptStroke means stroke deleted if no other pending windows */
#define ACCEPT_STROKE             0x01
   /* RejectStroke means stroke deleted if no other pending windows
      stroke may be turned into mouse events if no window accepts */
#define REJECT_STROKE             0x02
   /* WithdrawStroke means reject and send notification to window
      this is used to force stroke confirmation where the client is
      misbehaving */
#define WITHDRAW_STROKE           0x03
#define ACK_MASK_STROKE           0x03
   /* Copy flag may be used in conjunction, causes stroke reply to
      be generated with position data relative to requested window */
#define COPY_STROKE               0x04
#define ACCEPT_AND_COPY_STROKE    0x05
#define REJECT_AND_COPY_STROKE    0x06
#define WITHDRAW_AND_COPY_STROKE  0x07
 
/************ xPLIDrawStrokeReq ************/
/* Function:
      This request can be used to draw a stroke. It is identical to
      the PolyLine request except that the XYs are scaled coordinates
      not pixel coordinates. This is useful for clients that need
      to keep the full resolution strokes (e.g. for recognition)
      but don't want to keep a descaled copy just for drawing.
      The descaling is done in the server. */
/* Errors:
      Request - major or minor opcode is incorrect.
      IDChoice - stroke id unknown to server.
      Drawable - the drawable ID is bad.
      GContext - the gcontext ID is bad.
      Match - the gcontext does not match the drawable
*/
typedef struct {
    CARD8 	reqType;  /* xpli request code */
    CARD8 	extReqType; /* X_PLIDrawStroke */
    BYTE 	coordMode;
    CARD8       pad;
    CARD16 	length B16;
    Drawable 	drawable B32;
    GContext 	gc B32;
} xPLIDrawStrokeReq;
#define sz_xPLIDrawStrokeReq sizeof(xPLIDrawStrokeReq)

/************ xPLIPendingStrokesReq ************/
/* Function:
      If an application misbehaves and does not accept or reject
      a stroke, it will not be deleted from the server queue.
      This request retrieves the pending strokes and their ages
      so that a window manager task can force them to be deleted
      using the DeleteStroke subrequest of PLIManageStrokeXYReq. */
/* Errors:
      Request - major or minor opcode is incorrect.
*/
typedef struct {
        CARD8  type;  /* X_Reply */
        BYTE   pad0;
        CARD16 sequenceNumber B16;
        CARD32 length B32;
	CARD32 pad3 B32;
	CARD32 pad4 B32;
	CARD32 pad5 B32;
	CARD32 pad6 B32;
	CARD32 pad7 B32;
	CARD32 pad8 B32;
      } xPLIPendingStrokesReply;
/* following this structure are length/2 of: */
typedef struct {
        CARD16 strokeID;  /* ID of pending stroke */
        CARD16 age;       /* time in seconds stroke has been queued */
        Window window B32; /* window to which stroke was sent */
      } xPLIPendingStroke;

/* Errors:
      Request - major or minor opcode is incorrect.
      Value   - range requested for reply is incorrect
      Length  - request is the wrong length.
      IDChoice - stroke id unknown to server
*/
typedef struct {
        CARD8  reqType;  /* request code for xpli extension */
        CARD8  extReqType;  /* X_PLIManageStrokeXY */
        CARD16 length B16;
      } xPLIPendingStrokesReq;
#define sz_xPLIPendingStrokesReq sizeof(xPLIPendingStrokesReq)

/*
  Function:
     The server extension performs mouse emulation in two cases:
     (1) the stroke does not enter a stroke aware window;
     (2) the stroke starts in a window that is not receiving
         strokes, and remains motionless for a specified period
	 of time. This mode permits drag and drop via the pen.
     The dragtime value is the elapsed time in milliseconds
     since the start of the stroke to wait before determining
     that the stroke should be turned into mouse events. The
     dragx and dragy values determine the size of the box
     in which the pen must remain during this time interval
     expressed in millimeters. Examine the extension implementation
     and documentation to find out the default values for these
     parameters. Note that the time specification is only as
     accurate as the period in which the digitizer reports
     new positions.
*/
 
#define STROKE_ATTR_DRAGTIME 0x00000001
#define STROKE_ATTR_DRAGX    0x00000002
#define STROKE_ATTR_DRAGY    0x00000004
#define STROKE_ATTR_BUTTON   0x00000008

typedef struct {
        CARD8  reqType;  /* request code for xpli extension */
        CARD8  extReqType;  /* X_PLISetStrokeAttr */
        CARD16 length B16;
	CARD32 valueMask B32;
      } xPLISetStrokeAttrReq;
#define sz_xPLISetStrokeAttrReq sizeof(xPLISetStrokeAttrReq)

typedef struct {
        CARD8  type;  /* X_Reply */
        BYTE   pad0;
        CARD16 sequenceNumber B16;
        CARD32 length B32;
	CARD32 valueMask B32;
	CARD32 pad4 B32; /* values start here in r2l order of mask bits */
	CARD32 pad5 B32;
	CARD32 pad6 B32;
	CARD32 pad7 B32;
	CARD32 pad8 B32;
      } xPLIGetStrokeAttrReply;

typedef struct {
        CARD8  reqType;  /* request code for xpli extension */
        CARD8  extReqType;  /* X_PLIGetStrokeAttr */
        CARD16 length B16;
	CARD32 valueMask B32;
      } xPLIGetStrokeAttrReq;
#define sz_xPLIGetStrokeAttrReq sizeof(xPLIGetStrokeAttrReq)

 
#define X_PLIGrabStroke          1
#define X_PLIUngrabStroke        2
#define X_PLIAlign               3
#define X_PLIAlignAbort          4
#define X_PLIEnqueueStroke       5
#define X_PLIConfig              6
#define X_PLIManageStrokeXY      7
#define X_PLIDrawStroke          8
#define X_PLIPendingStrokes      9
#define X_PLISetStrokeAttr      10
#define X_PLIGetStrokeAttr      11
/* this must be last valid extension request + 1 */
#define X_PLILastMinorReq       12

/* 
 * swiped from dix/extension.c - necessary to modify Xt translation tables - 
 */
#define EXTENSION_BASE  128
#define EXTENSION_EVENT_BASE  64
#define LAST_EVENT  128
#define LAST_ERROR 255
#endif
 
 
 
 
