;;;
;;;                    Copyright (C) Shingo NISHIOKA, 1991
;;;                       nishioka@sanken.osaka-u.ac.jp
;;;

(defvar dserver-host-name)
(defvar od:*window-config*)
(defvar od:*body-fillcolumn* nil)
(defvar od:*select-unique-entry-immediately* t)
(defvar od:*scroll-step* 'full)
(defvar od:*eiwa-summary-mode* nil)
(defvar od:*default-jisyo* nil)
(defvar od:*fep-type* 'egg)		;egg, t-code or no-fep

(defvar dserv-protocol "ndtp")

;;; Caution: No variables allowed to be buffer-local
(defvar od:*previous-config* nil)
(defvar od:*dict* nil)
(defvar od:*headers*)

(defvar od:*current-entry* -1)

(defvar od:*input-by-kanji* nil)
(defvar od:*current-jisyo*)
(defvar od:*current-jisyo-pretty-name* "")
(defconst od:*jisyo* '(("eiwa" "eiwa") ("waei" "waei") ("kojien" "kojien")))
(defconst od:*dict-name*
    '(("eiwa" . "$@?71QOBCf<-E5(J")
      ("waei" . "$@?7OB1QCf<-E5(J")
      ("kojien" . "$@9-<-1q(J")))
(defconst od:*is-use-kanji* '((?a . nil) (?A . nil) (?k . t) (?K . t)))
(defconst od:*default-index-type-dict*
    '(("eiwa" . ?a) ("waei" . ?k) ("kojien" . ?k)))
(defvar od:*index-types*)
(defvar od:*default-index-type*)


;;; dict filter

(defvar od:*current-dict-filter-func* nil)

(defconst od:*dict-filter-func*
    '(("eiwa" . (eiwa-reformatter od:filter-buffer eiwa-make-summary))
      ("waei" . (od:filter-buffer))
      ("kojien" . (od:filter-buffer))))

;; eiwa
;;
;; "$@!%(J [1|2|3|4|5|6|7|8|9] " 
;; "\($@!%(J\) \([1-9]a? \)" -> "\1\n\n\2"
;;       $@A43Q%T%j%*%I(J, $@H>3Q%9%Z!<%9(J, $@H>3Q?t;z(J, $@H>3Q%9%Z!<%9$N%7!<(J
;;       $@%1%s%9$,$"$C$?$i?t;z$NA0$G2~9T#28D(J. 
;;       $@?t;z$N8e$K(J"a"$@$,F~$k$3$H$,$"$k(J. 
;; 
;; "$@!%(J [b|c|d|e|f|...] "
;; "\($@!%(J\) \([b-z] \)" -> "\1\n\2"
;;       $@A43Q%T%j%*%I(J, $@H>3Q%9%Z!<%9(J, a$@0J30$N<c$$H>3Q1Q;z(J, $@H>3Q(J
;;       $@%9%Z!<%9$N%7!<%1%s%9$,$"$C$?$iH>3Q1Q;z$NA0$G2~9T#18D(J. 

;; " $@!](J " -> "\n $@!](J "

(defvar od:*dict-filter*
    '(("eiwa" . (("\\($@!%(J\\) \\([1-9][0-9]?[a(]? \\)" . "\\1\n\n  \\2")
		 ("\\($@!%(J\\) \\([bcdefg] \\)" . "\\1\n  \\2")
		 ("\015\005" . "\n  ")
		 (" \\($@!](J[\\[ ]\\)" . "\n\n\n \\1")))
      ("waei" . (("\015\005" . "\n\n  ")
		 ("$@!?(J " . "\n $@!?(J ")
		 ("$@"y(J" . "\n$@"y(J")
		 ("$@!%!?(J" . "$@!%(J\n $@!?(J")))
      ("kojien" . (("\015\005" . "\n\n  ")
		   ("\\(([1-9][0-9]*)\\)" . "\n \\1")))))

(defvar od:*current-dict-filter* nil)

(defun od:filter-buffer ()
  (let ((tmp od:*current-dict-filter*))
    (while tmp
      (goto-char 1)
      (while (re-search-forward (car (car tmp)) nil t nil)
	(replace-match (cdr (car tmp)) t nil))
      (setq tmp (cdr tmp)))))

(defun eiwa-reformatter ()
  (goto-char 1)
  (if (re-search-forward "\\[[^]]*\\]" nil t nil)
      (insert "\n\n")))

(defun eiwa-make-summary ()
  (goto-char 1)
  (if od:*eiwa-summary-mode*
      (while (re-search-forward "$@!'(J" nil t nil)
	(backward-char 1)
	(kill-line))))



(defvar odic-mode-syntax-table nil
  "Syntax table used while in odic mode.")

(defvar odic-mode-abbrev-table nil
  "Abbrev table used while in odic mode.")
(define-abbrev-table 'odic-mode-abbrev-table ())

(if odic-mode-syntax-table
    ()
  (setq odic-mode-syntax-table (make-syntax-table)))

(defvar odic-mode-map nil "")
(if odic-mode-map
    ()
  (setq odic-mode-map (make-sparse-keymap))
  (define-key odic-mode-map "f" 'od:lookup-pattern)
  (define-key odic-mode-map "i" 'od:lookup-pattern-with-ispell)
  (define-key odic-mode-map "." 'od:current-entry)
  (define-key odic-mode-map " " 'od:scroll-entry-up)
  (define-key odic-mode-map "\177" 'od:scroll-entry-down)
  (define-key odic-mode-map "\C-d" 'od:scroll-entry-up-half)
  (define-key odic-mode-map "\C-u" 'od:scroll-entry-down-half)
  (define-key odic-mode-map "h" 'od:show-headers)
  (define-key odic-mode-map "p" 'od:previous-entry)
  (define-key odic-mode-map "n" 'od:next-entry)
  (define-key odic-mode-map "?" 'od:help)
  (define-key odic-mode-map "q" 'od:quit)
  (define-key odic-mode-map "Q" 'od:really-quit)
  (define-key odic-mode-map "R" 'od:really-restart)
  (define-key odic-mode-map "s" 'od:select-entry)
  (define-key odic-mode-map "C" 'od:change-dictionary)
  (define-key odic-mode-map "E" 'od:toggle-eiwa-summary-mode)
  (define-key odic-mode-map "H" 'od:show-hanrei)
  (define-key odic-mode-map "O" 'od:show-okuduke)
  (define-key odic-mode-map "1" 'od:direct-select-entry)
  (define-key odic-mode-map "2" 'od:direct-select-entry)
  (define-key odic-mode-map "3" 'od:direct-select-entry)
  (define-key odic-mode-map "4" 'od:direct-select-entry)
  (define-key odic-mode-map "5" 'od:direct-select-entry)
  (define-key odic-mode-map "6" 'od:direct-select-entry)
  (define-key odic-mode-map "7" 'od:direct-select-entry)
  (define-key odic-mode-map "8" 'od:direct-select-entry)
  (define-key odic-mode-map "9" 'od:direct-select-entry))

(defun od:set-mode-line-format ()
  (make-variable-buffer-local 'mode-line-format)
  (let ((dic-mode-line-format
	 '(""
	   mode-line-modified
	   mode-line-nemacs-header
	   " " od:*current-jisyo-pretty-name*
	   "   " global-mode-string "  %[("
	   (kanji-flag ("%c" ":")) mode-name minor-mode-alist "%n"
	   mode-line-process ")%]--" (-3 . "%p") "-%-")))
    (cond ((eql od:*fep-type* 'egg)
	   (setq mode-line-format
		 (cons '(kanji-flag
			 ((minibuffer-window-selected
			   (display-minibuffer-mode "m" " ")
			   " ")
			  "["
			  (minibuffer-window-selected
			   (display-minibuffer-mode
			    mode-line-egg-mode-in-minibuffer
			    mode-line-egg-mode)
			   mode-line-egg-mode)
			  "]"))
		       dic-mode-line-format)))
	  ((eql od:*fep-type* 't-code)
	   (if (not (eq 'kanji-flag
			(car (cdr (default-value 'mode-line-format)))))
	       (setq mode-line-format
		     (cons
		      (list 'kanji-flag
			    '("["
			      (tcode-on "T" "-")
			      (tcode-on-in-minibuffer "T" "-")
			      "]"))
		      dic-mode-line-format)))))))

(defun odic-mode ()
  "Major mode for editing odic intended for humans to read.  
Special commands:\\{odic-mode-map}
Turning on odic-mode calls the value of the variable odic-mode-hook,
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-mode-map)
  (setq mode-name "Dict")
  (setq major-mode 'odic-mode)
  (setq local-abbrev-table odic-mode-abbrev-table)
  (set-syntax-table odic-mode-syntax-table)
  (setq buffer-read-only t)
  (make-variable-buffer-local 'goal-column)
  (setq goal-column 4)
  (od:set-mode-line-format)
  (run-hooks 'odic-mode-hook))

(defvar odic-body-mode-syntax-table nil
  "Syntax table used while in odic body mode.")

(defvar odic-body-mode-abbrev-table nil
  "Abbrev table used while in odic body mode.")
(define-abbrev-table 'odic-body-mode-abbrev-table ())

(if odic-body-mode-syntax-table
    ()
  (setq odic-body-mode-syntax-table (make-syntax-table)))

(defvar odic-body-mode-map nil "")
(if odic-body-mode-map
    ()
  (setq odic-body-mode-map (make-sparse-keymap))
  (define-key odic-body-mode-map "9" 'od:direct-select-entry))

(defun odic-body-mode ()
  "Major mode for editing odic intended for humans to read.  
Special commands:\\{odic-body-mode-map}
Turning on odic-body-mode calls the value of the variable odic-body-mode-hook,
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map odic-body-mode-map)
  (setq mode-name "DictBody")
  (setq major-mode 'odic-body-mode)
  (setq local-abbrev-table odic-body-mode-abbrev-table)
  (set-syntax-table odic-body-mode-syntax-table)
  (setq buffer-read-only t)
  (make-variable-buffer-local 'fill-column)
  (setq fill-column (or od:*body-fillcolumn* 
			(let ((fillcol (- (window-width) 10)))
			  (if (< 0 fillcol)
			      fillcol))
			fill-column))
  (od:set-mode-line-format)
  (run-hooks 'odic-body-mode-hook))



(defun od:open-dictionary ()
  (if od:*dict*
      ()
      (setq od:*dict* (open-network-stream "*ndtp*" " *ndtp*" 
					   dserver-host-name 
					   dserv-protocol))
      (set-process-kanji-code od:*dict* 3)
      (set-process-sentinel od:*dict* 'od:watchdog)
      (set-buffer " *ndtp*")
      (erase-buffer)
      (process-send-string od:*dict*
			   (format "A%s@%s\n" (getenv "USER") (od:hostname)))
      (let ((ans (od:wait-until '("$A\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
	(if (not (string= "$A\n" ans))
	    (error "Cannot connect to jisyo server")))))

(defun od:wait-until (list proc)
  (catch 'od:wait-until
    (while t
      (let* ((str (buffer-string))
	     (len (length str))
	     (tmp list))
	(while tmp
	  (let* ((item (car tmp))
		 (ilen (length item)))
	    (cond ((< len ilen) nil)
		  ((string= item (substring str (- len ilen)))
		   (throw 'od:wait-until item))))
	  (setq tmp (cdr tmp)))
	(accept-process-output proc)))))

(defun od:setup-windows ()
  (delete-other-windows)
  (switch-to-buffer "*Dict-Header*")
  (switch-to-buffer "*Dict-Body*")
  (split-window (selected-window) od:*window-config*)
  (switch-to-buffer "*Dict-Header*"))

(defun od:kill-active-buffers ()
  (kill-buffer "*Dict-Header*")
  (kill-buffer "*Dict-Body*"))

(defun od:set-header ()
  (od:setup-windows)
  (set-buffer "*Dict-Header*"))

(defun od:set-body ()
  (od:setup-windows)
  (set-buffer "*Dict-Body*"))

(defun od:switch-to-header ()
  (od:setup-windows)
  (switch-to-buffer "*Dict-Header*"))

(defun od:switch-to-body ()
  (od:setup-windows)
  (switch-to-buffer "*Dict-Body*"))

(defun od:watchdog (process event)
  (if (string= "finished\n" event)
      (setq od:*dict* nil)))



(defun online-dictionary ()
  (interactive)
  (let ((is-first (not od:*dict*)))
    (setq od:*eiwa-current-display-mode*
	  (if od:*eiwa-summary-mode* "[Summary]" 
	      ""))
    (setq od:*previous-config* (current-window-configuration))
    (get-buffer-create " *ndtp*")
    (get-buffer-create " *od:hostname*")
    (get-buffer-create " *od:temp*")
    (get-buffer-create "*Dict-Header*")
    (get-buffer-create "*Dict-Body*")
    (od:set-header)
    (odic-mode)
    (od:set-body)
    (odic-body-mode)
    (od:set-header)
    (delete-other-windows)
    (let ((buffer-read-only nil))
      (erase-buffer)
      (od:show-title))
    (od:open-dictionary)
    (if is-first
	(if od:*default-jisyo*
	    (od:really-change-dictionary od:*default-jisyo*)
	    (od:change-dictionary)))
    (od:help)))

(defun od:help ()
  (interactive)
  (with-output-to-temp-buffer " *od:Help*"
    (princ "
                   $@%*%s%i%$%s<-=q%3%^%s%I%j%U%!%l%s%9(J

$@%-!<(J  $@F0:n(J

f     $@8!:w$r9T$J$$$^$9(J. 
      $@F~NO$7$?%Q%?!<%s$K40A4$K0lCW$9$k8uJd$,I=<($5$l$^$9(J.
      $@%Q%?!<%s$OH>3Q%"%k%U%!%Y%C%H(J($@%"%k%U%!%Y%C%H8!:w(J), 
      $@J?2>L>(J, $@JR2>L>(J($@$+$J8!:w(J) $@$GF~NO$7$^$9(J.
      $@$J$*(J, $@$+$J8!:w$,;XDj$5$l$F$$$k;~$K$O(J, $@<+F0E*$K(J
      fep$@$,5/F0$7$?>uBV$K$J$j$^$9(J.
      fep$@$K$O(J egg $@$+(J t-code $@$N$$$:$l$+$,;XDj$G$-$^$9(J.
      $@%Q%?!<%s$O8lF,$^$?$O8lHx$K(J1$@$D$N(J \"*\" $@$rIU$1$k$3$H$,$G$-$^$9(J.
      $@$=$N:](J, \"*\" $@$O(J0$@J8;z0J>e$NG$0U$NJ8;zNs$r0UL#$7$^$9(J.
      $@%Q%?!<%s$OG$0U$N0LCV$K(J \"?\" $@$r$$$/$D$G$b4^$`$3$H$,$G$-$^$9(J.
      $@$=$N:](J, \"?\" $@$OG$0U$N(J1$@J8;z$r0UL#$7$^$9(J.

i     ispell$@$rMQ$$$F$"$d$U$d$JDV$+$iF~NO%Q%?!<%s$rA*$S$^$9(J. ($@1QOB<-=q$N;~JXMx(J)

.     $@8=:_$N8uJd$N@bL@J8$rI=<($7$^$9(J.
p     $@D>A0$N8uJd$N@bL@J8$rI=<($7$^$9(J.
n     $@D>8e$N8uJd$N@bL@J8$rI=<($7$^$9(J.
s     $@8uJd$rHV9f$GA*$S(J, $@$=$N@bL@J8$rI=<($7$^$9(J.
1-9   1-9$@HV$N8uJd$rD>@\A*$S(J, $@$=$N@bL@J8$rI=<($7$^$9(J.
      10$@HV0J9_$O(J \"s\" $@$^$?$O(J \"n\" $@%3%^%s%I$r;H$C$F2<$5$$(J.

SPC   $@@bL@J8$r(J od:*scroll-step* $@$G<($5$l$?CM$@$1>e$K%9%/%m!<%k$7$^$9(J. 
DEL   $@@bL@J8$r(J od:*scroll-step* $@$G<($5$l$?CM$@$12<$K%9%/%m!<%k$7$^$9(J. 
C-d   $@@bL@J8$r(J od:*scroll-step* $@$G<($5$l$?CM$NH>J,$@$1>e$K%9%/%m!<%k$7$^$9(J. 
C-u   $@@bL@J8$r(J od:*scroll-step* $@$G<($5$l$?CM$NH>J,$@$12<$K%9%/%m!<%k$7$^$9(J. 
      $@$3$l$i$N%3%^%s%I$O@bL@J8$,I=<($5$l$F$$$J$1$l$P(J \".\" $@$HF1$8F0:n$r$7$^$9(J.
      od:*scroll-step* $@$K$O(J, $@@0?t(J, half, full $@$N$$$:$l$+$,;XDj$G$-$^$9(J.
      $@%G%U%)%k%H$O(J full $@$G$9(J.

h     $@%X%C%@$KLa$j$^$9(J. 
q     $@%*%s%i%$%s<-=q5/F0A0$N%&%$%s%I%&%3%s%U%#%0%l!<%7%g%s$KLa$j$^$9(J.
Q     $@%*%s%i%$%s<-=q$N;}$D%P%C%U%!$rA4$F:o=|$7(J, 
      $@%*%s%i%$%s<-=q5/F0A0$N%&%$%s%I%&%3%s%U%#%0%l!<%7%g%s$KLa$j$^$9(J.
R     $@%*%s%i%$%s<-=q$r:F5/F0$7$^$9(J. 
      $@<-=q%5!<%PEy$,%j%9%?!<%H$7$?;~$K%3%M%/%7%g%s$N:F3+$K;H$$$^$9(J.
      $@%P%C%U%!Ey$N%3%s%U%#%0%l!<%7%g%s$OA4$F<:$o$l$^$9(J.

C     $@<-=q$r@ZBX$($^$9(J. 
      $@8=:_(J eiwa($@1QOB<-E5(J), waei($@OB1Q<-E5(J), kojien($@9-<-1q(J) $@$,;H$($^$9(J.
      $@%3%s%W%j!<%7%g%sF~NO$,2DG=$G$9(J.
E     $@1QOB<-E5$G$NNcJ8$NI=<($N(J ON/OFF $@$r@ZBX$($^$9(J.

H     $@$b$7$"$l$P(J, $@K^Nc$rI=<($7$^$9(J. p, n, SPC, DEL, q $@$G$=$l$>$lA0%Z!<%8(J, 
      $@<!%Z!<%8(J, $@%9%/%m!<%k%"%C%W(J, $@%9%/%m!<%k%@%&%s(J, $@=*N;$H$J$j$^$9(J.
O     $@$b$7$"$l$P(J, $@1|IU$1$rI=<($7$^$9(J. q$@$G=*N;$7$^$9(J.

?     $@$3$N%a%C%;!<%8$rI=<($7$^$9(J.
")))

(defun od:show-title ()
  (insert "




		     $@EE;R%V%C%/HG<-=q8!:w%7%9%F%`(J


		       $@4dGH=qE9(J $@9-<-1q(J($@Bh;0HG(J)
	   $@8&5f<R(J $@?71QOBCf<-E5(J($@Bh(J5$@HG(J)$@!&?7OB1QCf<-E5(J($@Bh(J3$@HG(J)

		 Copyright (C) Shingo NISHIOKA, 1991
		    nishioka@sanken.osaka-u.ac.jp



* $@!V9-<-1q!W$O3t<02q<R4dGH=qE9$NEPO?>&I8$G$9(J.
")
  (goto-char 1))

(defun od:change-dictionary ()
  (interactive)
  (let ((new-dic (completing-read "Select jisyo: " od:*jisyo*)))
    (od:really-change-dictionary new-dic)))

(defun od:really-change-dictionary (new-dic)
  (od:open-dictionary)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (process-send-string od:*dict* (format "L%s\n" new-dic))
  (let ((ans (od:wait-until '("$*\n" "$&\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (if (not (string= ans "$*\n"))
	(error "Jisyo set failed")
	(setq od:*current-jisyo* new-dic)
	(setq od:*current-jisyo-pretty-name*
	      (cdr (assoc new-dic od:*dict-name*)))
	(setq od:*current-dict-filter*
	      (cdr (assoc new-dic od:*dict-filter*)))
	(setq od:*current-dict-filter-func*
	      (cdr (assoc new-dic od:*dict-filter-func*)))
	(message "$@<-=q$,!V(J%s$@!W$K%;%C%H$5$l$^$7$?(J."
		 od:*current-jisyo-pretty-name*)))
  (od:change-index-type)
  (setq od:*default-index-type*
	(cdr (assoc new-dic od:*default-index-type-dict*)))
  (setq od:*input-by-kanji*
	(cdr (assoc od:*default-index-type* od:*is-use-kanji*))))

(defun od:change-index-type ()
  (od:open-dictionary)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (process-send-string od:*dict* "I\n")
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (let ((lst))
	     (let ((tmp '(("BA" . ?A) ("BK" . ?K) ("IA" . ?a) ("IK" . ?k))))
	       (while tmp
		 (goto-char 1)
		 (if (re-search-forward (car (car tmp)) nil t)
		     (setq lst (cons (cdr (car tmp)) lst)))
		 (setq tmp (cdr tmp))))
	     (setq od:*index-types* lst)
	     (with-output-to-temp-buffer " *od:Selection*"
	       (princ (format "\n$@$3$N<-=q$O0J2<$N(J%d$@<oN`$N8!:w$,2DG=$G$9(J.\n"
			      (length lst)))
	       (while lst
		 (princ (cdr 
			 (assoc (car lst)
				'((?A . "$@%"%k%U%!%Y%C%H8eJ}(J ... $@8lF,$K(J * $@$rIU$1$k$3$H$,$G$-$k(J.")
				  (?K . "$@$+$J8eJ}(J           ... $@8lF,$K(J * $@$rIU$1$k$3$H$,$G$-$k(J.")
				  (?a . "$@%"%k%U%!%Y%C%HA0J}(J ... $@8lHx$K(J * $@$rIU$1$k$3$H$,$G$-$k(J.")
				  (?k . "$@$+$JA0J}(J           ... $@8lHx$K(J * $@$rIU$1$k$3$H$,$G$-$k(J.")))))
		 (princ "\n")
		 (setq lst (cdr lst)))))))))

(defun od:lookup-pattern ()
  (interactive)
  (let ((pat (od:read-string-with-fep (if od:*input-by-kanji*
					  "Pattern(in kana): "
					  "Pattern: ")
				      nil)))
    (od:really-lookup-pattern pat)))

(defun od:really-lookup-pattern (pat)
  (od:open-dictionary)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (let ((index (pattern-consistency-check pat)))
    (if (or (eql index ?A)
	    (eql index ?K))
	(setq pat (od:string-reverse pat)))
    (process-send-string od:*dict* (format "P%c%s\n" index pat)))
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (od:find-entries (buffer-string)))
	  (t (od:no-entries-found pat)))))

(defun pattern-consistency-check (pat)
  (let ((len (length pat))
	(i 0)
	(kana nil)
	(alfa nil)
	(other nil)
	(first nil)
	(last nil))
    (while (< i len)
      (let ((c (aref pat i)))
	(cond ((= ?* c)
	       (cond ((= i 0)
		      (setq first t))	;at beginning
		     ((= i (1- len))
		      (setq last t))	;at end
		     (t (error "\"*\"$@$O%Q%?!<%s$NC<0J30$G$O;H$($^$;$s(J."))))
	      ((= ?? c))		;do nothing
	      ((< 127 c)
	       (setq i (1+ i))
	       (let* ((cc (aref pat i))
		      (k (+ (* 256 c) cc)))
	       (cond ((or (= 164 c)	;hirakana
			  (= 165 c))	;katakana
		      (setq kana t))
		     ((= 163 c)		;eizi
		      (setq alfa t))
		     ((= k 41462)	;zenkaku *
		      (cond ((= i 1)
			     (setq first t))	;at beginning
			    ((= i (1- len))
			     (setq last t))	;at end
			    (t (error "\"*\"$@$O%Q%?!<%s$NC<0J30$G$O;H$($^$;$s(J."))))
		     (t)		;do nothing
		     )))
	      ((or (and (<= ?a c) (<= c ?z))
		   (and (<= ?A c) (<= c ?Z)))
	       (setq alfa t))
	      (t (setq other t))))
	(setq i (1+ i)))
    (if (and kana alfa)
	(message "$@2>L>$H1Q;z$,:.$6$C$F$$$^$9$,(J, $@2>L>$H$7$F8!:w$rB3$1$^$9(J."))
    (if (and first last)
	(error "\"*\"$@$O(J1$@8D$7$+;H$($^$;$s(J."))
    (let ((idx (cond (kana (if first ?K ?k))
		     (alfa (if first ?A ?a)))))
      (if (not (memq idx od:*index-types*))
	  (error "$@$=$N8!:wJ}K!$O;H$($^$;$s(J."))
      idx)))

(defun od:find-entries (str)
  (set-buffer " *od:temp*")
  (erase-buffer)
  (insert str)
  (setq od:*headers* nil)
  (setq od:*current-entry* -1)
  (goto-char 1)
  (if (looking-at "$0\n")		;this is $0 line
      (forward-char 3))			;just after $0
  (while (< (point) (- (point-max) 3))
    (let (entry body)
      (beginning-of-line)
      (let ((begin (point)))
	(end-of-line)
	(let ((end (point)))
	  (setq entry (buffer-substring begin end))))
      (next-line 1)
      (beginning-of-line)
      (let ((begin (point)))
	(end-of-line)
	(let ((end (point)))
	  (setq body (buffer-substring begin end))))
      (if (not (od:findp-equal (cons entry body) od:*headers*))
	  (setq od:*headers*
		(cons (cons entry body)
		      od:*headers*)))
      (next-line 1)))
  (if (null od:*headers*)
      (od:no-entries-found pat)
      (setq od:*headers* (reverse od:*headers*))
      (od:show-headers)
      (delete-other-windows)
      (if (and od:*select-unique-entry-immediately*
	       (= 1 (length od:*headers*)))
	  (od:select-entry 1))))

(defun od:show-headers ()
  (interactive)
  (od:set-header)
  (let ((buffer-read-only nil))
    (erase-buffer)
    (insert (format "%d entries.\n"
		    (length od:*headers*)))
    (let ((tmp od:*headers*)
	  (i 1))
      (while tmp
	(let ((ent (car tmp)))
	  (insert (format "%3d: %s\n" i (car ent))))
	(setq i (1+ i))
	(setq tmp (cdr tmp)))))
  (goto-line 2)
  (forward-char 4))

(defun od:select-entry (n)
  (interactive "nNumber:")
  (let ((len (length od:*headers*)))
    (if (and (<= 1 n) (<= n len))
	(progn
	  (setq od:*current-entry* n)
	  (let ((ent (od:get-entry (nth (1- n) od:*headers*))))
	    (od:show-entry ent))
	  (od:set-header)
	  (goto-line (1+ n))
	  (forward-char 4))
	(message (format "%d out of range" n)))))

(defun od:get-entry (pat)
  (od:open-dictionary)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (process-send-string od:*dict* (format "S%s\n" (cdr pat)))
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (buffer-substring 4 (- (point-max) 4)))
	  (t ""))))

(defun od:direct-select-entry ()
  (interactive)
  (let ((n (string-to-int (char-to-string last-command-char))))
    (od:select-entry n)))

(defun od:current-header-line ()
  (save-excursion
    (od:set-header)
    (1- (count-lines 1 (1+ (point))))))

(defun od:previous-entry ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (< 1 curr)
	(od:select-entry (1- curr))
	(message "No previous entry"))))

(defun od:current-entry ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (od:select-entry curr)))

(defun od:next-entry ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (< curr (length od:*headers*))
	(od:select-entry (1+ curr))
	(message "No following entry"))))

(defun od:scroll-step ()
  (cond ((integerp od:*scroll-step*)
	 od:*scroll-step*)
	((eql od:*scroll-step* 'full)
	 (- (window-height (next-window (selected-window))) 2))
	((eql od:*scroll-step* 'half)
	 (- (/ (window-height (next-window (selected-window))) 2) 1))))

(defun od:scroll-entry-up ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (od:scroll-step)))))

(defun od:scroll-entry-down ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (- (od:scroll-step))))))

(defun od:scroll-entry-up-half ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (/ (od:scroll-step) 2)))))

(defun od:scroll-entry-down-half ()
  (interactive)
  (let ((curr (od:current-header-line)))
    (if (/= od:*current-entry* curr)
	(od:select-entry curr)
	(scroll-other-window (- (/ (od:scroll-step) 2))))))

(defun od:show-entry (entry)
  (save-excursion
    (od:set-body)
    (let ((buffer-read-only nil))
      (erase-buffer)
      (insert entry)
      (let ((tmp od:*current-dict-filter-func*))
	(while tmp
	  (goto-char 1)
	  (funcall (car tmp))
	  (setq tmp (cdr tmp))))
      (fill-region 1 (point-max))
      (hang-indent-buffer))
    (goto-char 1)))

(defun od:no-entries-found (pat)
  (od:set-body)
  (let ((buffer-read-only nil))
    (erase-buffer)
    (insert (format "\"%s\" Not found\n" pat)))
  (od:set-header))



(defun od:toggle-eiwa-summary-mode ()
  (interactive)
  (setq od:*eiwa-summary-mode* (not od:*eiwa-summary-mode*))
  (if od:*eiwa-summary-mode*
      (setq od:*eiwa-current-display-mode* "[Summary]")
      (setq od:*eiwa-current-display-mode* ""))
  (od:current-entry))



(defun od:quit ()
  (interactive)
  (if od:*previous-config*
      (set-window-configuration od:*previous-config*))
  (setq od:*previous-config* nil))

(defun od:really-quit ()
  (interactive)
  (od:kill-active-buffers)
  (if od:*previous-config*
      (set-window-configuration od:*previous-config*))
  (setq od:*previous-config* nil))

(defun od:really-restart ()
  (interactive)
  (if (yes-or-no-p "Really restart? ")
      (progn
	(setq od:*dict* nil)
	(setq od:*headers* nil)
	(setq od:*current-entry* -1)
	(kill-buffer " *ndtp*")
	(kill-buffer " *od:hostname*")
	(kill-buffer " *od:temp*")
	(kill-buffer "*Dict-Header*")
	(kill-buffer "*Dict-Body*")
	(online-dictionary))))



(defun od:hostname ()
  (save-excursion
    (set-buffer " *od:hostname*")
    (erase-buffer)
    (call-process "/bin/hostname" nil " *od:hostname*")
    (let ((str (buffer-string)))
      (let ((i 0)
	    (len (length str)))
	(while (and (< i len)
		    (/= (aref str i) ?\n))
	  (setq i (1+ i)))
	(substring str 0 i)))))


(defvar od:ispell-program-name "ispell"
  "Program invoked by ispell-word and ispell-region commands.")

(defconst od:ispell-out-name " *ispell*"
  "Name of the buffer that is associated with the 'ispell' process")

(defvar od:ispell-process nil
  "Holds the process object for 'ispell'")

(defun od:ispell-init-process ()
  (if (and od:ispell-process
	   (eq (process-status od:ispell-process) 'run))
      (save-excursion
	(set-buffer od:ispell-out-name)
	(erase-buffer))
      (message "Starting new ispell process...")
      (and (get-buffer od:ispell-out-name) (kill-buffer od:ispell-out-name))
      (setq od:ispell-process (apply 'start-process "ispell"
                                   od:ispell-out-name od:ispell-program-name
				   (list "-A")))
      (process-kill-without-query od:ispell-process)
      (sleep-for 3)))

(defun od:lookup-pattern-with-ispell ()
  (interactive)
  (let ((orig-pat (read-string "Pattern: ")))
    (od:ispell-init-process)
    (send-string od:ispell-process orig-pat)
    (send-string od:ispell-process "\n")
    (sleep-for 3)
    (set-buffer od:ispell-out-name)
    (goto-char 1)
    (cond ((looking-at "\\*")
	   ;; correct!
	   (od:really-lookup-pattern orig-pat))
	  ((looking-at "+")
	   ;; a word follows
	   (od:really-lookup-pattern (od:select-candidate)))
	  ((looking-at "&")
	   ;; some words follows
	   (od:really-lookup-pattern (od:select-candidate)))
	  ((looking-at "#")
	   ;; cannot find any candidates
	   (message "No candidate.")))))

(defun od:select-candidate ()
  (goto-char 1)
  (forward-char 2)
  (let ((list nil))
    (while (looking-at "[^ ]")
      (let ((begin (point)))
	(forward-word 1)
	(setq list (cons (buffer-substring begin (point)) list)))
      (while (looking-at "[ \t\n]") (forward-char 1)))
    (with-output-to-temp-buffer " *od:Selection*"
      (let ((i 1)
	    (col 0)
	    (l list))
	(while l
	  (let ((str (format "(%d)  %s  " i (car l))))
	    (let ((len (length str)))
	      (if (< (- (screen-width) 2) (+ col len))
		  (progn
		    (princ "\n")
		    (setq col 0)))
	      (setq col (+ col len))
	      (princ str)))
	  (setq l (cdr l))
	  (setq i (1+ i)))))
    (let ((sel (od:read-integer)))
      (while (or (< sel 1) (< (length list) sel))
	(setq sel (od:read-integer)))
      (nth (1- sel) list))))

(defun od:read-integer ()
  (let ((n (string-to-int (read-from-minibuffer "Choose:"))))
    (while (not (integerp n))
      (setq n (string-to-int (read-from-minibuffer "Choose:"))))
    n))

(defun od:read-string-with-fep (prompt initial)
  (set-buffer (format " *Minibuf-%d*" (minibuffer-depth)))
  (cond ((or (not od:*input-by-kanji*)
	     (eql od:*fep-type* 'no-fep))
	 (format "%s" (read-minibuffer prompt (or initial ""))))
	((eql od:*fep-type* 'egg)
	 (toggle-egg-mode)
	 (format "%s" (read-minibuffer prompt (or initial ""))))
	((eql od:*fep-type* 't-code)
	 (setq tcode-on-in-minibuffer nil)
	 (setq unread-command-char ?\034)
	 (format "%s" (read-minibuffer prompt (or initial ""))))
	(t (error "Unknown fep-type %s" od:*fep-type*))))

(defun od:findp-equal (item sequence)
  (let ((is-true nil))
    (while (and sequence
		(not is-true))
      (if (equal item (car sequence))
	  (setq is-true t)
	  (setq sequence (cdr sequence))))
    is-true))

(defun hang-indent-buffer ()
  (goto-char 1)
  (beginning-of-line)
  (while (< (point) (point-max))
    (cond ((looking-at "[ \t]")
	   (while (looking-at "[ \t]")
	     (delete-char 1)))
	  ((looking-at "\n")
	   ;; do nothing
	   )
	  (t
	   (insert "  ")))
    (next-line 1)
    (beginning-of-line)))

(defun od:string-reverse (s)
  (let ((n (make-string (length s) ? ))
	(j (1- (length s)))
	(i 0))
    (while (<= 0 j)
      (let ((c (aref s i)))
	(if (<= c 127)
	    (progn
	      (aset n j c)
	      (setq i (1+ i)
		    j (1- j)))
	    (progn
	      (aset n j  (aref s (1+ i)))
	      (aset n (1- j) c)
	      (setq i (+ i 2)
		    j (- j 2))))))
    n))

;;; hanrei, okuduke

(defvar od:*previous-config-frm* nil)

(defun od:save-config-frm ()
  (if (null od:*previous-config-frm*)
      (setq od:*previous-config-frm*
	    (current-window-configuration))))

(defun od:restore-config-frm ()
  (interactive)
  (if od:*previous-config-frm*
      (set-window-configuration od:*previous-config-frm*))
  (setq od:*previous-config-frm* nil))

(defun od:show-okuduke ()
  (interactive)
  (od:save-config-frm)
  (od:open-dictionary)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (process-send-string od:*dict* "I\n")
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (goto-char 1)
	   (if (re-search-forward "OK " nil t nil)
	       (if (re-search-forward "[0-9a-f][0-9a-f]*")
		   (let ((frm (buffer-substring
			       (match-beginning 0) (match-end 0))))
		     (od:show-frame (od:xtoi frm)))))))))

(defun od:show-hanrei ()
  (interactive)
  (od:save-config-frm)
  (od:open-dictionary)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (process-send-string od:*dict* "I\n")
  (let ((ans (od:wait-until '("$$\n" "$?\n" "$N\n" "$<\n") od:*dict*)))
    (cond ((string= "$$\n" ans)
	   (goto-char 1)
	   (if (re-search-forward "HA " nil t nil)
	       (if (re-search-forward "[0-9a-f][0-9a-f]*")
		   (let ((frm (buffer-substring
			       (match-beginning 0) (match-end 0))))
		     (od:show-frame (od:xtoi frm)))))))))

(defvar od:*current-frame* nil)

(defun od:next-frame ()
  (interactive)
  (if od:*current-frame*
      (progn (setq od:*current-frame* (1+ od:*current-frame*))
	     (od:show-frame od:*current-frame*))))

(defun od:previous-frame ()
  (interactive)
  (if od:*current-frame*
      (progn (setq od:*current-frame* (1- od:*current-frame*))
	     (od:show-frame od:*current-frame*))))

(defun od:show-frame (frm)
  (setq od:*current-frame* frm)
  (set-buffer " *ndtp*")
  (erase-buffer)
  (process-send-string od:*dict* (format "F%x\n" frm))
  (sit-for 1)
  (let ((s (buffer-substring 3 (+ 2048 3))))
    (get-buffer-create " *od:Frame*")
    (set-buffer " *od:Frame*")
    (erase-buffer)
    (local-set-key "p" 'od:previous-frame)
    (local-set-key "n" 'od:next-frame)
    (local-set-key " " 'scroll-up)
    (local-set-key "\177" 'scroll-down)
    (local-set-key "q" 'od:restore-config-frm)
    (insert (format "frame=%x\n" frm))
    (od:format-frame s)
    (switch-to-buffer " *od:Frame*")
    (delete-other-windows)
    (goto-char 1)))

(defun od:format-frame (s)
  (let ((i 0))
    (while (< i 2048)
      (let ((hi (aref s i))
	    (lo (aref s (1+ i))))
	(cond ((= hi 31)
	       (cond ((= lo 10)
		      (insert "\n"))
		     ((= lo ?c)
		      (setq i (+ i 2))
		      (let ((num (+ (* (aref s (+ i 0)) (* 256))
				    (* (aref s (+ i 1)) (*))))
			    (frm (+ (* (aref s (+ i 2)) (* 256))
				    (* (aref s (+ i 3)) (*))))
			    (ofs (+ (* (aref s (+ i 4)) (* 256))
				    (* (aref s (+ i 5)) (*)))))
			(setq i (+ i 4))))))
	      ((or (<= 128 hi) (<= 128 lo)))
	      ((or (< hi 32) (< lo 32)))
	      (t (insert (format "%c%c" (+ hi 128) (+ lo 128)))))
	(setq i (+ i 2))))))

(defun od:xtoi (s)
  (let ((r 0)
	(i 0)
	(l (length s)))
    (while (< i l)
      (let ((c (aref s i)))
	(setq r (+ (* r 16)
		   (if (and (<= ?0 c) (<= c ?9))
		       (- c ?0)
		       (+ (- c ?a) 10))))
	(setq i (1+ i))))
    r))

;;;
;;; Additional new interface [lookup-current-word].
;;;  look up a current point word.
;;;
;;;  15 Nov 91  Tetsuya Nishimaki   (t-nishim@dn.softbank.co.jp)
;;;

(defun word-at-point ()
  (let ((pat "\\([^a-zA-Z][a-zA-Z]+\\|[^$@$"(J-$@$s(J][$@$"(J-$@$s(J]+\\|[^$@%"(J-$@%s(J][$@%"(J-$@%s(J]+\\)"))
  (condition-case ()
      (save-excursion
	(if (looking-at ".") (forward-char 1))
	(if (not (looking-at pat))
	    (re-search-backward pat nil t))
	(or (re-search-forward pat nil t)
	    (error nil))
	(goto-char (match-beginning 0))
	(forward-char 1)
	(buffer-substring (point) (match-end 0)))
    (error nil))))

(defun od:lookup-pattern-edit ()
  (interactive)
  (let* ((v (word-at-point))
	 (enable-recursive-minibuffers t)
	 (pat (od:read-string-with-fep (if od:*input-by-kanji*
					   "Pattern(in kana): "
					   "Pattern: ")
				       v)))
    (online-dictionary)
    (od:really-lookup-pattern pat)))
