/*
 *                    Copyright (C) Shingo NISHIOKA, 1991
 *                       nishioka@sanken.osaka-u.ac.jp
 *            Everyone is permitted to do anything on this program
 *         including copying, transplanting, debugging, and modifying.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/time.h>
#ifdef SYSLOG
#include <syslog.h>
#endif

#include "cdrom.h"
#include "dict.h"
#include "ndtp.h"

#define TRUE 1
#define FALSE 0

#ifdef DEBUG
extern int debug;
#endif

static client_wants_talk();

static int 	ls;		/* listen socket */

iserver_init()
{
	struct servent *sp;
	struct sockaddr_in myaddr_in;

#ifdef SYSLOG
	openlog("dserver", LOG_PID, LOG_USER);
#endif
#ifdef STANDALONE
	bzero((char *)&myaddr_in, sizeof(struct sockaddr_in));

	myaddr_in.sin_family = AF_INET;
	myaddr_in.sin_addr.s_addr = INADDR_ANY;

	sp = getservbyname(SERVICE, "tcp");
	if(sp == NULL)  {
#ifdef SYSLOG
		syslog(LOG_ERR,"Service not found in /etc/services\n");
#endif
		exit(1);
	}
	myaddr_in.sin_port = sp->s_port;

	ls = socket(AF_INET, SOCK_STREAM, 0);
	if(ls == -1)  {
#ifdef SYSLOG
		syslog(LOG_ERR,"Unable to create a socket /iserver_init\n");
#endif
		exit(1);
	}

	if( bind(ls, &myaddr_in, sizeof(struct sockaddr_in)) == -1)  {
#ifdef SYSLOG
		syslog(LOG_ERR,"Unable to bind address /iserver_init\n");
#endif
		exit(1);
	}

	if(listen(ls, 5) == -1)  {
#ifdef SYSLOG
		syslog(LOG_ERR,"Unable to listen on socket /iserver_init\n");
#endif
		exit(1);
	}
#ifdef DEBUG
	if(debug>=2) {
		printf("Listen socket=%d\n",ls);
	}
#endif
#endif /* STANDALONE */
}

static
iserver_accept()
{
#ifdef STANDALONE
	int s, addrlen;
	struct sockaddr_in peeraddr_in;

	addrlen = sizeof(struct sockaddr_in);
	bzero((char *)&peeraddr_in, addrlen);

	s = accept(ls, &peeraddr_in, &addrlen);
	if(s==-1) {
#ifdef SYSLOG
		syslog(LOG_ERR,"Unable to accept /iserver_accept\n");
#endif
		return -1;
	}
	return s;
#else /* STANDALONE */
	(void) close(1);
	(void) close(2);
	(void) dup(0);
	(void) dup(0);
	return (0);
#endif /* STANDALONE */
}

#define CLNUM 64

typedef struct _env {
	int d;
	int authority;
	char name[32];
	Dict *dict;
	int conv;
} ENV;

static ENV env[CLNUM];
static int clnum;

static Dict *select_dict();

command_loop()
{
	int width;
	fd_set readfds, writefds, exceptfds;
	int c;

#ifdef STANDALONE
	clnum=0;
	width = getdtablesize();
	for(;;) {
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(ls,&readfds);
		for(c=0; c<clnum; c++) {
			if(env[c].d!=-1) {
				FD_SET(env[c].d,&readfds);
			}
		}
		select(width,&readfds,&writefds,&exceptfds,NULL);
		if(FD_ISSET(ls,&readfds)) {
			int new_client;
			new_client = iserver_accept();
#ifdef DEBUG
			if(debug>=2) {
				printf("%d\n",new_client);
			}
#endif
			for(c=0; c<clnum && env[c].d!=-1; c++) ;

			if(c<clnum) {
				env[c].d = new_client;
				env[c].authority=0;
				env[c].conv=0;
			}
			else if(clnum<CLNUM) {
				env[clnum].d = new_client;
				env[clnum].authority=0;
				env[clnum].conv=0;
				clnum++;
			}
			else {
#ifdef SYSLOG
				syslog(LOG_ERR,"Too many clients\n");
#endif
			}
		}
		printf("%d\n",clnum);
		for(c=0; c<clnum; c++) {
			if(FD_ISSET(env[c].d, &readfds)) {
				client_wants_talk(c);
			}
		}
#else /* STANDALONE */
		iserver_accept();	
		env[0].d = 0;
		env[0].authority=0;
		env[0].conv=0;

	for(;;){
		client_wants_talk(0);
#endif /* STANDALONE */
	}
}

static
client_wants_talk(c)
int c;
{
	int isclosed;
#ifdef DEBUG
	if(debug>=2) {
		printf("client %d(%d) wants to talk\n", c,env[c].d);
	}
#endif
	isclosed=process_a_client(&env[c]);
	if(isclosed) {
#ifdef DEBUG
		if(debug>=2) {
			printf("Client %d connection closed\n",c);
		}
#endif
#ifdef SYSLOG
#ifdef DEBUG
		if(debug>=2) {
			printf("close connection %s\n",env[c].name);
		}
#endif
		if(env[c].authority) {
			syslog(LOG_INFO,
			       "user=%s, stat=Closed connection, conv=%d",
			        env[c].name, env[c].conv);
		}
#endif
		close(env[c].d);
		env[c].d = -1;
		strcpy(env[c].name,"");
		env[c].authority=0;
#ifndef STANDALONE
		exit(0);
#endif
	}
}

#define BUFSIZE	4096

static
process_a_client(p)
ENV *p;
{
	static char buf[BUFSIZE];
	int len;
	int s;

	(p->conv)++;

	s = p->d;
	bzero(buf,BUFSIZE);
	len = read(s,buf,BUFSIZE);
	buf[len] = '\0';
#ifdef DEBUG
	if(debug>=2) {
		printf("%s\n",buf);
	}
#endif
	if(len==0) return(1);

	/* A, Q and u can be used without authority */

	switch(buf[0]) {
		int c;		/* u */
		static char outbuf[128]; /* u */
	case 'A':
		if(!host_access(s)){
			int i;
#ifndef STANDALONE
			dic_inits();
#endif
			p->authority=1;
			p->dict = NULL;
			for(i=0; i<31&&buf[i]!='\n'; i++) ;
			buf[i]='\0';
			strcpy(p->name,buf+1);
			write(s,"$A\n",3);

#ifdef SYSLOG
#ifdef DEBUG
			if(debug>=2) {
				printf("new user %s\n",p->name);
			}
#endif
			syslog(LOG_INFO,"user=%s, stat=New connection",p->name);
#endif
			return(0);
		}
		else {
			write(s,"$N\n",3);
			return(1);
		}
	case 'Q':
		return(1);
	case 'u':
		strcpy(outbuf,"$U\n");
		write(s,outbuf,strlen(outbuf));
		for(c=0; c<clnum; c++) {
			if(env[c].d!=-1 && env[c].authority) {
				sprintf(outbuf,"%5d %s\n",c,env[c].name);
				write(s,outbuf,strlen(outbuf));
			}
		}
		strcpy(outbuf,"$$\n");
		write(s,outbuf,strlen(outbuf));
		return(0);
	case 'v':
		write_server_version(s);
		return(0);
	}

	if(p->authority==0) {
		write(s,"$N\n",3);
		return(0);
	}

	switch(buf[0]) {
		char *tmp;
		Dict *newdic;
	case 'L':
#ifdef DEBUG
		if(debug>=2) {
			fprintf(stderr,"Select dictionary ");
			fflush(stderr);
		}
#endif
		for(tmp=buf+1; *tmp && *tmp!='\n'; tmp++);
		*tmp = '\0';
#ifdef DEBUG
		if(debug>=2) {
			fprintf(stderr,"<%s>\n",buf+1);
			fflush(stderr);
		}
#endif
		if((newdic = select_dict(buf+1))!=NULL) {
			p->dict = newdic;
			write(s,"$*\n",3);
		}
		else {
			write(s,"$&\n",3);
		}
		break;
#if 0
	case 'p':
		if(p->dict==-1) {
			write(s,"$<\n",3);
		}
		else {
			search_pattern(buf+1,s,p->dict);
		}
		break;
#endif
	case 'P':
	case 'F':
	case 'S':
	case 'I':
		if(p->dict==NULL) {
			write(s,"$<\n",3);
		}
		else {
			exec_newcommands(p->dict,buf,s);
		}
		break;
	default:
		write(s,"$?\n",3);
		break;
	}
	return(0);
}

struct _dict_set {
	char *name;
	char *filename;
	int mount;
	Dict dict;
};

struct _dict_set dict_set[] 
	= {{"eiwa", EIWA, FALSE, NULL},
	   {"waei", WAEI, FALSE, NULL},
	   {"kojien", KOJIEN, FALSE, NULL}};


dic_inits()
{
	int i;
	FILE *dic;

	for(i=0; i<sizeof(dict_set)/sizeof(struct _dict_set); i++) {

		if(!(dic=fopen(dict_set[i].filename,"r"))) {
			fprintf(stderr,"Cannot open file %s\n",dict_set[i].filename);
			dict_set[i].mount = FALSE;
		}
		else {
			dict_set[i].mount = TRUE;
			init_dic(&(dict_set[i].dict),dic);
		}

#ifdef SYSLOG
			syslog(LOG_INFO,"Initializing %s\n",dict_set[i].name);
#endif
	}
}

static
Dict *
select_dict(name)
char *name;
{
	int i;
	for(i=0; i<sizeof(dict_set)/sizeof(struct _dict_set); i++) {
#ifdef DEBUG
		if(debug>=2) {
			fprintf(stderr,"(%d)\n",i);
			fflush(stderr);
		}
#endif
		if(!strcmp(dict_set[i].name,name) && dict_set[i].mount) {
			return &(dict_set[i].dict);
		}
	}
	return NULL;
}

#if 0
search_pattern(pattern,s,d)
char *pattern;
int s,d;
{
	char *p;

	for(p=pattern; *p && *p!='\n'; p++);
	*p = '\0';
	(*(dict_set[d].set_out))(s);
	(*(dict_set[d].proc_ent))(pattern);
}
#endif
