/*
 * Copyright (c) 1992 Stanford University
 * Copyright (c) 1992 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Stanford and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Stanford and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */

#include "wtable.h"
#include <InterViews/display.h>
#include <InterViews/patch.h>
#include <InterViews/window.h>
#include <IV-X11/Xlib.h>
#include <IV-X11/xatom.h>
#include <IV-X11/xdisplay.h>
#include <IV-X11/xselection.h>
#include <IV-X11/xwindow.h>
#include <OS/string.h>
#include <OS/table.h>
#include <stdlib.h>

/*
 * SelectionManager -- inter client communications mechanism
 */

SelectionManager::SelectionManager(Display* d, const char* name) {
    rep_ = new SelectionManagerRep(d, String(name));
}

SelectionManager::SelectionManager(Display* d, const String& name) {
    rep_ = new SelectionManagerRep(d, name);
}

SelectionManager::~SelectionManager() {
    delete rep_;
}

void SelectionManager::own(
    SelectionHandler* convert, SelectionHandler* lose, SelectionHandler* done
) {
    SelectionManagerRep& s = *rep();
    Resource::ref(convert);
    Resource::unref(s.convert_);
    s.convert_ = convert;
    Resource::ref(lose);
    Resource::unref(s.lose_);
    s.lose_ = lose;
    Resource::ref(done);
    Resource::unref(s.done_);
    s.done_ = done;

    XSetSelectionOwner(
	s.xdisplay_, s.atom_, s.owner_->rep()->xwindow_, CurrentTime
    );
}

void SelectionManager::retrieve(
    const String& type, SelectionHandler* ok, SelectionHandler* fail
) {
    SelectionManagerRep& s = *rep();
    Resource::ref(ok);
    Resource::unref(s.ok_);
    s.ok_ = ok;
    Resource::ref(fail);
    Resource::unref(s.fail_);
    s.fail_ = fail;

    Atom atom = XAtom::intern(s.xdisplay_, type);
    XConvertSelection(
	s.xdisplay_, s.atom_, atom, atom,
	s.owner_->rep()->xwindow_, CurrentTime
    );
}

void SelectionManager::put_value(
    const String& type, const void* data, int length, int format
) {
    SelectionManagerRep& s = *rep();
    XSelectionRequestEvent& req = s.event_.xselectionrequest;
    Atom atom = XAtom::intern(s.xdisplay_, type);
    XChangeProperty(
        s.xdisplay_, req.requestor, req.property,
	atom, format, PropModeReplace,
	(const unsigned char*)data, length
    );
    XEvent xe;
    XSelectionEvent& xs = xe.xselection;
    xs.type = SelectionNotify;
    xs.requestor = req.requestor;
    xs.selection = req.selection;
    xs.target = req.target;
    xs.property = req.property;
    xs.time = req.time;
    XSendEvent(s.xdisplay_, xs.requestor, False, 0, &xe);
}

void SelectionManager::get_value(
    String& type, void*& data, int& length, int& format
) {
    Atom		atom;
    unsigned long	after;
    unsigned long	nitems;
    unsigned char*	prop;

    SelectionManagerRep& s = *rep();
    XSelectionEvent& ev = s.event_.xselection;
    if (XGetWindowProperty(s.xdisplay_, ev.requestor, ev.property, 0L, 102400L,
			   True, AnyPropertyType, &atom, &format,
			   &nitems, &after, &prop) == Success) {
	type = XAtom::name(s.xdisplay_, atom);
	length = int(nitems);
	if (format == 8) {
	    char *src = (char *)(prop);
	    char *dst = new char[length + 1];
	    for (int i = 0 ; i < length; i++) {
		dst[i] = src[i];
	    }
	    dst[i] = '\0';
	    data = (void*) dst;
	} else if (format == 16) {
	    short *src = (short *)(prop);
	    short *dst = new short[length + 1];
	    for (int i = 0 ; i < length; i++) {
		dst[i] = src[i];
	    }
	    dst[i] = 0;
	    data = (void*) dst;
	} else if (format == 32) {
	    long *src = (long *)(prop);
	    long *dst = new long[length + 1];
	    for (int i = 0 ; i < length; i++) {
		dst[i] = src[i];
	    }
	    dst[i] = 0;
	    data = (void*) dst;
	}
	XFree((char*) prop);
    } else {
	length = 0;
	data = nil;
    }
}

SelectionManagerRep* SelectionManager::rep() const { return rep_; }

/* class SelectionManagerRep */

SelectionManagerRep::SelectionManagerRep(Display* d, const String& name) {
    DisplayRep& dr = *d->rep();
    xdisplay_ = dr.display_;
    name_ = new CopyString(name);
    atom_ = XAtom::intern(xdisplay_, name_->string());
    owner_ = new PopupWindow(new Patch(nil));
    WindowRep& wr = *owner_->rep();
    wr.display_ = d;
    wr.xwindow_ = XCreateSimpleWindow(
        xdisplay_, dr.root_, 0, 0, 1, 1, 0, 0, 0
    );
    dr.wtable_->insert(wr.xwindow_, owner_);
    wr.xtoplevel_ = wr.xwindow_;
    convert_ = nil;
    lose_ = nil;
    done_ = nil;
    ok_ = nil;
    fail_ = nil;
}

SelectionManagerRep::~SelectionManagerRep() {
    delete name_;
    delete owner_;
    Resource::unref(convert_);
    Resource::unref(lose_);
    Resource::unref(done_);
    Resource::unref(ok_);
    Resource::unref(fail_);
}

void SelectionManagerRep::request(SelectionManager* s, const XEvent& x) {
    if (convert_ != nil) {
	event_ = x;
	convert_->handle(s);
    }
}

void SelectionManagerRep::notify(SelectionManager* s, const XEvent& x) {
    if (event_.xselection.property != None) {
        if (ok_ != nil) {
	    event_ = x;
	    ok_->handle(s);
        }
    } else {
        if (fail_ != nil) {
	    event_ = x;
	    fail_->handle(s);
	}
    }
}

void SelectionManagerRep::clear(SelectionManager* s, const XEvent& x) {
    if (lose_ != nil) {
	event_ = x;
	lose_->handle(s);
    }
}

/* class SelectionHandler */

SelectionHandler::SelectionHandler(){}

SelectionHandler::~SelectionHandler(){}
