/*
 * Copyright (C) 1992 by Software Research Associates, Inc.
 *      Author: Y. Kawabe <kawabe@sra.co.jp>
 *
 * Permission to use, copy, modify, and distribute, and sell this software
 * and its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  Software Research Associates
 * makes no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 */

#include <NLS/charset.h>
#include <NLS/nlsfile.h>
#include <OS/table.h>
#include <OS/ustring.h>
#include <string.h>
#include <stdlib.h>

#ifndef IV_CHARSET_PATH
#define IV_CHARSET_PATH         "CharSet"
#endif

/*
 * Class CharSetRep
 */

CharSetRep::CharSetRep(
    const char* name, int bytes, int cols, int chars, char final
) {
    name_ = new UniqueString(name);
    bytes_ = bytes;
    cols_ = cols;
    chars_ = chars;
    final_ = final;
}

CharSetRep::~CharSetRep() {
    delete name_;
}
			
/*
 * Class CharSetRepTable
 */

declareTable(CharSetRepTable, int, const CharSetRep*);
implementTable(CharSetRepTable, int, const CharSetRep*);

/*
 * Class CharSet
 */

CharSetRepTable	*CharSet::table_;

void CharSet::initialize () {
    if (table_ == nil) {
	table_ = new CharSetRepTable(32);
	table_->insert(
	    CharSet::ascii(), new CharSetRep("ASCII", 1, 1, 94, 'B')
	);
	table_->insert(
	    CharSet::latin(), new CharSetRep("ISO8859-1", 1, 1, 96, 'A')
	);
	
	int		argc;
	const int	argsize = 64;
	char		*argv[argsize];
	nlsFile		file (IV_CHARSET_PATH);
	CharSetRep	*rep;
	int		charset;

	while ((argc = file.getline (argv, argsize)) >= 0) {
	    if (argc == 6) {
		charset = file.str2int(argv[0]);
		if (table_->find_and_remove(rep, charset)) {
		    delete rep;
		}
		rep = new CharSetRep(argv[1],
				     file.str2int(argv[2]),
				     file.str2int(argv[3]),
				     file.str2int(argv[4]),
				     argv[5][0]);
		table_->insert(charset, rep);
	    }
	}
    }
}

/*
 *	find CharSet ID
 */

CharSet_T CharSet::find (const char* name) {
    initialize();
    UniqueString uname(name);
    return find (uname);
}

CharSet_T CharSet::find (const String& name) {
    initialize();
    UniqueString uname(name);
    for (TableIterator(CharSetRepTable) i(*table_); i.more(); i.next()) {
	if (i.cur_value()->name() == uname)
	    return i.cur_key();
    }
    return -1;
}

CharSet_T CharSet::find (enum SorM n, int chars, char final) {
    initialize();
    if (n == Single) {
	for (TableIterator(CharSetRepTable) i(*table_); i.more(); i.next()) {
	    if (i.cur_value()->bytes() == 1 &&
		i.cur_value()->chars() == chars && 
		i.cur_value()->final() == final) {
		return i.cur_key();
	    }
	}
    } else {
	for (TableIterator(CharSetRepTable) i(*table_); i.more(); i.next()) {
	    if (i.cur_value()->bytes() > 1 &&
		i.cur_value()->chars() == chars &&
		i.cur_value()->final() == final) {
		return i.cur_key();
	    }
	}
    }
    return -1;
}

const CharSetRep* CharSet::rep (CharSet_T charset) {
    initialize();
    const CharSetRep *rep;
    if (table_->find(rep, charset)) {
	return rep;
    }
    return nil;
}
