/* vi:ts=4:sw=4
 *
 * VIM - Vi IMitation Japanese extension
 *
 * Code Contributions By:	Atsushi Nakamura		ann@mrit.mei.co.jp
 */

#include "vim.h"
#include "globals.h"
#include "proto.h"
#include "param.h"
#include "ops.h"
#ifdef JP
#include "jp.h"

#include "jptab.h"

#ifdef UNIX
#include "signal.h"

/*
 *	handle core dump by some bugs.
 */
	static void
core_handler(dummy)
	int dummy;
{
	static void abend_handler();
	abend_handler(TRUE);
}

	static void 
hup_handler(dummy)
	int dummy;
{
	static void abend_handler();
	abend_handler(FALSE);
}

	static void
abend_handler(core)
	int core;
{
	char *text;

	settmode(0);
	stoptermcap();
	flushbuf();
	stopscript();					/* remove autoscript file */
#ifdef JPFEP
	fep_freqsave();
#endif

	signal(SIGQUIT, SIG_DFL);
	signal(SIGHUP,  SIG_DFL);
	signal(SIGTERM, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);
#ifdef SIGILL
	signal(SIGILL,  SIG_DFL);
#endif
#ifdef SIGBUS
	signal(SIGBUS,  SIG_DFL);
#endif
#ifdef SIGLOST
	signal(SIGLOST, SIG_DFL);
#endif
#ifdef SIGPWR
	signal(SIGPWR,  SIG_DFL);
#endif

#ifdef JP
	printf("\n%s is killed by some bugs....\n", JpVersion);
#else
	printf("\n%s is killed by some bugs....\n", Version);
#endif
	text = modname(Filename, ".sav");
	if (writeit(text, NULL, (linenr_t)1, line_count, FALSE, FALSE, TRUE))
		printf("\nThe current text is saved into %s.\n", text);
	else
		printf("\nI couldn't save the current text.\n");

	if (core)
		kill(getpid(), SIGQUIT);
	else
		kill(getpid(), SIGKILL);
}

	static void
core_init()
{
	signal(SIGQUIT, core_handler);
	signal(SIGHUP,  hup_handler);
	signal(SIGTERM, hup_handler);
	signal(SIGSEGV, core_handler);
#ifdef SIGILL
	signal(SIGILL,  core_handler);
#endif
#ifdef SIGBUS
	signal(SIGBUS,  core_handler);
#endif
#ifdef SIGLOST
	signal(SIGLOST, hup_handler);
#endif
#ifdef SIGPWR
	signal(SIGPWR,  hup_handler);
#endif
}

#endif

	void
jp_init()
{
	JP_FCODE = JP_NEW;
	core_init();
}

/*
 * convert EUC letter into suitable letter.
 */
	void
kanjito(k1, k2, code)
	u_char	*k1, *k2;
	char	code;
{
	u_char c1, c2;

	switch(code)
	{
	  case JP_NJIS:
	  case JP_JIS:
		*k1 &= 0x7f;
		*k2 &= 0x7f;
		return;

	  case JP_SJIS:
	  	c1 = *k1 & 0x7f;
	  	c2 = *k2 & 0x7f;
	    *k1 = (((int)c1 - 1) >> 1) + ((c1 <= 0x5e) ? 0x71 : 0xb1);
		*k2 = c2 + ((c1 & 1) ? ((c2 < 0x60) ? 0x1f : 0x20) : 0x7e);
	    return;

	  default:
		return;
	}
}

/*
 * return kanji shift-in string
 */
	char *
kanjiin(code)
	char	code;
{
	switch(code)
	{
	  case JP_JIS:
		return "\033$@";

	  case JP_NJIS:
		return "\033$B";

	  default:
		return "";
	}
}

/*
 * return kanji shift-out string
 */
	char *
kanjiout(code)
	char	code;
{
	switch(code)
	{ 
	  case JP_NJIS:
		return "\033(B";

	  case JP_JIS:
		return "\033(J";

	  default:
		return "";
	}
}

/*
 * convert buffer to EUC code
 */
static int jis, euc, njis;
static int knj_asc;

	void
reset_jcount()
{
	jis = euc = njis =
	knj_asc = 0;
}

#ifdef JPFEP
	char*
guess_jiauto(ji, th)
	char* ji;
	int th;
{
	if (!th)
		return ji;
	if (knj_asc > th)
		return isupper(*ji) ? "J": "j";
	if (knj_asc < - th)
		return isupper(*ji) ? "A": "a";

	return ji;
}
#endif

	char
judge_jcode(code)
	char code;
{
	if (code == JP_ANY || code == JP_SANY)
	{
		char	judge;

		if (!jis && !euc && !njis)
			judge = JP_NONE;
		else if (njis)
			judge = JP_NJIS;
		else if (jis)
			judge = JP_JIS;
		else if (code == JP_ANY)
			judge = JP_EUC;
		else
			judge = JP_SJIS;
		return judge;
	}
	else
		return code;
}

static void sjis2euc(ptr)
	char *ptr;
{
	int i, j;

	i = (u_char) *ptr;
	j = (u_char) *(ptr + 1);

	i = (i * 2) - ((i <= 0x9f)? 0xe1 : 0x161);
	if (j < 0x9f)
		j -= ((j > 0x7f) ? 0x20 : 0x1f);
	else
	{
		i ++;
		j -= 0x7e;
	}
	*ptr++ = i | 0x80;
	*ptr   = j | 0x80;
}

	int
kanjiconvsfrom(ptr, code, kanji)
	char	*ptr;
	char	code;
	int		kanji;
{
	char	*top, *ptr2;
	u_char	c;
	int		sjislen = 0;
	static  char pre[3] = "";

	top = ptr;

	if (pre[0])
	{
		int ptrlen, prelen;
		ptrlen = strlen(ptr);
		prelen = strlen(pre);
		memmove(ptr + prelen, ptr, ptrlen + 1);
		memmove(ptr, pre, prelen);
		pre[0] = NUL;
	}

	switch(code)
	{
	  case JP_NONE:
		for(;*ptr; ptr++)
		{
			if (IsKanji(c = *ptr) && ! (c & 0x60))	/* if Function key */
				continue;

			 *ptr &= 0x7f;
			 knj_asc --;
		}
		return FALSE;

	  case JP_EUC:
		while(*ptr)
			if (IsKanji(c = *ptr++))
			{
				if (! (c & 0x60))	/* if Function key */
					continue;

				if (IsKanji(*ptr))
				{
					knj_asc ++;
					ptr++;
				}
				else if (*ptr)
					*ptr &= 0x7f;
				else
				{
					* -- ptr = NUL;
					pre[0] = c;
					pre[1] = NUL;
					return TRUE;
				}
			}
			else if (c > ' ')
				knj_asc --;

	  	return FALSE;
	
	  case JP_SJIS:
		while(*ptr)
			if (IsKanji(c = *ptr ++))
			{
				if (*ptr)
				{
					sjis2euc(ptr++ - 1);
					knj_asc ++;
				}
				else
				{
					* -- ptr = NUL;
					pre[0] = c;
					pre[1] = NUL;
					return TRUE;
				}
			}
			else if (c > ' ')
				knj_asc --;

	  	return FALSE;

	  case JP_NJIS:
	  case JP_JIS:
	  case JP_ANY:
	  case JP_SANY:

		ptr2 = ptr;
		while(*ptr)
		{
			if (IsKanji(c = *ptr ++))
			{
			 	if (! ((*ptr2 ++ = c) & 0x60) && code != JP_SANY)
					continue;					/* if Function key */

				if (code == JP_SANY && sjislen ++ == 0 && !*ptr)
					continue;					/* if Function key (SJIS) */

				if (! *ptr)						/* if not completed */
				{
					*-- ptr2 = NUL;
					pre[0] = c;
					pre[1] = NUL;
					return kanji;
				}

				*ptr2++ = *ptr++;

				euc++;
				knj_asc++;
				switch(code)
				{
					case JP_ANY:
						break;
					case JP_SANY:
						sjis2euc(ptr2 - 2);
						break;
					default:
						*(ptr2 - 2) &= 0x7f;
						*(ptr2 - 1) &= 0x7f;
				}
				continue;
			}

			if (kanji)
			{
				if (c == ESC)
				{
					if (*ptr == '(')
					{
					 	if (*++ptr == 'J' || *ptr == 'H' || *ptr == 'B' ||
						    code == JP_ANY) /* Kanji Out */
						{
							kanji = FALSE;
							ptr++;
							continue;
						}
						else if (! *ptr)
						{
							pre[0] = c;
							pre[1] = *(ptr - 1);
							pre[2] = NUL;
							*(ptr - 2) = NUL;
							return kanji;
						}
						else
							ptr --;
					}
					else if (! *ptr && ptr - 1 != top)
					{
						pre[0] = c;
						pre[1] = NUL;
						* -- ptr = NUL;
						return kanji;
					}
				}

				if (c > ' ')
				{
					*ptr2++ = c | 0x80;
					if (! *ptr)
					{
						* -- ptr2 = NUL;
						pre[0] = c;
						pre[1] = NUL;
						return kanji;
					}
					*ptr2++ = *ptr++ | 0x80;
					knj_asc ++;
				}
				else
				{
					*ptr2++ = c;
					if (*ptr == '\n' || *ptr == '\r')
						kanji = FALSE;
				}
			}

			else
			{
				if (c == ESC)
				{
					if (*ptr == '$')
					{
					 	if (*++ptr == '@' || *ptr == 'B' ||
							code == JP_ANY)		/* Kanji In */
						{
							if      (*ptr == '@') jis++;
							else if (*ptr == 'B') njis++;

							kanji = TRUE;
							ptr++;
							continue;
						}
						else if (!*ptr)
						{
							pre[0] = c;
							pre[1] = *(ptr - 1);
							pre[2] = NUL;
							*(ptr - 2) = NUL;
							return kanji;
						}
						else
							ptr--;
					}
					else if (! *ptr && ptr - 1 != top)
					{
						pre[0] = c;
						pre[1] = NUL;
						* -- ptr = NUL;
						return kanji;
					}
				}

				if (c > ' ')
					knj_asc --;

				*ptr2++ = c;
			}
		}
		*ptr2 = NUL;
		return kanji;
	}
	return FALSE;
}

/*
 * convert buffer from EUC code
 */
	void
kanjiconvto(lnum, lend, code)
	linenr_t	lnum, lend;
	char		code;
{
	char	*ptr, *new;

	for(; lnum <= lend; ++lnum)
	{
		ptr = nr2ptr(lnum);
		new = kanjiconvsto(ptr, code);
		if (new != ptr) free_line(replaceline(lnum, new));
	}
}

	char *
kanjiconvsto(ptr, code)
	char	*ptr;
	char	code;
{
	char	*top, *ptr2;
	char	*kin, *kout, *cp;
	int		kanji, nshift, nchar;

	top = ptr;

	switch(code)
	{
	  case JP_EUC:
	  	return top;
	
	  case JP_NONE:
		top = ptr2 = alloc_line(strlen(ptr));
		while(*ptr)
			 *ptr2 ++ = *ptr ++ & ~0x80;
		*ptr2 = NUL;
		return top;

	  case JP_SJIS:
		top = ptr2 = alloc_line(strlen(ptr));
		while(*ptr)
			if (IsKanji(*ptr2++ = *ptr++))
			{
				*ptr2 ++ = *ptr ++;
				kanjito(ptr2 - 2, ptr2 - 1, code);
			}
		*ptr2 = NUL;
		return top;

	  case JP_NJIS:
	  case JP_JIS:
	  case JP_ANY:

		kin  = kanjiin(code);
		kout = kanjiout(code);

		kanji = 0;
		for(nshift = 0; *ptr; ptr++)
		{
			if (kanji !=  (*ptr & 0x80))
			{
				nshift++;
				kanji ^= 0x80;
			}
		}
		if (kanji) nshift++;
		nchar = ptr - top;

		if (!nshift) return top;

		ptr = top;
		top = ptr2 = alloc_line(nchar + nshift * 3);

		kanji = 0;
		while(*ptr) 
		{
			if (kanji !=  (*ptr & 0x80))
			{
				if (kanji)
					for(cp = kout; *cp;)
						*ptr2++ = *cp++;
				else
					for(cp = kin; *cp;)
						*ptr2++ = *cp++;
				kanji ^= 0x80;
			}
			*ptr2++ = (*ptr++ & 0x7f);
		}
		if (kanji)
			for(cp = kout; *cp;)
				*ptr2++ = *cp++;
		*ptr2 = 0;
		return top;
	}
	return top;
}

	void
kanji_align()
{
	int col;
	char *line;

	line = nr2ptr(Curpos.lnum);
	for(col = 0; col < Curpos.col; line++, col++)
		if (IsKanji(*line))
		{
			line += 2;
			col  += 2;
		}
	Curpos.col = col;
}


/*
 *	Japanese Character class;
 *					Make sure this routine is consistent with search.c:cls().
 *
 * 	for Japanese
 * 3 - alphabet, digits
 * 4 - japanese hiragana
 * 5 - japanese katakana
 * 6 - symbols
 * 7 - other multi-char letter
 */

	int
jpcls(c, k)
	char c, k;
{
	if (c == ' ' || c == '\t' || c == NUL)
		return 0;

	if (IsKanji(c))
	{
		int		ret;
		return (ret = jptab[c & 0x7f].cls1) == JPC_KIGOU
				?	jptab[k & 0x7f].cls2
				:	ret;
	}

	if (isidchar(c))
		return 1;

	return -1;
}

/* 
 *	isjppunc(c, k) returns whether a kanji character ck necessary KINSOKU
 *	processing or not.
 */
	int
isjppunc(c, k, type)
	char c, k;
	int type;
{
	k &= 0x7f;
	switch(jptab[(int)c & 0x7f].cls1)
	{
	case JPC_KIGOU:
		return type ? jptab[(int)k].punccsym : jptab[(int)k].puncosym;

	case JPC_HIRA:
	case JPC_KATA:
		return type ? jptab[(int)k].puncckana : FALSE;

	default:
		return FALSE;
	}
}
/* 
 *	isaspunc(c, type) returns whether an ascii character ck necessary KINSOKU
 *	processing or not.
 */
	int
isaspunc(c, type)
	char c;
	int type;
{
	return(type ? jptab[(int)c].punccasc: jptab[(int)c].puncoasc);
}

/* 
 *	jptocase(&c, &k, tocase)
 *		modify c & k to case tocase
 *			tocase == UPPER : to upper
 *			tocase == LOWER : to lower
 *			tocase == others : swap case
 */
	void
jptocase(cp, kp, tocase)
	u_char *cp, *kp;
	int tocase;
{
	u_char k;

	k = *kp & 0x7f;
	switch(jptab[*cp & 0x7f].cls1)
	{
	case JPC_ALNUM:
		if (tocase != LOWER && islower(k))
			*kp = TO_UPPER(k) | 0x80;
		if (tocase != UPPER && isupper(k))
			*kp = TO_LOWER(k) | 0x80;
		return;

	case JPC_KIGOU:
		if (  (tocase != LOWER && jptab[k].scase == JLOS)
		   || (tocase != UPPER && jptab[k].scase == JUPS)
		   )
			*kp = jptab[k].swap;
		return;

	case JPC_KATA:
		if (tocase != -1)
			*cp = JP1_HIRA | 0x80;
		return;

	case JPC_HIRA:
		if (tocase != 1)
			*cp = JP1_KATA | 0x80;
		return;

	default:
		return;
	}
}

#endif /* JP */
