/*
 * Routines to manipulate the "line buffer".
 * The line buffer holds a line of output as it is being built
 * in preparation for output to the screen.
 */

#include "less.h"

/*
 * Common special variable for treating Kanji-code in line.c and output.c
 */
extern int treat_jis_code;	/* Treat JIS as one of the KANJI code */
extern int treat_ujis_code;	/* Treat UJIS as one of the KANJI code */
extern int treat_sjis_code;	/* Treat SJIS as one of the KANJI code */
extern int treat_all_kanji_code;/* Treat all KANJI code without KANA */
extern int output_jis;		/* Do you want to JIS as output kanji code? */

/*
 * Kanji convetion
 */
#define ISJIS(c)		(0x21 <= (c) && (c) <= 0x7e)
#define ISJISKANJI1(c,i)	((status & IN_JIS_KANJI) && (i) == 0 && \
				 ISJIS(c))
#define ISJISKANJI2(c,i)	((status & IN_JIS_KANJI) && (i) == 1 && \
				 ISJIS(c))
#define ISJISKANA1(c,i)		((status & IN_JIS_KANA) && ISJIS(c))
#define ISJISOTHER1(c,i)	((status & IN_JIS_OTHER) && 0)
#define ISJISOTHER2(c,i)	((status & IN_JIS_OTHER) && 0)
#define ISJISOTHER3(c,i)	((status & IN_JIS_OTHER) && 0)
#define ISJISOTHER4(c,i)	(0)
#define ISUJIS(c)		(0xa1 <= (c) && (c) <= 0xfe)
#define ISUJISKANJI1(c,i)	((i) == 0 && ISUJIS(c))
#define ISUJISKANJI2(c,i)	((i) == 1 && multibuf[0] != 0x8e && \
				 multibuf[0] != 0x8f && ISUJIS(c))
#define ISUJISKANA1(c,i)	((i) == 0 && (c) == 0x8e)
#define ISUJISKANA2(c,i)	((i) == 1 && multibuf[0] == 0x8e && ISUJIS(c))
#define ISUJISOTHER1(c,i)	((i) == 0 && (c) == 0x8f)
#define ISUJISOTHER2(c,i)	((i) == 1 && multibuf[0] == 0x8f && ISUJIS(c))
#define ISUJISOTHER3(c,i)	((i) == 2 && multibuf[0] == 0x8f && ISUJIS(c))
#define ISUJISOTHER4(c,i)	(0)
#define ISSJISKANJI1(c,i)	((i) == 0 && ((0x81 <= (c) && (c) <= 0x9f) || \
					      (0xe0 <= (c) && (c) <= 0xfc)))
#define ISSJISKANJI2(c,i)	((i) == 1 && \
				 0x40 <= (c) && (c) <= 0xfc && (c) != 0x7f)
#define ISSJISKANA1(c,i)	((i) == 1 && 0xa1 <= (c) && (c) <= 0xdf)
#define ISSJISOTHER1(c,i)	(0)
#define ISSJISOTHER2(c,i)	(0)
#define ISSJISOTHER3(c,i)	(0)
#define ISSJISOTHER4(c,i)	(0)

static int status = 0;
#define ESC_		(0001 << 8)
#define ESC_DOL 	(0002 << 8)
#define ESC_PAR		(0004 << 8)
#define ESC_MASK	(ESC_ | ESC_DOL | ESC_PAR)
#define IN_JIS_KANJI	(0010 << 8)
#define IN_JIS_KANA	(0020 << 8)
#define IN_JIS_OTHER	(0030 << 8)
static unsigned char multibuf[5];
static unsigned char multiattr[5];
static int multiindex = 0;	/* Index for multi bytes character */
static int multilen = 1;	/* Max length of currnet multi bytes char. */
static int first_ujis = 1;	/* If treat all kanji, first we check ujis code */

	static int
what_jis(c)
	int c;
{
	switch (multiindex)
	{
	case 0:
		if (ISJISKANJI1(c, multiindex))
		{
			multilen = 2;
			return KANJI1;
		}
		multilen = 1;
		if (ISJISKANA1(c, multiindex))
			return KANA1;
		if (ISJISOTHER1(c, multiindex))
			return OTHER1;
		return NORMAL;
	case 1:
		if (ISJISKANJI2(c, multiindex))
			return KANJI2;
		if (ISJISOTHER2(c, multiindex))
			return OTHER2;
		return NORMAL;
	case 2:
		if (ISJISOTHER3(c, multiindex))
			return OTHER3;
		return NORMAL;
	default:
		if (ISJISOTHER4(c, multiindex))
			return OTHER4;
		return NORMAL;
	}
}

	static int
what_ujis(c)
	int c;
{
	switch (multiindex)
	{
	case 0:
		if (ISUJISKANJI1(c, multiindex))
		{
			multilen = 2;
			return KANJI1;
		}
		if (ISUJISKANA1(c, multiindex))
		{
			multilen = 2;
			return KANA1;
		}
		if (ISUJISOTHER1(c, multiindex))
		{
			multilen = 3;
			return OTHER1;
		}
		multilen = 1;
		return NORMAL;
	case 1:
		if (ISUJISKANJI2(c, multiindex))
			return KANJI2;
		if (ISUJISKANA2(c, multiindex))
			return KANA2;
		if (ISUJISOTHER2(c, multiindex))
			return OTHER2;
		return NORMAL;
	case 2:
		if (ISUJISOTHER3(c, multiindex))
			return OTHER3;
		return NORMAL;
	default:
		if (ISUJISOTHER4(c, multiindex))
			return OTHER4;
		return NORMAL;
	}
}

	static int
what_sjis(c)
	int c;
{
	switch (multiindex)
	{
	case 0:
		if (ISSJISKANJI1(c, multiindex))
		{
			multilen = 2;
			return KANJI1;
		}
		multilen = 1;
		if (ISSJISKANA1(c, multiindex))
			return KANA1;
		if (ISSJISOTHER1(c, multiindex))
			return OTHER1;
		return NORMAL;
	case 1:
		if (ISSJISKANJI2(c, multiindex))
			return KANJI2;
		if (ISSJISOTHER2(c, multiindex))
			return OTHER2;
		return NORMAL;
	case 2:
		if (ISSJISOTHER3(c, multiindex))
			return OTHER3;
		return NORMAL;
	default:
		if (ISSJISOTHER4(c, multiindex))
			return OTHER4;
		return NORMAL;
	}
}

	static int
what_kanji(c)
	int c;
{
	if (status & ESC_MASK)
		return NORMAL;
	switch (multiindex)
	{
	case 0:
		if (((status & IN_JIS_KANJI) && 0x21 <= c && c <= 0x7e) ||
		    (0x81 <= c && c <= 0xfe && c != 0xa0))
		{
			multilen = 2;
			return KANJI1;
		}
		multilen = 1;
		return NORMAL;
	case 1:
		if (0x21 <= c && c <= 0xfe && c != 0x7f)
			return KANJI2;
		return NORMAL;
	default:
		return NORMAL;
	}
}

	static void
jis2ujis()
{
	register unsigned char *s = multibuf;
	register unsigned char *a = multiattr;

	if (ISJISKANJI1(*s, 0))
	{
		s[0] |= 0x80;
		s[1] |= 0x80;
	} else if (ISJISKANA1(*s, 0))
	{
		s[1] = s[0] | 0x80;
		s[0] = 0x8e;
		a[1] = KANA2;
		s[2] = '\0';
	} else if (ISJISOTHER1(*s, 0))
	{
		/*
		 * How to convert this?? all change to binary.
		 */
		a[0] = NORMAL;
		a[1] = NORMAL;
		a[2] = NORMAL;
		a[3] = NORMAL;
	}
}

	static void
jis2sjis()
{
	register int c1, c2;
	register unsigned char *s = multibuf;
	register unsigned char *a = multiattr;

	if (ISJISKANJI1(*s, 0))
	{
		c1 = (s[0] - 0x21) / 2 + 0x81;
		c2 = s[1] + ((s[0] & 1) ? 0x40 - 0x21 : 0x9e - 0x21);
		s[0] = c1 + (c1 >= 0xa0 ? 0x40 : 0);
		s[1] = c2 + (c2 >= 0x7f ? 1 : 0);
	} else if (ISJISKANA1(*s, 0))
	{
		s[0] |= 0x80;
	} else if (ISJISOTHER1(*s, 0))
	{
		/*
		 * How to convert this?? all change to binary.
		 */
		a[0] = NORMAL;
		a[1] = NORMAL;
		a[2] = NORMAL;
		a[3] = NORMAL;
	}
}

	static void
ujis2jis()
{
	register unsigned char *s = multibuf;
	register unsigned char *a = multiattr;

	if (ISUJISKANJI1(*s, 0))
	{
		s[0] &= ~0x80;
		s[1] &= ~0x80;
	} else if (ISUJISKANA1(*s, 0))
	{
		s[0] = s[1] & ~0x80;
		s[1] = '\0';
	} else if (ISUJISOTHER1(*s, 0))
	{
		/*
		 * How to convert this?? all change to binary.
		 */
		a[0] = NORMAL;
		a[1] = NORMAL;
		a[2] = NORMAL;
		a[3] = NORMAL;
	}
}

	static void
ujis2sjis()
{
	register int c1, c2;
	register unsigned char *s = multibuf;
	register unsigned char *a = multiattr;

	if (ISUJISKANJI1(*s, 0))
	{
		s[0] &= ~0x80;
		s[1] &= ~0x80;
		c1 = (s[0] - 0x21) / 2 + 0x81;
		c2 = s[1] + ((s[0] & 1) ? 0x40 - 0x21 : 0x9e - 0x21);
		s[0] = c1 + (c1 >= 0xa0 ? 0x40 : 0);
		s[1] = c2 + (c2 >= 0x7f ? 1 : 0);
	} else if (ISUJISKANA1(*s, 0))
	{
		s[0] = s[1];
		s[1] = '\0';
	} else if (ISUJISOTHER1(*s, 0))
	{
		/*
		 * How to convert this?? all change to binary.
		 */
		a[0] = NORMAL;
		a[1] = NORMAL;
		a[2] = NORMAL;
		a[3] = NORMAL;
	}
}

	static void
sjis2jis()
{
	register int c1, c2;
	register unsigned char *s = multibuf;
	register unsigned char *a = multiattr;

	if (ISSJISKANJI1(*s, 0))
	{
		c1 = s[0] - (s[0] >= 0xe0 ? 0x40 : 0);
		c2 = s[1] - (s[1] >= 0x80 ? 1 : 0);
		s[0] = (c1 - 0x81) * 2 + (c2 >= 0x9e ? 1 + 0x21 : 0x21);
		s[1] = c2 - (c2 >= 0x9e ? 0x9e - 0x21 : 0x40 - 0x21);
	} else if (ISSJISKANA1(*s, 0))
	{
		*s &= ~0x80;
	} else if (ISSJISOTHER1(*s, 0))
	{
		/*
		 * How to convert this?? all change to binary.
		 */
		a[0] = NORMAL;
		a[1] = NORMAL;
		a[2] = NORMAL;
		a[3] = NORMAL;
	}
}

	static void
sjis2ujis()
{
	register int c1, c2;
	register unsigned char *s = multibuf;
	register unsigned char *a = multiattr;

	if (ISSJISKANJI1(*s, 0))
	{
		c1 = s[0] - (s[0] >= 0xe0 ? 0x40 : 0);
		c2 = s[1] - (s[1] >= 0x80 ? 1 : 0);
		s[0] = ((c1 - 0x81) * 2 + (c2 >= 0x9e ? 1 + 0x21 : 0x21)) | 0x80;
		s[1] = (c2 - (c2 >= 0x9e ? 0x9e - 0x21 : 0x40 - 0x21)) | 0x80;
	} else if (ISSJISKANA1(*s, 0))
	{
		s[1] = s[0];
		s[0] = 0x8e;
		a[1] = KANA2;
		s[2] = '\0';
	} else if (ISSJISOTHER1(*s, 0))
	{
		/*
		 * How to convert this?? all change to binary.
		 */
		a[0] = NORMAL;
		a[1] = NORMAL;
		a[2] = NORMAL;
		a[3] = NORMAL;
	}
}

	static int
kanji2kanji()
{
	register int c1 = multibuf[0];
	register int c2 = multibuf[1];
	enum { none, jis, ujis, sjis, } kanji;

	kanji = none;
	if (ISJISKANJI1(c1, 0) && ISJISKANJI2(c2, 1))
		kanji = jis;
	else
	{
		if (first_ujis)
		{
			if (ISUJISKANJI1(c1, 0) && ISUJISKANJI2(c2, 1))
			{
				first_ujis = 1;
				kanji = ujis;
			} else if (ISSJISKANJI1(c1, 0) && ISSJISKANJI2(c2, 1))
			{
				first_ujis = 0;
				kanji = sjis;
			}
		} else
		{
			if (ISSJISKANJI1(c1, 0) && ISSJISKANJI2(c2, 1))
			{
				first_ujis = 0;
				kanji = sjis;
			} else if (ISUJISKANJI1(c1, 0) && ISUJISKANJI2(c2, 1))
			{
				first_ujis = 1;
				kanji = ujis;
			}
		}
	}
	if (kanji == none)
		return (-1);	/* this is wrong multi bytes character */
	if (output_jis)
	{
		if (kanji == ujis)
			ujis2jis();
		else if (kanji == sjis)
			sjis2jis();
	} else if (treat_ujis_code)
	{
		if (kanji == jis)
			jis2ujis();
		else if (kanji == sjis)
			sjis2ujis();
	} else if (treat_sjis_code)
	{
		if (kanji == jis)
			jis2sjis();
		else if (kanji == ujis)
			ujis2sjis();
	}
	return (0);
}

	static MULTI
wrong_multi(strbuf, attrbuf)
	unsigned char **strbuf;
	unsigned char **attrbuf;
{
	register int i;

	status = 0;
	multilen = multiindex + 1;	/* for adjusting the buffer length */
	multibuf[multiindex] = '\0';
	for (i = 0; i < multiindex; i++)
		multiattr[i] = NORMAL;
	*strbuf = multibuf;
	*attrbuf = multiattr;
	multiindex = 0;
	return (SOME_CHARS);
}

	static MULTI
check_jis(c, strbuf, attrbuf)
	register int c;
	unsigned char **strbuf;
	unsigned char **attrbuf;
{
	if (multiindex)
	{
		if (status & ESC_DOL)
		{
			if (c == 'B' || c == '@')
				status = IN_JIS_KANJI;
			else
				return (wrong_multi(strbuf, attrbuf));
			multiindex = 0;
			return (ESCAPE_SEQUENCE);
		} else if (status & ESC_PAR)
		{
			if (c == 'J' || c == 'B')
				status = 0;
			else if (c == 'I')
				status = IN_JIS_KANA;
			else
				return (wrong_multi(strbuf, attrbuf));
			multiindex = 0;
			return (ESCAPE_SEQUENCE);
		} else if (status & ESC_)
		{
			if (c == '$')
				status = ESC_DOL;
			else if (c == '(')
				status = ESC_PAR;
			else
				return (wrong_multi(strbuf, attrbuf));
		} else
		{
			return (wrong_multi(strbuf, attrbuf));
		}
	} else if (c == '\016')
	{
		status = IN_JIS_KANA;
		multiindex = 0;
		return (ESCAPE_SEQUENCE);
	} else if (c == '\017')
	{
		status = 0;
		multiindex = 0;
		return (ESCAPE_SEQUENCE);
	} else if (c == '\033')
	{
		status = ESC_;
	} else
	{
		return (wrong_multi(strbuf, attrbuf));
	}
	multibuf[multiindex++] = c;
	return (IN_MULTI_CHAR);
}

	public void
init_multi()
{
	multiindex = 0;
	multilen = 1;
}

	public MULTI
conv_multi(c, strbuf, attrbuf)
	int c;
	unsigned char **strbuf;
	unsigned char **attrbuf;
{
	if (c < 0)
	{
		/*
		 * Flush out buffering characters.
		 */
		if (multiindex)
			return (wrong_multi(strbuf, attrbuf));
		multilen = 1;
		return (NO_FLUSH);
	}

	multibuf[multiindex] = c;

	if (treat_all_kanji_code)
	{
		if ((multiattr[multiindex] = what_kanji(c)) != NORMAL)
			goto multi;
		else if ((status & ESC_MASK) || c == '\033' ||
			 c == '\016' || c == '\017')
			return (check_jis(c, strbuf, attrbuf));
	} else
	{
		if (treat_jis_code)
		{
			if ((multiattr[multiindex] = what_jis(c)) != NORMAL)
				goto multi;
			else if ((status & ESC_MASK) || c == '\033' ||
				 c == '\016' || c == '\017')
				return (check_jis(c, strbuf, attrbuf));
		}
		if (treat_ujis_code)
		{
			if ((multiattr[multiindex] = what_ujis(c)) != NORMAL)
				goto multi;
		}
		if (treat_sjis_code)
		{
			if ((multiattr[multiindex] = what_sjis(c)) != NORMAL)
				goto multi;
		}
	}

	if (multiindex > 0)
		return (wrong_multi(strbuf, attrbuf));
	multiattr[0] = NORMAL;
	multibuf[1] = '\0';
	multilen = 1;
	*strbuf = multibuf;
	*attrbuf = multiattr;
	return (CHAR);
multi:
	multiindex++;
	if (multiindex == multilen)
	{
		multibuf[multiindex] = '\0';
		if (treat_all_kanji_code)
		{
			if (kanji2kanji() < 0)
				return (wrong_multi(strbuf, attrbuf));
		} else if (output_jis)
		{
			if (treat_ujis_code)
				ujis2jis();
			else if (treat_sjis_code)
				sjis2jis();
		} else if (treat_ujis_code)
		{
			if (treat_jis_code)
				jis2ujis();
		} else if (treat_sjis_code)
		{
			if (treat_jis_code)
				jis2sjis();
		}
		multiindex = 0;
		*strbuf = multibuf;
		*attrbuf = multiattr;
		return (MULTI_CHAR);
	}
	return (IN_MULTI_CHAR);
}

	public POSITION
adjust_multi()
{
	return (POSITION) multilen - 1;
}
