/*
 * Copyright (c) 1984,1985,1989,1994,1995  Mark Nudelman
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice in the documentation and/or other materials provided with 
 *    the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * Routines to manipulate the "line buffer".
 * The line buffer holds a line of output as it is being built
 * in preparation for output to the screen.
 */

#include "less.h"

public char linebuf[1024];	/* Buffer which holds the current output line */
public int size_linebuf = sizeof(linebuf);

static char attr[1024];		/* Extension of linebuf to hold attributes */
static int curr;		/* Index into linebuf */
static int column;		/* Printable length, accounting for
				   backspaces, etc. */
static int lno_indent;		/* Number of chars used for line number */
static int overstrike;		/* Next char should overstrike previous char */
static int is_null_line;	/* There is no current line */
static char pendc;
static POSITION pendpos;

static int do_append();
static int real_do_append();

extern int bs_mode;
extern int tabstop;
extern int linenums;
extern int ctldisp;
extern int twiddle;
extern int binattr;
extern int auto_wrap, ignaw;
extern int bo_s_width, bo_e_width;
extern int ul_s_width, ul_e_width;
extern int bl_s_width, bl_e_width;
extern int so_s_width, so_e_width;
extern int sc_width, sc_height;

/*
 * Rewind the line buffer.
 */
	public void
prewind(pos)
	POSITION pos;
{
	curr = 0;
	column = 0;
	overstrike = 0;
	is_null_line = 0;
	lno_indent = 0;
	pendc = '\0';
#if ISO
	prewind_multi(pos);
#endif
}

/*
 * Insert the line number (of the given position) into the line buffer.
 */
	public void
plinenum(pos)
	POSITION pos;
{
	register int lno;
	register int i;
	register int n;

	/*
	 * We display the line number at the start of each line
	 * only if the -N option is set.
	 */
	if (linenums != OPT_ONPLUS)
		return;

	/*
	 * Get the line number and put it in the current line.
	 * {{ Note: since find_linenum calls forw_raw_line,
	 *    it may seek in the input file, requiring the caller 
	 *    of plinenum to re-seek if necessary. }}
	 */
	lno = find_linenum(pos);

	sprintf(&linebuf[curr], "%6d", lno);
	n = strlen(&linebuf[curr]);
	column += n;
	for (i = 0;  i < n;  i++)
		attr[curr++] = 0;

	/*
	 * Append enough spaces to bring us to the next tab stop.
	 * {{ We could avoid this at the cost of adding some
	 *    complication to the tab stop logic in pappend(). }}
	 */
	if (tabstop == 0)
		tabstop = 1;
	do
	{
		linebuf[curr] = ' ';
		attr[curr++] = AT_NORMAL;
		column++;
	} while ((column % tabstop) != 0);
	lno_indent = column;
}

/*
 * Return the printing width of the start (enter) sequence
 * for a given character attribute.
 */
	int
attr_swidth(a)
	int a;
{
	switch (a & AT_MASK)
	{
	case AT_BOLD:		return (bo_s_width);
	case AT_UNDERLINE:	return (ul_s_width);
	case AT_BLINK:		return (bl_s_width);
	case AT_STANDOUT:	return (so_s_width);
	}
	return (0);
}

/*
 * Return the printing width of the end (exit) sequence
 * for a given character attribute.
 */
	int
attr_ewidth(a)
	int a;
{
	switch (a & AT_MASK)
	{
	case AT_BOLD:		return (bo_e_width);
	case AT_UNDERLINE:	return (ul_e_width);
	case AT_BLINK:		return (bl_e_width);
	case AT_STANDOUT:	return (so_e_width);
	}
	return (0);
}

/*
 * Return the printing width of a given character and attribute,
 * if the character were added to the current position in the line buffer.
 * Adding a character with a given attribute may cause an enter or exit
 * attribute sequence to be inserted, so this must be taken into account.
 */
	static int
pwidth(c, a)
	int c;
	int a;
{
	register int w;

#if ISO
	if (a & ESCAPE_SEQUENCE)
		/*
		 * Escape sequences aren't printable ones.
		 * Only status of terminal will be changed.
		 */
		return (0);
#endif

	if (c == '\b')
		/*
		 * Backspace moves backwards one position.
		 */
		return (-1);

	if (control_char(c))
		/*
		 * Control characters do unpredicatable things,
		 * so we don't even try to guess; say it doesn't move.
		 * This can only happen if the -r flag is in effect.
		 */
		return (0);

#if ISO
	/*
	 * Other characters may take some spaces,
	 * plus the width of any attribute enter/exit sequence.
	 * E.g.: the width of a first byte of 2-width multi bytes
	 * character must be 2, and width of rest must be 0.
	 */
	switch (a & WIDTH_MASK)
	{
	case HEAD_W1_CHAR:
	default:		w = 1; break;
	case HEAD_W2_CHAR:	w = 2; break;
	case HEAD_W3_CHAR:	w = 3; break;
	case REST_CHAR:		w = 0; break;
	}
#else
	/*
	 * Other characters take one space,
	 * plus the width of any attribute enter/exit sequence.
	 */
	w = 1;
#endif
	if (curr > 0 && (attr[curr-1] & AT_MASK) != (a & AT_MASK))
		w += attr_ewidth(attr[curr-1]);
	if (a && (curr == 0 || (attr[curr-1] & AT_MASK) != (a & AT_MASK)))
		w += attr_swidth(a);
	return (w);
}

/*
 * Delete the previous character in the line buffer.
 */
	static void
backc()
{
	curr--;
	column -= pwidth(linebuf[curr], attr[curr]);
}

/*
 * Append a character and attribute to the line buffer.
 */
	static int
storec(c, a, pos)
	int c;
	int a;
	POSITION pos;
{
	register int w;

#if HILITE_SEARCH
	if (is_hilited(pos, pos+1, 0))
		/*
		 * This character should be highlighted.
		 * Override the attribute passed in.
		 */
		a = AT_STANDOUT;
#endif
	w = pwidth(c, a);
	if (ctldisp > 0 && column + w + attr_ewidth(a) > sc_width)
		/*
		 * Won't fit on screen.
		 */
		return (1);

	if (curr >= sizeof(linebuf)-2)
		/*
		 * Won't fit in line buffer.
		 */
		return (1);

	/*
	 * Special handling for "magic cookie" terminals.
	 * If an attribute enter/exit sequence has a printing width > 0,
	 * and the sequence is adjacent to a space, delete the space.
	 * We just mark the space as invisible, to avoid having too
	 * many spaces deleted.
	 * {{ Note that even if the attribute width is > 1, we
	 *    delete only one space.  It's not worth trying to do more.
	 *    It's hardly worth doing this much. }}
	 */
	if (curr > 0 && (a & AT_MASK) != AT_NORMAL && 
		linebuf[curr-1] == ' ' && (attr[curr-1] & AT_MASK) == AT_NORMAL &&
		attr_swidth(a) > 0)
	{
		/*
		 * We are about to append an enter-attribute sequence
		 * just after a space.  Delete the space.
		 */
		attr[curr-1] = AT_INVIS;
		column--;
	} else if (curr > 0 && (attr[curr-1] & AT_MASK) != AT_NORMAL && 
		(attr[curr-1] & AT_MASK) != AT_INVIS && c == ' ' &&
		(a & AT_MASK) == AT_NORMAL && attr_ewidth(attr[curr-1]) > 0)
	{
		/*
		 * We are about to append a space just after an 
		 * exit-attribute sequence.  Delete the space.
		 */
		a = AT_INVIS;
		column--;
	}
	/* End of magic cookie handling. */

	linebuf[curr] = c;
	attr[curr] = a;
	column += w;
	return (0);
}

/*
 * Append a character to the line buffer.
 * Expand tabs into spaces, handle underlining, boldfacing, etc.
 * Returns 0 if ok, 1 if couldn't fit in buffer.
 */
	public int
pappend(c, pos)
	register int c;
	POSITION pos;
{
	if (pendc)
	{
		if (do_append(pendc, pendpos))
			/*
			 * Oops.  We've probably lost the char which
			 * was in pendc, since caller won't back up.
			 */
			return (1);
		pendc = '\0';
	}

	if (c == '\r' && bs_mode == BS_SPECIAL)
	{
		/*
		 * Don't put the CR into the buffer until we see 
		 * the next char.  If the next char is a newline,
		 * discard the CR.
		 */
		pendc = c;
		pendpos = pos;
		return (0);
	}

	return (do_append(c, pos));
}

	static int
do_append(c, pos)
	int c;
	POSITION pos;
{
#if ISO
	char *cbuf;
	char *abuf;
	int i, j;

	switch (buffering_multi(c, &cbuf, &abuf, &i))
	{
	case BIN_CHARS:
		/*
		 * Cannot understand buffering characters.  It would be
		 * some control or binary characters.  Flushs them, and
		 * retry to understanding by do_append with current new one.
		 */
		j = i;
		while (--i >= 0)
			if (real_do_append(*cbuf++, *abuf++, pos))
			{
				adjust_buffering_bytes(j - (i + 1) - 1);
				return (1);
			}
		return (do_append(c));
	case CHARS:
		/*
		 * Understand buffering character(s).  Check to fit it
		 * on screen, and flush them.
		 */
		while (--i >= 0)
			if (real_do_append(*cbuf++, *abuf++, pos))
				return (1);
	case NO_OUTPUT:
	default:
		/*
		 * Buffered this character.  To check as right sequence.
		 */
		return (0);
	}
#else
	return real_do_append(c, AT_NORMAL, pos);
#endif
}

	static int
real_do_append(c, a, pos)
	register int c, a;
	POSITION pos;
{
	register char *s;

#define	STOREC(c,a) \
	if (storec((c),(a),pos)) return (1); else curr++

#if ISO
	if (a & ESCAPE_SEQUENCE) {
		/*
		 * Must insert all escape sequence characters in linebuf.
		 * When overstriking, escape sequence is INSERTED.
		 */
		if (overstrike)
		{
			register int i;
			while ((attr[curr] & WIDTH_MASK) == REST_CHAR)
			{
				backc();
				overstrike++;
			}
			for (i = overstrike; i > 0; --i)
			{
				linebuf[curr + i] = linebuf[curr + i - 1];
				attr[curr + i] = attr[curr + i - 1];
			}
		}
		STOREC(c, a);
	} else
#endif
	if (overstrike && !(a & WRONG_CHAR))
	{
		/*
		 * Check about multi '\b' for multi bytes character.
		 */
		if (c == '\b')
			goto do_bs_char;
		else if (control_char(c))
			goto do_control_char;
		--overstrike;
		/*
		 * Overstrike the character at the current position
		 * in the line buffer.  This will cause either 
		 * underline (if a "_" is overstruck), 
		 * bold (if an identical character is overstruck),
		 * or just deletion of the character in the buffer.
		 */
		if (c != '_' && (a & WIDTH_MASK) != REST_CHAR &&
		    (attr[curr] & WIDTH_MASK) == REST_CHAR)
		{
			register int i, j;
			for (i = 1; (attr[curr - i] & WIDTH_MASK) == REST_CHAR;
			     i++)
				;
			if ((char)c == linebuf[curr - i])
			{
				for (j = 0; j < i; j++)
				{
					attr[curr] = (attr[curr] & WIDTH_MASK)
						| AT_BOLD;
					backc();
					overstrike++;
				}
				STOREC(c, (attr[curr] & WIDTH_MASK) | AT_BOLD);
			}
		} else if ((char)c == linebuf[curr] && 
			   (attr[curr] & WIDTH_MASK) == (a & WIDTH_MASK))
			STOREC(c, (attr[curr] & WIDTH_MASK) | AT_BOLD);
		else if (c == '_' && (a & WIDTH_MASK) == HEAD_W1_CHAR)
		{
			if ((attr[curr] & WIDTH_MASK) == REST_CHAR)
			{
				register int i = curr;
				while ((attr[--i] & WIDTH_MASK) == REST_CHAR)
					attr[i] = (attr[i] & WIDTH_MASK) |
						  AT_UNDERLINE;
				attr[i] = (attr[i] & WIDTH_MASK) |
					  AT_UNDERLINE;
			}
			STOREC(linebuf[curr], (attr[curr] & WIDTH_MASK) |
					      AT_UNDERLINE);
		} else if (linebuf[curr] == '_' &&
			   (attr[curr] & WIDTH_MASK) == HEAD_W1_CHAR)
			STOREC(c, (a & WIDTH_MASK) | AT_UNDERLINE);
		else if ((a & WIDTH_MASK) == REST_CHAR)
			STOREC(c, (a & WIDTH_MASK) |
				  (attr[curr - 1] & AT_MASK));
		else
			STOREC(c, a);
#if ISO
		/*
		 * When overstriking, if there are any escape sequence,
		 * it must be overwritten.  Delete them.
		 */
		if (overstrike && (attr[curr] & ESCAPE_SEQUENCE))
		{
			register int i, j;
			for (i = 0;
			     overstrike > 0 &&
			     (attr[curr + i] & ESCAPE_SEQUENCE);
			     i++, --overstrike)
				;
			for (j = 0; j < overstrike; j++)
			{
				linebuf[curr + j] = linebuf[curr + j + i];
				attr[curr + j] = attr[curr + j + i];
			}
		}
#endif
	} else if (c == '\b')
	{
	do_bs_char:
		switch (bs_mode)
		{
		case BS_NORMAL:
			STOREC(c, AT_NORMAL);
			break;
		case BS_CONTROL:
			goto do_control_char;
		case BS_SPECIAL:
			if (curr == 0)
				break;
#if ISO
			/*
			 * Back a position.  If there are escape sequence,
			 * back to a first place of them.
			 */
			do
			{
				backc();
				overstrike++;
			} while ((attr[curr] & ESCAPE_SEQUENCE) && curr > 0);
#else
			backc();
			overstrike++;
#endif
			break;
		}
	} else if (c == '\t') 
	{
		/*
		 * Expand a tab into spaces.
		 */
		if (tabstop == 0)
			tabstop = 1;
		do
		{
			STOREC(' ', AT_NORMAL);
		} while ((column % tabstop) != 0);
	} else if (control_char(c) || (a & WRONG_CHAR))
	{
	do_control_char:
#if ISO
		if (a & WRONG_CHAR)
		{
			s = prchar_binfmt(c);
			goto binout;
		} else
#endif
		if (ctldisp == 0)
		{
			/*
			 * Output as a normal character.
			 */
			STOREC(c, AT_NORMAL);
		} else 
		{
			/*
			 * Convert to printable representation.
			 */
			s = prchar(c);  
		binout:	a = binattr;

			/*
			 * Make sure we can get the entire representation
			 * of the character on this line.
			 */
			if (column + (int) strlen(s) + 
			    attr_swidth(a) + attr_ewidth(a) > sc_width)
				return (1);

			for ( ;  *s != 0;  s++)
				STOREC(*s, a);
		}
#if ISO
	} else if ((a & WIDTH_MASK) == REST_CHAR)
	{
		STOREC(c, (a & WIDTH_MASK) | (attr[curr - 1] & AT_MASK));
#endif
	} else
	{
		STOREC(c, a);
	}

	return (0);
}

/*
 * Terminate the line in the line buffer.
 */
	public void
pdone(endline, pos)
	int endline;
	POSITION pos;
{
#if ISO
	/*
	 * Check and flush some buffering characters.
	 */
	(void) do_append(-1);
#endif

	if (pendc && (pendc != '\r' || !endline))
		/*
		 * If we had a pending character, put it in the buffer.
		 * But discard a pending CR if we are at end of line
		 * (that is, discard the CR in a CR/LF sequence).
		 */
		(void) do_append(pendc, pendpos);

	/*
	 * Add a newline if necessary,
	 * and append a '\0' to the end of the line.
	 */
	if (column < sc_width || !auto_wrap || ignaw || ctldisp == 0)
	{
		linebuf[curr] = '\n';
		attr[curr] = AT_NORMAL;
		curr++;
	}
	linebuf[curr] = '\0';
	attr[curr] = AT_NORMAL;
#if ISO
	pdone_multi(pos);
#endif
}

/*
 * Get a character from the current line.
 * Return the character as the function return value,
 * and the character attribute in *ap.
 */
	public int
gline(i, ap)
	register int i;
	register int *ap;
{
	char *s;
	
	if (is_null_line)
	{
		/*
		 * If there is no current line, we pretend the line is
		 * either "~" or "", depending on the "twiddle" flag.
		 */
		*ap = AT_NORMAL;
		s = (twiddle) ? "~\n" : "\n";
		return (s[i]);
	}

	*ap = attr[i];
	return (linebuf[i] & 0377);
}

/*
 * Indicate that there is no current line.
 */
	public void
null_line()
{
	is_null_line = 1;
}

#if 1
/*
 * Analogous to forw_line(), but deals with "raw lines":
 * lines which are not split for screen width.
 * {{ This is supposed to be more efficient than forw_line(). }}
 */
	public POSITION
forw_raw_line(curr_pos, linep)
	POSITION curr_pos;
	char **linep;
{
	register char *p;
	register int c;
	POSITION new_pos;

	if (curr_pos == NULL_POSITION || ch_seek(curr_pos) ||
		(c = ch_forw_get()) == EOI)
		return (NULL_POSITION);

	p = linebuf;

	for (;;)
	{
		if (c == '\n' || c == EOI)
		{
			new_pos = ch_tell();
			break;
		}
		if (p >= &linebuf[sizeof(linebuf)-1])
		{
			/*
			 * Overflowed the input buffer.
			 * Pretend the line ended here.
			 * {{ The line buffer is supposed to be big
			 *    enough that this never happens. }}
			 */
			new_pos = ch_tell() - 1;
			break;
		}
		*p++ = c;
		c = ch_forw_get();
	}
	*p = '\0';
	if (linep != NULL)
		*linep = linebuf;
	return (new_pos);
}

/*
 * Analogous to back_line(), but deals with "raw lines".
 * {{ This is supposed to be more efficient than back_line(). }}
 */
	public POSITION
back_raw_line(curr_pos, linep)
	POSITION curr_pos;
	char **linep;
{
	register char *p;
	register int c;
	POSITION new_pos;

	if (curr_pos == NULL_POSITION || curr_pos <= ch_zero() ||
		ch_seek(curr_pos-1))
		return (NULL_POSITION);

	p = &linebuf[sizeof(linebuf)];
	*--p = '\0';

	for (;;)
	{
		c = ch_back_get();
		if (c == '\n')
		{
			/*
			 * This is the newline ending the previous line.
			 * We have hit the beginning of the line.
			 */
			new_pos = ch_tell() + 1;
			break;
		}
		if (c == EOI)
		{
			/*
			 * We have hit the beginning of the file.
			 * This must be the first line in the file.
			 * This must, of course, be the beginning of the line.
			 */
			new_pos = ch_zero();
			break;
		}
		if (p <= linebuf)
		{
			/*
			 * Overflowed the input buffer.
			 * Pretend the line ended here.
			 */
			new_pos = ch_tell() + 1;
			break;
		}
		*--p = c;
	}
	if (linep != NULL)
		*linep = p;
	return (new_pos);
}
#endif
