/*
#  poster.c  -  a generic PS scaling program
#  This program outputs a PostScript page to a given size, create tiles if necessary.
#	each tile will be 7.5" by 10" with crop marks.
#
######################
#  CopyRight Notice: #
######################
#
#  Feel free to distribute or use, provided that nothing is changed.
#  Please send comments, bug fixes to pyramid!daisy!cplai
#
#  Author:  C.P. Lai  (All rights reserved)
#  Creation Date: Dec 22, 1987
#  Modification History:  
#		by C.P. Lai on Aug 18, 1989  for BoundingBox Handling.
#
#
#  Usage:  Poster ps-file width height [row col] | lpr -Pps
#	where ps-file is the file name of the PostScript page
#		width and height are the size in inches
#		row and col are option if only one tile is to be printed.
#		default is to print all tiles
*/

#include <stdio.h>
#include "bool.h"
#define BUFSIZE 512
/* tile size */
#define PAGEW 7.5
#define PAGEH 10.0

main (argc, argv)
int argc;
char *argv[];
{
char *pagename;
float width, height;
int row, col, llx, lly, urx, ury;

	row = col = 0;
	/* accepts only two command formats */
	if (!(argc == 4 || argc == 6)) usage(argv[0]);	
	pagename = argv[1];
	if (sscanf (argv[2], "%f", &width) != 1) usage(argv[0]);
	if (sscanf (argv[3], "%f", &height) != 1) usage(argv[0]);
	if (argc == 6) {
		if (sscanf (argv[4], "%d", &row) != 1) usage(argv[0]);
		if (sscanf (argv[5], "%d", &col) != 1) usage(argv[0]);
		if (row<1 || col<1) usage(argv[0]);
	}
	getBoundingBox (pagename, &llx, &lly, &urx, &ury);
	printposter (pagename, llx, lly, urx, ury, width, height, row, col);

	exit (0);
}

/**************************************/
/* output usage message and terminate */
/**************************************/
usage (prog_name)
char *prog_name;
{
	fprintf (stderr, "Usage:\t%s ps-file width height [row col]\n", prog_name);
	fprintf (stderr, "\twhere ps-file is the file name of the PostScript page\n");
	fprintf (stderr, "\twidth and height are the size in inches\n");
	fprintf (stderr, "\trow and col are option if only one tile is to be printed.\n");
	fprintf (stderr, "\tdefault is to print all tiles.\n");
	exit (1);
}

/*********************************************/
/* output the poster, create tiles if needed */
/*********************************************/
printposter (pagename, llx, lly, urx, ury, width, height, row, col)
char *pagename;
int llx, lly, urx, ury, row, col;
float width, height; 
{
int nrow, ncol, imagew, imageh; 
float xscale, yscale, scale;

	imagew = urx - llx;
	imageh = ury - lly;
	xscale = (width * 72) / imagew;
	yscale = (height * 72) / imageh;
	scale = (xscale<yscale)?xscale:yscale;

	/* adjust actual width and height based on aspect ratio */
	width = imagew * scale / 72;
	height = imageh * scale / 72;

	ncol = (int) (width / PAGEW);
	/* add one column if exceed by one user space unit */
	if ((width-(ncol*PAGEW))*72 > 1) { ncol++; }
	nrow = (int) (height / PAGEH);
	/* add one row if exceed by one user space unit */
	if ((height-(nrow*PAGEH))*72 > 1) { nrow++; }

	if (row>nrow || col>ncol) {
		fprintf (stderr, "The image does not fall on grid (%d,%d)\n", row, col);
		exit (1);
	}

	printf ("%%!PS\t");
	if (row == 0 || col == 0)
 		printf ("All grids ");
	else
		printf ("Grid (%d,%d) ", row, col);
	printf ("of poster %s in %dx%d tiles at %f scale\n", 
		pagename, nrow, ncol, scale);

	printprolog (scale, llx, lly);
	if (row == 0 || col == 0) {
		/* all tile */
		for (row = 1; row <= nrow; row++)
			for (col = 1; col <= ncol; col++)
				tile (pagename, row, col);
	} else tile (pagename, row, col);
	printf ("%% End of Poster %s\n", pagename);
	return;
}

/*******************************************************/
/* output PS prolog of the scaling and tiling routines */
/*******************************************************/
printprolog (scale, llx, lly)
float scale;
int llx, lly;
{
printf ("%%%%Creator: poster - a tiling program by C.P. Lai\n\n");

printf ("\
/cropmark	%% x y cropmark -\n\
{	%%def\n\
	/y exch def\n\
	/x exch def\n\
	gsave\n\
	0 y rmoveto\n\
	0 y 2 mul rlineto \n\
	gsave\n\
	1 setgray 3 setlinewidth stroke	%% paint background white to highlite mark\n\
	grestore\n\
	0 setgray 0 setlinewidth stroke	%% use thinnest line on device\n\
	grestore\n\
	x 0 rmoveto\n\
	x 2 mul 0 rlineto \n\
	gsave\n\
	1 setgray 3 setlinewidth stroke	%% paint background white to highlite mark\n\
	grestore\n\
	0 setgray 0 setlinewidth stroke	%% use thinnest line on device\n\
} bind def\n\n");

printf ("\
%% usage: 	row col tileprolog ps-code tilepilog\n\
%% these procedures output the tile specified by row & col\n\
/tileprolog\n\
{ 	%%def\n\
	gsave\n\
	/colcount exch def\n\
	/rowcount exch def\n\
	gsave\n\
	pagewidth colcount 1 sub mul neg\n\
	pageheight rowcount 1 sub mul neg\n\
	translate\n\
	sfactor dup scale\n\
	llx neg lly neg translate\n\
	tiledict begin\n\
} bind def\n\n");

printf("\
/tilepilog\n\
{	%%def\n\
	end %% of tiledict\n\
	grestore\n\
	%% output the crop marks\n\
	0 0 moveto\n\
	-9 -9 cropmark\n\
	0 pageheight moveto\n\
	-9 9 cropmark\n\
	pagewidth 0 moveto\n\
	9 -9 cropmark\n\
	pagewidth pageheight moveto\n\
	9 9 cropmark\n\n\
	1 setgray\n\
	18 -18 moveto\n\
	0 9 rlineto\n\
	72 0 rlineto\n\
	0 -9 rlineto\n\
	closepath\n\
	fill\n\n\
	0 setgray\n\
	18 -18 moveto\n\
	(Grid \\( ) show\n\
	rowcount strg cvs show\n\
	( , ) show\n\
	colcount strg cvs show\n\
	( \\)) show\n\n\
	showpage\n\
	grestore\n\
} bind def\n\n");

printf("\
/sfactor %f def\n\
/llx %d def\n\
/lly %d def\n\
/leftmargin 36 def\n\
/botmargin 36 def\n\
/pagewidth %d def\n\
/pageheight %d def\n\n\
/strg 10 string def\n\
/tiledict 250 dict def\n\
tiledict begin\n\
/showpage {} def	%% delay users showpage until cropmark is printed.\n\
end\n\
%% end of prolog\n\n", scale, llx, lly, (int) (PAGEW * 72), (int) (PAGEH * 72));

printf ("\
/Times-Roman findfont 9 scalefont setfont\n\
%% readjust the origin to coincides with the cropping mark\n\
leftmargin botmargin translate\n\n");
return;
}

/*****************************/
/* output one tile at a time */
/*****************************/
tile (pagename, row, col)
char *pagename;
int row, col;
{
printf ("%d %d tileprolog\n", row, col);
printf ("%% Start of included ps code\n%% vvvvvvvvvvvv\n");
printfile (pagename);
printf ("%% ^^^^^^^^^^^^\n%% End of included ps code\n");
printf ("tilepilog\n");
return;
}

/******************************/
/* copy the PS file to output */
/******************************/
printfile (filename)
char* filename;
{
char tmp_byte;

	if (freopen (filename, "r", stdin) == NULL) {
		fprintf (stderr, "fail to open file %s\n", filename);
		printf ("/systemdict /showpage get exec\n");
		exit (1);
	}
	while ((tmp_byte = getchar()) != EOF)
		putchar (tmp_byte);
	return;
}

/*********************************************/
/* extract BoundingBox info from the PS file */
/*********************************************/
getBoundingBox (filename, llxp, llyp, urxp, uryp)
char *filename;
int *llxp, *llyp, *urxp, *uryp;
{
char buf[BUFSIZE];
int not_found;

	/* initialize to 0 0 612 792 for defaults */
	*llxp = *llyp = 0;
	*urxp = 612; *uryp = 792;

	if (freopen (filename, "r", stdin) == NULL) {
		fprintf (stderr, "fail to open file %s\n", filename);
		exit (1);
	}
	not_found = TRUE;
	while (not_found && (gets(buf) != NULL)) {
		/* cut short if not found */
		if (strncmp (buf, "%%EndComments", 13) == 0) {
			not_found = FALSE;
			break;
		}
		if (sscanf (buf, "%%%%BoundingBox: %d %d %d %d", llxp, llyp, urxp, uryp) == 4)
			not_found = FALSE;
	}
	return;
}
