/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/arc.c,v 2.0 91/03/05 12:46:37 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "color.e"
#include "cursor.e"
#include "file.e"
#include "grid.e"
#include "obj.e"
#include "pattern.e"
#include "poly.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#ifdef UC
#include "showsize.e"
#endif /* UC */

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define EXPAND_BBOX(bbox,x,y) \
   if ((x)<(bbox)->ltx) (bbox)->ltx=(x); if ((y)<(bbox)->lty) (bbox)->lty=(y); \
   if ((x)>(bbox)->rbx) (bbox)->rbx=(x); if ((y)>(bbox)->rby) (bbox)->rby=(y)

int	arcDrawn = FALSE;
#ifdef UC
#define	OFFSET 20
#endif /* UC */

static
int ArcDirection (xc, yc, x1, y1, x2, y2)
   int	xc, yc, x1, y1, x2, y2;
{
   register int		dx, dy, radius;
   register double	theta1, theta2;

   dx = x1-xc; dy = y1-yc;
   radius = (int)sqrt((double)(dx*dx+dy*dy));
   theta1 = atan2 ((double)(dy), (double)(dx));
   theta2 = atan2 ((double)(y2-yc), (double)(x2-xc));
   if (theta1 < 0) theta1 += 2*M_PI;
   if (theta2 < 0) theta2 += 2*M_PI;

   if (theta2 > theta1)
   {
      if (theta2-theta1 >= 2*M_PI-theta2+theta1)
         return (ARC_CCW);
      else
         return (ARC_CW);
   }
   else if (theta1 > theta2)
   {
      if (theta1-theta2 >= 2*M_PI-theta1+theta2)
         return (ARC_CW);
      else
         return (ARC_CCW);
   }
   else
      return (ARC_CCW);
}

void PointsToArc (xc, yc, x1, y1, x2, y2, dir, ltx, lty, w, h, angle1, angle2)
   int	xc, yc, x1, y1, x2, y2, dir;
   int	* ltx, * lty, * w, * h, * angle1, * angle2;
{
   register int	dx, dy, radius, theta1, theta2, d_theta;

   dx = x1-xc; dy = y1-yc;
   radius = (int)sqrt((double)(dx*dx+dy*dy));
   *ltx = xc-radius; *lty = yc-radius;
   *w = *h = 2*radius;
   theta1 = (int)(atan2 ((double)(dy),(double)(dx))/M_PI*(-180));
   theta2 = (y2-yc == 0 && x2-xc == 0) ?
         0 : (int)(atan2 ((double)(y2-yc),(double)(x2-xc))/M_PI*(-180));
   /* NOTE:  *angle1 must be between -180 degrees and +180 degrees */
   *angle1 = theta1*64;
   d_theta = theta2-theta1;
   switch (dir)
   {
      case ARC_CCW: if (d_theta < 0) d_theta = 360 + d_theta; break;
      case ARC_CW:  if (d_theta > 0) d_theta = d_theta - 360; break;
   }
   *angle2 = d_theta * 64;
}

void ArcRealX2Y2 (ArcPtr, RealX2, RealY2)
   register struct ArcRec	* ArcPtr;
   int				* RealX2, * RealY2;
{
   register double	angle_in_radian;
   int			w = ArcPtr->w, h = ArcPtr->h;

   angle_in_radian = (ArcPtr->angle1+ArcPtr->angle2)*M_PI/180/64;
   *RealX2 = ArcPtr->xc + round((w/2)*cos(angle_in_radian));
   *RealY2 = ArcPtr->yc - round((h/2)*sin(angle_in_radian));
}

void CalcArcBBox (ArcPtr, bbox)
   struct ArcRec		* ArcPtr;
   register struct BBRec	* bbox;
{
   register int	theta1, theta2;
   int		real_x2, real_y2, dir = ArcPtr->dir;
   int		ltx = ArcPtr->ltx, lty = ArcPtr->lty;
   int		w = ArcPtr->w, h = ArcPtr->h;
   int		pass_theta1 = FALSE, coverage = 0, angle;

   theta1 = (ArcPtr->angle1)/64;
   theta2 = theta1 + (ArcPtr->angle2)/64;

   ArcRealX2Y2 (ArcPtr, &real_x2, &real_y2);

   if (ArcPtr->fill == NONEPAT)
   {  /* don't counter the center of the arc */
      bbox->ltx = min(ArcPtr->x1,real_x2);
      bbox->lty = min(ArcPtr->y1,real_y2);
      bbox->rbx = max(ArcPtr->x1,real_x2);
      bbox->rby = max(ArcPtr->y1,real_y2);
   }
   else
   {
      bbox->ltx = min(ArcPtr->xc,min(ArcPtr->x1,real_x2));
      bbox->lty = min(ArcPtr->yc,min(ArcPtr->y1,real_y2));
      bbox->rbx = max(ArcPtr->xc,max(ArcPtr->x1,real_x2));
      bbox->rby = max(ArcPtr->yc,max(ArcPtr->y1,real_y2));
   }

   if (theta2 < -180) theta2 += 360;
   if (theta2 > 180) theta2 -= 360;

   if (theta1 < 0) theta1 += 360;
   if (theta2 < 0) theta2 += 360;

   if (dir == ARC_CCW)
   {
      angle = 0;
      while (angle < theta2 || !pass_theta1)
      {
         if (angle >= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            if (theta2 > theta1 && angle >= theta2) break;
            if (theta2 < theta1) angle -= 360;
         }
         if (pass_theta1) coverage |= 1 << (((angle+360)/90) % 4);
         angle = (angle == 360) ? 0 : (angle+90);
      }
   }
   else
   {
      angle = 360;
      while (angle > theta2 || !pass_theta1)
      {
         if (angle <= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            if (theta2 < theta1 && angle <= theta2) break;
            if (theta2 > theta1) angle += 360;
         }
         if (pass_theta1) coverage |= 1 << ((angle/90) % 4);
         angle = (angle == 0) ? 360 : (angle-90);
      }
   }
   if (coverage & 0x1) { EXPAND_BBOX(bbox,(int)(ltx+w),(int)(lty+h/2)); }
   if (coverage & 0x2) { EXPAND_BBOX(bbox,(int)(ltx+w/2),lty); }
   if (coverage & 0x4) { EXPAND_BBOX(bbox,ltx,(int)(lty+h/2)); }
   if (coverage & 0x8) { EXPAND_BBOX(bbox,(int)(ltx+w/2),(int)(lty+h)); }
}

static
void DumpArcPSPath (FP, xc, yc, xr, yr, dir, a1, a2, outline, blank1, blank2)
   FILE	* FP;
   int	xc, yc, xr, yr, dir, a1, a2, outline;
   char	* blank1, * blank2;
{
   fprintf (FP, "%snewpath\n", blank1);
   if (outline) fprintf (FP, "%s%1d %1d moveto\n", blank2, xc, yc);
   switch (dir)
   {
      case ARC_CCW:
         fprintf (FP, "%s%1d %1d %1d %1d %1d %1d tgifarc\n", blank2,
               xc, yc, xr, yr, a1, a2);
         break;
      case ARC_CW:
         fprintf (FP, "%s%1d %1d %1d %1d %1d %1d tgifarcn\n", blank2,
               xc, yc, xr, yr, a1, a2);
         break;
   }
   if (outline) fprintf (FP, "%s%1d %1d lineto ", blank2, xc, yc);
}

void DumpArcObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register struct ArcRec	* arc_ptr = ObjPtr->detail.a;
   int				i;
   int				fill, width, pen, dash, color_index;
   int				xc, yc, xr, yr, dir, angle1, angle2;

   fill = arc_ptr->fill;
   width = arc_ptr->width;
   pen = arc_ptr->pen;
   dash = arc_ptr->dash;
   xc = arc_ptr->xc; yc = arc_ptr->yc;
   xr = (int)(arc_ptr->w/2); yr = (int)(arc_ptr->h/2);
   dir = arc_ptr->dir;
   angle1 = -(int)(arc_ptr->angle1/64);
   angle2 = -(int)(arc_ptr->angle2/64) + angle1;

   if (fill == NONEPAT && pen == NONEPAT) return;

   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   switch (fill)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,TRUE,"","   ");
         fprintf (FP, "   fill\n");
         break;
      case BACKPAT:
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,TRUE,"","   ");
         fprintf (FP, "   closepath 1 setgray fill\n");
         if (colorDump)
            fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                  ((float)tgifColors[color_index].red/maxRGB),
                  ((float)tgifColors[color_index].green/maxRGB),
                  ((float)tgifColors[color_index].blue/maxRGB));
         else
            fprintf (FP, "   0 setgray\n");
         break;
      default:
         fprintf (FP, "gsave\n");
         if (!colorDump)
            fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n",fill);
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,TRUE,"   ","      ");
         if (colorDump)
         {
            fprintf (FP, "   closepath clip\n");
            DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "   fill\n");
         fprintf (FP, "grestore\n");
         break;
   }

   fprintf (FP, "%1d setlinewidth\n", widthOfLine[width]);
   if (dash != 0)
   {
      fprintf (FP, "[");
      for (i = 0; i < dashListLength[dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[dash][dashListLength[dash]-1]));
   }

   switch (pen)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,FALSE,"","   ");
         fprintf (FP, "stroke\n");
         break;
      case BACKPAT:
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,FALSE,"","   ");
         fprintf (FP, "1 setgray stroke 0 setgray\n");
         break;
      default:
         fprintf (FP, "gsave\n");
         if (!colorDump)
            fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n", pen);
         DumpArcPSPath (FP,xc,yc,xr,yr,dir,angle1,angle2,FALSE,"   ","      ");
         if (colorDump)
         {
            fprintf (FP, "   strokepath clip\n");
            DumpPatFill (FP, pen, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "   stroke\n");
         fprintf (FP, "grestore\n");
         break;
   }
   if (dash != 0) fprintf (FP, "[] 0 setdash\n");
   fprintf (FP, "1 setlinewidth\n\n");
}
 
void DrawArcObj (window, XOff, YOff, ObjPtr)
   Window		window;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   struct ArcRec	* arc_ptr = ObjPtr->detail.a;
   int			fill, width, pen, dash, pixel, real_x_off, real_y_off;
   int			ltx, lty, w, h, angle1, angle2;
   XGCValues		values;

#ifndef UC
   real_x_off = (XOff >> zoomScale) << zoomScale;
   real_y_off = (YOff >> zoomScale) << zoomScale;

   ltx = (arc_ptr->ltx - real_x_off) >> zoomScale;
   lty = (arc_ptr->lty - real_y_off) >> zoomScale;
   w = (arc_ptr->w) >> zoomScale;
   h = (arc_ptr->h) >> zoomScale;
#else /* UC */
   real_x_off = RealSize((ScreenSize(XOff ,  zoomScale)),  zoomScale);
   real_y_off = RealSize((ScreenSize(YOff ,  zoomScale)),  zoomScale);

   ltx = ScreenSize((arc_ptr->ltx - real_x_off) ,  zoomScale);
   lty = ScreenSize((arc_ptr->lty - real_y_off) ,  zoomScale);
   w = ScreenSize((arc_ptr->w) ,  zoomScale);
   h = ScreenSize((arc_ptr->h) ,  zoomScale);
#endif /* UC */
   angle1 = arc_ptr->angle1;
   angle2 = arc_ptr->angle2;

   fill = ObjPtr->detail.a->fill;
   width = ObjPtr->detail.a->width;
   pen = ObjPtr->detail.a->pen;
   dash = ObjPtr->detail.a->dash;
   pixel = colorPixels[ObjPtr->color];

   if (fill != 0)
   {
      values.foreground = (fill == 2) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillArc (mainDisplay, window, drawGC, ltx, lty, w, h, angle1, angle2);
   }
   if (pen != 0)
   {
      values.foreground = (pen == 2) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[pen];
#ifndef UC
      values.line_width = widthOfLine[width] >> zoomScale;
#else /* UC */
      values.line_width =ScreenSize( widthOfLine[width] ,  zoomScale);
#endif /* UC */
      if (dash != 0)
      {
         XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
               dashListLength[dash]);
         values.line_style = LineOnOffDash;
      }
      else
         values.line_style = LineSolid;
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple | GCLineWidth |
            GCLineStyle, &values);
      XDrawArc (mainDisplay, window, drawGC, ltx, lty, w, h, angle1, angle2);
   }
}

void UpdArcBBox (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   struct ArcRec	* arc_ptr = ObjPtr->detail.a;
   struct BBRec		bbox;
   int			w;

   ObjPtr->x = arc_ptr->xc;
   ObjPtr->y = arc_ptr->yc;

   CalcArcBBox (arc_ptr, &bbox);

   ObjPtr->bbox.ltx = ObjPtr->obbox.ltx = bbox.ltx;
   ObjPtr->bbox.lty = ObjPtr->obbox.lty = bbox.lty;
   ObjPtr->bbox.rbx = ObjPtr->obbox.rbx = bbox.rbx;
   ObjPtr->bbox.rby = ObjPtr->obbox.rby = bbox.rby;

   w = widthOfLine[arc_ptr->width];

   ObjPtr->bbox.ltx -= w;
   ObjPtr->bbox.lty -= w;
   ObjPtr->bbox.rbx += w;
   ObjPtr->bbox.rby += w;
}

static
void CreateArcObj (xc, yc, x1, y1, x2, y2, dir, ltx, lty, w, h, angle1, angle2)
   int	xc, yc, x1, y1, x2, y2, dir, ltx, lty, w, h, angle1, angle2;
{
   struct ArcRec	* arc_ptr;
   struct ObjRec	* obj_ptr;

   arc_ptr = (struct ArcRec *) calloc (1, sizeof(struct ArcRec));
   arc_ptr->fill = objFill;
   arc_ptr->width = lineWidth;
   arc_ptr->pen = penPat;
   arc_ptr->dash = curDash;

#ifndef UC
   arc_ptr->xc = (xc<<zoomScale)+drawOrigX;
   arc_ptr->yc = (yc<<zoomScale)+drawOrigY;
   arc_ptr->x1 = (x1<<zoomScale)+drawOrigX;
   arc_ptr->y1 = (y1<<zoomScale)+drawOrigY;
   arc_ptr->x2 = (x2<<zoomScale)+drawOrigX;
   arc_ptr->y2 = (y2<<zoomScale)+drawOrigY;
   arc_ptr->dir = dir;
   arc_ptr->ltx = (ltx<<zoomScale)+drawOrigX;
   arc_ptr->lty = (lty<<zoomScale)+drawOrigY;
   arc_ptr->w = w<<zoomScale;
   arc_ptr->h = h<<zoomScale;
#else /* UC */
   arc_ptr->xc = (RealSize(xc, zoomScale))+drawOrigX;
   arc_ptr->yc = (RealSize(yc, zoomScale))+drawOrigY;
   arc_ptr->x1 = (RealSize(x1, zoomScale))+drawOrigX;
   arc_ptr->y1 = (RealSize(y1, zoomScale))+drawOrigY;
   arc_ptr->x2 = (RealSize(x2, zoomScale))+drawOrigX;
   arc_ptr->y2 = (RealSize(y2, zoomScale))+drawOrigY;
   arc_ptr->dir = dir;
   arc_ptr->ltx = (RealSize(ltx, zoomScale))+drawOrigX;
   arc_ptr->lty = (RealSize(lty, zoomScale))+drawOrigY;
   arc_ptr->w = RealSize(w, zoomScale);
   arc_ptr->h = RealSize(h, zoomScale);
#endif /* UC */
   arc_ptr->angle1 = angle1; arc_ptr->angle2 = angle2;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   obj_ptr->detail.a = arc_ptr;

   UpdArcBBox (obj_ptr);

   obj_ptr->type = OBJ_ARC;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   AddObj (NULL, topObj, obj_ptr);
}
 
static
void ContinueArc (OrigX, OrigY)
   int	OrigX, OrigY;
{
   int 		grid_x, grid_y, first_x = 0, first_y = 0;
   int 		end_x, end_y, saved_x, saved_y;
   int 		done = FALSE, drawing_arc = FALSE;
   int		dir = INVALID, ltx, lty, w, h, angle1, angle2;
   char		msg[80];
   XGCValues	values;
   XEvent	input;
   XMotionEvent	* motion_ev;
#ifdef UC
   register int	showsize = showSize;
   register int	hidedragcursor = hideDragCursor;
   register int cursordefined = FALSE;
#endif /* UC */

   values.foreground = xorColorPixels[colorIndex];
   values.function = GXxor;
   values.fill_style = FillSolid;
   values.line_width = 0;
   values.line_style = LineSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = saved_x = OrigX;
   grid_y = saved_y = OrigY; 
   XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, saved_x, saved_y);

   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
#ifndef UC
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
#else /* UC */
         GrabModeAsync, GrabModeAsync, None, ((hidedragcursor)?None:handCursor), CurrentTime);

   if (showsize) showPoint (OrigX, OrigY, RIGHTUP);
#endif /* UC */
   
   Msg ("Please specify the start of an arc.");
   while (!done)
   {
      XNextEvent (mainDisplay, &input);
      if (input.type == ButtonPress)
      {
         if (drawing_arc)
         {
            XUngrabPointer (mainDisplay, CurrentTime);
            XDrawArc (mainDisplay, drawWindow, drawGC, ltx, lty, w, h,
                  angle1, angle2);
#ifdef UC
	    if (showsize) eraseSizeString ();
	    if (hidedragcursor && cursordefined)
	    {
		XDefineCursor (mainDisplay, drawWindow, drawCursor);
		cursordefined = FALSE;
		XFlush (mainDisplay);
	    }
#endif /* UC */
            done = TRUE;
            Msg ("");
         }
         else
         {
            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                  saved_x, saved_y);
            first_x = saved_x;
            first_y = saved_y;
            drawing_arc = TRUE;
            if (OrigX == grid_x && OrigY == grid_y)
            {  /* fake it as if the 1st point is ok but the 2nd point is bad */
               XUngrabPointer (mainDisplay, CurrentTime);
               grid_x = first_x;
               grid_y = first_y;
               done = TRUE;
            }
#ifdef UC
	    if (showsize) eraseSizeString ();
	    if (hidedragcursor && cursordefined)
	    {
		XDefineCursor (mainDisplay, drawWindow, drawCursor);
		cursordefined = FALSE;
	    }
#endif /* UC */
            Msg ("Please specify the end of the arc.");
         }
      }
      else if (input.type == MotionNotify)
      {
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
         if (grid_x != saved_x || grid_y != saved_y)
         {
            if (drawing_arc)
            {  /* finished with the center and the first point on the arc */
               if (dir == INVALID)
               {
#ifdef UC
		  if (showsize) showAngle (saved_x + OFFSET, saved_y, 0);
#endif /* UC */
                  dir = ArcDirection (OrigX, OrigY, first_x, first_y,
                        grid_x, grid_y);
                  ltx = OrigX; lty = OrigY; w = 0; h = 0; angle1 = angle2 = 0;
                  if (dir == ARC_CW)
                     sprintf (msg, "Please specify the end of the arc.  (%s).",
                           "clockwise");
                  else
                     sprintf (msg, "Please specify the end of the arc.  (%s).",
                           "counter-clockwise");
                  Msg (msg);
               }
               XDrawArc (mainDisplay, drawWindow, drawGC, ltx, lty, w, h,
                     angle1, angle2);
#ifdef UC
	       if (showsize) eraseSizeString ();
#endif /* UC */
               saved_x = grid_x;
               saved_y = grid_y;
               PointsToArc (OrigX, OrigY, first_x, first_y, saved_x, saved_y,
                     dir, &ltx, &lty, &w, &h, &angle1, &angle2);
               XDrawArc (mainDisplay, drawWindow, drawGC, ltx, lty, w, h,
                     angle1, angle2);
#ifdef UC
	       if (showsize) showAngle (saved_x + OFFSET, saved_y, angle2);
	       if (hidedragcursor && !cursordefined)
	       {
		   XDefineCursor (mainDisplay, drawWindow, dragCursor);
		   cursordefined = TRUE;
	       }
#endif /* UC */
            }
            else
            {  /* looking for the first point on the arc */
#ifdef UC
		if (showsize) eraseSizeString ();
		if (hidedragcursor &&
		    ! cursordefined && (OrigX != grid_x || OrigY != grid_y))
		{
		    XDefineCursor (mainDisplay, drawWindow, dragCursor);
		    cursordefined = TRUE;
		}
#endif /* UC */
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                     saved_x, saved_y);
               saved_x = grid_x;
               saved_y = grid_y;
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY,
                     saved_x, saved_y);
#ifdef UC
	       if (showsize) showLength (OrigX, OrigY, saved_x, saved_y, RIGHTUP);
#endif /* UC */
            }
         }
         MarkRulers (grid_x, grid_y);
      }
   }
   if (first_x != grid_x || first_y != grid_y)
   {
      CreateArcObj (OrigX, OrigY, first_x, first_y, saved_x, saved_y, dir,
            ltx, lty, w, h, angle1, angle2);
      DrawArcObj (drawWindow, drawOrigX, drawOrigY, topObj);
      arcDrawn = TRUE;
      SetFileModified (TRUE);
   }
}

void DrawArc (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   int		mouse_x, mouse_y, grid_x, grid_y;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      ContinueArc (grid_x, grid_y);
   }
}

void SaveArcObj (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   register struct ArcRec	* arc_ptr = ObjPtr->detail.a;

   fprintf (FP, "arc('%s',", colorMenuItems[ObjPtr->color]);
   fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,\
         %1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,",
         arc_ptr->fill, arc_ptr->width, arc_ptr->pen, arc_ptr->dash,
         arc_ptr->ltx, arc_ptr->lty, arc_ptr->xc, arc_ptr->yc,
         arc_ptr->x1, arc_ptr->y1, arc_ptr->x2, arc_ptr->y2,
         arc_ptr->dir, arc_ptr->w, arc_ptr->h, arc_ptr->angle1, arc_ptr->angle2,
         ObjPtr->id);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

void ReadArcObj (Inbuf, ObjPtr)
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register struct ArcRec	* arc_ptr;
   char				color_str[20], * s;
   int				fill, width, pen, dash, ltx, lty, w, h;
   int				xc, yc, x1, y1, x2, y2, dir, angle1, angle2;

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);
   arc_ptr = (struct ArcRec *) calloc (1, sizeof(struct ArcRec));

   if (fileVersion > 8)
   {
      sscanf (s, "%d , %d , %d , %d , %d , %d , %d , %d , %d , \
            %d, %d , %d , %d , %d , %d , %d , %d , %d",
            &fill, &width, &pen, &dash, &ltx, &lty, &xc, &yc, &x1, &y1,
            &x2, &y2, &dir, &w, &h, &angle1, &angle2, &((*ObjPtr)->id));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
   }

   arc_ptr->fill = fill;
   arc_ptr->width = width;
   arc_ptr->pen = pen;
   arc_ptr->dash = dash;

   arc_ptr->xc = xc;         arc_ptr->yc = yc;
   arc_ptr->x1 = x1;         arc_ptr->y1 = y1;
   arc_ptr->x2 = x2;         arc_ptr->y2 = y2;
   arc_ptr->dir = dir;
   arc_ptr->ltx = ltx;       arc_ptr->lty = lty;
   arc_ptr->w = w;           arc_ptr->h = h;
   arc_ptr->angle1 = angle1; arc_ptr->angle2 = angle2;

   (*ObjPtr)->detail.a = arc_ptr;
   UpdArcBBox (*ObjPtr);

   (*ObjPtr)->type = OBJ_ARC;
   (*ObjPtr)->color = FindColorIndex (color_str);
   (*ObjPtr)->dirty = FALSE;
}

void FreeArcObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   cfree (ObjPtr->detail.a);
   cfree (ObjPtr);
}
