/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/drawing.c,v 2.0 91/03/05 15:26:27 william Exp $";
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "align.e"
#include "animate.e"
#include "arc.e"
#include "attr.e"
#include "box.e"
#include "choice.e"
#include "copypaste.e"
#include "cursor.e"
#include "dialog.e"
#include "dup.e"
#include "edit.e"
#include "file.e"
#include "font.e"
#include "grid.e"
#include "group.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "obj.e"
#include "oval.e"
#include "pattern.e"
#include "poly.e"
#include "polygon.e"
#include "raster.e"
#include "rcbox.e"
#include "rect.e"
#include "ruler.e"
#include "scroll.e"
#include "select.e"
#include "setup.e"
#include "special.e"
#include "stk.e"
#include "stretch.e"
#include "text.e"
#include "xbitmap.e"
#ifdef KINPUT
#include "kconvert.e"
#endif /* KINPUT */
#ifdef UC
#include "mainmenu.e"
#endif /* UC */

#define O_VIS 4
#define O_INVIS 4
#define O_GRID (O_VIS+O_INVIS)

void SetDefaultDrawWinClipRecs ()
{
   XRectangle	recs[1];

#ifndef UC
   SetRecVals (recs[0], 0, 0, drawWinW, drawWinH);
#else /* UC */
   SetRecVals (recs[0], 0, 0,
	       ((zoomScale<0)? drawWinW<<(-zoomScale):drawWinW),
	       ((zoomScale<0)? drawWinH<<(-zoomScale):drawWinH));
#endif /* UC */
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, 1, YXBanded);
}

void SetDefaultIconWinClipRecs ()
{
   XRectangle	recs[1];

   SetRecVals (recs[0], 0, 0, iconWindowW, iconWindowH);
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, 1, YXBanded);
}

static
void DrawHorizOutline (Win, Y, X1, X2, XStart, XEnd, Pixel)
   Window	Win;
   int		Y, X1, X2, XStart, XEnd, Pixel;
   /* XStart and XEnd are the real place, X1 and X2 are on outline grid */
{
   register int	i;

   if (XStart-X1 < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, XStart, Y, X1+O_VIS-1, Y);
   for (i = X1+O_GRID; i < X2-O_GRID; i+= O_GRID)
      XDrawLine (mainDisplay, Win, defaultGC, i, Y, i+O_VIS-1, Y);
   if (X2-XEnd < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, X2-O_GRID, Y, XEnd, Y);
   else
      XDrawLine (mainDisplay, Win, defaultGC, X2-O_GRID, Y, X2-O_INVIS-1, Y);
}

static
void DrawVertOutline (Win, X, Y1, Y2, YStart, YEnd, Pixel)
   Window	Win;
   int		X, Y1, Y2, YStart, YEnd, Pixel;
   /* YStart and YEnd are the real place, Y1 and Y2 are on outline grid */
{
   register int	i;

   if (YStart-Y1 < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, X, YStart, X, Y1+O_VIS-1);
   for (i = Y1+O_GRID; i < Y2-O_GRID; i+= O_GRID)
      XDrawLine (mainDisplay, Win, defaultGC, X, i, X, i+O_VIS-1);
   if (Y2-YEnd < O_VIS)
      XDrawLine (mainDisplay, Win, defaultGC, X, Y2-O_GRID, X, YEnd);
   else
      XDrawLine (mainDisplay, Win, defaultGC, X, Y2-O_GRID, X, Y2-O_INVIS-1);
}

static
void DrawSymOutline (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   int	ltx, lty, rbx, rby, x_start, x_end, y_start, y_end, pixel;

   pixel = myFgPixel;

#ifndef UC
   ltx = ((ObjPtr->obbox.ltx - XOff - QUARTER_INCH) >> zoomScale) + 1;
   lty = ((ObjPtr->obbox.lty - YOff - QUARTER_INCH) >> zoomScale) + 1;
   rbx = ((ObjPtr->obbox.rbx - XOff + QUARTER_INCH) >> zoomScale) - 1;
   rby = ((ObjPtr->obbox.rby - YOff + QUARTER_INCH) >> zoomScale) - 1;
#else /* UC */
   ltx = (ScreenSize((ObjPtr->obbox.ltx - XOff - QUARTER_INCH),  zoomScale)) + 1;
   lty = (ScreenSize((ObjPtr->obbox.lty - YOff - QUARTER_INCH),  zoomScale)) + 1;
   rbx = (ScreenSize((ObjPtr->obbox.rbx - XOff + QUARTER_INCH),  zoomScale)) - 1;
   rby = (ScreenSize((ObjPtr->obbox.rby - YOff + QUARTER_INCH),  zoomScale)) - 1;
#endif /* UC */

   x_start = (ltx % O_GRID == 0) ? ltx : (int)(ltx / O_GRID) * O_GRID;
   x_end = (rbx % O_GRID == 0) ? rbx : ((int)(rbx / O_GRID) + 1) * O_GRID;
   DrawHorizOutline (Win, lty, x_start, x_end, ltx, rbx, pixel);
   DrawHorizOutline (Win, rby, x_start, x_end, ltx, rbx, pixel);
   y_start = (lty % O_GRID == 0) ? lty : (int)(lty / O_GRID) * O_GRID;
   y_end = (rby % O_GRID == 0) ? rby : ((int)(rby / O_GRID) + 1) * O_GRID;
   DrawVertOutline (Win, ltx, y_start, y_end, lty, rby, pixel);
   DrawVertOutline (Win, rbx, y_start, y_end, lty, rby, pixel);
}

void DrawObj (Win, ObjPtr)
   Window			Win;
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         DrawPolyObj (Win, drawOrigX, drawOrigY, ObjPtr); 
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_BOX:
         DrawBoxObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_OVAL:
         DrawOvalObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_TEXT: DrawTextObj (Win, drawOrigX, drawOrigY, ObjPtr); break;
      case OBJ_POLYGON:
         DrawPolygonObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_ARC:
         DrawArcObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_RCBOX:
         DrawRCBoxObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;
      case OBJ_XBM:
         DrawXBmObj (Win, drawOrigX, drawOrigY, ObjPtr);
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         break;

      case OBJ_SYM:
      case OBJ_ICON:
      case OBJ_GROUP:
         obj_ptr = ObjPtr->detail.r->last;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
            if (BBoxIntersect (obj_ptr->bbox, drawWinBBox))
               DrawObj (Win, obj_ptr);
         if (ObjPtr->type == OBJ_ICON && ObjPtr->dirty)
         {
            attr_ptr = ObjPtr->fattr;
            for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
               UpdTextBBox (attr_ptr->obj);
            AdjObjBBox (ObjPtr);
            UpdSelBBox ();
            ObjPtr->dirty = FALSE;
         }
         DrawAttrs(Win, drawOrigX, drawOrigY, ObjPtr->fattr);
         if (ObjPtr->type == OBJ_SYM) DrawSymOutline (Win, drawOrigX, drawOrigY,
               ObjPtr);
         break;
   }
}

void DrawPaperBoundary ()
{
   register int	x_end, y_end;

   if (drawOrigX+drawWinW > paperWidth)
   {
      x_end = OFFSET_X(paperWidth);
      if (drawOrigY+drawWinH > paperHeight)
      {
         y_end = OFFSET_Y(paperHeight);
         XDrawLine (mainDisplay, drawWindow, defaultGC, x_end, 0, x_end, y_end);
         XDrawLine (mainDisplay, drawWindow, defaultGC, 0, y_end, x_end, y_end);
      }
      else
         XDrawLine (mainDisplay, drawWindow, defaultGC, x_end, 0, x_end,
#ifndef UC
               drawWinH>>zoomScale);
#else /* UC */
               ScreenSize(drawWinH, zoomScale));
#endif /* UC */
   }
   else if (drawOrigY+drawWinH > paperHeight)
   {
      y_end = OFFSET_Y(paperHeight);
      XDrawLine (mainDisplay, drawWindow, defaultGC, 0, y_end,
#ifndef UC
            drawWinW>>zoomScale, y_end);
#else /* UC */
            ScreenSize(drawWinW, zoomScale), y_end);
#endif /* UC */
   }
}

void RedrawAnArea (BotObj, LtX, LtY, RbX, RbY)
   struct ObjRec	* BotObj;
   int			LtX, LtY, RbX, RbY;
   /* LtX, LtY, RbX, RbY are absolute coordinates */
{
   register struct ObjRec	* obj_ptr;
   struct BBRec			bbox;
   XRectangle			recs[1];

   bbox.ltx = LtX; bbox.lty = LtY;
   bbox.rbx = min(RbX, paperWidth);
   bbox.rby = min(RbY, paperHeight);

#ifndef UC
   SetRecVals (recs[0], OFFSET_X(LtX), OFFSET_Y(LtY), ((RbX-LtX)>>zoomScale)+1,
         ((RbY-LtY)>>zoomScale)+1);
#else /* UC */
   SetRecVals (recs[0], OFFSET_X(LtX), OFFSET_Y(LtY), (ScreenSize((RbX-LtX), zoomScale))+1,
         (ScreenSize((RbY-LtY), zoomScale))+1);
#endif /* UC */
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, 1, YXBanded);

   XClearArea (mainDisplay, drawWindow, OFFSET_X(LtX), OFFSET_Y(LtY),
#ifndef UC
         ((RbX-LtX)>>zoomScale)+1, ((RbY-LtY)>>zoomScale)+1, FALSE);
#else /* UC */
         (ScreenSize((RbX-LtX), zoomScale))+1, (ScreenSize((RbY-LtY), zoomScale))+1, FALSE);
#endif /* UC */

   if (paperWidth >= LtX && paperWidth < RbX ||
         paperHeight >= LtY && paperHeight < RbY)
      DrawPaperBoundary ();

#ifndef UC
   DrawGridLines (drawWindow, LtX, LtY, RbX+(1<<zoomScale), RbY+(1<<zoomScale));
#else /* UC */
   DrawGridLines (drawWindow, LtX, LtY, RbX+(RealSize(1, zoomScale)), RbY+(RealSize(1, zoomScale)));
#endif /* UC */

   for (obj_ptr = BotObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      if (BBoxIntersect (obj_ptr->bbox, bbox) &&
            BBoxIntersect (obj_ptr->bbox, drawWinBBox))
         DrawObj (drawWindow, obj_ptr);

   SetDefaultDrawWinClipRecs ();
}

static
int Inside (BBox1, BBox2)
   struct BBRec	BBox1, BBox2;
{
   return (BBox1.ltx >= BBox2.ltx && BBox1.lty >= BBox2.lty &&
         BBox1.rbx <= BBox2.rbx && BBox1.rby <= BBox2.rby);
}

void RedrawAreas (BotObj, LtX1, LtY1, RbX1, RbY1, LtX2, LtY2, RbX2, RbY2)
   struct ObjRec	* BotObj;
   int			LtX1, LtY1, RbX1, RbY1, LtX2, LtY2, RbX2, RbY2;
   /* note:  these coordinates are absolute */
{
   register struct ObjRec	* obj_ptr;
   int				ltx, lty, rbx, rby, rec1_slot, num;
   struct BBRec			bbox1, bbox2;
   XRectangle			recs[4];

   bbox1.ltx = LtX1; bbox1.lty = LtY1;
   bbox1.rbx = min(RbX1, paperWidth);
   bbox1.rby = min(RbY1, paperHeight);
   bbox2.ltx = LtX2; bbox2.lty = LtY2;
   bbox2.rbx = min(RbX2, paperWidth);
   bbox2.rby = min(RbY2, paperHeight);

   if (Inside (bbox1, bbox2))
   {
      RedrawAnArea (BotObj, LtX2, LtY2, RbX2, RbY2);
      return;
   }
   else if (Inside (bbox2, bbox1))
   {
      RedrawAnArea (BotObj, LtX1, LtY1, RbX1, RbY1);
      return;
   }

   XClearArea (mainDisplay, drawWindow, OFFSET_X(LtX1), OFFSET_Y(LtY1),
#ifndef UC
         ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY1)>>zoomScale)+1, FALSE);
#else /* UC */
         (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY1), zoomScale))+1, FALSE);
#endif /* UC */

   if (BBoxIntersect (bbox1, bbox2))
   {
      ltx = min(LtX1,LtX2); lty = min(LtY1,LtY2);
      rbx = max(RbX1,RbX2); rby = max(RbY1,RbY2);
      RedrawAnArea (BotObj, ltx, lty, rbx, rby);
      return;
   }

   if (LtY1 == LtY2)
   {
      rec1_slot = (LtX1 <= LtX2) ? 0 : 1;
#ifndef UC
      SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
            ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY1)>>zoomScale)+1);
      SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
            ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY2)>>zoomScale)+1);
#else /* UC */
      SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
            (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY1), zoomScale))+1);
      SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
            (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY2), zoomScale))+1);
#endif /* UC */
      num = 2;
   }
   else
   {
      if (LtY1 < LtY2)
      {
         if (RbY1 <= LtY2)
         {  /* y-extents do not intersect */
#ifndef UC
            SetRecVals (recs[0], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY1)>>zoomScale)+1);
            SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY2)>>zoomScale)+1);
#else /* UC */
            SetRecVals (recs[0], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY1), zoomScale))+1);
            SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY2), zoomScale))+1);
#endif /* UC */
            num = 2;
         }
         else if (RbY1 >= RbY2)
         {  /* box 2's y-extents is inside box 1's y-extents */
            rec1_slot = (LtX1 < LtX2) ? 0 : 1;  
#ifndef UC
            SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY1)>>zoomScale)+1);
            SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY2)>>zoomScale)+1);
#else /* UC */
            SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY1), zoomScale))+1);
            SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY2), zoomScale))+1);
#endif /* UC */
            num = 2;
         }
         else
         {  
#ifndef UC
            SetRecVals (recs[0], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ((RbX1-LtX1)>>zoomScale)+1, ((LtY2-LtY1)>>zoomScale)+1);
            if (LtX1 < LtX2)
            {
               SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY2),
                     ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY2)>>zoomScale)+1);
               SetRecVals (recs[2], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                     ((RbX2-LtX2)>>zoomScale)+1, ((RbY1-LtY2)>>zoomScale)+1);
#else /* UC */
            SetRecVals (recs[0], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((LtY2-LtY1), zoomScale))+1);
            if (LtX1 < LtX2)
            {
               SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY2),
                     (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY2), zoomScale))+1);
               SetRecVals (recs[2], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                     (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY1-LtY2), zoomScale))+1);
#endif /* UC */
            }
            else
            {
#ifndef UC
               SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                     ((RbX2-LtX2)>>zoomScale)+1, ((RbY1-LtY2)>>zoomScale)+1);
               SetRecVals (recs[2], OFFSET_X(LtX1), OFFSET_Y(LtY2),
                     ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY2)>>zoomScale)+1);
            }
            SetRecVals (recs[3], OFFSET_X(LtX2), OFFSET_Y(RbY1),
                  ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-RbY1)>>zoomScale)+1);
#else /* UC */
               SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                     (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY1-LtY2), zoomScale))+1);
               SetRecVals (recs[2], OFFSET_X(LtX1), OFFSET_Y(LtY2),
                     (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY2), zoomScale))+1);
            }
            SetRecVals (recs[3], OFFSET_X(LtX2), OFFSET_Y(RbY1),
                  (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-RbY1), zoomScale))+1);
#endif /* UC */
            num = 4;
         }
      }
      else
      {
         if (RbY2 <= LtY1)
         {  /* y-extents do not intersect */
#ifndef UC
            SetRecVals (recs[0], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY2)>>zoomScale)+1);
            SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY1)>>zoomScale)+1);
#else /* UC */
            SetRecVals (recs[0], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY2), zoomScale))+1);
            SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY1), zoomScale))+1);
#endif /* UC */
            num = 2;
         }
         else if (RbY2 >= RbY1)
         {  /* box 1's y-extents is inside box 2's y-extents */
            rec1_slot = (LtX1 < LtX2) ? 0 : 1;  
#ifndef UC
            SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-LtY1)>>zoomScale)+1);
            SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY2)>>zoomScale)+1);
#else /* UC */
            SetRecVals (recs[rec1_slot], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                  (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-LtY1), zoomScale))+1);
            SetRecVals (recs[!rec1_slot], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY2), zoomScale))+1);
#endif /* UC */
            num = 2;
         }
         else
         {  
#ifndef UC
            SetRecVals (recs[0], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  ((RbX2-LtX2)>>zoomScale)+1, ((LtY1-LtY2)>>zoomScale)+1);
#else /* UC */
            SetRecVals (recs[0], OFFSET_X(LtX2), OFFSET_Y(LtY2),
                  (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((LtY1-LtY2), zoomScale))+1);
#endif /* UC */
            if (LtX1 < LtX2)
            {
#ifndef UC
               SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                     ((RbX1-LtX1)>>zoomScale)+1, ((RbY2-LtY1)>>zoomScale)+1);
               SetRecVals (recs[2], OFFSET_X(LtX2), OFFSET_Y(LtY1),
                     ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY1)>>zoomScale)+1);
#else /* UC */
               SetRecVals (recs[1], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                     (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY2-LtY1), zoomScale))+1);
               SetRecVals (recs[2], OFFSET_X(LtX2), OFFSET_Y(LtY1),
                     (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY1), zoomScale))+1);
#endif /* UC */
            }
            else
            {
#ifndef UC
               SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY1),
                     ((RbX2-LtX2)>>zoomScale)+1, ((RbY2-LtY1)>>zoomScale)+1);
               SetRecVals (recs[2], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                     ((RbX1-LtX1)>>zoomScale)+1, ((RbY2-LtY1)>>zoomScale)+1);
            }
            SetRecVals (recs[3], OFFSET_X(LtX1), OFFSET_Y(RbY2),
                  ((RbX1-LtX1)>>zoomScale)+1, ((RbY1-RbY2)>>zoomScale)+1);
#else /* UC */
               SetRecVals (recs[1], OFFSET_X(LtX2), OFFSET_Y(LtY1),
                     (ScreenSize((RbX2-LtX2), zoomScale))+1, (ScreenSize((RbY2-LtY1), zoomScale))+1);
               SetRecVals (recs[2], OFFSET_X(LtX1), OFFSET_Y(LtY1),
                     (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY2-LtY1), zoomScale))+1);
            }
            SetRecVals (recs[3], OFFSET_X(LtX1), OFFSET_Y(RbY2),
                  (ScreenSize((RbX1-LtX1), zoomScale))+1, (ScreenSize((RbY1-RbY2), zoomScale))+1);
#endif /* UC */
            num = 4;
         }
      }
   }
   XSetClipRectangles (mainDisplay, drawGC, 0, 0, recs, num, YXSorted);

   if (paperWidth >= LtX1 && paperWidth < RbX1 ||
         paperHeight >= LtY1 && paperHeight < RbY1 ||
         paperWidth >= LtX2 && paperWidth < RbX2 ||
         paperHeight >= LtY2 && paperHeight < RbY2)
      DrawPaperBoundary ();

#ifndef UC
   DrawGridLines (drawWindow, LtX1, LtY1, RbX1+(1<<zoomScale),
         RbY1+(1<<zoomScale));
   DrawGridLines (drawWindow, LtX2, LtY2, RbX2+(1<<zoomScale),
         RbY2+(1<<zoomScale));
#else /* UC */
   DrawGridLines (drawWindow, LtX1, LtY1, RbX1+(RealSize(1, zoomScale)),
         RbY1+(RealSize(1, zoomScale)));
   DrawGridLines (drawWindow, LtX2, LtY2, RbX2+(RealSize(1, zoomScale)),
         RbY2+(RealSize(1, zoomScale)));
#endif /* UC */

   for (obj_ptr = BotObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      if (BBoxIntersect (obj_ptr->bbox, drawWinBBox) &&
            (BBoxIntersect (obj_ptr->bbox, bbox1) ||
            BBoxIntersect (obj_ptr->bbox, bbox2)))
         DrawObj (drawWindow, obj_ptr);
   }

   SetDefaultDrawWinClipRecs ();
}

Pixmap DrawAllOnPixmap (LtX, LtY, W, H)
   int	*LtX, *LtY, *W, *H;
{
   register int			i;
   register struct ObjRec	* obj_ptr;
   int				len, ltx, lty, rbx, rby, w, h;
   int				saved_draw_orig_x, saved_draw_orig_y;
   int				saved_draw_win_w, saved_draw_win_h;
   int                          saved_zoom_scale;
   char				msg[MAXSTRING];
   Pixmap			pixmap;
   XGCValues			values;

   ltx = botObj->bbox.ltx; lty = botObj->bbox.lty;
   rbx = botObj->bbox.rbx; rby = botObj->bbox.rby;
   for (obj_ptr = botObj->prev; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      if (obj_ptr->bbox.ltx < ltx) ltx = obj_ptr->bbox.ltx;
      if (obj_ptr->bbox.lty < lty) lty = obj_ptr->bbox.lty;
      if (obj_ptr->bbox.rbx > rbx) rbx = obj_ptr->bbox.rbx;
      if (obj_ptr->bbox.rby > rby) rby = obj_ptr->bbox.rby;
   }
   *W = w = rbx - ltx;
   *H = h = rby - lty;
   *LtX = ltx;
   *LtY = lty;

   saved_draw_orig_x = drawOrigX; saved_draw_orig_y = drawOrigY;
   saved_draw_win_w = drawWinW; saved_draw_win_h = drawWinH;
   saved_zoom_scale = zoomScale;

   drawOrigX = ltx; drawOrigY = lty;
   drawWinW = w; drawWinH = h;
   zoomScale = 0;

   pixmap = XCreatePixmap (mainDisplay, mainWindow, w, h, mainDepth);

   if (pixmap == None)
   {
      sprintf (msg, "Can not allocate pixmap of size %1dx%1d.", w, h);
      Msg (msg);
      return (None);
   }

   values.foreground = myBgPixel;
   values.function = GXcopy;
   values.fill_style = FillSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle, &values);
   XFillRectangle (mainDisplay, pixmap, drawGC, 0, 0, w, h);

   AdjSplineVs ();

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      DrawObj (pixmap, obj_ptr);

   drawOrigX = saved_draw_orig_x; drawOrigY = saved_draw_orig_y;
   drawWinW = saved_draw_win_w; drawWinH = saved_draw_win_h;
   zoomScale = saved_zoom_scale;

   AdjSplineVs ();

   return (pixmap);
}

void RedrawDrawWindow (BotObj)
   struct ObjRec	* BotObj;
{
   register struct ObjRec	* obj_ptr;

   DrawPaperBoundary ();
   RedrawGridLines ();

   for (obj_ptr = BotObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      if (PointInBBox (obj_ptr->x, obj_ptr->y, drawWinBBox) ||
            BBoxIntersect (obj_ptr->bbox, drawWinBBox))
         DrawObj (drawWindow, obj_ptr);
}

void ClearAndRedrawDrawWindow ()
{
   XClearWindow (mainDisplay, drawWindow);
   RedrawDrawWindow (botObj);
   RedrawCurText ();
   HighLightForward ();
}

void CleanUpDrawingWindow ()
{
   HighLightReverse ();
   RemoveAllSel ();
   DelAllObj ();
   ClearCurText ();
   SetCurChoice (NOTHING);
}

int ShortHand (input)
   XEvent	* input;
   /* returns BAD if the character is a <CONTROL> or a <META> character */
   /* returns INVALID if the character is a normal character */
   /* otherwise, returns the value of sub-functions, such as QuitProc () */
{
   int			x, y;
   char			buf[80];
   KeySym		key_sym;
   XComposeStatus	c_stat;
   XKeyEvent		* key_ev;

   key_ev = &(input->xkey);
   XLookupString (key_ev, buf, 80, &key_sym, &c_stat);
   if ((key_sym>='\040' && key_sym<='\177' ||
         key_sym>='\240' && key_sym<='\377') &&
#ifndef KINPUT       
         (key_ev->state & (ControlMask | Mod1Mask)))
#else /* KINPUT */
         (key_ev->state & (ControlMask | Mod1Mask | ShiftMask)))
#endif /* KINPUT */
   {
      Msg ("");
      if ((key_ev->state & ControlMask) && (!(key_ev->state & Mod1Mask)))
      {
         switch (buf[0])
         {
            case '\001': /*^A*/ SelAllObj (); break;
            case '\002': /*^B*/ BackProc (); break;
            case '\003': /*^C*/ ChangeDomain (); break;
            case '\004': /*^D*/ DupSelObj (); break;
#ifndef UC
            case '\005': /*^E*/ break;
#else /* UC */
            case '\005': /*^E*/ PushCurChoice (); break;
#endif /* UC */
            case '\006': /*^F*/ FrontProc (); break;
            case '\007': /*^G*/ GroupSelObj (); break;
            case '\010': /*^H*/ return (INVALID);
            case '\011': /*^I*/ Instantiate (); break;
            case '\012': /*^J*/ return (INVALID);
            case '\013': /*^K*/ PopIcon (); break;
            case '\014': /*^L*/ AlignSelObjs (); break;
            case '\015': /*^M*/ return (INVALID);
            case '\016': /*^N*/ NewProc (); break;
            case '\017': /*^O*/ OpenProc (); break;
            case '\020': /*^P*/ Dump (FALSE, ""); break;
            case '\021': /*^Q*/ return (QuitProc ());
            case '\022': /*^R*/ ClearAndRedrawDrawWindow (); break;
            case '\023': /*^S*/ SaveFile (); break;
            case '\024': /*^T*/ AlignSelToGrid (); break;
            case '\025': /*^U*/ UngroupSelObj (); break;
            case '\026': /*^V*/ PushIcon (); break;
            case '\027': /*^W*/ SetCurChoice (DRAWTEXT); break;
            case '\030': /*^X*/ DelAllSelObj (); break;
            case '\031': /*^Y*/ CopyToCutBuffer (); break;
            case '\032': /*^Z*/ return (AnimateProc ());
            case ',': /*^,*/ ScrollLeft (); break;
            case '.': /*^.*/ ScrollRight (); break;
            case '-': /*^-*/ PrintWithCommand (FALSE, ""); break;
#ifdef UC
	    case '\037': /*^/*/ ToggleDotOnGrid (); break;
#endif /* UC */
#ifdef KINPUT
	    case ';': /* ^; */ ToggleAutoKinput (); break;
#endif /* KINPUT */
         }
      }
      else if ((key_ev->state & Mod1Mask) && (!(key_ev->state & ControlMask)))
      {
         switch (buf[0])
         {
            case 'a': /*#A*/ AddAttrs (); break;
            case 'b': /*#B*/ return (ProbeProc ());
            case 'c': /*#C*/ RotateCounter (); break;
            case 'd': /*#D*/ DecGrid (); break;
            case 'e': /*#E*/ AnimateSel (); break;
            case 'f': /*#F*/ FlashSelColor (); break;
            case 'g': /*#G*/ ToggleGridShown (); break;
            case 'h': /*#H*/ FlipHorizontal (); break;
            case 'i': /*#I*/ IncGrid (); break;
            case 'j': /*#J*/ HideAllAttrNames (); break;
            case 'k': /*#K*/ SetCurChoice (NOTHING); break;
            case 'l': /*#L*/ CornerLoop (&x, &y); LineStyleMenu (x, y); break;
            case 'm': /*#M*/ MoveAttr (); break;
            case 'n': /*#N*/ ShowAllAttrNames (); break;
            case 'o': /*#O*/ ZoomOut (); break;
            case 'p': /*#P*/ ImportFile (); break;
            case 'q': /*#Q*/ SetCurChoice (DRAWPOLY); break;
            case 'r': /*#R*/ SetCurChoice (DRAWBOX); break;
            case 's': /*#S*/ return (SolveProc ());
            case 't': /*#T*/ DetachAttrs (); break;
            case 'u': /*#U*/ UndoDelete (); break;
            case 'v': /*#V*/ FlipVertical (); break;
            case 'w': /*#W*/ RotateClockWise (); break;
            case 'x': /*#X*/ return (EscapeProc ());
            case 'y': /*#Y*/ return (SimulateProc ());
            case 'z': /*#Z*/ ZoomIn (); break;
            case '0': /*#0*/ ChangeFontSize (0); break;
            case '1': /*#1*/ ChangeFontSize (1); break;
            case '2': /*#2*/ ChangeFontSize (2); break;
            case '3': /*#3*/ ChangeFontSize (3); break;
            case '4': /*#4*/ ChangeFontSize (4); break;
            case '5': /*#5*/ ChangeFontSize (5); break;
            case ',': /*#,*/ ScrollUp (); break;
            case '.': /*#.*/ ScrollDown (); break;
#ifdef UC
	    case '/': /*#/*/ ToggleShowSize (); break;
#endif /* UC */
         }
      }
      else if ((key_ev->state & Mod1Mask) && (key_ev->state & ControlMask))
      {
         switch (buf[0])
         {
            case '\001': /*^#A*/ AddPoint (); break;
            case '\002': /*^#B*/ ChangeFontStyle (STYLE_BR); break;
            case '\003': /*^#C*/ ChangeFontJust (JUST_C); break;
            case '\004': /*^#D*/ DeletePoint (); break;
            case '\005': /*^#E*/ SetCurChoice (DRAWRCBOX); break;
            case '\006': /*^#F*/ InvertXBitmaps (); break;
            case '\007': /*^#G*/ ToggleSnapOn (); break;
            case '\010': /*^#H*/ break;
            case '\011': /*^#I*/ MakeIconic (); break;
            case '\012': /*^#J*/ UnMakeIconic (); break;
            case '\013': /*^#K*/ ToggleColorPostScript (); break;
            case '\014': /*^#L*/ ChangeFontJust (JUST_L); break;
            case '\015': /*^#M*/ MakeSymbolic (); break;
            case '\016': /*^#N*/ UnMakeSymbolic (); break;
            case '\017': /*^#O*/ ChangeFontStyle (STYLE_NR); break;
            case '\020': /*^#P*/ ChangeFontStyle (STYLE_BI); break;
            case '\021': /*^#Q*/ SetCurChoice (DRAWPOLYGON); break;
            case '\022': /*^#R*/ ChangeFontJust (JUST_R); break;
            case '\023': /*^#S*/ SaveNewFile (); break;
            case '\024': /*^#T*/ ChangeFontStyle (STYLE_NI); break;
            case '\025': /*^#U*/ UpdateSymbols (); break;
            case '\026': /*^#V*/ SetCurChoice (DRAWCIRCLE); break;
            case '\027': /*^#W*/ ToggleAllSelLineType (); break;
            case '\030': /*^#X*/ ToggleWhereToPrint (); break;
            case '\031': /*^#Y*/ PasteFromCutBuffer (); break;
            case '\032': /*^#Z*/ SetCurChoice (DRAWARC); break;
            case '.': /*^#.*/ ImportXBitmapFile (); break;
         }
      }
#ifdef KINPUT
      else if (! noKanji && !(key_ev->state & Mod1Mask) && !(key_ev->state & ControlMask) && (key_ev->state & ShiftMask))
      {	  
         switch (buf[0])
         {
	    case ' ': /* ShiftSpace */
	     BeginKanjiConversion (mainDisplay, drawWindow, input);
	     break;
	    default:
	     return (INVALID);
	 }
      }
#endif /* KINPUT */
      return (BAD);
   }
   return (INVALID);
}

static
int SomethingDirty ()
{
   register struct ObjRec	* obj_ptr = topObj;

   for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->dirty)
         return (TRUE);
   return (FALSE);
}

int DrawingEventHandler (input)
   XEvent	* input;
{
   int		mouse_x, mouse_y, grid_x, grid_y;
   XEvent	ev;
   XButtonEvent	* button_ev;

#ifdef KINPUT
   if (enableNoEventMask)
      kconvertEventHandler (NoEventMask, mainDisplay, drawWindow, input);
   if (enablePropertyChangeMask)
      kconvertEventHandler (PropertyChangeMask, mainDisplay, drawWindow, input);
   if (enableClientMessage)
      kconvertEventHandler (ClientMessage, mainDisplay, drawWindow, input);
#endif /* KINPUT */   
   if (input->type == Expose)
   {
      XSync (mainDisplay, FALSE);
      while (XCheckWindowEvent (mainDisplay, drawWindow, ExposureMask, &ev)) ;

      if (topSel != NULL || SomethingDirty ())
         ClearAndRedrawDrawWindow ();
      else
      {
         RedrawDrawWindow (botObj);
         RedrawCurText ();
      }
      return (INVALID);
   }
   else if (input->type == MotionNotify)
   {
      while (XCheckWindowEvent (mainDisplay,drawWindow,PointerMotionMask,&ev)) ;

      mouse_x = (input->xmotion).x;
      mouse_y = (input->xmotion).y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      MarkRulers (grid_x, grid_y);
      return (INVALID);
   }

   if (input->type == ButtonPress)
   {
      button_ev = &(input->xbutton);
      if ((button_ev->button == Button3) && (button_ev->state & ShiftMask))
      {
         SetCurChoice (NOTHING);
         return (INVALID);
      }
      else if (button_ev->button == Button3)
         return (MainMenu (button_ev));
#ifdef UC
      else if (curChoice != DRAWTEXT && button_ev->button == Button2){
	      ModeMenu (button_ev->x_root, button_ev->y_root);
	      return (INVALID);
      }
#endif /* UC */
   }
#ifdef DEBUG
/* #ifdef WIRED
 *   else if (input->type ==VisibilityNotify)
 *   {
 *       if (mainMenuWiredDown && !createdMainMenu &&
 *	   input->xvisibility.state != VisibilityUnobscured)
 *       {
 *	   LowerMainMenuWindows();
 *       } else if (createdMainMenu)
 *	   createdMainMenu = FALSE;
 *       return (INVALID);
 *   }
 *#endif /_* INVALID *_/
 */
#endif /* DEBUG */

   Msg ("");
   switch(curChoice)
   {
      case NOTHING: Select (input); break;
      case DRAWTEXT: DrawText (input); break;
      case DRAWBOX: DrawBox (input); break;
      case DRAWCIRCLE: DrawOval (input); break;
      case DRAWPOLY: DrawPoly (input); break;
      case DRAWPOLYGON: DrawPolygon (input); break;
      case DRAWARC: DrawArc (input); break;
      case DRAWRCBOX: DrawRCBox (input); break;
   }
   return (INVALID);
}
