/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/file.c,v 2.0 91/03/05 14:17:43 william Exp $";
#endif

#include <sys/types.h>
#include <sys/file.h>
#include <stdio.h>
#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "align.e"
#include "arc.e"
#include "attr.e"
#include "box.e"
#include "button.e"
#include "choice.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "font.e"
#include "grid.e"
#include "group.e"
#include "mainloop.e"
#include "menu.e"
#include "msg.e"
#include "names.e"
#include "pattern.e"
#include "poly.e"
#include "polygon.e"
#include "prtgif.e"
#include "obj.e"
#include "oval.e"
#include "rcbox.e"
#include "rect.e"
#include "ruler.e"
#include "scroll.e"
#include "select.e"
#include "setup.e"
#include "special.e"
#include "stk.e"
#include "text.e"
#include "xbitmap.e"  
#ifdef KINPUT
#include "kconvert.e" 
#endif /* KINPUT */

#define CUR_VERSION 10
#define START_HAVING_ATTRS 8

extern char	* getenv ();

char	curFileName[MAXPATHLENGTH];
int	curFileDefined = FALSE;
int	fileVersion = INVALID;
int	importingFile = FALSE;
#ifndef UC
int	psDotsPerInch = 72;
char	*psXOffStr[MAXPAGESTYLES]={"0.55","0.4","0.28","0.2","0.565","0.4"};
float	psXOff[MAXPAGESTYLES] =   { 0.55,  0.4,  0.28,  0.2,  0.565,  0.4 };
char	*psYOffStr[MAXPAGESTYLES]={"10.4","-0.6","10.7","-0.3","10.4","-0.6"};
float	psYOff[MAXPAGESTYLES] =   { 10.4,  -0.6,  10.7,  -0.3,  10.4,  -0.6 };
char	*psScaleStr[MAXPAGESTYLES]={"0.565","0.565","0.28","0.28","1.13","1.13"};
float	psScale[MAXPAGESTYLES] =   { 0.565,  0.565,  0.28,  0.28,  1.13,  1.13 };
float	psPageWidthInInch[MAXPAGESTYLES] = { 7.5, 10, 15.5, 21, 3.75, 5 };
float	psPageHeightInInch[MAXPAGESTYLES] = { 10, 7.5, 21, 15.5, 5, 3.75 };
#else /* UC */
float	psDotsPerInch[INCH_CM]= {72.0, 72.567};
char	*psXOffStr[INCH_CM][MAXPAGESTYLES]=
		{{"0.55",   "0.4",    "0.28", "0.2",    "0.565",  "0.4"},
		 {"0.5543", "0.4031", "0.2822", "0.2016", "0.5694", "0.4031"}};
float	psXOff[INCH_CM][MAXPAGESTYLES] =
		{{0.55,   0.4,    0.28,  0.2,    0.565,  0.4 },
		 {0.5543, 0.4031, 0.2822, 0.2016, 0.5694, 0.4031}};
char	*psYOffStr[INCH_CM][MAXPAGESTYLES]=
		{{"10.4",   "-0.6",     "10.7",   "-0.3",    "10.4",   "-0.6"},
		 {"10.48", "-0.6047", "10.78", "-0.3024", "10.48", "-0.6047"}};
float	psYOff[INCH_CM][MAXPAGESTYLES] =
		{{10.4,   -0.6,     10.7,   -0.3,    10.4,   -0.6 },
		 {10.48, -0.6047, 10.78, -0.3024, 10.48, -0.6047}};
char	*psScaleStr[INCH_CM][MAXPAGESTYLES]=
		{{"0.565",  "0.565",  "0.28", "0.28", "1.13",   "1.13"},
		 {"0.5694", "0.5694", "0.2822", "0.2822", "1.1389", "1.1389"}};
float	psScale[INCH_CM][MAXPAGESTYLES] =
		{{0.565,  0.565,  0.28,  0.28,  1.13,  1.13 },
		 {0.5694, 0.5694, 0.2822, 0.2822, 1.1389, 1.1389}};
float	psPageWidthInInch[INCH_CM][MAXPAGESTYLES] =
		{{7.5,    10,   15.5,    21,     3.75,   5 },
		 {19,	  25.5, 39.5,    53.5,   9.55,   12.5}};
float	psPageHeightInInch[INCH_CM][MAXPAGESTYLES] =
		{{10,    7.5,    21,     15.5,   5,      3.75 },
		 {25.5,  19,     53.5,   39.5,   12.5,   9.55}};

extern  void Dump();
int	saveWithEps = FALSE;
#endif /* UC */

char	printCommand[MAXSTRING];

#ifdef KANJI
int	noKanji = FALSE;
int 	noEuc = TRUE;
#endif /* KANJI */
void ClearFileInfo ()
{
   curFileName[0] = '\0';
   curFileDefined = FALSE;
   *curSymDir = '\0';
}

int OkayToCreateFile (FileName)
   char	* FileName;
{
   FILE	* fp;

   if ((fp = fopen (FileName, "r")) == NULL) return (TRUE);
   fclose (fp);
   switch (YesNoCancel ("File exists, okay to overwrite? [ync](y)"))
   {
      case CONFIRM_YES: break;
      case CONFIRM_NO: return (FALSE);
      case CONFIRM_CANCEL: return (FALSE);
   }
   unlink (FileName);
   return (TRUE);
}

void Save (FP, BotObjPtr, Level)
   FILE			* FP;
   struct ObjRec	* BotObjPtr;
   int			Level;
{
   struct ObjRec	* obj_ptr;

   TieLooseEnds ();

   if (Level == 0)
   {
      fprintf (FP, "state(%1d,%1d,", pageStyle, CUR_VERSION);
      fprintf (FP, "%1d,%1d,%1d,", drawOrigX, drawOrigY, zoomScale);
      fprintf (FP, "%1d,%1d,%1d,", xyGrid, gridOn, colorIndex);
      fprintf (FP, "%1d,%1d,%1d,", horiAlign, vertAlign, lineWidth);
      fprintf (FP, "%1d,%1d,%1d,%1d,", curSpline, lineStyle, objFill, penPat);
      fprintf (FP, "%1d,%1d,%1d,%1d,", textJust, curFont, curStyle, curSize);
      fprintf (FP, "%1d,%1d).\n", curFontDPI, curDash);
      fprintf (FP, "%%\n");
      fprintf (FP, "%% @%s%s\n", "(#)$H", "eader$");
      fprintf (FP, "%%\n");
   }

   for (obj_ptr = BotObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_POLY: SavePolyObj (FP, obj_ptr); break;
         case OBJ_BOX: SaveBoxObj (FP, obj_ptr); break;
         case OBJ_OVAL: SaveOvalObj (FP, obj_ptr); break;
         case OBJ_TEXT: SaveTextObj (FP, obj_ptr); break;
         case OBJ_POLYGON: SavePolygonObj (FP, obj_ptr); break;
         case OBJ_ARC: SaveArcObj (FP, obj_ptr); break;
         case OBJ_RCBOX: SaveRCBoxObj (FP, obj_ptr); break;
         case OBJ_XBM: SaveXBmObj (FP, obj_ptr); break;
         case OBJ_GROUP: SaveGroupObj (FP, obj_ptr, Level); break;
         case OBJ_SYM: SaveCompObj (FP, obj_ptr, Level); break;
         case OBJ_ICON: SaveIconObj (FP, obj_ptr, Level); break;
      }
      if (obj_ptr->prev == NULL)
         if (Level == 0)
            fprintf (FP, ".\n");
         else
            fprintf (FP, "\n");
      else
         if (Level == 0)
            fprintf (FP, ".\n");
         else
            fprintf (FP, ",\n");
   }
   SetCurChoice (NOTHING);
}

#define OBJ_FILE_SAVED 0
#define SYM_FILE_SAVED 1

static
int SaveTmpFile (NewFileName)
   char	* NewFileName;
   /* return TRUE if file successfully saved */
{
   char			new_file_name[MAXPATHLENGTH], s[MAXPATHLENGTH];
   FILE			* fp;
   int			count = 0, status = INVALID;
   struct ObjRec	* obj_ptr;

   strcpy (new_file_name, NewFileName);

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->type == OBJ_SYM) count++;

   switch (count)
   {
      case 0: strcat (new_file_name, ".obj"); status = OBJ_FILE_SAVED; break;
      case 1: strcat (new_file_name, ".sym"); status = SYM_FILE_SAVED; break;
      default:
         Msg ("TOO MANY SYMBOLS!  Symbol file not saved.");
         return (INVALID);
   }

   unlink (new_file_name);

   if ((fp = fopen (new_file_name, "w")) == NULL)
   {
      sprintf (s, "Can not open '%s', file not saved.", new_file_name);
      Msg (s);
      return (INVALID);
   }

   sprintf (s, "Saving temporary file '%s' ...", new_file_name);
   Msg (s);
   Save (fp, botObj, 0);
   fclose (fp);
   sprintf (s, "Temporary file '%s' saved.", new_file_name);
   Msg (s);

   chmod (new_file_name, 0777);
   return (status);
}

void SaveNewFile ()
{
   char			new_file_name[MAXPATHLENGTH], s[MAXPATHLENGTH];
   char			new_full_name[MAXPATHLENGTH], tmp_str[MAXPATHLENGTH];
   char			name_without_ext[MAXPATHLENGTH];
   FILE			* fp;
   int			count = 0, len;
   struct ObjRec	* obj_ptr;

   Dialog ("Please Enter New File Name:", new_file_name);
   if (*new_file_name == '\0') return;
   len = strlen (new_file_name);

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->type == OBJ_SYM) count++;

   switch (count)
   {
      case 0:
         if (len >= 4)
         {
            if (strcmp (&new_file_name[len-4], ".sym") == 0)
            {
               Msg ("Can not save as a .sym file, no symbol defined.");
               return;
            }
            else if (strcmp (&new_file_name[len-4], ".obj") != 0)
            {
               strcpy (name_without_ext, new_file_name);
               strcat (new_file_name, ".obj");
            }
            else
            {
               strcpy (name_without_ext, new_file_name);
               name_without_ext[len-4] = '\0';
            }

            if (strlen (new_file_name) == 4)
            {
               Msg ("No file name specified.  File not saved.");
               return;
            }
         }
         else
         {
            strcpy (name_without_ext, new_file_name);
            strcat (new_file_name, ".obj");
         }
         break;
      case 1:
         if (len >= 4)
         {
            if (strcmp (&new_file_name[len-4], ".obj") == 0)
            {
               Msg ("Can not save as a .obj file, ther is a symbol defined.");
               return;
            }
            else if (strcmp (&new_file_name[len-4], ".sym") != 0)
            {
               strcpy (name_without_ext, new_file_name);
               strcat (new_file_name, ".sym");
            }
            else
            {
               strcpy (name_without_ext, new_file_name);
               name_without_ext[len-4] = '\0';
            }

            if (strlen (new_file_name) == 4)
            {
               Msg ("No file name specified.  File not saved.");
               return;
            }
         }
         else
         {
            strcpy (name_without_ext, new_file_name);
            strcat (new_file_name, ".sym");
         }
         break;
      default:
         Msg ("TOO MANY SYMBOLS!  Symbol file not saved.");
         return;
   }

   if (*new_file_name == '/')
      strcpy (new_full_name, new_file_name);
   else
      sprintf (new_full_name, "%s/%s", curDir, new_file_name);

   if (!OkayToCreateFile (new_full_name)) return;

   if ((fp = fopen (new_full_name, "w")) == NULL)
   {
      sprintf (s, "Can not open '%s', file not saved.", new_full_name);
      Msg (s);
      return;
   }

   strcpy (tmp_str, curDir);
   SetCurDir (new_full_name);
   curFileDefined = TRUE;

   switch (count)
   {
      case 0:
         *curSymDir = '\0';
         if ((strcmp (tmp_str, curDir) != 0) || (!NameInCurDir (curFileName)))
            UpdateDirInfo ();
         break;
      case 1:
         strcpy (curSymDir, curDir);
         if (!DirInSymPath (curDir)) UpdateSymInfo ();
         break;
   }

   sprintf (s, "Saving '%s/%s' ...", curDir, curFileName);
   Msg (s);
   Save (fp, botObj, 0);
   fclose (fp);
   sprintf (s, "File '%s/%s' saved.", curDir, curFileName);
   Msg (s);
   SetFileModified (FALSE);

#ifdef UC
   if (saveWithEps)
   {
       int saved_whereToPrint = whereToPrint;

       whereToPrint = LATEX_FIG;
       Dump (FALSE, "");
       whereToPrint = saved_whereToPrint;
   }
#endif /* UC */
   RedrawTitleWindow ();
}

void SaveFile ()
{
   int			i, len, count;
   struct ObjRec	* obj_ptr;
   FILE			* fp;
   char			ext[MAXPATHLENGTH], s[MAXPATHLENGTH];
   char			full_name[MAXPATHLENGTH];

   if (!curFileDefined)
   {
      SaveNewFile ();
      return;
   }

   len = strlen (curFileName);
   for (i = len-1; curFileName[i] != '.'; i--) ;
   strcpy (ext, &curFileName[i+1]);

   for (obj_ptr = topObj, count = 0; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (obj_ptr->type == OBJ_SYM) count++;
   switch (count)
   {
      case 0:
         if (strcmp (ext, "sym") == 0)
         {
            Msg ("No SYMBOL defined in symbol file.  Symbol file not saved.");
            return;
         }
         break;
      case 1:
         if (strcmp (ext, "obj") == 0)
         {
            Msg ("One SYMBOL defined in OBJECT file.  Object file not saved.");
            return;
         }
         break;
      default:
         if (strcmp (ext, "sym") == 0)
         {
            Msg ("Too many SYMBOLS in symbol file!  Symbol file not saved.");
            return;
         }
         break;
   }

   if (strcmp (ext, "sym") == 0)
      sprintf (full_name, "%s/%s", curSymDir, curFileName);
   else if (strcmp (ext, "obj") == 0)
      sprintf (full_name, "%s/%s", curDir, curFileName);

   if ((fp = fopen (full_name, "w")) == NULL)
   {
      sprintf (s, "Can not open '%s', file not saved.", full_name);
      Msg (s);
      return;
   }

   sprintf (s, "Saving '%s' ...", full_name);
   Msg (s);

   Save (fp, botObj, 0);

   fclose (fp);
   sprintf (s, "File '%s' saved.", full_name);
   Msg (s);
   SetFileModified (FALSE);
   if (!NameInCurDir (curFileName)) UpdateDirInfo ();
#ifdef UC
   if (saveWithEps)
   {
       int saved_whereToPrint = whereToPrint;

       whereToPrint = LATEX_FIG;
       Dump (FALSE, "");
       whereToPrint = saved_whereToPrint;
   }
#endif /* UC */
}

char * ParseStr (Str, C, Left)
   char	* Str, C, * Left;
{
   register char	* s = Str, * l = Left;
   register int		len = 0;

   while (*s != '\0' && *s != C)
   {
      *l++ = *s++;
      len++;
   }

   if (*s == C) s++;
   *l = '\0';

   while (len >= 2 && *Left == '\'' && *(l-1) == '\'')
   {
      *(--l) = '\0';
      strcpy (Left, &Left[1]);
      len -= 2;
   }
   return (s);
}

char * FindChar (C, Str)
   char	C;
   char	* Str;
   /* returns the address of the character right after C of the string Str */
{
   register char	* s = Str;

   while (*s != '\0' && *s != C) s++;

   if (*s == C) s++;
   return (s);
}

void ReadState (Inbuf, PRTGIF)
   char	* Inbuf;
   int	PRTGIF;
{
   char	* s;
   int	page_style;

   s = FindChar ('(', Inbuf);
   sscanf (s, "%d", &page_style);
   s = FindChar (',', s);
   if (*s == '\0')
      fileVersion = INVALID;
   else
      sscanf (s, "%d", &fileVersion);

   if (PRTGIF)
   {
      pageStyle = page_style;
      return;
   }

   if (!importingFile)
   {
      pageStyle = page_style;
      if (fileVersion >= 2)
      {
         s = FindChar (',', s);
         if (fileVersion <= 3)
         {
            sscanf (s, "%d , %d , %d , %d , %d , %d , %d , %d , %d , %d , \
                  %d , %d , %d , %d , %d , %d",
                  &drawOrigX, &drawOrigY, &zoomScale, &xyGrid, &gridOn,
                  &colorIndex, &horiAlign, &vertAlign, &lineWidth, &lineStyle,
                  &objFill, &penPat, &textJust, &curFont, &curStyle, &curSize);
            if (lineWidth == LINE_CURVED)
            {
               lineWidth = 0;
               curSpline = LT_SPLINE;
            }
            else
               curSpline = LT_STRAIGHT;
            curFontDPI = FONT_DPI_75;
            curDash = 0;
         }
         else if (fileVersion <= 7)
         {
            sscanf (s, "%d , %d , %d , %d , %d , %d , %d , %d , %d , \
                  %d , %d , %d , %d , %d , %d , %d , %d",
                  &drawOrigX, &drawOrigY, &zoomScale, &xyGrid, &gridOn,
                  &colorIndex, &horiAlign, &vertAlign, &lineWidth, &curSpline,
                  &lineStyle, &objFill, &penPat, &textJust, &curFont,
                  &curStyle, &curSize);
            curFontDPI = FONT_DPI_75;
            curDash = 0;
         }
         else if (fileVersion <= 8)
         {
            sscanf (s, "%d , %d , %d , %d , %d , %d , %d , %d , %d , \
                  %d , %d , %d , %d , %d , %d , %d , %d , %d",
                  &drawOrigX, &drawOrigY, &zoomScale, &xyGrid, &gridOn,
                  &colorIndex, &horiAlign, &vertAlign, &lineWidth, &curSpline,
                  &lineStyle, &objFill, &penPat, &textJust, &curFont,
                  &curStyle, &curSize, &curFontDPI);
            curDash = 0;
         }
         else
         {
            sscanf (s, "%d , %d , %d , %d , %d , %d , %d , %d , %d , \
                  %d , %d , %d , %d , %d , %d , %d , %d , %d , %d",
                  &drawOrigX, &drawOrigY, &zoomScale, &xyGrid, &gridOn,
                  &colorIndex, &horiAlign, &vertAlign, &lineWidth, &curSpline,
                  &lineStyle, &objFill, &penPat, &textJust, &curFont,
                  &curStyle, &curSize, &curFontDPI, &curDash);
         }
         if (colorIndex >= maxColors)
         {
            printf ("Can not find color #%1d, use '%s' instead.\n", colorIndex,
                  colorMenuItems[defaultColorIndex]);
            colorIndex = defaultColorIndex;
         }
         SetCanvasFont ();
      }

      UpdDrawWinWH ();
      UpdPageStyle (pageStyle);
      UpdDrawWinBBox ();

      DrawPaperBoundary ();
      RedrawGridLines ();
      RedrawRulers ();
      RedrawChoiceWindow ();
   }
}

void ReadObjAttrs (MinFileVersion, FP, ObjPtr, PRTGIF)
   FILE			* FP;
   struct ObjRec	* * ObjPtr;
   int			PRTGIF;
{
   struct AttrRec	* top_attr = NULL, * bot_attr = NULL, * attr_ptr;

   if (fileVersion <= MinFileVersion) return;

   while (ReadAttr (FP, &attr_ptr, PRTGIF))
   {
      attr_ptr->owner = *ObjPtr;
      attr_ptr->prev = NULL;
      attr_ptr->next = top_attr;
      if (top_attr == NULL)
         bot_attr = attr_ptr;
      else
         top_attr->prev = attr_ptr;
      top_attr = attr_ptr;
   }
   if (bot_attr != NULL) bot_attr->next = NULL;
   (*ObjPtr)->fattr = top_attr;
   (*ObjPtr)->lattr = bot_attr;
}

int ReadObj (FP, ObjPtr, PRTGIF)
   FILE			* FP;
   struct ObjRec	* * ObjPtr;
   int			PRTGIF;
{
   char			inbuf[MAXSTRING+1], obj_name[10], tmp_str[MAXSTRING+1];
   char			* line = NULL, * c_ptr, * s, * s1;
   int			len, id, cur_size, done = FALSE, allocated = FALSE;

   while (fgets (inbuf, MAXSTRING, FP) != NULL)
   {
      if (inbuf[0] == ']') return (FALSE);

      len = strlen(inbuf);
      if (inbuf[len-1] != '\r' && inbuf[len-1] != '\n')
      {  /* line longer than MAXSTRING characters */
         /* inbuf[MAXSTRING-1] == '\0' and len == MAXSTRING-1 now */
         cur_size = 2*(MAXSTRING-1);
         allocated = TRUE;
         line = (char *) calloc (cur_size, sizeof(char));
         strcpy (line, inbuf);
         c_ptr = &(line[MAXSTRING-1]);
         while (!done && fgets (inbuf, MAXSTRING, FP) != NULL)
         {
            len = strlen(inbuf);
            if (inbuf[len-1] == '\r' || inbuf[len-1] == '\n')
            {
               done = TRUE;
               inbuf[len-1] = '\0';
               strcpy (c_ptr, inbuf);
            }
            else
            {
               cur_size += MAXSTRING-1;
               line = (char *) realloc (line, cur_size);
               strcpy (c_ptr, inbuf);
               c_ptr += MAXSTRING-1;
            }
         }
         if (!done)
         {
            if (allocated) cfree (line);
            return (FALSE);
         }
      }
      else
      {
         line = inbuf;
         line[len-1] = '\0';
      }

      if (line[0] == '%')
      {  /* comment line, skip */
         continue;
      }

      ParseStr (line, '(', obj_name);
      if (strcmp (obj_name, "poly") == 0)
      {
         ReadPolyObj (line, ObjPtr);
         ReadObjAttrs (INVALID, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "box") == 0)
      {
         ReadBoxObj (line, ObjPtr);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "oval") == 0)
      {
         ReadOvalObj (line, ObjPtr);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "text") == 0)
      {
         ReadTextObj (FP, line, ObjPtr, PRTGIF);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "polygon") == 0)
      {
         ReadPolygonObj (line, ObjPtr);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "arc") == 0)
      {
         ReadArcObj (line, ObjPtr);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "rcbox") == 0)
      {
         ReadRCBoxObj (line, ObjPtr);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "xbm") == 0)
      {
         ReadXBmObj (FP, line, ObjPtr, PRTGIF);
         ReadObjAttrs (START_HAVING_ATTRS-1, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "group") == 0)
      {
         ReadGroupObj (FP, ObjPtr, PRTGIF);
         ReadObjAttrs (INVALID, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "sym") == 0)
      {
         ReadGroupObj (FP, ObjPtr, PRTGIF);
         (*ObjPtr)->type = OBJ_SYM;
         ReadObjAttrs (INVALID, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "icon") == 0)
      {
         ReadGroupObj (FP, ObjPtr, PRTGIF);
         (*ObjPtr)->type = OBJ_ICON;
         if (fgets (line, MAXSTRING, FP) == NULL)
         {
            if (allocated) cfree (line);
            return (FALSE);
         }

         strcpy(tmp_str, FindChar ('"', line));
         s = FindChar ('"', tmp_str);
         if (fileVersion != INVALID)
         {
            s1 = FindChar (',', s);
            sscanf (s1, "%d", &id);
            if (id >= objId) objId = id+1;
            (*ObjPtr)->id = id;
         }
         (*ObjPtr)->dirty = FALSE;
         *(--s) = '\0';
         strcpy ((*ObjPtr)->detail.r->s, tmp_str);
         ReadObjAttrs (INVALID, FP, ObjPtr, PRTGIF);
         AdjObjBBox (*ObjPtr);
         if (allocated) cfree (line);
         return (TRUE);
      }
      else if (strcmp (obj_name, "state") == 0)
      {
         ReadState (line, PRTGIF);
         *ObjPtr = NULL;
         if (allocated) cfree (line);
         return (TRUE);
      }
   }
   if (allocated) cfree (line);
   return (FALSE);
}

void ChangeDomain ()
{
   char 	domain_name[MAXPATHLENGTH], env_str[MAXPATHLENGTH];
   char 	s[MAXSTRING], s1[MAXSTRING], * c_ptr;

   if (SelectDomain (domain_name) == INVALID) return;

   sprintf (env_str, "TGIF_%s", domain_name);
   if ((c_ptr = getenv (env_str)) == NULL)
      ParseSymPath (".");
   else
      ParseSymPath (c_ptr);

   UpdateSymInfo ();

   strcpy (curDomainName, domain_name);
   sprintf (s, "Current domain is '%s'.", curDomainName);
   sprintf (s1, "Symbol path set to '%s'.", curDomainPath);
   TwoLineMsg (s, s1);
   RedrawTitleWindow ();
}

void ImportFile ()
{
   struct ObjRec	* obj_ptr, * saved_top_obj, * saved_bot_obj;
   char 		file_name[MAXPATHLENGTH], s[MAXPATHLENGTH];
   FILE			* fp;

   importingFile = TRUE;
   if (SelectFileNameToImport ("Please select a file to IMPORT ...",
         "obj", file_name) == INVALID)
   {
      importingFile = FALSE;
      return;
   }

   strcat (file_name, ".obj");

   if ((fp = fopen (file_name, "r")) == NULL)
   {
      sprintf (s, "Can not import '%s'.", file_name);
      Msg (s);
      importingFile = FALSE;
      return;
   }

   TieLooseEnds ();
   if (topSel != NULL) { HighLightReverse (); RemoveAllSel (); }
   saved_top_obj = topObj;
   saved_bot_obj = botObj;
   topObj = botObj = NULL;

   sprintf (s, "Importing '%s' ...", file_name);
   Msg (s);

   while (ReadObj (fp, &obj_ptr, FALSE))
      if (obj_ptr != NULL)
         AddObj (NULL, topObj, obj_ptr);

   fclose (fp);
   importingFile = FALSE;
   if (topObj != NULL) SetFileModified (TRUE);

   RedrawDrawWindow (botObj);
   SelAllObj ();

   if (botObj != NULL)
      botObj->next = saved_top_obj;
   else
      topObj = saved_top_obj;

   if (saved_top_obj != NULL)
   {
      saved_top_obj->prev = botObj;
      botObj = saved_bot_obj;
   }

   sprintf (s, "'%s' imported.", file_name);
   Msg (s);
}

void ImportXBitmapFile ()
{
   char 		file_name[MAXPATHLENGTH], s[MAXPATHLENGTH];
   unsigned int		w, h;
   int			rc, x_hot, y_hot;
   Pixmap		bitmap;

   importingFile = TRUE;
   if (SelectFileNameToImport ("Please select a XBitmap file to IMPORT ...",
         XBM_FILE_EXT, s) == INVALID)
   {
      importingFile = FALSE;
      return;
   }

   sprintf (file_name, "%s.%s", s, XBM_FILE_EXT);

   rc = XReadBitmapFile (mainDisplay, mainWindow, file_name, &w, &h, &bitmap,
         &x_hot, &y_hot);

   if (rc != BitmapSuccess)
   {
      sprintf (s, "Can not import XBitmap file '%s'.", file_name);
      Msg (s);
      importingFile = FALSE;
      return;
   }

   CreateXBmObj (w, h, bitmap);

   TieLooseEnds ();
   if (topSel != NULL) { HighLightReverse (); RemoveAllSel (); }
   SetNullCursor (drawWindow);

   PlaceTopObj ();
   SelectTopObj ();
   SetFileModified (TRUE);
   justDupped = FALSE;
   ShowCursor ();

   sprintf (s, "XBitmap file (%1dx%1d) '%s' imported.", w, h, file_name);
   Msg (s);
   importingFile = FALSE;
}

void LoadFile ()
{
   struct ObjRec	* obj_ptr;
   char 		file_name[MAXPATHLENGTH], s[MAXPATHLENGTH];
   char 		saved_cur_dir[MAXPATHLENGTH];
   FILE			* fp;

   strcpy (saved_cur_dir, curDir);
   if (SelectFileName ("Please select a file to OPEN ...",file_name) == INVALID)
      return;

   strcat (file_name, ".obj");

   if ((fp = fopen (file_name, "r")) == NULL)
   {
      sprintf (s, "Can not open '%s'.", file_name);
      Msg (s);
      return;
   }

   TieLooseEnds ();
   CleanUpDrawingWindow ();
   SetFileModified (FALSE);
   sprintf (s, "Loading '%s' ...", file_name);
   Msg (s);

   XClearWindow (mainDisplay, drawWindow);

   while (ReadObj (fp, &obj_ptr, FALSE))
      if (obj_ptr != NULL)
      {
         AddObj (NULL, topObj, obj_ptr);
         if (PointInBBox (obj_ptr->x, obj_ptr->y, drawWinBBox) ||
               BBoxIntersect (obj_ptr->bbox, drawWinBBox))
            DrawObj (drawWindow, obj_ptr);
      }

   if (topObj == NULL)
   {
      DrawPaperBoundary ();
      RedrawGridLines ();
   }

   fclose (fp);
   SetCurDir (file_name);
   *curSymDir = '\0';
   curFileDefined = TRUE;

   if (strcmp (saved_cur_dir, curDir) != 0 && DirInSymPath ("."))
      UpdateSymInfo ();

   sprintf (s, "Current file is '%s'.", file_name);
   Msg (s);

   RedrawTitleWindow ();
}

void DumpPatFill (FP, Fill, CellSize, BBox, Blanks)
   FILE		* FP;
   int		Fill, CellSize;
   struct BBRec	BBox;
   char		* Blanks;
{
   int	ltx, lty, rbx, rby;

   ltx = ((BBox.ltx % CellSize) == 0) ? BBox.ltx :
         ((BBox.ltx > 0) ? ((int)(BBox.ltx / CellSize))*CellSize :
         ((int)(BBox.ltx / CellSize)-1)*CellSize);
   lty = ((BBox.lty % CellSize) == 0) ? BBox.lty :
         ((BBox.lty > 0) ? ((int)(BBox.lty / CellSize))*CellSize :
         ((int)(BBox.lty / CellSize)-1)*CellSize);
   rbx = ((BBox.rbx % CellSize) == 0) ? BBox.rbx :
         ((BBox.rbx > 0) ? ((int)(BBox.rbx / CellSize)+1)*CellSize :
         ((int)(BBox.rbx / CellSize))*CellSize);
   rby = ((BBox.rby % CellSize) == 0) ? BBox.rby :
         ((BBox.rby > 0) ? ((int)(BBox.rby / CellSize)+1)*CellSize :
         ((int)(BBox.rby / CellSize))*CellSize);

   fprintf (FP, "%spat%1d %1d %1d %1d %1d %1d patfill\n",
         Blanks, Fill, CellSize, ltx, lty, rbx-ltx, rby-lty);
}

void DumpSymOutline (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   int  ltx, lty, rbx, rby;

   ltx = ObjPtr->obbox.ltx - QUARTER_INCH + 1;
   lty = ObjPtr->obbox.lty - QUARTER_INCH + 1;
   rbx = ObjPtr->obbox.rbx + QUARTER_INCH - 1;
   rby = ObjPtr->obbox.rby + QUARTER_INCH - 1;

   fprintf (FP, "gsave\n");
   fprintf (FP, "   0 setgray\n");
   fprintf (FP, "   [4 4] 0 setdash\n");
   fprintf (FP, "   newpath\n   %1d %1d moveto\n", ltx, lty);
   fprintf (FP, "      %1d %1d lineto\n", rbx, lty);
   fprintf (FP, "      %1d %1d lineto\n", rbx, rby);
   fprintf (FP, "      %1d %1d lineto\n", ltx, rby);
   fprintf (FP, "   closepath stroke\n");
   fprintf (FP, "grestore\n");
}

void DumpAttrs (FP, AttrPtr, PRTGIF)
   FILE				* FP;
   register struct AttrRec	* AttrPtr;
   int				PRTGIF;
{
   for ( ; AttrPtr != NULL; AttrPtr = AttrPtr->prev)
      if (AttrPtr->shown)
         DumpTextObj (FP, AttrPtr->obj, PRTGIF);
}

void DumpAllObj (FP, ObjPtr, PRTGIF)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
   int				PRTGIF;
{
   register struct ObjRec	* obj_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         DumpPolyObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_BOX:
         DumpBoxObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_OVAL:
         DumpOvalObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_TEXT: DumpTextObj (FP, ObjPtr, PRTGIF); break;
      case OBJ_POLYGON:
         DumpPolygonObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_ARC:
         DumpArcObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_RCBOX:
         DumpRCBoxObj (FP, ObjPtr);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_XBM:
         DumpXBmObj (FP, ObjPtr, PRTGIF);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         break;
      case OBJ_SYM:
      case OBJ_ICON:
      case OBJ_GROUP:
         obj_ptr = ObjPtr->detail.r->last;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
            DumpAllObj (FP, obj_ptr, PRTGIF);
         DumpAttrs (FP, ObjPtr->lattr, PRTGIF);
         if (ObjPtr->type == OBJ_SYM) DumpSymOutline (FP, ObjPtr);
         break;
   }
}

void DumpBBox (fp)
   FILE	* fp;
{
   register struct ObjRec	* obj_ptr;
   int				ltx, lty, rbx, rby;
   double			llx1, lly1, urx1, ury1;
   double			real_ps_dots_per_inch = 0.0;

   if ((obj_ptr = topObj) == NULL)
   {
      fprintf (fp, "%%%%BoundingBox: 0 0 0 0\n");
      printf ("Warning:  The PostScript bounding box is empty!\n");
      Msg ("Warning:  The PostScript bounding box is empty!\n");
      return;
   }

   ltx = obj_ptr->bbox.ltx; lty = obj_ptr->bbox.lty;
   rbx = obj_ptr->bbox.rbx; rby = obj_ptr->bbox.rby;

   for (obj_ptr = topObj->next; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      if (obj_ptr->bbox.ltx < ltx) ltx = obj_ptr->bbox.ltx;
      if (obj_ptr->bbox.lty < lty) lty = obj_ptr->bbox.lty;
      if (obj_ptr->bbox.rbx > rbx) rbx = obj_ptr->bbox.rbx;
      if (obj_ptr->bbox.rby > rby) rby = obj_ptr->bbox.rby;
   }

   switch (pageStyle)
   {
#ifndef UC
      case PORTRAIT: real_ps_dots_per_inch = psDotsPerInch; break;
      case LANDSCAPE: real_ps_dots_per_inch = psDotsPerInch; break;
      case HIGHPORT: real_ps_dots_per_inch = psDotsPerInch/2.0; break;
      case HIGHLAND: real_ps_dots_per_inch = psDotsPerInch/2.0; break;
      case SLIDEPORT: real_ps_dots_per_inch = psDotsPerInch*2.0; break;
      case SLIDELAND: real_ps_dots_per_inch = psDotsPerInch*2.0; break;
#else /* UC */
      case PORTRAIT: real_ps_dots_per_inch = psDotsPerInch[cmUnit]; break;
      case LANDSCAPE: real_ps_dots_per_inch = psDotsPerInch[cmUnit]; break;
      case HIGHPORT: real_ps_dots_per_inch = psDotsPerInch[cmUnit]/2.0; break;
      case HIGHLAND: real_ps_dots_per_inch = psDotsPerInch[cmUnit]/2.0; break;
      case SLIDEPORT: real_ps_dots_per_inch = psDotsPerInch[cmUnit]*2.0; break;
      case SLIDELAND: real_ps_dots_per_inch = psDotsPerInch[cmUnit]*2.0; break;
#endif /* UC */
   }

#ifndef UC
   llx1 = 1.0*ltx*real_ps_dots_per_inch/(PIX_PER_INCH);
   lly1 = -1.0*rby*real_ps_dots_per_inch/(PIX_PER_INCH);
   urx1 = 1.0*rbx*real_ps_dots_per_inch/(PIX_PER_INCH);
   ury1 = -1.0*lty*real_ps_dots_per_inch/(PIX_PER_INCH);
#else /* UC */
   llx1 = 1.0*ltx*real_ps_dots_per_inch/(PIX_PER_INCH_OR_CM);
#ifndef DEBUG
   lly1 = -1.0*(rby*real_ps_dots_per_inch/(PIX_PER_INCH_OR_CM) + 1.999);
   urx1 = 1.0*(rbx*real_ps_dots_per_inch/(PIX_PER_INCH_OR_CM) + 1.999);
#else /* DEBUG */
   lly1 = -1.0*(rby*real_ps_dots_per_inch/(PIX_PER_INCH_OR_CM) + 0.001);
   urx1 = 1.0*(rbx*real_ps_dots_per_inch/(PIX_PER_INCH_OR_CM) + 0.001);
#endif /* DEBUG */   
   ury1 = -1.0*lty*real_ps_dots_per_inch/(PIX_PER_INCH_OR_CM);
#endif /* UC */
#ifdef DEBUG
   printf("%.10f, %.10f, %.10f, %.10f\n", llx1,lly1,urx1,ury1);
#endif /* DEBUG */   
   fprintf (fp,"%%%%BoundingBox: %.3f %.3f %.3f %.3f\n",llx1,lly1,urx1,ury1);
}

static
void GenDump (PRTGIF, FileName)
   int	PRTGIF;
   char	* FileName;
{
   register struct ObjRec	* obj_ptr;
   char				cmd[MAXSTRING+1], tmp_str[MAXSTRING+1];
   char				tmp_file[MAXSTRING+1], ps_file[MAXSTRING+1];
   char				message[MAXSTRING+1];
   int				i, len;
   FILE				* fp, * fps;

#ifndef UC
   if (botObj == NULL) { Msg ("No object to print."); return; }
#else /* UC */
   if (botObj == NULL)
   {
       if (PRTGIF)
       {   printf ("No object to print"); return;}
       else
       {   Msg ("No object to print."); return;}
   }
#endif /* UC */

   if (!PRTGIF) Msg ("Generating print file ...");

   if (whereToPrint == XBM_FILE)
   {
      if (!curFileDefined)
      {
         if (colorDump)
            Dialog ("No current file.  Can not generate X11 pitmap output!",
                  cmd);
         else
            Dialog ("No current file.  Can not generate X11 bitmap output!",
                  cmd);
      }
      else
         DumpXBitmapFile ();
      return;
   }

   strcpy (tmp_file, "/tmp/TgifXXXXXX");
   mktemp (tmp_file);
   unlink (tmp_file);

   if ((fp = fopen (tmp_file, "w")) == NULL)
   {
      if (PRTGIF)
         printf ("Can not create '%s', print aborted.", tmp_file);
      else
      {
         sprintf (tmp_str, "Can not create '%s', print aborted.", tmp_file);
         Msg (tmp_str);
      }
      return;
   }

   if (PRTGIF) printf ("Writing to '%s' ...\n", tmp_file);

   fprintf (fp, "%%!\n");
   DumpBBox (fp);
   sprintf(ps_file, "%s/.psmac", drawPath);
   if ((fps = fopen (ps_file, "r")) == NULL) 
   {
      if (PRTGIF)
         printf ("Can not find '%s', print aborted.\n", ps_file);
      else
      {
         sprintf (message, "Can not find '%s', print aborted.", ps_file);
         Msg (message);
      }
      fclose (fp);
      unlink (tmp_file);
      return;
   }
   while (fgets (tmp_str, MAXSTRING, fps) != NULL) /* copy the header file */
      fputs (tmp_str, fp);
   fclose (fps);

#ifdef KANJI
   if (!noKanji)
   {
       sprintf(ps_file, noEuc ? "%s/.pskanjiEXT": "%s/.pskanjiEUC", drawPath);

       if ((fps = fopen (ps_file, "r")) == NULL) 
       {
	   sprintf (message, "Can not find %s, print aborted.", ps_file);
	   Msg (message);
	   fclose (fp);
	   unlink (tmp_file);
	   return;
       }
	   /* copy the header file */
       while (fgets (tmp_str, MAXSTRING, fps) != NULL)
	   fputs (tmp_str, fp);
   }
#endif /* KANJI */
   fprintf (fp, "gsave\n\n");
   switch (pageStyle)
   {
      case LANDSCAPE:
      case HIGHLAND:
      case SLIDELAND: fprintf (fp, "90 rotate\n"); break;
   }
   switch (whereToPrint)
   {
      case PRINTER:
      case PS_FILE:
#ifndef UC
         fprintf (fp, "%1d %s mul %1d %s mul translate\n", psDotsPerInch,
               psXOffStr[pageStyle], psDotsPerInch, psYOffStr[pageStyle]);
#else /* UC */
         fprintf (fp, "%1f %s mul %1f %s mul translate\n", psDotsPerInch[cmUnit],
	       psXOffStr[cmUnit][pageStyle], psDotsPerInch[cmUnit],
	       psYOffStr[cmUnit][pageStyle]);	 
#endif /* UC */
         break;
      case LATEX_FIG: break;
   }

   fprintf (fp, "%s -%s scale\n\n",
#ifndef UC
         psScaleStr[pageStyle], psScaleStr[pageStyle]);
#else /* UC */
	   psScaleStr[cmUnit][pageStyle], psScaleStr[cmUnit][pageStyle]);
#endif /* UC */
   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      DumpAllObj (fp, obj_ptr, PRTGIF);

   fprintf (fp, "grestore\n\n");
   switch (whereToPrint)
   {
      case PRINTER:
      case PS_FILE: fprintf (fp, "showpage\n"); break;
      case LATEX_FIG: break;
   }
   fclose (fp);

   switch (whereToPrint)
   {
      case PRINTER:
         if (PRTGIF)
         {
            if (lastFile)
               sprintf (cmd, "%s %s 2>&1", printCommand, tmp_file);
            else
               sprintf (cmd, "%s -h %s 2>&1", printCommand, tmp_file);
         }
         else
         {
            sprintf (cmd, "%s %s 2>&1", printCommand, tmp_file);
            sprintf (message, "Printing with '%s' command.", printCommand);
            Msg (message);
         }
         break;
      case LATEX_FIG:
         if (PRTGIF)
            sprintf (ps_file, "%s.%s", FileName, EPSF_FILE_EXT);
         else
         {
            if (!curFileDefined)
            {
               Dialog ("No current file.  Can not generate LaTeX output!", cmd);
               unlink (tmp_file);
               return;
            }
            sprintf (ps_file, "%s/%s", curDir, curFileName);
            len = strlen (ps_file);
            for (i = len-1; ps_file[i] != '.'; i--) ;
            sprintf (&ps_file[i], ".%s", EPSF_FILE_EXT);
            sprintf (cmd, "Printing into '%s'.", ps_file);
            Msg (cmd);
         }
         sprintf (cmd, "cat %s 1> %s 2>&1; chmod %s %s", tmp_file, ps_file,
               PSFILE_MOD, ps_file);
         break;
      case PS_FILE:
         if (PRTGIF)
            sprintf (ps_file, "%s.%s", FileName, PS_FILE_EXT);
         else
         {
            if (!curFileDefined)
            {
               Dialog ("No current file.  Can not generate PostScript output!",
                  cmd);
               unlink (tmp_file);
               return;
            }
            sprintf (ps_file, "%s/%s", curDir, curFileName);
            len = strlen (ps_file);
            for (i = len-1; ps_file[i] != '.'; i--) ;
            sprintf (&ps_file[i], ".%s", PS_FILE_EXT);
            sprintf (cmd, "Printing into '%s'.", ps_file);
            Msg (cmd);
         }
         sprintf (cmd, "cat %s 1> %s 2>&1; chmod %s %s", tmp_file, ps_file,
               PSFILE_MOD, ps_file);
         break;
   }

#ifdef KINPUT
   if (kinputPid)
   {
#ifndef SYSV
       signal (SIGCHLD, SIG_DFL);
#else /* SYSV */
       signal (SIGCLD, SIG_DFL);
#endif /* SYSV */
   }
#endif /* KINPUT */
   if ((fp = popen (cmd, "r")) == NULL)
   {
      if (PRTGIF)
         printf ("Can not execute '%s', print aborted.\n", cmd);
      else
      {
         sprintf (message, "Can not execute '%s', print aborted.", cmd);
         Msg (message);
      }
      unlink (tmp_file);
      return;
   }
   while (fgets(tmp_str, MAXSTRING, fp) != NULL)
   {
      if (PRTGIF)
         printf ("%s", tmp_str);
      else
         Msg (tmp_str);
      sleep (5);
   }
   switch (whereToPrint)
   {
      case PRINTER:
         if (PRTGIF)
            printf ("'%s' printed.\n\n", tmp_file);
         else
            Msg ("Print completed.");
         break;
      case PS_FILE:
         if (PRTGIF)
            printf ("PostScript file printed into '%s'.\n\n", ps_file);
         else
         {
            sprintf (message, "PostScript file printed into '%s'.", ps_file);
            Msg (message);
         }
         break;
      case LATEX_FIG:
         if (PRTGIF)
            printf ("LaTeX figure printed into '%s'.\n\n", ps_file);
         else
         {
            sprintf (message, "LaTeX figure printed into '%s'.", ps_file);
            Msg (message); 
         }
         break;
   }
   pclose (fp);
#ifdef KINPUT
   if (kinputPid)
   {
#ifndef SYSV
       signal (SIGCHLD, MyExecError);
#else /* SYSV */
       signal (SIGCLD, MyExecError);
#endif /* SYSV */
   }
#endif /* KINPUT */
   unlink (tmp_file);
}

void Dump (PRTGIF, FileName)
   int	PRTGIF;
   char	* FileName;
{
   char	* c_ptr;
   int	len;

   if (PRTGIF)
   {
      len = strlen (FileName);
      if (len >= 4 && strcmp (&FileName[len-4], ".obj") == 0)
         FileName[len-4] = '\0';
   }

   if (whereToPrint == PRINTER && !PRTGIF)
   {
      if ((c_ptr = XGetDefault (mainDisplay, "Tgif", "PrintCommand")) != NULL)
         strcpy (printCommand, c_ptr);
      else
         strcpy (printCommand, "lpr");
   }
   GenDump (PRTGIF, FileName);
}

void PrintWithCommand (PRTGIF, FileName)
   int	PRTGIF;
   char	* FileName;
{
   if (whereToPrint == PRINTER)
   {
      Dialog ("Please Enter Print Command Name:", printCommand);
      if (*printCommand == '\0') return;
   }
   GenDump (PRTGIF, FileName);
}

void NewProc ()
{
   if (fileModified)
   {
      switch (YesNoCancel ("File modified, save file before clear? [ync](y)"))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO: TieLooseEnds (); break;
         case CONFIRM_CANCEL: return;
      }
      SetFileModified (FALSE);
   }
   CleanUpDrawingWindow ();
   ClearFileInfo ();
   ClearAndRedrawDrawWindow ();
   Msg ("Editing no file.");
   objId = 0;
   RedrawTitleWindow ();
}

void OpenProc ()
{
   if (fileModified)
   {
      switch (YesNoCancel ("File modified, save file before open? [ync](y)"))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO: break;
         case CONFIRM_CANCEL: return;
      }
   }
   LoadFile ();
}

#define FILE_NEW 0
#define FILE_OPEN 1
#define FILE_SAVE 2
#define FILE_SAVENEW 3
#define FILE_IMPORT 4
#define FILE_IMPORTXBM 5
#define FILE_DOMAIN 6
#define FILE_DUMP 7
#define FILE_USR_DUMP 8
#define FILE_SOLVE 9
#define FILE_SIMULATE 10
#define FILE_PROBE 11
#define FILE_ANIMATE 12
#define FILE_ESCAPE 13
#define FILE_QUIT 14

#define FILEMENUENTRIES 15

char * fileMenuStr[] =
      { "New            ^N",
        "Open           ^O",
        "Save           ^S",
        "SaveNew       ^#S",
        "Import         #P",
        "ImportXBitmap ^#.",
        "ChangeDomain   ^C",
        "Print          ^P",
        "PrintWithCmd   ^-",
        "Solve          #S",
        "Simulate       #Y",
        "Probe          #B",
        "Animate        ^Z",
        "Escape         #X",
        "Quit           ^Q"
      };

int QuitProc ()
{
#ifdef KINPUT
      if (kinputStatus == EXECING)
      {
	  switch (YesNoCancel ("Kinput is running, quit it? [ync](y)"))
	  {
	     case CONFIRM_YES: kill (kinputPid, SIGKILL);
             case CONFIRM_NO: 
             case CONFIRM_CANCEL: break;
	  }
      }
#endif /* KINPUT */
   if (fileModified)
   {
      switch (YesNoCancel ("File modified, save file before quit? [ync](y)"))
      {
         case CONFIRM_YES: SaveFile (); break;
         case CONFIRM_NO: break;
         case CONFIRM_CANCEL: return (INVALID);
      }
   }
   if (AncesterModified ())
   {
      switch (YesNoCancel ("Ancester file modified, still quitting? [ync](y)"))
      {
         case CONFIRM_YES: return (FILE_QUIT);
         case CONFIRM_NO: return (INVALID);
         case CONFIRM_CANCEL: return (INVALID);
      }
   }
   return (FILE_QUIT);
}

int SolveProc ()
{
   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_SOLVE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
}

int SimulateProc ()
{
   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_SIMULATE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
}

int ProbeProc ()
{
   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_PROBE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
}

int AnimateProc ()
{
   switch (SaveTmpFile ("tmpmodel"))
   {
      case OBJ_FILE_SAVED: return (FILE_ANIMATE);
      case SYM_FILE_SAVED: return (INVALID);
      case INVALID: return (INVALID);
   }
}

int EscapeProc ()
{
   return (FILE_ESCAPE);
}

int FileMenu (X, Y)
   int	X, Y;
{
   register int index;
   int		* fore_colors, * valid;

   DefaultColorArrays (FILEMENUENTRIES, &fore_colors, &valid);
   index = TextMenuLoop (X, Y, fileMenuStr, FILEMENUENTRIES, fore_colors,
#ifndef UC
         valid, SINGLECOLOR);
#else /* UC */
         valid, SINGLECOLOR, MENU_FILE);
#endif /* UC */
   switch (index)
   {
      case FILE_NEW: NewProc (); break;
      case FILE_OPEN: OpenProc (); break;
      case FILE_SAVE: SaveFile (); break;
      case FILE_SAVENEW: SaveNewFile (); break;
      case FILE_IMPORT: ImportFile (); break;
      case FILE_IMPORTXBM: ImportXBitmapFile (); break;
      case FILE_DOMAIN: ChangeDomain (); break;
      case FILE_DUMP: Dump (FALSE, ""); break;
      case FILE_USR_DUMP: PrintWithCommand (FALSE, ""); break;
      case FILE_SOLVE: return (SolveProc ());
      case FILE_SIMULATE: return (SimulateProc ());
      case FILE_PROBE: return (ProbeProc ());
      case FILE_ANIMATE: return (AnimateProc ());
      case FILE_ESCAPE: return (EscapeProc ());
      case FILE_QUIT: return (QuitProc ());
   }
   return (INVALID);
}

void CleanUpFiles ()
{
   ClearFileInfo ();
   fileModified = FALSE;
}

void EmergencySave ()
{
   if (exitNormally) return;

   signal (SIGHUP, SIG_DFL);
   signal (SIGFPE, SIG_DFL);
   signal (SIGBUS, SIG_DFL);
   signal (SIGSEGV, SIG_DFL);

#ifndef SYSV
   switch (SaveTmpFile ("EmergencySave"))
#else /* SYSV */
   switch (SaveTmpFile ("Emergency"))
#endif /* SYSV */
   {
      case OBJ_FILE_SAVED:
#ifndef SYSV
         fprintf (stderr, "Saved to EmergencySave.obj.\n");
#else /* SYSV */
         fprintf (stderr, "Saved to Emergency.obj.\n");
#endif /* SYSV */
         break;
      case SYM_FILE_SAVED:
#ifndef SYSV
         fprintf (stderr, "Saved to EmergencySave.sym.\n");
#else /* SYSV */
         fprintf (stderr, "Saved to Emergency.sym.\n");
#endif /* SYSV */
         break;
      case INVALID:
         fprintf (stderr, "Unable to save working file.\n");
         break;
   }
#ifdef DEBUG
   kill (getpid(), SIGQUIT);
#endif /* DEBUG */
   exitNormally = TRUE;
   exit (0);
}

int EmergencySaveForX (dsp, ev)
   Display	* dsp;
   XErrorEvent	* ev;
{

   if (exitNormally) return (0);

   signal (SIGHUP, SIG_DFL);
   signal (SIGFPE, SIG_DFL);
   signal (SIGBUS, SIG_DFL);
   signal (SIGSEGV, SIG_DFL);

#ifndef SYSV
   switch (SaveTmpFile ("EmergencySave"))
#else /* SYSV */
   switch (SaveTmpFile ("Emergency"))
#endif /* SYSV */
   {
      case OBJ_FILE_SAVED:
         fprintf (stderr, "Saved to EmergencySave.obj.\n");
         break;
      case SYM_FILE_SAVED:
         fprintf (stderr, "Saved to EmergencySave.sym.\n");
         break;
      case INVALID:
         fprintf (stderr, "Unable to save working file.\n");
         break;
   }
   exitNormally = TRUE;
#ifndef UC
   return (-1);
#else /* UC */
   exit (-1);
#endif /* UC */
}
