/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/poly.c,v 2.0 91/03/05 12:47:54 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "attr.e"
#include "color.e"
#include "cursor.e"
#include "drawing.e"
#include "file.e"
#include "grid.e"
#include "obj.e"
#include "pattern.e"
#include "raster.e"
#include "rect.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "spline.e"
#ifdef UC
#include "showsize.e"
#endif /* UC */

int	polyDrawn = FALSE;

/* short	widthOfLine[] = { 0, 3,  6,  0 }; */
/* short	arrowHeadH[]  = { 3, 5,  10, 3 }; */
/* short	arrowHeadW[]  = { 8, 12, 22, 8 }; */

short	widthOfLine[] = { 1, 2,  3,  4,  5,  6,  7 };
short	arrowHeadH[]  = { 3, 4,  5,  6,  7,  8,  9 };
short	arrowHeadW[]  = { 8, 10, 12, 14, 18, 20, 22 };

double	penGrayScale[] = { 1.0, 0.0, 1.0, 0.9, 0.85, 0.8, 0.7, 0.5 };

static struct PtRec	* lastPtPtr = NULL;
static XPoint pv[11];

void UpdPolyBBox (ObjPtr, NumPts, V)
   struct ObjRec	* ObjPtr;
   int			NumPts;
   XPoint		* V;
{
   register int	i;
   int		ltx, lty, rbx, rby, w = 0;

   ltx = rbx = V[0].x;
   lty = rby = V[0].y;

   for (i = 1; i < NumPts; i++)
   {
      if (V[i].x < ltx) ltx = V[i].x; if (V[i].y < lty) lty = V[i].y;
      if (V[i].x > rbx) rbx = V[i].x; if (V[i].y > rby) rby = V[i].y;
   }

   switch (ObjPtr->type)
   {
      case OBJ_POLY: w = arrowHeadH[ObjPtr->detail.p->width]; break;
      case OBJ_POLYGON: w = widthOfLine[ObjPtr->detail.g->width]; break;
   }

   ObjPtr->x = ltx;
   ObjPtr->y = lty;
   ObjPtr->bbox.ltx = ltx - w;
   ObjPtr->bbox.lty = lty - w;
   ObjPtr->bbox.rbx = rbx + w;
   ObjPtr->bbox.rby = rby + w;
   ObjPtr->obbox.ltx = ltx;
   ObjPtr->obbox.lty = lty;
   ObjPtr->obbox.rbx = rbx;
   ObjPtr->obbox.rby = rby;
   if (ObjPtr->type == OBJ_POLY) AdjObjBBox (ObjPtr); /* adjust for attrs */
}

static
void CreatePolyObj (NumPts)
   int	NumPts;
{
   struct PtRec		* pt_ptr;
   struct PolyRec	* poly_ptr;
   struct ObjRec	* obj_ptr;
   register int		i;
   XPoint		* v;
   int			ltx, lty, rbx, rby, w;

   poly_ptr = (struct PolyRec *) calloc (1, sizeof(struct PolyRec));
   poly_ptr->n = NumPts;
   v = (XPoint *) calloc (NumPts+1, sizeof(XPoint));
   pt_ptr = lastPtPtr;
   ltx = rbx = pt_ptr->x;
   lty = rby = pt_ptr->y;
   for (i = NumPts-1; i >= 0; i--, lastPtPtr = lastPtPtr->next)
   {
#ifndef UC
      v[i].x = (lastPtPtr->x << zoomScale) + drawOrigX;
      v[i].y = (lastPtPtr->y << zoomScale) + drawOrigY;
#else /* UC */
      v[i].x = (RealSize(lastPtPtr->x ,  zoomScale)) + drawOrigX;
      v[i].y = (RealSize(lastPtPtr->y ,  zoomScale)) + drawOrigY;
#endif /* UC */
      if (lastPtPtr->x < ltx) ltx = lastPtPtr->x;
      if (lastPtPtr->y < lty) lty = lastPtPtr->y;
      if (lastPtPtr->x > rbx) rbx = lastPtPtr->x;
      if (lastPtPtr->y > rby) rby = lastPtPtr->y;
      cfree (lastPtPtr);
   }
   poly_ptr->vlist = v;
   if (curSpline == LT_SPLINE)
      poly_ptr->svlist = MakeSplinePolyVertex (&(poly_ptr->sn), drawOrigX,
            drawOrigY, NumPts, v);
   poly_ptr->style = lineStyle;
   poly_ptr->width = lineWidth;
   poly_ptr->pen = penPat;
   poly_ptr->curved = curSpline;
   poly_ptr->fill = objFill;
   poly_ptr->dash = curDash;
   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   obj_ptr->color = colorIndex;
   obj_ptr->type = OBJ_POLY;
   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x =
#ifndef UC
         (ltx << zoomScale) + drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y =
         (lty << zoomScale) + drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = (rbx << zoomScale) + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = (rby << zoomScale) + drawOrigY;
#else /* UC */
         (RealSize(ltx ,  zoomScale)) + drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y =
         (RealSize(lty ,  zoomScale)) + drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = (RealSize(rbx ,  zoomScale)) + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = (RealSize(rby ,  zoomScale)) + drawOrigY;
#endif /* UC */
   w = arrowHeadH[lineWidth];
   obj_ptr->bbox.ltx -= w;
   obj_ptr->bbox.lty -= w;
   obj_ptr->bbox.rbx += w;
   obj_ptr->bbox.rby += w;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->detail.p = poly_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   AddObj (NULL, topObj, obj_ptr);
}

/* line shapes: */
/* 0  |---|     */
/* 1 -|---|     */
/* 2  |---|-    */
/* 3 -|---|-    */
/* 4  |<--|     */
/* 5  |<--|-    */
/* 6  |-->|     */
/* 7 -|-->|     */
/* 8  |<->|     */

/*
 *                             Cont == 0
 *
 *      ||           ||           ||           ||           ||
 *      ||  Rb == 1  ||  Rb == 0  ||  Rb == 1  ||  Rb == 0  ||
 *      ||  Pt == 1  ||  Pt == 1  ||  Pt == 0  ||  Pt == 0  ||
 *      ||           ||           ||           ||           ||
 *  --- ||   |---|   ||   |---|   ||  -|---|   ||  -|---|   ||
 *  <-- ||   |<--|   ||   |<--|   ||  -|---|   ||  -|---|   ||
 *  --> ||   |-->|   ||   |-->|   ||  -|-->|   ||  -|-->|   ||
 *  <-> ||   |<->|   ||   |<->|   ||  -|-->|   ||  -|-->|   ||
 *      ||           ||           ||           ||           ||
 *
 *                             Cont == 1
 *
 *      ||           ||           ||           ||           ||
 *      ||  Rb == 1  ||  Rb == 0  ||  Rb == 1  ||  Rb == 0  ||
 *      ||  Pt == 1  ||  Pt == 1  ||  Pt == 0  ||  Pt == 0  ||
 *      ||           ||           ||           ||           ||
 *  --- ||   |---|   ||   |---|-  ||  -|---|   ||  -|---|-  ||
 *  <-- ||   |<--|   ||   |<--|-  ||  -|---|   ||  -|---|-  ||
 *  --> ||   |-->|   ||   |---|-  ||  -|-->|   ||  -|---|-  ||
 *  <-> ||   |<->|   ||   |<--|-  ||  -|-->|   ||  -|---|-  ||
 *      ||           ||           ||           ||           ||
 */

void MyLine (Win,Cont,Rb,N,StartX,StartY,EndX,EndY,W,S,P,Pix,Dash)
   Window	Win;
   int		Cont, Rb, N, StartX, StartY, EndX, EndY, W, S, P, Pix, Dash;
   /* 'Cont' means continue */
   /* 'Rb' means rubber-band */
   /* StartX, StartY, EndX, EndY are screen coordinates within Win, */
   /*   that is, scaled and translated. */
{
   int		shape = 0, aw, ah, x1, y1, x2, y2;
   double	dx, dy, len, sin, cos;
   XGCValues	values;

   if (StartX == EndX && StartY == EndY) return;

   if (Cont)
      if (Rb)
         if (N == 1)
            switch (S)
            {
               case LS_PLAIN: shape = 0; break;
               case LS_LEFT: shape = 4; break;
               case LS_RIGHT: shape = 6; break;
               case LS_DOUBLE: shape = 8; break;
            }
         else /* Cont && Rb && N != 1 */
            switch (S)
            {
               case LS_PLAIN: shape = 1; break;
               case LS_LEFT: shape = 1; break;
               case LS_RIGHT: shape = 7; break;
               case LS_DOUBLE: shape = 7; break;
            }
      else /* Cont && !Rb */
         if (N == 1)
            switch (S)
            {
               case LS_PLAIN: shape = 2; break;
               case LS_LEFT: shape = 5; break;
               case LS_RIGHT: shape = 2; break;
               case LS_DOUBLE: shape = 5; break;
            }
         else /* Cont && !Rb && N != 1 */
            shape = 3;
   else
      if (N == 1)
         switch (S)
         {
            case LS_PLAIN: shape = 0; break;
            case LS_LEFT: shape = 4; break;
            case LS_RIGHT: shape = 6; break;
            case LS_DOUBLE: shape = 8; break;
         }
      else /* !Cont && N != 1 */
         switch (S)
         {
            case LS_PLAIN: shape = 1; break;
            case LS_LEFT: shape = 1; break;
            case LS_RIGHT: shape = 7; break;
            case LS_DOUBLE: shape = 7; break;
         }

#ifndef UC
   ah = arrowHeadH[W] >> zoomScale; if (ah == 0) ah = 1;
   aw = arrowHeadW[W] >> zoomScale; if (aw == 0) aw = 1;
#else /* UC */
   ah = ScreenSize(arrowHeadH[W] ,  zoomScale); if (ah == 0) ah = 1;
   aw = ScreenSize(arrowHeadW[W] ,  zoomScale); if (aw == 0) aw = 1;
#endif /* UC */

   if (Rb)
   {
      values.foreground = Pix;
      values.function = GXxor;
      values.fill_style = FillSolid;
      values.line_width = 0;
      values.line_style = LineSolid;
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
            &values);
   }
   else if (P != 0)
   {
      values.foreground = (P == 2) ? myBgPixel : Pix;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[P];
#ifndef UC
      values.line_width = widthOfLine[W] >> zoomScale;
#else /* UC */
      values.line_width = ScreenSize(widthOfLine[W] ,  zoomScale);
#endif /* UC */
      if (Dash != 0)
      {
         XSetDashes (mainDisplay, drawGC, 0, dashList[Dash],
               dashListLength[Dash]);
         values.line_style = LineOnOffDash;
      }
      else
         values.line_style = LineSolid;
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple | GCLineWidth |
            GCLineStyle, &values);
   }
   else
      return;

   dx = (double)(EndX - StartX);
   dy = (double)(EndY - StartY);
   len = (double) sqrt((double)(dx * dx + dy * dy));
   sin = dy / len;
   cos = dx / len;

   switch (shape)
   {
      case 0:
      case 1:
      case 2:
      case 3:
         XDrawLine (mainDisplay, Win, drawGC, StartX, StartY, EndX, EndY);
         break;
      case 4:
      case 5:
         x1 = round(StartX+aw*cos);
         y1 = round(StartY+aw*sin);
         XDrawLine (mainDisplay, Win, drawGC, x1, y1, EndX, EndY);
         break;
      case 6:
      case 7:
         x2 = round(EndX-aw*cos);
         y2 = round(EndY-aw*sin);
         XDrawLine (mainDisplay, Win, drawGC, StartX, StartY, x2, y2);
         break;
      case 8:
         x1 = round(StartX+aw*cos);
         y1 = round(StartY+aw*sin);
         x2 = round(EndX-aw*cos);
         y2 = round(EndY-aw*sin);
         XDrawLine (mainDisplay, Win, drawGC, x1, y1, x2, y2);
         break;
   }
   if (shape==4 || shape==5 || shape==8)
   {
      pv[0].x = pv[3].x = StartX;
      pv[0].y = pv[3].y = StartY;
      pv[1].x = round(StartX + aw*cos - ah*sin);
      pv[1].y = round(StartY + aw*sin + ah*cos);
      pv[2].x = round(StartX + aw*cos + ah*sin);
      pv[2].y = round(StartY + aw*sin - ah*cos);
      XFillPolygon (mainDisplay, Win, drawGC, pv, 4, Convex, CoordModeOrigin);
   }
   if (shape==6 || shape==7 || shape==8)
   {
      pv[0].x = pv[3].x = EndX;
      pv[0].y = pv[3].y = EndY;
      pv[1].x = round(EndX - aw*cos + ah*sin);
      pv[1].y = round(EndY - aw*sin - ah*cos);
      pv[2].x = round(EndX - aw*cos - ah*sin);
      pv[2].y = round(EndY - aw*sin + ah*cos);
      XFillPolygon (mainDisplay, Win, drawGC, pv, 4, Convex, CoordModeOrigin);
   }
}

static
void ContinuePoly (OrigX, OrigY)
   int 	OrigX, OrigY;
   /* OrigX and OrigY are screen coordinates (scaled and translated). */
   /* OrigX and OrigY are also on grid. */
{
   XGCValues	values;
   XEvent	input, ev;
   XButtonEvent	* button_ev;
   XMotionEvent	* motion_ev;
   int		pixel, xor_pixel;
   int 		end_x, end_y, grid_x, grid_y, done = FALSE, num_pts = 1;
   int		last_x = OrigX, last_y = OrigY;
   struct PtRec	* pt_ptr;
#ifdef UC
   register int	showsize = showSize;
   register int	hidedragcursor = hideDragCursor;
   register int cursordefined = FALSE;
#endif /* UC */

   pixel = colorPixels[colorIndex];
   xor_pixel = xorColorPixels[colorIndex];

   values.foreground = xor_pixel;
   values.function = GXxor;
   values.fill_style = FillSolid;
   values.line_width = 0;
   values.line_style = LineSolid;

   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = end_x = OrigX;
   grid_y = end_y = OrigY; 
   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask,
#ifndef UC
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
#else /* UC */
         GrabModeAsync, GrabModeAsync, None,
         ((hidedragcursor)? None: handCursor), CurrentTime);

#endif /* UC */
#ifdef UC
   if (showsize) showPoint (OrigX, OrigY, RIGHTUP);
#endif /* UC */
   
   while (!done)
   {
      XNextEvent (mainDisplay, &input);
      if (input.type == MotionNotify)
      {
         motion_ev = &(input.xmotion);
         MyLine (drawWindow, CONT, RB, num_pts, OrigX, OrigY, grid_x,
               grid_y, lineWidth, lineStyle, penPat, xor_pixel,0);

#ifdef UC
	 if (showsize) eraseSizeString ();
#endif /* UC */
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
#ifdef UC
	 if (hidedragcursor && !cursordefined &&
			(grid_x != OrigX || grid_y != OrigY))
	 {
	     XDefineCursor (mainDisplay, drawWindow, dragCursor);
	     cursordefined = TRUE;
	 }
#endif /* UC */
         MarkRulers (grid_x, grid_y);
         MyLine (drawWindow, CONT, RB, num_pts, OrigX, OrigY, grid_x,
               grid_y, lineWidth, lineStyle, penPat, xor_pixel,0);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
#ifdef UC
	 if (showsize) showLength (OrigX, OrigY, grid_x, grid_y, RIGHTUP);
#endif /* UC */
      }
      else if (input.type == ButtonPress)
      {
         button_ev = &(input.xbutton);
         MyLine (drawWindow, CONT, RB, num_pts, OrigX, OrigY, grid_x,
               grid_y, lineWidth, lineStyle, penPat, xor_pixel,0);

#ifdef UC
	 if (showsize) eraseSizeString ();
#endif /* UC */
         GridXY (end_x, end_y, &grid_x, &grid_y);
         if (grid_x != last_x || grid_y != last_y)
         {
            num_pts++;
            pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
            pt_ptr->next = lastPtPtr;
            lastPtPtr = pt_ptr;
            pt_ptr->x = last_x = grid_x;
            pt_ptr->y = last_y = grid_y;
         }
         switch (button_ev->button)
         {
            case Button1 :
               if (penPat != 0)
                  MyLine (drawWindow, CONT, NORB, num_pts-1, OrigX, OrigY,
                        grid_x, grid_y, lineWidth, lineStyle, penPat,
                        pixel,0);
               OrigX = grid_x;
               OrigY = grid_y;
#ifdef UC
	       if (showsize) showPoint (grid_x, grid_y, RIGHTUP);
#endif /* UC */
               break;
            case Button2 :
            case Button3 :
               if (penPat != 0 || (OrigX == grid_x && OrigY == grid_y &&
                     (lineStyle & LS_RIGHT)))
                  MyLine (drawWindow, NOCONT, NORB, num_pts-1, OrigX,
                        OrigY, grid_x, grid_y, lineWidth, lineStyle,
                        penPat, pixel,0);

               done = TRUE;
               break;
         }
      }
   }
   XUngrabPointer (mainDisplay, CurrentTime);

#ifdef UC
   if (hidedragcursor && cursordefined)
   {
       XDefineCursor (mainDisplay, drawWindow, drawCursor);
       cursordefined = FALSE;
       XFlush (mainDisplay);
   }
#endif /* UC */
   if (num_pts > 1)
   {
      CreatePolyObj (num_pts);
#ifndef UC
      RedrawAnArea (botObj, topObj->bbox.ltx-(1<<zoomScale),
            topObj->bbox.lty-(1<<zoomScale), topObj->bbox.rbx+(1<<zoomScale),
            topObj->bbox.rby+(1<<zoomScale));
#else /* UC */
      RedrawAnArea (botObj, topObj->bbox.ltx-(RealSize(1, zoomScale)),
            topObj->bbox.lty-(RealSize(1, zoomScale)), topObj->bbox.rbx+(RealSize(1, zoomScale)),
            topObj->bbox.rby+(RealSize(1, zoomScale)));
#endif /* UC */
      polyDrawn = TRUE;
      SetFileModified (TRUE);
   }
}

void DrawPoly (input)
   XEvent	* input;
{
   int		mouse_x, mouse_y, grid_x, grid_y;
   XButtonEvent	* button_ev;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = input->xbutton.x;
      mouse_y = input->xbutton.y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      lastPtPtr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
      lastPtPtr->x = grid_x;
      lastPtPtr->y = grid_y;
      lastPtPtr->next = NULL;
      ContinuePoly (grid_x, grid_y);
   } 
}

static
void DumpArrow (FP, TailV, HeadV, ArrowW, ArrowH, Pen)
   FILE			* FP;
   XPoint		TailV, HeadV;
   int			ArrowW, ArrowH, Pen;
{
   int		i, dx, dy;
   struct BBRec	bbox;
   XPoint	v[2];
   double	len, sin, cos;

   dx = HeadV.x - TailV.x;
   dy = HeadV.y - TailV.y;

   fprintf (FP, "gsave\n");

   if (colorDump)
   {
      len = (double) sqrt ((double)(dx*dx + dy*dy));
      sin = ((double)dy) / len;
      cos = ((double)dx) / len;

      v[0].x = round(HeadV.x - ArrowW*cos + ArrowH*sin);
      v[0].y = round(HeadV.y - ArrowW*sin - ArrowH*cos);
      v[1].x = round(HeadV.x - ArrowW*cos - ArrowH*sin);
      v[1].y = round(HeadV.y - ArrowW*sin + ArrowH*cos);

      bbox.ltx = bbox.rbx = HeadV.x;
      bbox.lty = bbox.rby = HeadV.y;

      for (i = 0; i < 2; i++)
      {
         if (v[i].x < bbox.ltx) bbox.ltx = v[i].x;
         if (v[i].y < bbox.lty) bbox.lty = v[i].y;
         if (v[i].x > bbox.rbx) bbox.rbx = v[i].x;
         if (v[i].y > bbox.rby) bbox.rby = v[i].y;
      }
   }
   else
      fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n", Pen);

   fprintf (FP, "   newpath\n");
   fprintf (FP, "      %1d %1d %1d %1d %1d %1d arrowtip\n", HeadV.x, HeadV.y,
         ArrowW, ArrowH, dx, dy);
   switch (Pen)
   {
      case NONEPAT: break;
      case SOLIDPAT:
      case BACKPAT: fprintf (FP, "   closepath fill\n"); break;
      default:
         if (colorDump)
         {
            fprintf (FP, "   closepath eoclip\n");
            DumpPatFill (FP, Pen, 8, bbox, "   ");
         }
         else
            fprintf (FP, "   closepath fill\n");
         break;
   }
   fprintf (FP, "grestore\n");
}

void DumpPolyObj (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   XPoint	* v;
   int		i, num_pts, fill, pen, width, curved, dash, color_index;
   int		w, aw, ah, dy, dx;

   fill = ObjPtr->detail.p->fill;
   width = ObjPtr->detail.p->width;
   pen = ObjPtr->detail.p->pen;
   curved = ObjPtr->detail.p->curved;
   dash = ObjPtr->detail.p->dash;
   v = ObjPtr->detail.p->vlist;
   num_pts = ObjPtr->detail.p->n;

   if (fill == NONEPAT && pen == NONEPAT) return;

   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   if (fill != NONEPAT && num_pts > 2)
   {
      switch (curved)
      {
         case LT_STRAIGHT:
            switch (fill)
            {
               case SOLIDPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                     fprintf (FP,
                           "   pat%1d 8 1 0 72 300 32 div div setpattern\n",
                           fill);
                  fprintf (FP, "   newpath\n");
                  fprintf (FP, "      %1d %1d moveto\n", v[0].x,v[0].y);
                  DumpPoints (FP, num_pts, v, 6);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
         case LT_SPLINE:
            switch (fill)
            {
               case SOLIDPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpCurvedPolyPoints (FP, num_pts, v, 3);
                  fprintf (FP, "   %1d %1d curveto\n",
                        v[num_pts-1].x, v[num_pts-1].y);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpCurvedPolyPoints (FP, num_pts, v, 3);
                  fprintf (FP, "   %1d %1d curveto\n",
                        v[num_pts-1].x, v[num_pts-1].y);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                     fprintf (FP,
                           "   pat%1d 8 1 0 72 300 32 div div setpattern\n",
                           fill);
                  fprintf (FP, "   newpath\n");
                  fprintf (FP, "      %1d %1d moveto\n", v[0].x,v[0].y);
                  DumpCurvedPolyPoints (FP, num_pts, v, 6);
                  fprintf (FP, "   %1d %1d curveto\n",
                        v[num_pts-1].x, v[num_pts-1].y);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
      }
   }

   if (pen == NONEPAT) { fprintf (FP, "\n"); return; }

   fprintf (FP, "gsave\n");

   w = widthOfLine[width];
   aw = arrowHeadW[width];
   ah = arrowHeadH[width];

   fprintf (FP, "   %1d setlinewidth", w);
   if (dash != 0)
   {
      fprintf (FP, " [");
      for (i = 0; i < dashListLength[dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[dash][dashListLength[dash]-1]));
   }
   else
      fprintf (FP, "\n");

   fprintf (FP, "   newpath\n      %1d %1d moveto\n", v[0].x, v[0].y);
   if (ObjPtr->detail.p->style & LS_LEFT)
   {
      dx = v[1].x - v[0].x;
      dy = v[1].y - v[0].y;
      fprintf (FP, "      %1d %1d atan dup cos %1d mul exch sin %1d mul rmoveto\n",
            dy, dx, aw, aw);
   }
   if (ObjPtr->detail.p->style & LS_RIGHT)
   {
      if (curved == LT_SPLINE && num_pts != 2)
         DumpCurvedPolyPoints (FP, num_pts, v, 6);
      else
         DumpPoints (FP, num_pts-1, v, 6);

      dx = v[num_pts-1].x - v[num_pts-2].x;
      dy = v[num_pts-1].y - v[num_pts-2].y;
      fprintf (FP, "      %1d %1d atan dup cos %1d mul %1d exch sub\n",
            dy, dx, aw, v[num_pts-1].x);
      fprintf (FP, "      exch sin %1d mul %1d exch sub",
            aw, v[num_pts-1].y);

      if (curved == LT_SPLINE && num_pts != 2)
         fprintf (FP, " curveto\n");
      else
         fprintf (FP, " lineto\n");
   }
   else
      if (curved == LT_SPLINE && num_pts != 2)
      {
         DumpCurvedPolyPoints (FP, num_pts, v, 6);
         fprintf (FP, "      %1d %1d curveto\n",v[num_pts-1].x,v[num_pts-1].y);
      }
      else
         DumpPoints (FP, num_pts, v, 6);

   switch (pen)
   {
      case SOLIDPAT: fprintf (FP, "   stroke\n"); break;
      case BACKPAT: fprintf (FP, "   1 setgray stroke\n"); break;
      default:
         if (colorDump)
         {
            fprintf (FP, "   strokepath clip\n");
            DumpPatFill (FP, pen, 8, ObjPtr->bbox, "   ");
         }
         else
         {
            fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n", pen);
            fprintf (FP, "   stroke\n");
         }
         break;
   }

   fprintf (FP, "grestore\n");

   switch (ObjPtr->detail.p->style)
   {
      case LS_PLAIN: break;
      case LS_LEFT: DumpArrow (FP, v[1], v[0], aw, ah, pen); break;
      case LS_RIGHT:
         DumpArrow (FP, v[num_pts-2], v[num_pts-1], aw, ah, pen);
         break;
      case LS_DOUBLE:
         DumpArrow (FP, v[1], v[0], aw, ah, pen);
         DumpArrow (FP, v[num_pts-2], v[num_pts-1], aw, ah, pen);
         break;
   }
   fprintf (FP, "\n");
}

void DrawPolyObj (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   register int			i;
   register struct PolyRec	* poly_ptr = ObjPtr->detail.p;
   XPoint			* pv = poly_ptr->vlist, * v;
   int				pen, width, pixel, fill, curved, n, dash;
   int				x1, y1, x2, y2, real_x_off, real_y_off;
   XGCValues			values;

   n = poly_ptr->n;
   fill = poly_ptr->fill;
   width = poly_ptr->width;
   pen = poly_ptr->pen;
   curved = poly_ptr->curved;
   dash = poly_ptr->dash;
   pixel = colorPixels[ObjPtr->color];

   if (curved == LT_SPLINE)
      DrawSplinePolyObj (Win, XOff, YOff, fill, width, pen, dash, pixel,
            poly_ptr);
   else
   {
#ifndef UC
      real_x_off = (XOff >> zoomScale) << zoomScale;
      real_y_off = (YOff >> zoomScale) << zoomScale;
#else /* UC */
      real_x_off = RealSize((ScreenSize(XOff,  zoomScale)),  zoomScale);
      real_y_off = RealSize((ScreenSize(YOff,  zoomScale)),  zoomScale);
#endif /* UC */

      if (fill != NONEPAT)
      {
         v = (XPoint *) calloc (n+1, sizeof(XPoint));
         for (i = 0; i < n; i++)
         {
#ifndef UC
            v[i].x = (pv[i].x - real_x_off) >> zoomScale;
            v[i].y = (pv[i].y - real_y_off) >> zoomScale;
#else /* UC */
            v[i].x = ScreenSize((pv[i].x - real_x_off),  zoomScale);
            v[i].y = ScreenSize((pv[i].y - real_y_off),  zoomScale);
#endif /* UC */
         }
         v[n].x = v[0].x; v[n].y = v[0].y;
         values.foreground = (fill == BACKPAT) ? myBgPixel : pixel;
         values.function = GXcopy;
         values.fill_style = FillOpaqueStippled;
         values.stipple = patPixmap[fill];
         XChangeGC (mainDisplay, drawGC,
               GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
         XFillPolygon (mainDisplay, Win, drawGC, v, n+1, Complex,
               CoordModeOrigin);
         cfree (v);
      }

      if (pen == NONEPAT) return;

      for (i = 0; i < poly_ptr->n-2; i++)
      {
#ifndef UC
         x1 = (pv[i].x-real_x_off)>>zoomScale;
         y1 = (pv[i].y-real_y_off)>>zoomScale;
         x2 = (pv[i+1].x-real_x_off)>>zoomScale;
         y2 = (pv[i+1].y-real_y_off)>>zoomScale;
#else /* UC */
         x1 = ScreenSize((pv[i].x-real_x_off), zoomScale);
         y1 = ScreenSize((pv[i].y-real_y_off), zoomScale);
         x2 = ScreenSize((pv[i+1].x-real_x_off), zoomScale);
         y2 = ScreenSize((pv[i+1].y-real_y_off), zoomScale);
#endif /* UC */

         MyLine (Win, CONT, NORB, i+1, x1, y1, x2, y2, poly_ptr->width,
               poly_ptr->style, pen, pixel, dash);
      }
#ifndef UC
      x1 = (pv[i].x-real_x_off)>>zoomScale;
      y1 = (pv[i].y-real_y_off)>>zoomScale;
      x2 = (pv[i+1].x-real_x_off)>>zoomScale;
      y2 = (pv[i+1].y-real_y_off)>>zoomScale;
#else /* UC */
      x1 = ScreenSize((pv[i].x-real_x_off), zoomScale);
      y1 = ScreenSize((pv[i].y-real_y_off), zoomScale);
      x2 = ScreenSize((pv[i+1].x-real_x_off), zoomScale);
      y2 = ScreenSize((pv[i+1].y-real_y_off), zoomScale);
#endif /* UC */

      MyLine (Win, NOCONT, NORB, i+1, x1, y1, x2, y2, poly_ptr->width,
            poly_ptr->style, pen, pixel, dash);
   }
}

void SavePolyObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int		i, n;
   struct PolyRec	* poly_ptr = ObjPtr->detail.p;

   n = poly_ptr->n;
   fprintf (FP, "poly('%s',%1d,[", colorMenuItems[ObjPtr->color], poly_ptr->n);
   for (i = 0; i < n-1; i++)
   {
      fprintf (FP, "%1d,%1d,", poly_ptr->vlist[i].x, poly_ptr->vlist[i].y);
   }
   fprintf (FP, "%1d,%1d", poly_ptr->vlist[n-1].x, poly_ptr->vlist[n-1].y);
   fprintf (FP, "],%1d,%1d,%1d,%1d,%1d,%1d,%1d,", poly_ptr->style, poly_ptr->width,
         poly_ptr->pen, ObjPtr->id, poly_ptr->curved, poly_ptr->fill, poly_ptr->dash);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

void ReadPolyObj (Inbuf, ObjPtr)
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register int		i;
   struct PolyRec	* poly_ptr;
   XPoint		* v;
   char			color_str[20], * s;
   int			num_pts, ltx, lty, rbx, rby, x, y, w;

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);
   sscanf (s, "%d,", &num_pts);
   poly_ptr = (struct PolyRec *) calloc (1, sizeof(struct PolyRec));
   poly_ptr->n = num_pts;
   v = (XPoint *) calloc (num_pts+1, sizeof(XPoint));
   ltx = 20 * PIX_PER_INCH; rbx = 0;
   lty = 20 * PIX_PER_INCH; rby = 0;
   s = FindChar ('[', s);
   for (i = 0; i < num_pts; i++)
   {
      sscanf (s, "%d , %d", &x, &y);
      v[i].x = x;
      v[i].y = y;
      if (x < ltx) ltx = x;
      if (y < lty) lty = y;
      if (x > rbx) rbx = x;
      if (y > rby) rby = y;
      s = FindChar (',',s);
      s = FindChar (',',s);
   }

   if (fileVersion == INVALID)
   {
      sscanf (s, "%d , %d , %d", &(poly_ptr->style), &(poly_ptr->width),
            &(poly_ptr->pen));
      (*ObjPtr)->id = objId++;
      poly_ptr->fill = NONEPAT;
      if (poly_ptr->width == LINE_CURVED)
      {
         poly_ptr->width = 0;
         poly_ptr->curved = TRUE;
      }
      else
         poly_ptr->curved = FALSE;
      switch (poly_ptr->width)
      {
         case 1: poly_ptr->width = 3; break;
         case 2: poly_ptr->width = 6; break;
      }
      poly_ptr->dash = 0;
   }
   else if (fileVersion <= 3)
   {
      sscanf (s, "%d , %d , %d , %d", &(poly_ptr->style), &(poly_ptr->width),
            &(poly_ptr->pen), &((*ObjPtr)->id));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
      poly_ptr->fill = NONEPAT;
      if (poly_ptr->width == LINE_CURVED)
      {
         poly_ptr->width = 0;
         poly_ptr->curved = TRUE;
      }
      else
         poly_ptr->curved = FALSE;
      switch (poly_ptr->width)
      {
         case 1: poly_ptr->width = 3; break;
         case 2: poly_ptr->width = 6; break;
      }
      poly_ptr->dash = 0;
   }
   else if (fileVersion <= 4)
   {
      sscanf (s, "%d , %d , %d , %d, %d", &(poly_ptr->style),
            &(poly_ptr->width), &(poly_ptr->pen), &((*ObjPtr)->id),
            &(poly_ptr->curved));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
      poly_ptr->fill = NONEPAT;
      switch (poly_ptr->width)
      {
         case 1: poly_ptr->width = 3; break;
         case 2: poly_ptr->width = 6; break;
      }
      poly_ptr->dash = 0;
   }
   else if (fileVersion <= 5)
   {
      sscanf (s, "%d , %d , %d , %d, %d, %d", &(poly_ptr->style),
            &(poly_ptr->width), &(poly_ptr->pen), &((*ObjPtr)->id),
            &(poly_ptr->curved), &(poly_ptr->fill));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
      switch (poly_ptr->width)
      {
         case 1: poly_ptr->width = 3; break;
         case 2: poly_ptr->width = 6; break;
      }
      poly_ptr->dash = 0;
   }
   else if (fileVersion <= 8)
   {
      sscanf (s, "%d , %d , %d , %d, %d, %d", &(poly_ptr->style),
            &(poly_ptr->width), &(poly_ptr->pen), &((*ObjPtr)->id),
            &(poly_ptr->curved), &(poly_ptr->fill));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
      poly_ptr->dash = 0;
   }
   else
   {
      sscanf (s, "%d , %d , %d , %d , %d , %d , %d", &(poly_ptr->style),
            &(poly_ptr->width), &(poly_ptr->pen), &((*ObjPtr)->id),
            &(poly_ptr->curved), &(poly_ptr->fill), &(poly_ptr->dash));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
   }

   poly_ptr->vlist = v;
   if (poly_ptr->curved == LT_SPLINE)
      poly_ptr->svlist = MakeSplinePolyVertex (&(poly_ptr->sn), drawOrigX,
            drawOrigY, num_pts, v);
   w = arrowHeadH[poly_ptr->width];
   (*ObjPtr)->x = ltx;
   (*ObjPtr)->y = lty;
   (*ObjPtr)->color = FindColorIndex (color_str);
   (*ObjPtr)->type = OBJ_POLY;
   (*ObjPtr)->bbox.ltx = ltx - w;
   (*ObjPtr)->bbox.lty = lty - w;
   (*ObjPtr)->bbox.rbx = rbx + w;
   (*ObjPtr)->bbox.rby = rby + w;
   (*ObjPtr)->obbox.ltx = ltx;
   (*ObjPtr)->obbox.lty = lty;
   (*ObjPtr)->obbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = rby;
   (*ObjPtr)->detail.p = poly_ptr;
}

void FreePolyObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   if (ObjPtr->detail.p->curved) cfree (ObjPtr->detail.p->svlist);
   cfree (ObjPtr->detail.p->vlist);
   cfree (ObjPtr->detail.p);
   cfree (ObjPtr);
}
