/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/polygon.c,v 2.0 91/03/05 12:47:59 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "grid.e"
#include "file.e"
#include "obj.e"
#include "pattern.e"
#include "poly.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "spline.e"
#ifdef UC
#include "showsize.e"
#endif /* UC */

int	polygonDrawn = FALSE;

static struct PtRec	* lastPtPtr = NULL;
static int		startPolygonX, startPolygonY;

void DumpPoints (FP, NumPts, V, Indent)
   FILE		* FP;
   int		NumPts, Indent;
   XPoint	* V;
{
   register int	i, j;

   for (i = 1; i < NumPts; i++)
   {
      for (j = 0; j < Indent; j++) fprintf (FP, " ");
      fprintf (FP, "%1d %1d lineto\n", V[i].x, V[i].y);
   }
}

void DumpPolygonObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   XPoint	* v = ObjPtr->detail.g->vlist;
   int		num_pts = ObjPtr->detail.g->n, i;
   int		fill, width, pen, curved, dash, color_index;

   fill = ObjPtr->detail.g->fill;
   width = ObjPtr->detail.g->width;
   pen = ObjPtr->detail.g->pen;
   curved = ObjPtr->detail.g->curved;
   dash = ObjPtr->detail.g->dash;

   if (fill == NONEPAT && pen == NONEPAT) return;

   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   if (fill != NONEPAT)
   {
      switch (curved)
      {
         case LT_STRAIGHT:
            switch (fill)
            {
               case SOLIDPAT: /* solid black polygon */
                  fprintf (FP, "newpath\n   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts-1, v, 3);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT: /* solid white polygon */
                  fprintf (FP, "newpath\n   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts-1, v, 3);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                     fprintf (FP,
                           "   pat%1d 8 1 0 72 300 32 div div setpattern\n",
                           fill);
                  fprintf (FP, "   newpath\n");
                  fprintf (FP, "      %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts-1, v, 6);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
         case LT_SPLINE:
            switch (fill)
            {
               case SOLIDPAT: /* solid black polygon */
                  fprintf (FP, "newpath\n");
                  DumpCurvedPolygonPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT: /* solid white polygon */
                  fprintf (FP, "newpath\n");
                  DumpCurvedPolygonPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                     fprintf (FP,
                           "   pat%1d 8 1 0 72 300 32 div div setpattern\n",
                           fill);
                  fprintf (FP, "   newpath\n");
                  DumpCurvedPolygonPoints (FP, num_pts, v, 6);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
      }
   }

   if (pen == NONEPAT) { fprintf (FP, "\n"); return; }

   fprintf (FP, "%1d setlinewidth", widthOfLine[width]);
   if (dash != 0)
   {
      fprintf (FP, " [");
      for (i = 0; i < dashListLength[dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[dash][dashListLength[dash]-1]));
   }
   else
      fprintf (FP, "\n");

   switch (curved)
   {
      case LT_STRAIGHT:
         switch (pen)
         {
            case SOLIDPAT:
               fprintf (FP, "newpath\n   %1d %1d moveto\n", v[0].x, v[0].y);
               DumpPoints (FP, num_pts-1, v, 3);
               fprintf (FP, "closepath stroke\n");
               break;
            case BACKPAT:
               fprintf (FP, "newpath\n   %1d %1d moveto\n", v[0].x, v[0].y);
               DumpPoints (FP, num_pts-1, v, 3);
               fprintf (FP, "closepath 1 setgray stroke\n");
               if (colorDump)
                  fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                        ((float)tgifColors[color_index].red/maxRGB),
                        ((float)tgifColors[color_index].green/maxRGB),
                        ((float)tgifColors[color_index].blue/maxRGB));
               else
                  fprintf (FP, "0 setgray\n");
               break;
            default:
               fprintf (FP, "gsave\n");
               if (!colorDump)
                  fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n",
                        pen);
               fprintf (FP, "   newpath\n");
               fprintf (FP, "      %1d %1d moveto\n", v[0].x, v[0].y);
               DumpPoints (FP, num_pts-1, v, 6);
               if (colorDump)
               {
                  fprintf (FP, "   closepath strokepath clip\n");
                  DumpPatFill (FP, pen, 8, ObjPtr->bbox, "   ");
               }
               else
                  fprintf (FP, "   closepath stroke\n");
               fprintf (FP, "grestore\n");
               break;
         }
         break;
      case LT_SPLINE:
         switch (pen)
         {
            case SOLIDPAT:
               fprintf (FP, "newpath\n");
               DumpCurvedPolygonPoints (FP, num_pts, v, 3);
               fprintf (FP, "closepath stroke\n");
               break;
            case BACKPAT:
               fprintf (FP, "newpath\n");
               DumpCurvedPolygonPoints (FP, num_pts, v, 3);
               fprintf (FP, "closepath 1 setgray stroke 0 setgray\n");
               break;
            default:
               fprintf (FP, "gsave\n");
               if (!colorDump)
                  fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n",
                        pen);
               fprintf (FP, "   newpath\n");
               DumpCurvedPolygonPoints (FP, num_pts, v, 6);
               if (colorDump)
               {
                  fprintf (FP, "   closepath strokepath clip\n");
                  DumpPatFill (FP, pen, 8, ObjPtr->bbox, "   ");
               }
               else
                  fprintf (FP, "   closepath stroke\n");
               fprintf (FP, "grestore\n");
               break;
         }
         break;
   }
   if (dash != 0) fprintf (FP, "[] 0 setdash\n");
   fprintf (FP, "1 setlinewidth\n\n");
}

void DrawPolygonObj (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   register int		i;
   XPoint		* v;
   struct PolygonRec	* polygon_ptr = ObjPtr->detail.g;
   int			fill, width, pen, curved, dash, pixel;
   int			real_x_off, real_y_off;
   XGCValues		values;

   fill = polygon_ptr->fill;
   width = polygon_ptr->width;
   pen = polygon_ptr->pen;
   curved = polygon_ptr->curved;
   dash = polygon_ptr->dash;
   pixel = colorPixels[ObjPtr->color];

   if (curved == LT_SPLINE)
      DrawSplinePolygonObj (Win, XOff, YOff, fill, width, pen, dash, pixel,
            polygon_ptr);
   else
   {
#ifndef UC
      real_x_off = (XOff >> zoomScale) << zoomScale;
      real_y_off = (YOff >> zoomScale) << zoomScale;
#else /* UC */
      real_x_off = RealSize((ScreenSize(XOff ,  zoomScale)) ,  zoomScale);
      real_y_off = RealSize((ScreenSize(YOff ,  zoomScale)) ,  zoomScale);
#endif /* UC */

      v = (XPoint *) calloc (polygon_ptr->n+1, sizeof(XPoint));
      for (i = 0; i < polygon_ptr->n; i++)
      {
#ifndef UC
         v[i].x = (polygon_ptr->vlist[i].x - real_x_off) >> zoomScale;
         v[i].y = (polygon_ptr->vlist[i].y - real_y_off) >> zoomScale;
#else /* UC */
         v[i].x = ScreenSize((polygon_ptr->vlist[i].x - real_x_off) ,  zoomScale);
         v[i].y = ScreenSize((polygon_ptr->vlist[i].y - real_y_off) ,  zoomScale);
#endif /* UC */
      }

      if (fill != 0)
      {
         values.foreground = (fill == 2) ? myBgPixel : pixel;
         values.function = GXcopy;
         values.fill_style = FillOpaqueStippled;
         values.stipple = patPixmap[fill];
         XChangeGC (mainDisplay, drawGC,
               GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
         XFillPolygon (mainDisplay, Win, drawGC, v, polygon_ptr->n, Complex,
               CoordModeOrigin);
      }

      if (pen != 0)
      {
         values.foreground = (pen == 2) ? myBgPixel : pixel;
         values.function = GXcopy;
         values.fill_style = FillOpaqueStippled;
         values.stipple = patPixmap[pen];
#ifndef UC
         values.line_width = widthOfLine[width] >> zoomScale;
#else /* UC */
         values.line_width = ScreenSize(widthOfLine[width] ,  zoomScale);
#endif /* UC */
         if (dash != 0)
         {
            XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
                  dashListLength[dash]);
            values.line_style = LineOnOffDash;
         }
         else
            values.line_style = LineSolid;
         XChangeGC (mainDisplay, drawGC,
               GCForeground | GCFunction | GCFillStyle | GCStipple |
               GCLineWidth | GCLineStyle, &values);

         XDrawLines (mainDisplay, Win, drawGC, v, polygon_ptr->n,
               CoordModeOrigin);
      }
      cfree (v);
   }
}

static
void CreatePolygonObj (NumPts)
   int	NumPts;
{
   register int		i;
   struct PtRec		* pt_ptr;
   struct PolygonRec	* polygon_ptr;
   struct ObjRec	* obj_ptr;
   XPoint		* v;
   int			ltx, lty, rbx, rby, w;

   polygon_ptr = (struct PolygonRec *) calloc (1, sizeof(struct PolygonRec));
   polygon_ptr->n = NumPts;
   v = (XPoint *) calloc (NumPts+1, sizeof(XPoint));
   pt_ptr = lastPtPtr;
   ltx = rbx = pt_ptr->x;
   lty = rby = pt_ptr->y;
   for (i = NumPts-1; i >= 0; i--, lastPtPtr = lastPtPtr->next)
   {
#ifndef UC
      v[i].x = (lastPtPtr->x << zoomScale) + drawOrigX;
      v[i].y = (lastPtPtr->y << zoomScale) + drawOrigY;
#else /* UC */
      v[i].x = (RealSize(lastPtPtr->x ,  zoomScale)) + drawOrigX;
      v[i].y = (RealSize(lastPtPtr->y ,  zoomScale)) + drawOrigY;
#endif /* UC */
      if (lastPtPtr->x < ltx) ltx = lastPtPtr->x;
      if (lastPtPtr->y < lty) lty = lastPtPtr->y;
      if (lastPtPtr->x > rbx) rbx = lastPtPtr->x;
      if (lastPtPtr->y > rby) rby = lastPtPtr->y;
      cfree (lastPtPtr);
   }
   polygon_ptr->vlist = v;
   if (curSpline == LT_SPLINE)
      polygon_ptr->svlist = MakeSplinePolygonVertex (&(polygon_ptr->sn),
            drawOrigX, drawOrigY, NumPts, v);
   polygon_ptr->fill = objFill;
   polygon_ptr->width = lineWidth;
   polygon_ptr->pen = penPat;
   polygon_ptr->curved = curSpline;
   polygon_ptr->dash = curDash;
   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   obj_ptr->color = colorIndex;
   obj_ptr->type = OBJ_POLYGON;
   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x =
#ifndef UC
         (ltx << zoomScale) + drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y =
         (lty << zoomScale) + drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = (rbx << zoomScale) + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = (rby << zoomScale) + drawOrigY;
#else /* UC */
         (RealSize(ltx ,  zoomScale)) + drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y =
         (RealSize(lty ,  zoomScale)) + drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = (RealSize(rbx ,  zoomScale)) + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = (RealSize(rby ,  zoomScale)) + drawOrigY;
#endif /* UC */
   w = widthOfLine[lineWidth];
   obj_ptr->bbox.ltx -= w;
   obj_ptr->bbox.lty -= w;
   obj_ptr->bbox.rbx += w;
   obj_ptr->bbox.rby += w;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->detail.g = polygon_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   AddObj (NULL, topObj, obj_ptr);
}

static
void ContinuePolygon (OrigX, OrigY)
   int 	OrigX, OrigY;
   /* OrigX and OrigY are screen coordinates (scaled and translated). */
{
   XGCValues	values;
   XEvent	input;
   XButtonEvent	* button_ev;
   XMotionEvent	* motion_ev;
   int 		end_x, end_y, grid_x, grid_y, done = FALSE;
   int		ltx, lty, rbx, rby;
   int		pixel, xor_pixel, num_pts = 1;
   struct PtRec	* pt_ptr;
#ifdef UC
   register int	showsize = showSize;
   register int	hidedragcursor = hideDragCursor;
   register int cursordefined = FALSE;
#endif /* UC */

   pixel = colorPixels[colorIndex];
   xor_pixel = xorColorPixels[colorIndex];

   values.foreground = xor_pixel;
   values.function = GXxor;
   values.fill_style = FillSolid;
   values.line_width = 0;
   values.line_style = LineSolid;

   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = end_x = OrigX;
   grid_y = end_y = OrigY; 
   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonPressMask,
#ifndef UC
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
#else /* UC */
         GrabModeAsync, GrabModeAsync, None,
         ((hidedragcursor)? None: handCursor), CurrentTime);
#endif /* UC */
#ifdef UC
   if (showsize) showPoint (OrigX, OrigY, RIGHTUP);
#endif /* UC */
   
   while (!done)
   {
      XNextEvent (mainDisplay, &input);
      if (input.type == MotionNotify)
      {
         XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
               grid_y);
#ifdef UC
	 if (showsize) eraseSizeString ();
#endif /* UC */
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
         MarkRulers (grid_x, grid_y);
         XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
               grid_y);
#ifdef UC
	 if (showsize) showLength (OrigX, OrigY, grid_x, grid_y, RIGHTUP);
	 if (hidedragcursor && ! cursordefined && 
			(grid_x != OrigX && grid_y != OrigY))
	 {
	     XDefineCursor (mainDisplay, drawWindow, dragCursor);
	     cursordefined = TRUE;
	 }
#endif /* UC */
      }
      else if (input.type == ButtonPress)
      {
         button_ev = &(input.xbutton);
         XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
               grid_y);
         end_x = button_ev->x;
         end_y = button_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);

         if (penPat != 0 && (button_ev->button == Button1 || num_pts != 1))
         {
            values.foreground = (penPat == 2) ? myBgPixel : pixel;
            values.function = GXcopy;
            values.fill_style = FillOpaqueStippled;
#ifndef UC
            values.line_width = widthOfLine[lineWidth] >> zoomScale;
#else /* UC */
            values.line_width = ScreenSize(widthOfLine[lineWidth] ,  zoomScale);
#endif /* UC */
            values.stipple = patPixmap[penPat];
            XChangeGC (mainDisplay, drawGC,
                  GCForeground | GCFunction | GCFillStyle | GCLineWidth |
                  GCStipple, &values);

            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                  grid_y);

            values.foreground = xor_pixel;
            values.function = GXxor;
            values.fill_style = FillSolid;
            values.line_width = 0;
            XChangeGC (mainDisplay, drawGC,
                  GCForeground | GCFunction | GCFillStyle | GCLineWidth,
                  &values);
#ifdef UC
	    if (hidedragcursor && cursordefined)
	    {
		XDefineCursor (mainDisplay, drawWindow, drawCursor);
		cursordefined = FALSE;
	    }
#endif /* UC */
         }

         if ((grid_x != OrigX || grid_y != OrigY) &&
               (button_ev->button == Button1 || num_pts != 1))
         {
            num_pts++;
            pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
            pt_ptr->next = lastPtPtr;
            lastPtPtr = pt_ptr;
            pt_ptr->x = grid_x;
            pt_ptr->y = grid_y;
         }

         if (grid_x == startPolygonX && grid_y == startPolygonY)
            done = TRUE;
         else
         {
            switch(button_ev->button)
            {
               case Button1 : OrigX = grid_x; OrigY = grid_y; break;
               case Button2 :
               case Button3 :
                  if (num_pts == 1)
                  {
                     done = TRUE;
                     break;
                  }

                  values.foreground = (penPat == 2) ? myBgPixel : pixel;
                  values.function = GXcopy;
                  values.fill_style = FillOpaqueStippled;
#ifndef UC
                  values.line_width = widthOfLine[lineWidth] >> zoomScale;
#else /* UC */
                  values.line_width = ScreenSize(widthOfLine[lineWidth] ,  zoomScale);
#endif /* UC */
                  values.stipple = patPixmap[penPat];
                  XChangeGC (mainDisplay, drawGC,
                        GCForeground | GCFunction | GCFillStyle | GCLineWidth |
                        GCStipple, &values);

                  XDrawLine (mainDisplay, drawWindow, drawGC, grid_x, grid_y,
                        startPolygonX, startPolygonY);

                  num_pts++;
                  pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
                  pt_ptr->next = lastPtPtr;
                  lastPtPtr = pt_ptr;
                  pt_ptr->x = startPolygonX;
                  pt_ptr->y = startPolygonY;
                  done = TRUE;
                  break;
            }
         }
      }
   }
   XUngrabPointer (mainDisplay, CurrentTime);
#ifdef UC
   if (hidedragcursor && cursordefined)
   {
       XDefineCursor (mainDisplay, drawWindow, drawCursor);
       cursordefined = FALSE;
   }
#endif /* UC */

   switch (num_pts)
   {
      case 1:
#ifndef UC
         RedrawAnArea (botObj, (OrigX+drawOrigX)-(1<<zoomScale),
               (OrigY+drawOrigY)-(1<<zoomScale),
               (OrigX+drawOrigX)+(1<<zoomScale),
               (OrigY+drawOrigY)+(1<<zoomScale));
#else /* UC */
         RedrawAnArea (botObj, (OrigX+drawOrigX)-(RealSize(1, zoomScale)),
               (OrigY+drawOrigY)-(RealSize(1, zoomScale)),
               (OrigX+drawOrigX)+(RealSize(1, zoomScale)),
               (OrigY+drawOrigY)+(RealSize(1, zoomScale)));
#endif /* UC */
         cfree (lastPtPtr);
         break;
      case 2:
         CalcBBox (lastPtPtr->x, lastPtPtr->y, lastPtPtr->next->x,
               lastPtPtr->next->y, &ltx, &lty, &rbx, &rby);
#ifndef UC
         RedrawAnArea (botObj, (ltx+drawOrigX)-(1<<zoomScale),
               (lty+drawOrigY)-(1<<zoomScale), (rbx+drawOrigX)+(1<<zoomScale),
               (rby+drawOrigY)+(1<<zoomScale));
#else /* UC */
         RedrawAnArea (botObj, (ltx+drawOrigX)-(RealSize(1, zoomScale)),
               (lty+drawOrigY)-(RealSize(1, zoomScale)), (rbx+drawOrigX)+(RealSize(1, zoomScale)),
               (rby+drawOrigY)+(RealSize(1, zoomScale)));
#endif /* UC */
         cfree (lastPtPtr->next);
         cfree (lastPtPtr);
         break;
      default:
         CreatePolygonObj (num_pts);
#ifndef UC
         RedrawAnArea (botObj, topObj->bbox.ltx-(1<<zoomScale),
               topObj->bbox.lty-(1<<zoomScale),
               topObj->bbox.rbx+(1<<zoomScale),
               topObj->bbox.rby+(1<<zoomScale));
#else /* UC */
         RedrawAnArea (botObj, topObj->bbox.ltx-(RealSize(1, zoomScale)),
               topObj->bbox.lty-(RealSize(1, zoomScale)),
               topObj->bbox.rbx+(RealSize(1, zoomScale)),
               topObj->bbox.rby+(RealSize(1, zoomScale)));
#endif /* UC */
         polygonDrawn = TRUE;
         SetFileModified (TRUE);
         break;
   }
}

void DrawPolygon (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   int		mouse_x, mouse_y, grid_x, grid_y;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      lastPtPtr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
      lastPtPtr->x = startPolygonX = grid_x;
      lastPtPtr->y = startPolygonY = grid_y;
      lastPtPtr->next = NULL;
      ContinuePolygon (grid_x, grid_y);
   } 
}

void SavePolygonObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int		i, n;
   struct PolygonRec	* polygon_ptr = ObjPtr->detail.g;

   n = polygon_ptr->n;
   fprintf (FP, "polygon('%s',%1d,[", colorMenuItems[ObjPtr->color],
         polygon_ptr->n);
   for (i = 0; i < n-1; i++)
      fprintf (FP, "%1d,%1d,", polygon_ptr->vlist[i].x, polygon_ptr->vlist[i].y);

   fprintf (FP, "%1d,%1d],%1d,%1d,%1d,%1d,%1d,%1d,", polygon_ptr->vlist[n-1].x,
         polygon_ptr->vlist[n-1].y, polygon_ptr->fill, polygon_ptr->width,
         polygon_ptr->pen, polygon_ptr->curved, ObjPtr->id, polygon_ptr->dash);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

void ReadPolygonObj (Inbuf, ObjPtr)
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register int		i;
   struct PolygonRec	* polygon_ptr;
   XPoint		* v;
   char			color_str[20], * s;
   int			num_pts, ltx, lty, rbx, rby, x, y, fill, width, pen, w;
   int			curved, dash;

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);
   sscanf (s, "%d,", &num_pts);
   polygon_ptr = (struct PolygonRec *) calloc (1, sizeof(struct PolygonRec));
   polygon_ptr->n = num_pts;
   v = (XPoint *) calloc (num_pts+1, sizeof(XPoint));
   ltx = 20 * PIX_PER_INCH; rbx = 0;
   lty = 20 * PIX_PER_INCH; rby = 0;
   s = FindChar ('[', s);
   for (i = 0; i < num_pts; i++)
   {
      sscanf (s, "%d , %d", &x, &y);
      v[i].x = x;
      v[i].y = y;
      if (x < ltx) ltx = x;
      if (y < lty) lty = y;
      if (x > rbx) rbx = x;
      if (y > rby) rby = y;
      s = FindChar (',',s);
      s = FindChar (',',s);
   }

   if (fileVersion <= 3)
   {
      sscanf (s, "%d , %d , %d", &fill, &width, &pen);
      if (width == LINE_CURVED)
      {
         width = 0;
         curved = TRUE;
      }
      else
         curved = FALSE;
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
      (*ObjPtr)->id = objId++;
      dash = 0;
   }
   else if (fileVersion <= 5)
   {
      sscanf (s, "%d , %d , %d , %d", &fill, &width, &pen, &curved);
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
      (*ObjPtr)->id = objId++;
      dash = 0;
   }
   else if (fileVersion <= 7)
   {
      sscanf (s, "%d , %d , %d , %d", &fill, &width, &pen, &curved);
      (*ObjPtr)->id = objId++;
      dash = 0;
   }
   else if (fileVersion <= 8)
   {
      sscanf (s, "%d , %d , %d , %d , %d", &fill, &width, &pen, &curved,
            &((*ObjPtr)->id));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
      dash = 0;
   }
   else
   {
      sscanf (s, "%d , %d , %d , %d , %d , %d", &fill, &width, &pen, &curved,
            &((*ObjPtr)->id), &dash);
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
   }

   polygon_ptr->vlist = v;
   if (curved == LT_SPLINE)
      polygon_ptr->svlist = MakeSplinePolygonVertex (&(polygon_ptr->sn),
            drawOrigX, drawOrigY, num_pts, v);
   polygon_ptr->fill = fill;
   polygon_ptr->width = width;
   polygon_ptr->pen = pen;
   polygon_ptr->curved = curved;
   polygon_ptr->dash = dash;
   (*ObjPtr)->x = ltx;
   (*ObjPtr)->y = lty;
   (*ObjPtr)->color = FindColorIndex (color_str);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->type = OBJ_POLYGON;
   (*ObjPtr)->obbox.ltx = ltx;
   (*ObjPtr)->obbox.lty = lty;
   (*ObjPtr)->obbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = rby;
   if (width == 0)
   {
      (*ObjPtr)->bbox.ltx = ltx;
      (*ObjPtr)->bbox.lty = lty;
      (*ObjPtr)->bbox.rbx = rbx;
      (*ObjPtr)->bbox.rby = rby;
   }
   else
   {
      w = widthOfLine[width];
      (*ObjPtr)->bbox.ltx = ltx - w;
      (*ObjPtr)->bbox.lty = lty - w;
      (*ObjPtr)->bbox.rbx = rbx + w;
      (*ObjPtr)->bbox.rby = rby + w;
   }
   (*ObjPtr)->detail.g = polygon_ptr;
}

void FreePolygonObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   if (ObjPtr->detail.g->curved) cfree (ObjPtr->detail.g->svlist);
   cfree (ObjPtr->detail.g->vlist);
   cfree (ObjPtr->detail.g);
   cfree (ObjPtr);
}
