/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/rcbox.c,v 2.0 91/03/05 12:48:08 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "color.e"
#include "cursor.e"
#include "file.e"
#include "grid.e"
#include "obj.e"
#include "pattern.e"
#include "poly.e"
#include "raster.e"
#include "ruler.e"
#include "setup.e"
#ifdef UC
#include "showsize.e"
#endif /* UC */

#define RC_DEF_R (EIGHTH_INCH)

int	rcBoxDrawn = FALSE;

static XSegment	rcbSegs[4];
static XArc	rcbArcs[4];
static int	rcbArcsInitialized = FALSE;

void SetRCBoxVertex (x1, y1, x2, y2, r)
   int	x1, y1, x2, y2, r;
{
   register int	inc_x, inc_y, d=2*r;

   inc_x = (x2 > x1);
   inc_y = (y2 > y1);
   rcbSegs[0].x1 = (inc_x) ? (x1+r) : (x1-r); rcbSegs[0].y1 = y1;
   rcbSegs[0].x2 = (inc_x) ? (x2-r) : (x2+r); rcbSegs[0].y2 = y1;
   rcbSegs[1].x1 = x2; rcbSegs[1].y1 = (inc_y) ? (y1+r) : (y1-r);
   rcbSegs[1].x2 = x2; rcbSegs[1].y2 = (inc_y) ? (y2-r) : (y2+r);
   rcbSegs[2].x1 = (inc_x) ? (x1+r) : (x1-r); rcbSegs[2].y1 = y2;
   rcbSegs[2].x2 = (inc_x) ? (x2-r) : (x2+r); rcbSegs[2].y2 = y2;
   rcbSegs[3].x1 = x1; rcbSegs[3].y1 = (inc_y) ? (y1+r) : (y1-r);
   rcbSegs[3].x2 = x1; rcbSegs[3].y2 = (inc_y) ? (y2-r) : (y2+r);

   if (!rcbArcsInitialized)
   {
      rcbArcsInitialized = TRUE;

      rcbArcs[0].angle1 = 90*64;   rcbArcs[0].angle2 = 90*64;
      rcbArcs[1].angle1 = 0;       rcbArcs[1].angle2 = 90*64;
      rcbArcs[2].angle1 = -90*64;  rcbArcs[2].angle2 = 90*64;
      rcbArcs[3].angle1 = -180*64; rcbArcs[3].angle2 = 90*64;
   }
   rcbArcs[0].width=rcbArcs[1].width=rcbArcs[2].width=rcbArcs[3].width=d;
   rcbArcs[0].height=rcbArcs[1].height=rcbArcs[2].height=rcbArcs[3].height=d;

   if (inc_x)
   {
      if (inc_y)
      {
         rcbArcs[0].x=x1; rcbArcs[0].y=y1;
         rcbArcs[1].x=x2-d; rcbArcs[1].y=y1;
         rcbArcs[2].x=x2-d; rcbArcs[2].y=y2-d;
         rcbArcs[3].x=x1; rcbArcs[3].y=y2-d;
      }
      else
      {
         rcbArcs[0].x=x1; rcbArcs[0].y=y2;
         rcbArcs[1].x=x2-d; rcbArcs[1].y=y2;
         rcbArcs[2].x=x2-d; rcbArcs[2].y=y1-d;
         rcbArcs[3].x=x1; rcbArcs[3].y=y1-d;
      }
   }
   else
   {
      if (inc_y)
      {
         rcbArcs[0].x=x2; rcbArcs[0].y=y1;
         rcbArcs[1].x=x1-d; rcbArcs[1].y=y1;
         rcbArcs[2].x=x1-d; rcbArcs[2].y=y2-d;
         rcbArcs[3].x=x2; rcbArcs[3].y=y2-d;
      }
      else
      {
         rcbArcs[0].x=x2; rcbArcs[0].y=y2;
         rcbArcs[1].x=x1-d; rcbArcs[1].y=y2;
         rcbArcs[2].x=x1-d; rcbArcs[2].y=y1-d;
         rcbArcs[3].x=x2; rcbArcs[3].y=y1-d;
      }
   }
}

void MyRCBox (window, gc, x1, y1, x2, y2, r)
   Window	window;
   GC		gc;
   int		x1, y1, x2, y2, r;
{
   if (abs(x1-x2) < 2*r || abs(y1-y2) < 2*r)
      MyBox (window, gc, x1, y1, x2, y2);
   else
   {
      XDrawSegments (mainDisplay, window, gc, rcbSegs, 4);
      XDrawArcs (mainDisplay, window, gc, rcbArcs, 4);
   }
}
 
static
void MyFillRCBox (window, gc, x1, y1, x2, y2, r)
   Window	window;
   GC		gc;
   int		x1, y1, x2, y2, r;
{
   if (abs(x1-x2) < 2*r || abs(y1-y2) < 2*r)
      XFillRectangle (mainDisplay, window, gc, x1, y1, x2-x1, y2-y1);
   else
   {
      XFillRectangle (mainDisplay, window, gc, x1+r, y1, x2-x1-2*r, y2-y1);
      XFillRectangle (mainDisplay, window, gc, x1, y1+r, x2-x1, y2-y1-2*r);
      XFillArcs (mainDisplay, window, gc, rcbArcs, 4);
   }
}

static
void DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, blank1, blank2)
   FILE	* FP;
   int	ltx, lty, rbx, rby, r;
   char	* blank1, * blank2;
{
   if (abs(ltx-rbx) < 2*r || abs(lty-rby) < 2*r)
   {
      fprintf (FP, "%snewpath\n%s%1d %1d moveto\n", blank1, blank2, rbx, lty);
      fprintf (FP, "%s%1d %1d lineto\n", blank2, rbx, rby);
      fprintf (FP, "%s%1d %1d lineto\n", blank2, ltx, rby);
      fprintf (FP, "%s%1d %1d lineto\n", blank2, ltx, lty);
   }
   else
   {
      fprintf (FP, "%snewpath\n%s%1d %1d moveto\n", blank1, blank2, rbx-r, lty);
      fprintf (FP, "%s%1d %1d %1d %1d %1d arcto 4 {pop} repeat\n", blank2,
            rbx, lty, rbx, rby, r);
      fprintf (FP, "%s%1d %1d lineto\n", blank2, rbx, rby-r);
      fprintf (FP, "%s%1d %1d %1d %1d %1d arcto 4 {pop} repeat\n", blank2,
            rbx, rby, ltx, rby, r);
      fprintf (FP, "%s%1d %1d lineto\n", blank2, ltx+r, rby);
      fprintf (FP, "%s%1d %1d %1d %1d %1d arcto 4 {pop} repeat\n", blank2,
            ltx, rby, ltx, lty, r);
      fprintf (FP, "%s%1d %1d lineto\n", blank2, ltx, lty+r);
      fprintf (FP, "%s%1d %1d %1d %1d %1d arcto 4 {pop} repeat\n", blank2,
            ltx, lty, rbx, lty, r);
   }
}
 
void DumpRCBoxObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int	ltx, lty, rbx, rby, i;
   int		fill, width, pen, dash, color_index, r;

   ltx = ObjPtr->obbox.ltx;
   lty = ObjPtr->obbox.lty;
   rbx = ObjPtr->obbox.rbx;
   rby = ObjPtr->obbox.rby;

   fill = ObjPtr->detail.rcb->fill;
   pen = ObjPtr->detail.rcb->pen;
   width = ObjPtr->detail.rcb->width;
   dash = ObjPtr->detail.rcb->dash;
   r = ObjPtr->detail.rcb->radius;

   if (fill == NONEPAT && pen == NONEPAT) return;

   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   switch (fill)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, "", "   ");
         fprintf (FP, "closepath fill\n");
         break;
      case BACKPAT:
         DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, "", "   ");
         fprintf (FP, "closepath 1 setgray fill\n");
         if (colorDump)
            fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                  ((float)tgifColors[color_index].red/maxRGB),
                  ((float)tgifColors[color_index].green/maxRGB),
                  ((float)tgifColors[color_index].blue/maxRGB));
         else
            fprintf (FP, "0 setgray\n");
         break;
      default:
         fprintf (FP, "gsave\n");
         if (!colorDump)
            fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n",fill);
         DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, "   ", "      ");
         if (colorDump)
         {
            fprintf (FP, "   closepath eoclip\n");
            DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "   closepath fill\n");
         fprintf (FP, "grestore\n");
         break;
   }

   fprintf (FP, "%1d setlinewidth\n", widthOfLine[width]);
   if (dash != 0)
   {
      fprintf (FP, "[");
      for (i = 0; i < dashListLength[dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[dash][dashListLength[dash]-1]));
   }

   switch (pen)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, "", "   ");
         fprintf (FP, "closepath stroke\n");
         break;
      case BACKPAT:
         DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, "", "   ");
         fprintf (FP, "closepath 1 setgray stroke 0 setgray\n");
         break;
      default:
         fprintf (FP, "gsave\n");
         if (!colorDump)
            fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n", pen);
         DumpRCBoxPSPath (FP, ltx, lty, rbx, rby, r, "   ", "      ");
         if (colorDump)
         {
            fprintf (FP, "   closepath strokepath clip\n");
            DumpPatFill (FP, pen, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "   closepath stroke\n");
         fprintf (FP, "grestore\n");
         break;
   }
   if (dash != 0) fprintf (FP, "[] 0 setdash\n");
   fprintf (FP, "1 setlinewidth\n\n");
}

void DrawRCBoxObj (win, XOff, YOff, ObjPtr)
   Window		win;
   struct ObjRec	* ObjPtr;
{
   struct RCBoxRec	* rcbox_ptr = ObjPtr->detail.rcb;
   int			fill, pen, pixel, func, ltx, lty, rbx, rby, width, dash;
   int			real_x_off, real_y_off, radius;
   XGCValues		values;

   pen = rcbox_ptr->pen;
   fill = rcbox_ptr->fill;
   width = rcbox_ptr->width;
   dash = rcbox_ptr->dash;
   radius = rcbox_ptr->radius;
   pixel = colorPixels[ObjPtr->color];

   if (fill == 0 && pen == 0) return;

#ifndef UC
   real_x_off = (XOff >> zoomScale) << zoomScale;
   real_y_off = (YOff >> zoomScale) << zoomScale;
   ltx = (ObjPtr->obbox.ltx - real_x_off) >> zoomScale;
   lty = (ObjPtr->obbox.lty - real_y_off) >> zoomScale;
   rbx = (ObjPtr->obbox.rbx - real_x_off) >> zoomScale;
   rby = (ObjPtr->obbox.rby - real_y_off) >> zoomScale;
#else /* UC */
   real_x_off = RealSize((ScreenSize(XOff ,  zoomScale)) ,  zoomScale);
   real_y_off = RealSize((ScreenSize(YOff ,  zoomScale)) ,  zoomScale);
   ltx = ScreenSize((ObjPtr->obbox.ltx - real_x_off) ,  zoomScale);
   lty = ScreenSize((ObjPtr->obbox.lty - real_y_off) ,  zoomScale);
   rbx = ScreenSize((ObjPtr->obbox.rbx - real_x_off) ,  zoomScale);
   rby = ScreenSize((ObjPtr->obbox.rby - real_y_off) ,  zoomScale);
#endif /* UC */
   func = GXcopy;

   SetRCBoxVertex (ltx, lty, rbx, rby, radius);

   if (fill != 0)
   {
      values.foreground = (fill == 2) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      MyFillRCBox (win, drawGC, ltx, lty, rbx, rby, radius);
   }

   if (pen != 0)
   {
      values.foreground = (pen == 2) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[pen];
#ifndef UC
      values.line_width = widthOfLine[width] >> zoomScale;
#else /* UC */
      values.line_width = ScreenSize(widthOfLine[width] ,  zoomScale);
#endif /* UC */
      if (dash != 0)
      {
         XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
               dashListLength[dash]);
         values.line_style = LineOnOffDash;
      }
      else
         values.line_style = LineSolid;
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple | GCLineWidth |
            GCLineStyle, &values);
      MyRCBox (win, drawGC, ltx, lty, rbx, rby, radius);
   }
}

static
void CreateRCBoxObj (X1, Y1, X2, Y2)
   int	X1, Y1, X2, Y2;
{
   struct RCBoxRec	* rcbox_ptr;
   struct ObjRec	* obj_ptr;
   int			w, ltx, lty, rbx, rby;

   rcbox_ptr = (struct RCBoxRec *) calloc (1, sizeof(struct RCBoxRec));
   rcbox_ptr->fill = objFill;
   rcbox_ptr->width = lineWidth;
   rcbox_ptr->pen = penPat;
   rcbox_ptr->dash = curDash;
   rcbox_ptr->radius = RC_DEF_R;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));

   if (X1 < X2)
      if (Y1 < Y2)
      {
         ltx = X1; lty = Y1; rbx = X2; rby = Y2;
      }
      else
      {
         ltx = X1; lty = Y2; rbx = X2; rby = Y1;
      }
   else
      if (Y1 < Y2)
      {
         ltx = X2; lty = Y1; rbx = X1; rby = Y2;
      }
      else
      {
         ltx = X2; lty = Y2; rbx = X1; rby = Y1;
      }

#ifndef UC
   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = (ltx << zoomScale) +
         drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = (lty << zoomScale) +
         drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = (rbx << zoomScale) + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = (rby << zoomScale) + drawOrigY;
#else /* UC */
   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = (RealSize(ltx ,  zoomScale)) +
         drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = (RealSize(lty ,  zoomScale)) +
         drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = (RealSize(rbx ,  zoomScale)) + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = (RealSize(rby ,  zoomScale)) + drawOrigY;
#endif /* UC */
   w = widthOfLine[lineWidth];
   obj_ptr->bbox.ltx -= w;
   obj_ptr->bbox.lty -= w;
   obj_ptr->bbox.rbx += w;
   obj_ptr->bbox.rby += w;
   obj_ptr->type = OBJ_RCBOX;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->detail.rcb = rcbox_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   AddObj (NULL, topObj, obj_ptr);
}
 
static
void ContinueRCBox (OrigX, OrigY)
   int 	OrigX, OrigY;
{
   int 		end_x, end_y, grid_x, grid_y, saved_x, saved_y;
   int 		done = FALSE;
   int		pixel, xor_pixel;
   XGCValues	values;
   XEvent	input;
   XMotionEvent	* motion_ev;
#ifdef UC
   register int showsize = showSize;
   register int hidedragcursor = hideDragCursor;
   register int cursordefined = FALSE;
#endif /* UC */

   pixel = colorPixels[colorIndex];
   xor_pixel = xorColorPixels[colorIndex];

   values.foreground = xor_pixel;
   values.function = GXxor;
   values.fill_style = FillSolid;
   values.line_width = 0;
   values.line_style = LineSolid;

   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   saved_x = grid_x = OrigX;
   saved_y = grid_y = OrigY; 
   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonReleaseMask,
#ifndef UC
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
#else /* UC */
         GrabModeAsync, GrabModeAsync, None,
   	 ((hidedragcursor)? None: handCursor), CurrentTime);
#endif /* UC */
#ifdef UC
   if (showsize) showPoint (OrigX, OrigY, RIGHTUP);
#endif /* UC */
   
   SetRCBoxVertex (OrigX, OrigY, saved_x, saved_y, RC_DEF_R);
   while (!done)
   {
      XNextEvent (mainDisplay, &input);
      if (input.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         MyRCBox (drawWindow,drawGC,OrigX,OrigY,saved_x,saved_y,RC_DEF_R);
#ifdef UC
	 if (showsize) eraseSizeString ();
	 if (hidedragcursor && cursordefined)
	 {
	     XDefineCursor (mainDisplay, drawWindow, drawCursor);
	     cursordefined = FALSE;
	 }
#endif /* UC */
         done = TRUE;
      }
      else if (input.type == MotionNotify)
      {
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
#ifdef UC
	 if (hidedragcursor && !cursordefined && (OrigX != grid_x || OrigY != grid_y))
	 {
	     XDefineCursor (mainDisplay, drawWindow, dragCursor);
	     cursordefined = TRUE;
	 }
#endif /* UC */
         if (grid_x != saved_x || grid_y != saved_y)
         {
            MyRCBox (drawWindow,drawGC,OrigX,OrigY,saved_x,saved_y,RC_DEF_R);
#ifdef UC
	    if (showsize) eraseSizeString ();
#endif /* UC */
            saved_x = grid_x;
            saved_y = grid_y;
            SetRCBoxVertex (OrigX, OrigY, saved_x, saved_y, RC_DEF_R);
            MyRCBox (drawWindow,drawGC,OrigX,OrigY,saved_x,saved_y,RC_DEF_R);
#ifdef UC
	    if (showsize) showHeightWidth (OrigX, OrigY, saved_x, saved_y, RIGHTDOWN);
#endif /* UC */
         }
         MarkRulers (grid_x, grid_y);
      }
   }
   if (OrigX != grid_x && OrigY != grid_y)
   {
      CreateRCBoxObj (OrigX, OrigY, grid_x, grid_y);
      DrawRCBoxObj (drawWindow, drawOrigX, drawOrigY, topObj);
      rcBoxDrawn = TRUE;
      SetFileModified (TRUE);
   }
}

void DrawRCBox (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   int		mouse_x, mouse_y, grid_x, grid_y;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      ContinueRCBox (grid_x, grid_y);
   }
}

void SaveRCBoxObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   fprintf (FP, "rcbox('%s',", colorMenuItems[ObjPtr->color]);
   fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,", ObjPtr->obbox.ltx,
         ObjPtr->obbox.lty, ObjPtr->obbox.rbx, ObjPtr->obbox.rby,
         ObjPtr->detail.rcb->fill, ObjPtr->detail.rcb->width,
         ObjPtr->detail.rcb->pen, ObjPtr->detail.rcb->dash,
         ObjPtr->detail.rcb->radius, ObjPtr->id);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

void ReadRCBoxObj (Inbuf, ObjPtr)
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   struct RCBoxRec	* rcbox_ptr;
   char			color_str[20], * s;
   int			ltx, lty, rbx, rby, fill, width, pen, dash, w, radius;

   *ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);
   rcbox_ptr = (struct RCBoxRec *) calloc (1, sizeof(struct RCBoxRec));

   if (fileVersion > 8)
   {
      sscanf (s, "%d , %d , %d , %d , %d , %d , %d , %d , %d , %d",
            &ltx, &lty, &rbx, &rby, &fill, &width, &pen, &dash, &radius,
            &((*ObjPtr)->id));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
   }

   rcbox_ptr->fill = fill;
   rcbox_ptr->width = width;
   rcbox_ptr->pen = pen;
   rcbox_ptr->dash = dash;
   rcbox_ptr->radius = radius;
   (*ObjPtr)->x = ltx;
   (*ObjPtr)->y = lty;
   (*ObjPtr)->color = FindColorIndex (color_str);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->type = OBJ_RCBOX;
   (*ObjPtr)->obbox.ltx = ltx;
   (*ObjPtr)->obbox.lty = lty;
   (*ObjPtr)->obbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = rby;
   w = widthOfLine[width];
   (*ObjPtr)->bbox.ltx = ltx - w;
   (*ObjPtr)->bbox.lty = lty - w;
   (*ObjPtr)->bbox.rbx = rbx + w;
   (*ObjPtr)->bbox.rby = rby + w;
   (*ObjPtr)->detail.rcb = rcbox_ptr;
}

void FreeRCBoxObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   cfree (ObjPtr->detail.rcb);
   cfree (ObjPtr);
}
