/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/rect.c,v 2.0 91/03/05 12:48:11 william Exp $";
#endif

#include <X11/Xlib.h>
#include <math.h>
#include "const.h"
#include "types.h"

#include "arc.e"
#include "poly.e"
#include "setup.e"
#include "spline.e"

int BBoxIntersect (Rect1, Rect2)
   struct BBRec	Rect1, Rect2;
{
   if (Rect1.ltx < Rect2.ltx)
      if (Rect1.lty < Rect2.lty)
         return (Rect1.rbx >= Rect2.ltx && Rect1.rby >= Rect2.lty);
      else
         return (Rect1.rbx >= Rect2.ltx && Rect1.lty <= Rect2.rby);
   else
      if (Rect1.lty < Rect2.lty)
         return (Rect1.ltx <= Rect2.rbx && Rect1.rby >= Rect2.lty);
      else
         return (Rect1.ltx <= Rect2.rbx && Rect1.lty <= Rect2.rby);
}

int PointInBBox (X, Y, Rect)
   int	X, Y;
   struct BBRec	Rect;
{
   return (X >= Rect.ltx && Y >= Rect.lty && X <= Rect.rbx && Y <= Rect.rby);
}

int PointInPolygon (X, Y, NumPts, V)
   int		X, Y, NumPts;
   XPoint	* V;
{
   register double	x1, x2, y1, y2;
   double		m, y_int;
   int			n, count = 0;

   x2 = (double)V[0].x;
   y2 = (double)V[0].y;
   for (n = 0; n < NumPts-1; n++)
   {
      x1 = x2;
      y1 = y2;
      x2 = (double)V[n+1].x;
      y2 = (double)V[n+1].y;
      if (x2 == x1 && X == x1)
      {
         if (Y <= min(y1,y2)) count++;
         continue;
      }
      if (x2 > x1)
      {
         if (X >= x2 || X < x1) continue;
      }
      else
      {
         if (X > x1 || X <= x2) continue;
      }

      m = (y1 - y2) / (x1 - x2);
      y_int = m * X + (y1 - m * x1);
      if (Y <= y_int) count++;
   }
   return (count & 0x1);
}

int PointInPoly (X, Y, NumPts, V, W)
   int		X, Y, NumPts, W;
   XPoint	* V;
{
   register double	x1, x2, y1, y2;
   double		m, m1, b, b1, x_int, y_int;
   int			n;

   x2 = (double)V[0].x;
   y2 = (double)V[0].y;
   for (n = 0; n < NumPts-1; n++)
   {
      x1 = x2;
      y1 = y2;
      x2 = (double)V[n+1].x;
      y2 = (double)V[n+1].y;
      if (x2 >= x1)
      {
         if (X < x1-W || X > x2+W) continue;
      }
      else
      {
         if (X < x2-W || X > x1+W) continue;
      }
      if (y2 >= y1)
      {
         if (Y < y1-W || Y > y2+W) continue;
      }
      else
      {
         if (Y < y2-W || Y > y1+W) continue;
      }

      if (x1 == x2 || y1 == y2)
         return (TRUE);
      else
      {
         m = (y1 - y2) / (x1 - x2);
         b = y1 - m * x1;
         m1 = (x2 - x1) / (y1 - y2);
         b1 = Y - m1 * X;
         x_int = (b1 - b) / (m - m1);
         y_int = m * x_int + b;
         if (sqrt ((double)(x_int-X)*(x_int-X)+(y_int-Y)*(y_int-Y)) <= (double)W)
            return (TRUE);
      }
   }
   return (FALSE);
}

int FindGoodBox (X, Y, BoxObj)
   int			X, Y;
   struct ObjRec	* BoxObj;
   /* X and Y are absolute coordinates */
{
   struct BBRec	bbox;
   int		w;

   if (BoxObj->detail.b->fill != 0) return (TRUE);

#ifndef UC
   w = (widthOfLine[BoxObj->detail.b->width]+3) << zoomScale;
#else /* UC */
   w = RealSize((widthOfLine[BoxObj->detail.b->width]+3) ,  zoomScale);
#endif /* UC */
   bbox.ltx = BoxObj->obbox.ltx + w;
   bbox.lty = BoxObj->obbox.lty + w;
   bbox.rbx = BoxObj->obbox.rbx - w;
   bbox.rby = BoxObj->obbox.rby - w;
   return (!PointInBBox (X, Y, bbox));
}

int FindGoodRCBox (X, Y, RCBoxObj)
   int			X, Y;
   struct ObjRec	* RCBoxObj;
   /* X and Y are absolute coordinates */
{
   register struct BBRec	* obbox;
   register struct BBRec	bbox;
   int				w, r;

   if (RCBoxObj->detail.b->fill != 0) return (TRUE);

   r = RCBoxObj->detail.rcb->radius;
#ifndef UC
   w = (widthOfLine[RCBoxObj->detail.rcb->width]+3) << zoomScale;
#else /* UC */
   w = RealSize((widthOfLine[RCBoxObj->detail.rcb->width]+3) ,  zoomScale);
#endif /* UC */
   obbox = &(RCBoxObj->obbox);

   bbox.ltx = obbox->ltx + w; bbox.lty = obbox->lty + r;
   bbox.rbx = obbox->rbx - w; bbox.rby = obbox->rby - r;

   if (PointInBBox (X, Y, bbox))
      return (FALSE);
   else
   {
      bbox.ltx = obbox->ltx + r; bbox.lty = obbox->lty + w;
      bbox.rbx = obbox->rbx - r; bbox.rby = obbox->rby - w;
      return (!PointInBBox (X, Y, bbox));
   }
}

int FindGoodOval (X, Y, OvalObj)
   int			X, Y;
   struct ObjRec	* OvalObj;
   /* X and Y are absolute coordinates */
{
   struct OvalRec	* oval_ptr = OvalObj->detail.o;
   int			w, ltx, lty, rbx, rby, delta;
   double		cx, cy, rx, ry, tmp_x, tmp_y;
   double		x1 = 0.0, x2 = 0.0, y1 = 0.0, y2 = 0.0;
   int			fill = oval_ptr->fill;

   ltx = OvalObj->obbox.ltx; lty = OvalObj->obbox.lty;
   rbx = OvalObj->obbox.rbx; rby = OvalObj->obbox.rby;

   cx = (ltx + rbx) / 2; cy = (lty + rby) / 2;
   rx = (rbx - ltx) / 2; ry = (rby - lty) / 2;

   if (rx >= ry)
   {  /* flat oval */
      tmp_y = sqrt (fabs ((double)(ry*ry*(1-(X-cx)*(X-cx)/rx/rx))));
      y1 = cy - tmp_y;
      y2 = cy + tmp_y;
      delta = (int)(3.0*(ry/rx));
   }
   else
   {  /* tall oval */
      tmp_x = sqrt (fabs ((double)(rx*rx*(1-(Y-cy)*(Y-cy)/ry/ry))));
      x1 = cx - tmp_x;
      x2 = cx + tmp_x;
      delta = (int)(3.0*(rx/ry));
   }

   if (fill != NONEPAT)
   {
      if (rx >= ry)
      {  /* flat oval */
         if (Y >= y1 && y2 >= Y) return (TRUE);
      }
      else
      {  /* tall oval */
         if (X >= x1 && x2 >= X) return (TRUE);
      }
   }
#ifndef UC
   w = (widthOfLine[oval_ptr->width]+3+delta) << zoomScale;
#else /* UC */
   w = RealSize((widthOfLine[oval_ptr->width]+3+delta), zoomScale);
#endif /* UC */
   if (rx >= ry)
      return ((fabs ((double)(Y-y1)) <= w) || (fabs ((double)(Y-y2)) <= w));
   else
      return ((fabs ((double)(X-x1)) <= w) || (fabs ((double)(X-x2)) <= w));
}

int FindGoodPoly (X, Y, PolyObj)
   int			X, Y;
   struct ObjRec	* PolyObj;
   /* X and Y are absolute coordinates */
{
   int			w, n;
   XPoint		* v;
   struct PolyRec	* poly_ptr;

   poly_ptr = PolyObj->detail.p;
#ifndef UC
   w = (arrowHeadH[poly_ptr->width]+3) << zoomScale;
#else /* UC */
   w = RealSize((arrowHeadH[poly_ptr->width]+3) ,  zoomScale);
#endif /* UC */
   if (poly_ptr->curved == LT_SPLINE)
      return (PointInSplinePoly (X, Y, poly_ptr, w));
   else
   {
      if (poly_ptr->fill != NONEPAT)
      {
         n = poly_ptr->n;
         v = poly_ptr->vlist;
         v[n].x = v[0].x; v[n].y = v[0].y;
         if (PointInPolygon (X, Y, n+1, v))
            return (TRUE);
      }
      return (PointInPoly (X, Y, poly_ptr->n, poly_ptr->vlist, w));
   }
}

int FindGoodPolygon (X, Y, PolygonObj)
   int			X, Y;
   struct ObjRec	* PolygonObj;
   /* X and Y are absolute coordinates */
{
   int			w;
   struct PolygonRec	* polygon_ptr;

   polygon_ptr = PolygonObj->detail.g;
#ifndef UC
   w = (widthOfLine[polygon_ptr->width]+3) << zoomScale;
#else /* UC */
   w = RealSize((widthOfLine[polygon_ptr->width]+3) ,  zoomScale);
#endif /* UC */
   if (polygon_ptr->curved == LT_SPLINE)
      return (PointInSplinePolygon (X, Y, polygon_ptr, w));
   else
   {
      if (polygon_ptr->fill != NONEPAT)
         if (PointInPolygon (X, Y, polygon_ptr->n, polygon_ptr->vlist))
            return (TRUE);
      return (PointInPoly (X, Y, polygon_ptr->n, polygon_ptr->vlist, w));
   }
}

static
int PointInFlatPie (X,Y,dir,angle,ov_int_y1,ov_int_y2,rad_int_y1,rad_int_y2)
   int		X, Y, dir, angle;
   double	ov_int_y1, ov_int_y2, rad_int_y1, rad_int_y2;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y2 && ov_int_y2>=Y) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=ov_int_y1 && rad_int_y1>=Y) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=ov_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y1 && ov_int_y2>=Y) return (TRUE);
               }
               break;
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=ov_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_y1 < rad_int_y2)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y1 < rad_int_y1)
               {
                  if (Y>=ov_int_y1 && rad_int_y1>=Y) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y2 && rad_int_y1>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y2 && ov_int_y2>=Y) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_y2 > rad_int_y2)
               {
                  if (Y>=rad_int_y1 && rad_int_y2>=Y) return (TRUE);
               }
               else if (ov_int_y2 > rad_int_y1)
               {
                  if (Y>=rad_int_y1 && ov_int_y2>=Y) return (TRUE);
               }
               break;
         }
         break;
   }
   return (FALSE);
}

static
int PointInTallPie (X,Y,dir,angle,ov_int_x1,ov_int_x2,rad_int_x1,rad_int_x2)
   int		X, Y, dir, angle;
   double	ov_int_x1, ov_int_x2, rad_int_x1, rad_int_x2;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x1 && ov_int_x2>=X) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x2 && ov_int_x2>=X) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=rad_int_x1 && ov_int_x1>=X) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=ov_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               break;
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
               if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x1 && ov_int_x2>=X) return (TRUE);
               }
               break;
            case 90:
               if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=rad_int_x1 && rad_int_x2>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=rad_int_x2 && ov_int_x1>=X) return (TRUE);
               }
               break;
            case 180:
               if (ov_int_x1 < rad_int_x2)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x1 < rad_int_x1)
               {
                  if (X>=ov_int_x1 && rad_int_x1>=X) return (TRUE);
               }
               break;
            case 270:
               if (ov_int_x2 > rad_int_x1)
               {
                  if (X>=rad_int_x2 && rad_int_x1>=X) return (TRUE);
               }
               else if (ov_int_x2 > rad_int_x2)
               {
                  if (X>=rad_int_x2 && ov_int_x2>=X) return (TRUE);
               }
               break;
         }
         break;
   }
   return (FALSE);
}

static
int XInPieRange (X, dir, angle, cx, rx)
   int		X, dir, angle;
   double	cx, rx;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
            case 90: return (X>=cx && cx+rx>=X);

            case 180:
            case 270: return (X>=cx-rx && cx>=X);
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
            case 270: return (X>=cx && cx+rx>=X);

            case 90:
            case 180: return (X>=cx-rx && cx>=X);
         }
         break;
   }
}

static
int YInPieRange (Y, dir, angle, cy, ry)
   int		Y, dir, angle;
   double	cy, ry;
{
   switch (dir)
   {
      case ARC_CCW:
         switch ((angle+360)%360)
         {
            case 0:
            case 270: return (Y>=cy && cy+ry>=Y);

            case 90:
            case 180: return (Y>=cy-ry && cy>=Y);
         }
         break;
      case ARC_CW:
         switch ((angle+360)%360)
         {
            case 0:
            case 90: return (Y>=cy-ry && cy>=Y);

            case 180:
            case 270: return (Y>=cy && cy+ry>=Y);
         }
         break;
   }
}

int FindGoodArc (X, Y, ArcObj)
   int			X, Y;
   struct ObjRec	* ArcObj;
   /* X and Y are absolute coordinates */
{
   struct ArcRec	* arc_ptr = ArcObj->detail.a;
   int			w, ltx, lty, rbx, rby, delta;
   double		cx, cy, rx, ry, tmp_x, tmp_y, x = 0.0, y = 0.0;
   double		ov_int_x1 = 0.0, ov_int_x2 = 0.0;
   double		ov_int_y1 = 0.0, ov_int_y2 = 0.0;
   double		rad_int_x1 = 0.0, rad_int_x2 = 0.0;
   double		rad_int_y1 = 0.0, rad_int_y2 = 0.0;
   int			fill = arc_ptr->fill;
   int			arc_x1, arc_y1, arc_x2, arc_y2, theta1, theta2;
   int			pass_theta1, just_pass_theta1, angle, dir;

   ltx = arc_ptr->ltx; lty = arc_ptr->lty;
   rbx = ltx+arc_ptr->w; rby = lty+arc_ptr->h;
   rx = arc_ptr->w/2; ry = arc_ptr->h/2;

   cx = arc_ptr->xc; cy = arc_ptr->yc;
   arc_x1 = arc_ptr->x1; arc_y1 = arc_ptr->y1;
   dir = arc_ptr->dir;

   theta1 = (int)(arc_ptr->angle1)/64;
   theta2 = theta1 + (int)(arc_ptr->angle2)/64;
   ArcRealX2Y2 (arc_ptr, &arc_x2, &arc_y2);

   if (theta2 < -180) theta2 += 360;
   if (theta2 > 180) theta2 -= 360;

   if (theta1 < 0) theta1 += 360;
   if (theta2 <= 0) theta2 += 360;

   if (rx >= ry)
   {  /* flat oval */
      tmp_y = sqrt (fabs ((double)(ry*ry*(1-(X-cx)*(X-cx)/rx/rx))));
      ov_int_y1 = cy - tmp_y;
      ov_int_y2 = cy + tmp_y;
      delta = (int)(3.0*(ry/rx));
   }
   else
   {  /* tall oval */
      tmp_x = sqrt (fabs ((double)(rx*rx*(1-(Y-cy)*(Y-cy)/ry/ry))));
      ov_int_x1 = cx - tmp_x;
      ov_int_x2 = cx + tmp_x;
      delta = (int)(3.0*(rx/ry));
   }

#ifndef UC
   w = (widthOfLine[arc_ptr->width]+3+delta) << zoomScale;
#else /* UC */
   w = RealSize((widthOfLine[arc_ptr->width]+3+delta) ,  zoomScale);
#endif /* UC */

   if (rx >= ry)
   {  /* flat oval */
#ifndef UC
      rad_int_y1 = cy + (X-cx)*(arc_y1-cy)/(arc_x1-cx);
      rad_int_y2 = cy + (X-cx)*(arc_y2-cy)/(arc_x2-cx);
#else /* UC */
      if (arc_x1 != cx && arc_x2 != cx)
      {
	  rad_int_y1 = cy + (X-cx)*(arc_y1-cy)/(arc_x1-cx);
	  rad_int_y2 = cy + (X-cx)*(arc_y2-cy)/(arc_x2-cx);
      }	  
#endif /* UC */
   }
   else
   {  /* tall oval */
#ifndef UC
      rad_int_x1 = cx + (Y-cy)*(arc_x1-cx)/(arc_y1-cy);
      rad_int_x2 = cx + (Y-cy)*(arc_x2-cx)/(arc_y2-cy);
#else /* UC */
      if (arc_y1 != cy && arc_y2 != cy)
      {
	  rad_int_x1 = cx + (Y-cy)*(arc_x1-cx)/(arc_y1-cy);
	  rad_int_x2 = cx + (Y-cy)*(arc_x2-cx)/(arc_y2-cy);
      }
#endif /* UC */
   }
   if (dir == ARC_CCW)
   {
      angle = 0;
      pass_theta1 = FALSE;
      just_pass_theta1 = FALSE;
      while (angle < theta2 || !pass_theta1)
      {
         if (angle >= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            just_pass_theta1 = TRUE;
            if (theta2 > theta1 && angle >= theta2)
            {  /* theta1 and theta2 are in the same quadrant */
               if (fill != NONEPAT)
               {
                  if (rx >= ry)
                  {
                     if (PointInFlatPie (X, Y, dir, angle, ov_int_y1,
                           ov_int_y2, rad_int_y1, rad_int_y2));
                        return (TRUE);
                  }
                  else
                  {
                     if (PointInTallPie (X, Y, dir, angle, ov_int_x1,
                           ov_int_x2, rad_int_x1, rad_int_x2));
                        return (TRUE);
                  }
               }
               if (rx >= ry)
               {
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        return (X>=arc_x1 && arc_x2>=X &&
                              (fabs ((double)(Y-ov_int_y2)) <= w));
                     case 90:
                     case 180:
                        return (X>=arc_x2 && arc_x1>=X &&
                              (fabs ((double)(Y-ov_int_y1)) <= w));
                  }
               }
               else
               {
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        return (Y>=arc_y2 && arc_y1>=Y &&
                              (fabs ((double)(X-ov_int_x2)) <= w));
                     case 180:
                     case 270:
                        return (Y>=arc_y1 && arc_y2>=Y &&
                              (fabs ((double)(X-ov_int_x1)) <= w));
                  }
               }
            }
            if (theta2 < theta1) angle -= 360;
         }
         if (just_pass_theta1)
         {
            just_pass_theta1 = FALSE;
            if (rx >= ry)
            {
               switch ((angle+360)%360)
               {
                  case 0: y = cy; break;
                  case 90: y = cy-ry; break;
                  case 180: y = cy; break;
                  case 270: y = cy+ry; break;
               }
               if (fill != NONEPAT)
               {
                  if (XInPieRange (X, dir, angle, cx, rx) &&
                        PointInFlatPie (X, Y, dir, angle, ov_int_y1,
                        ov_int_y2, rad_int_y1, y))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (X>=arc_x1 && cx+rx>=X &&
                           (fabs ((double)(Y-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (X>=cx && arc_x1>=X &&
                           (fabs ((double)(Y-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (X>=cx-rx && arc_x1>=X &&
                           (fabs ((double)(Y-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (X>=arc_x1 && cx>=X &&
                           (fabs ((double)(Y-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
               }
            }
            else
            {
               switch ((angle+360)%360)
               {
                  case 0: x = cx+rx; break;
                  case 90: x = cx; break;
                  case 180: x = cx-rx; break;
                  case 270: x = cx; break;
               }
               if (fill != NONEPAT)
               {
                  if (YInPieRange (Y, dir, angle, cy, ry) &&
                        PointInTallPie (X, Y, dir, angle, ov_int_x1,
                        ov_int_x2, rad_int_x1, x))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (Y>=cy && arc_y1>=Y &&
                           (fabs ((double)(X-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (Y>=cy-ry && arc_y1>=Y &&
                           (fabs ((double)(X-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (Y>=arc_y1 && cy>=Y &&
                           (fabs ((double)(X-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (Y>=arc_y1 && cy+ry>=Y &&
                           (fabs ((double)(X-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
               }
            }
         }
         else if (pass_theta1)
         {  /* see if point is in the quadrant */
            if (rx >= ry)
            {
               if (XInPieRange (X, dir, angle, cx, rx))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 90:
                        case 180:
                           if (Y>=ov_int_y1 && cy>=Y) return (TRUE);
                           break;

                        case 0:
                        case 270:
                           if (Y>=cy && ov_int_y2>=Y) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        if (fabs ((double)(Y-ov_int_y2)) <= w) return (TRUE);
                        break;
                     case 90:
                     case 180:
                        if (fabs ((double)(Y-ov_int_y1)) <= w) return (TRUE);
                        break;
                  }
               }
            }
            else
            {
               if (YInPieRange (Y, dir, angle, cy, ry))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 0:
                        case 90:
                           if (X>=cx && ov_int_x2>=X) return (TRUE);
                           break;

                        case 180:
                        case 270:
                           if (X>=ov_int_x1 && cx>=X) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        if (fabs ((double)(X-ov_int_x2)) <= w) return (TRUE);
                        break;
                     case 180:
                     case 270:
                        if (fabs ((double)(X-ov_int_x1)) <= w) return (TRUE);
                        break;
                  }
               }
            }
         }
         angle = (angle == 360) ? 0 : (angle+90);
      }
      if (rx >= ry)
      {
         switch ((angle+360)%360)
         {
            case 0: y = cy+ry; break;
            case 180: y = cy-ry; break;

            case 90:
            case 270: y = cy; break;
         }
         if (fill != NONEPAT)
         {
            if (XInPieRange (X, dir, angle, cx, rx) &&
                  PointInFlatPie (X, Y, dir, angle, ov_int_y1, ov_int_y2,
                  y, rad_int_y2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (X>=cx && arc_x2>=X &&
                     (fabs ((double)(Y-ov_int_y2)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (X>=arc_x2 && cx+rx>=X &&
                     (fabs ((double)(Y-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (X>=arc_x2 && cx>=X &&
                     (fabs ((double)(Y-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (X>=cx-rx && arc_x2>=X &&
                     (fabs ((double)(Y-ov_int_y2)) <= w))
                  return (TRUE);
               break;
         }
      }
      else
      {
         switch ((angle+360)%360)
         {
            case 0:
            case 180: x = cx; break;

            case 90: x = cx+rx; break;
            case 270: x = cx-rx; break;
         }
         if (fill != NONEPAT)
         {
            if (YInPieRange (Y, dir, angle, cy, ry) &&
                  PointInTallPie (X, Y, dir, angle, ov_int_x1, ov_int_x2,
                  x, rad_int_x2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (Y>=arc_y2 && cy+ry>=Y &&
                     (fabs ((double)(X-ov_int_x2)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (Y>=arc_y2 && cy>=Y &&
                     (fabs ((double)(X-ov_int_x2)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (Y>=cy-ry && arc_y2>=Y &&
                     (fabs ((double)(X-ov_int_x1)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (Y>=cy && arc_y2>=Y &&
                     (fabs ((double)(X-ov_int_x1)) <= w))
                  return (TRUE);
               break;
         }
      }
   }
   else
   {
      angle = 360;
      pass_theta1 = FALSE;
      just_pass_theta1 = FALSE;
      while (angle > theta2 || !pass_theta1)
      {
         if (angle <= theta1 && !pass_theta1)
         {
            pass_theta1 = TRUE;
            just_pass_theta1 = TRUE;
            if (theta2 < theta1 && angle <= theta2)
            {  /* theta1 and theta2 are in the same quadrant */
               if (fill != NONEPAT)
               {
                  if (rx >= ry)
                  {
                     if (PointInFlatPie (X, Y, dir, angle, ov_int_y1,
                           ov_int_y2, rad_int_y1, rad_int_y2));
                        return (TRUE);
                  }
                  else
                  {
                     if (PointInTallPie (X, Y, dir, angle, ov_int_x1,
                           ov_int_x2, rad_int_x1, rad_int_x2));
                        return (TRUE);
                  }
               }
               if (rx >= ry)
               {
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        return (X>=arc_x1 && arc_x2>=X &&
                              (fabs ((double)(Y-ov_int_y1)) <= w));
                     case 180:
                     case 270:
                        return (X>=arc_x2 && arc_x1>=X &&
                              (fabs ((double)(Y-ov_int_y2)) <= w));
                  }
               }
               else
               {
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        return (Y>=arc_y1 && arc_y2>=Y &&
                              (fabs ((double)(X-ov_int_x2)) <= w));
                     case 90:
                     case 180:
                        return (Y>=arc_y2 && arc_y1>=Y &&
                              (fabs ((double)(X-ov_int_x1)) <= w));
                  }
               }
            }
            if (theta2 > theta1) angle += 360;
         }
         if (just_pass_theta1)
         {
            just_pass_theta1 = FALSE;
            if (rx >= ry)
            {
               switch ((angle+360)%360)
               {
                  case 0: y = cy; break;
                  case 90: y = cy-ry; break;
                  case 180: y = cy; break;
                  case 270: y = cy+ry; break;
               }
               if (fill != NONEPAT)
               {
                  if (XInPieRange (X, dir, angle, cx, rx) &&
                        PointInFlatPie (X, Y, dir, angle, ov_int_y1,
                        ov_int_y2, rad_int_y1, y))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (X>=arc_x1 && cx+rx>=X &&
                           (fabs ((double)(Y-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (X>=arc_x1 && cx>=X &&
                           (fabs ((double)(Y-ov_int_y1)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (X>=cx-rx && arc_x1>=X &&
                           (fabs ((double)(Y-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (X>=cx && arc_x1>=X &&
                           (fabs ((double)(Y-ov_int_y2)) <= w))
                        return (TRUE);
                     break;
               }
            }
            else
            {
               switch ((angle+360)%360)
               {
                  case 0: x = cx+rx; break;
                  case 90: x = cx; break;
                  case 180: x = cx-rx; break;
                  case 270: x = cx; break;
               }
               if (fill != NONEPAT)
               {
                  if (YInPieRange (Y, dir, angle, cy, ry) &&
                        PointInTallPie (X, Y, dir, angle, ov_int_x1,
                        ov_int_x2, rad_int_x1, x))
                     return (TRUE);
               }
               switch ((angle+360)%360)
               {
                  case 0:
                     if (Y>=arc_y1 && cy>=Y &&
                           (fabs ((double)(X-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
                  case 90:
                     if (Y>=cy-ry && arc_y1>=Y &&
                           (fabs ((double)(X-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 180:
                     if (Y>=arc_y1 && cy+ry>=Y &&
                           (fabs ((double)(X-ov_int_x1)) <= w))
                        return (TRUE);
                     break;
                  case 270:
                     if (Y>=cy && arc_y1>=Y &&
                           (fabs ((double)(X-ov_int_x2)) <= w))
                        return (TRUE);
                     break;
               }
            }
         }
         else if (pass_theta1)
         {  /* see if point is in the quadrant */
            if (rx >= ry)
            {
               if (XInPieRange (X, dir, angle, cx, rx))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 0:
                        case 90:
                           if (Y>=ov_int_y1 && cy>=Y) return (TRUE);
                           break;
      
                        case 180:
                        case 270:
                           if (Y>=cy && ov_int_y2>=Y) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 90:
                        if (fabs ((double)(Y-ov_int_y1)) <= w) return (TRUE);
                        break;
                     case 180:
                     case 270:
                        if (fabs ((double)(Y-ov_int_y2)) <= w) return (TRUE);
                        break;
                  }
               }
            }
            else
            {
               if (YInPieRange (Y, dir, angle, cy, ry))
               {
                  if (fill != NONEPAT)
                  {
                     switch ((angle+360)%360)
                     {
                        case 0:
                        case 270:
                           if (X>=cx && ov_int_x2>=X) return (TRUE);
                           break;
                        case 90:
                        case 180: 
                           if (X>=ov_int_x1 && cx>=X) return (TRUE);
                           break;
                     }
                  }
                  switch ((angle+360)%360)
                  {
                     case 0:
                     case 270:
                        if (fabs ((double)(X-ov_int_x2)) <= w) return (TRUE);
                        break;
                     case 90:
                     case 180:
                        if (fabs ((double)(X-ov_int_x1)) <= w) return (TRUE);
                        break;
                  }
               }
            }
         }
         angle = (angle == 0) ? 360 : (angle-90);
      }
      if (rx >= ry)
      {
         switch ((angle+360)%360)
         {
            case 0: y = cy-ry; break;
            case 180: y = cy+ry; break;

            case 90:
            case 270: y = cy; break;
         }
         if (fill != NONEPAT)
         {
            if (XInPieRange (X, dir, angle, cx, rx) &&
                  PointInFlatPie (X, Y, dir, angle, ov_int_y1, ov_int_y2,
                  y, rad_int_y2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (X>=cx && arc_x2>=X &&
                     (fabs ((double)(Y-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (X>=cx-rx && arc_x2>=X &&
                     (fabs ((double)(Y-ov_int_y1)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (X>=arc_x2 && cx>=X &&
                     (fabs ((double)(Y-ov_int_y2)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (X>=arc_x2 && cx+rx>=X &&
                     (fabs ((double)(Y-ov_int_y2)) <= w))
                  return (TRUE);
               break;
         }
      }
      else
      {
         switch ((angle+360)%360)
         {
            case 0:
            case 180: x = cx; break;

            case 90: x = cx-rx; break;
            case 270: x = cx+rx; break;
         }
         if (fill != NONEPAT)
         {
            if (YInPieRange (Y, dir, angle, cy, ry) &&
                  PointInTallPie (X, Y, dir, angle, ov_int_x1, ov_int_x2,
                  x, rad_int_x2))
               return (TRUE);
         }
         switch ((angle+360)%360)
         {
            case 0:
               if (Y>=cy-ry && arc_y2>=Y &&
                     (fabs ((double)(X-ov_int_x2)) <= w))
                  return (TRUE);
               break;
            case 90:
               if (Y>=arc_y2 && cy>=Y &&
                     (fabs ((double)(X-ov_int_x1)) <= w))
                  return (TRUE);
               break;
            case 180:
               if (Y>=arc_y2 && cy+ry>=Y &&
                     (fabs ((double)(X-ov_int_x1)) <= w))
                  return (TRUE);
               break;
            case 270:
               if (Y>=cy && arc_y2>=Y &&
                     (fabs ((double)(X-ov_int_x2)) <= w))
                  return (TRUE);
               break;
         }
      }
   }
   return (FALSE);
}

int FindGoodObj (X, Y, FirstObj)
   int			X, Y;
   struct ObjRec	* FirstObj;
   /* X and Y are absolute coordinates */
{
   register struct ObjRec	* obj_ptr;

   for (obj_ptr = FirstObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      if (X >= obj_ptr->bbox.ltx-3 && X <= obj_ptr->bbox.rbx+3 &&
            Y >= obj_ptr->bbox.lty-3 && Y <= obj_ptr->bbox.rby+3)
      {
         switch (obj_ptr->type)
         {
            case OBJ_TEXT: return (TRUE);
            case OBJ_XBM: return (TRUE);
            case OBJ_BOX:
               if (FindGoodBox (X, Y, obj_ptr)) return (TRUE);
               break;
            case OBJ_RCBOX:
               if (FindGoodRCBox (X, Y, obj_ptr)) return (TRUE);
               break;
            case OBJ_OVAL:
               if (FindGoodOval (X, Y, obj_ptr)) return (TRUE);
               break;
            case OBJ_POLY:
               if (FindGoodPoly (X, Y, obj_ptr)) return (TRUE);
               break;
            case OBJ_POLYGON:
               if (FindGoodPolygon (X, Y, obj_ptr)) return (TRUE);
               break;
            case OBJ_ARC:
               if (FindGoodArc (X, Y, obj_ptr)) return (TRUE);
               break;

            case OBJ_GROUP:
            case OBJ_SYM:
            case OBJ_ICON:
               if (FindGoodObj (X, Y, obj_ptr->detail.r->first)) return (TRUE);
               break;
         }
      }
   return (FALSE);
}
