/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, William Cheng.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /tmp_mnt/n/kona/tangram/u/william/X11/TGIF2/RCS/xbitmap.c,v 2.1 91/03/06 17:00:57 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "color.e"
#include "file.e"
#include "mark.e"
#include "names.e"
#include "obj.e"
#include "pattern.e"
#include "raster.e"
#include "select.e"
#include "setup.e"

#define SAVE_XBM 0
#define DUMP_XBM 1

GC	xbmGC = NULL;

static Pixmap	dummyBitmap;

static char	hexValue[] = "0123456789abcdef";

static int	numColorsToDump = 0;
static int	* pixelValue = NULL;
static int	* colorIndexToDumpIndex = NULL;
static char	* colorChar = NULL;
static char	* * colorStr = NULL;

void InitXBm ()
{
   XGCValues	values;

   dummyBitmap = XCreatePixmap (mainDisplay, mainWindow, 1, 1, 1);

   values.foreground = 1;
   values.background = 0;
   values.function = GXcopy;
   values.fill_style = FillSolid;
   xbmGC = XCreateGC (mainDisplay, dummyBitmap,
         GCForeground | GCBackground | GCFunction | GCFillStyle, &values);
}

void CleanUpXBm ()
{
   register int	i;

   if (colorChar != NULL)
   {
      for (i = 0; i < numColorsToDump+2; i++)
         if (colorStr[i] != NULL)
            cfree (colorStr[i]);
      cfree (colorStr);
      cfree (colorChar);
      cfree (pixelValue);
      cfree (colorIndexToDumpIndex);
   }

   if (xbmGC != NULL) XFreeGC (mainDisplay, xbmGC);
   XFreePixmap (mainDisplay, dummyBitmap);
}

void FlipXBmHorizontal (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		i, left, right;
   int			w, h, left_pixel, right_pixel;
   Pixmap		bitmap;
   XImage		* image;

   bitmap = ObjPtr->detail.xbm->bitmap;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;
   image = XGetImage (mainDisplay, bitmap, 0, 0, w, h, 1, ZPixmap);

   for (i = 0; i < h; i++)
      for (left=0, right=w-1; left < right; left++, right--)
      {
         left_pixel = XGetPixel (image, left, i);
         right_pixel = XGetPixel (image, right, i);
         if (left_pixel != right_pixel)
         {
            XPutPixel (image, right, i, left_pixel);
            XPutPixel (image, left, i, right_pixel);
         }
      }
   XPutImage (mainDisplay, bitmap, xbmGC, image, 0, 0, 0, 0, w, h);
   XDestroyImage (image);
}

void FlipXBmVertical (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		j, top, bottom;
   int			w, h, top_pixel, bottom_pixel;
   Pixmap		bitmap;
   XImage		* image;

   bitmap = ObjPtr->detail.xbm->bitmap;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;
   image = XGetImage (mainDisplay, bitmap, 0, 0, w, h, 1, ZPixmap);

   for (j = 0; j < w; j++)
      for (top=0, bottom=h-1; top < bottom; top++, bottom--)
      {
         top_pixel = XGetPixel (image, j, top);
         bottom_pixel = XGetPixel (image, j, bottom);
         if (top_pixel != bottom_pixel)
         {
            XPutPixel (image, j, bottom, top_pixel);
            XPutPixel (image, j, top, bottom_pixel);
         }
      }
   XPutImage (mainDisplay, bitmap, xbmGC, image, 0, 0, 0, 0, w, h);
   XDestroyImage (image);
}

void RotateXBmClockWise (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		j, i;
   int			w, h, num_bytes;
   char			* xbm_data;
   Pixmap		src_bitmap, dest_bitmap;
   XImage		* src_image, * dest_image;

   src_bitmap = ObjPtr->detail.xbm->bitmap;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;
   src_image = XGetImage (mainDisplay, src_bitmap, 0, 0, w, h, 1, ZPixmap);

   num_bytes = ((h % 8) == 0) ? (int)(h>>3) : (int)(h>>3)+1;

   xbm_data = (char *) calloc (w*num_bytes, sizeof(char));
   dest_bitmap = XCreateBitmapFromData (mainDisplay, dummyBitmap, xbm_data,
         h, w);
   dest_image = XGetImage (mainDisplay, dest_bitmap, 0, 0, h, w, 1, ZPixmap);
   cfree (xbm_data);

   for (i = 0; i < h; i++)
      for (j = 0; j < w; j++)
         if (XGetPixel (src_image, j, i) == 1)
            XPutPixel (dest_image, h-i-1, j, 1);

   XPutImage (mainDisplay, dest_bitmap, xbmGC, dest_image, 0, 0, 0, 0, h, w);
   ObjPtr->detail.xbm->bitmap = dest_bitmap;

   XDestroyImage (src_image);
   XDestroyImage (dest_image);
   XFreePixmap (mainDisplay, src_bitmap);
}

void RotateXBmCounter (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		j, i;
   int			w, h, num_bytes;
   char			* xbm_data;
   Pixmap		src_bitmap, dest_bitmap;
   XImage		* src_image, * dest_image;

   src_bitmap = ObjPtr->detail.xbm->bitmap;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;
   src_image = XGetImage (mainDisplay, src_bitmap, 0, 0, w, h, 1, ZPixmap);

   num_bytes = ((h % 8) == 0) ? (int)(h>>3) : (int)(h>>3)+1;

   xbm_data = (char *) calloc (w*num_bytes, sizeof(char));
   dest_bitmap = XCreateBitmapFromData (mainDisplay, dummyBitmap, xbm_data,
         h, w);
   dest_image = XGetImage (mainDisplay, dest_bitmap, 0, 0, h, w, 1, ZPixmap);
   cfree (xbm_data);

   for (i = 0; i < h; i++)
      for (j = 0; j < w; j++)
         if (XGetPixel (src_image, j, i) == 1)
            XPutPixel (dest_image, i, w-j-1, 1);

   XPutImage (mainDisplay, dest_bitmap, xbmGC, dest_image, 0, 0, 0, 0, h, w);
   ObjPtr->detail.xbm->bitmap = dest_bitmap;

   XDestroyImage (src_image);
   XDestroyImage (dest_image);
   XFreePixmap (mainDisplay, src_bitmap);
}

static
void InvertXBmObject (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		i, j;
   int			w, h, pixel;
   Pixmap		bitmap;
   XImage		* image;

   bitmap = ObjPtr->detail.xbm->bitmap;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;
   image = XGetImage (mainDisplay, bitmap, 0, 0, w, h, 1, ZPixmap);

   for (i = 0; i < h; i++)
      for (j = 0; j < w; j++)
      {
         pixel = XGetPixel (image, j, i);
         XPutPixel (image, j, i, ((pixel==1) ? 0 : 1));
      }
   XPutImage (mainDisplay, bitmap, xbmGC, image, 0, 0, 0, 0, w, h);
   XDestroyImage (image);
}

static
int InvertObjXBitmap (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;
   int				changed = FALSE;

   for (obj_ptr = ObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      switch (obj_ptr->type)
      {
         case OBJ_XBM: InvertXBmObject (obj_ptr); changed = TRUE; break;

         case OBJ_SYM:
         case OBJ_GROUP:
            if (InvertObjXBitmap (obj_ptr->detail.r->last)) changed = TRUE;
            break;
      }
   return (changed);
}

void InvertXBitmaps ()
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE;

   if (topSel == NULL) return;

   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
      switch (sel_ptr->obj->type)
      {
         case OBJ_XBM: InvertXBmObject (sel_ptr->obj); changed = TRUE; break;

         case OBJ_SYM:
         case OBJ_GROUP:
            if (InvertObjXBitmap (sel_ptr->obj->detail.r->last)) changed = TRUE;
            break;
      }

   if (changed)
   {
      SetFileModified (TRUE);
      HighLightReverse ();
#ifndef UC
      RedrawAnArea (botObj, selLtX-(1<<zoomScale), selLtY-(1<<zoomScale),
            selRbX+(1<<zoomScale), selRbY+(1<<zoomScale));
#else /* UC */
      RedrawAnArea (botObj, selLtX-(RealSize(1, zoomScale)), selLtY-(RealSize(1, zoomScale)),
            selRbX+(RealSize(1, zoomScale)), selRbY+(RealSize(1, zoomScale)));
#endif /* UC */
      HighLightForward ();
   }
}

static
void DumpXBmData (FP, bitmap, W, H, Mode)
   FILE		* FP;
   Pixmap	bitmap;
   int		W, H, Mode;
{
   register int	nibble_count, bit_count, data, i, j;
   int		num_nibbles = ((W % 4) == 0) ? (int)(W>>2) : (int)(W>>2)+1;
   XImage	* image;

   image = XGetImage (mainDisplay, bitmap, 0, 0, W, H, 1, ZPixmap);

   nibble_count = 0;

   for (i = 0; i < H; i++)
   {
      bit_count = 0;
      data = 0;

      for (j = 0; j < W; j++)
      {
         data = (XGetPixel (image, j, i) == 1) ? (data<<1) | 1 : (data<<1);

         if (++bit_count == 4)
         {
            if (nibble_count++ == 64)
            {
               nibble_count = 1;
               fprintf (FP, "\n     ");
            }
            fprintf (FP, "%c", hexValue[data]);
            bit_count = 0;
            data = 0;
         }
      }
      if ((W % 4) != 0)
      {
         data <<= (4 - (W % 4));
         if (nibble_count++ == 64)
         {
            nibble_count = 1;
            fprintf (FP, "\n     ");
         }
         fprintf (FP, "%c", hexValue[data]);
      }
      if (Mode == DUMP_XBM && (num_nibbles & 0x1) == 1)
      {
         if (nibble_count++ == 64)
         {
            nibble_count = 1;
            fprintf (FP, "\n     ");
         }
         fprintf (FP, "0");
      }
   }
   XDestroyImage (image);
}

static
void BuildXPmColors ()
{
   register int			i, color_index;
   register struct ObjRec	* obj_ptr;

   if (colorChar == NULL)
   {
      pixelValue = (int *) calloc (maxColors+2, sizeof(int));
      colorIndexToDumpIndex = (int *) calloc (maxColors, sizeof(int));
      colorChar = (char *) calloc (maxColors+2, sizeof(char));
      colorStr = (char * *) calloc (maxColors+2, sizeof(char *));

      for (i = 0; i < maxColors+2; i++) colorStr[i] = NULL;
   }

   for (i = 0; i < maxColors; i++) colorIndexToDumpIndex[i] = INVALID;

   pixelValue[0] = myBgPixel;
   colorChar[0] = '`';
   colorStr[0] = (char *) calloc (strlen(myBgColorStr), sizeof(char));
   strcpy (colorStr[0], myBgColorStr);
   numColorsToDump = 1;

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      color_index = obj_ptr->color;
      if (colorIndexToDumpIndex[color_index] == INVALID)
      {
         if (colorStr[numColorsToDump] != NULL)
            cfree (colorStr[numColorsToDump]);

         pixelValue[numColorsToDump] = colorPixels[color_index];
         colorIndexToDumpIndex[color_index] = numColorsToDump;
         colorChar[numColorsToDump] = (char)(((int)('a'))+numColorsToDump-1);
         colorStr[numColorsToDump] = (char *) calloc
               (strlen(colorMenuItems[color_index]), sizeof(char));
         strcpy (colorStr[numColorsToDump], colorMenuItems[color_index]);
         numColorsToDump++;
      }
   }
}

static
void DumpXPmColors (FP)
   FILE	* FP;
{
   register int	i;

   for (i = 0; i < numColorsToDump-1; i++)
      fprintf (FP, "   \"%c\", \"%s\",\n", colorChar[i], colorStr[i]);
   fprintf (FP, "   \"%c\", \"%s\"\n};\n", colorChar[i], colorStr[i]);
}

void DumpXBitmapFile ()
{
   register int	j, bit_count, index, data;
   int		i, len, ltx, lty, w, h, byte_count;
   char		xbm_file_name[MAXPATHLENGTH], msg[MAXSTRING];
   char		name[MAXPATHLENGTH];
   FILE		* fp;
   Pixmap	pixmap;
   XImage	* image;

   strcpy (name, curFileName);
   len = strlen (name);
   for (j = len-1; name[j] != '.'; j--) ;
   name[j] = '\0';

   sprintf (xbm_file_name, "%s/%s", curDir, curFileName);
   len = strlen (xbm_file_name);
   for (j = len-1; xbm_file_name[j] != '.'; j--) ;
   if (colorDump)
      sprintf (&xbm_file_name[j], ".%s", XPM_FILE_EXT);
   else
      sprintf (&xbm_file_name[j], ".%s", XBM_FILE_EXT);

   if ((fp = fopen (xbm_file_name, "w")) == NULL)
   {
      sprintf (msg, "Can not open '%s', print aborted.", xbm_file_name);
      Msg (msg);
      return;
   }
   sprintf (msg, "Printing into '%s'.", xbm_file_name);
   Msg (msg);

   if ((pixmap = DrawAllOnPixmap (&ltx, &lty, &w, &h)) == None) return;

   if (colorDump)
   {
      BuildXPmColors ();
      fprintf (fp, "#define %s_format 1\n", name);
      fprintf (fp, "#define %s_width %1d\n", name, w);
      fprintf (fp, "#define %s_height %1d\n", name, h);
      fprintf (fp, "#define %s_ncolors %1d\n", name, numColorsToDump);
      fprintf (fp, "#define %s_chars_per_pixel 1\n", name);
      fprintf (fp, "static char *%s_colors[] = {\n", name);
      DumpXPmColors (fp);
      fprintf (fp, "static char *%s_pixels[] = {\n", name);
   }
   else
   {
      fprintf (fp, "#define %s_width %1d\n", name, w);
      fprintf (fp, "#define %s_height %1d\n", name, h);
      fprintf (fp, "#define %s_x_hot 0\n", name);
      fprintf (fp, "#define %s_y_hot 0\n", name);
      fprintf (fp, "static char %s_bits[] = {\n   ", name);
   }
   image = XGetImage (mainDisplay, pixmap, 0, 0, w, h, AllPlanes, ZPixmap);

   if (colorDump)
   {
      for (i = 0; i < h; i++)
      {
         fprintf (fp, "\"");
         for (j = 0; j < w; j++)
         {
            data = XGetPixel (image,j,i);
            if (data == myBgPixel)
               fprintf (fp, "`");
            else
            {
               for (index = 0; index < numColorsToDump; index++)
                  if (pixelValue[index] == data)
                     break;
               if (index == numColorsToDump)
               {
                  sprintf (msg, "Unrecognized pixel value %1d!  %s %s.",
                        data, "Print aborted!");
                  Msg (msg);

                  fclose (fp);
                  XDestroyImage (image);
                  XFreePixmap (mainDisplay, pixmap);
                  return;
               }
               fprintf (fp, "%c", colorChar[index]);
            }
         }
         if (i == h-1)
            fprintf (fp, "\"\n};\n");
         else
            fprintf (fp, "\",\n");
      }
   }
   else
   {
      byte_count = 0;
      for (i = 0; i < h; i++)
      {
         bit_count = 0;
         data = 0;

         for (j = 0; j < w; j++)
         {
            if (XGetPixel (image,j,i) != myBgPixel) data |= (1<<bit_count);

            if (++bit_count == 8)
            {
               if (byte_count++ == 12)
               {
                  byte_count = 1;
                  fprintf (fp, "\n   ");
               }
               fprintf (fp, "0x%c", hexValue[(data>>4) & 0xf]);

               if (i == h-1 && j == w-1)
                  fprintf (fp, "%c};\n", hexValue[data & 0xf]);
               else
                  fprintf (fp, "%c, ", hexValue[data & 0xf]);

               bit_count = 0;
               data = 0;
            }
         }
         if ((w % 8) != 0)
         {
            if (byte_count++ == 12)
            {
               byte_count = 1;
               fprintf (fp, "\n   ");
            }
            fprintf (fp, "0x%c", hexValue[(data>>4) & 0xf]);
            if (i == h-1)
               fprintf (fp, "%c};\n", hexValue[data & 0xf]);
            else
               fprintf (fp, "%c, ", hexValue[data & 0xf]);
         }
      }
   }
   fclose (fp);

   XDestroyImage (image);
   XFreePixmap (mainDisplay, pixmap);
}

void DumpXBmObj (FP, ObjPtr, PRTGIF)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
   int			PRTGIF;
{
   register int	i, j;
   int		ltx, lty, rbx, rby, w, h, color_index;
   int		num_nibbles, nibble_count, fill;
   Pixmap	bitmap;
   char		* xbm_data;

   ltx = ObjPtr->obbox.ltx;
   lty = ObjPtr->obbox.lty;
   rbx = ObjPtr->obbox.rbx;
   rby = ObjPtr->obbox.rby;

   fill = ObjPtr->detail.xbm->fill;

   w = rbx - ltx;
   h = rby - lty;

   fprintf (FP, "gsave\n");

   color_index = ObjPtr->color;
   if (colorDump)
   {
      fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));
   }
   else
      fprintf (FP, "   0 setgray\n");

   switch (fill)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         fprintf (FP, "   newpath\n");
         fprintf (FP, "      %1d %1d moveto\n", ltx, lty);
         fprintf (FP, "      %1d %1d lineto\n", rbx, lty);
         fprintf (FP, "      %1d %1d lineto\n", rbx, rby);
         fprintf (FP, "      %1d %1d lineto\n", ltx, rby);
         fprintf (FP, "   closepath fill\n\n");
         break;
      case BACKPAT:
         fprintf (FP, "   newpath\n");
         fprintf (FP, "      %1d %1d moveto\n", ltx, lty);
         fprintf (FP, "      %1d %1d lineto\n", rbx, lty);
         fprintf (FP, "      %1d %1d lineto\n", rbx, rby);
         fprintf (FP, "      %1d %1d lineto\n", ltx, rby);
         fprintf (FP, "   closepath 1 setgray fill\n\n");
         if (colorDump)
            fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                  ((float)tgifColors[color_index].red/maxRGB),
                  ((float)tgifColors[color_index].green/maxRGB),
                  ((float)tgifColors[color_index].blue/maxRGB));
         else
            fprintf (FP, "   0 setgray\n");
         break;
      default:
         /* patterned */
         fprintf (FP, "   gsave\n");
         if (!colorDump)
            fprintf (FP, "   pat%1d 8 1 0 72 300 32 div div setpattern\n",fill);
         fprintf (FP, "      newpath\n");
         fprintf (FP, "         %1d %1d moveto\n", ltx, lty);
         fprintf (FP, "         %1d %1d lineto\n", rbx, lty);
         fprintf (FP, "         %1d %1d lineto\n", rbx, rby);
         fprintf (FP, "         %1d %1d lineto\n", ltx, rby);
         if (colorDump)
         {
            fprintf (FP, "      closepath eoclip\n");
            DumpPatFill (FP, fill, 8, ObjPtr->bbox, "      ");
         }
         else
            fprintf (FP, "      closepath fill\n");
         fprintf (FP, "   grestore\n\n");
         break;
   }

   fprintf (FP, "   %1d %1d translate\n", ltx, lty);
   fprintf (FP, "   %1d %1d true [1 0 0 1 0 0]\n   {<", w, h);
   if (PRTGIF)
   {
      xbm_data = ObjPtr->detail.xbm->data;

      num_nibbles = ((w % 4) == 0) ? (int)(w>>2) : (int)(w>>2)+1;
      nibble_count = 0;

      for (i = 0; i < h; i++)
      {
         for (j = 0; j < num_nibbles; j++)
         {
            if (nibble_count++ == 64)
            {
               nibble_count = 1;
               fprintf (FP, "\n     ");
            }
            fprintf (FP, "%c", xbm_data[i*num_nibbles+j]);
         }
         if ((num_nibbles & 0x1) == 1)
         {
            if (nibble_count++ == 64)
            {
               nibble_count = 1;
               fprintf (FP, "\n     ");
            }
            fprintf (FP, "0");
         }
      }
   }
   else
   {
      bitmap = ObjPtr->detail.xbm->bitmap;
      DumpXBmData (FP, bitmap, w, h, DUMP_XBM);
   }
   fprintf (FP, ">}\n   imagemask\n");
   fprintf (FP, "grestore\n\n");

}

void DrawXBmObj (win, XOff, YOff, ObjPtr)
   Window		win;
   struct ObjRec	* ObjPtr;
{
   register int		i, j;
   int			r, c, ltx, lty, rbx, rby, w, h, scr_w, scr_h;
   int			real_x_off, real_y_off, scale, num_cols, num_rows;
   int			fg_pixel, row_start, row_end, col_start, col_end;
   int			count, threshold, fill;
   struct XBmRec	* xbm_ptr = ObjPtr->detail.xbm;
   XGCValues		values;
   XImage		* image;
#ifdef UC
   int dotW, dotH;
#endif /* UC */
   fill = xbm_ptr->fill;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;

#ifndef UC
   real_x_off = (XOff >> zoomScale) << zoomScale;
   real_y_off = (YOff >> zoomScale) << zoomScale;
   ltx = (ObjPtr->obbox.ltx - real_x_off) >> zoomScale;
   lty = (ObjPtr->obbox.lty - real_y_off) >> zoomScale;
   rbx = (ObjPtr->obbox.rbx - real_x_off) >> zoomScale;
   rby = (ObjPtr->obbox.rby - real_y_off) >> zoomScale;
#else /* UC */
   real_x_off = RealSize((ScreenSize(XOff, zoomScale)), zoomScale);
   real_y_off = RealSize((ScreenSize(YOff, zoomScale)), zoomScale);
   ltx =  ScreenSize((ObjPtr->obbox.ltx - real_x_off), zoomScale);
   lty =  ScreenSize((ObjPtr->obbox.lty - real_y_off), zoomScale);
   rbx =  ScreenSize((ObjPtr->obbox.rbx - real_x_off), zoomScale);
   rby =  ScreenSize((ObjPtr->obbox.rby - real_y_off), zoomScale);
#endif /* UC */
   scr_w = rbx - ltx;
   scr_h = rby - lty;

   if (fill != 0)
   {
      values.foreground = (fill == 2) ? myBgPixel : colorPixels[ObjPtr->color];
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillRectangle (mainDisplay, win, drawGC, ltx, lty, scr_w, scr_h);
   }

   if (zoomScale == 0)
   {
      values.foreground = colorPixels[ObjPtr->color];
      values.function = GXcopy;
      values.fill_style = FillStippled;
      values.stipple = xbm_ptr->bitmap;
      values.ts_x_origin = ltx;
      values.ts_y_origin = lty;
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple |
            GCTileStipXOrigin | GCTileStipYOrigin, &values);
      XFillRectangle (mainDisplay, win, drawGC, ltx, lty, scr_w, scr_h);
      XSetTSOrigin (mainDisplay, drawGC, 0, 0);
   }
#ifdef UC
   else if (zoomScale < 0)
   {
      values.function = GXcopy;
      values.fill_style = FillSolid;
      XChangeGC (mainDisplay, drawGC, GCFunction | GCFillStyle, &values);

      fg_pixel = colorPixels[ObjPtr->color];

      dotH = dotW = 1<<(-zoomScale);
    
      image = XGetImage (mainDisplay, xbm_ptr->bitmap, 0, 0, w, h, 1, ZPixmap);

      for (i = 0 ; i < h ; i++)
	  for (j = 0 ; j < w ; j++)
	      if (XGetPixel(image, j, i) == 1)
	      {
		  XSetForeground (mainDisplay, drawGC, fg_pixel);
		  XFillRectangle (mainDisplay, win, drawGC,
				  (j<<(-zoomScale))+ltx, (i<<(-zoomScale))+lty, dotH, dotW);
	      }
/*	      else
	      {
		  XSetForeground (mainDisplay, drawGC, myBgPixel);
		  XFillRectangle (mainDisplay, win, drawGC,
				  (j<<(-zoomScale))+ltx, (i<<(-zoomScale))+lty, dotH, dotW);
	       }*/
   }
#endif /* UC */
   else
   {
      values.function = GXcopy;
      values.fill_style = FillSolid;
      XChangeGC (mainDisplay, drawGC, GCFunction | GCFillStyle, &values);

      fg_pixel = colorPixels[ObjPtr->color];

      scale = 1<<zoomScale;
      threshold = scale*scale/2;
      num_cols = ((w % scale) == 0) ? (w/scale) : (w/scale+1);
      num_rows = ((h % scale) == 0) ? (h/scale) : (h/scale+1);

      image = XGetImage (mainDisplay, xbm_ptr->bitmap, 0, 0, w, h, 1, ZPixmap);

      for (r = 0; r < num_rows-1; r++)
      {
         row_start = scale*r;
         row_end = min(scale*(r+1), w);
         for (c = 0; c < num_cols-1; c++)
         {
            col_start = scale*c;
            col_end = min(scale*(c+1), w);

            count = 0;
            for (i = row_start; count < threshold && i < row_end; i++)
               for (j = col_start; count < threshold && j < col_end; j++)
                  if (XGetPixel (image, j, i) == 1)
                     count++;
            if (count >= threshold)
            {
               XSetForeground (mainDisplay, drawGC, fg_pixel);
#ifdef sun
               XDrawPoint (mainDisplay, win, drawGC, c+ltx, r+lty);
#else /* UC */
#ifdef ultrix
               XDrawPoint (mainDisplay, win, drawGC, c+ltx, r+lty);
#else /* UC */
               XDrawLine (mainDisplay, win, drawGC, c+ltx, r+lty, c+ltx, r+lty);
#endif
#endif
            }
            else
            {
               XSetForeground (mainDisplay, drawGC, myBgPixel);
#ifdef sun
               XDrawPoint (mainDisplay, win, drawGC, c+ltx, r+lty);
#else /* UC */
#ifdef ultrix
               XDrawPoint (mainDisplay, win, drawGC, c+ltx, r+lty);
#else /* UC */
               XDrawLine (mainDisplay, win, drawGC, c+ltx, r+lty, c+ltx, r+lty);
#endif
#endif
            }
         }
      }
      XDestroyImage (image);
   }
}

void CreateXBmObj (W, H, bitmap)
   int		W, H;
   Pixmap	bitmap;
{
   struct XBmRec	* xbm_ptr;
   struct ObjRec	* obj_ptr;

   xbm_ptr = (struct XBmRec *) calloc (1, sizeof(struct XBmRec));
   xbm_ptr->bitmap = bitmap;
   xbm_ptr->data = NULL;
   xbm_ptr->fill = objFill;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));

   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = W + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = H + drawOrigY;
   obj_ptr->type = OBJ_XBM;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->detail.xbm = xbm_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   AddObj (NULL, topObj, obj_ptr);
}
 
void SaveXBmObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   int	ltx, lty, rbx, rby, w, h;

   ltx = ObjPtr->obbox.ltx; lty = ObjPtr->obbox.lty;
   rbx = ObjPtr->obbox.rbx; rby = ObjPtr->obbox.rby;
   w = rbx - ltx;
   h = rby - lty;
   fprintf (FP, "xbm('%s',", colorMenuItems[ObjPtr->color]);
   fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,\n    \"",
         ltx, lty, rbx, rby, ObjPtr->detail.xbm->fill, ObjPtr->id);
   DumpXBmData (FP, ObjPtr->detail.xbm->bitmap, w, h, SAVE_XBM);
   fprintf (FP, "\",");

   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

void ReadXBmObj (FP, Inbuf, ObjPtr, PRTGIF)
   FILE			* FP;
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
   int			PRTGIF;
{
   struct XBmRec	* xbm_ptr;
   char			color_str[20], * s, inbuf[MAXSTRING], * c_ptr;
   int			ltx, lty, rbx, rby, w, h, i, j, k, data=0, color_index;
   int			nibble_count, bit_count, num_nibbles, num_bytes, fill;
   Pixmap		bitmap;
   char			* xbm_data;
   XImage		* image;

   *ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);
   xbm_ptr = (struct XBmRec *) calloc (1, sizeof(struct XBmRec));

   if (fileVersion > 8)
   {
      sscanf (s, "%d , %d , %d , %d , %d , %d",
            &ltx, &lty, &rbx, &rby, &fill, &((*ObjPtr)->id));
      if ((*ObjPtr)->id >= objId) objId = (*ObjPtr)->id + 1;
   }

   color_index = FindColorIndex (color_str);

   (*ObjPtr)->color = color_index;
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->type = OBJ_XBM;
   (*ObjPtr)->obbox.ltx = (*ObjPtr)->bbox.ltx = (*ObjPtr)->x = ltx;
   (*ObjPtr)->obbox.lty = (*ObjPtr)->bbox.lty = (*ObjPtr)->y = lty;
   (*ObjPtr)->obbox.rbx = (*ObjPtr)->bbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = (*ObjPtr)->bbox.rby = rby;
   (*ObjPtr)->detail.xbm = xbm_ptr;

   xbm_ptr->fill = fill;

   w = rbx - ltx;
   h = rby - lty;
   num_nibbles = ((w % 4) == 0) ? (int)(w>>2) : (int)(w>>2)+1;
   num_bytes = ((w % 8) == 0) ? (int)(w>>3) : (int)(w>>3)+1;

   fgets (inbuf, MAXSTRING, FP);
   c_ptr = &inbuf[5];
   nibble_count = 0;

   if (PRTGIF)
   {
      xbm_data = (char *) calloc (h*num_nibbles, sizeof(char));
      for (i = 0; i < h; i++)
      {
         for (j = 0; j < num_nibbles; j++)
         {
            if (nibble_count++ == 64)
            {
               fgets (inbuf, MAXSTRING, FP);
               c_ptr = &inbuf[5];
               nibble_count = 1;
            }
            xbm_data[i*num_nibbles+j] = *c_ptr++;
         }
      }
      xbm_ptr->data = xbm_data;
   }
   else
   {
      xbm_data = (char *) calloc (h*num_bytes, sizeof(char));
      bitmap = XCreateBitmapFromData (mainDisplay, dummyBitmap, xbm_data, w, h);
      image = XGetImage (mainDisplay, bitmap, 0, 0, w, h, 1, ZPixmap);
      cfree (xbm_data);

      for (i = 0; i < h; i++)
      {
         bit_count = 0;
         for (j = 0; j < num_nibbles; j++)
         {
            if (nibble_count++ == 64)
            {
               fgets (inbuf, MAXSTRING, FP);
               c_ptr = &inbuf[5];
               nibble_count = 1;
            }

            if (*c_ptr >= '0' && *c_ptr <= '9')
               data = (int)(*c_ptr++) - (int)('0');
            else if (*c_ptr >= 'a' && *c_ptr <= 'f')
               data = (int)(*c_ptr++) - (int)('a') + 10;

            for (k = 0; k < 4; k++)
            {
               if (bit_count++ == w) break;

               if (data & (1<<(3-k)))
                  XPutPixel (image, j*4+k, i, 1);
            }
         }
      }
      XPutImage (mainDisplay, bitmap, xbmGC, image, 0, 0, 0, 0, w, h);
      xbm_ptr->bitmap = bitmap;

      XDestroyImage (image);
   }
}

void FreeXBmObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   if (ObjPtr->detail.xbm->bitmap != NULL)
       XFreePixmap (mainDisplay, ObjPtr->detail.xbm->bitmap);
   if (ObjPtr->detail.xbm->data != NULL) cfree (ObjPtr->detail.xbm->data);
   cfree (ObjPtr->detail.xbm);
   cfree (ObjPtr);
}
