/*
 * Author:	Yoichiro Ueno (ueno@cs.titech.ac.jp)
 *
 * Copyright (C) 1991, 1992, Yoichiro Ueno.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "choice.e"
#include "convkinput.e"
#include "setup.e"
#include "text.e"

#define CONV_OFF		0
#define WAIT_CONV_ON		1
#define CONV_ON			2
#define WAIT_CONV_OFF		3

#define KANJI0			0
#define KANJI1			1
#define KANJI2			2
#define KANJI3			3
#define KANJI4			4
#define KANJI5			5

#define LATIN			0
#define ESC			1
#define ESC_24			2
#define ESC_28			3
#define ESC_GL			4
#define ESC_GR			5
#define KANJI_GL		6
#define KANJI_GR		7

#define KINPUT2_SELECTION	"_JAPANESE_CONVERSION"
#define KINPUT_SELECTION	"JAPANESE_CONVERSION"
#define TARGET			"COMPOUND_TEXT"
#define CONV_REQUEST		"CONVERSION_REQUEST"
#define CONV_NOTIFY		"CONVERSION_NOTIFY"
#define CONV_END		"CONVERSION_END"
#define CONV_END_REQ		"CONVERSION_END_REQUEST"

static int		conv_mode = CONV_OFF;
static Atom		conv_selection = None;
static Window		conv_owner_win = None;
static Atom		conv_target = None;
static Atom		conv_property = None;
static Atom		conv_req = None;
static Atom		conv_notify = None;
static Atom		conv_end = None;
static Atom		conv_end_req = None;
static Window		conv_win;

int	imProtocol = IM_NONE;
char	kinputConvSelName[MAXSTRING] = KINPUT2_SELECTION;

void KinputBeginConversion ()
{
	XClientMessageEvent	xcme;
	Window			owner_win;
	char			msg[80];
	char *			str1;

	if(imProtocol != IM_KINPUT)
		return;

#ifdef DEBUG
	fprintf(stderr, "KinputBeginConversion() : %d\n", conv_mode);
#endif
	if(curChoice != DRAWTEXT || !textCursorShown)
		return;

	if(conv_selection == None &&
			(conv_selection = InternAtom(kinputConvSelName)) == None)
		return;

	if(conv_target == None && (conv_target = InternAtom(TARGET)) == None)
		return;

	conv_property = None;

	if(conv_req == None && (conv_req = InternAtom(CONV_REQUEST)) == None)
		return;

	if(conv_notify == None &&
			(conv_notify = InternAtom(CONV_NOTIFY)) == None)
		return;

	if(conv_end == None && (conv_end = InternAtom(CONV_END)) == None)
		return;

	if(conv_end_req == None &&
			(conv_end_req = InternAtom(CONV_END_REQ)) == None)
		return;

	if((owner_win = XGetSelectionOwner(mainDisplay, conv_selection))
			== None) {
		str1 = XGetAtomName(mainDisplay, conv_selection);
		sprintf(msg, "There is no Selection Owner of %s.\n", str1);
		Msg(msg);
		XFree(str1);
		conv_owner_win = None;
		conv_mode = CONV_OFF;
#ifdef DEBUG
		fprintf(stderr, "0x%08x\n", owner_win);
#endif
		return;
	}
#ifdef DEBUG
	fprintf(stderr, "0x%08x\n", owner_win);
#endif

	switch(conv_mode) {
	    case WAIT_CONV_ON :
		Msg("Waiting for start conversion.\n");
		return;
	    case CONV_ON :
		if (conv_owner_win == owner_win)
		    return;
		break;
	}

	xcme.type = ClientMessage;
	xcme.display = mainDisplay;
	xcme.window = owner_win;
	xcme.message_type = conv_req;
	xcme.format = 32;
	xcme.data.l[0] = conv_selection;
	xcme.data.l[1] = drawWindow;
	xcme.data.l[2] = conv_target;
	xcme.data.l[3] = conv_property;
	xcme.data.l[4] = None;
	if(XSendEvent(mainDisplay, owner_win, False, NoEventMask, (XEvent *)&xcme)
			== 0) {
		Msg("Failing in connecting owner.\n");
		return;
	}

	conv_owner_win = owner_win;
	conv_mode = WAIT_CONV_ON;
}

void KinputCheckClientMessage (xclient)
XClientMessageEvent *	xclient;
{
	if(imProtocol != IM_KINPUT)
		return;

	if(xclient->message_type == conv_notify) {
#ifdef DEBUG
		fprintf(stderr, "KinputCheckClientMessage() : conv_notify : %d\n", conv_mode);
#endif
		if( conv_mode != WAIT_CONV_ON ||
		    xclient->window != drawWindow ||
		    xclient->format != 32 ||
		    xclient->data.l[0] != conv_selection) {
			return;
		}
		if( xclient->data.l[2] == None ||
		    xclient->data.l[1] != conv_target) {
			conv_mode = CONV_OFF;
			return;
		}
		conv_mode = CONV_ON;
		conv_property = xclient->data.l[2];
		conv_win = xclient->data.l[3];
	}
	else if(xclient->message_type == conv_end) {
#ifdef DEBUG
		fprintf(stderr, "KinputCheckClientMessage() : conv_end : %d\n", conv_mode);
#endif
		if((conv_mode != WAIT_CONV_OFF && conv_mode != CONV_ON) ||
		    xclient->window != drawWindow ||
		    xclient->format != 32 ||
		    xclient->data.l[0] != conv_selection ||
		   (xclient->data.l[1] != conv_owner_win &&
		    xclient->data.l[1] != conv_win)) {
			return;
		}
		conv_mode = CONV_OFF;
	}
}

void KinputCheckConvProperty (xprop, dst)
XPropertyEvent *	xprop;
register char *		dst;
{
	Atom		act_target;
	int		act_format;
	unsigned long	nitems;
	unsigned long	remain_bytes;
	unsigned char *		data;

	if(imProtocol != IM_KINPUT)
		return;

	if( xprop->window != drawWindow ||
	    xprop->atom != conv_property ||
	    xprop->state != PropertyNewValue ||
	    conv_mode != CONV_ON) {
		return;
	}

	if(XGetWindowProperty(mainDisplay, drawWindow, conv_property,
			0, MAXSTRING/sizeof(long),
			True, conv_target, &act_target, &act_format,
			&nitems, &remain_bytes, &data) != Success) {
		Msg("XGetWindowProperty failed.\n");
		return;
	}
	if (remain_bytes > 0)
		XDeleteProperty(mainDisplay, drawWindow, conv_property);

	if(act_target == None || conv_target != act_target)
		return;
	if(act_format != 8) {
		XFree(data);
		return;
	}

	CvtCompoundTextToEuc(dst, data);

	XFree(data);
}

void KinputEndConversion ()
{
	Window			owner_win;
	char			msg[80];
	char *			str1;
	XClientMessageEvent	xcme;

	if(imProtocol != IM_KINPUT)
		return;

#ifdef DEBUG
	fprintf(stderr, "KinputEndConversion() : %d\n", conv_mode);
#endif
	switch(conv_mode) {
	    case WAIT_CONV_ON :
		Msg("Waiting for start conversion.\n");
		return;
	    case WAIT_CONV_OFF :
	    case CONV_OFF :
		return;
	}

	if((owner_win = XGetSelectionOwner(mainDisplay, conv_selection))
			== None) {
		str1 = XGetAtomName(mainDisplay, conv_selection);
		sprintf(msg, "There is no Selection Owner of %s.\n", str1);
		Msg(msg);
		XFree(str1);
		conv_owner_win = None;
		conv_mode = CONV_OFF;
#ifdef DEBUG
		fprintf(stderr, "0x%08x\n", owner_win);
#endif
		return;
	}
#ifdef DEBUG
	fprintf(stderr, "0x%08x\n", owner_win);
#endif

	if(conv_owner_win != owner_win) {
		conv_mode = CONV_OFF;
		return;
	}

	xcme.type = ClientMessage;
	xcme.display = mainDisplay;
	xcme.window = owner_win;
	xcme.message_type = conv_end_req;
	xcme.format = 32;
	xcme.data.l[0] = conv_selection;
	xcme.data.l[1] = drawWindow;
	if(XSendEvent(mainDisplay, owner_win, False, NoEventMask, (XEvent *)&xcme)
			== 0) {
		Msg("Failing in disconnecting owner.\n");
		conv_mode = CONV_OFF;
		return;
	}

	conv_mode = WAIT_CONV_OFF;
}

Atom InternAtom(name)
char *	name;
{
	Atom	ret;
	char	msg[80];

	if((ret = XInternAtom(mainDisplay, name, False)) == None) {
		sprintf(msg, "Can't Intern Atom from %s.\n", name);
		Msg(msg);
	}

	return ret;
}

void CvtCompoundTextToEuc(dst, src)
register char *	dst;
register char *	src;
{
	int	status;

	status = LATIN;

	while(*src != '\0') {
		switch(status) {
		    case LATIN :
			if(*src == '\033')
				status = ESC;
			else
				*dst++= *src;
			break;
		    case ESC :
			if(*src == '$')
				status = ESC_24;
			else if(*src == '(')
				status = ESC_28;
			else {
				status = LATIN;
				*dst++= *src;
			}
			break;
		    case ESC_28 :
			if(*src == 'B' || *src == 'J')	/* noda */
				status = LATIN;
			else {
				status = LATIN;
				*dst++= *src;
			}
			break;
		    case ESC_24 :
			if(*src == '(')
				status = ESC_GL;
			else if(*src == ')')
				status = ESC_GR;
			else {
				status = LATIN;
				*dst++= *src;
			}
			break;
		    case ESC_GL :
			if(*src == 'B' || *src == '@')
				status = KANJI_GL;
			else {
				status = LATIN;
				*dst++= *src;
			}
			break;
		    case ESC_GR :
			if(*src == 'B' || *src == '@')
				status = KANJI_GR;
			else {
				status = LATIN;
				*dst++= *src;
			}
			break;
		    case KANJI_GL :
			if(*src == '\033')
				status = ESC;
			else
				*dst++= (*src | 0x80);
			break;
		    case KANJI_GR :
			if(*src == '\033')
				status = ESC;
			else
				*dst++= *src;
			break;
		    default :
			status = LATIN;
			break;
		}
		src ++;
	}

	*dst = '\0';
}

void CvtJisToEuc(dst, src)
register char *	dst;
register char *	src;
{
	register int	kanjiMode;
	register int	len;

	kanjiMode = KANJI0;
	len = 0;

	while(*src != '\0') {
		switch(kanjiMode) {
		    case KANJI0 :
			if(*src == '\033')
				kanjiMode = KANJI1;
			else
				*dst ++ = *src;
			break;
		    case KANJI1 :
			if(*src == '$')
				kanjiMode = KANJI2;
			else
				kanjiMode = KANJI0;
			break;
		    case KANJI2 :
			if(*src == '@' || *src == 'B') {
				len = 0;
				kanjiMode = KANJI3;
			}
			else
				kanjiMode = KANJI0;
			break;
		    case KANJI3 :
			if(*src == '\033')
				kanjiMode = KANJI4;
			else {
			   len ++;
			   if(*(src + 1) != '\033' || (len & 1) == 0)
				*dst ++ = (*src | 0x80);
			}
			break;
		    case KANJI4 :
			if(*src == '(')
				kanjiMode = KANJI5;
			else {
				len = 0;
				kanjiMode = KANJI3;
			}
			break;
		    case KANJI5 :
			if(*src == 'J' || *src == 'B')
				kanjiMode = KANJI0;
			else {
				len = 0;
				kanjiMode = KANJI3;
			}
			break;
		    default :
			break;
		}
		src ++;
	}

	*dst = '\0';
}

int CvtEucToJis(dst, src)
register char	*dst;
register char	*src;
{
	int	status;
	int	len;

	status = LATIN;
	len = 0;

	while(*src != '\0') {
		switch(status) {
		    case LATIN :
			while((*src & 0x80) == 0 && *src != '\0') {
				if(dst)
					*dst ++ = *src;
				src ++;
				len ++;
			}
			status = KANJI_GR;
			break;
		    case KANJI_GR :
			if(dst) {
				*dst ++ = '\033';
				*dst ++ = '$';
				*dst ++ = 'B';
			}
			len += 3;
			while((*src & 0x80) == 0x80 && *src != '\0') {
				if(dst)
					*dst ++ = *src & 0x7f;
				src ++;
				len ++;
			}
			if(dst) {
				*dst ++ = '\033';
				*dst ++ = '(';
				*dst ++ = 'B';
			}
			len += 3;
			status = LATIN;
			break;
		    default :
			break;
		}
	}

	if(dst)
		*dst = '\0';

	return len;
}
