/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/edit.c,v 2.87.1.4 1993/03/08 02:18:37 william Exp $";
#endif

#include <stdio.h>
#include <math.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "align.e"
#include "arc.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "copypaste.e"
#include "cursor.e"
#include "drawing.e"
#include "dup.e"
#include "eps.e"
#include "font.e"
#include "grid.e"
#include "group.e"
#include "mainloop.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "obj.e"
#include "page.e"
#include "pattern.e"
#include "poly.e"
#include "raster.e"
#include "select.e"
#include "setup.e"
#include "special.e"
#include "spline.e"
#include "stretch.e"
#include "text.e"
#include "xbitmap.e"
#include "xpixmap.e"

char * editMenuStr[] =
      { "Redraw         ^r",
        "Duplicate      ^d",
        "Delete         ^x",
        "SelectAll      ^a",
        "Undo           #u",
        "Redo           #*",
        "DeletePoint   ^#d",
        "AddPoint      ^#a",
        "Copy           ^y",
        "Paste         ^#y",
        "Update         #0",
        "PreciseScale   #)",
        "FlushUndoBuffer  ",
        "PrintMsgBuffer   ",
        "InvertXBitmap ^#f",
        "PushCurChoice  ^e",
        "SpecifyAnArc   #9",
        "CutBit/Pixmap  #;",
        "RestoreImageWH   ",
        "UpdateEPS        ",
        "MakeRegPolygon #\"",
        "BreakUpText    ##",
        "SelectionPaste^#Y"		/* noda */
      };

static struct ObjRec	* tmpTopObj, * tmpBotObj;
static struct SelRec	* tmpTopSel, * tmpBotSel;
static int		tmpCount;

static
void BreakATextObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   struct ObjRec	* prototype, * char_obj;
   struct StrRec	* s_ptr, * prev_str;
   struct TextRec	* text_ptr;
   struct SelRec	* new_sel_ptr;
   char			* c_ptr;
   int			w, x, y, xinc=0, yinc=0, x_pos, y_pos;
   int			dx, dy, x_dist, y_dist, char_w, len;

   tmpTopSel = tmpBotSel = NULL;
   tmpTopObj = tmpBotObj = NULL;
   tmpCount = 0;

   prototype = DupObj (ObjPtr);

   text_ptr = prototype->detail.t;
   for (s_ptr=text_ptr->last->prev; s_ptr!=NULL; s_ptr=prev_str)
   {
      prev_str = s_ptr->prev;
      cfree (s_ptr);
   }
   text_ptr->first = text_ptr->last;
   text_ptr->first->s[1] = '\0';
   text_ptr->lines = 1;

   text_ptr = ObjPtr->detail.t;

   SaveCurFont ();
   curKanjiFont = text_ptr->kanji_font;
   curFont = text_ptr->font;
   curFontDPI = text_ptr->dpi;
   curStyle = text_ptr->style;
   curSize = text_ptr->size;
   textJust = text_ptr->just;
   textVSpace = text_ptr->v_space;
   curRotate = text_ptr->rotate;
   SetCanvasFont ();

   x = ObjPtr->x;
   y = ObjPtr->y;
   switch (curRotate)
   {
      case ROTATE0: xinc = 0; yinc = textCursorH+textVSpace; break;
      case ROTATE90: xinc = -(textCursorH+textVSpace); yinc = 0; break;
      case ROTATE180: xinc = 0; yinc = -(textCursorH+textVSpace); break;
      case ROTATE270: xinc = textCursorH+textVSpace; yinc = 0; break;
   }

   for (s_ptr=text_ptr->first; s_ptr!=NULL; s_ptr=s_ptr->next)
   {
      c_ptr = s_ptr->s;
      len = strlen (c_ptr);
      w = TextWidth (canvasFontPtr, canvasKanjiFontPtr, c_ptr, len);

      x_pos = x;
      y_pos = y;
      switch (textJust)
      {
         case JUST_L: break;
         case JUST_C:
            switch (curRotate)
            {
               case ROTATE0: x_pos -= w/2; break;
               case ROTATE90: y_pos -= w/2; break;
               case ROTATE180: x_pos += w/2; break;
               case ROTATE270: y_pos += w/2; break;
            }
            break;
         case JUST_R:
            switch (curRotate)
            {
               case ROTATE0: x_pos -= w; break;
               case ROTATE90: y_pos -= w; break;
               case ROTATE180: x_pos += w; break;
               case ROTATE270: y_pos += w; break;
            }
            break;
      }

      len = 1;
      for ( ; *c_ptr != '\0'; c_ptr += len)
      {
         char_obj = DupObj (prototype);
	 if ((*c_ptr & 0x80) && curKanjiFont != KANJI_FONT_NONE) {
	    char_obj->detail.t->first->s[0] = *c_ptr;
	    char_obj->detail.t->first->s[1] = *(c_ptr + 1);
	    char_obj->detail.t->first->s[2] = '\0';
	    len = 2;
	 }
	 else {
	    char_obj->detail.t->first->s[0] = *c_ptr;
	    char_obj->detail.t->first->s[1] = '\0';
	    len = 1;
	 }
         char_w = TextWidth (canvasFontPtr, canvasKanjiFontPtr, c_ptr, len);

         dx = dy = 0;
         switch (textJust)
         {
            case JUST_L: break;
            case JUST_C:
               switch (curRotate)
               {
                  case ROTATE0: dx = char_w/2; break;
                  case ROTATE90: dy = char_w/2; break;
                  case ROTATE180: dx = -(char_w/2); break;
                  case ROTATE270: dy = -(char_w/2); break;
               }
               break;
            case JUST_R:
               switch (curRotate)
               {
                  case ROTATE0: dx = char_w; break;
                  case ROTATE90: dy = char_w; break;
                  case ROTATE180: dx = -char_w; break;
                  case ROTATE270: dy = char_w; break;
               }
               break;
         }
         x_dist = x_pos+dx-prototype->x;
         y_dist = y_pos+dy-prototype->y;
         MoveObj (char_obj, x_dist, y_dist);
         SetTextBBox (char_obj, textJust, char_w, textCursorH, curRotate);

         switch (curRotate)
         {
            case ROTATE0: x_pos += char_w; break;
            case ROTATE90: y_pos += char_w; break;
            case ROTATE180: x_pos -= char_w; break;
            case ROTATE270: y_pos -= char_w; break;
         }
         new_sel_ptr = (struct SelRec *) calloc (1, sizeof(struct SelRec));
         new_sel_ptr->obj = char_obj;

         new_sel_ptr->next = tmpTopSel;
         char_obj->next = tmpTopObj;
         if (tmpTopSel == NULL)
         {
            tmpBotSel = new_sel_ptr;
            tmpBotObj = char_obj;
         }
         else
         {
            tmpTopSel->prev = new_sel_ptr;
            tmpTopObj->prev = char_obj;
         }
         tmpTopSel = new_sel_ptr;
         tmpTopObj = char_obj;
         tmpTopSel->prev = NULL;
         tmpTopObj->prev = NULL;
         tmpCount++;
      }
      x += xinc;
      y += yinc;
   }

   RestoreCurFont ();
   FreeTextObj (prototype);
}

void BreakUpText ()
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr, * next_sel;
   int				sel_ltx, sel_lty, sel_rbx, sel_rby;
   int				changed=FALSE;

   if (topSel == NULL) return;

   HighLightReverse ();
   sel_ltx = selLtX; sel_lty = selLtY;
   sel_rbx = selRbX; sel_rby = selRbY;

   StartCompositeCmd ();
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = next_sel)
   {
      next_sel = sel_ptr->next;
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->type == OBJ_TEXT && !obj_ptr->locked)
      {
         changed = TRUE;

         PrepareToReplaceAnObj (obj_ptr);
         BreakATextObj (obj_ptr);

         tmpTopSel->obj->prev = obj_ptr->prev;
         if (obj_ptr->prev == NULL)
            curPage->top = topObj = tmpTopSel->obj;
         else
            obj_ptr->prev->next = tmpTopSel->obj;
         tmpBotSel->obj->next = obj_ptr->next;
         if (obj_ptr->next == NULL)
            curPage->bot = botObj = tmpBotSel->obj;
         else
            obj_ptr->next->prev = tmpBotSel->obj;

         RecordCmd (CMD_ONE_TO_MANY, NULL, tmpTopSel, tmpBotSel, tmpCount);

         tmpTopSel->prev = sel_ptr->prev;
         if (sel_ptr->prev == NULL)
            topSel = tmpTopSel;
         else
            sel_ptr->prev->next = tmpTopSel;
         tmpBotSel->next = sel_ptr->next;
         if (sel_ptr->next == NULL)
            botSel = tmpBotSel;
         else
            sel_ptr->next->prev = tmpBotSel;

         FreeObj (obj_ptr);
         cfree (sel_ptr);
      }
   }
   EndCompositeCmd ();

   if (changed)
   {
      UpdSelBBox ();
      RedrawAreas (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1),
            sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
            sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
      justDupped = FALSE;
      Msg ("Text string is broken up into characters.");
   }
   HighLightForward ();
}

void MakeRegularPolygon ()
{
   register int		i;
   int			vertex_at_right, xc, yc;
   int			sel_ltx, sel_lty, sel_rbx, sel_rby, sides, radius;
   int			ltx, lty, rbx, rby;
   double		inc, angle, r;
   char			msg[MAXSTRING];
   struct ObjRec	* obj_ptr, * new_obj_ptr;
   struct PolygonRec	* polygon_ptr;

   if (topSel == NULL) return;
   if (topSel != botSel || topSel->obj->type != OBJ_POLYGON)
   {
      Msg ("Please select one polygon object to make it regular.");
      return;
   }
   if (topSel->obj->locked)
   {
      Msg ("Polygon locked.");
      return;
   }

   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);

   sel_ltx = selLtX; sel_lty = selLtY;
   sel_rbx = selRbX; sel_rby = selRbY;

   obj_ptr = topSel->obj;

   radius = (min(obj_ptr->obbox.rbx-obj_ptr->obbox.ltx,
         obj_ptr->obbox.rby-obj_ptr->obbox.lty))>>1;
   if (radius < 1)
   {
      Msg ("Selected polygon is too small to make regular.");
      return;
   }

   sprintf (msg, "Do you want a vertex at the 3 o'clock position?  [ync](y)");
   if ((vertex_at_right = YesNoCancel (msg, CONFIRM_YES)) == CONFIRM_CANCEL)
      return;

   tmpTopObj = tmpBotObj = NULL;
   tmpTopSel = tmpBotSel = NULL;

   HighLightReverse ();

   new_obj_ptr = DupObj (obj_ptr);

   UnlinkObj (obj_ptr);

   polygon_ptr = new_obj_ptr->detail.g;
   sides = polygon_ptr->n-1;
   inc = 2*M_PI/sides;
   angle = (vertex_at_right==CONFIRM_YES) ? 0 : inc/2;

   xc = obj_ptr->obbox.ltx+radius;
   yc = obj_ptr->obbox.lty+radius;

   if ((sides%4)==0 && vertex_at_right==CONFIRM_NO)
      r = ((double)(min(obj_ptr->obbox.rbx-obj_ptr->obbox.ltx,
            obj_ptr->obbox.rby-obj_ptr->obbox.lty)) / cos(angle)) / 2;
   else
      r = radius;

   ltx = obj_ptr->obbox.rbx;
   lty = obj_ptr->obbox.rby;
   rbx = obj_ptr->obbox.ltx;
   rby = obj_ptr->obbox.lty;
   for (i = 0; i < sides; i++, angle+=inc)
   {
      polygon_ptr->vlist[i].x = xc + round(r*cos(angle));
      polygon_ptr->vlist[i].y = yc - round(r*sin(angle));
      if (polygon_ptr->vlist[i].x < ltx) ltx = polygon_ptr->vlist[i].x;
      if (polygon_ptr->vlist[i].y < lty) lty = polygon_ptr->vlist[i].y;
      if (polygon_ptr->vlist[i].x > rbx) rbx = polygon_ptr->vlist[i].x;
      if (polygon_ptr->vlist[i].y > rby) rby = polygon_ptr->vlist[i].y;
   }
   polygon_ptr->vlist[sides].x = polygon_ptr->vlist[0].x;
   polygon_ptr->vlist[sides].y = polygon_ptr->vlist[0].y;
   new_obj_ptr->obbox.ltx = ltx;
   new_obj_ptr->obbox.lty = lty;
   new_obj_ptr->obbox.rbx = rbx;
   new_obj_ptr->obbox.rby = rby;
   AdjObjSplineVs (new_obj_ptr);
   AdjObjBBox (new_obj_ptr);

   topSel->obj = botSel->obj = new_obj_ptr;
   AddObj (NULL, topObj, new_obj_ptr);
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   FreeObj (obj_ptr);

   UpdSelBBox ();
   RedrawAnArea (botObj, sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
         sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1));
   SetFileModified (TRUE);
   justDupped = FALSE;
   HighLightForward ();
}

void DeletePoint ()
{
   register int			i;
   register struct ObjRec	* obj_ptr;
   struct PolyRec		* poly_ptr=NULL;
   struct PolygonRec		* polygon_ptr=NULL;
   int				index, n, pt_deleted=FALSE, deleting=TRUE;
   int				root_x, root_y, old_x, old_y;
   unsigned int			status;
   Window			root_win, child_win;
   XEvent			input, ev;

   if (!(topSel != NULL && topSel == botSel &&
         (topSel->obj->type == OBJ_POLY || topSel->obj->type == OBJ_POLYGON)))
   {
      Msg ("Please select only one POLY or POLYGON object.");
      return;
   }
   if (topSel->obj->locked)
   {
      Msg ("Can not delete points for locked object.");
      return;
   }
   if (curChoice == VERTEXMODE)
   {
      HighLightReverse ();
      JustRemoveAllVSel ();
      HighLightForward ();
   }
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   obj_ptr = topSel->obj;
   switch (obj_ptr->type)
   {
      case OBJ_POLY: poly_ptr = obj_ptr->detail.p; break;
      case OBJ_POLYGON: polygon_ptr = obj_ptr->detail.g; break;
   }
   TwoLineMsg ("Click left mouse button to DELETE points.",
         "Click other buttons to quit.");

   XGrabPointer (mainDisplay, drawWindow, False,
         PointerMotionMask | ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, defaultCursor, CurrentTime);
   XQueryPointer (mainDisplay, drawWindow, &root_win, &child_win,
         &root_x, &root_y, &old_x, &old_y, &status);
   XSetFont (mainDisplay, revDefaultGC, defaultFontPtr->fid);
   XDrawString (mainDisplay, drawWindow, revDefaultGC,
         old_x+4, old_y+defaultFontAsc, "DEL", 3);
   MarkRulers (old_x, old_y);

   while (deleting)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonPress)
      {
         if (input.xbutton.button == Button1)
         {
            if (obj_ptr->type == OBJ_POLY &&
                  PtInPolyMark (input.xbutton.x, input.xbutton.y,
                  poly_ptr->n, poly_ptr->vlist, &index) ||
                  obj_ptr->type == OBJ_POLYGON &&
                  PtInPolyMark (input.xbutton.x, input.xbutton.y,
                  polygon_ptr->n-1, polygon_ptr->vlist, &index))
            {
               pt_deleted = TRUE;
               HighLightReverse ();
               if (obj_ptr->type == OBJ_POLY && poly_ptr->n == 2 ||
                     obj_ptr->type == OBJ_POLYGON && polygon_ptr->n == 4)
               {
                  DelObj (obj_ptr);
                  deleting = FALSE;
                  obj_ptr = NULL;
                  cfree (topSel);
                  topSel = botSel = NULL;
               }
               else
               {
                  switch (obj_ptr->type)
                  {
                     case OBJ_POLY:
                        n = poly_ptr->n;
                        for (i = index+1; i < n; i++)
                           poly_ptr->vlist[i-1] = poly_ptr->vlist[i];
                        poly_ptr->n--;
                        AdjObjSplineVs (obj_ptr);
                        UpdPolyBBox (obj_ptr, poly_ptr->n, poly_ptr->vlist);
                        break;
                     case OBJ_POLYGON:
                        n = polygon_ptr->n;
                        for (i = index+1; i < n; i++)
                           polygon_ptr->vlist[i-1] = polygon_ptr->vlist[i];
                        polygon_ptr->n--;
                        n--;
                        if (index == 0)
                           polygon_ptr->vlist[n-1] = polygon_ptr->vlist[0];
                        AdjObjSplineVs (obj_ptr);
                        UpdPolyBBox (obj_ptr, polygon_ptr->n,
                              polygon_ptr->vlist);
                        break;
                  }
                  AdjObjBBox (obj_ptr);
               }
               XDrawString (mainDisplay, drawWindow, revDefaultGC, old_x+4,
                     old_y+defaultFontAsc, "DEL", 3);
               old_x = input.xbutton.x;
               old_y = input.xbutton.y;
               RedrawAnArea (botObj,
                     selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
                     selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
               HighLightForward ();
               if (obj_ptr != NULL)
                  XDrawString (mainDisplay, drawWindow, revDefaultGC, old_x+4,
                        old_y+defaultFontAsc, "DEL", 3);
               UpdSelBBox ();
               SetFileModified (TRUE);
               justDupped = FALSE;
            }
         }
         else
         {
            XUngrabPointer (mainDisplay, CurrentTime);
            Msg ("");
            deleting = FALSE;
            XDrawString (mainDisplay, drawWindow, revDefaultGC,
                  old_x+4, old_y+defaultFontAsc, "DEL", 3);
         }
      }
      else if (input.type == MotionNotify)
      {
         XDrawString (mainDisplay, drawWindow, revDefaultGC,
               old_x+4, old_y+defaultFontAsc, "DEL", 3);
         old_x = input.xmotion.x;
         old_y = input.xmotion.y;
         XDrawString (mainDisplay, drawWindow, revDefaultGC,
               old_x+4, old_y+defaultFontAsc, "DEL", 3);
         MarkRulers (old_x, old_y);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   if (pt_deleted)
   {
      if (topSel == NULL)
         ChangeReplaceOneCmdToDeleteCmd ();
      else
         RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   }
   else
      AbortPrepareCmd (CMD_REPLACE);
}

static
int ContinueAddPolyPoint (ObjPtr, MouseX, MouseY, Index, PolyPtr,
      LastMouseX, LastMouseY)
   struct ObjRec	* ObjPtr;
   int			MouseX, MouseY, Index;
   struct PolyRec	* PolyPtr;
   int			* LastMouseX, * LastMouseY;
   /* (MouseX,MouseY) is the mouse's origin in screen offsets */
{
   int		n = PolyPtr->n, sn = 0, curved=PolyPtr->curved;
   int		already_moved = FALSE, done = FALSE, before = FALSE;
   XPoint	* vs = PolyPtr->vlist, v[3], * sv = NULL, * pv = NULL;
   int		prev_x, prev_y, x, y, next_x, next_y, new_x, new_y;
   int		orig_x, orig_y, grid_x, grid_y, new_mouse_x, new_mouse_y;
   int		sel_ltx, sel_lty, sel_rbx, sel_rby, num = 0, i;
   double	prev_angle, next_angle, new_angle, theta_1, theta_2;
   XEvent	input, ev;

   MARK(drawWindow, revDefaultGC, OFFSET_X(vs[Index].x), OFFSET_Y(vs[Index].y));

   sel_ltx =  selLtX; sel_lty = selLtY;
   sel_rbx =  selRbX; sel_rby = selRbY;

   x = vs[Index].x;
   y = vs[Index].y;

   if (Index == 0)
   {
      next_x = vs[1].x;    next_y = vs[1].y;
      prev_x = 2*x-next_x; prev_y = 2*y-next_y;
   }
   else if (Index == n-1)
   {
      prev_x = vs[n-2].x;  prev_y = vs[n-2].y;
      next_x = 2*x-prev_x; next_y = 2*y-prev_y;
   }
   else
   {
      prev_x = vs[Index-1].x; prev_y = vs[Index-1].y;
      next_x = vs[Index+1].x; next_y = vs[Index+1].y;
   }
   prev_angle = (prev_x==x) ? ((prev_y>=y) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(prev_y-y), (double)(prev_x-x));
   next_angle = (next_x==x) ? ((next_y>=y) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(next_y-y), (double)(next_x-x));

   if (curved && splineRubberband)
   {
      pv = (XPoint *) calloc (n+2, sizeof (XPoint));
      for (i = 0; i <= Index; i++)
      {
         pv[i].x = vs[i].x;
         pv[i].y = vs[i].y;
      }
      for (i = Index; i < n; i++)
      {
         pv[i+1].x = vs[i].x;
         pv[i+1].y = vs[i].y;
      }
   }

   GridXY (MouseX, MouseY, &orig_x, &orig_y);
   new_mouse_x = MouseX; new_mouse_y = MouseY;
   XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
         new_mouse_y+defaultFontAsc, "ADD", 3);

   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == MotionNotify)
      {
         XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
               new_mouse_y+defaultFontAsc, "ADD", 3);
         new_mouse_x = input.xmotion.x;
         new_mouse_y = input.xmotion.y;
         XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
               new_mouse_y+defaultFontAsc, "ADD", 3);

         GridXY (new_mouse_x, new_mouse_y, &grid_x, &grid_y);
         new_x = ABS_SIZE(new_mouse_x-MouseX) + x;
         new_y = ABS_SIZE(new_mouse_y-MouseY) + y;
         if (!already_moved)
         {
            already_moved = TRUE;

            new_angle = (new_x==x) ? ((new_y>=y) ? M_PI/2.0 : -M_PI/2.0) :
                  atan2 ((double)(new_y-y), (double)(new_x-x));
            theta_1 = fabs (prev_angle - new_angle);
            theta_2 = fabs (next_angle - new_angle);
            if (theta_1 > M_PI) theta_1 = 2*M_PI-theta_1;
            if (theta_2 > M_PI) theta_2 = 2*M_PI-theta_2;
            before = (theta_1 <= theta_2);

            if (before)
            {  /* Add a point between the current and the previous point */
               if (Index == 0)
               {
                  num = 2;
                  v[0].x = OFFSET_X(x); v[0].y = OFFSET_Y(y);
                  v[1].x = OFFSET_X(x); v[1].y = OFFSET_Y(y);
               }
               else
               {
                  num = 3;
                  v[0].x = OFFSET_X(prev_x); v[0].y = OFFSET_Y(prev_y);
                  v[1].x = OFFSET_X(x);      v[1].y = OFFSET_Y(y);
                  v[2].x = OFFSET_X(x);      v[2].y = OFFSET_Y(y);
               }
            }
            else
            {  /* Add a point between the current and the next point */
               if (Index == n-1)
               {
                  num = 2;
                  v[0].x = OFFSET_X(x);      v[0].y = OFFSET_Y(y);
                  v[1].x = OFFSET_X(x);      v[1].y = OFFSET_Y(y);
               }
               else
               {
                  num = 3;
                  v[0].x = OFFSET_X(x);      v[0].y = OFFSET_Y(y);
                  v[1].x = OFFSET_X(x);      v[1].y = OFFSET_Y(y);
                  v[2].x = OFFSET_X(next_x); v[2].y = OFFSET_Y(next_y);
               }
            }
            if (curved && splineRubberband)
            {
               sv = MakeSplinePolyVertex (&sn,drawOrigX,drawOrigY,n+1,pv);
               XDrawLines (mainDisplay, drawWindow, revDefaultGC,
                     PolyPtr->svlist, PolyPtr->sn, CoordModeOrigin);
            }
         }
         else
         {
            if (curved && splineRubberband)
               XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
                     CoordModeOrigin);
            else
               MyDashedLine (drawWindow, revDefaultGC, v, num);
            v[1].x = OFFSET_X(x) + grid_x - orig_x;
            v[1].y = OFFSET_Y(y) + grid_y - orig_y;
            if (curved && splineRubberband)
            {
               cfree (sv);
               if (before)
               {
                  pv[Index].x = x + ABS_SIZE(grid_x-orig_x);
                  pv[Index].y = y + ABS_SIZE(grid_y-orig_y);
               }
               else
               {
                  pv[Index+1].x = x + ABS_SIZE(grid_x-orig_x);
                  pv[Index+1].y = y + ABS_SIZE(grid_y-orig_y);
               }
               sv = MakeSplinePolyVertex (&sn,drawOrigX,drawOrigY,n+1,pv);
               XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
                     CoordModeOrigin);
            }
            else
               MyDashedLine (drawWindow, revDefaultGC, v, num);
            MarkRulers (v[1].x, v[1].y);
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
      else if (input.type == ButtonRelease)
      {
         done = TRUE;
         *LastMouseX = new_mouse_x; *LastMouseY = new_mouse_y;
         MARK(drawWindow, revDefaultGC, OFFSET_X(vs[Index].x),
               OFFSET_Y(vs[Index].y));
         XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
               new_mouse_y+defaultFontAsc, "ADD", 3);

         if (!already_moved)
            return (FALSE);
         else
         {
            if (curved && splineRubberband)
               XDrawLines (mainDisplay, drawWindow, revDefaultGC,
                     PolyPtr->svlist, PolyPtr->sn, CoordModeOrigin);
            if (grid_x == orig_x && grid_y == orig_y)
               return (FALSE);
         }

         HighLightReverse ();
         vs = (XPoint *) realloc (vs, (n+2)*sizeof(XPoint));
         if (vs == NULL)
         {
            Warning ("ContinueAddPolyPoint()", "Can not realloc()");
            return (FALSE);
         }
         PolyPtr->vlist = vs;
         if (before)
         {
            for (i = n-1; i >= Index; i--) vs[i+1] = vs[i];
            vs[Index].x = x + ABS_SIZE(grid_x-orig_x);
            vs[Index].y = y + ABS_SIZE(grid_y-orig_y);
         }
         else
         {
            for (i = n-1; i > Index; i--) vs[i+1] = vs[i];
            vs[Index+1].x = x + ABS_SIZE(grid_x-orig_x);
            vs[Index+1].y = y + ABS_SIZE(grid_y-orig_y);
         }
         if (curved && splineRubberband) { cfree (sv); cfree (pv); }
         PolyPtr->n++;
         n++;
         AdjObjSplineVs (ObjPtr);
         UpdPolyBBox (ObjPtr, PolyPtr->n, PolyPtr->vlist);
         AdjObjBBox (ObjPtr);

         UpdSelBBox ();
         RedrawAreas (botObj,
               sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
               sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1),
               selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
               selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
         HighLightForward ();
         SetFileModified (TRUE);
         justDupped = FALSE;
      }
   }
   return (TRUE);
}

static
int ContinueAddPolygonPoint (ObjPtr, MouseX, MouseY, Index, PolygonPtr,
      LastMouseX, LastMouseY)
   struct ObjRec	* ObjPtr;
   int			MouseX, MouseY, Index;
   struct PolygonRec	* PolygonPtr;
   int			* LastMouseX, * LastMouseY;
   /* (MouseX,MouseY) is the mouse's origin in screen offsets */
{
   int		n = PolygonPtr->n, sn = 0, curved=PolygonPtr->curved;
   int		already_moved = FALSE, done = FALSE, before = FALSE;
   XPoint	* vs = PolygonPtr->vlist, v[3], * sv = NULL, * pv = NULL;
   int		prev_x, prev_y, x, y, next_x, next_y, new_x, new_y;
   int		orig_x, orig_y, grid_x, grid_y, new_mouse_x, new_mouse_y;
   int		sel_ltx, sel_lty, sel_rbx, sel_rby, i;
   double	prev_angle, next_angle, new_angle, theta_1, theta_2;
   XEvent	input, ev;

   MARK(drawWindow, revDefaultGC, OFFSET_X(vs[Index].x), OFFSET_Y(vs[Index].y));

   sel_ltx =  selLtX; sel_lty = selLtY;
   sel_rbx =  selRbX; sel_rby = selRbY;

   x = vs[Index].x;
   y = vs[Index].y;

   if (Index == 0 || Index == n-1)
   {
      next_x = vs[1].x;   next_y = vs[1].y;
      prev_x = vs[n-2].x; prev_y = vs[n-2].y;
   }
   else
   {
      prev_x = vs[Index-1].x; prev_y = vs[Index-1].y;
      next_x = vs[Index+1].x; next_y = vs[Index+1].y;
   }
   prev_angle = (prev_x==x) ? ((prev_y>=y) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(prev_y-y), (double)(prev_x-x));
   next_angle = (next_x==x) ? ((next_y>=y) ? M_PI/2.0 : -M_PI/2.0) :
         atan2 ((double)(next_y-y), (double)(next_x-x));

   if (curved && splineRubberband)
   {
      pv = (XPoint *) calloc (n+2, sizeof (XPoint));
      for (i = 0; i <= Index; i++)
      {
         pv[i].x = vs[i].x;
         pv[i].y = vs[i].y;
      }
      for (i = Index; i < n; i++)
      {
         pv[i+1].x = vs[i].x;
         pv[i+1].y = vs[i].y;
      }
   }

   GridXY (MouseX, MouseY, &orig_x, &orig_y);
   new_mouse_x = MouseX; new_mouse_y = MouseY;
   XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
         new_mouse_y+defaultFontAsc, "ADD", 3);

   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == MotionNotify)
      {
         XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
               new_mouse_y+defaultFontAsc, "ADD", 3);
         new_mouse_x = input.xmotion.x;
         new_mouse_y = input.xmotion.y;
         XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
               new_mouse_y+defaultFontAsc, "ADD", 3);

         GridXY (new_mouse_x, new_mouse_y, &grid_x, &grid_y);
         new_x = ABS_SIZE(new_mouse_x-MouseX) + x;
         new_y = ABS_SIZE(new_mouse_y-MouseY) + y;
         if (!already_moved)
         {
            already_moved = TRUE;

            new_angle = (new_x==x) ? ((new_y>=y) ? M_PI/2.0 : -M_PI/2.0) :
                  atan2 ((double)(new_y-y), (double)(new_x-x));
            theta_1 = fabs (prev_angle - new_angle);
            theta_2 = fabs (next_angle - new_angle);
            if (theta_1 > M_PI) theta_1 = 2*M_PI-theta_1;
            if (theta_2 > M_PI) theta_2 = 2*M_PI-theta_2;
            before = (theta_1 <= theta_2);

            if (before)
            {  /* Add a point between the current and the previous point */
               v[0].x = OFFSET_X(prev_x); v[0].y = OFFSET_Y(prev_y);
               v[1].x = OFFSET_X(x);      v[1].y = OFFSET_Y(y);
               v[2].x = OFFSET_X(x);      v[2].y = OFFSET_Y(y);
            }
            else
            {  /* Add a point between the current and the next point */
               v[0].x = OFFSET_X(x);      v[0].y = OFFSET_Y(y);
               v[1].x = OFFSET_X(x);      v[1].y = OFFSET_Y(y);
               v[2].x = OFFSET_X(next_x); v[2].y = OFFSET_Y(next_y);
            }
            if (curved && splineRubberband)
            {
               sv = MakeSplinePolygonVertex (&sn,drawOrigX,drawOrigY,n+1,pv);
               XDrawLines (mainDisplay, drawWindow, revDefaultGC,
                     PolygonPtr->svlist, PolygonPtr->sn, CoordModeOrigin);
            }
         }
         else
         {
            if (curved && splineRubberband)
               XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
                     CoordModeOrigin);
            else
               MyDashedLine (drawWindow, revDefaultGC, v, 3);
            v[1].x = OFFSET_X(x) + grid_x - orig_x;
            v[1].y = OFFSET_Y(y) + grid_y - orig_y;
            if (curved && splineRubberband)
            {
               cfree (sv);
               if (before)
               {
                  pv[Index].x = x + ABS_SIZE(grid_x-orig_x);
                  pv[Index].y = y + ABS_SIZE(grid_y-orig_y);
                  if (Index == 0)
                  {
                     pv[n].x = x + ABS_SIZE(grid_x-orig_x);
                     pv[n].y = y + ABS_SIZE(grid_y-orig_y);
                  }
               }
               else
               {
                  pv[Index+1].x = x + ABS_SIZE(grid_x-orig_x);
                  pv[Index+1].y = y + ABS_SIZE(grid_y-orig_y);
                  if (Index == n-1)
                  {
                     pv[0].x = x + ABS_SIZE(grid_x-orig_x);
                     pv[0].y = y + ABS_SIZE(grid_y-orig_y);
                  }
               }
               sv = MakeSplinePolygonVertex (&sn,drawOrigX,drawOrigY,n+1,pv);
               XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
                     CoordModeOrigin);
            }
            else
               MyDashedLine (drawWindow, revDefaultGC, v, 3);
            MarkRulers (v[1].x, v[1].y);
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
      else if (input.type == ButtonRelease)
      {
         done = TRUE;
         *LastMouseX = new_mouse_x; *LastMouseY = new_mouse_y;
         MARK(drawWindow, revDefaultGC, OFFSET_X(vs[Index].x),
               OFFSET_Y(vs[Index].y));
         XDrawString (mainDisplay, drawWindow, revDefaultGC, new_mouse_x+4,
               new_mouse_y+defaultFontAsc, "ADD", 3);

         if (!already_moved)
            return (FALSE);
         else
         {
            if (curved && splineRubberband)
               XDrawLines (mainDisplay, drawWindow, revDefaultGC,
                     PolygonPtr->svlist, PolygonPtr->sn, CoordModeOrigin);
            if (grid_x == orig_x && grid_y == orig_y)
               return (FALSE);
         }

         HighLightReverse ();
         vs = (XPoint *) realloc (vs, (n+2)*sizeof(XPoint));
         if (vs == NULL)
         {
            Warning ("ContinueAddPolygonPoint()", "Can not realloc()");
            return (FALSE);
         }
         PolygonPtr->vlist = vs;
         if (Index == 0 || Index == n-1)
         {
            if (before)
            {
               vs[n].x = vs[n-1].x;
               vs[n].y = vs[n-1].y;
               vs[n-1].x = x + ABS_SIZE(grid_x-orig_x);
               vs[n-1].y = y + ABS_SIZE(grid_y-orig_y);
            }
            else
            {
               for (i = n-1; i > 0; i--) vs[i+1] = vs[i];
               vs[1].x = x + ABS_SIZE(grid_x-orig_x);
               vs[1].y = y + ABS_SIZE(grid_y-orig_y);
            }
         }
         else
         {
            if (before)
            {
               for (i = n-1; i >= Index; i--) vs[i+1] = vs[i];
               vs[Index].x = x + ABS_SIZE(grid_x-orig_x);
               vs[Index].y = y + ABS_SIZE(grid_y-orig_y);
            }
            else
            {
               for (i = n-1; i > Index; i--) vs[i+1] = vs[i];
               vs[Index+1].x = x + ABS_SIZE(grid_x-orig_x);
               vs[Index+1].y = y + ABS_SIZE(grid_y-orig_y);
            }
         }
         if (curved && splineRubberband) { cfree (sv); cfree (pv); }
         PolygonPtr->n++;
         n++;
         AdjObjSplineVs (ObjPtr);
         UpdPolyBBox (ObjPtr, PolygonPtr->n, PolygonPtr->vlist);
         AdjObjBBox (ObjPtr);

         UpdSelBBox ();
         RedrawAreas (botObj,
               sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
               sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1),
               selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
               selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
         HighLightForward ();
         SetFileModified (TRUE);
         justDupped = FALSE;
      }
   }
   return (TRUE);
}

void AddPoint ()
{
   register struct ObjRec	* obj_ptr;
   struct PolyRec		* poly_ptr=NULL;
   struct PolygonRec		* polygon_ptr=NULL;
   int				index, adding=TRUE, pt_added=FALSE;
   int				root_x, root_y, old_x, old_y;
   unsigned int			status;
   Window			root_win, child_win;
   XEvent			input, ev;

   if (!(topSel != NULL && topSel == botSel &&
         (topSel->obj->type == OBJ_POLY || topSel->obj->type == OBJ_POLYGON)))
   {
      Msg ("Please select only one POLY or POLYGON object.");
      return;
   }
   if (topSel->obj->locked)
   {
      Msg ("Can not add points for locked object.");
      return;
   }
   if (curChoice == VERTEXMODE)
   {
      HighLightReverse ();
      JustRemoveAllVSel ();
      HighLightForward ();
   }
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   obj_ptr = topSel->obj;
   switch (obj_ptr->type)
   {
      case OBJ_POLY: poly_ptr = obj_ptr->detail.p; break;
      case OBJ_POLYGON: polygon_ptr = obj_ptr->detail.g; break;
   }
   TwoLineMsg ("Drag left mouse button to ADD points.",
         "Click other buttons to quit.");

   XGrabPointer (mainDisplay, drawWindow, False,
         PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
         GrabModeAsync, GrabModeAsync, None, defaultCursor, CurrentTime);
   XQueryPointer (mainDisplay, drawWindow, &root_win, &child_win,
         &root_x, &root_y, &old_x, &old_y, &status);
   XSetFont (mainDisplay, revDefaultGC, defaultFontPtr->fid);
   XDrawString (mainDisplay, drawWindow, revDefaultGC,
         old_x+4, old_y+defaultFontAsc, "ADD", 3);
   MarkRulers (old_x, old_y);

   while (adding)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonPress)
      {
         if (input.xbutton.button == Button1)
         {
            XDrawString (mainDisplay, drawWindow, revDefaultGC,
                  old_x+4, old_y+defaultFontAsc, "ADD", 3);
            if (obj_ptr->type == OBJ_POLY &&
                  PtInPolyMark (input.xbutton.x, input.xbutton.y,
                  poly_ptr->n, poly_ptr->vlist, &index))
            {
               if (ContinueAddPolyPoint (obj_ptr, input.xbutton.x,
                     input.xbutton.y, index, poly_ptr, &old_x, &old_y))
                  pt_added = TRUE;
            }
            else if (obj_ptr->type == OBJ_POLYGON &&
                  PtInPolyMark (input.xbutton.x, input.xbutton.y,
                  polygon_ptr->n-1, polygon_ptr->vlist, &index))
            {
               if (ContinueAddPolygonPoint (obj_ptr, input.xbutton.x,
                     input.xbutton.y, index, polygon_ptr, &old_x, &old_y))
                  pt_added = TRUE;
            }
            XDrawString (mainDisplay, drawWindow, revDefaultGC,
                  old_x+4, old_y+defaultFontAsc, "ADD", 3);
         }
         else
         {
            XUngrabPointer (mainDisplay, CurrentTime);
            Msg ("");
            adding = FALSE;
            XDrawString (mainDisplay, drawWindow, revDefaultGC,
                  old_x+4, old_y+defaultFontAsc, "ADD", 3);
         }
      }
      else if (input.type == MotionNotify)
      {
         XDrawString (mainDisplay, drawWindow, revDefaultGC,
               old_x+4, old_y+defaultFontAsc, "ADD", 3);
         old_x = input.xmotion.x;
         old_y = input.xmotion.y;
         XDrawString (mainDisplay, drawWindow, revDefaultGC,
               old_x+4, old_y+defaultFontAsc, "ADD", 3);
         MarkRulers (old_x, old_y);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   if (pt_added)
      RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   else
      AbortPrepareCmd (CMD_REPLACE);
}

void RestoreImageWH ()
{
   struct XBmRec	* xbm_ptr=NULL;
   struct XPmRec	* xpm_ptr=NULL;
   int			w, h, image_w=0, image_h=0, rotate=ROTATE0;
   int			ltx, lty, rbx, rby;

   if (topSel == NULL) return;
   if (topSel != botSel || (!(topSel->obj->type==OBJ_XBM ||
         topSel->obj->type==OBJ_XPM)))
   {
      Msg ("Please select one xbitmap or xpixmap object to restore.");
      return;
   }
   if (topSel->obj->locked)
   {
      Msg ("Can not restore locked object.");
      return;
   }
   w = topSel->obj->obbox.rbx - topSel->obj->obbox.ltx;
   h = topSel->obj->obbox.rby - topSel->obj->obbox.lty;
   switch (topSel->obj->type)
   {
      case OBJ_XBM:
         xbm_ptr = topSel->obj->detail.xbm;
         if (xbm_ptr->real_type==XBM_EPS && xbm_ptr->bitmap==None)
         {
            image_w = xbm_ptr->eps_w;
            image_h = xbm_ptr->eps_h;
         }
         else
         {
            image_w = xbm_ptr->image_w;
            image_h = xbm_ptr->image_h;
         }
         if (w == image_w && h == image_h) return;
         rotate = xbm_ptr->rotate;
         break;
      case OBJ_XPM:
         xpm_ptr = topSel->obj->detail.xpm;
         image_w = xpm_ptr->image_w; image_h = xpm_ptr->image_h;
         if (w == image_w && h == image_h) return;
         rotate = xpm_ptr->rotate;
         break;
   }
   ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
   HighLightReverse ();

   PrepareToReplaceAnObj (topSel->obj);
   switch (rotate)
   {
      case ROTATE0:
      case ROTATE180:
         topSel->obj->obbox.rbx = topSel->obj->obbox.ltx+image_w;
         topSel->obj->obbox.rby = topSel->obj->obbox.lty+image_h;
         break;

      case ROTATE90:
      case ROTATE270:
         topSel->obj->obbox.rbx = topSel->obj->obbox.ltx+image_h;
         topSel->obj->obbox.rby = topSel->obj->obbox.lty+image_w;
         break;
   }
   switch (topSel->obj->type)
   {
      case OBJ_XBM:
         if (xbm_ptr->cached_bitmap != None)
            XFreePixmap (mainDisplay, xbm_ptr->cached_bitmap);
         xbm_ptr->cached_bitmap = None;
         xbm_ptr->cached_zoom = 0;
         xbm_ptr->cached_rotate = INVALID;
         break;
      case OBJ_XPM:
         if (xpm_ptr->cached_pixmap != None)
            XFreePixmap (mainDisplay, xpm_ptr->cached_pixmap);
         xpm_ptr->cached_pixmap = None;
         xpm_ptr->cached_zoom = 0;
         xpm_ptr->cached_rotate = INVALID;
         break;
   }
   AdjObjBBox (topSel->obj);
   RecordReplaceAnObj (topSel->obj);

   UpdSelBBox ();
   RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
         rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}

void CutMaps ()
{
   if (topSel == NULL) return;
   if (topSel != botSel)
   {
      Msg ("Please select a xbitmap or xpixmap object to cut.");
      return;
   }
   if (topSel->obj->locked)
   {
      Msg ("Can not cut locked object.");
      return;
   }
   switch (topSel->obj->type)
   {
      case OBJ_XBM: CutXBitmap (); break;
      case OBJ_XPM: CutXPixmap (); break;
      default: Msg ("Please select a xbitmap or xpixmap object to cut."); break;
   }
}

void EditSubMenu (index)
   int	index;
{
   switch (index)
   {
      case EDIT_REDRAW: ClearAndRedrawDrawWindow (); break;
      case EDIT_DUP: DupSelObj (); break;
      case EDIT_DELETE: DelAllSelObj (); break;
      case EDIT_SELALL: SelAllObj (TRUE); break;
      case EDIT_UNDO: UndoCmd (); break;
      case EDIT_REDO: RedoCmd (); break;
      case EDIT_DEL_POINT: DeletePoint (); break;
      case EDIT_ADD_POINT: AddPoint (); break;
      case EDIT_COPY: CopyToCutBuffer (); break;
      case EDIT_PASTE: PasteFromCutBuffer (); break;
      case EDIT_UPDATE: UpdateSelObjs (); break;
      case EDIT_SCALE: ScaleAllSelObj (); break;
      case EDIT_FLUSH_UNDO:
         CleanUpCmds ();
         if (FlushColormap ())
            Msg ("Undo buffer and Colormap flushed.");
         else
            Msg ("Undo buffer flushed.");
         break;
      case EDIT_PRINT_MSG_BUF: PrintMsgBuffer (); break;
      case EDIT_INV_XBM: InvertXBitmaps (); break;
      case EDIT_PUSH_CUR_CHOICE: PushCurChoice (); break;
      case EDIT_MAKE_PREC_ARC: MakePreciseArc (); break;
      case EDIT_CUT_MAPS: CutMaps (); break;
      case EDIT_RESTORE_MAPS: RestoreImageWH (); break;
      case EDIT_UPDATE_EPS: UpdateEPS (); break;
      case EDIT_MAKE_REGULAR: MakeRegularPolygon (); break;
      case EDIT_BREAK_TEXT: BreakUpText (); break;
      case EDIT_SELECTION_PASTE: PasteFromSelection (); break;	/* noda */
   }
}

void EditMenu (X, Y)
   int	X, Y;
{
   int		index, * fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXEDITMENUS, &fore_colors, &valid, &init_rv);
   activeMenu = MENU_EDIT;
   index = TextMenuLoop (X, Y, editMenuStr, MAXEDITMENUS, fore_colors, valid,
         init_rv, SINGLECOLOR);

   if (index != INVALID) EditSubMenu (index);
}

void FrontProc ()
{
   if (topSel != NULL)
   {
      HighLightReverse ();
      MoveSelToTop ();
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
      HighLightForward ();
      SetFileModified (TRUE);
   }
}

void BackProc ()
{
   if (topSel != NULL)
   {
      HighLightReverse ();
      MoveSelToBot ();
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
      HighLightForward ();
      SetFileModified (TRUE);
   }
}

void AlignObjsTop ()
{
   register int saved_h_align = horiAlign, saved_v_align = vertAlign;

   horiAlign = ALIGN_N; vertAlign = ALIGN_T;
   AlignSelObjs ();
   horiAlign = saved_h_align; vertAlign = saved_v_align;
}

void AlignObjsMiddle ()
{
   register int saved_h_align = horiAlign, saved_v_align = vertAlign;

   horiAlign = ALIGN_N; vertAlign = ALIGN_M;
   AlignSelObjs ();
   horiAlign = saved_h_align; vertAlign = saved_v_align;
}

void AlignObjsBottom ()
{
   register int saved_h_align = horiAlign, saved_v_align = vertAlign;

   horiAlign = ALIGN_N; vertAlign = ALIGN_B;
   AlignSelObjs ();
   horiAlign = saved_h_align; vertAlign = saved_v_align;
}

void AlignObjsLeft ()
{
   register int saved_h_align = horiAlign, saved_v_align = vertAlign;

   horiAlign = ALIGN_L; vertAlign = ALIGN_N;
   AlignSelObjs ();
   horiAlign = saved_h_align; vertAlign = saved_v_align;
}

void AlignObjsCenter ()
{
   register int saved_h_align = horiAlign, saved_v_align = vertAlign;

   horiAlign = ALIGN_C; vertAlign = ALIGN_N;
   AlignSelObjs ();
   horiAlign = saved_h_align; vertAlign = saved_v_align;
}

void AlignObjsRight ()
{
   register int saved_h_align = horiAlign, saved_v_align = vertAlign;

   horiAlign = ALIGN_R; vertAlign = ALIGN_N;
   AlignSelObjs ();
   horiAlign = saved_h_align; vertAlign = saved_v_align;
}

static
void Abut (Dir)
   int	Dir;
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr;
   struct SelRec		* top_sort, * bot_sort, * sort_ptr;
   struct SelRec		* new_sort_ptr, * tmp_sel_ptr, * next_sort;
   struct ObjRec		* sorted_obj, * locked_obj=NULL;
   struct MoveSubCmdRec		* move_cmd;
   struct SubCmdRec		* sub_cmd;
   int				sel_ltx, sel_lty, sel_rbx, sel_rby, rbx, rby;
   int				found, delta, dx=0, dy=0;

   if (topSel == NULL) return;
   if (curChoice == VERTEXMODE)
   {
      Msg ("Can not abut in vertex mode.");
      return;
   }
   if (numObjLocked > 1)
   {
      Msg ("Can not abut.  Too many objects locked.");
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   sel_ltx = selLtX; sel_lty = selLtY;
   sel_rbx = selRbX; sel_rby = selRbY;

   top_sort = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   top_sort->next = top_sort->prev = NULL;

   top_sort->obj = sorted_obj = botSel->obj;
   if (botSel->obj->locked) locked_obj = botSel->obj;
   for (sel_ptr = botSel->prev; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->locked) locked_obj = obj_ptr;
      switch (Dir)
      {
         case ABUT_HORIZONTAL:
            if (obj_ptr->obbox.ltx < sorted_obj->obbox.ltx ||
                  (obj_ptr->obbox.ltx == sorted_obj->obbox.ltx &&
                  obj_ptr->obbox.lty < sorted_obj->obbox.lty))
               top_sort->obj = sorted_obj = sel_ptr->obj;
            break;
         case ABUT_VERTICAL:
            if (obj_ptr->obbox.lty < sorted_obj->obbox.lty ||
                  (obj_ptr->obbox.lty == sorted_obj->obbox.lty &&
                  obj_ptr->obbox.ltx < sorted_obj->obbox.ltx))
               top_sort->obj = sorted_obj = sel_ptr->obj;
            break;
      }
   }
   bot_sort = top_sort;

   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      if (sel_ptr->obj == top_sort->obj) continue;

      obj_ptr = sel_ptr->obj;
      new_sort_ptr = (struct SelRec *) calloc (1, sizeof(struct SelRec));
      new_sort_ptr->obj = obj_ptr;
      found = FALSE;
      for (sort_ptr=top_sort->next; sort_ptr!=NULL; sort_ptr=sort_ptr->next)
      {
         switch (Dir)
         {
            case ABUT_HORIZONTAL:
               if (sort_ptr->obj->obbox.ltx > obj_ptr->obbox.ltx ||
                     (sort_ptr->obj->obbox.ltx == obj_ptr->obbox.ltx &&
                     sort_ptr->obj->obbox.lty > obj_ptr->obbox.lty))
                  found = TRUE;
               break;
            case ABUT_VERTICAL:
               if (sort_ptr->obj->obbox.lty > obj_ptr->obbox.lty ||
                     (sort_ptr->obj->obbox.lty == obj_ptr->obbox.lty &&
                     sort_ptr->obj->obbox.ltx > obj_ptr->obbox.ltx))
                  found = TRUE;
               break;
         }
         if (found) break;
      }
      new_sort_ptr->next = sort_ptr;
      if (sort_ptr == NULL)
      {
         new_sort_ptr->prev = bot_sort;
         bot_sort->next = new_sort_ptr;
         bot_sort = new_sort_ptr;
      }
      else
      {
         new_sort_ptr->prev = sort_ptr->prev;
         sort_ptr->prev->next = new_sort_ptr;
         sort_ptr->prev = new_sort_ptr;
      }
   }

   tmp_sel_ptr = (struct SelRec *) calloc (1,sizeof(struct SelRec));
   tmp_sel_ptr->next = tmp_sel_ptr->prev = NULL;

   if (locked_obj != NULL)
   {
      switch (Dir)
      {
         case ABUT_HORIZONTAL:
            rbx = top_sort->obj->obbox.rbx;
            for (sort_ptr=top_sort; sort_ptr->obj!=locked_obj &&
                  sort_ptr->next!=NULL; sort_ptr=next_sort)
            {
               next_sort = sort_ptr->next;
               delta = rbx-next_sort->obj->obbox.ltx;
               dx = (-delta);
               rbx = next_sort->obj->obbox.rbx+delta;
            }
            break;
         case ABUT_VERTICAL:
            rby = top_sort->obj->obbox.rby;
            for (sort_ptr=top_sort; sort_ptr->obj!=locked_obj &&
                  sort_ptr->next!=NULL; sort_ptr=next_sort)
            {
               next_sort = sort_ptr->next;
               delta = rby-next_sort->obj->obbox.lty;
               dy = (-delta);
               rby = next_sort->obj->obbox.rby+delta;
            }
            break;
      }
   }

   move_cmd = (struct MoveSubCmdRec *) calloc (1,sizeof(struct MoveSubCmdRec));
   sub_cmd = (struct SubCmdRec *) calloc (1,sizeof(struct SubCmdRec));
   sub_cmd->detail.mv = move_cmd;

   rbx = top_sort->obj->obbox.rbx;
   rby = top_sort->obj->obbox.rby;
   found = (locked_obj == NULL);
   if (!found && locked_obj != top_sort->obj)
   {
      tmp_sel_ptr->obj = top_sort->obj;
      switch (Dir)
      {
         case ABUT_HORIZONTAL:
            move_cmd->dx = dx;
            move_cmd->dy = 0;
            PrepareToRecord (CMD_MOVE, NULL, NULL, 0);
            RecordCmd (CMD_MOVE, sub_cmd, tmp_sel_ptr, tmp_sel_ptr, 1);
            MoveObj (top_sort->obj, dx, 0);
            break;
         case ABUT_VERTICAL:
            move_cmd->dx = 0;
            move_cmd->dy = dy;
            PrepareToRecord (CMD_MOVE, NULL, NULL, 0);
            RecordCmd (CMD_MOVE, sub_cmd, tmp_sel_ptr, tmp_sel_ptr, 1);
            MoveObj (top_sort->obj, 0, dy);
            break;
      }
   }
   for (sort_ptr=top_sort; sort_ptr->next!=NULL; sort_ptr=next_sort)
   {
      next_sort = sort_ptr->next;
      tmp_sel_ptr->obj = next_sort->obj;
      switch (Dir)
      {
         case ABUT_HORIZONTAL:
            delta = rbx-next_sort->obj->obbox.ltx;
            rbx = next_sort->obj->obbox.rbx+delta;
            if (!found) delta += dx;
            move_cmd->dx = delta;
            move_cmd->dy = 0;
            PrepareToRecord (CMD_MOVE, NULL, NULL, 0);
            RecordCmd (CMD_MOVE, sub_cmd, tmp_sel_ptr, tmp_sel_ptr, 1);
            MoveObj (next_sort->obj, delta, 0);
            break;
         case ABUT_VERTICAL:
            delta = rby-next_sort->obj->obbox.lty;
            rby = next_sort->obj->obbox.rby+delta;
            if (!found) delta += dy;
            move_cmd->dx = 0;
            move_cmd->dy = delta;
            PrepareToRecord (CMD_MOVE, NULL, NULL, 0);
            RecordCmd (CMD_MOVE, sub_cmd, tmp_sel_ptr, tmp_sel_ptr, 1);
            MoveObj (next_sort->obj, 0, delta);
            break;
      }
      cfree (sort_ptr);
   }
   EndCompositeCmd ();
   cfree (sort_ptr);
   cfree (move_cmd);
   cfree (sub_cmd);
   cfree (tmp_sel_ptr);

   UpdSelBBox ();
   RedrawAreas (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1),
         sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
         sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1));
   HighLightForward ();
   justDupped = FALSE;
   switch (Dir)
   {
      case ABUT_HORIZONTAL: Msg ("Objects are abutted horizontally."); break;
      case ABUT_VERTICAL: Msg ("Objects are abutted vertically."); break;
   }
   SetFileModified (TRUE);
}

void AbutHorizontal ()
{
   Abut (ABUT_HORIZONTAL);
}

void AbutVertical ()
{
   Abut (ABUT_VERTICAL);
}

char * arrangeMenuStr[] =
      { "Front           ^f",
        "Back            ^b",
        "Group           ^g",
        "UnGroup         ^u",
        "Lock            #<",
        "UnLock          #>",
        "AlignObjs       ^l",
        "AlignToGrid     ^t",
        "AlignToPage     #&",
        "DistributeObjs  #l",
        "FlipHorizontal  #h",
        "FlipVertical    #v",
        "RotateClockWise #w",
        "RotateCounter   #c",
        "AlignObjsTop    #{",
        "AlignObjsMiddle #+",
        "AlignObjsBottom #}",
        "AlignObjsLeft   #[",
        "AlignObjsCenter #=",
        "AlignObjsRight  #]",
        "AbutHorizontal  #_",
        "AbutVertical    #|",
        "CenterAnEndPoint  "
      };

void ArrangeSubMenu (index)
   int	index;
{
   switch (index)
   {
      case ARRANGE_FRONT: FrontProc (); break;
      case ARRANGE_BACK: BackProc (); break;
      case ARRANGE_GROUP: GroupSelObj (); break;
      case ARRANGE_UNGROUP: UngroupSelObj (); break;
      case ARRANGE_LOCK: LockSelObj (); break;
      case ARRANGE_UNLOCK: UnlockSelObj (); break;
      case ARRANGE_ALIGNOBJ: AlignSelObjs (); break;
      case ARRANGE_ALIGNGRID: AlignSelToGrid (); break;
      case ARRANGE_ALIGNPAGE: AlignSelToPage (); break;
      case ARRANGE_DISTROBJ: DistrSelObjs (); break;
      case FLIP_HORIZONTAL: FlipHorizontal (); break;
      case FLIP_VERTICAL: FlipVertical (); break;
      case ROTATE_CLOCKWISE: RotateClockWise (); break;
      case ROTATE_COUNTER: RotateCounter (); break;
      case ALIGN_OBJ_TOP: AlignObjsTop (); break;
      case ALIGN_OBJ_MIDDLE: AlignObjsMiddle (); break;
      case ALIGN_OBJ_BOTTOM: AlignObjsBottom (); break;
      case ALIGN_OBJ_LEFT: AlignObjsLeft (); break;
      case ALIGN_OBJ_CENTER: AlignObjsCenter (); break;
      case ALIGN_OBJ_RIGHT: AlignObjsRight (); break;
      case ABUT_HORIZONTAL: AbutHorizontal (); break;
      case ABUT_VERTICAL: AbutVertical (); break;
      case ALIGN_CENTERANENDPOINT: CenterAnEndPoint (); break;
   }
}

void ArrangeMenu (X, Y)
   int	X, Y;
{
   int		index, * fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXARRANGEMENUS, &fore_colors, &valid, &init_rv);
   activeMenu = MENU_ARRANGE;
   index = TextMenuLoop (X, Y, arrangeMenuStr, MAXARRANGEMENUS, fore_colors,
         valid, init_rv, SINGLECOLOR);

   if (index != INVALID) ArrangeSubMenu (index);
}

void UpdateSymbols ()
{
   int			dx = 0, dy = 0, changed=FALSE;
   int			sel_ltx, sel_lty, sel_rbx, sel_rby;
   char			path_name[MAXPATHLENGTH], sym_name[MAXPATHLENGTH];
   struct ObjRec	* obj_ptr, * new_obj_ptr;
   struct SelRec	* sel_ptr;
   struct GroupRec	* icon_ptr;

   if (topSel == NULL) return;

   tmpTopObj = tmpBotObj = NULL;
   tmpTopSel = tmpBotSel = NULL;

   sel_ltx = selLtX; sel_lty = selLtY;
   sel_rbx = selRbX; sel_rby = selRbY;

   HighLightReverse ();

   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->type!=OBJ_ICON || obj_ptr->locked) continue;

      icon_ptr = obj_ptr->detail.r;
      strcpy (sym_name, icon_ptr->s);
      if (GetSymbolPath (icon_ptr->s, path_name))
      {
         if ((new_obj_ptr = GetObjRepresentation (path_name, sym_name)) != NULL)
         {
            PrepareToReplaceAnObj (obj_ptr);
            if (icon_ptr->rotate != ROTATE0 || icon_ptr->flip != NO_FLIP)
            {
               if (icon_ptr->flip & HORI_EVEN) FlipIconHorizontal (new_obj_ptr);
               if (icon_ptr->flip & VERT_EVEN) FlipIconVertical (new_obj_ptr);
               if (icon_ptr->rotate == ROTATE0)
               {
                  if (icon_ptr->flip & (HORI_ODD | VERT_ODD))
                  {
                     RotateIconClockWise (new_obj_ptr);
                     if (icon_ptr->flip & HORI_ODD)
                        FlipIconHorizontal (new_obj_ptr);
                     if (icon_ptr->flip & VERT_ODD)
                        FlipIconVertical (new_obj_ptr);
                     RotateIconCounter (new_obj_ptr);
                  }
               }
               else
               {
                  switch (icon_ptr->rotate)
                  {
                     case ROTATE90:
                        RotateIconClockWise (new_obj_ptr);
                        if (icon_ptr->flip & HORI_ODD)
                           FlipIconHorizontal (new_obj_ptr);
                        if (icon_ptr->flip & VERT_ODD)
                           FlipIconVertical (new_obj_ptr);
                        break;
                     case ROTATE180:
                        RotateIconClockWise (new_obj_ptr);
                        if (icon_ptr->flip & HORI_ODD)
                           FlipIconHorizontal (new_obj_ptr);
                        if (icon_ptr->flip & VERT_ODD)
                           FlipIconVertical (new_obj_ptr);
                        RotateIconClockWise (new_obj_ptr);
                        break;
                     case ROTATE270:
                        RotateIconClockWise (new_obj_ptr);
                        if (icon_ptr->flip & HORI_ODD)
                           FlipIconHorizontal (new_obj_ptr);
                        if (icon_ptr->flip & VERT_ODD)
                           FlipIconVertical (new_obj_ptr);
                        RotateIconClockWise (new_obj_ptr);
                        RotateIconClockWise (new_obj_ptr);
                        break;
                  }
               }
            }
            switch (horiAlign)
            {
               case ALIGN_L:
                  dx = obj_ptr->obbox.ltx - new_obj_ptr->obbox.ltx;
                  break;
               case ALIGN_N:
               case ALIGN_S:
               case ALIGN_C:
                  dx = (int)(((obj_ptr->obbox.ltx+obj_ptr->obbox.rbx) -
                        (new_obj_ptr->obbox.ltx+new_obj_ptr->obbox.rbx))/2);
                  break;
               case ALIGN_R:
                  dx = obj_ptr->obbox.rbx - new_obj_ptr->obbox.rbx;
                  break;
            }
            switch (vertAlign)
            {
               case ALIGN_T:
                  dy = obj_ptr->obbox.lty - new_obj_ptr->obbox.lty;
                  break;
               case ALIGN_N:
               case ALIGN_S:
               case ALIGN_M:
                  dy = (int)(((obj_ptr->obbox.lty+obj_ptr->obbox.rby) -
                        (new_obj_ptr->obbox.lty+new_obj_ptr->obbox.rby))/2);
                  break;
               case ALIGN_B:
                  dy = obj_ptr->obbox.rby - new_obj_ptr->obbox.rby;
                  break;
            }
            MoveObj (new_obj_ptr, dx, dy);

            changed = TRUE;

            UnlinkObj (obj_ptr);
            CopyAndUpdateAttrs (new_obj_ptr, obj_ptr);

            if (new_obj_ptr->bbox.ltx < selLtX) selLtX = new_obj_ptr->bbox.ltx;
            if (new_obj_ptr->bbox.lty < selLtY) selLtY = new_obj_ptr->bbox.lty;
            if (new_obj_ptr->bbox.rbx < selRbX) selRbX = new_obj_ptr->bbox.rbx;
            if (new_obj_ptr->bbox.rby < selRbY) selRbY = new_obj_ptr->bbox.rby;
            if (new_obj_ptr->obbox.ltx < selObjLtX)
                  selObjLtX = new_obj_ptr->obbox.ltx;
            if (new_obj_ptr->obbox.lty < selObjLtY)
                  selObjLtY = new_obj_ptr->obbox.lty;
            if (new_obj_ptr->obbox.rbx < selObjRbX)
                  selObjRbX = new_obj_ptr->obbox.rbx;
            if (new_obj_ptr->obbox.rby < selObjRbY)
                  selObjRbY = new_obj_ptr->obbox.rby;

            sel_ptr->obj = new_obj_ptr;
            AddObj (NULL, topObj, new_obj_ptr);
            RecordReplaceAnObj (new_obj_ptr);
            FreeObj (obj_ptr);
         }
      }
   }
   EndCompositeCmd ();

   if (changed)
   {
      UpdSelBBox ();
      RedrawAreas (botObj, sel_ltx-GRID_ABS_SIZE(1), sel_lty-GRID_ABS_SIZE(1),
            sel_rbx+GRID_ABS_SIZE(1), sel_rby+GRID_ABS_SIZE(1),
            selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
      justDupped = FALSE;
      Msg ("Selected icons are brought up to date.");
   }
   HighLightForward ();
}
