/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/font.c,v 2.61.1.2 1993/06/03 07:14:01 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "auxtext.e"
#include "choice.e"
#include "color.e"
#include "cursor.e"
#include "drawing.e"
#include "font.e"
#include "mainmenu.e"
#include "mark.e"
#include "menu.e"
#include "obj.e"
#include "pattern.e"
#include "prtgif.e"
#include "raster.e"
#include "select.e"
#include "setup.e"
#include "text.e"

#define FONTS_PER_DPI (((MAXFONTS-1)*MAXFONTSTYLES+1)*MAXFONTSIZES)
#define FONTTABLESIZE (MAXFONTDPIS*FONTS_PER_DPI)

#define KANJI_FONTS_PER_DPI ((MAX_KANJI_FONTS - 1)*MAXFONTSIZES)
#define KANJI_FONTTABLESIZE (MAXFONTDPIS*KANJI_FONTS_PER_DPI)

#define COUR8R_75  FontIndex(FONT_DPI_75,FONT_COU,0,STYLE_NR)
#define COUR10R_75 FontIndex(FONT_DPI_75,FONT_COU,1,STYLE_NR)
#define COUR12R_75 FontIndex(FONT_DPI_75,FONT_COU,2,STYLE_NR)
#define COUR14R_75 FontIndex(FONT_DPI_75,FONT_COU,3,STYLE_NR)
#define COUR18R_75 FontIndex(FONT_DPI_75,FONT_COU,4,STYLE_NR)
#define COUR24R_75 FontIndex(FONT_DPI_75,FONT_COU,5,STYLE_NR)

#define COUR11R_100 FontIndex(FONT_DPI_100,FONT_COU,0,STYLE_NR)
#define COUR14R_100 FontIndex(FONT_DPI_100,FONT_COU,1,STYLE_NR)
#define COUR17R_100 FontIndex(FONT_DPI_100,FONT_COU,2,STYLE_NR)
#define COUR20R_100 FontIndex(FONT_DPI_100,FONT_COU,3,STYLE_NR)
#define COUR25R_100 FontIndex(FONT_DPI_100,FONT_COU,4,STYLE_NR)
#define COUR34R_100 FontIndex(FONT_DPI_100,FONT_COU,5,STYLE_NR)

#define IsRyumin(font)	((font == KANJI_FONT_RYUMIN || font == KANJI_FONT_RYUMIN_V) ? TRUE : FALSE)
#define IsGothic(font)	((font == KANJI_FONT_GOTHIC || font == KANJI_FONT_GOTHIC_V) ? TRUE : FALSE)
#define IsVert(font)	((font == KANJI_FONT_RYUMIN_V || font == KANJI_FONT_GOTHIC_V) ? TRUE : FALSE)

#define ROT_NRM	0
#define ROT_MOV	1
#define ROT_NON	2
#define ROT_INV	3

static void	InitRotTbl();
static char	RotTbl21XX[256];

struct MyFontRec {
   XFontStruct	* xfs;
   int		valid;
   int		checked;
   Bool		bold;
};

XFontStruct	* canvasFontPtr;
int	canvasFontHeight;
int	canvasFontAsc;
int	canvasFontDes;

XFontStruct	* rulerFontPtr;
KanjiFontStruct	* canvasKanjiFontPtr;
int	rulerFontWidth;
int	rulerFontHeight;
int	rulerFontAsc;
int	rulerFontDes;

XFontStruct	* defaultFontPtr;
int	defaultFontWidth;
int	defaultFontHeight;
int	defaultFontAsc;
int	defaultFontDes;

char	* defBaseKanjiFontName = "*--14-*-jisx0208.1983-0";
char	* baseRyuminFontName = NULL;
char	* baseGothicFontName = NULL;
XFontStruct	* baseRyuminFontPtr = NULL;
XFontStruct	* baseGothicFontPtr = NULL;
int	drawScaledKanji = TRUE;

static int	textBitmapWidth = INVALID;
static int	textBitmapHeight = INVALID;
static Pixmap	textBitmap;
static char	* scaleBuf = NULL;
static GC	baseGC = NULL;

int	* pointSize;
int	curFont = FONT_COU;
int	curKanjiFont = KANJI_FONT_NONE;
int	curSize = 4;
int	curStyle = STYLE_NR;
int	curFontDPI = FONT_DPI_75;
int	curRotate = ROTATE0;

int	pointSize75[] = { 8, 10, 12, 14, 18, 24 };
int	pointSize100[] = { 11, 14, 17, 20, 25, 34 };

/*
 * /8  ->     1
 * /4  ->     2   -   -   3
 * /2  ->     4   5   -   6   7   -   9
 * x1  ->     8  10  11  12  14  17  18  20  24  25  34
 * x2  ->    16   -  22   -  28   -  36  40  48  50  68
 * x4  ->    32   -  44   -  56   -  72  80  96 100 136
 * x8  ->    64   -  88   - 112   - 144 ...
 * x16 ->   128
 */

char	* fontMenuStr[] =
      {
         "Times",
	 "Courier",
	 "Helvetica",
	 "NewCentury",
	 "Symbol",
	 "",
	 "Ryumin",
	 "Gothic",
	 "Ryumin-V",
	 "Gothic-V"
      };
char	* sizeMenuStr[] =
      {
         "8 ",
         "10",
         "11",
         "12",
         "14",
         "17",
         "18",
         "20",
         "24",
         "25",
         "34"
      };
char	* styleMenuStr[] =
      {
         "Roman             ^#o",
         "Bold              ^#b",
         "Italic            ^#t",
         "BoldItalic        ^#p",
         "---------------------",
         "Left              ^#l",
         "Center            ^#c",
         "Right             ^#r",
	 "---------------------",
	 "Scale/Rotate Japanese"
      };

static struct MyFontRec	myFontInfo[FONTTABLESIZE];
static struct MyFontRec myKanjiFontInfo[KANJI_FONTTABLESIZE];
static char	* fontDPIMenuStr[] = { "75dpi", "100dpi" };

static char	* fontNameStr[] =
{
/* 75 dpi fonts */
  "-*-times-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--24-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--24-*-*-*-*-*-iso8859-1",

  "-*-courier-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--24-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--24-*-*-*-*-*-iso8859-1",

  "-*-helvetica-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--24-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--8-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--10-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--12-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--18-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--24-*-*-*-*-*-iso8859-1",

  "-*-new century schoolbook-medium-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--24-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--24-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--8-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--10-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--12-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--18-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--24-*-*-*-*-*-iso8859-1",

  "-*-symbol-medium-r-normal--8-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--10-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--12-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--14-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--18-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--24-*-*-*-*-*-*-*",

/* 100 dpi fonts */
  "-*-times-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-medium-i-normal--34-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-times-bold-i-normal--34-*-*-*-*-*-iso8859-1",

  "-*-courier-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-medium-o-normal--34-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-courier-bold-o-normal--34-*-*-*-*-*-iso8859-1",

  "-*-helvetica-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-medium-o-normal--34-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--11-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--14-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--17-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--20-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--25-*-*-*-*-*-iso8859-1",
  "-*-helvetica-bold-o-normal--34-*-*-*-*-*-iso8859-1",

  "-*-new century schoolbook-medium-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-r-normal--34-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-medium-i-normal--34-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--11-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--14-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--17-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--20-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--25-*-*-*-*-*-iso8859-1",
  "-*-new century schoolbook-bold-i-normal--34-*-*-*-*-*-iso8859-1",

  "-*-symbol-medium-r-normal--11-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--14-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--17-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--20-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--25-*-*-*-*-*-*-*",
  "-*-symbol-medium-r-normal--34-*-*-*-*-*-*-*"
};

static char	* kanjiFontNameStr[KANJI_FONTTABLESIZE];
static char	* kanjiFontRes[] =
{
  /* 75 dpi kanji Ryumin fonts */
  "KanjiRyumin-8-75",
  "KanjiRyumin-10-75",
  "KanjiRyumin-12-75",
  "KanjiRyumin-14-75",
  "KanjiRyumin-18-75",
  "KanjiRyumin-24-75",

  /* 75 dpi kanji Gothic fonts */
  "KanjiGothic-8-75",
  "KanjiGothic-10-75",
  "KanjiGothic-12-75",
  "KanjiGothic-14-75",
  "KanjiGothic-18-75",
  "KanjiGothic-24-75",

  /* 75 dpi kanji Ryumin Vertical fonts */
  "KanjiRyumin-8-75",
  "KanjiRyumin-10-75",
  "KanjiRyumin-12-75",
  "KanjiRyumin-14-75",
  "KanjiRyumin-18-75",
  "KanjiRyumin-24-75",

  /* 75 dpi kanji Gothic Vertical fonts */
  "KanjiGothic-8-75",
  "KanjiGothic-10-75",
  "KanjiGothic-12-75",
  "KanjiGothic-14-75",
  "KanjiGothic-18-75",
  "KanjiGothic-24-75",

  /* 100 dpi kanji Ryumin fonts */
  "KanjiRyumin-11-100",
  "KanjiRyumin-14-100",
  "KanjiRyumin-17-100",
  "KanjiRyumin-20-100",
  "KanjiRyumin-25-100",
  "KanjiRyumin-34-100",

  /* 100 dpi kanji Gothic fonts */
  "KanjiGothic-11-100",
  "KanjiGothic-14-100",
  "KanjiGothic-17-100",
  "KanjiGothic-20-100",
  "KanjiGothic-25-100",
  "KanjiGothic-34-100",

  /* 100 dpi kanji Ryumin Vertical fonts */
  "KanjiRyumin-11-100",
  "KanjiRyumin-14-100",
  "KanjiRyumin-17-100",
  "KanjiRyumin-20-100",
  "KanjiRyumin-25-100",
  "KanjiRyumin-34-100",

  /* 100 dpi kanji Gothic Vertical fonts */
  "KanjiGothic-11-100",
  "KanjiGothic-14-100",
  "KanjiGothic-17-100",
  "KanjiGothic-20-100",
  "KanjiGothic-25-100",
  "KanjiGothic-34-100"
};

static char	* charCodeToName[] =
{
/* \200 */ "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
/* \220 */ "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
/* \240 */ "x", "", "", "", "\\250", "", "x", "",
           "x", "x", "\\343", "", "x", "x", "x", "x",
/* \260 */ "x", "x", "x", "x", "\\302", "x", "", "\\264",
           "\\313", "x", "\\353", "", "x", "x", "x", "",
/* \300 */ "8#260 /Agrave", "8#265 /Aacute", "8#276 /Acircumflex",
           "8#300 /Atilde", "8#311 /Adieresis", "8#314 /Aring", "\\341",
           "8#321 /Ccedilla", "8#322 /Egrave", "8#323 /Eacute",
           "8#324 /Ecircumflex", "8#325 /Edieresis", "8#326 /Igrave",
           "8#327 /Iacute", "8#330 /Icircumflex", "8#331 /Idieresis",
/* \320 */ "x", "8#332 /Ntilde", "8#333 /Ograve", "8#334 /Oacute",
           "8#335 /Ocircumflex", "8#336 /Otilde", "8#337 /Odieresis", "x",
           "\\351", "8#340 /Ugrave", "8#342 /Uacute", "8#344 /Ucircumflex",
           "8#345 /Udieresis", "x", "x", "\\373",
/* \340 */ "8#346 /agrave", "8#347 /aacute", "8#354 /acircumflex",
           "8#355 /atilde", "8#356 /adieresis", "8#357 /aring", "\\361",
           "8#360 /ccedilla", "8#362 /egrave", "8#363 /eacute",
           "8#364 /ecircumflex", "8#366 /edieresis", "8#367 /igrave",
           "8#374 /iacute", "8#375 /icircumflex", "8#376 /idieresis",
/* \360 */ "x", "8#254 /ntilde", "8#255 /ograve", "8#256 /oacute",
           "8#257 /ocircumflex", "8#271 /otilde", "8#274 /odieresis", "x",
           "\\371", "8#275 /ugrave", "8#350 /uacute", "8#352 /ucircumflex",
           "8#370 /udieresis", "x", "x", "8#372 /ydieresis"
};

XFontStruct * GetFontStruct (font_index)
   register int	font_index;
{
   return (myFontInfo[font_index].xfs);
}

int FontIndex (dpi_index, font_index, size_index, style_index)
   register int	dpi_index, font_index, size_index, style_index;
{
   if (font_index == FONT_SYM)
      return (size_index+MAXFONTSIZES*(MAXFONTSTYLES*FONT_SYM) +
            dpi_index*FONTS_PER_DPI);
   else
      return (size_index+MAXFONTSIZES*(style_index+MAXFONTSTYLES*font_index) +
            dpi_index*FONTS_PER_DPI);
}

int KanjiFontIndex (dpi_index, kanji_font_index, size_index, style_index)
   register int	dpi_index, kanji_font_index, size_index, style_index;
{
   if(kanji_font_index == KANJI_FONT_NONE) {
      return -1;
   }
   else
      return (size_index+MAXFONTSIZES*(kanji_font_index - 1) +
            dpi_index*KANJI_FONTS_PER_DPI);
}

int ValidCharCode (c_ptr)
   char	* c_ptr;
{
   register int	index = (int)(*c_ptr - (char)'\200');
   char		msg[MAXSTRING];

   if (*charCodeToName[index] == '\0' || *charCodeToName[index] == '8' ||
         *charCodeToName[index] == '\\')
      return (TRUE);
   else
   {
      sprintf (msg, "Unrecognized character code \\%o.  Character discarded!",
         (*c_ptr)&0xff);
      Msg (msg);
      return (FALSE);
   }
}

char * CharCodeTranslate (c_ptr)
   char	* c_ptr;
{
   register int	index = (int)(*c_ptr - (char)'\200');

   if (*charCodeToName[index] == '\\' || *charCodeToName[index] == '8')
      return (charCodeToName[index]);
   else
      return (NULL);
}

static
void RemoveIllegalChars (TextPtr)
   struct TextRec	* TextPtr;
{
   register char	* c_ptr, * dest_c_ptr;
   struct StrRec	* str_ptr;

   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      for (c_ptr = dest_c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
         if (!(((*c_ptr) & 0x80) && !ValidCharCode (c_ptr)))
            *dest_c_ptr++ = *c_ptr;
      *dest_c_ptr = '\0';
   }
}

static
void CheckLength (TextPtr)
   struct TextRec     * TextPtr;
{
   register char      * c_ptr, * dest_c_ptr;
   struct StrRec      * str_ptr;
   register           len;

   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      len = 0;
      for (c_ptr = dest_c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
      if (*c_ptr & 0x80) {
         len ++;
         if ((*(c_ptr + 1) & 0x80) || (len & 1) == 0)
            *dest_c_ptr++ = *c_ptr;
      }
      else {
         len = 0;
         *dest_c_ptr++ = *c_ptr;
      }
      *dest_c_ptr = '\0';
   }
}

static int	encodeFont[MAXFONTS*MAXFONTSTYLES];
static short	* encodeCharFlags[MAXFONTS*MAXFONTSTYLES];
static int	encodeCharFlagsAllocated = FALSE;

static
void PrepareText (TextPtr)
   register struct TextRec	* TextPtr;
{
   register char	* c_ptr;
   struct StrRec	* str_ptr;
   int			index, byte_index;
   short		* flag_ptr;

   if (TextPtr->font == FONT_SYM || TextPtr->kanji_font != KANJI_FONT_NONE) return;

   index = TextPtr->font*MAXFONTSTYLES + TextPtr->style;
   for (str_ptr = TextPtr->last; str_ptr != NULL; str_ptr = str_ptr->prev)
   {
      for (c_ptr = str_ptr->s; *c_ptr != '\0'; c_ptr++)
         if (((*c_ptr)&0x80) && *charCodeToName[(int)(*c_ptr-(char)'\200')]=='8')
         {
            encodeFont[index] = TRUE;
            flag_ptr = encodeCharFlags[index];
            byte_index = (int)(((*c_ptr) & 0x7f) / 8);
            flag_ptr[byte_index] |= (1<<(((*c_ptr) & 0x7f) % 8));
         }
   }
}

static
void PrepareObjFontInfo (ObjPtr)
   register struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;

   for (obj_ptr = ObjPtr; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT: PrepareText (obj_ptr->detail.t); break;

         case OBJ_GROUP:
         case OBJ_SYM: PrepareObjFontInfo (obj_ptr->detail.r->last); break;
      }
      for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
         PrepareText (attr_ptr->obj->detail.t);
   }
}

void PrepareEightBitFontInfo ()
{
   register struct ObjRec	* obj_ptr;
   register struct AttrRec	* attr_ptr;
   register int			j, i;
   short			* flag_ptr;

   if (!encodeCharFlagsAllocated)
   {
      for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++)
         encodeCharFlags[i] = (short *) calloc (16, sizeof(short));
      encodeCharFlagsAllocated = TRUE;
   }

   for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++)
   {
      encodeFont[i] = FALSE;
      flag_ptr = encodeCharFlags[i];
      for (j = 0; j < 16; j++) flag_ptr[j] = 0;
   }

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT: PrepareText (obj_ptr->detail.t); break;

         case OBJ_GROUP:
         case OBJ_SYM: PrepareObjFontInfo (obj_ptr->detail.r->last); break;
      }
      for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
         PrepareText (attr_ptr->obj->detail.t);
   }
}

int NeedEncode (FontIndex, KanjiFontIndex, StyleIndex)
   int	FontIndex, StyleIndex;
{
   if (FontIndex == FONT_SYM || KanjiFontIndex != KANJI_FONT_NONE)
      return (FALSE);
   else
      return (encodeFont[FontIndex*MAXFONTSTYLES+StyleIndex]);
}

void GetPSFontStr (FontIndex, StyleIndex, FontStr)
   int	FontIndex, StyleIndex;
   char	* FontStr;
{
   if (FontIndex == FONT_SYM)
   {
      strcpy (FontStr, "/Symbol");
      return;
   }
   switch (FontIndex)
   {
      case FONT_TIM: strcpy (FontStr, "/Times"); break;
      case FONT_COU: strcpy (FontStr, "/Courier"); break;
      case FONT_HEL: strcpy (FontStr, "/Helvetica"); break;
      case FONT_CEN: strcpy (FontStr, "/NewCenturySchlbk"); break;
   }
   switch (StyleIndex)
   {
      case STYLE_BI:
         switch (FontIndex)
         {
            case FONT_TIM: strcat (FontStr, "-BoldItalic"); break;
            case FONT_COU: strcat (FontStr, "-BoldOblique"); break;
            case FONT_HEL: strcat (FontStr, "-BoldOblique"); break;
            case FONT_CEN: strcat (FontStr, "-BoldItalic"); break;
         }
         break;
      case STYLE_BR: strcat (FontStr, "-Bold"); break;
      case STYLE_NI:
         switch (FontIndex)
         {
            case FONT_TIM: strcat (FontStr, "-Italic"); break;
            case FONT_COU: strcat (FontStr, "-Oblique"); break;
            case FONT_HEL: strcat (FontStr, "-Oblique"); break;
            case FONT_CEN: strcat (FontStr, "-Italic"); break;
         }
         break;
      case STYLE_NR:
         switch (FontIndex)
         {
            case FONT_TIM: strcat (FontStr, "-Roman"); break;
            case FONT_COU: break;
            case FONT_HEL: break;
            case FONT_CEN: strcat (FontStr, "-Roman"); break;
         }
         break;
   }
}

static char   eucFontTable[MAXFONTS * (MAX_KANJI_FONTS - 1) * MAXFONTSTYLES];

void PrepareEUCFontTable()
{
   int        i;

   for(i = 0; i < MAXFONTS * (MAX_KANJI_FONTS - 1) * MAXFONTSTYLES; i ++)
      eucFontTable[i] = FALSE;
}

void DefineEUCFont(FontIndex, KanjiFontIndex, StyleIndex)
   int        FontIndex, KanjiFontIndex, StyleIndex;
{
   eucFontTable[FontIndex * (MAX_KANJI_FONTS - 1) * MAXFONTSTYLES +
      (KanjiFontIndex - 1) *  MAXFONTSTYLES + StyleIndex] = TRUE;
}

int PreDefineEUCFont(FontIndex, KanjiFontIndex, StyleIndex)
   int        FontIndex, KanjiFontIndex, StyleIndex;
{
   return eucFontTable[FontIndex * (MAX_KANJI_FONTS - 1) * MAXFONTSTYLES +
      (KanjiFontIndex - 1) *  MAXFONTSTYLES + StyleIndex];
}

void GetPSKanjiFontStr (KanjiFontIndex, StyleIndex, FontStr)
   int        KanjiFontIndex, StyleIndex;
   char       * FontStr;
{
   switch(KanjiFontIndex) {
      case KANJI_FONT_RYUMIN :
         strcpy(FontStr, "Ryumin-Light-EUC-H");
         break;
      case KANJI_FONT_GOTHIC :
         strcpy(FontStr, "GothicBBB-Medium-EUC-H");
         break;
      case KANJI_FONT_RYUMIN_V :
         strcpy(FontStr, "Ryumin-Light-EUC-V");
         break;
      case KANJI_FONT_GOTHIC_V :
         strcpy(FontStr, "GothicBBB-Medium-EUC-V");
         break;
      case KANJI_FONT_NONE :
      default :
         break;
   }
}

void CurFontMsg ()
{
   char	msg[MAXSTRING], style[20];

   switch (curStyle)
   {
      case STYLE_BI: strcpy (style, "BoldItalic"); break;
      case STYLE_BR: strcpy (style, "Bold"); break;
      case STYLE_NI: strcpy (style, "Italic"); break;
      case STYLE_NR: strcpy (style, "Roman"); break;
   }

   sprintf (msg, "%s-%s-%s-%1dpt-%s", fontMenuStr[curFont], style,
         fontMenuStr[curKanjiFont + MAXFONTS],
         pointSize[curSize], fontDPIMenuStr[curFontDPI]);
   Msg (msg);
}

void DumpEightBitFontInfo (FP)
   FILE	* FP;
{
   register int		k, j;
   register short	* flag_ptr, flag;
   int			font_index, style_index;
   char			font_str[MAXSTRING];

   for (font_index = FONT_TIM; font_index < FONT_SYM; font_index++)
   {
      for (style_index = 0; style_index < MAXFONTSTYLES; style_index++)
      {
         if (NeedEncode (font_index, KANJI_FONT_NONE, style_index))
         {
            GetPSFontStr (font_index, style_index, font_str);
            fprintf (FP, "%s-vec [\n", font_str);
            flag_ptr = encodeCharFlags[font_index*MAXFONTSTYLES+style_index];
            for (j = 0; j < 16; j++, flag_ptr++)
            {
               flag = *flag_ptr;
               if ((flag & 0xff) != 0)
               {
                  for (k = 0; k < 8; k++)
                     if (flag & (1<<k))
                        fprintf (FP, " %s\n", charCodeToName[j*8+k]);
               }
            }
            fprintf (FP, " ] def\n");
            fprintf (FP, "%s %s-8 %s-vec tgifReEncodeSmall\n\n", font_str,
                  font_str, &font_str[1]);
         }
      }
   }
}

void SetCanvasFont ()
{
   register int	index, old_index;
   register int	kanji_index, idx;
   char		msg[MAXSTRING];
   int		watch_cursor;
   char		** baseKanjiFontName;
   XFontStruct	** baseKanjiFontPtr;

   index = FontIndex (curFontDPI, curFont, curSize, curStyle);
   if (!myFontInfo[index].valid)
   {
      watch_cursor = watchCursorOnMainWindow;
      if (!watch_cursor)
      {
         SetWatchCursor (drawWindow);
         SetWatchCursor (mainWindow);
      }
      myFontInfo[index].checked = TRUE;
      if ((myFontInfo[index].xfs =
            XLoadQueryFont (mainDisplay, fontNameStr[index])) == NULL)
      {
         old_index = index;
         curFontDPI = (curFontDPI==FONT_DPI_75) ? FONT_DPI_100 : FONT_DPI_75;
         index = FontIndex (curFontDPI, curFont, curSize, curStyle);
         if (!myFontInfo[index].valid)
         {
            myFontInfo[index].checked = TRUE;
            if ((myFontInfo[index].xfs =
                  XLoadQueryFont (mainDisplay, fontNameStr[index])) == NULL)
            {
               sprintf (msg, "Can not open '%s' nor\n\t'%s'!",
                     fontNameStr[old_index], fontNameStr[index]);
               Error ("SetCanvasFont()", msg);
            }
            fprintf (stderr, "%s '%s' font,\n\tuse '%s' instead.\n",
                  "Warning:  can not open", fontNameStr[old_index],
                  fontNameStr[index]);
            SetFileModified (TRUE);
         }
      }
      myFontInfo[index].valid = TRUE;
      if (!watch_cursor)
      {
         SetDefaultCursor (mainWindow);
         ShowCursor ();
      }
   }

   canvasFontPtr = myFontInfo[index].xfs;

   kanji_index = KanjiFontIndex (curFontDPI, curKanjiFont, curSize, curStyle);
   if (kanji_index == -1)
   {  /* KANJI_FONT_NONE */
      canvasKanjiFontPtr->xfs = canvasFontPtr;
      canvasKanjiFontPtr->ascent = canvasFontPtr->max_bounds.ascent;
      canvasKanjiFontPtr->descent = canvasFontPtr->max_bounds.descent;
      canvasKanjiFontPtr->vert = FALSE;
      canvasKanjiFontPtr->bold = FALSE;
   }
   else if (myKanjiFontInfo[kanji_index].valid)
   {
      canvasKanjiFontPtr->xfs = myKanjiFontInfo[kanji_index].xfs;
      canvasKanjiFontPtr->ascent = pDrawKanjiFontAsc[kanji_index];
      canvasKanjiFontPtr->descent = pDrawKanjiFontDes[kanji_index];
      canvasKanjiFontPtr->vert = IsVert(curKanjiFont);
      canvasKanjiFontPtr->bold = myKanjiFontInfo[kanji_index].bold;
   }
   else
   {
      SetWatchCursor (drawWindow);
      SetWatchCursor (mainWindow);
      myKanjiFontInfo[kanji_index].checked = TRUE;
      if (kanjiFontNameStr[kanji_index] != NULL &&
              (myKanjiFontInfo[kanji_index].xfs = XLoadQueryFont (mainDisplay,
            kanjiFontNameStr[kanji_index])) != NULL)
      {
         myKanjiFontInfo[kanji_index].valid = TRUE;
         myKanjiFontInfo[kanji_index].bold = FALSE;
	 if(IsGothic(curKanjiFont)) {
	    idx = KanjiFontIndex (curFontDPI, KANJI_FONT_RYUMIN, curSize,
		  curStyle);
	    if (kanjiFontNameStr[idx] != NULL &&
		!strcmp(kanjiFontNameStr[kanji_index], kanjiFontNameStr[idx]))
	       myKanjiFontInfo[kanji_index].bold = TRUE;
	 }
      }
      else
      {
	 if (IsRyumin(curKanjiFont)) {
	    baseKanjiFontName = &baseRyuminFontName;
	    baseKanjiFontPtr = &baseRyuminFontPtr;
	 }
	 else {
	    baseKanjiFontName = &baseGothicFontName;
	    baseKanjiFontPtr = &baseGothicFontPtr;
	 }

         if (*baseKanjiFontPtr != NULL) {
            myKanjiFontInfo[kanji_index].valid = TRUE;
            myKanjiFontInfo[kanji_index].xfs = *baseKanjiFontPtr;
	    myKanjiFontInfo[kanji_index].bold = FALSE;
	    if (IsGothic(curKanjiFont) && baseRyuminFontName != NULL &&
		!strcmp(baseGothicFontName, baseRyuminFontName))
	       myKanjiFontInfo[kanji_index].bold = TRUE;
         }
         else {
            if(*baseKanjiFontName != NULL && (*baseKanjiFontPtr =
                  XLoadQueryFont (mainDisplay, *baseKanjiFontName)) != NULL)
            {
               myKanjiFontInfo[kanji_index].valid = TRUE;
               myKanjiFontInfo[kanji_index].xfs = *baseKanjiFontPtr;
	       myKanjiFontInfo[kanji_index].bold = FALSE;
	       if (IsGothic(curKanjiFont) && baseRyuminFontName != NULL &&
		   !strcmp(baseGothicFontName, baseRyuminFontName))
		  myKanjiFontInfo[kanji_index].bold = TRUE;
            }
            else if((*baseKanjiFontPtr =
                  XLoadQueryFont (mainDisplay, defBaseKanjiFontName)) != NULL)
            {
	       *baseKanjiFontName = defBaseKanjiFontName;
               myKanjiFontInfo[kanji_index].valid = TRUE;
               myKanjiFontInfo[kanji_index].xfs = *baseKanjiFontPtr;
	       myKanjiFontInfo[kanji_index].bold = IsGothic(curKanjiFont);
            }
            else {
               if (kanjiFontNameStr[kanji_index] == NULL &&
                     *baseKanjiFontName == NULL) {
                  sprintf (msg, "Can not open '%s'!", defBaseKanjiFontName);
               }
               else if (kanjiFontNameStr[kanji_index] != NULL &&
                     *baseKanjiFontName == NULL)
               {
                  sprintf (msg, "Can not open '%s' nor\n\t'%s'!",
                        kanjiFontNameStr[kanji_index], defBaseKanjiFontName);
               }
               else if (kanjiFontNameStr[kanji_index] == NULL &&
                     *baseKanjiFontName != NULL)
               {
                  sprintf (msg, "Can not open '%s' nor\n\t'%s'!",
                        *baseKanjiFontName, defBaseKanjiFontName);
               }
               else
               {
                  sprintf (msg, "Can not open '%s' nor\n\t'%s' nor\n\t'%s'!",
                        kanjiFontNameStr[kanji_index], *baseKanjiFontName,
                        defBaseKanjiFontName);
               }
               Error ("SetCanvasFont()", msg);
               myKanjiFontInfo[kanji_index].xfs = NULL;
            }
         }
      }
      SetDefaultCursor (mainWindow);
      SetDefaultCursor (drawWindow);

      canvasKanjiFontPtr->xfs = myKanjiFontInfo[kanji_index].xfs;
      canvasKanjiFontPtr->ascent = pDrawKanjiFontAsc[kanji_index];
      canvasKanjiFontPtr->descent = pDrawKanjiFontDes[kanji_index];
      canvasKanjiFontPtr->vert = IsVert(curKanjiFont);
      canvasKanjiFontPtr->bold = myKanjiFontInfo[kanji_index].bold;
   }

   canvasFontAsc = max(canvasFontPtr->max_bounds.ascent,
       canvasKanjiFontPtr->ascent);
   canvasFontDes = max(canvasFontPtr->max_bounds.descent,
       canvasKanjiFontPtr->descent);
   canvasFontHeight = canvasFontAsc + canvasFontDes;

   XSetFont (mainDisplay, drawGC, canvasFontPtr->fid);
   if(canvasKanjiFontPtr->xfs != NULL)
       XSetFont (mainDisplay, drawKanjiGC, canvasKanjiFontPtr->xfs->fid);

   textCursorH = canvasFontHeight;

   pointSize = (curFontDPI==FONT_DPI_75) ? pointSize75 : pointSize100;
}

void InitFonts ()
{
   register int	i;
   int		ruler_index, default_index;
   int		old_ruler_index, old_default_index, len, dpi;
   char		* s;

   for (i = 0; i < FONTTABLESIZE; i++)
   {
      myFontInfo[i].valid = FALSE;
      myFontInfo[i].checked = FALSE;
   }
   for (i = 0; i < KANJI_FONTTABLESIZE; i++)
   {
      myKanjiFontInfo[i].valid = FALSE;
      myKanjiFontInfo[i].checked = FALSE;
      myKanjiFontInfo[i].bold = FALSE;
      kanjiFontNameStr[i] = NULL;
   }
   canvasKanjiFontPtr = (KanjiFontStruct *)calloc(1, sizeof(KanjiFontStruct));
   canvasKanjiFontPtr->vert = FALSE;
   canvasKanjiFontPtr->bold = FALSE;

   for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++)
      encodeCharFlags[i] = (short *) calloc (16, sizeof(short));
   encodeCharFlagsAllocated = TRUE;

   curFont = FONT_COU;
   curKanjiFont = KANJI_FONT_NONE;
   curStyle = STYLE_NR;
   curSize = 4;
   curFontDPI = FONT_DPI_75;

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFontDPI")) != NULL)
   {
      dpi = atoi (s);
      switch (dpi)
      {
         case 75: curFontDPI = FONT_DPI_75; curSize = 4; break;
         case 100: curFontDPI = FONT_DPI_100; curSize = 2; break;
         default:
            fprintf (stderr,"Warning:  can not set InitialFontDPI to '%s'\n",s);
            break;
      }
   }

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "MsgFontSizeIndex")) != NULL)
   {
      if (atoi (s) < MAXFONTSIZES)
         curSize = atoi (s);
      else
         fprintf (stderr, "%s '%s', '%1d' is used.\n",
               "Warning:  can not set MsgFontSizeIndex to", s, curSize);
   }

   default_index = FontIndex (curFontDPI, curFont, curSize, curStyle);
   myFontInfo[default_index].checked = TRUE;
   if ((myFontInfo[default_index].xfs =
         XLoadQueryFont (mainDisplay, fontNameStr[default_index])) == NULL)
   {
      old_default_index = default_index;
      curSize = 2;
      curFontDPI = FONT_DPI_100;
      default_index = FontIndex (curFontDPI, curFont, curSize, curStyle);
      myFontInfo[default_index].checked = TRUE;
      if ((myFontInfo[default_index].xfs =
            XLoadQueryFont (mainDisplay, fontNameStr[default_index])) == NULL)
      {
         fprintf (stderr, "Can not open '%s' nor\n\t'%s'!  %s aborted!\n\n",
               fontNameStr[old_default_index], fontNameStr[default_index],
               TOOL_NAME);
         exit (-1);
      }
   }
   myFontInfo[default_index].valid = TRUE;

   ruler_index = FontIndex (FONT_DPI_75, curFont, 1, curStyle);
   myFontInfo[ruler_index].checked = TRUE;
   if ((myFontInfo[ruler_index].xfs =
         XLoadQueryFont (mainDisplay, fontNameStr[ruler_index])) == NULL)
   {
      old_ruler_index = ruler_index;
      ruler_index = FontIndex (FONT_DPI_100, curFont, 0, curStyle);
      myFontInfo[ruler_index].checked = TRUE;
      if ((myFontInfo[ruler_index].xfs =
            XLoadQueryFont (mainDisplay, fontNameStr[ruler_index])) == NULL)
      {
         fprintf (stderr, "Can not open '%s' nor\n\t'%s'!  %s aborted!\n\n",
               fontNameStr[old_ruler_index], fontNameStr[ruler_index],
               TOOL_NAME);
         exit (-1);
      }
   }
   myFontInfo[ruler_index].valid = TRUE;

   defaultFontPtr = myFontInfo[default_index].xfs;
   defaultFontWidth = defaultFontPtr->max_bounds.width;
   defaultFontAsc = defaultFontPtr->max_bounds.ascent;
   defaultFontDes = defaultFontPtr->max_bounds.descent;
   defaultFontHeight = defaultFontAsc + defaultFontDes;

   rulerFontPtr = myFontInfo[ruler_index].xfs;
   rulerFontWidth = rulerFontPtr->max_bounds.width;
   rulerFontAsc = rulerFontPtr->max_bounds.ascent;
   rulerFontDes = rulerFontPtr->max_bounds.descent;
   rulerFontHeight = rulerFontAsc + rulerFontDes;

   pointSize = (curFontDPI==FONT_DPI_75) ? pointSize75 : pointSize100;

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFont")) != NULL)
   {
      for (i = 0; i < MAXFONTS; i++)
         if (strcmp (s, fontMenuStr[i]) == 0)
            break;
      if (i != MAXFONTS)
         curFont = i;
      else
         fprintf (stderr, "Warning:  can not set InitialFont to '%s'\n", s);
   }
   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialKanjiFont")) != NULL)
   {
      for (i = 0; i < MAX_KANJI_FONTS; i++)
         if (strcmp (s, fontMenuStr[i+MAXFONTS]) == 0)
            break;
      if (i != MAX_KANJI_FONTS)
         curKanjiFont = i;
      else
         fprintf (stderr, "Warning:  can not set InitialKanjiFont to '%s'\n", s);
   }
   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFontStyle")) != NULL)
   {
      len = strlen (s);
      if (len < strlen (styleMenuStr[0]))
      {
         for (i = 0; i < MAXFONTSTYLES; i++)
            if (strncmp (s, styleMenuStr[i], len) == 0)
               break;
         if (i != MAXFONTSTYLES)
            curStyle = i;
         else
            fprintf (stderr, "%s '%s'\n",
                  "Warning:  can not set InitialFontStyle to", s);
      }
      else
         fprintf (stderr,"Warning:  can not set InitialFontStyle to '%s'\n",s);
   }
   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "InitialFontJust")) != NULL)
   {
      len = strlen (s);
      if (len < strlen (styleMenuStr[0]))
      {
         for (i = MAXFONTSTYLES+1; i < MAXFONTSTYLES+1+MAXJUSTS; i++)
            if (strncmp (s, styleMenuStr[i], len) == 0)
               break;
         if (i != MAXFONTSTYLES+1+MAXJUSTS)
            textJust = i-MAXFONTSTYLES-1;
         else
            fprintf (stderr, "%s '%s'\n",
                  "Warning:  can not set InitialFontJust to", s);
      }
      else
         fprintf (stderr, "Warning:  can not set InitialFontJust to '%s'\n", s);
   }
   if ((s = XGetDefault (mainDisplay,TOOL_NAME,"InitialFontSizeIndex")) != NULL)
   {
      if (atoi (s) < MAXFONTSIZES)
         curSize = atoi (s);
      else
         fprintf (stderr, "%s '%s'\n",
               "Warning:  can not set InitialFontSizeIndex to", s);
   }

   baseRyuminFontName = XGetDefault (mainDisplay, TOOL_NAME, "BaseRyuminFont");
   baseGothicFontName = XGetDefault (mainDisplay, TOOL_NAME, "BaseGothicFont");
   for (i = 0; i < KANJI_FONTTABLESIZE; i++)
   {
      if ((s = XGetDefault (mainDisplay, TOOL_NAME, kanjiFontRes[i])) != NULL)
         kanjiFontNameStr[i] = s;
   }

   if ((s = XGetDefault (mainDisplay, TOOL_NAME, "ScaleRotateJapanese")) != NULL)
      if ((strcmp (s, "false") == 0) || (strcmp (s, "False") == 0))
	 drawScaledKanji = FALSE;

   InitRotTbl();
}

int ChangeObjTextStyle (ObjPtr, StyleIndex)
   register struct ObjRec	* ObjPtr;
   register int			StyleIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->style != StyleIndex)
         {
            ObjPtr->detail.t->style = StyleIndex;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextStyle (obj_ptr, StyleIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontStyle (StyleIndex)
   int	StyleIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				changed = FALSE, text_cursor_shown;

   if (StyleIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curStyle = StyleIndex;
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_STYLE);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextStyle (sel_ptr->obj, StyleIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int ChangeObjTextJust (ObjPtr, JustIndex)
   register struct ObjRec	* ObjPtr;
   register int			JustIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->just != JustIndex)
         {
            ObjPtr->detail.t->just = JustIndex;
            UpdTextBBox (ObjPtr);

            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextJust (obj_ptr, JustIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontJust (JustIndex)
   int JustIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby;
   int				text_obj_created, text_cursor_shown;
   int				changed = FALSE;

   if (JustIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      textJust = JustIndex;
      ShowJust ();
      UpdateSubMenu (MENU_STYLE);
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextJust (sel_ptr->obj, JustIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

static void FreeKanjiTextCache (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register struct ObjRec	* ptr;
   register struct AttrRec	* attr_ptr;
   struct TextRec		* text_ptr;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
	 text_ptr = ObjPtr->detail.t;

	 if (text_ptr->kanji_font != KANJI_FONT_NONE) {
	    if (text_ptr->cached_bitmap != None)
	       XFreePixmap (mainDisplay, text_ptr->cached_bitmap);
	    text_ptr->cached_zoom = 0;
	    text_ptr->cached_bitmap = None;
	 }
         break;

      case OBJ_SYM:
      case OBJ_ICON:
      case OBJ_GROUP:
         for (ptr = ObjPtr->detail.r->first; ptr != NULL; ptr = ptr->next)
            FreeKanjiTextCache (ptr);
         break;
   }
   for (attr_ptr=ObjPtr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
      if (attr_ptr->shown)
         FreeKanjiTextCache (attr_ptr->obj);
}

static void FreeKanjiTextCaches ()
{
   register struct ObjRec	* obj_ptr, * ptr;
   register struct AttrRec	* attr_ptr;

   if (topObj == NULL) return;

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      switch (obj_ptr->type)
      {
         case OBJ_TEXT:
            FreeKanjiTextCache (obj_ptr);
            break;
         case OBJ_SYM:
         case OBJ_ICON:
         case OBJ_GROUP:
            for (ptr = obj_ptr->detail.r->first; ptr != NULL; ptr = ptr->next)
               FreeKanjiTextCache (ptr);
            break;
      }
      for (attr_ptr=obj_ptr->fattr; attr_ptr!=NULL; attr_ptr=attr_ptr->next)
         if (attr_ptr->shown)
            FreeKanjiTextCache (attr_ptr->obj);
   }
}

void StyleSubMenu (index)
   int	index;
{
   if (index < MAXFONTSTYLES)
      ChangeFontStyle (index);
   else if (index > MAXFONTSTYLES && index < MAXFONTSTYLES+1+MAXJUSTS)
      ChangeFontJust (index - MAXFONTSTYLES - 1);
   else if (index == MAXFONTSTYLES+1+MAXJUSTS+1) {
      drawScaledKanji = (drawScaledKanji)?FALSE:TRUE;
      FreeKanjiTextCaches();
      UpdateSubMenu (MENU_STYLE);
      ClearAndRedrawDrawWindow ();
   }
}

void StyleMenu (X, Y)
   int  X, Y;
{
   register int	index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXFONTSTYLES+MAXJUSTS+3, &fore_colors, &valid,
         &init_rv);

   init_rv[curStyle] = TRUE;
   init_rv[MAXFONTSTYLES+1+textJust] = TRUE;
   init_rv[MAXFONTSTYLES+1+MAXJUSTS+1] = drawScaledKanji;
   activeMenu = MENU_STYLE;
   index = TextMenuLoop (X, Y, styleMenuStr, MAXFONTSTYLES+MAXJUSTS+3,
         fore_colors, valid, init_rv, SINGLECOLOR);

   if (index != INVALID) StyleSubMenu (index);
}

int ChangeObjTextSize (ObjPtr, SizeIndex)
   register struct ObjRec	* ObjPtr;
   register int			SizeIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->size != SizeIndex ||
               ObjPtr->detail.t->dpi != curFontDPI)
         {
            ObjPtr->detail.t->size = SizeIndex;
            ObjPtr->detail.t->dpi = curFontDPI;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextSize (obj_ptr, SizeIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFontSize (SizeIndex)
   int	SizeIndex;
{
   register struct SelRec	* sel_ptr;
   int				ltx, lty, rbx, rby, text_obj_created;
   int				text_cursor_shown, changed = FALSE;

   if (SizeIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curSize = SizeIndex;
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_SIZE);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextSize (sel_ptr->obj, SizeIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int GetSizeMenuIndex ()
{
   switch (curFontDPI)
   {
      case FONT_DPI_75:
         switch (curSize)
         {
            case 0: return (0);
            case 1: return (1);
            case 2: return (3);
            case 3: return (4);
            case 4: return (6);
            case 5: return (8);
         }
         break;
      case FONT_DPI_100:
         switch (curSize)
         {
            case 0: return (2);
            case 1: return (4);
            case 2: return (5);
            case 3: return (7);
            case 4: return (9);
            case 5: return (10);
         }
         break;
   }
   return (INVALID);
}

int ParseSizeMenuIndex (index, dpi)
   int	index, *dpi;
{
   switch (index)
   {
      case 0: *dpi = FONT_DPI_75; return (0);
      case 1: *dpi = FONT_DPI_75; return (1);
      case 2: *dpi = FONT_DPI_100; return (0);
      case 3: *dpi = FONT_DPI_75; return (2);
      case 4: *dpi = FONT_DPI_75; return (3);
      case 5: *dpi = FONT_DPI_100; return (2);
      case 6: *dpi = FONT_DPI_75; return (4);
      case 7: *dpi = FONT_DPI_100; return (3);
      case 8: *dpi = FONT_DPI_75; return (5);
      case 9: *dpi = FONT_DPI_100; return (4);
      case 10: *dpi = FONT_DPI_100; return (5);
   }
   return (INVALID);
}

void SizeSubMenu (index)
   int	index;
{
   int	saved_dpi=curFontDPI, new_index;

   if (topSel != NULL) saved_dpi = curFontDPI;
   new_index = ParseSizeMenuIndex (index, &curFontDPI);
   ChangeFontSize (new_index);
   if (topSel != NULL) curFontDPI = saved_dpi;
}

void SizeMenu (X, Y)
   int  X, Y;
{
   register int	index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (2*MAXFONTSIZES-1, &fore_colors, &valid, &init_rv);

   init_rv[GetSizeMenuIndex ()] = TRUE;
   activeMenu = MENU_SIZE;
   index = TextMenuLoop (X, Y, sizeMenuStr, 2*MAXFONTSIZES-1, fore_colors,
         valid, init_rv, SINGLECOLOR);

   if (index != INVALID) SizeSubMenu (index);
}

int ChangeObjTextFont (ObjPtr, FontIndex)
   register struct ObjRec	* ObjPtr;
   register int			FontIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->font != FontIndex)
         {
            if (ObjPtr->detail.t->font == FONT_SYM &&
		ObjPtr->detail.t->kanji_font == KANJI_FONT_NONE)
               RemoveIllegalChars (ObjPtr->detail.t);
            ObjPtr->detail.t->font = FontIndex;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextFont (obj_ptr, FontIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeFont (FontIndex)
   int	FontIndex;
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE, text_cursor_shown;
   int				ltx, lty, rbx, rby, text_obj_created;

   if (FontIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curFont = FontIndex;
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      ShowCurFont ();
      ShowTextSize ();
      UpdateSubMenu (MENU_FONT);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextFont (sel_ptr->obj, FontIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

int ChangeObjTextKanjiFont (ObjPtr, KanjiFontIndex)
   register struct ObjRec	* ObjPtr;
   register int			KanjiFontIndex;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->kanji_font != KanjiFontIndex)
         {
            if (ObjPtr->detail.t->kanji_font != KANJI_FONT_NONE &&
		ObjPtr->detail.t->font != FONT_SYM &&
		KanjiFontIndex == KANJI_FONT_NONE)
               RemoveIllegalChars (ObjPtr->detail.t);
	    if (ObjPtr->detail.t->kanji_font == KANJI_FONT_NONE)
	       CheckLength (ObjPtr->detail.t);
            ObjPtr->detail.t->kanji_font = KanjiFontIndex;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("    That object's vertical spacing reset to 0.");
               ObjPtr->detail.t->v_space = 0;
               UpdTextBBox (ObjPtr);
            }
            if (ObjPtr->detail.t->cached_bitmap != None)
               XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

            ObjPtr->detail.t->cached_zoom = 0;
            ObjPtr->detail.t->cached_bitmap = None;

            changed = TRUE;
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjTextKanjiFont (obj_ptr, KanjiFontIndex))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeKanjiFont (KanjiFontIndex)
   int	KanjiFontIndex;
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE, text_cursor_shown;
   int				ltx, lty, rbx, rby, text_obj_created;

   if (KanjiFontIndex == INVALID) return;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curKanjiFont = KanjiFontIndex;
      SetCanvasFont ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      /* There is no need for update Panel (Choice) Window
      ShowCurFont ();
      ShowTextSize ();
      */
      UpdateSubMenu (MENU_FONT);
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjTextKanjiFont (sel_ptr->obj, KanjiFontIndex))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

void FontSubMenu (index)
   int	index;
{
   if (index < MAXFONTS)
      ChangeFont (index);
   else if(index >= MAXFONTS && index < MAXFONTS+MAX_KANJI_FONTS)
      ChangeKanjiFont (index - MAXFONTS);
}

void FontMenu (X, Y)
   int  X, Y;
{
   register int			index;
   int				* fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXFONTS + MAX_KANJI_FONTS, &fore_colors,
	 &valid, &init_rv);

   init_rv[curFont] = TRUE;
   init_rv[curKanjiFont + MAXFONTS] = TRUE;
   activeMenu = MENU_FONT;
   index = TextMenuLoop (X, Y, fontMenuStr, MAXFONTS+MAX_KANJI_FONTS,
	 fore_colors, valid, init_rv, SINGLECOLOR);

   if (index != INVALID) FontSubMenu (index);
}

int ChangeObjVSpace (ObjPtr, VSpace)
   register struct ObjRec	* ObjPtr;
   register int			VSpace;
{
   register struct ObjRec	* obj_ptr;
   register int			changed=FALSE, obj_changed;
   int				saved_v_space;

   switch (ObjPtr->type)
   {
      case OBJ_TEXT:
         if (ObjPtr->detail.t->v_space != VSpace)
         {
            saved_v_space = ObjPtr->detail.t->v_space;
            ObjPtr->detail.t->v_space = VSpace;
            if (!UpdTextBBox (ObjPtr))
            {
               Msg ("Invalid vertical spacing for a text object.");
               Msg ("Vertical spacing for that object not changed.");
               ObjPtr->detail.t->v_space = saved_v_space;
               UpdTextBBox (ObjPtr);
            }
            else
            {
               if (ObjPtr->detail.t->cached_bitmap != None)
                  XFreePixmap (mainDisplay,ObjPtr->detail.t->cached_bitmap);

               ObjPtr->detail.t->cached_zoom = 0;
               ObjPtr->detail.t->cached_bitmap = None;

               changed = TRUE;
            }
         }
         break;

      case OBJ_GROUP:
      case OBJ_SYM:
         obj_changed = FALSE;
         for (obj_ptr = ObjPtr->detail.r->last; obj_ptr != NULL;
               obj_ptr = obj_ptr->prev)
            if (ChangeObjVSpace (obj_ptr, VSpace))
               obj_changed = TRUE;
         if (obj_changed)
         {
            changed = TRUE;
            AdjObjBBox (ObjPtr);
         }
         break;
   }
   return (changed);
}

void ChangeVSpace (VSpace)
   int	VSpace;
{
   register struct SelRec	* sel_ptr;
   int				changed = FALSE, text_cursor_shown;
   int				ltx, lty, rbx, rby, text_obj_created;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      if (textCursorH+textVSpace <= 0)
         Msg ("Text vertical spacing too small.  No change.");
      else
      {
         textVSpace = VSpace;
         SetCanvasFont ();
         ShowTextVSpace ();
      }
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
   {
      PrepareToReplaceAnObj (sel_ptr->obj);
      if (ChangeObjVSpace (sel_ptr->obj, VSpace))
      {
         changed = TRUE;
         RecordReplaceAnObj (sel_ptr->obj);
      }
      else
         AbortPrepareCmd (CMD_REPLACE);
   }
   EndCompositeCmd ();

   if (changed)
   {
      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1), selLtX-GRID_ABS_SIZE(1),
            selLtY-GRID_ABS_SIZE(1), selRbX+GRID_ABS_SIZE(1),
            selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
   }
   HighLightForward ();
}

static int	savedKanjiFont, savedFont, savedSize, savedStyle, savedDPI;
static int	savedJust, savedRotate, savedPen, savedFill, savedVSpace;

void SaveCurFont ()
{
   savedKanjiFont = curKanjiFont;
   savedFont = curFont;
   savedSize = curSize;
   savedStyle = curStyle;
   savedDPI = curFontDPI;
   savedJust = textJust;
   savedRotate = curRotate;
   savedPen = penPat;
   savedFill = objFill;
   savedVSpace = textVSpace;
}

void RestoreCurFont ()
{
   curKanjiFont = savedKanjiFont;
   curFont = savedFont;
   curSize = savedSize;
   curStyle = savedStyle;
   curFontDPI = savedDPI;
   textJust = savedJust;
   curRotate = savedRotate;
   penPat = savedPen;
   objFill = savedFill;
   textVSpace = savedVSpace;

   SetCanvasFont ();
}

static int	pushedKanjiFont;
static int	pushedFont, pushedSize, pushedStyle, pushedDPI, pushedJust;
static int	pushedColorIndex, pushedRotate, pushedPen, pushedVSpace;
static int	pushedFill;

void PushCurFont ()
{
   pushedKanjiFont = curKanjiFont;
   pushedFont = curFont;
   pushedSize = curSize;
   pushedStyle = curStyle;
   pushedDPI = curFontDPI;
   pushedJust = textJust;
   pushedRotate = curRotate;
   pushedPen = penPat;
   pushedFill = objFill;
   pushedVSpace = textVSpace;
   pushedColorIndex = colorIndex;
}

void PopCurFont ()
{
   curKanjiFont = pushedKanjiFont;
   curFont = pushedFont;
   curSize = pushedSize;
   curStyle = pushedStyle;
   curFontDPI = pushedDPI;
   textJust = pushedJust;
   curRotate = pushedRotate;
   penPat = pushedPen;
   objFill = pushedFill;
   textVSpace = pushedVSpace;
   colorIndex = pushedColorIndex;

   SetCanvasFont ();
}

void CleanUpFonts ()
{
   register int	i;

   for (i = 0; i < FONTTABLESIZE; i++)
   {
      if (myFontInfo[i].valid)
      {
         XFreeFont (mainDisplay, myFontInfo[i].xfs);
         myFontInfo[i].valid = FALSE;
      }
   }
   for (i = 0; i < KANJI_FONTTABLESIZE; i++)
   {
      if (myKanjiFontInfo[i].valid &&
          myKanjiFontInfo[i].xfs != baseRyuminFontPtr &&
          myKanjiFontInfo[i].xfs != baseGothicFontPtr)
      {
         XFreeFont (mainDisplay, myKanjiFontInfo[i].xfs);
         myKanjiFontInfo[i].valid = FALSE;
      }
   }
   for (i = 0; i < MAXFONTS*MAXFONTSTYLES; i++) cfree (encodeCharFlags[i]);
   encodeCharFlagsAllocated = FALSE;

   if(baseRyuminFontPtr != NULL)
      XFreeFont (mainDisplay, baseRyuminFontPtr);
   if(baseGothicFontPtr != NULL)
      XFreeFont (mainDisplay, baseGothicFontPtr);
   if (textBitmapWidth != INVALID && textBitmapHeight != INVALID)
      XFreePixmap (mainDisplay, textBitmap);
   if (baseGC != NULL) XFreeGC (mainDisplay, baseGC);
}

/* The following procedure is used to generate pdrawFontAsc[] and */
/*    pDrawFontDes[], to be used in ``prtgif.c''.  It is supposed */
/*    to be called within dbx and not tgif.                       */

/*
 * static
 * void GenerateFontInfo ()
 * {
 *    register int	i, j, num_rows;
 * 
 *    for (i = 0; i < FONTTABLESIZE; i++)
 *       if (!myFontInfo[i].valid)
 *       {
 *          if ((myFontInfo[i].xfs =
 *                XLoadQueryFont (mainDisplay, fontNameStr[i])) == NULL)
 *          {
 *             printf ("Can not open %s.  Abort.\n\n", fontNameStr[i]);
 *             exit (-1);
 *          }
 *          myFontInfo[i].valid = TRUE;
 *       }
 * 
 *    num_rows = FONTTABLESIZE / MAXFONTSIZES;
 *    printf ("short\tpDrawFontAsc[] =\n{\n");
 *    for (i = 0; i < num_rows; i++)
 *    {
 *       printf ("   ");
 *       for (j = 0; j < MAXFONTSIZES; j++)
 *          if (i == num_rows-1 && j == MAXFONTSIZES-1)
 *             printf ("%2d ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.ascent);
 *          else
 *             printf ("%2d, ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.ascent);
 *       printf ("\n");
 *    }
 *    printf ("};\n\n");
 * 
 *    printf ("short\tpDrawFontDes[] =\n{\n");
 *    for (i = 0; i < num_rows; i++)
 *    {
 *       printf ("   ");
 *       for (j = 0; j < MAXFONTSIZES; j++)
 *          if (i == num_rows-1 && j == MAXFONTSIZES-1)
 *             printf ("%2d ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.descent);
 *          else
 *             printf ("%2d, ",
 *                   (myFontInfo[i*MAXFONTSIZES+j].xfs)->max_bounds.descent);
 *       printf ("\n");
 *    }
 *    printf ("};\n");
 * }
 */

int
TextWidth(xfs, kfs, str, len)
XFontStruct *	xfs;
KanjiFontStruct *	kfs;
char *	str;
int	len;
{
   int	width = 0;
   register char *	head;
   register char *	tail;
   register char	check;
   register int		i;
   XChar2b		str16[MAXSTRING / 2];

   head = str;
   check = *head & 0x80;

   while(head - str < len) {
      for(tail = head, i = 0; tail - str < len && (*tail & (char)0x80) == check; tail ++, i ++) {
	 if(i & 1)
	    str16[i >> 1].byte2 = (*tail & 0x7f);
	 else
	    str16[i >> 1].byte1 = (*tail & 0x7f);
      }
      if(check == 0x00) {
	 width += XTextWidth(xfs, head, tail - head);
      }
      else if(kfs->xfs == NULL) {
	 width += (kfs->ascent + kfs->descent) * ((tail - head) >> 1);
      }
      else if(kfs->xfs->min_byte1 == 0 && kfs->xfs->max_byte1 == 0) {
	 width += XTextWidth(kfs->xfs, head, tail - head);
      }
      else if(kfs->xfs->ascent + kfs->xfs->descent ==
	    kfs->ascent + kfs->descent) {
	 width += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1);
      }
      else {
	 width += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1) *
	       (kfs->ascent + kfs->descent) /
	       (kfs->xfs->ascent + kfs->xfs->descent);
      }
      head = tail;
      check = *head & 0x80;
   }

   return width;
}

static void
AllocTextBitmapAndScaleBuf(display, d, width, height)
Display		*display;
Drawable	d;
int		width;
int		height;
{
   if(height > textBitmapHeight || width > textBitmapWidth) {
      if(textBitmapHeight != INVALID && textBitmapWidth != INVALID) {
	 XFreePixmap(display, textBitmap);
	 free(scaleBuf);
      }
      
      textBitmapHeight = height;
      textBitmapWidth = width;
      textBitmap = XCreatePixmap(display, d,
	    textBitmapWidth, textBitmapHeight, 1);
      scaleBuf = (char *)malloc(textBitmapHeight * sizeof(char));
   }
}

static void
CreateBaseGC(display)
Display	*display;
{
   XGCValues		values;

   values.foreground = 1;
   values.background = 0;
   values.fill_style = FillSolid;
   values.function = GXcopy;
   baseGC = XCreateGC (display, textBitmap,
	 GCForeground | GCBackground | GCFillStyle | GCFunction, &values);
}

static void
InitRotTbl()
{
   int	i;

   for(i = 0; i < 256; i ++)
      RotTbl21XX[i] = ROT_NRM;

   RotTbl21XX[0x22] = ROT_MOV; RotTbl21XX[0x23] = ROT_MOV;

   RotTbl21XX[0x31] = ROT_NON; RotTbl21XX[0x32] = ROT_NON;
   RotTbl21XX[0x3c] = ROT_INV;
   RotTbl21XX[0x3d] = ROT_NON; RotTbl21XX[0x3e] = ROT_NON;

   RotTbl21XX[0x41] = ROT_INV;
   RotTbl21XX[0x42] = ROT_NON; RotTbl21XX[0x43] = ROT_NON;
   RotTbl21XX[0x44] = ROT_NON; RotTbl21XX[0x45] = ROT_NON;
   RotTbl21XX[0x4a] = ROT_NON; RotTbl21XX[0x4b] = ROT_NON;
   RotTbl21XX[0x4c] = ROT_NON; RotTbl21XX[0x4d] = ROT_NON;
   RotTbl21XX[0x4e] = ROT_NON; RotTbl21XX[0x4f] = ROT_NON;

   RotTbl21XX[0x50] = ROT_NON; RotTbl21XX[0x51] = ROT_NON;
   RotTbl21XX[0x52] = ROT_NON; RotTbl21XX[0x53] = ROT_NON;
   RotTbl21XX[0x54] = ROT_NON; RotTbl21XX[0x55] = ROT_NON;
   RotTbl21XX[0x56] = ROT_NON; RotTbl21XX[0x57] = ROT_NON;
   RotTbl21XX[0x58] = ROT_NON; RotTbl21XX[0x59] = ROT_NON;
   RotTbl21XX[0x5a] = ROT_NON; RotTbl21XX[0x5b] = ROT_NON;

   RotTbl21XX[0x61] = ROT_NON;
}

static void
RotChars(num_chars, from_image, to_image, size, str16)
int	num_chars;
XImage	*from_image;
XImage	*to_image;
int	size;
XChar2b	*str16;
{
   register int	i;
   register int	x1;
   register int	x2;
   register int	y2;
   register int size2;

   size2 = size >> 1;
   for(i = num_chars, x1 = 0; i > 0; i --, x1 += size, str16++) {
      switch((str16->byte1 == 0x21)?RotTbl21XX[str16->byte2]:ROT_NRM) {
	 case ROT_NON :
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, x2 + x1, y2, 1);
	       }
	    }
	    break;
	 case ROT_INV :
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, x2 + x1, size - y2, 1);
	       }
	    }
	    break;
	 case ROT_MOV:
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, (y2 + size2) % size + x1,
			   size - (x2 + size2) % size, 1);
	       }
	    }
	    break;
	 case ROT_NRM :
	 default :
	    for(x2 = 0; x2 < size ; x2 ++) {
	       for(y2 = 0; y2 < size; y2 ++) {
		  if(XGetPixel(from_image, x2 + x1, y2))
		     XPutPixel(to_image, y2 + x1, size - x2, 1);
	       }
	    }
	    break;
      }
   }
}

static void
ScaleUp(from_image, to_image, width, base_size, size)
XImage	*from_image;
XImage	*to_image;
int	width;
int	base_size;
int	size;
{
   register int	i;
   register int	j;
   register int	x1;
   register int	y1;
   register int	x2;
   register int	y2;

   i = -1;
   for(x2 = 0; x2 < width; x2 ++) {
      x1 = x2 * base_size / size;
      if(x1 != i) {
	 j = -1;
	 for(y2 = 0; y2 < size; y2 ++) {
	    y1= y2 * base_size / size;
	    if(y1 != j) {
	       if((scaleBuf[y2] = XGetPixel(from_image, x1, y1)) == 1)
		  XPutPixel(to_image, x2, y2, 1);
	       j = y1;
	    }
	    else {
	       if((scaleBuf[y2] = scaleBuf[y2 - 1]) == 1)
		  XPutPixel(to_image, x2, y2, 1);
	    }
	 }
	 i = x1;
      }
      else {
	 for(y2 = 0; y2 < size; y2 ++) {
	    if(scaleBuf[y2] == 1)
	       XPutPixel(to_image, x2, y2, 1);
	 }
      }
   }
}

static void
ScaleDown(from_image, to_image, width, base_size, size)
XImage	*from_image;
XImage	*to_image;
int	width;
int	base_size;
int	size;
{
   register int	x1;
   register int	y1;
   register int	x2;
   register int	y2;

   for(x2 = 0; x2 < width; x2 ++) {
      x1 = x2 * base_size / size;
      for(y2 = 0; y2 < size; y2 ++) {
	 y1= y2 * base_size / size;
	 if(XGetPixel(from_image, x1, y1) == 1)
	    XPutPixel(to_image, x2, y2, 1);
      }
   }
}

static int
DrawScaleRotateString16(display, d, kgc, x, y, str16, len, kfs, clipGC)
Display		*display;
Drawable	d;
GC		kgc;
int		x;
int		y;
XChar2b		*str16;
int		len;
KanjiFontStruct	*kfs;
GC		*clipGC;
{
   int		base_size;
   int		size;
   int		base_width;
   int		width;
   XImage	*from_image;
   XImage	*via_image;
   XImage	*to_image;
   XGCValues	values;


   base_size = kfs->xfs->ascent + kfs->xfs->descent;
   size = kfs->ascent + kfs->descent;
   base_width = XTextWidth16(kfs->xfs, str16, len);
   width = base_width * size / base_size;

   AllocTextBitmapAndScaleBuf(display, d, max(width, base_width),
	 max(size, base_size));

   if(baseGC == NULL)
      CreateBaseGC(display);

   XSetForeground(display, baseGC, 0);
   XFillRectangle(display, textBitmap, baseGC, 0, 0,
	 textBitmapWidth, textBitmapHeight);
   if(kfs->vert && size != base_size)
      via_image = XGetImage(display, textBitmap, 0, 0,
	    textBitmapWidth, textBitmapHeight, 1, ZPixmap);
   to_image = XGetImage(display, textBitmap, 0, 0,
	 textBitmapWidth, textBitmapHeight, 1, ZPixmap);
   
   values.foreground = 1;
   values.font = kfs->xfs->fid;
   XChangeGC(display, baseGC, GCForeground | GCFont, &values);

   XDrawString16(display, textBitmap, baseGC, 0, kfs->xfs->ascent, str16, len);
   if(kfs->bold)
      XDrawString16(display, textBitmap, baseGC, 1, kfs->xfs->ascent, str16, len);
   from_image = XGetImage(display, textBitmap, 0, 0,
	 textBitmapWidth, textBitmapHeight, 1, ZPixmap);

   if(size == base_size) {
      RotChars(len, from_image, to_image, size, str16);
   }
   else if(size > base_size) { /* Scale up */
      if(kfs->vert) {
	 RotChars(len, from_image, via_image, base_size, str16);
	 ScaleUp(via_image, to_image, width, base_size, size);
      }
      else {
	 ScaleUp(from_image, to_image, width, base_size, size);
      }
   }
   else { /* Scale Down */
      if(kfs->vert) {
	 ScaleDown(from_image, via_image, width, base_size, size);
	 RotChars(len, via_image, to_image, size, str16);
      }
      else {
	 ScaleDown(from_image, to_image, width, base_size, size);
      }
   }

   XPutImage(display, textBitmap, baseGC, to_image, 0, 0, 0, 0, width, size);

   if(*clipGC == NULL) {
      values.clip_x_origin = x;
      values.clip_y_origin = y - kfs->ascent;
      values.clip_mask = textBitmap;
      *clipGC = XCreateGC(display, d, 
	    GCClipMask | GCClipXOrigin | GCClipYOrigin, &values);
      XCopyGC(display, kgc,
	    ~0L & ~GCClipMask & ~GCClipXOrigin & ~GCClipYOrigin, *clipGC);
   }
   else {
      values.clip_x_origin = x;
      values.clip_y_origin = y - kfs->ascent;
      values.clip_mask = textBitmap;
      XChangeGC(display, *clipGC, GCClipMask | GCClipXOrigin | GCClipYOrigin,
	    &values);
   }

   XFillRectangle(display, d, *clipGC, x, y - kfs->ascent, width, size);

   XDestroyImage(from_image);
   if(kfs->vert && size != base_size)
      XDestroyImage(via_image);
   XDestroyImage(to_image);

   return width;
}

void
DrawString (display, d, gc, kgc, x, y, str, len, xfs, kfs)
Display *	display;
Drawable	d;
GC		gc;
GC		kgc;
int		x;
int		y;
char *		str;
int		len;
XFontStruct *	xfs;
KanjiFontStruct *	kfs;
{
   register char *	head;
   register char *	tail;
   register char	check;
   register int		i;
   XChar2b		str16[MAXSTRING / 2];
   int			size;
   int			x1;
   int			y1;
   GC			clipGC = NULL;

   head = str;
   check = *head & 0x80;

   while(head - str < len) {
      for(tail = head, i = 0; tail - str < len && (*tail & (char)0x80) == check; tail ++, i ++) {
	 if(i & 1)
	    str16[i >> 1].byte2 = (*tail & 0x7f);
	 else
	    str16[i >> 1].byte1 = (*tail & 0x7f);
      }

      if(check == 0x00) {
	 /* ISO characters in GL*/
	 XDrawString(display, d, gc, x, y, head, tail - head);
	 x += XTextWidth(xfs, head, tail - head);
      }
      else if(kfs->xfs == NULL) {
	 /* JAPANESE characters, but no font */
	 size = kfs->ascent + kfs->descent;
	 x1 = x;
	 y1 = y - kfs->ascent;
	 for(i = (tail - head) >> 1; i > 0; i --) {
	    XFillRectangle(display, d, kgc, x1 + 1, y1 + 1, size - 2, size - 2);
	    x1 += size;
	 }
	 x += size * ((tail - head) >> 1);
      }
      else if(kfs->xfs->min_byte1 == 0 && kfs->xfs->max_byte1 == 0) {
	 /* ISO characters in GR */
	 XDrawString(display, d, kgc, x, y, head, tail - head);
	 x += XTextWidth(kfs->xfs, head, tail - head);
      }
      else if(kfs->xfs->ascent + kfs->xfs->descent ==
	    kfs->ascent + kfs->descent && !kfs->vert) {
	 /* JAPANESE characters, no scale or rotate */
	 XDrawString16(display, d, kgc, x, y, str16, (tail - head) >> 1);
	 if(kfs->bold)
	    XDrawString16(display, d, kgc, x + 1, y, str16, (tail - head) >> 1);
	 x += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1);
      }
      else if(!drawScaledKanji) {
	 /* JAPANESE characters, scale or rotate with black box */
	 size = kfs->ascent + kfs->descent;
	 x1 = x;
	 y1 = y - kfs->ascent;
	 for(i = (tail - head) >> 1; i > 0; i --) {
	    XFillRectangle(display, d, kgc, x1 + 1, y1 + 1, size - 2, size - 2);
	    x1 += size;
	 }
	 x += XTextWidth16(kfs->xfs, str16, (tail - head) >> 1) *
	       (kfs->ascent + kfs->descent) /
	       (kfs->xfs->ascent + kfs->xfs->descent);
      }
      else {
	 /* JAPANESE characters, scale or rotate */
	 x += DrawScaleRotateString16(display, d, kgc, x, y, str16,
	       (tail - head) >> 1, kfs, &clipGC);
      }
      head = tail;
      check = *head & 0x80;
   }

   if(clipGC != NULL)
      XFreeGC(display, clipGC);
}
