/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/page.c,v 2.9 1993/06/12 19:13:01 william Exp $";
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "button.e"
#include "cmd.e"
#include "choice.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "file.e"
#include "font.e"
#include "grid.e"
#include "mark.e"
#include "mainloop.e"
#include "mainmenu.e"
#include "menu.e"
#include "move.e"
#include "msg.e"
#include "names.e"
#include "obj.e"
#include "raster.e"
#include "select.e"
#include "setup.e"

struct PageRec	* firstPage=NULL, * lastPage=NULL, * curPage=NULL;
int		curPageNum=1, lastPageNum=1;
int		paperRow=1, paperCol=1;
int		pageLayoutMode=PAGE_STACK;
int		pageLineShownInTileMode=TRUE;

char * pageStackMenuStr[] =
      { "NextPage     ",
        "PrevPage     ",
        "NamePages    ",
        "GotoPage     ",
        "AddPageBefore",
        "AddPageAfter ",
        "DeleteCurPage",
        "PrintOnePage "
      };

char * pageTileMenuStr[] =
      { "TogglePageLineShown",
        "SpecifyDrawingSize ",
        "PrintOnePage       "
      };

void InitPage ()
   /* given lastPageNum, allocate enough pages */
{
   int	i;

   firstPage = lastPage = curPage = NULL;
   for (i = 1; i <= lastPageNum; i++)
   {
      curPage = (struct PageRec *) calloc (1, sizeof(struct PageRec));
      curPage->name = NULL;
      curPage->top = curPage->bot = topObj = botObj = NULL;
      curPage->next = NULL;
      curPage->prev = lastPage;
      curPage->draw_orig_x = drawOrigX;
      curPage->draw_orig_y = drawOrigY;
      curPage->zoom_scale = zoomScale;
      curPage->zoomed_in = zoomedIn;
      if (firstPage == NULL)
         firstPage = curPage;
      else
         lastPage->next = curPage;
      lastPage = curPage;
   }
   curPageNum = (lastPageNum > 0 ? 1 : 0);
   curPage = (lastPageNum > 0 ? firstPage : NULL);
}

void GotoPageNum (new_page_num)
   int	new_page_num;
{
   int			i=1;
   struct PageRec	* page_ptr;

   curPage->draw_orig_x = drawOrigX;
   curPage->draw_orig_y = drawOrigY;
   curPage->zoom_scale = zoomScale;
   curPage->zoomed_in = zoomedIn;

   for (page_ptr = firstPage; page_ptr != NULL; page_ptr = page_ptr->next, i++)
      if (i == new_page_num)
         break;
   curPageNum = new_page_num;
   curPage = page_ptr;
   topObj = curPage->top;
   botObj = curPage->bot;
   if (curPage->draw_orig_x != drawOrigX || curPage->draw_orig_y != drawOrigY ||
         curPage->zoom_scale != zoomScale || curPage->zoomed_in != zoomedIn)
   {
      AdjSplineVs();
      AdjCaches();
      curPage->draw_orig_x = drawOrigX;
      curPage->draw_orig_y = drawOrigY;
      curPage->zoom_scale = zoomScale;
      curPage->zoomed_in = zoomedIn;
   }
}

void SetCurPage (page_number)
   int	page_number;
{
   if (pageLayoutMode == PAGE_TILE) return;

   MakeQuiescent ();
   PrepareToRecord (CMD_GOTO_PAGE, NULL, NULL, curPageNum);
   GotoPageNum (page_number);
   RecordCmd (CMD_GOTO_PAGE, NULL, NULL, NULL, curPageNum);
   ClearAndRedrawDrawWindow ();
   RedrawTitleWindow ();
   ShowPage ();
}

void NextPage ()
{
   if (pageLayoutMode == PAGE_TILE)
   {
      Msg ("Can not do NextPage() in TILED page mode.");
      return;
   }
   if (curPageNum == lastPageNum)
   {
      Msg ("Already at last page.");
      return;
   }
   SetCurPage (curPageNum+1);
}

void PrevPage ()
{
   if (pageLayoutMode == PAGE_TILE)
   {
      Msg ("Can not do PrevPage() in TILED page mode.");
      return;
   }
   if (curPageNum == 1)
   {
      Msg ("Already at first page.");
      return;
   }
   SetCurPage (curPageNum-1);
}

static XComposeStatus	c_stat;
static int		leadingChars=(-1);
static char		formatStr[20];
static DspList		* pageNameDspPtr=NULL;

static
DspList * PageNameListing ()
{
   register int		i;
   DspList		* dsp_ptr;
   struct PageRec	* page_ptr;

   for (leadingChars=1, i=lastPageNum; ;leadingChars++, i /= 10)
      if (i < 10)
         break;

   sprintf (formatStr, "%%%1dd %%s", leadingChars++);
   pageNameDspPtr = (DspList *) calloc (lastPageNum, sizeof(DspList));
   for (i = 1, dsp_ptr = pageNameDspPtr, page_ptr = firstPage; i <= lastPageNum;
         i++, dsp_ptr++, page_ptr = page_ptr->next)
   {
      sprintf (dsp_ptr->itemstr, formatStr, i,
            ((page_ptr->name == NULL) ? "" : page_ptr->name));
      dsp_ptr->next = ((i == lastPageNum) ? NULL : &dsp_ptr[1]);
   }
   nameEntries = lastPageNum;
   return (pageNameDspPtr);
}

static
int EditOrSelectPageNames (TopStr, Which)
   char	* TopStr;
   int	Which; /* either PAGE_GOTO or PAGE_NAME */
{
   int			button_widths, str_width, graph_width;
   int			str_start, button_start, graph_start;
   int			dsp_w, dsp_h, w, h, i, button_selected = INVALID;
   XEvent		input, ev;
   int			changing = TRUE, name_index, exposed = FALSE;
   char			buf[80], name[MAXPATHLENGTH];
   XKeyEvent		* key_ev;
   XButtonEvent		* button_ev;
   KeySym		key_sym;
   XWMHints		wmhints;
   XSizeHints		sizehints;
   XSetWindowAttributes	win_attrs;
   int			win_x, win_y;

   dsp_w = DisplayWidth (mainDisplay, mainScreen);
   dsp_h = DisplayHeight (mainDisplay, mainScreen);

   button_widths = ButtonWidth("OK", 8) + ButtonWidth("CANCEL", 8) +
         defaultFontWidth;
   numButtons = 2;
   strcpy (buttonStr[0], "OK");
   strcpy (buttonStr[1], "CANCEL");

   str_width = defaultFontWidth * strlen (TopStr);
   graph_width = nameDspWinW + scrollBarW + 2 * brdrW;

   if (str_width > graph_width)
   {
      w = str_width + 4 * defaultFontWidth;
      str_start = 2 * defaultFontWidth;
      graph_start = (w - graph_width) / 2;
   }
   else
   {
      w = graph_width + 4 * defaultFontWidth;
      str_start = (w - str_width) / 2;
      graph_start = 2 * defaultFontWidth;
   }
   button_start = (w - button_widths) / 2;
   h = (8 + ITEM_DSPED) * ROW_HEIGHT;

   win_x = (w > dsp_w) ? 0 : (dsp_w - w)/2;
   win_y = (h > dsp_h) ? 0 : (dsp_h - h)/3;

   if ((nameBaseWin = XCreateSimpleWindow (mainDisplay, rootWindow,
         win_x, win_y, w, h, brdrW, myBorderPixel, myBgPixel)) == 0)
      Error ("ChooseAName()", "Can not XCreateSimpleWindow() for nameBaseWin");

   XDefineCursor (mainDisplay, nameBaseWin, defaultCursor);

   if ((nameDspWin = XCreateSimpleWindow (mainDisplay, nameBaseWin, graph_start,
         5*ROW_HEIGHT, nameDspW, nameDspH, brdrW, myBorderPixel,
         myBgPixel)) == 0)
      Error ("ChooseAName()", "Can not XCreateSimpleWindow() for nameDspWin");

   if ((nameScrollWin = XCreateSimpleWindow (mainDisplay, nameBaseWin,
         graph_start+nameDspWinW, 5*ROW_HEIGHT, scrollBarW, nameDspH,
         brdrW, myBorderPixel, myBgPixel)) == 0)
      Error ("ChooseAName()","Can not XCreateSimpleWindow() for nameScrollWin");

   win_attrs.save_under = True;
   XChangeWindowAttributes (mainDisplay, nameBaseWin, CWSaveUnder, &win_attrs);

   wmhints.flags = InputHint | StateHint;
   wmhints.input = True;
   wmhints.initial_state = NormalState;
   XSetWMHints (mainDisplay, nameBaseWin, &wmhints);

   sizehints.flags = PPosition | PSize | USPosition | PMinSize | PMaxSize;
   sizehints.x = win_x;
   sizehints.y = win_y;
   sizehints.width = sizehints.min_width = sizehints.max_width = w;
   sizehints.height = sizehints.min_height = sizehints.max_height = h;
#ifdef NOTR4MODE
   XSetNormalHints (mainDisplay, nameBaseWin, &sizehints);
#else
   XSetWMNormalHints (mainDisplay, nameBaseWin, &sizehints);
#endif

   XSetTransientForHint (mainDisplay, nameBaseWin, mainWindow);

#ifdef MAPBEFORESELECT
   XMapWindow (mainDisplay, nameBaseWin);
   XSelectInput (mainDisplay, nameBaseWin,
         KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow (mainDisplay, nameDspWin);
   XSelectInput (mainDisplay, nameDspWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow (mainDisplay, nameScrollWin);
   XSelectInput (mainDisplay, nameScrollWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
#else
   XSelectInput (mainDisplay, nameBaseWin,
         KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow (mainDisplay, nameBaseWin);
   XSelectInput (mainDisplay, nameDspWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow (mainDisplay, nameDspWin);
   XSelectInput (mainDisplay, nameScrollWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow (mainDisplay, nameScrollWin);
#endif

   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, nameBaseWin, 0, 0, 0, 0,
            (int)(w/2), (int)(h/2));

   XSync (mainDisplay, False);

   justClicked = FALSE;

   Msg ("");

   if (nameMarked == INVALID)
   {
      name[0] = '\0';
      name_index = 0;
   }
   else
   {
      if (nameMarked >= ITEM_DSPED)
      {
         if (nameMarked < nameEntries-ITEM_DSPED)
            nameFirst = nameMarked;
         else
            nameFirst = nameEntries-ITEM_DSPED;
      }
      strcpy (name, &(nameDspPtr[nameMarked])[leadingChars]);
      name_index = strlen (name);
   }

   while (changing)
   {
      XNextEvent (mainDisplay, &input);

      if ((input.type==MapNotify && input.xany.window==nameBaseWin) ||
            (input.type==Expose && (input.xany.window==nameBaseWin ||
            input.xany.window==nameScrollWin ||
            input.xany.window==nameDspWin)) ||
            (!exposed &&
            (XCheckWindowEvent (mainDisplay,nameBaseWin,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,nameScrollWin,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,nameDspWin,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,nameBaseWin,StructureNotifyMask,
            &ev))))
      {
         while (XCheckWindowEvent (mainDisplay,nameBaseWin,ExposureMask,&ev)) ;
         while (XCheckWindowEvent (mainDisplay,nameScrollWin,ExposureMask,
               &ev))
            ;
         while (XCheckWindowEvent (mainDisplay,nameDspWin,ExposureMask,&ev)) ;
         while (XCheckWindowEvent (mainDisplay,nameBaseWin,StructureNotifyMask,
               &ev))
            ;

         RedrawNameBaseWindow (TopStr, name, str_start, graph_start,
               button_start, w, h);
         RedrawNameScrollWin ();
         RedrawDspWindow ();

         exposed = TRUE;
         XSync (mainDisplay, False);

         if ((input.type==MapNotify && input.xany.window==nameBaseWin) ||
               (input.type==Expose && (input.xany.window==nameBaseWin ||
               input.xany.window==nameScrollWin ||
               input.xany.window==nameDspWin)))
            continue;
      }

      if (input.type==Expose)
         ExposeEventHandler (&input, FALSE);
      else if (input.type==VisibilityNotify && input.xany.window==mainWindow &&
            input.xvisibility.state==VisibilityUnobscured)
      {
         while (XCheckWindowEvent (mainDisplay, mainWindow,
               VisibilityChangeMask, &ev)) ;
         if (pinnedMainMenu) XMapRaised (mainDisplay, mainMenuWindow);
         for (i = 0; i < numExtraWins; i++)
            if (extraWinInfo[i].mapped && extraWinInfo[i].raise &&
                  extraWinInfo[i].window != None)
               XMapRaised (mainDisplay, extraWinInfo[i].window);
         XMapRaised (mainDisplay, nameBaseWin);
      }
      else if (input.type == KeyPress)
      {
         key_ev = &(input.xkey);
         XLookupString (key_ev, buf, 80-1, &key_sym, &c_stat);
         TranslateKeys (buf, &key_sym);

         if ((buf[0]=='\r' && (key_sym & 0xff)=='\r') ||
             (buf[0]=='\n' && (key_sym & 0xff)=='\n'))
         {
            changing = FALSE;
            button_selected = BUTTON_OK;
         }
         else if (buf[0]=='\033' && (key_sym & 0xff)=='\033')
         {
            changing = FALSE;
            button_selected = BUTTON_CANCEL;
         }
         else if ((buf[0] == '\b'  && (key_sym & 0xff)=='\b') ||
               (buf[0] == '\b' && (key_sym & 0xff)=='h') ||
               (buf[0] == '\177' && (key_sym & 0x7f)=='\177') ||
               key_sym==XK_Left)
         {
            switch (Which)
            {
               case PAGE_NAME:
                  if (nameMarked != INVALID && name_index != 0)
                  {
                     name[--name_index] = '\0';
                     if (nameMarked != INVALID)
                        strcpy (&(nameDspPtr[nameMarked])[leadingChars], name);

                     if (exposed)
                     {
                        RedrawNamePath (name, graph_start,
                              3*ROW_HEIGHT+defaultFontAsc+2);
                        RedrawDspWindow ();
                     }
                  }
                  break;
            }
         }
         else if (nameEntries != 0 && ((key_sym>'\040' && key_sym<='\177' &&
               (key_ev->state & ControlMask)) || key_sym==XK_Up ||
               key_sym==XK_Down))
         {
            if ((i = ControlChar (key_ev, key_sym)) != BAD)
            {
               if (i == INVALID)
               {
                  name[0] = '\0';
                  name_index = 0;
                  nameFirst = 0;
                  nameMarked = INVALID;
               }
               else
               {
                  strcpy (name, &(nameDspPtr[i])[leadingChars]);
                  name_index = strlen (name);

                  if (i < nameFirst)
                     nameFirst = i;
                  else if (i >= nameFirst+ITEM_DSPED)
                  {
                     if (i < nameEntries-ITEM_DSPED)
                        nameFirst = i;
                     else
                        nameFirst = nameEntries-ITEM_DSPED;
                  }
                  nameMarked = i;
               }

               if (exposed)
               {
                  RedrawNamePath (name, graph_start,
                        3*ROW_HEIGHT+defaultFontAsc+2);
                  RedrawNameScrollWin ();
                  RedrawDspWindow ();
               }
            }
         }
         else if (nameMarked != INVALID &&
               key_sym>='\040' && key_sym<='\177' && nameEntries != 0)
         {
            switch (Which)
            {
               case PAGE_NAME:
                  name[name_index++] = buf[0];
                  name[name_index] = '\0';
                  strcpy (&(nameDspPtr[nameMarked])[leadingChars], name);

                  if (exposed)
                  {
                     RedrawNamePath (name, graph_start,
                           3*ROW_HEIGHT+defaultFontAsc+2);
                     RedrawNameScrollWin ();
                     RedrawDspWindow ();
                  }
                  break;
            }
         }
      }
      else if (input.type == ButtonPress)
      {
         button_ev = &(input.xbutton);
         if (button_ev->window == nameBaseWin)
         {
            if (PointInBBox (button_ev->x, button_ev->y, buttonBBox[0]))
            {
               changing = FALSE;
               button_selected = BUTTON_OK;
            }
            else if (PointInBBox (button_ev->x, button_ev->y, buttonBBox[1]))
            {
               changing = FALSE;
               button_selected = BUTTON_CANCEL;
            }
         }
         else if (button_ev->window == nameScrollWin)
            NameScrollHandler (button_ev);
         else if (button_ev->window == nameDspWin)
         {
            int tmp_rc=NameDspHandler (button_ev);

            if (nameMarked != INVALID)
            {
               switch (Which)
               {
                  case PAGE_GOTO:
                     if (tmp_rc != INVALID)
                     {
                        changing = FALSE;
                        button_selected = BUTTON_OK;
                     }
                     break;
                  case PAGE_NAME:
                     strcpy (name, &(nameDspPtr[nameMarked])[leadingChars]);
                     name_index = strlen (name);
                     RedrawNamePath (name, graph_start,
                           3*ROW_HEIGHT+defaultFontAsc+2);
                     break;
               }
            }
         }
      }
   }

   if (button_selected == BUTTON_CANCEL) button_selected = 1;
   if (exposed && button_selected != INVALID)
      DisplayButton (nameBaseWin, buttonStr[button_selected], 8,
            &buttonBBox[button_selected], BUTTON_INVERT);

   XDestroyWindow (mainDisplay, nameBaseWin);
   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, drawWindow, 0, 0, 0, 0,
            (int)(ZOOMED_SIZE(drawWinW)>>1), (int)(ZOOMED_SIZE(drawWinH)>>1));
   return (button_selected == BUTTON_OK);
}

static
int BlankStr (s)
   register char	* s;
{
   while (*s == ' ') s++;
   return (*s == '\0');
}

void NamePages ()
{
   register int	i;
   DspList	* dsp_ptr;

   if (pageLayoutMode == PAGE_TILE)
   {
      Msg ("Can not do NamePages() in TILED page mode.");
      return;
   }
   MakeQuiescent ();

   dsp_ptr = PageNameListing ();
   nameDspPtr = MakeNameDspItemArray (nameEntries, dsp_ptr);
   nameFirst = 0;
   nameMarked = curPageNum-1;
   if (EditOrSelectPageNames ("Edit page names...", PAGE_NAME))
   {
      int		modified=FALSE;
      struct PageRec	* page_ptr;

      for (page_ptr=firstPage, i=0; page_ptr!=NULL;
            page_ptr=page_ptr->next, i++)
      {
         int	blank_str=BlankStr(&(nameDspPtr[i])[leadingChars]);

         if (page_ptr->name == NULL)
         {
            if (!blank_str)
            {
               modified = TRUE;
               page_ptr->name = (char*) calloc (strlen (
                     &(nameDspPtr[i])[leadingChars])+1, sizeof(char));
               strcpy (page_ptr->name, &(nameDspPtr[i])[leadingChars]);
            }
         }
         else
         {
            if (blank_str || strcmp (page_ptr->name,
                  &(nameDspPtr[i])[leadingChars]) != 0)
            {
               modified = TRUE;
               cfree (page_ptr->name);
               if (blank_str)
                  page_ptr->name = NULL;
               else
               {
                  page_ptr->name = (char*) calloc (strlen (
                        &(nameDspPtr[i])[leadingChars])+1, sizeof(char));
                  strcpy (page_ptr->name, &(nameDspPtr[i])[leadingChars]);
               }
            }
         }
      }
      if (modified)
      {
         SetFileModified (TRUE);
         RedrawTitleWindow ();
      }
   }
   cfree (dsp_ptr);
   cfree (*nameDspPtr);
   cfree (nameDspPtr);
   nameDspPtr = NULL;
   Msg ("");
}

void GotoPage ()
{
   char		msg[MAXSTRING+1];
   DspList	* dsp_ptr;
   int		rc;

   if (pageLayoutMode == PAGE_TILE)
   {
      Msg ("Can not do GotoPage() in TILED page mode.");
      return;
   }
   MakeQuiescent ();

   dsp_ptr = PageNameListing ();
   nameDspPtr = MakeNameDspItemArray (nameEntries, dsp_ptr);
   nameFirst = 0;
   nameMarked = curPageNum-1;

   rc = EditOrSelectPageNames ("Goto page...", PAGE_GOTO);

   cfree (dsp_ptr);
   cfree (*nameDspPtr);
   cfree (nameDspPtr);
   nameDspPtr = NULL;

   if (rc)
   {
      if (nameMarked++ == (-1))
      {
         Msg ("No page selected.");
         return;
      }
      if (nameMarked < 1 || nameMarked > lastPageNum)
      {
         sprintf (msg, "Invalid page number '%1d'.", nameMarked);
         Msg (msg);
         return;
      }
      if (nameMarked == curPageNum)
      {
         sprintf (msg, "Already at page %1d.", curPageNum);
         Msg (msg);
         return;
      }
      SetCurPage (nameMarked);

      sprintf (msg, "Current page is page %1d.", curPageNum);
      Msg (msg);
   }
   else
      Msg ("");
}

#define BEFORE_CUR_PAGE 0
#define AFTER_CUR_PAGE 1

static
void AddPage (AfterCurPage)
   int AfterCurPage;
{
   char			msg[MAXSTRING+1];
   int			n;
   struct PageRec	* page_ptr;

   if (pageLayoutMode == PAGE_TILE)
   {
      Msg ("Can not add a page in TILED page mode.");
      return;
   }
   if (firstCmd != NULL)
   {
      sprintf (msg, "Adding a page %s page %1d can not be undone.  %s",
            (AfterCurPage ? "after" : "before"), curPageNum,
            "Ok to proceed? [ync](y)");
      if (YesNoCancel (msg, CONFIRM_YES) != CONFIRM_YES) return;
      CleanUpCmds ();
   }
   n = (AfterCurPage ? curPageNum : curPageNum-1);

   MakeQuiescent ();

   for (curPageNum=1, curPage=firstPage; curPageNum <= n; curPageNum++)
      curPage=curPage->next;
   page_ptr = (struct PageRec *) calloc (1, sizeof(struct PageRec));
   page_ptr->top = page_ptr->bot = topObj = botObj = NULL;
   page_ptr->next = curPage;
   page_ptr->name = NULL;
   page_ptr->draw_orig_x = drawOrigX;
   page_ptr->draw_orig_y = drawOrigY;
   page_ptr->zoom_scale = zoomScale;
   page_ptr->zoomed_in = zoomedIn;
   if (curPage == NULL)
   {
      page_ptr->prev = lastPage;
      lastPage->next = page_ptr;
      lastPage = page_ptr;
   }
   else
   {
      page_ptr->prev = curPage->prev;
      if (curPage->prev == NULL)
         firstPage = page_ptr;
      else
         curPage->prev->next = page_ptr;
      curPage->prev = page_ptr;
   }
   curPage = page_ptr;
   lastPageNum++;
   topObj = curPage->top;
   botObj = curPage->bot;

   ClearAndRedrawDrawWindow ();
   RedrawTitleWindow ();
   SetFileModified (TRUE);
   ShowPage ();

   sprintf (msg, "Page %1d added.", curPageNum);
   Msg (msg);
}

void AddPageBefore ()
{
   AddPage (BEFORE_CUR_PAGE);
}

void AddPageAfter ()
{
   AddPage (AFTER_CUR_PAGE);
}

void DeleteCurPage ()
{
   char			msg[MAXSTRING+1];
   int			n;
   struct PageRec	* page_ptr;

   if (pageLayoutMode == PAGE_TILE)
   {
      Msg ("Can not delete a page in TILED page mode.");
      return;
   }
   if (lastPageNum == 1)
   {
      Msg ("Can not delete (the only) page.");
      return;
   }
   if (firstCmd != NULL || topObj != NULL)
   {
      sprintf (msg, "Deleting page %1d can not be undone.  %s",
            curPageNum, "Ok to proceed? [ync](y)");
      if (YesNoCancel (msg, CONFIRM_YES) != CONFIRM_YES) return;
      CleanUpCmds ();
   }
   n = curPageNum;
   SetFileModified (TRUE);

   MakeQuiescent ();

   if (curPage == firstPage)
   {
      page_ptr = firstPage = curPage->next;
      firstPage->prev = NULL;
   }
   else if (curPage == lastPage)
   {
      page_ptr = lastPage = curPage->prev;
      lastPage->next = NULL;
      curPageNum--;
   }
   else
   {
      curPage->next->prev = curPage->prev;
      curPage->prev->next = curPage->next;
      page_ptr = curPage->next;
   }
   curPage = page_ptr;
   lastPageNum--;
   topObj = curPage->top;
   botObj = curPage->bot;

   ClearAndRedrawDrawWindow ();
   RedrawTitleWindow ();
   ShowPage ();

   sprintf (msg, "Page %1d deleted.", n);
   Msg (msg);
}

void TogglePageLineShown ()
{
   if (pageLayoutMode == PAGE_STACK)
   {
      Msg ("Can not toggle page line shown in STACK page mode.");
      return;
   }
   pageLineShownInTileMode = !pageLineShownInTileMode;
   ClearAndRedrawDrawWindow ();
}

static
int JustSpecifyDrawingSize ()
{
   char	spec[MAXSTRING+1], * c_ptr, * part_1, msg[MAXSTRING+1];
   int	cols=0, rows=0;

   if (Dialog ("Please enter drawing size specification: [WxH]",
         "( <CR>: accept, <ESC>: cancel )", spec) == INVALID)
      return (FALSE);
   for (c_ptr = spec; *c_ptr != '\0'; c_ptr++)
      if (*c_ptr != ' ') break;
   if (*c_ptr == '\0') return (FALSE);
   part_1 = c_ptr;
   while (*c_ptr != '\0' && *c_ptr != 'x' && *c_ptr != ' ') c_ptr++;
   if (*c_ptr == 'x' || *c_ptr == ' ')
   {
      *c_ptr++ = '\0';
      cols = atoi (part_1);
      rows = atoi (c_ptr);
   }
   if (cols > 0 && rows > 0 && (cols*rows >= lastPageNum))
   {
      if (cols*rows < lastPageNum)
      {
         sprintf (msg, "Invalid drawing size:  W times H should be >= %1d",
               lastPageNum);
         Dialog (msg, "( <CR> or <ESC> to continue )", spec);
      }
      else
      {
         paperCol = cols;
         paperRow = rows;
         return (TRUE);
      }
   }
   else if (cols*rows < lastPageNum)
   {
      sprintf (msg, "Invalid drawing size:  W times H should be >= %1d",
            lastPageNum);
      Dialog (msg, "( <CR> or <ESC> to continue )", spec);
   }
   else
      Dialog ("Invalid drawing size specified.",
            "( <CR> or <ESC> to continue )", spec);
   return (FALSE);
}

void SpecifyDrawingSize ()
{
   if (pageLayoutMode != PAGE_TILE)
   {
      Msg ("Can only do SpecifyDrawingSize() in TILED page mode.");
      return;
   }
   if (JustSpecifyDrawingSize ())
   {
      UpdPageStyle (pageStyle);
      RedrawScrollBars ();
      UpdDrawWinBBox ();
      AdjSplineVs ();
      ClearAndRedrawDrawWindow ();

      ShowPage ();
   }
}

void PrintOnePage ()
{
   int	x, y;
   char	msg[81];

   if (whereToPrint == XBM_FILE)
   {
      sprintf (msg, "Can not do PrintOnePage() in %s format.",
            colorDump ? "XPM" : "XBM");
      Msg (msg);
      return;
   }
   MakeQuiescent ();

   switch (pageLayoutMode)
   {
      case PAGE_TILE:
         Msg ("Please point and click on a page...");
         PickAPoint (&x, &y);
         if (x>=0 && x<ZOOMED_SIZE(drawWinW) && y>=0 && y<ZOOMED_SIZE(drawWinH))
         {
            int	abs_x=ABS_X(x), abs_y=ABS_X(y);
            int	col = (int)(abs_x / onePageWidth);
            int	row = (int)(abs_y / onePageHeight);

            if (col >= paperCol || row >= paperRow)
               Msg ("Must select a point on the paper.");
            else
               DumpOnePageInTileMode (row, col);
         }
         else
            Msg ("Must select a point from the drawing canvas.");
         break;
      case PAGE_STACK:
         DumpOnePageInStackMode ();
         break;
   }
}

void PageSubMenu (index)
   int	index;
{
   switch (pageLayoutMode)
   {
      case PAGE_STACK:
         switch (index)
         {
            case PAGE_NEXT: NextPage (); break;
            case PAGE_PREV: PrevPage (); break;
            case PAGE_NAME: NamePages (); break;
            case PAGE_GOTO: GotoPage (); break;
            case PAGE_ADDBEFORE: AddPageBefore (); break;
            case PAGE_ADDAFTER: AddPageAfter (); break;
            case PAGE_DEL: DeleteCurPage (); break;
            case PAGE_PRINT_ONE_IN_STACK: PrintOnePage (); break;
         }
         break;
      case PAGE_TILE:
         switch (index)
         {
            case PAGE_TOGGLELINE: TogglePageLineShown (); break;
            case PAGE_SIZE: SpecifyDrawingSize (); break;
            case PAGE_PRINT_ONE: PrintOnePage (); break;
         }
         break;
   }
   UpdateSubMenu (MENU_PAGE);
}

void PageMenu (X, Y)
   int	X, Y;
{
   int	index=INVALID, * fore_colors, * valid, * init_rv;

   switch (pageLayoutMode)
   {
      case PAGE_STACK:
         DefaultColorArrays (MAXPAGESTACKMENUS, &fore_colors, &valid, &init_rv);
         activeMenu = MENU_PAGE;
         index = TextMenuLoop (X, Y, pageStackMenuStr, MAXPAGESTACKMENUS,
               fore_colors, valid, init_rv, SINGLECOLOR);
         break;
      case PAGE_TILE:
         DefaultColorArrays (MAXPAGETILEMENUS, &fore_colors, &valid, &init_rv);
         activeMenu = MENU_PAGE;
         index = TextMenuLoop (X, Y, pageTileMenuStr, MAXPAGETILEMENUS,
               fore_colors, valid, init_rv, SINGLECOLOR);
         break;
   }

   if (index != INVALID) PageSubMenu (index);
}

void DelAllPages ()
{
   register struct PageRec	* next_page=NULL;

   for (curPage = firstPage; curPage != NULL; curPage = next_page)
   {
      next_page = curPage->next;

      topObj = curPage->top;
      botObj = curPage->bot;
      DelAllObj ();
      if (curPage->name != NULL) cfree (curPage->name);
      cfree (curPage);
   }
   lastPageNum = 0;
   InitPage ();
}

static
int AnyObjOnPageBoundary ()
{
   register struct ObjRec	* obj_ptr;
   int				ltx, lty, rbx, rby, next_page_x, next_page_y;
   char				msg[MAXSTRING+1], dummy_str[MAXSTRING+1];

   for (obj_ptr = topObj; obj_ptr != NULL; obj_ptr = obj_ptr->next)
   {
      ltx = obj_ptr->obbox.ltx; lty = obj_ptr->obbox.lty;
      rbx = obj_ptr->obbox.rbx; rby = obj_ptr->obbox.rby;
      if (ltx < 0 || lty < 0 || rbx >= onePageWidth*paperCol ||
            rby >= onePageHeight*paperRow)
      {
         sprintf (msg, "%s.  %s.",
               "There are objects outside the page boundary",
               "Can not switch to TILED mode");
         Dialog (msg, "( <CR> or <ESC> to continue )", dummy_str);
         return (TRUE);
      }
      next_page_x = ((ltx % onePageWidth) == 0) ?
            onePageWidth * (round(ltx / onePageWidth) + 1) :
            onePageWidth * (((int)(ltx / onePageWidth)) + 1);
      next_page_y = ((lty % onePageHeight) == 0) ?
            onePageHeight * (round(lty / onePageHeight) + 1) :
            onePageHeight * (((int)(lty / onePageHeight)) + 1);
      if (rbx >= next_page_x || rby >= next_page_y)
      {
         sprintf (msg, "%s.  %s.",
               "There are objects on the page boundary",
               "Can not switch to TILED mode");
         Dialog (msg, "( <CR> or <ESC> to continue )", dummy_str);
         return (TRUE);
      }
   }
   return (FALSE);
}

static
int CalcStackPageNum (obj_ptr)
   struct ObjRec	* obj_ptr;
{
   int	ltx, lty, col, row;

   ltx = obj_ptr->obbox.ltx; lty = obj_ptr->obbox.lty;
   col = ((ltx % onePageWidth) == 0) ? round(ltx / onePageWidth) + 1 :
         ((int)(ltx / onePageWidth)) + 1;
   row = ((lty % onePageHeight) == 0) ? round(lty / onePageHeight) + 1 :
         ((int)(lty / onePageHeight)) + 1;
   return ((row-1) * paperCol + col);
}

static
void TileToStack ()
{
   register int			i;
   register struct ObjRec	* obj_ptr, * prev_obj;
   int				* dx=NULL, * dy=NULL, index;
   struct PageRec		* * page_handle=NULL;
   struct ObjRec		* saved_top_obj, * saved_bot_obj;

   saved_top_obj=curPage->top;
   saved_bot_obj=curPage->bot;
   curPage->top = curPage->bot = NULL;
   DelAllPages ();

   firstPage = lastPage = curPage = NULL;
   lastPageNum = paperCol * paperRow;
   InitPage ();

   page_handle = (struct PageRec **) calloc (lastPageNum,
         sizeof(struct PageRec *));
   dx = (int *)calloc(lastPageNum,sizeof(int));
   dy = (int *)calloc(lastPageNum,sizeof(int));
   for (curPage = firstPage, i=0; curPage != NULL; curPage = curPage->next, i++)
   {
      dx[i] = (i % paperCol) * onePageWidth;
      dy[i] = (((i % paperCol) == 0) ? round(i / paperCol):
            ((int)(i / paperCol))) * onePageHeight;
      page_handle[i] = curPage;
   }

   for (obj_ptr=saved_bot_obj; obj_ptr != NULL; obj_ptr = prev_obj)
   {
      prev_obj = obj_ptr->prev;
      index = CalcStackPageNum(obj_ptr)-1;
      curPage = page_handle[index];
      AddObj (NULL, curPage->top, obj_ptr);
      MoveObj (obj_ptr, -(dx[index]), -(dy[index]));
   }
   curPage = firstPage;
   topObj = curPage->top;
   botObj = curPage->bot;
   curPageNum = 1;
   paperCol = paperRow = 1;
   cfree (dx);
   cfree (dy);
   cfree (page_handle);
}

static
void StackToTile ()
{
   register struct ObjRec	* obj_ptr, * prev_obj;
   int				i, dx, dy;
   struct PageRec		* saved_first_page, * saved_last_page;
   struct PageRec		* tmp_page, * page_ptr;

   if (paperCol * paperRow < lastPageNum && !JustSpecifyDrawingSize ()) return;

   saved_first_page = firstPage;
   saved_last_page = lastPage;
   firstPage = lastPage = NULL;
   lastPageNum = 1;
   InitPage ();
   tmp_page = firstPage;

   i = 0;
   for (page_ptr = saved_first_page; page_ptr != NULL;
         page_ptr = page_ptr->next, i++)
   {
      dx = (i % paperCol) * onePageWidth;
      dy = (((i % paperCol) == 0) ? round(i / paperCol):
            ((int)(i / paperCol))) * onePageHeight;
      for (obj_ptr = page_ptr->bot; obj_ptr != NULL; obj_ptr = prev_obj)
      {
         prev_obj = obj_ptr->prev;
         AddObj (NULL, topObj, obj_ptr);
         MoveObj (obj_ptr, dx, dy);
      }
      page_ptr->top = page_ptr->bot = NULL;
   }
   firstPage = saved_first_page;
   lastPage = saved_last_page;
   DelAllPages ();
   firstPage = lastPage = curPage = tmp_page;
   curPageNum = lastPageNum = 1;
   pageLineShownInTileMode = TRUE;
   topObj = curPage->top;
   botObj = curPage->bot;
}

static
int OkToFlushUndoBuffer ()
{
   if (firstCmd != NULL)
   {
      char	msg[MAXSTRING+1];

      sprintf (msg, "%s.  %s? [ync](y)",
            "Changing the page layout mode can not be undone", "Ok to proceed");
      if (YesNoCancel (msg, CONFIRM_YES) == CONFIRM_YES)
      {
         CleanUpCmds ();
         return (TRUE);
      }
      return (FALSE);
   }
   return (TRUE);
}

static
int PreservePageNames ()
{
   struct PageRec	* page_ptr;

   for (page_ptr = firstPage; page_ptr != NULL; page_ptr = page_ptr->next)
      if (page_ptr->name != NULL && *page_ptr->name != '\0')
         break;
   if (page_ptr != NULL)
   {
      char msg[MAXSTRING];

      sprintf (msg, "%s  %s [ync](y)",
            "Switching from TILED to STACK mode loses all page names.",
            "Ok to proceed?" );
      if (YesNoCancel (msg, CONFIRM_YES) != CONFIRM_YES)
         return (FALSE);
   }
   return (TRUE);
}

void PageLayoutSubMenu (index)
   int	index;
{
   MakeQuiescent ();
   if (pageLayoutMode != index)
   {
      switch (index)
      {
         case PAGE_STACK:
            if (AnyObjOnPageBoundary ()) return;
            if (!OkToFlushUndoBuffer ()) return;

            pageLayoutMode = index;
            TileToStack ();
            Msg ("Page mode is STACKED.");
            break;
         case PAGE_TILE:
            if (!PreservePageNames ()) return;
            if (!JustSpecifyDrawingSize ()) return;
            if (!OkToFlushUndoBuffer ()) return;

            pageLayoutMode = index;
            StackToTile ();
            Msg ("Page mode is TILED.");
            break;
      }
      ShowPage ();
      ShowPageLayout ();
      UpdateSubMenu (MENU_PAGELAYOUT);
      DestroySubMenu (MENU_PAGE);

      UpdPageStyle (pageStyle);
      RedrawScrollBars ();
      UpdDrawWinBBox ();
      AdjSplineVs ();
      ClearAndRedrawDrawWindow ();
      RedrawTitleWindow ();
      SetFileModified (TRUE);
   }
}

void PageLayoutMenu (X, Y)
   int	X, Y;
{
   int	index, * fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXPAGELAYOUTMODES, &fore_colors, &valid, &init_rv);
   cfree (valid);
   init_rv[pageLayoutMode] = TRUE;
   activeMenu = MENU_PAGELAYOUT;
   index = PxMpMenuLoop (X, Y, choiceImageW, choiceImageH, MAXPAGELAYOUTMODES,
         1, MAXPAGELAYOUTMODES, fore_colors, pageLayoutPixmap, init_rv,
         SINGLECOLOR);

   if (index != INVALID) PageLayoutSubMenu (index);
}

void CleanUpPage ()
{
   DelAllPages ();
}

static int savedPageLayoutMode, savedPaperCol, savedPaperRow;
static int savedCurPageNum, savedLastPageNum;
static struct PageRec *savedFirstPage, *savedLastPage, *savedCurPage;

void PushPageInfo ()
{
   savedPageLayoutMode = pageLayoutMode;
   savedPaperCol = paperCol;
   savedPaperRow = paperRow;
   savedFirstPage = firstPage;
   savedLastPage = lastPage;
   savedCurPage = curPage;
   savedCurPageNum = curPageNum;
   savedLastPageNum = lastPageNum;
}

void PopPageInfo ()
{
   pageLayoutMode = savedPageLayoutMode;
   paperCol = savedPaperCol;
   paperRow = savedPaperRow;
   firstPage = savedFirstPage;
   lastPage = savedLastPage;
   curPage = savedCurPage;
   curPageNum = savedCurPageNum;
   lastPageNum = savedLastPageNum;
}
