/*
 * Author:	William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990, 1991, 1992, William Cheng.
 * 
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/kona/tangram/u/william/X11/TGIF2/RCS/poly.c,v 2.59.1.4 1993/01/03 23:34:24 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "attr.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "drawing.e"
#include "file.e"
#include "grid.e"
#include "mainloop.e"
#include "msg.e"
#include "obj.e"
#include "pattern.e"
#include "polygon.e"
#include "raster.e"
#include "rect.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "spline.e"
#include "xpixmap.e"

#define RETREAT (0.8)

int	polyDrawn = FALSE;

/* short	widthOfLine[] = { 0, 3,  6,  0 }; */
/* short	arrowHeadH[]  = { 3, 5,  10, 3 }; */
/* short	arrowHeadW[]  = { 8, 12, 22, 8 }; */

short	origWidthOfLine[] = { 1, 2,  3,  4,  5,  6,  7 };
short	origArrowHeadH[]  = { 3, 4,  5,  6,  7,  8,  9 };
short	origArrowHeadW[]  = { 8, 10, 12, 14, 18, 20, 22 };

short	* curWidthOfLine;
short	* curArrowHeadH;
short	* curArrowHeadW;

static struct PtRec	* lastPtPtr = NULL;

XPoint	* MakePolyVertex (XOff, YOff, NumVs, Vs)
   int			XOff, YOff, NumVs;
   register XPoint	* Vs;
{
   register XPoint	* v;
   register int		i;
   int			real_x_off, real_y_off;

   real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);

   v = (XPoint *) calloc (NumVs+1, sizeof(XPoint));
   if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   for (i = 0; i < NumVs; i++)
   {
      v[i].x = ZOOMED_SIZE(Vs[i].x-real_x_off);
      v[i].y = ZOOMED_SIZE(Vs[i].y-real_y_off);
   }
   return (v);
}

void CalcPolyBBox (OBBox, NumPts, V, Style, W, AW, AH, LtX, LtY, RbX, RbY)
   struct BBRec	OBBox;
   int		NumPts, Style, W, AW, AH, * LtX, * LtY, * RbX, * RbY;
   XPoint	* V;
{
   register int	x, y;
   int		ltx=OBBox.ltx, lty=OBBox.lty, rbx=OBBox.rbx, rby=OBBox.rby;
   int		dx, dy;
   double	len, sin, cos, w, h;

   dx = V[1].x - V[0].x;
   dy = V[1].y - V[0].y;
   if ((Style & LS_LEFT) && (dx != 0 || dy != 0))
   {
      len = (double) sqrt ((double)(dx*dx + dy*dy));
      sin = ((double)dy) / len;
      cos = ((double)dx) / len;

      w = (double)AW; h = (double)AH;

      x = round(V[0].x + w*cos - h*sin);
      y = round(V[0].y + w*sin + h*cos);
      if (x < ltx) ltx = x; if (y < lty) lty = y;
      if (x > rbx) rbx = x; if (y > rby) rby = y;

      x = round(V[0].x + w*cos + h*sin);
      y = round(V[0].y + w*sin - h*cos);
      if (x < ltx) ltx = x; if (y < lty) lty = y;
      if (x > rbx) rbx = x; if (y > rby) rby = y;

      w = ((double)AW)*RETREAT; h = ((double)W)/2.0;

      x = round(V[0].x + w*cos - h*sin);
      y = round(V[0].y + w*sin + h*cos);
      if (x < ltx) ltx = x; if (y < lty) lty = y;
      if (x > rbx) rbx = x; if (y > rby) rby = y;

      x = round(V[0].x + w*cos + h*sin);
      y = round(V[0].y + w*sin - h*cos);
      if (x < ltx) ltx = x; if (y < lty) lty = y;
      if (x > rbx) rbx = x; if (y > rby) rby = y;
   }

   dx = V[NumPts-1].x - V[NumPts-2].x;
   dy = V[NumPts-1].y - V[NumPts-2].y;
   if ((Style & LS_RIGHT) && (dx != 0 || dy != 0))
   {
      len = (double) sqrt ((double)(dx*dx + dy*dy));
      sin = ((double)dy) / len;
      cos = ((double)dx) / len;

      w = (double)AW; h = (double)AH;

      x = round(V[NumPts-1].x - w*cos + h*sin);
      y = round(V[NumPts-1].y - w*sin - h*cos);
      if (x < ltx) ltx = x; if (y < lty) lty = y;
      if (x > rbx) rbx = x; if (y > rby) rby = y;

      x = round(V[NumPts-1].x - w*cos - h*sin);
      y = round(V[NumPts-1].y - w*sin + h*cos);
      if (x < ltx) ltx = x; if (y < lty) lty = y;
      if (x > rbx) rbx = x; if (y > rby) rby = y;
   }

   *LtX = min(ltx, OBBox.ltx-W/2);
   *LtY = min(lty, OBBox.lty-W/2);
   *RbX = max(rbx, OBBox.rbx+W/2);
   *RbY = max(rby, OBBox.rby+W/2);
}

void UpdPolyBBox (ObjPtr, NumPts, V)
   struct ObjRec	* ObjPtr;
   int			NumPts;
   XPoint		* V;
{
   register int	i;
   int		ltx, lty, rbx, rby;

   ltx = rbx = V[0].x;
   lty = rby = V[0].y;

   for (i = 1; i < NumPts; i++)
   {
      if (V[i].x < ltx) ltx = V[i].x; if (V[i].y < lty) lty = V[i].y;
      if (V[i].x > rbx) rbx = V[i].x; if (V[i].y > rby) rby = V[i].y;
   }

   ObjPtr->x = ltx;
   ObjPtr->y = lty;
   ObjPtr->obbox.ltx = ltx;
   ObjPtr->obbox.lty = lty;
   ObjPtr->obbox.rbx = rbx;
   ObjPtr->obbox.rby = rby;
   AdjObjBBox (ObjPtr);
}

static
void CreatePolyObj (NumPts)
   int	NumPts;
{
   struct PtRec		* pt_ptr, * next_pt;
   struct PolyRec	* poly_ptr;
   struct ObjRec	* obj_ptr;
   register int		i;
   XPoint		* v;
   int			ltx, lty, rbx, rby;

   poly_ptr = (struct PolyRec *) calloc (1, sizeof(struct PolyRec));
   poly_ptr->n = NumPts;
   v = (XPoint *) calloc (NumPts+1, sizeof(XPoint));
   if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   pt_ptr = lastPtPtr;
   ltx = rbx = pt_ptr->x;
   lty = rby = pt_ptr->y;
   for (i = NumPts-1; i >= 0; i--, lastPtPtr = next_pt)
   {
      next_pt = lastPtPtr->next;
      v[i].x = ABS_X(lastPtPtr->x);
      v[i].y = ABS_Y(lastPtPtr->y);
      if (lastPtPtr->x < ltx) ltx = lastPtPtr->x;
      if (lastPtPtr->y < lty) lty = lastPtPtr->y;
      if (lastPtPtr->x > rbx) rbx = lastPtPtr->x;
      if (lastPtPtr->y > rby) rby = lastPtPtr->y;
      cfree (lastPtPtr);
   }

   poly_ptr->vlist = v;
   poly_ptr->svlist = poly_ptr->asvlist = NULL;
   poly_ptr->style = lineStyle;
   poly_ptr->width = curWidthOfLine[lineWidth];
   poly_ptr->aw = curArrowHeadW[lineWidth];
   poly_ptr->ah = curArrowHeadH[lineWidth];
   poly_ptr->pen = penPat;
   poly_ptr->curved = curSpline;
   poly_ptr->fill = objFill;
   poly_ptr->dash = curDash;
   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   obj_ptr->color = colorIndex;
   obj_ptr->type = OBJ_POLY;
   obj_ptr->obbox.ltx = obj_ptr->x = ABS_X(ltx);
   obj_ptr->obbox.lty = obj_ptr->y = ABS_Y(lty);
   obj_ptr->obbox.rbx = ABS_X(rbx);
   obj_ptr->obbox.rby = ABS_Y(rby);
   CalcPolyBBox (obj_ptr->obbox, NumPts, v, lineStyle, poly_ptr->width,
         poly_ptr->aw, poly_ptr->ah, &(obj_ptr->bbox.ltx), &(obj_ptr->bbox.lty),
         &(obj_ptr->bbox.rbx), &(obj_ptr->bbox.rby));
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->rotation = 0;
   obj_ptr->locked = FALSE;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   obj_ptr->detail.p = poly_ptr;
   AdjObjSplineVs (obj_ptr);
   AddObj (NULL, topObj, obj_ptr);
}

static
void ContinuePoly (OrigX, OrigY)
   int 	OrigX, OrigY;
   /* OrigX and OrigY are screen coordinates (scaled and translated). */
   /* OrigX and OrigY are also on grid. */
{
   XGCValues		values;
   XEvent		input, ev;
   XButtonEvent		* button_ev;
   XMotionEvent		* motion_ev;
   int			xor_pixel;
   int 			end_x, end_y, grid_x, grid_y, done = FALSE, num_pts = 1;
   int			last_x=OrigX, last_y=OrigY, n=2, sn=0, max_n=40;
   struct PtRec		* pt_ptr;
   XPoint		* v = NULL, * sv = NULL;

   xor_pixel = xorColorPixels[colorIndex];

   values.foreground = xor_pixel;
   values.function = GXxor;
   values.fill_style = FillSolid;
#ifdef NO_THIN_LINE
   values.line_width = 1;
#else
   values.line_width = 0;
#endif
   values.line_style = LineSolid;

   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = end_x = OrigX;
   grid_y = end_y = OrigY;
   if (curSpline == LT_SPLINE && splineRubberband)
   {
      v = (XPoint *) calloc (max_n+1, sizeof (XPoint));
      if (v == NULL) fprintf (stderr, "Can not calloc().\n");
      v[0].x = v[1].x = v[2].x = ABS_X(OrigX);
      v[0].y = v[1].y = v[2].y = ABS_Y(OrigY);
      sv = MakeSplinePolyVertex (&sn, drawOrigX, drawOrigY, n, v);
   }

   if (curChoice == FREEHAND)
   {
      Msg ("Release button to terminate freehand.");
      XGrabPointer (mainDisplay, drawWindow, FALSE,
            PointerMotionMask | ButtonReleaseMask,
            GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   }
   else
   {
      Msg ("Click middle or right button to end poly.");
      XGrabPointer (mainDisplay, drawWindow, FALSE,
            PointerMotionMask | ButtonPressMask,
            GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   }
   
   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == MotionNotify)
      {
         if (curSpline==LT_SPLINE && splineRubberband)
         {
#ifdef HP_LINE_BUG
            if (sn == 2)
            {
               XPoint hp_sv[3];

               hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
               hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
               hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
               XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                     CoordModeOrigin);
            }
            else
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
#else
            XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                  CoordModeOrigin);
#endif
         }
         else
            XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                  grid_y);
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         if (curChoice == FREEHAND)
         {
            grid_x = end_x;
            grid_y = end_y;
            MarkRulers (grid_x, grid_y);
            if (curSpline == LT_SPLINE && splineRubberband)
            {
               cfree (sv);
               v[n-1].x = v[n].x = ABS_X(grid_x);
               v[n-1].y = v[n].y = ABS_Y(grid_y);
               sv = MakeSplinePolyVertex (&sn, drawOrigX, drawOrigY, n, v);
#ifdef HP_LINE_BUG
               if (sn == 2)
               {
                  XPoint hp_sv[3];

                  hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
                  hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
                  hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
                  XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                        CoordModeOrigin);
               }
               else
                  XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                        CoordModeOrigin);
#else
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
#endif
            }
            else
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                     grid_y);
            while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;

            if (grid_x != last_x || grid_y != last_y)
            {
               num_pts++;
               pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
               pt_ptr->next = lastPtPtr;
               lastPtPtr = pt_ptr;
               pt_ptr->x = last_x = grid_x;
               pt_ptr->y = last_y = grid_y;
               if (curSpline == LT_SPLINE && splineRubberband)
               {
                  if (n >= max_n-2)
                  {
                     max_n += 40;
                     v = (XPoint *) realloc (v, sizeof(XPoint)*max_n+1);
                     if (v == NULL) fprintf (stderr, "Can not realloc().\n");
                  }
#ifdef HP_LINE_BUG
                  if (sn == 2)
                  {
                     XPoint hp_sv[3];

                     hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
                     hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
                     hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
                     XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                           CoordModeOrigin);
                  }
                  else
                     XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                           CoordModeOrigin);
#else
                  XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                        CoordModeOrigin);
#endif
                  cfree (sv);
                  v[n].x = v[n+1].x = ABS_X(grid_x);
                  v[n].y = v[n+1].y = ABS_Y(grid_y);
                  n++;
                  sv = MakeSplinePolyVertex (&sn, drawOrigX, drawOrigY, n, v);
#ifdef HP_LINE_BUG
                  if (sn == 2)
                  {
                     XPoint hp_sv[3];

                     hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
                     hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
                     hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
                     XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                           CoordModeOrigin);
                  }
                  else
                     XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                           CoordModeOrigin);
#else
                  XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                        CoordModeOrigin);
#endif
               }
            }
            OrigX = grid_x; OrigY = grid_y;
         }
         else
         {
            GridXY (end_x, end_y, &grid_x, &grid_y);
            MarkRulers (grid_x, grid_y);
            if (curSpline == LT_SPLINE && splineRubberband)
            {
               cfree (sv);
               v[n-1].x = v[n].x = ABS_X(grid_x);
               v[n-1].y = v[n].y = ABS_Y(grid_y);
               sv = MakeSplinePolyVertex (&sn, drawOrigX, drawOrigY, n, v);
#ifdef HP_LINE_BUG
               if (sn == 2)
               {
                  XPoint hp_sv[3];

                  hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
                  hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
                  hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
                  XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                        CoordModeOrigin);
               }
               else
                  XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                        CoordModeOrigin);
#else
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
#endif
            }
            else
               XDrawLine (mainDisplay, drawWindow, drawGC, OrigX, OrigY, grid_x,
                     grid_y);
            while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
         }
      }
      else if (input.type == ButtonPress && curChoice != FREEHAND)
      {
         button_ev = &(input.xbutton);

         end_x = button_ev->x;
         end_y = button_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);

         if (grid_x != last_x || grid_y != last_y)
         {
            num_pts++;
            pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
            pt_ptr->next = lastPtPtr;
            lastPtPtr = pt_ptr;
            pt_ptr->x = last_x = grid_x;
            pt_ptr->y = last_y = grid_y;
            if (curSpline == LT_SPLINE && splineRubberband)
            {
               if (n >= max_n-2)
               {
                  max_n += 40;
                  v = (XPoint *) realloc (v, sizeof(XPoint)*max_n+1);
                  if (v == NULL) fprintf (stderr, "Can not realloc().\n");
               }
#ifdef HP_LINE_BUG
               if (sn == 2)
               {
                  XPoint hp_sv[3];

                  hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
                  hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
                  hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
                  XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                        CoordModeOrigin);
               }
               else
                  XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                        CoordModeOrigin);
#else
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
#endif
               cfree (sv);
               v[n].x = v[n+1].x = ABS_X(grid_x);
               v[n].y = v[n+1].y = ABS_Y(grid_y);
               n++;
               sv = MakeSplinePolyVertex (&sn, drawOrigX, drawOrigY, n, v);
#ifdef HP_LINE_BUG
               if (sn == 2)
               {
                  XPoint hp_sv[3];

                  hp_sv[0].x = sv[0].x; hp_sv[0].y = sv[0].y;
                  hp_sv[1].x = sv[0].x; hp_sv[1].y = sv[0].y;
                  hp_sv[2].x = sv[1].x; hp_sv[2].y = sv[1].y;
                  XDrawLines (mainDisplay, drawWindow, drawGC, hp_sv, 3,
                        CoordModeOrigin);
               }
               else
                  XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                        CoordModeOrigin);
#else
               XDrawLines (mainDisplay, drawWindow, drawGC, sv, sn,
                     CoordModeOrigin);
#endif
            }
         }

         switch (button_ev->button)
         {
            case Button1 : OrigX = grid_x; OrigY = grid_y; break;
            case Button2 : done = TRUE; break;
            case Button3 : done = TRUE; break;
         }
      }
      else if (input.type == ButtonRelease && curChoice == FREEHAND)
      {
         button_ev = &(input.xbutton);

         if (grid_x != last_x || grid_y != last_y)
         {
            num_pts++;
            pt_ptr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
            pt_ptr->next = lastPtPtr;
            lastPtPtr = pt_ptr;
            pt_ptr->x = last_x = grid_x;
            pt_ptr->y = last_y = grid_y;
         }
         done = TRUE;
      }
   }
   XUngrabPointer (mainDisplay, CurrentTime);
   Msg ("");

   if (curSpline == LT_SPLINE && splineRubberband) { cfree (v); cfree (sv); }

   if (num_pts > 1)
   {
      CreatePolyObj (num_pts);
      RecordNewObjCmd ();
      RedrawAnArea (botObj, topObj->bbox.ltx-GRID_ABS_SIZE(1),
            topObj->bbox.lty-GRID_ABS_SIZE(1),
            topObj->bbox.rbx+GRID_ABS_SIZE(1),
            topObj->bbox.rby+GRID_ABS_SIZE(1));
      polyDrawn = TRUE;
      SetFileModified (TRUE);
   }
}

void DrawPoly (input)
   XEvent	* input;
{
   int		mouse_x, mouse_y, grid_x, grid_y;
   XButtonEvent	* button_ev;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = input->xbutton.x;
      mouse_y = input->xbutton.y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      lastPtPtr = (struct PtRec *) calloc (1, sizeof(struct PtRec));
      lastPtPtr->x = grid_x;
      lastPtPtr->y = grid_y;
      lastPtPtr->next = NULL;
      ContinuePoly (grid_x, grid_y);
   } 
}

static
void DumpArrow (FP, TailV, HeadV, ArrowW, ArrowH, Pen, ColorIndex)
   FILE		* FP;
   XPoint	TailV, HeadV;
   int		ArrowW, ArrowH, Pen, ColorIndex;
{
   int		i, dx, dy;
   struct BBRec	bbox;
   XPoint	v[2];
   double	len, sin, cos;

   dx = HeadV.x - TailV.x;
   dy = HeadV.y - TailV.y;

   if (dx == 0 && dy == 0) return;

   fprintf (FP, "gsave\n");

   if (colorDump)
   {
      len = (double) sqrt ((double)(dx*dx + dy*dy));
      sin = ((double)dy) / len;
      cos = ((double)dx) / len;

      v[0].x = round(HeadV.x - ArrowW*cos + ArrowH*sin);
      v[0].y = round(HeadV.y - ArrowW*sin - ArrowH*cos);
      v[1].x = round(HeadV.x - ArrowW*cos - ArrowH*sin);
      v[1].y = round(HeadV.y - ArrowW*sin + ArrowH*cos);

      bbox.ltx = bbox.rbx = HeadV.x;
      bbox.lty = bbox.rby = HeadV.y;

      for (i = 0; i < 2; i++)
      {
         if (v[i].x < bbox.ltx) bbox.ltx = v[i].x;
         if (v[i].y < bbox.lty) bbox.lty = v[i].y;
         if (v[i].x > bbox.rbx) bbox.rbx = v[i].x;
         if (v[i].y > bbox.rby) bbox.rby = v[i].y;
      }
      fprintf (FP, "   newpath\n");
      fprintf (FP, "      %1d %1d %1d %1d %1d %1d tgifarrowtip\n",
            HeadV.x, HeadV.y, ArrowW, ArrowH, dx, dy);
      fprintf (FP, "   1 setgray closepath fill\n");
      fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[ColorIndex].red/maxRGB),
            ((float)tgifColors[ColorIndex].green/maxRGB),
            ((float)tgifColors[ColorIndex].blue/maxRGB));
   }
   else
   {
      switch (Pen)
      {
         case SOLIDPAT: break;
         case BACKPAT: break;
         default:
            GrayCheck (Pen);
            if (useGray)
               fprintf (FP, "   %s setgray\n", GrayStr(Pen));
            else
               fprintf (FP, "   pat%1d %s\n", Pen, patternStr);
            break;
      }
   }

   if (!(colorDump && Pen==BACKPAT))
   {
      fprintf (FP, "   newpath\n");
      fprintf (FP, "      %1d %1d %1d %1d %1d %1d tgifarrowtip\n",
            HeadV.x, HeadV.y, ArrowW, ArrowH, dx, dy);
   }

   if (colorDump)
   {
      switch (Pen)
      {
         case SOLIDPAT: fprintf (FP, "   closepath fill\n"); break;
         case BACKPAT: break;
         default:
            fprintf (FP, "   closepath eoclip\n");
            DumpPatFill (FP, Pen, 8, bbox, "   ");
            break;
      }
   }
   else
   {
      switch (Pen)
      {
         case SOLIDPAT: fprintf (FP, "   closepath fill\n"); break;
         case BACKPAT: fprintf (FP, "   closepath 1 setgray fill\n"); break;
         default: fprintf (FP, "   closepath fill\n"); break;
      }
   }
   fprintf (FP, "grestore\n");
}

static
void DumpPolyPath (FP, ObjPtr, Vs, NumPts, Style, Width, ArrowWidth,
      ArrowHeight, Pen, Curved, Dash)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
   XPoint			* Vs;
   int				NumPts, Style, Width, ArrowWidth, ArrowHeight;
   int				Pen, Curved, Dash;
{
   register int	i, dx, dy;
   int		w, aw;

   w = Width;
   aw = ArrowWidth;

   if (Width != 1) fprintf (FP, "   %1d setlinewidth\n", w);
   if (Dash != 0)
   {
      fprintf (FP, "   [");
      for (i = 0; i < dashListLength[Dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[Dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[Dash][dashListLength[Dash]-1]));
   }

   fprintf (FP, "   newpath\n      %1d %1d moveto\n", Vs[0].x, Vs[0].y);
   if (Style & LS_LEFT)
   {
      dx = Vs[1].x - Vs[0].x;
      dy = Vs[1].y - Vs[0].y;
      if (dx != 0 || dy != 0)
         fprintf (FP, "      %1d %1d atan dup cos %1d mul exch sin %1d %s\n",
               dy, dx, aw, aw, "mul rmoveto");
   }
   if (Style & LS_RIGHT)
   {
      if (Curved == LT_SPLINE && NumPts != 2)
         DumpCurvedPolyPoints (FP, NumPts, Vs, 6);
      else
         DumpPoints (FP, NumPts-1, Vs, 6);

      dx = Vs[NumPts-1].x - Vs[NumPts-2].x;
      dy = Vs[NumPts-1].y - Vs[NumPts-2].y;
      if (dx != 0 || dy != 0)
      {
         fprintf (FP, "      %1d %1d atan dup cos %1d mul %1d exch sub\n",
               dy, dx, aw, Vs[NumPts-1].x);
         fprintf (FP, "      exch sin %1d mul %1d exch sub",
               aw, Vs[NumPts-1].y);

         if (Curved == LT_SPLINE && NumPts != 2)
            fprintf (FP, " curveto");
         else
            fprintf (FP, " lineto");
      }
      fprintf (FP, "\n");
   }
   else if (Curved == LT_SPLINE && NumPts != 2)
   {
      DumpCurvedPolyPoints (FP, NumPts, Vs, 6);
      fprintf (FP, "      %1d %1d curveto\n",Vs[NumPts-1].x,Vs[NumPts-1].y);
   }
   else
      DumpPoints (FP, NumPts, Vs, 6);

   switch (Pen)
   {
      case SOLIDPAT: fprintf (FP, "   stroke\n"); break;
      case BACKPAT: fprintf (FP, "   1 setgray stroke\n"); break;
      default:
         if (colorDump)
         {
            fprintf (FP, "   flattenpath strokepath clip\n");
            DumpPatFill (FP, Pen, 8, ObjPtr->bbox, "   ");
         }
         else
         {
            GrayCheck (Pen);
            if (useGray)
               fprintf (FP, "   %s setgray\n", GrayStr(Pen));
            else
               fprintf (FP, "   pat%1d %s\n", Pen, patternStr);
            fprintf (FP, "   stroke\n");
         }
         break;
   }
   if (Dash != 0) fprintf (FP, "   [] 0 setdash\n");
   if (Width != 1) fprintf (FP, "   1 setlinewidth\n");
}

void DumpPolyObj (FP, ObjPtr)
   FILE				* FP;
   register struct ObjRec	* ObjPtr;
{
   XPoint	* v;
   int		num_pts, fill, pen, width, curved, dash, color_index, style;
   int		aw, ah, rotation;

   fill = ObjPtr->detail.p->fill;
   width = ObjPtr->detail.p->width;
   aw = ObjPtr->detail.p->aw;
   ah = ObjPtr->detail.p->ah;
   pen = ObjPtr->detail.p->pen;
   style = ObjPtr->detail.p->style;
   curved = ObjPtr->detail.p->curved;
   dash = ObjPtr->detail.p->dash;
   rotation = ObjPtr->rotation;
   v = ObjPtr->detail.p->vlist;
   num_pts = ObjPtr->detail.p->n;

   if (fill == NONEPAT && pen == NONEPAT) return;

   fprintf (FP, "%% POLY/OPEN-SPLINE\n");
   color_index = ObjPtr->color;
   if (colorDump)
      fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
            ((float)tgifColors[color_index].red/maxRGB),
            ((float)tgifColors[color_index].green/maxRGB),
            ((float)tgifColors[color_index].blue/maxRGB));

   if (fill != NONEPAT && num_pts > 2)
   {
      switch (curved)
      {
         case LT_STRAIGHT:
            switch (fill)
            {
               case SOLIDPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpPoints (FP, num_pts, v, 3);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                  {
                     GrayCheck (fill);
                     if (useGray)
                        fprintf (FP, "   %s setgray\n", GrayStr(fill));
                     else
                        fprintf (FP, "   pat%1d %s\n", fill, patternStr);
                  }
                  else
                  {
                     fprintf (FP, "   newpath\n");
                     fprintf (FP, "      %1d %1d moveto\n", v[0].x, v[0].y);
                     DumpPoints (FP, num_pts, v, 6);
                     fprintf (FP, "   closepath 1 setgray eofill\n");
                     fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  }
                  fprintf (FP, "   newpath\n");
                  fprintf (FP, "      %1d %1d moveto\n", v[0].x,v[0].y);
                  DumpPoints (FP, num_pts, v, 6);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
         case LT_SPLINE:
            switch (fill)
            {
               case SOLIDPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpCurvedPolyPoints (FP, num_pts, v, 3);
                  fprintf (FP, "   %1d %1d curveto\n",
                        v[num_pts-1].x, v[num_pts-1].y);
                  fprintf (FP, "closepath eofill\n");
                  break;
               case BACKPAT:
                  fprintf (FP, "newpath\n");
                  fprintf (FP, "   %1d %1d moveto\n", v[0].x, v[0].y);
                  DumpCurvedPolyPoints (FP, num_pts, v, 3);
                  fprintf (FP, "   %1d %1d curveto\n",
                        v[num_pts-1].x, v[num_pts-1].y);
                  fprintf (FP, "closepath 1 setgray eofill\n");
                  if (colorDump)
                     fprintf (FP, "%.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  else
                     fprintf (FP, "0 setgray\n");
                  break;
               default: /* patterned */
                  fprintf (FP, "gsave\n");
                  if (!colorDump)
                  {
                     GrayCheck (fill);
                     if (useGray)
                        fprintf (FP, "   %s setgray\n", GrayStr(fill));
                     else
                        fprintf (FP, "   pat%1d %s\n", fill, patternStr);
                  }
                  else
                  {
                     fprintf (FP, "   newpath\n");
                     fprintf (FP, "      %1d %1d moveto\n", v[0].x, v[0].y);
                     DumpCurvedPolyPoints (FP, num_pts, v, 6);
                     fprintf (FP, "      %1d %1d curveto\n",
                           v[num_pts-1].x, v[num_pts-1].y);
                     fprintf (FP, "   closepath 1 setgray eofill\n");
                     fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
                           ((float)tgifColors[color_index].red/maxRGB),
                           ((float)tgifColors[color_index].green/maxRGB),
                           ((float)tgifColors[color_index].blue/maxRGB));
                  }
                  fprintf (FP, "   newpath\n");
                  fprintf (FP, "      %1d %1d moveto\n", v[0].x,v[0].y);
                  DumpCurvedPolyPoints (FP, num_pts, v, 6);
                  fprintf (FP, "   %1d %1d curveto\n",
                        v[num_pts-1].x, v[num_pts-1].y);
                  if (colorDump)
                  {
                     fprintf (FP, "   closepath eoclip\n");
                     DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
                  }
                  else
                     fprintf (FP, "   closepath eofill\n");
                  fprintf (FP, "grestore\n");
                  break;
            }
            break;
      }
   }

   if (pen == NONEPAT) { fprintf (FP, "\n"); return; }

   fprintf (FP, "gsave\n");

   if (colorDump && pen > BACKPAT)
   {
      DumpPolyPath (FP,ObjPtr,v,num_pts,style,width,aw,ah,BACKPAT,curved,0);
      if (colorDump)
         fprintf (FP, "   %.3f %.3f %.3f setrgbcolor\n",
               ((float)tgifColors[color_index].red/maxRGB),
               ((float)tgifColors[color_index].green/maxRGB),
               ((float)tgifColors[color_index].blue/maxRGB));
   }
   DumpPolyPath (FP,ObjPtr,v,num_pts,style,width,aw,ah,pen,curved,dash);

   fprintf (FP, "grestore\n");

   switch (style)
   {
      case LS_PLAIN: break;
      case LS_LEFT: DumpArrow (FP, v[1], v[0], aw, ah, pen, color_index); break;
      case LS_RIGHT:
         DumpArrow (FP, v[num_pts-2], v[num_pts-1], aw, ah, pen, color_index);
         break;
      case LS_DOUBLE:
         DumpArrow (FP, v[1], v[0], aw, ah, pen, color_index);
         DumpArrow (FP, v[num_pts-2], v[num_pts-1], aw, ah, pen, color_index);
         break;
   }
   fprintf (FP, "\n");
}

void DrawPolyObj (Win, XOff, YOff, ObjPtr)
   Window		Win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   register struct PolyRec	* poly_ptr = ObjPtr->detail.p;
   XPoint			* v, tmp_v[4];
   XPoint			v0, v1, vnminus2, vnminus1;
   int				pen, width, pixel, fill, n, dash;
   int				real_x_off, real_y_off;
   int				style, aw, ah, num_pts;
   int				left_dx,left_dy,right_dx,right_dy;
   double			len, sin, cos;
   XGCValues			values;

   n = poly_ptr->n;
   fill = poly_ptr->fill;
   width = poly_ptr->width;
   aw = poly_ptr->aw;
   ah = poly_ptr->ah;
   pen = poly_ptr->pen;
   style = poly_ptr->style;
   dash = poly_ptr->dash;
   pixel = colorPixels[ObjPtr->color];

   if (fill == NONEPAT && pen == NONEPAT) return;

   real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);

   v = poly_ptr->svlist;
   num_pts = poly_ptr->sn;

   v[num_pts].x = v[0].x; v[num_pts].y = v[0].y;

   if (fill != NONEPAT)
   {
      values.foreground = (fill == BACKPAT) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillPolygon (mainDisplay, Win, drawGC, v, num_pts+1, Complex,
            CoordModeOrigin);
   }

   if (pen == NONEPAT) return;

   v0.x = ZOOMED_SIZE(poly_ptr->vlist[0].x-real_x_off);
   v0.y = ZOOMED_SIZE(poly_ptr->vlist[0].y-real_y_off);
   v1.x = ZOOMED_SIZE(poly_ptr->vlist[1].x-real_x_off);
   v1.y = ZOOMED_SIZE(poly_ptr->vlist[1].y-real_y_off);
   vnminus2.x = ZOOMED_SIZE(poly_ptr->vlist[n-2].x-real_x_off);
   vnminus2.y = ZOOMED_SIZE(poly_ptr->vlist[n-2].y-real_y_off);
   vnminus1.x = ZOOMED_SIZE(poly_ptr->vlist[n-1].x-real_x_off);
   vnminus1.y = ZOOMED_SIZE(poly_ptr->vlist[n-1].y-real_y_off);

   aw = ZOOMED_SIZE(aw); if (aw == 0) aw = 1;
   ah = ZOOMED_SIZE(ah); if (ah == 0) ah = 1;

   values.foreground = (pen == BACKPAT) ? myBgPixel : pixel;
   values.function = GXcopy;
   values.fill_style = FillOpaqueStippled;
   values.stipple = patPixmap[pen];
   values.line_width = ZOOMED_SIZE(width);
#ifdef NO_THIN_LINE
   if (values.line_width < 1) values.line_width = 1;
#endif
   values.join_style = JoinBevel;
   if (dash != 0)
   {
      XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
            dashListLength[dash]);
      values.line_style = LineOnOffDash;
   }
   else
      values.line_style = LineSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCStipple | GCLineWidth |
         GCLineStyle | GCJoinStyle, &values);

   left_dx = v1.x - v0.x;
   left_dy = v1.y - v0.y;

   if ((style & LS_LEFT) && (left_dx != 0 || left_dy != 0))
   {  /* adjust the first point */
      len = (double) sqrt((double)(left_dx*left_dx+left_dy*left_dy));
      sin = ((double)left_dy)/len;
      cos = ((double)left_dx)/len;

      tmp_v[0].x = tmp_v[3].x = v0.x;
      tmp_v[0].y = tmp_v[3].y = v0.y;
      tmp_v[1].x = round(v0.x + aw*cos - ah*sin);
      tmp_v[1].y = round(v0.y + aw*sin + ah*cos);
      tmp_v[2].x = round(v0.x + aw*cos + ah*sin);
      tmp_v[2].y = round(v0.y + aw*sin - ah*cos);

      XFillPolygon (mainDisplay, Win, drawGC, tmp_v, 4, Convex,
            CoordModeOrigin);
   }

   right_dx = vnminus1.x - vnminus2.x;
   right_dy = vnminus1.y - vnminus2.y;

   if ((style & LS_RIGHT) && (right_dx != 0 || right_dy != 0))
   {  /* adjust the last point */
      len = (double) sqrt((double)(right_dx*right_dx+right_dy*right_dy));
      sin = ((double)right_dy)/len;
      cos = ((double)right_dx)/len;

      tmp_v[0].x = tmp_v[3].x = vnminus1.x;
      tmp_v[0].y = tmp_v[3].y = vnminus1.y;
      tmp_v[1].x = round(vnminus1.x - aw*cos + ah*sin);
      tmp_v[1].y = round(vnminus1.y - aw*sin - ah*cos);
      tmp_v[2].x = round(vnminus1.x - aw*cos - ah*sin);
      tmp_v[2].y = round(vnminus1.y - aw*sin + ah*cos);

      XFillPolygon (mainDisplay, Win, drawGC, tmp_v, 4, Convex,
            CoordModeOrigin);
   }

   if (style != LS_PLAIN)
   {
#ifdef HP_LINE_BUG
      if (poly_ptr->asn == 2)
      {
         XPoint hp_sv[3];

         hp_sv[0].x=poly_ptr->asvlist[0].x; hp_sv[0].y=poly_ptr->asvlist[0].y;
         hp_sv[1].x=poly_ptr->asvlist[0].x; hp_sv[1].y=poly_ptr->asvlist[0].y;
         hp_sv[2].x=poly_ptr->asvlist[1].x; hp_sv[2].y=poly_ptr->asvlist[1].y;
         XDrawLines (mainDisplay, Win, drawGC, hp_sv, 3, CoordModeOrigin);
      }
      else
         XDrawLines (mainDisplay, Win, drawGC, poly_ptr->asvlist, poly_ptr->asn,
               CoordModeOrigin);
#else
      XDrawLines (mainDisplay, Win, drawGC, poly_ptr->asvlist, poly_ptr->asn,
            CoordModeOrigin);
#endif
   }
   else
   {
#ifdef HP_LINE_BUG
      if (num_pts == 2)
      {
         XPoint hp_sv[3];

         hp_sv[0].x = v[0].x; hp_sv[0].y = v[0].y;
         hp_sv[1].x = v[0].x; hp_sv[1].y = v[0].y;
         hp_sv[2].x = v[1].x; hp_sv[2].y = v[1].y;
         XDrawLines (mainDisplay, Win, drawGC, hp_sv, 3, CoordModeOrigin);
      }
      else
         XDrawLines (mainDisplay, Win, drawGC, v, num_pts, CoordModeOrigin);
#else
      XDrawLines (mainDisplay, Win, drawGC, v, num_pts, CoordModeOrigin);
#endif
   }
}

void SavePolyObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int		i, n;
   int			count;
   struct PolyRec	* poly_ptr = ObjPtr->detail.p;

   n = poly_ptr->n;
   fprintf (FP, "poly('%s',%1d,[\n\t",
         colorMenuItems[ObjPtr->color], poly_ptr->n);
   for (i = 0, count = 0; i < n-1; i++)
   {
      fprintf (FP, "%1d,%1d,", poly_ptr->vlist[i].x, poly_ptr->vlist[i].y);
      if (++count == 8)
      {
         count = 0;
         fprintf (FP, "\n\t");
      }
   }
   fprintf (FP, "%1d,%1d],", poly_ptr->vlist[n-1].x, poly_ptr->vlist[n-1].y);

   fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,", poly_ptr->style,
         poly_ptr->width, poly_ptr->pen, ObjPtr->id, poly_ptr->curved,
         poly_ptr->fill, poly_ptr->dash, ObjPtr->rotation, poly_ptr->aw,
         poly_ptr->ah, ObjPtr->locked);
   SaveAttrs (FP, ObjPtr->lattr);
   fprintf (FP, ")");
}

#define GETVALUE(val,name) ScanValue("%d", (char *) &(val), name, "poly")

void ReadPolyObj (FP, Inbuf, ObjPtr)
   FILE			* FP;
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   register int		i;
   struct PolyRec	* poly_ptr;
   XPoint		* v;
   char			color_str[20], * s, inbuf[MAXSTRING+1], msg[MAXSTRING];
   int			num_pts, ltx=0, lty=0, rbx=0, rby=0, x, y, id=0;
   int			initialized, rotation, count, new_alloc;
   int			style, width=0, pen, curved, fill, dash, locked=FALSE;
   int			aw=origArrowHeadW[6], ah=origArrowHeadH[6];

   *ObjPtr = NULL;

   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_str);

   InitScan (s, "\t\n, []");

   if (GETVALUE (num_pts, "number of points") == INVALID)
      return;

   if (num_pts <= 0)
   {
      (void) sprintf (msg, "%s, %d:  Invalid number of points in poly",
            scanFileName, scanLineNum);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      return;
   }

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   poly_ptr = (struct PolyRec *) calloc (1, sizeof(struct PolyRec));

   if (num_pts == 1)
   {
      v = (XPoint *) calloc (4, sizeof(XPoint));
      if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   }
   else
   {
      v = (XPoint *) calloc (num_pts+1, sizeof(XPoint));
      if (v == NULL) fprintf (stderr, "Can not calloc().\n");
   }

   initialized = FALSE;

   if (fileVersion <= 13)
   {
      for (i = 0; i < num_pts; i++)
      {
         if (GETVALUE (x, "x") == INVALID || GETVALUE (y, "y") == INVALID)
         {
            cfree (*ObjPtr);
            cfree (poly_ptr);
            cfree (v);
            *ObjPtr = NULL;
            return;
         }
         v[i].x = x; v[i].y = y;
         if (!initialized)
         {
            initialized = TRUE;
            ltx = rbx = x; lty = rby = y;
         }
         else
         {
            if (x < ltx) ltx = x; if (y < lty) lty = y;
            if (x > rbx) rbx = x; if (y > rby) rby = y;
         }
      }
   }
   else
   {
      fgets (inbuf, MAXSTRING, FP);
      scanLineNum++;
      s = inbuf;
      InitScan (s, "\t\n, []");
      for (i = 0, count = 0; i < num_pts; i++)
      {
         if (GETVALUE (x, "x") == INVALID || GETVALUE (y, "y") == INVALID)
         {
            cfree (*ObjPtr);
            cfree (poly_ptr);
            cfree (v);
            *ObjPtr = NULL;
            return;
         }
         v[i].x = x; v[i].y = y;
         if (!initialized)
         {
            initialized = TRUE;
            ltx = rbx = x; lty = rby = y;
         }
         else
         {
            if (x < ltx) ltx = x; if (y < lty) lty = y;
            if (x > rbx) rbx = x; if (y > rby) rby = y;
         }
         if (++count == 8 && i != num_pts-1)
         {
            count = 0;
            fgets (inbuf, MAXSTRING, FP);
            scanLineNum++;
            s = inbuf;
            InitScan (s, "\t\n, []");
         }
      }
   }

   if (num_pts == 1)
   {
      sprintf (msg, "%s (%1d,%1d) converted to double point poly.",
            "Single point poly", v[0].x, v[0].y);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      v[1].x = v[0].x;
      v[1].y = v[0].y;
      num_pts = 2;
   }

   poly_ptr->n = num_pts;

   dash = 0;
   rotation = 0;
   if (fileVersion == INVALID)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      id = objId++;
      fill = NONEPAT;
      if (width == LINE_CURVED)
      {
         width = 0;
         curved = TRUE;
      }
      else
         curved = FALSE;
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
   }
   else if (fileVersion <= 3)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
      fill = NONEPAT;
      if (width == LINE_CURVED)
      {
         width = 0;
         curved = TRUE;
      }
      else
         curved = FALSE;
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
   }
   else if (fileVersion <= 4)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
      fill = NONEPAT;
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
   }
   else if (fileVersion <= 5)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
   }
   else if (fileVersion <= 8)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 13)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 16)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 25)
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID ||
          GETVALUE (aw,       "arrow head width") == INVALID ||
          GETVALUE (ah,       "arrow head height") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else
   {
      if (GETVALUE (style,    "style") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (curved,   "curved") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID ||
          GETVALUE (aw,       "arrow head width") == INVALID ||
          GETVALUE (ah,       "arrow head height") == INVALID ||
          GETVALUE (locked,   "locked") == INVALID)
      {
         cfree (*ObjPtr);
         cfree (poly_ptr);
         cfree (v);
         *ObjPtr = NULL;
         return;
      }
      if (id >= objId) objId = id+1;
   }

   if (fileVersion <= 16 && width <= 6)
   {
      aw = origArrowHeadW[width];
      ah = origArrowHeadH[width];
      width = origWidthOfLine[width];
   }
   fill = UpgradePenFill (fill);
   pen = UpgradePenFill (pen);

   poly_ptr->style = style;
   poly_ptr->width = width;
   poly_ptr->aw = aw;
   poly_ptr->ah = ah;
   poly_ptr->pen = pen;
   poly_ptr->curved = curved;
   poly_ptr->fill = fill;
   poly_ptr->dash = dash;

   poly_ptr->vlist = v;
   poly_ptr->svlist = poly_ptr->asvlist = NULL;

   (*ObjPtr)->x = ltx;
   (*ObjPtr)->y = lty;
   (*ObjPtr)->color = QuickFindColorIndex (color_str, &new_alloc);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->id = id;
   (*ObjPtr)->rotation = rotation;
   (*ObjPtr)->locked = locked;
   (*ObjPtr)->type = OBJ_POLY;
   (*ObjPtr)->obbox.ltx = ltx;
   (*ObjPtr)->obbox.lty = lty;
   (*ObjPtr)->obbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = rby;
   CalcPolyBBox ((*ObjPtr)->obbox, num_pts, v, style, width, aw, ah,
         &((*ObjPtr)->bbox.ltx), &((*ObjPtr)->bbox.lty), &((*ObjPtr)->bbox.rbx),
         &((*ObjPtr)->bbox.rby));
   (*ObjPtr)->detail.p = poly_ptr;
   AdjObjSplineVs (*ObjPtr);
}

void FreePolyObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   if (ObjPtr->detail.p->svlist != NULL) cfree (ObjPtr->detail.p->svlist);
   if (ObjPtr->detail.p->asvlist != NULL) cfree (ObjPtr->detail.p->asvlist);
   cfree (ObjPtr->detail.p->vlist);
   cfree (ObjPtr->detail.p);
   cfree (ObjPtr);
}
